//
// tool.h: Interface to "tools"
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#ifndef shadow_tool_h
#define shadow_tool_h

// ====================================================================== //

#include <list>
#include <map>

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

#include "mystring.h"
#include "xrtypes.h"
#include "shadow-nml.h"

// ====================================================================== //
//
//  TOOLS
//
// ====================================================================== //

// 
// class Tool: Used to store definition of tool (1 for each tool)
//

class Tool {
public:
    typedef ShadowNamelist::VariableList VariableList;
public:
    Tool(const string& name);
    virtual ~Tool();

    virtual const string& name() const;

    virtual const ShadowNamelist& namelist() const;
    virtual const VariableList& variables() const;
    virtual bool exists_var(const string& varname) const;
    virtual const Var* get_var(const string& varname) const;
    virtual int add_var(Var* var); 

    virtual void dump() const;

private:
    string name_;
    ShadowNamelist nml_;
};

// ====================================================================== //
//
//  TOOL INSTANCE
//
// ====================================================================== //

//
// class ToolInstance: used to  store an instance of a tool in an experiment
//
class ToolInstance {
public:
    ToolInstance(const Tool* tool, int instance);
    ToolInstance(const ToolInstance&);
    virtual ~ToolInstance();

    virtual ToolInstance* clone() const;
    virtual int copy(const ToolInstance& from);

    const string& name() const;
    virtual const string& instance_name() const;
    const Tool* tool() const;
    int instance() const;

    virtual void set_instance_name(const string&);

    bool exists_var(const string& varname) const;
    const Var* get_var(const string& varname) const;
    const Value* get_varval(const string& varname) const;

    int add_var(Var* var);
    int set_varval(const string& varname, Value* val);

    virtual void dump() const;

private:
    const Tool* tool_;
    int instance_;
    string instname_;
    ShadowNamelist nml_;
};

// ====================================================================== //
//
//  TOOL Manager
//
// ====================================================================== //

class ToolMgr {
public:
    ToolMgr();
    virtual ~ToolMgr();

    const Tool* find(const string& name) const;
    Tool* find(const string& name);
    Tool* add(const string& name);
    int add_alias(const string& alias, const string& realname);

    //
    // for Tcl interpreter access.
    //
    const list<string>& tools() const;

    void dump() const;

private:
    int unalias(string& name) const;

private:
    typedef map<string, void*, less<string> > ToolMap;
    typedef map<Mystring, Mystring, less<Mystring> > ToolAliasMap;
    ToolMap tools_;
    ToolAliasMap tool_aliases_;
    list<string> toolnames_;
};

// ====================================================================== //

#endif/*shadow_tool_h*/
