//
// synonym.h:
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

#ifndef synonym_h
#define synonym_h

// =================================================================== //

#include "xrtypes.h"

// =================================================================== //
//
// SynonymTable defines the mapping of Symbolic names to integer values
// (and vice versa) used extensively within SHADOW. Symbols are tagged
// by the variable it belongs to and each variable can have a list of
// symbols, each with it's associated integer value (in the future, it'll
// be generalized to Value type, not just integer). When querying the
// value corresponding to the Symbolic name, SynonymTable does substring
// matching, so unambiguous substrings is all that's needed.
//
// eg., the variable FDISTR (in SRC tool), has the following symbols in
// the MENU description:
//
//    FDISTR          FLAT            01              
//    FDISTR          UNIFORM         02              
//    FDISTR          GAUSSIAN        03              
//    FDISTR          SYNCHR          04              
//    FDISTR          CONICAL         05              
//    FDISTR          EXACT-SR        06       
//    FDISTR          PHASE-ELLIPSE   07   
//
// we can now query using symbolic name "UNIFORM", "UNIFOR", or any unique
// substring of UNIFORM and get 02 in return. 
//    int value = synonym_table.translate("FDISTR", "UNIFORM");
//    int value = synonym_table.translate("FDISTR", "UNI");
// both would return 2 in this case.
//
// for correctness, the client should use "exists" method *before* calling
// translate, since translate assume unambigous existence of the symbol
// for that variable.
//    
//    char char* symbol = "UNI";
//    ...
//    SynonymTable::MatchType match;
//    bool exists = synonym_table.exists("FDISTR", symbol, match);
//    if (match == SynonymTable::UNIQUE) {
//        int value = translate("FDISTR", symbol);
//    }
// 
// If match == SynonymTable::NONE or SynonymTable::AMBIGUOUS, it is upto
// to the client to do the right thing and retry.
//
// reverse translation (from integer to symbolic representation) also
// exists.
//
// SynonymTable is basically a super-container for Synonym objects, which
// is a private class defined in synonym.cc.
//
// =================================================================== //

class Stq;
class Hashtable;

class SynonymTable {
public:
    enum {NONE, UNIQUE, AMBIGUOUS};
    typedef unsigned MatchType;
public:
    SynonymTable(unsigned hash_tabsize = 17);
    virtual ~SynonymTable();

    int add(const char* variable, const char* symbol, int value);
    bool exists(const char* variable, const char* symbol) const;
    bool exists(const char* variable, int value) const;
    bool exists(const char* variable, const char* symbol, MatchType&) const;
    bool exists(const char* variable, int value, MatchType&) const;
    int translate(const char* variable, const char* symbol) const;
    const char* translate(const char* variable, int value) const;
    int translate(
	const char* variable, const char* symbol, MatchType&
    ) const;
    const char* translate(
	const char* variable, int value, MatchType&
    ) const;
    const Stq* get_symbols(const char* variable) const;
    void dump() const;
private:
    Stq* syn_list_;
    Hashtable* syn_tab_;
};

// =================================================================== //

#endif/*synonym_h*/
