//
// menu.cc: reader/manipulator for MENU files.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

// =================================================================== //

#ifdef __GNUG__
# include <iostream.h>
# include <cassert>
# include <cctype>
# include <cstdio>
# include <cstdlib>
# include <cstring>
#else
# include <iostream>
# include <assert>
# include <ctype>
# include <stdio>
# include <stdlib>
# include <string>
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif


#include "menu.h"
#include "shadow-nml.h"
#include "reader.h"
#include "relation.h"
#include "synonym.h"
#include "beamline.h"
#include "utils.h"

// =================================================================== //

Menu::Menu(const string& name, ToolMgr& toolmgr, bool debug) : 
    name_(name), 
    toolmgr_(toolmgr),
    debug_(debug),
    syn_table_(new SynonymTable),
    loaded_(false)
{ }

Menu::~Menu() { 
    delete syn_table_;
}

int Menu::load(const string& file) {
    if (loaded_ || !fileexists(file.c_str()))
	return -1;
    MenuReader reader(toolmgr_, const_cast<Menu*>(this), debug_);
    loaded_ = reader.load(file) == 0;
    if (loaded_) {
	file_ = file;
    }
    return (loaded_) ? 0 : 1;
}

const char* Menu::name() const { return name_.c_str(); }

MenuPage* Menu::find_page(const string& name) {
    map<string, void*, less<string> >::iterator it = page_tab_.find(name);
    MenuPage* page = nil;
    if (it != page_tab_.end()) {
	page = static_cast<MenuPage*>((*it).second);
    }
    return page;
}

const MenuPage* Menu::find_page(const string& name) const {
    map<string, void*, less<string> >::const_iterator it =
	page_tab_.find(name);
    MenuPage const* page = nil;
    if (it != page_tab_.end()) {
	page = reinterpret_cast<const MenuPage*>((*it).second);
    }
    return page;
}

int Menu::add_page(MenuPage* page) {
    if (find_page(page->name()) != nil)
	return -1;
    page_list_.push_back(page->name());
    page_tab_[page->name()] = page;
    return 0;
}

const list<string>& Menu::get_pages() const { return page_list_; }

SynonymTable* Menu::get_synonym_table() const { return syn_table_; }

void Menu::dump() const {
    cerr << "Dumping MENU \"" << name() << "\" pages ..." << endl;
    list<string>::const_iterator it = page_list_.begin();
    for(; it != page_list_.end(); ++it) {
	const string& page_name = *it;
	const MenuPage* page = find_page(page_name);
	page->dump();
    }
    cerr << "done." << endl;
}

// =================================================================== //
 
