/*
 * common.h: C interface to SHADOW FORTRAN common block data.
 *
 * Source: src/utils/cnamelist/common.h
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * ------------------------------------------------
 *             SHADOW
 *    Center for X-ray Lithography
 *   University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 */

/*
 * 
 * The common blocks are accessed as global structures in the C part of
 * SHADOW. Note that for most systems (ultrix, sunos), the compiler tacks
 * on a underscore at the end of ALL fortran names, so the linker gets 
 * very confused if the C code calls the FORTRAN routines or accesses
 * FORTRAN variables w/out this trailing underscore.
 *
 * EXCEPTION: AIX 3.x on IBM RS/6000 DOES NOT need this underscore; in
 * fact CAN'T have it. Probably the same for HP-UX, VMS, etc.
 *
 * The other gotcha on i486-linux is due to g77's problems with common
 * blocks (such as no debugging info). Also notice the extra underscores
 * after ROT_MIRR and RIPP_2. Have no idea why!
 *
 */

#ifndef shadow_common_h
#define shadow_common_h

#if __cplusplus
extern "C" {
#endif

#if !NO_FORT_UNDERSCORE
# define F77_NAME(f) f##_
#else
# define F77_NAME(f) f
#endif

/*
 * G77 (GNU FORTRAN, upto and including v.0.5.18) adds a SECOND underscore
 * to names that already contain an underscore. Annoying as well, and
 * hence the G77_HACK_NAME(f) hack.
 */

#define G77_HACK_NAME(f) f##__

#define MATHBLK		F77_NAME(mathblk)
#define FLAGS 		F77_NAME(flags)
#define CALC 		F77_NAME(calc)
#define SYSTEM1 	F77_NAME(system1)
#define MIRROR 		F77_NAME(mirror)
#define GRATING 	F77_NAME(grating)
#define XTAL 		F77_NAME(xtal)
#if G77
# define ROT_MIR 	G77_HACK_NAME(rot_mir)
#else
# define ROT_MIR 	F77_NAME(rot_mir)
#endif
#define SLIT 		F77_NAME(slit)
#define SOURCE 		F77_NAME(source)
#define ALADDIN 	F77_NAME(aladdin)
#define TRIG 		F77_NAME(trig)
#define IMAGE 		F77_NAME(image)
#define AXIS 		F77_NAME(axis)
#define RIPPLE 		F77_NAME(ripple)
#define SEGMENT		F77_NAME(segment)
#define FACET		F77_NAME(facet)
#define KOMA		F77_NAME(koma)
#define NAMES 		F77_NAME(names )
#if G77
# define RIPP_2 	G77_HACK_NAME(ripp_2)
#else
# define RIPP_2 	F77_NAME(ripp_2)
#endif
#define LIGHT 		F77_NAME(light)
#define SCREENS 	F77_NAME(screens)
#define HOLO 		F77_NAME(holo)
#if G77
# define OE_NUMBER 	G77_HACK_NAME(oe_number)
#else
# define OE_NUMBER 	F77_NAME(oe_number)
#endif
#define DUMMY 		F77_NAME(dummy)
#define IDUMMY 		F77_NAME(idummy)
#if G77
# define SCR_NUMBER	G77_HACK_NAME(scr_number)
#else
# define SCR_NUMBER	F77_NAME(scr_number)
#endif

#define FNAME_LEN	80

/*
 * Now map each FORTRAN common block onto a C struct and off we go!
 */

struct XMATHBLK {
    double pi, twopi, pihalf, todeg, torad, origin[3],
	x_vrs[3], y_vrs[3], z_vrs[3], tocm, toangs;
};
extern struct XMATHBLK MATHBLK;

struct XFLAGS {
    int fmirr, f_torus, fcyl, fang, fstat, f_coher,
	fang1, fslit, fgrid, f_new, fsource_depth,
	fsour, fdistr, fwrite, f_bragg_a, f_johansson,
	f_pol, f_polar, f_ripple, f_mosaic,
	f_move, f_how, f_g_s, f_read_rip, f_r_ran,
	f_grating, f_side, f_central, f_virtual,
	f_color, f_convex, f_reflec, f_ruling,
	f_rul_abs, f_bound_sour, f_thick,
	f_ext, f_phot, f_screen, f_plate, fshape,
	fhit_c, f_pw, f_mono, f_default, f_refl,
	f_hunt, f_phot_cent, f_crystal, f_refrac,
	f_pw_c, f_opd, f_wiggler, fzp, f_sr_type,
	f_roughness, fdummy, f_angle, f_segment;
}; 
extern struct XFLAGS FLAGS;

struct XCALC {
    int npoint, ncol, istar1, ido_x_s, ido_z_s,
	ido_y_s, ido_vz, ido_vx, ido_xl, ido_xn,
	ido_zl, ido_zn, n_plates, ig_seed, n_color,
	n_circle, n_cone, n_screen, i_screen[10],
	i_slit[10], k_slit[10], i_stop[10],
	i_abs[10], mpurge[2], ndeg, mosaic_seed;
};
extern struct XCALC CALC;

struct XSYSTEM1 {
    double alpha, ssour, theta, psour[3], simag,
	rdsour, rtheta, psreal[3], timdis,
	delta, rdelta, off_soux, off_souy, off_souz,
	alpha_s;
};
extern struct XSYSTEM1 SYSTEM1;

struct XMIRROR {
    double rlen, rlen1, rlen2, ccc[10], rmirr, cone_a,
	axmaj, axmin, afoci, eccent, r_maj, r_min,
	rwidx, rwidx1, rwidx2, param,
	pcoeff[5][5][5], cil_ang, ell_the;
};
extern struct XMIRROR MIRROR;

struct XGRATING {
    double ruling, order, beta, phot_cent, r_lambda,
	hunt_l, hunt_h, blaze, d_spacing, azim_fan,
	dist_fan, coma_fac, rul_a1, rul_a2, rul_a3,
	rul_a4, a_bragg;
};
extern struct XGRATING GRATING;


struct XXTAL {
    double spread_mos,  r_johansson,  thickness;
};
extern struct XXTAL XTAL;

struct XROT_MIR {
    double x_rot, y_rot, z_rot, offx, offy, offz,
	u_mir[3], v_mir[3], w_mir[3];
};
extern struct XROT_MIR ROT_MIR;

struct XSLIT {
    double sllen, slwid, sltilt, cod_len, cod_wid;
};
extern struct XSLIT SLIT;

struct XSOURCE {
    double wxsou, wysou, wzsou, sigmax, sigmay, sigmaz,
	hdiv1, hdiv2, vdiv1, vdiv2, sigdix, sigdiz,
	conv_fact, cone_max, cone_min, x_sour, y_sour,
	z_sour, x_sour_rot, y_sour_rot, z_sour_rot,
	u_sour[3], v_sour[3], w_sour[3], plasma_angle;
};
extern struct XSOURCE SOURCE;

struct XALADDIN {
    double photon[10], bener, r_aladdin,
	epsi_x, epsi_z, epsi_dx, epsi_dz, r_magnet;
};
extern struct XALADDIN ALADDIN;

struct XTRIG {
    double cosdel, sindel, costhe, sinthe, costhr,
	sinthr, cosder, sinder, cosal, sinal,
	cosal_s, sinal_s, costhe_i, sinthe_i,
	cosal_i, sinal_i;
};
extern struct XTRIG TRIG;

struct XIMAGE {
    double rimcen[3], vnimag[3], uxim[3], vzim[3],
	d_plate[5], c_star[3], c_plate[3],
	ux_pl[3], vz_pl[3], wy_pl[3],
	theta_i, alpha_i;
};
extern struct XIMAGE IMAGE;

struct XAXIS {
    double central[20][24], t_incidence, t_source,
	t_image, t_reflection;
};
extern struct XAXIS AXIS;

struct XRIPPLE {
    double x_rip_amp, x_rip_wav, x_phase,
	y_rip_amp, y_rip_wav, y_phase,
	rough_x, rough_y;
};
extern struct XRIPPLE RIPPLE;

struct XSEGMENT {
    int iseg_xnum, iseg_ynum;

    double seg_lenx, seg_leny;
};
extern struct XSEGMENT SEGMENT;

struct XFACET {
    int f_polsel, f_facet, ifac_x, ifac_y,
	f_fac_orient, f_fac_latt;

    double rfac_lenx, rfac_leny, rfac_phax, rfac_phay,
	rfac_delx1, rfac_delx2, rfac_dely1, rfac_dely2;
};
extern struct XFACET FACET;

struct XKOMA {
    int f_koma, koxx, i_koma, f_dot;

    double zko_length, rkoma_cx, rkoma_cy;

    int f_koma_ca, f_koma_bounce,
	f_exit_shape, f_inc_mnor_ang;
};
extern struct XKOMA KOMA;

struct XNAMES {
    char file_source[FNAME_LEN], file_traj[FNAME_LEN],
	file_rip[FNAME_LEN], file_refl[FNAME_LEN],
	file_segment[FNAME_LEN], file_segp[FNAME_LEN],
#if HP_F77_BUGGY_NAMELIST
	filabs[10][FNAME_LEN],
#else 
	file_abs[10][FNAME_LEN],
#endif
	file_mir[FNAME_LEN],
	file_bound[FNAME_LEN],
	file_rough[FNAME_LEN], file_fac[FNAME_LEN],
	file_koma[FNAME_LEN], file_koma_ca[FNAME_LEN];
};
extern struct XNAMES NAMES; 

struct XRIPP_2 {
    double ampli[10], x_gr[10], y_gr[10], signum[10],
	sig_xmin[10], sig_xmax[10], sig_x[10],
	sig_ymin[10], sig_ymax[10], sig_y[10],
	ampl_in[10];

    int n_rip;
};
extern struct XRIPP_2 RIPP_2;

struct XLIGHT {
    double alfa, gamma, pol_angle, pol_deg,
	r_ind_obj, r_ind_ima,
	ph1, ph2, ph3, ph4, ph5, ph6, ph7, ph8, ph9, ph10,
	sigxl1, sigxl2, sigxl3, sigxl4, sigxl5,
	sigxl6, sigxl7, sigxl8, sigxl9, sigxl10,
	sigzl1, sigzl2, sigzl3, sigzl4, sigzl5,
	sigzl6, sigzl7, sigzl8, sigzl9, sigzl10;
};
extern struct XLIGHT LIGHT;

struct XSCREENS {
    double rx_slit[10], rz_slit[10], sl_dis[10],
	ux_scr[3][2], wy_scr[3][2], vz_scr[3][2],
	thick[10];
};
extern struct XSCREENS SCREENS;

struct XHOLO {
    double holo_r1, holo_r2, holo_del, holo_gam,
	holo_w, holo1[3], holo2[3], holo_rt1,
	holo_rt2;
};
extern struct XHOLO HOLO;


/*
 * defined in namelist.c.
 */
extern int OE_NUMBER;
extern double DUMMY;
extern int IDUMMY;
extern int SCR_NUMBER[10];

#if __cplusplus
}
#endif

#endif /*shadow_common_h*/
