/*
 * nml-modify.c: modify parameter value(s) in FORTRAN namelist file 
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * ------------------------------------------------
 *              SHADOW
 *    Center for X-ray Lithography
 *   University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "common.h"
#include "namelist.h"

typedef struct ParamListRec ParamListRec, *ParamList;
struct ParamListRec {
    char	*name;
    char	*value;
    ParamList	next;
};

static int parse_args (int argc, char** argv, const char* prog_name, 
    char** infile, char** outfile, NmlType* type, ParamList* params
);
static void usage(const char* prog_name);
static int do_modify (Namelist* namelist, ParamList params);
static char* str_lo_case(char *str);
static char* str_up_case(char *str);


int main (int argc, char** argv) {
    Namelist* namelist = nil;
    ParamList params = NULL;
    char* prog_name = nil;
    char* nml_file = nil;
    char* out_file = nil;
    NmlType type;
    int status = 0;

    char *ptr = strrchr(argv[0], '/');

    if (ptr) 
	prog_name = ptr + 1;
    else 
	prog_name = argv[0];

    parse_args(argc, argv, prog_name, &nml_file, &out_file, &type, &params);
 
    namelist = read_namelist(nml_file, type);
    if (namelist == nil) {
	fprintf(stderr, "ERROR: Cannot reade namelist file `%s'\n", nml_file);
	exit(1);
    }
    if (status = do_modify(namelist, params)) {
	fprintf(stderr, "ERROR: Cannot modify some parameters\n");
    } else {
	if (status = write_namelist(out_file, type)) {
	    fprintf(stderr, 
	        "ERROR: Cannot write namelist file `%s'\n", out_file
	    );
	}
    }
    exit(status);
}

int do_modify (Namelist* namelist, ParamList params) {
    int err = 0;
    for (; params; params = params -> next) {
	NamelistData *nml_data = get_namelist_data(params->name, namelist);

	if (!nml_data) {
	    error ("%%%% variable \"%s\" not found in namelist",
	       params -> name);
	    ++err;
	}
	else if (change_value(params->name, params->value, namelist)) {
	    error ("%%%% Illegal value \"%s\" given for \"%s\"",
	       params -> value, params -> name);
	    ++err;
	}
    }
    return err;
}


int parse_args (int argc, char** argv, const char* prog_name, char** infile, 
    char** outfile, NmlType* type, ParamList* paramsp
) {
    extern int optind;
    extern char *optarg;

    boolean type_supplied = FALSE;
    int c;

    *infile = nil;
    *outfile = nil;

    while ((c = getopt (argc, argv, "hf:t:o:")) != EOF) {
	switch (c) {
	    case 't':
		if (optarg[0] == 's' || optarg[0] == 'S') {
		    *type = NML_source;
		}
		else if (optarg[0] == 'o' || optarg[0] == 'O') {
		    *type = NML_oe;
		}
		else {
		    fprintf(stderr, "Illegal namelist type --  %s\n", 
			optarg
		    );
		    usage(prog_name);
		    exit(1);
		}
		type_supplied = TRUE;
		break;

	    case 'o':
		*outfile = strcpy (malloc (strlen (optarg) + 1), optarg);
		break;
	    
	    case 'f':
		*infile = strcpy (malloc (strlen (optarg) + 1), optarg);
		break;

	    case 'h':
		usage (prog_name);
		exit (0);
		break;
	    
	    default:
		usage (prog_name);
		exit (1);
		break;
	}
    }

    if (*infile == nil || *outfile == nil || type_supplied == FALSE) {
	fprintf(stderr, "Must supply all required parameters.\n");
	usage(prog_name);
	exit(1);
    }

    if (argc > optind) {
	char tmpbuf[BUFSIZ];
	char *eq = NULL;
	ParamList phead = NULL;
	ParamList ptail = NULL;
	int i = optind;
	for (; i < argc; i++) {
	    strcpy (tmpbuf, argv[i]);
	    if (!(eq = strrchr (tmpbuf, '='))) {
		error ("Error in name=value syntax. No \'=\' sign\n");
		usage (prog_name);
		exit (1);
	    }
	    if (!phead) {
		phead = ptail = (ParamList) malloc (sizeof (ParamListRec));
		phead -> next = ptail -> next = NULL;
	    }
	    else {
		ptail -> next = (ParamList) malloc (sizeof (ParamListRec));
		ptail = ptail -> next;
		ptail -> next = NULL;
	    }

	    ptail -> value = strcpy (malloc (strlen (eq+1) + 1), eq+1);
	    *eq = '\0';
	    ptail -> name = strcpy (malloc (strlen (tmpbuf) + 1), tmpbuf);
	    str_lo_case (ptail -> name);
	}

	if (!phead) {
	    fprintf (stderr, "Warning: no name=value pair to modify\n");
	}
	else {
	    *paramsp = phead;
	}
    }
    return 0;
}


void usage (const char* prog_name) {
    fprintf (stderr, "\n"
    "Usage:%s [-h] -f Namelist -t NamelistType -o output [name=value ...]\n\n\
    -h:			print this info\n\
    -b:			run in batch mode (ie., no screen interface)\n\
    -f NamelistFile:	SHADOW namelist (eg., start.00, start.01, etc)\n\
    -o Output:          Output namelist with modified value(s)\n\
    -t NamelistType:	\"source\"/\"oe\" (also accepts \"s\"/\"o\")\n\n\
    name=value:		where variable \"name\" is assigned \"value\" and the\n\
			namelist file is written out with the new value.\n\
			valid only when BATCH mode [-b] is specified. No\n\
			spaces are allowed around \'=\'\n\n",
    prog_name);

    fprintf (stderr, 
    "Examples:\n\
    %s -b -f start.00 -o new.00 -t source npoint=4000 fdistr=1\n\
    %s -b -f start.01 -o new.01 -t oe mosaic_seed=12351\n\n",
    prog_name, prog_name);

    return;
}

static char* str_up_case(char *str) {
    char *str_ptr = str;
    for (; *str_ptr; ++str_ptr) {
	if (islower (*str_ptr)) 
	    *str_ptr = toupper (*str_ptr);
	if (*str_ptr == '-') 
	    *str_ptr = '_';
    }
    return str;
}


static char* str_lo_case(char *str) {
    char *str_ptr = str;
    for (; *str_ptr; ++str_ptr) {
	if (isupper (*str_ptr)) 
	    *str_ptr = tolower (*str_ptr);
	if (*str_ptr == '-') 
	    *str_ptr = '_';
    }
    return str;
}
