//
// exp2report.cc: create a simple report from EXP file
//
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1996 Mumit Khan
//
//

// ====================================================================== //

#include <cassert>
#include <cstdlib>
#include <cstring>

#include <fstream>
#include <string>

//
// On Linux/glibc2, gettext macro causes problems with Value::gettext()
// in value.h, so we just undefine it here.
//
#ifdef gettext
# undef gettext
#endif

#include "exp.h"
#include "exp2report.h"
#include "freevars.h"
#include "global.h"
#include "logger.h"
#include "job.h"
#include "misc.h"
#include "pmu.h"
#include "tool.h"
#include "utils.h"
#include "value.h"
#include "variable.h"

using namespace std;

// ====================================================================== //

//
// global variables. Need this for the EXCON main globals. FIX/CLEAN/TODO.
//

bool g_verbose = false;

// ====================================================================== //

static int write_sequence(ostream& os, const Stq* job) {
    os << "\n================= Experiment sequence ===================\n\n";
    for(int i = 0; i < job->size(); ++i) {
	ToolInstance* ti = (ToolInstance*)job->cycle();
	if (i) 
	    os << " -> ";
	os << ti->getinstname();
    }
    os << endl;
    return 0;
}

//
// write_parameters: Write out the parameter values from the current
// Tool Instance. 
//
static int write_parameters(ostream& os, const ToolInstance* ti) {
    string toolname(ti->getname());
    string toolinstance(ti->getinstname());
    if (g_verbose) {
	EXCON_LOG_ALWAYS
	    << "Writing Tool `" << ti->getinstname() << "' params." 
	    << endl;
    }

    os << "\n=== TOOL: " << toolname << ", Instance: " << toolinstance 
	<< " ===\n\n";

    const Tool* tool = ti->gettool();

    // 
    // Query all variables defined by the Tool, but get the variables
    // from the ToolInstance so we can tell which one was overridden
    // and which one came as a default from the TOOLS file.
    //

    int found = 0;
    os << "*** Model (input) parameters. Defaults from TOOLS file:\n\n";
    for (int i = 1; i <= tool->nvars(); ++i) {
	const char* varname = tool->getvar(i)->getname();
	const Var* var = ti->getvar(varname);
	assume (var != 0);
	if ( 
	    var->hasprop("$INPUTS") && !var->hasprop("$OVERRIDDEN_VALUE") &&
	    !var->hasprop("$LINKS") && !var->hasprop("$OUTPUTS")
	) {
	    const Value* val = ti->getvarval(var->getname());  
	    assume (val != 0);
	    os << var->getname() << " = " << val->getstrval() << endl;
	    ++found;
	}
    }
    if (found == 0) {
	os << "(NONE)" << endl;
    }
    os << endl; 
    found = 0;
    os << "*** Model (input) parameters. Overridden from EXP file:\n\n";
    for (int i = 1; i <= tool->nvars(); ++i) {
	const char* varname = tool->getvar(i)->getname();
	const Var* var = ti->getvar(varname);
	assume (var != 0);
	if ( 
	    var->hasprop("$INPUTS") && var->hasprop("$OVERRIDDEN_VALUE") &&
	    !var->hasprop("$LINKS") && !var->hasprop("$OUTPUTS")
	) {
	    const Value* val = ti->getvarval(var->getname());  
	    assume (val != 0);
	    os << var->getname() << " = " << val->getstrval() << endl;
	    ++found;
	}
    }
    if (found == 0) {
	os << "(NONE)" << endl;
    }
    os << endl; 
    found = 0;
    os << "*** Linked (from previous tools in the sequence) parameters:\n\n";
    for (int i = 1; i <= tool->nvars(); ++i) {
	const Var* var = tool->getvar(i);  
	assume (var != 0);
	if (var->hasprop("$LINKS")) {
	    const Value* val = ti->getvarval(var->getname());  
	    assume (val != 0);
	    os << var->getname() << " = " << val->getstrval() << endl;
	    ++found;
	}
    }
    if (found == 0) {
	os << "(NONE)" << endl;
    }
    os << endl; 
    found = 0;
    os << "*** Output (auto-generated) parameters:\n\n";
    for (int i = 1; i <= tool->nvars(); ++i) {
	const Var* var = tool->getvar(i);
	assume (var != 0);
	if (var->hasprop("$OUTPUTS")) {
	    const Value* val = ti->getvarval(var->getname());  
	    assume (val != 0);
	    os << var->getname() << " = " << val->getstrval() << endl;
	    ++found;
	}
    }
    if (found == 0) {
	os << "(NONE)" << endl;
    }
    os << endl; 
    return 0;
}

static int write_freevar(ostream& os, const FreeVariable* fv) {
    const char* varname = fv->getvar();
    const char* toolname = fv->gettool();
    switch(fv->type()) {
    case FreeVariable::loop:
	os << "Loop: " << toolname << ":" << varname << endl;
	break;
    case FreeVariable::set:
	os << "Set: " << toolname << ":" << varname << endl;
	break;
    case FreeVariable::random:
	os << "Random: " << toolname << ":" << varname << endl;
	break;
    case FreeVariable::unknown:
    default:
	os << "ILLEGAL: " << toolname << ":" << varname << endl;
	break;
    }
    
    return 0;
}

//
// write_variable_parameters: Write out the VARY section.
//
static int write_variable_parameters(ostream& os, const Exp* exp) {
    os << "\n=== VARY section ===\n\n";

    const Stq* fvgroups = exp->get_freevar_groups();
    int nfvgroups = fvgroups->size();
    for(int i = 0; i < nfvgroups; ++i) {
	const FreeVariableGroup* fvgr = (FreeVariableGroup*)fvgroups->cycle();
	const Stq* freevars = fvgr->getgroup();
	int nfv = freevars->size();
	for(int j = 0; j < nfv; ++j) {
	    const FreeVariable* fv = (FreeVariable*)freevars->cycle();
	    write_freevar(os, fv);
	}
    }
    return 0;
}

//
// write_report: writes out all the parameters from the current exp.
//
static int write_report(Exp2ReportMgr& expmgr, const string& output) {
    //
    // now create the job sequence for the first iteration. That's all
    // we can do w/out having to actually run the job.
    //
    Exp* exp = expmgr.exp();
    exp->setup_links(exp->get_job_sequence(0));
    Stq* job = exp->get_next_job(1);
    assert(job != 0);

    exp->resolveunique(job);
    unsigned jobsize = job->size();
    while(jobsize--) {
	ToolInstance* ti = (ToolInstance*)job->cycle();
	exp->resolvelinks(ti);
    }

    ofstream ofp(output.c_str());
    if (!ofp) {
	EXCON_LOG_ALWAYS
	    << endl
	    << "exp2report: "
	    << "Cannot open output file `" << output << "'.";
	exit(1);
    }

    write_sequence(ofp, job);

    jobsize = job->size();
    while(jobsize--) {
	ToolInstance* ti = (ToolInstance*)job->cycle();
	write_parameters(ofp, ti);
    }

    write_variable_parameters(ofp, exp);
    return 0;
}

// ====================================================================== //

int main(int argc, char** argv) {
    string toolsfile, expfile, output;
    exp2report_init(argc, argv, toolsfile, expfile, output);

    if (Globals::toolsmgr->load (toolsfile.c_str())) {
	EXCON_LOG_ALWAYS
	    << "Exp2Report: Errors found int TOOLS file(s). Fix and re-run"
	    << endl;
	exit(1);
    }

    if (Globals::expmgr->load (expfile.c_str())) {
	EXCON_LOG_ALWAYS
	    << "Exp2Report: Errors found in EXPERIMENT file(s). Fix and re-run"
	    << endl;
	exit(1);
    }

    Exp2ReportMgr& expmgr = (Exp2ReportMgr&)*Globals::expmgr;
    return write_report(expmgr, output);
}


// ====================================================================== //

