/*
 * hash.c: hash table stuff for parameter reader/writer of EXCON.
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * Source: src/utils/io/hash.c
 *
 * ------------------------------------------------
 * Mumit Khan
 * Center for X-ray Lithography
 * University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 * Log:	hash.c
 * Revision 1.3  90/08/30  18:03:03  khan
 * About to add my own hashing routines.
 * 
 * Revision 1.2  90/08/21  20:50:10  khan
 * Added allocation rules to the hash tables. Specified at creation 
 * time.
 * 
 * Revision 1.1  90/08/18  18:41:36  khan
 * Initial revision
 * 
 *
 */

#include <stdio.h> 
#include <stdlib.h>
#include "hash.h"

#ifdef __STDC__
#include <memory.h>
/*#define bcopy(src, dest, len) memcpy((dest), (src), (len))*/
/*#define bzero (src, len) memset((src), 0, (len))*/
#endif



static char *lo_case (char *str) 
{
    char *p = str;

#ifdef WEIRD
    for (; *p ; *p = (char) (isupper (*p)) ? tolower ((int) *p) : *p, p++)
	;
#else
    for (; *p ; p++) {
	*p = (isupper (*p)) ? tolower (*p) : *p;
    }
#endif
    return str;
}




static int mystrcmp (char *str1, char *str2, int case_sensitive) {

    int retval = 0;
    if (case_sensitive)
	retval = strcmp (str1, str2);
    else {
	static char *loc_str1 = NULL;
	static unsigned lstr1 = 0;
	static char *loc_str2 = NULL;
	static unsigned lstr2 = 0;
	int same = 0;
	unsigned len1 = strlen (str1);
	unsigned len2 = strlen (str2);

	if (len1 != len2) {
	    retval = 1;
	}
	else {
	    if (len1 >= lstr1) {
		lstr1 += BUFSIZ;
		loc_str1 = (!loc_str1) ? 
		    malloc (lstr1) : realloc (loc_str1, lstr1);
		if (!loc_str1) {
		    fprintf (
			stderr, "mystrcmp (libcxrl): cannot allocate memory\n"
		    );
		    exit (1);
		}
	    }
	    strcpy (loc_str1, str1);
	    (void) lo_case (loc_str1);
	    if (len2 >= lstr2) {
		lstr2 += BUFSIZ;
		loc_str2 = (!loc_str2) ? 
		    malloc (lstr2) : realloc (loc_str2, lstr2);
		if (!loc_str2) {
		    fprintf (
			stderr, "mystrcmp (libcxrl): cannot allocate memory\n"
		    );
		    exit (1);
		}
	    }
	    strcpy (loc_str2, str2);
	    (void) lo_case (loc_str2);
	    retval = strcmp (loc_str1, loc_str2);
	}
    }
    return retval;
}



/* 
 * Make a new hash table for EXCON. Number of buckets is given by the input
 * parameter num_buckets. It returns a pointer to the hash table just
 * created. 
 *
 * alloc_rules defines what the hashing functions should allocate when an 
 * item is added to a table. 
 *     alloc_rules : ALLOCATE_KEY (allocate room for key).
 *                 : ALLOCATE_DATA (allocate room for data).
 *                 : ALLOCATE_BOTH (allocate room for both).
 *                 : ALLOCATE_NONE (don't allocate, but simply point).
 */


htable make_htable (num_buckets, alloc_rules)
int num_buckets;			/* number of buckets to use */
unsigned int alloc_rules;		/* allocation rules (see above) */
{
    htable table = (htable) xmalloc (sizeof (htable_rec));

    table -> buckets = 
	(hbucket *) xmalloc (num_buckets * sizeof (hbucket));
    
#ifdef __STDC__
    memset ((char *) table -> buckets, 0, num_buckets * sizeof (hbucket));
#else
    bzero ((char *) table -> buckets, num_buckets * sizeof (hbucket));
#endif
    table -> num_buckets = num_buckets;
    table -> num_entries = 0;
    table -> alloc_rules = alloc_rules;

    return table;
}

/* 
 * Delete the hash table given by "table". This requires going thru all the
 * elements to free memory allocated to the fields in each bucket.
 *
 * Note that only elements that were allocated by the hashing functions are
 * deallocated (as determined by the htable -> alloc_rules in make_htable).
 */

void delete_htable (table)
htable table;
{
    register unsigned int i = 0;
    hbucket bptr = NULLBUCKET;
    hbucket temp = NULLBUCKET;

    for (; i < table -> num_buckets; i++) {
	for (bptr = table -> buckets[i]; bptr;) {
	    if (table -> alloc_rules & ALLOCATE_KEY)
                free (bptr -> key);
	    if ((table -> alloc_rules & ALLOCATE_DATA) && bptr -> data)
		free (bptr -> data);
	    temp = bptr;
	    bptr = bptr -> next;
	    free (temp);
	}
    }
    free (table -> buckets);
    free (table);
}
    

/* 
 * Get the hashing position for the given key in the hash
 * table.
 */
 
int hash_key (table, key)
htable table;
char *key;
{
    register unsigned int i = 0;

    while (*key) 
	i += *key++;
    
    return i %= table -> num_buckets;
}

/* 
 * Return a pointer to the hashed item, or NULL if the item can't 
 * be found. 
 */

hbucket find_hitem (table, key)
htable table;
char *key;
{
  hbucket list = table -> buckets[hash_key (table, key)];
  hbucket ret = NULLBUCKET;

  while (list) {
    if (mystrcmp (list -> key, key, 0) == 0) {
      ret = list;
      break;
    }
    else list = list->next;
  }
  return ret;
}

/* 
 * Remove the item specified by STRING from the hash table TABLE.
 * The item removed is returned, so you can free its contents.  If
 * the item isn't in this table NULL is returned. 
 */

hbucket remove_hitem (table, key)
htable table;
char *key;
{
    int the_bucket = hash_key (table, key);
    hbucket prev = NULLBUCKET;
    hbucket temp = table -> buckets[the_bucket];

    while (temp) {
	if (mystrcmp (temp -> key, key, 0) == 0) {
	    if (prev) 
		prev -> next = temp -> next;
	    else 
		table -> buckets[the_bucket] = temp -> next;
	    table -> num_entries--;
	    return temp;
	}
	prev = temp;
	temp = temp->next;
    }
    return NULLBUCKET;
}

/* 
 * Add the given key to the hash table and return the bucket
 * it ends up in. If it already exists, just return the bucket.
 */

hbucket add_hitem (table, key, data, dsize)
htable table;			/* The hash table to dig thru */
char *key;			/* The key to search for */
char *data;			/* The data associated with the key */
unsigned int dsize;		/* The size of the data item */
{
    hbucket item;
    char *loc_key;		/* local copies of the pointers */
    char *loc_data;

    if (!(item = find_hitem (table, key))) {
	int bucket = hash_key (table, key);
	item = table -> buckets[bucket];

	while (item && item -> next) 
	    item = item -> next;
	if (item) {
	    item -> next = (hbucket) xmalloc (sizeof (hbucket_rec));
	    item = item -> next;
	} 
	else {
	    table -> buckets[bucket] = (hbucket) xmalloc (sizeof (hbucket_rec));
	    item = table -> buckets[bucket];
	}

	loc_key = key;
	loc_data = data;

        /* 
         * Note how the alloc_rules are used here. Allocate only if asked 
         * for in make_htable, else simply point to the originals.
         */

	if (table -> alloc_rules & ALLOCATE_KEY) 
	    loc_key = (char *) strcpy ((char *) xmalloc (strlen (key) + 1), key);
	
	if (table -> alloc_rules & ALLOCATE_DATA && dsize) 
#ifdef __STDC__
	    memcpy (loc_data = (char *) xmalloc (dsize), data, dsize);
#else
	    bcopy (data, loc_data = (char *) xmalloc (dsize), dsize);
#endif

	item -> key = loc_key;
	item -> data = loc_data;
	item -> next = NULLBUCKET;
	table -> num_entries++;
    }
    return item;
}



/* 
 * Return the bucket_contents list of bucket BUCKET in TABLE.  If
 * TABLE doesn't have BUCKET buckets, return NULL. 
 */

hbucket get_hbucket (table, buckets)
htable table;
int buckets;
{
    if (buckets >= 0 && buckets < table -> num_buckets) 
	return table -> buckets[buckets];
    else 
	return NULLBUCKET;
}


#ifdef TESTING

main ()
{
    char key[133];
    char val[133];
    char string[256];
    int count = 0;
    hbucket tt;
    htable table;

    table = make_htable (107, ALLOCATE_BOTH);

    printf ("Enter some data to be hashed, a word at a time.\n\
	Type a blank line when done:\n\n");

    for (;;) {
	gets (string);
	if (!*string) 
	    break;
	sscanf (string, "%s%s", key, val);
        if (tt = find_hitem (table, key)) 
	    printf ("Item %s already added. Data = %s\n", tt -> key, tt -> data);
	else {
	    tt = add_hitem (table, key, val, strlen (val) + 1);
	    count++;
	}
    }

    printf ("\nYou have entered %d (%d) items.  The items are:\n\n",
	table -> num_entries, count);

    for (count = 0; count < table->num_buckets; count++) {
	register hbucket list = get_hbucket (table, count);

	if (list) {
	printf ("%3d slot: ", count);
	      while (list) {
		printf ("%s:%s\n          ", list->key, list->data);
		list = list->next;
	      }
	  printf ("\n");
	}
    }
    delete_htable (table);
}
#endif  /* TESTING */
