//
// utils.cc: simple utility classes (linked list and hashing functions).
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

/*****************************************************************************
 *
 *  INCLUDES
 *
 ****************************************************************************/


#include <cstdio>
#include <iostream>
#include <cstring>
#include <cstdlib>
#include <cassert>
#include <fcntl.h>
#include <sys/stat.h>
#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#include <sys/types.h>
#include <pwd.h>			// for username stuff

#include "global.h"
#include "utils.h"

using namespace std;

/************************************************************************/

// NOTE: USER RESPONSIBLE FOR MEMORY MANAGEMENT OF item
void Stq::push (void const* item) {
    Stqitem* lp;
    if (free_ == 0) 
	lp = new Stqitem();
    else { 
	lp = free_;  
	lp->item = lp->next = 0;  
	free_ = 0; 
    }
    lp->item = (void*)item;	// UNCONST
    if (head_ == 0) 
	tail_ = lp;
    else 
	lp->next = head_;
    head_ = lp;
    nitems_++;
}

void* Stq::get (int ith) const {
    assume (ith > 0 && ith <= nitems_);
    void* ret = 0;
    Stqitem* lp = head_;
    int j = 0;
    while (lp != 0) { 
	j++; 
	if (ith == j) {
	    ret =  lp->item;  
	    break;
	}
	else {
	    lp = lp->next;  
	}
    }
    return ret;
}


// NOTE: USER RESPONSIBLE FOR MEMORY MANAGEMENT OF item
void* Stq::pop () {
    void* item;
    Stqitem* lp;
    if (head_ == 0) 
	return 0;
    item = head_->item;
    lp = head_;
    head_ = head_->next;
    if (head_ == 0) 
	tail_ = 0;
    if (free_ == 0) 
	free_ = lp;  
    else 
	delete lp;
    nitems_--;
    return item;
}

// NOTE: USER RESPONSIBLE FOR MEMORY MANAGEMENT OF item
void Stq::enq (void const* item) {
    Stqitem* lp;
    if (free_ == 0) 
	lp = new Stqitem();
    else { 
	lp = free_;  lp->item = lp->next = 0;  free_ = 0; 
    }
    lp->item = (void*)item;	// UNCONST
    if (tail_ == 0) 
	head_ = lp;
    else 
	tail_->next = lp;
    tail_ = lp;
    nitems_++;
}

void* Stq::cycle() const { 
    Stq* thislist  = (Stq*)this;
    void* item = thislist->pop(); thislist->enq(item); 
    return item; 
}

void Stq::insert (int ith, void const* item) {
    if(ith == 1) {
	push(item);
    }
    else if (ith == nitems_ + 1) {
	enq(item);
    }
    else {
	Stqitem* lp;
	if (free_ == 0) 
	    lp = new Stqitem();
	else { 
	    lp = free_;  lp->item = lp->next = 0;  free_ = 0; 
	}
	lp->item = (void*)item;	// UNCONST

	// get the item no. ith-1
	Stqitem* prev = head_;
	for (int i = 1; prev && i < ith-1; prev = prev->next, i++)
	    ;
	assume(prev != 0);
	lp->next = prev->next;
	prev->next = lp;
	nitems_++;
    }
}

void* Stq::remove (int ith) {
    assume (ith > 0 && ith <= nitems_);
    if (ith == 1) {
	return pop();
    }

    // get the item no. ith-1
    Stqitem* prev = head_;
    for (int i = 1; prev && i < ith-1; prev = prev->next, i++)
	;
    assume(prev != 0);
    Stqitem* lp = prev->next;
    void* item = lp->item;
    prev->next = lp->next;
    if (prev->next == 0)
	tail_ = prev;

    if (free_ == 0)
	free_ = lp;
    else
	delete lp;
    nitems_--;

    return item;
}

Stq* Stq::clone() const {
    Stq* buddy = new Stq();
    Stqitem* hp = head_;
    while (hp != 0) { 
	buddy->enq(hp->item);
	hp = hp->next;
    }
    return buddy;
}

// NOTE: ONLY Stqitems ARE DELETED...NOT items ADDED BY USERS
void Stq::clear() {
    for (int i=1;i<=nitems_;i++) 
	pop();
    if (free_ != 0) 
	delete free_;
}

/**************************************************************************/

Hashtable::Hashtable(int tablesize) {
    tablesize_ = tablesize;
    table_ = new Hashlink*[tablesize_];
    for (int i = 0; i < tablesize_; i++) 
	table_[i] = 0;
    nitems_ = 0;
}

Hashtable::~Hashtable() {  
    clear();  
}

void Hashtable::clear() {
    for (int i = 0; i < tablesize_; i++) {
	Hashlink* cp = table_[i];
	while (cp != 0) { 
	    Hashlink* lp = cp; 
	    cp = cp->next;  
	    // for debugging .... CHECK
	    lp->key = 0; lp->item = 0; lp->next = 0;
	    delete lp;
	}
	table_[i] = 0;
    }
    nitems_ = 0; 
}

int Hashtable::hash(char const* key) const {
    unsigned g;  unsigned h = 0;
    for (char const* s=key; *s != '\0'; s++) {
	h = (h << 4) + (toupper(*s));
	if ((g = h&0xf0000000)) { 
	    h = h ^ (g >> 24);  h = h ^ g; 
	}
    }
    return (h % tablesize_);
}

bool Hashtable::compare (char const* key1, char const* key2) const {
    bool isequal = false;
    char const* k1 = key1;  char const* k2 = key2;
    for (; toupper(*k1) == toupper(*k2); k1++, k2++) {
	if (*k1 == '\0' && *k2 == '\0') {
	    isequal = true;
	    break;
	}
    }
    return isequal;
}

void Hashtable::insert (char const* key, void const* item) {
    int result = hash (key);
    Hashlink* cp = new Hashlink(key, item);
    cp->next = table_[result];
    table_[result] = cp;
    nitems_++;
}

void* Hashtable::find (char const* key) const {
    void const* item = 0;
    int result = hash (key);
    for (Hashlink* cp = table_[result]; cp; cp = cp->next) {
	if (compare(key, cp->key)) {
	    item = cp->item;
	    break;
	}
    }
    return (void*)item;			// UNCONST
}

void* Hashtable::remove (char const* key) {
    int result = hash(key);
    Hashlink* cp = table_[result];  Hashlink* lp = 0;
    while (cp != 0) {
	if (compare(key, cp->key)) {
	    if (lp != 0) 
		lp->next = cp->next;
	    else 
		table_[result] = cp->next;
	    void const* item = cp->item;
	    // for debugging .... CHECK
	    cp->key = 0; cp->item = 0; cp->next = 0;
	    delete cp;
	    nitems_--;
	    return (void*)item;		// UNCONST
	}
	lp = cp;  cp = cp->next;
    }
    return 0;
}
void* Hashtable::get(int ith) const {
    assume (ith > 0);  assume (ith <= nitems_);
    int count = 0; 
    for (int i = 0; i <= tablesize_; i++) {
	if (table_[i] != 0) {
	    Hashlink* cp = table_[i];
	    while (cp != 0) { 
		count++;  
		if (count == ith) 
		    return (void*)cp->item; // UNCONST
		cp = cp->next;
	    } 
	}
    }
    assume (0);
    return 0;
}

Hashtable* Hashtable::clone (COPIER_FP clonerfunc) const {
    Hashtable* buddy = new Hashtable(tablesize_);
    for (int i=0; i<=tablesize_; i++) {
	if (table_[i] != 0) {
	    Hashlink* cp = table_[i];
	    while (cp != 0) { 
		buddy->insert(
		    cp->key, (clonerfunc) ? (*clonerfunc)(cp->item) : cp->item
		);
		cp = cp->next;
	    }
	}
    }
    return buddy;
}

/*******************************************************************************
*
*  OTHER UTILITIES
*
*******************************************************************************/

// REPLACE THE STRING replacethis WITH withthin IN inthisline
// PUT THE RESULTS IN results
// RETURN POSIITON IN results OF 1st CHAR WHERE THE SUBSTITUTION TOOK PLACE
// OR 0 IF THERE WAS NO SUBSTITUTION
// IF NO SUBSTITUTION TAKES PLACE results CONTAINS A COPY OF inthisline
// MATCH MUST BE EXACT..."Hi" DOES NOT MATCH "hi"
int strsub (
    const char* replacethis, const char* withthis, const char* inthisline, 
    char* results
) {
    int withlen,replacelen,subpos,resultspos,ret,stop;
    char const* k1; 
    char const* k2;

   strcpy (results, inthisline);
   replacelen = strlen (replacethis);  if (replacelen == 0) return 0;
   withlen = strlen (withthis);
   stop = strlen (inthisline) - replacelen;
   subpos = resultspos = ret = 0;
   while (1) {
      k1 = replacethis;  k2 = &inthisline[subpos];
      for (;*k1==*k2;k1++,k2++) if (*k1 == '\0') break;
      if (*k1 == '\0') {
         // MATCH...COPY INTO results AND CONTINUE
         // NOTE: k2 IS ONE PAST THE MATCH CHARS
         strncpy (&results[resultspos],withthis,withlen);
         ret = resultspos;
         resultspos += withlen-1;  subpos += replacelen-1;
         strcpy (&results[resultspos+1],&inthisline[subpos+1]);
      }
      subpos++;  resultspos++;  if (subpos > stop) break;
   }
   return ret;
}

//
// expandpath: expand a PATH by looking at embedded env variables.
//
// algorithm:
//   repeat until there are no '$token' or TILDE left in pathname
//       Start with ~/ or ~username?
//           yes - replace with user homedir
//       else extract $token and scan the environment and translate
//       found?
//           yes - replace $token with env value
//           no - complain and return with error code
//   done
//
// good features:
//    1. handles reference to references (ie., recursive expansion)
//
// bad features:
//    1. No checking for infinite recursion in expansion
//    

int expandpath(const char* orig_path, char path[]) {
    if (Globals::debug) {
	cerr << "expandpath: Expanding " << orig_path << " ... ";
    }
    char replace[1024];
    char tmppath[1024];
    strcpy(path, orig_path);
    strcpy(tmppath, orig_path);
    while(1) {
	//
	// see if tilde expansion required here.
	//
	if (path[0] == '~') {
	    char* start = &path[1];
	    char* end = start;
	    for ( ; 
		end && *end && *end != '/' && *end != ' ' && *end != '\t'; 
		++end
	    ) ;
	    if (end == start) {                 // hey, this is me!
		struct passwd* pw_entry = getpwuid(getuid());
		if (!pw_entry) {
		    cerr 
			<< endl
			<< "ERROR: Cannot get uid to expand ~ " 
			<< "found in path `" << orig_path << "'."
			<< endl;
		    return 1;
		}
		strcpy(tmppath, pw_entry->pw_dir);
		strcat(tmppath, &path[1]);
	    } else {				// ~username format.
		char username[1024];
		strncpy(username, start, end-start);
		username[end-start] = '\0';
		struct passwd* pw_entry = getpwnam(username);
		if (!pw_entry) {
		    cerr 
			<< endl
			<< "ERROR: No such user `" << username << "' "
			<< "found in path `" << orig_path << "'."
			<< endl;
		    return 1;
		}
		strcpy(tmppath, pw_entry->pw_dir);
		strcat(tmppath, end);
	    }
	    strcpy(path, tmppath);
	} else {
	    char* start = strchr(path, '$');
	    char* end = start;
	    for ( ; 
		end && *end && *end != '/' && *end != ' ' && *end != '\t'; 
		++end
	    ) ;
	    if (end == start)
		break;
	    strncpy(replace, start, end-start);
	    replace[end-start] = '\0';
	    char* variable = &replace[1];
	    const char* env_value;
	    if (!(env_value = getenv(variable))) {
		cerr << endl << "ERROR: environment variable `" << variable
		    << "' not found." << endl;
		return 1;
	    }
	    strsub(replace, env_value, tmppath, path);
	    strcpy(tmppath, path);
	}
    }
    if (Globals::debug) {
	cerr << path << endl;
    }
    return 0;
}

//
// find_include_file: look for the include file in the set of standard
// search directories supplied.
//

int find_include_file(
    const char* orig_path, const Stq* file_search_path, char path[]
) {
    if (Globals::debug) {
	cerr << "find_include_file: searching for " << orig_path << " ... ";
    }
    bool found = false;
    unsigned ndirs = file_search_path->size();
    while(ndirs--) {
	if (found)
	    continue;		// busy loop to finish cycle. work done.
	char buf[1024];
	const char* directory = (const char*)file_search_path->cycle();
	sprintf(buf, "%s/%s", directory, orig_path);
	if (fileexists(buf)) {
	    found = true;
	    strcpy(path, buf);
	}
    }
    return (found) ? 0 : 1;
}

//
// search_file: scans the filename and figures out if it should be 
// searched for in a set of standard places or not. If so, scan and
// return the full pathname where found. Uses find_include_file.
//
// return code:
//    0: AOK
//    1: couldn't find file anywhere
//    2: embedded env variable not found
//

int search_file(
    const char* orig_path, const Stq* file_search_path, char path[]
) {
    char pathname[1024];
    int error_code = 0;
    bool do_search = false;
    if (orig_path[0] == '<' && orig_path[strlen(orig_path)-1] == '>') {
	//
	// this is standard include directive, so set a flag
	// we scan later, except when the filename has a 
	// leading '/', in which case we don't scan.
	//
	const int namelen = strlen(orig_path) - 2;
	pathname[namelen] = '\0';
	strncpy(pathname, &orig_path[1], namelen);
	do_search = true;
    } else {
	strcpy(pathname, orig_path);
    }
    //
    // now expand embedded env vars.
    //
    if (expandpath(pathname, path)) {
	error_code = 2;
    } else {
	if (do_search && path[0] != '/') {
	    strcpy(pathname, path);
	    if (find_include_file(pathname, file_search_path, path)) {
		error_code = 1;
	    }
	} 
    }
    return error_code;
}

//
// search_program: scans the set of directories in prog_search_path to
// find the program. Make sure it's executable by the user.
//
// return code:
//    0: AOK
//    1: couldn't find program anywhere
//    2: embedded env variable not found
//    3: program not executable by user
//

int search_program(
    const char* orig_path, const Stq* program_search_path, char* path
) {
    int error_code = 0;
    char pathname[1024];
    strcpy(pathname, orig_path);
    //
    // now expand embedded env vars.
    //
    if (expandpath(pathname, path)) {
	error_code = 2;
    } else {
	bool do_search = (path[0] != '/');
	if (do_search) {
	    strcpy(pathname, path);
	    if (find_include_file(pathname, program_search_path, path)) {
		error_code = 1;
	    }
	} else {
	    if (!fileexists(path))
		error_code = 1;
	}
    }
    if (error_code == 0) {
	if (!fileexecutable(path))
	    error_code = 3;
    }

    return error_code;
}

// COPY INTO path A FILE NAME THAT IS NOT CURRENTLY BEING USED
// IF prefix IS NOT 0 USE IT AS A PREFIX
// IF create == 1 CREATE THE FILE (EMPTY)
void getunusedpath (char* path, char* prefix, bool create) {

    char buf[1024];
    char buf2[128];
    int  suffix = 0;
    // KEEP CONSTRUCTING FILES TILL YOU GET A UNIQUE ONE
    while (1) {
       strcpy  (buf,prefix);
       sprintf (buf2,"%d",suffix);  suffix++;
       strcat (buf,buf2);
       if (access(buf,F_OK)==-1) break;
    }
    strcpy (path,buf);
    if (create) { 
	int fp = creat(path,0777); 
	close(fp); 
    }
}

char* getprogramname (char* argv[]) {
// RETURN POINTER TO PROGRAM NAME
   
   return argv[0];
}
int getflag (char flag, int argc, char* argv[]) {
// RETURN 1 IF FLAG FOUND; OTHERWISE RETURN 0
   for (int i=0;i<argc;i++) 
      if (argv[i][0] == '-') 
         if (argv[i][1] == flag) return 1;
   return 0;
}
int getflagval (char flag, int argc, char* argv[], char* buf) {
// RETURN VALUE OF FLAG flag INTO buf
// RETURN 1 IF FLAG FOUND; OTHERWISE RETURN 0
   strcpy (buf,"");
   for (int i=0;i<argc;i++) 
      if (argv[i][0] == '-') if (argv[i][1] == flag) if (i+1 < argc)
         { strcpy (buf,argv[i+1]); return 1; }
   return 0;
}
void promptforval (const char* prompt, char* buf) {
// PROMPT USER FOR INPUT (prompt) AND PUT RESULT IN buf
// strlen(buf) == 0 MEANS THERE WAS NO INPUT
   strcpy (buf,"");
   printf ("%s",prompt);
   int pos = 0;
   while (1) { 
      int ch = getchar();
      if (ch == EOF) break;
      if (ch == 13) break;
      if (ch == 10) break;
      buf[pos++] = ch;
    }
   buf[pos] = '\0';
}

// RETURN true IF FILE EXISTS, OTHERWISE RETURNS false
bool fileexists (const char* path) {
    if (path == 0) 
	return false;

    struct stat sb;
    if (stat(path,&sb) < 0) 
	return false;

    // make sure it's not a directory!
    bool regular_file = ((sb.st_mode & S_IFMT) == S_IFREG);  
    if (!regular_file) 
	return false;
    return true;
}

bool fileexecutable(const char* path) {
   return (access(path, X_OK) != -1);
}

char* strcpy_with_alloc (char const* str) {
    char* dup = 0;
    if (str) {
	char* copy = new char[::strlen(str) + 1];
	::strcpy (copy, str);
	dup = copy;
    }
    return dup;
}

void assume_fail (const char* x, const char* file, int line)
{
    fprintf (stderr, "The assertion '%s' failed: FILE=%s LINE=%d.\n",x,file,line);
    assert (0);
}
