//
// freevars.cc: Free variable (loops, sets) definition.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#include <cassert>
#include <cstdio>
#include <cstdlib>
#include <cstring>

#ifdef __CYGWIN32__
# include <ctime>
#else
# include <sys/types.h>			// for clock_t
# include <sys/time.h>			//   and time() for random seed
#endif

#include "utils.h"
#include "value.h"
#include "variable.h"
#include "freevars.h"

using namespace std;

//
// Thanks a lot, SunOS, for forcing me to do this under GCC.
//
extern "C" double drand48();
extern "C" void srand48(long);

/***************************************************************************/


FreeVariableGroup::FreeVariableGroup() {
    group_ = new Stq;
}

FreeVariableGroup::~FreeVariableGroup() {
    unsigned groupsize = group_->size();
    while(groupsize--) {
	FreeVariable* fv = (FreeVariable*) group_->pop();
	delete fv; fv = 0;
    }
    delete group_; group_ = 0;
}

void FreeVariableGroup::add(const FreeVariable* fv) {
    bool exists = false;
    unsigned groupsize = group_->size();
    while(groupsize--) {
	const FreeVariable* member = (FreeVariable*) group_->cycle();
	if(fv == member) {
	    exists = true;
	}
    }
    if(!exists) {
	group_->enq(fv);
    }
}

void FreeVariableGroup::remove(const FreeVariable* fv) {
    unsigned groupsize = group_->size();
    while(groupsize--) {
	const FreeVariable* member = (FreeVariable*) group_->pop();
	if(fv != member) {
	    group_->enq(member);
	}
    }
}

int FreeVariableGroup::getnruns() const {
    int nruns = 0;
    unsigned groupsize = group_->size();
    while(groupsize--) {
	const FreeVariable* member = (FreeVariable*) group_->cycle();
	int thisrun = member->getnruns();
	if (thisrun > nruns)
	    nruns = thisrun;
    }
    return nruns;
}

bool FreeVariableGroup::done() const {
    bool alldone = true;
    unsigned groupsize = group_->size();
    while(groupsize--) {
	const FreeVariable* member = (FreeVariable*) group_->cycle();
	alldone = alldone && member->done();
    }
    return alldone;
}

void FreeVariableGroup::increment() {
    unsigned groupsize = group_->size();
    while(groupsize--) {
	FreeVariable* member = (FreeVariable*) group_->cycle();
	if(!member->done())
	    member->increment();
    }
}

void FreeVariableGroup::reset() {
    unsigned groupsize = group_->size();
    while(groupsize--) {
	FreeVariable* member = (FreeVariable*) group_->cycle();
	member->reset();
    }
}

/***************************************************************************/

FreeVariable::FreeVariable(const char* tool, const char* var) {
    tool_ = strcpy_with_alloc(tool);
    var_  = strcpy_with_alloc(var);
}

FreeVariable::~FreeVariable() {
    delete[] tool_; tool_ = 0;
    delete[] var_; var_ = 0;
}

const char* FreeVariable::gettool() const { return tool_; }
const char* FreeVariable::getvar() const { return var_; }

const Value* FreeVariable::getcurrent() const { return 0; }
bool FreeVariable::done() const { return true; }
void FreeVariable::increment() { return; }
void FreeVariable::reset() { return; }

int FreeVariable::getnruns() const { return 0; }

FreeVariable::FreeVariableType FreeVariable::type() const { return unknown; }

/***************************************************************************/

LoopVariable::LoopVariable(const char* tool, const char* var, 
    Value* start, Value* stop, Value* increment
) : FreeVariable(tool, var) {
    start_ = start;
    stop_ = stop;
    increment_ = increment;
    current_ = start_->clone();
    done_ = false;
}

LoopVariable::~LoopVariable() {
    delete start_; start_ = 0;
    delete stop_; stop_ = 0;
    delete increment_; increment_ = 0;
    delete current_; current_ = 0;
}

int LoopVariable::getnruns() const { 
    Value* tmpval = stop_->clone();
    tmpval->add(increment_);
    tmpval->subtract(start_);
    tmpval->div(increment_);

    int nruns = -1;
    if (tmpval->isainteger())
	nruns = tmpval->getinteger();
    else if (tmpval->isareal())
	nruns = (int)tmpval->getreal();
    
    delete tmpval;
    return nruns;
}

const Value* LoopVariable::getcurrent() const { return current_; }
bool LoopVariable::done() const { return done_; }
void LoopVariable::increment() { 
    Value* newval = current_->clone(); 
    newval->add(increment_);

    //
    // take care of increment vs decrement
    // FIX: CHECK: add isle and islt and isge 
    //
    if (!start_->isgt(stop_) && newval->isgt(stop_)) {
	done_ = true;
	delete newval;
    }
    else if (start_->isgt(stop_) && 
	(!newval->isgt(stop_) && !newval->isequal(stop_))
    ) {
	 done_ = true;
	 delete newval;
    }
    else {
	delete current_;
	current_ = newval;
    }
}

void LoopVariable::reset() { 
    done_ = false; 
    delete current_; 
    current_ = start_->clone();
}

FreeVariable::FreeVariableType LoopVariable::type() const { return loop; }

/***************************************************************************/

SetVariable::SetVariable(
    const char* tool, const char* var, Stq* values
) : FreeVariable(tool, var) {
    values_ = values;
    counter_ = 1;
    done_ = false;
}

SetVariable::~SetVariable() {
    unsigned nvalues = values_->size();
    while(nvalues--) {
	Value* value = (Value*)values_->pop();
	delete value; value = 0;
    }
    delete values_; values_ = 0;
    counter_ = 1;
}

int SetVariable::getnruns() const { return values_->size(); }
const Value* SetVariable::getcurrent() const { 
    return (const Value*)values_->get(counter_); 
}
bool SetVariable::done() const { return done_; }
void SetVariable::increment() { 
    ++counter_;
    if (counter_ > values_->size()) {
	counter_ = values_->size();
	done_ = true;
    }
}

void SetVariable::reset() { done_ = false; counter_ = 1; }

FreeVariable::FreeVariableType SetVariable::type() const { return set; }

/**************************************************************************/

extern "C" {
static int compare_values(const void* uu, const void* vv) {
#if CXX_NO_STATIC_CAST
    const Value* u = *(Value**)(uu);
    const Value* v = *(Value**)(vv);
#else
    const Value* u = *reinterpret_cast<Value* const*>(uu);
    const Value* v = *reinterpret_cast<Value* const*>(vv);
#endif
    return u->isgt(v) ? 1 : -1;
}
}

/**************************************************************************/

RandomVariable::RandomVariable(const char* tool, const char* var, 
    long seed, Value* start, Value* stop, int nvalues, bool sorted
) : FreeVariable(tool, var) {
    start_ = start;
    stop_ = stop;
    nvalues_ = nvalues;
    counter_ = 0;
    done_ = false;
    sorted_ = sorted;

    // get seed from clock and set seed
    seed_ = seed;
    srand48(seed_);
    //
    // now pre-compute the random values. Avoid having to have multiple
    // independent streams.
    //
    values_ = new Value*[nvalues];
    for(int i = 0; i < nvalues; ++i) {
	values_[i] = create_random();
    }
    if (sorted) { 
	qsort(values_, nvalues_, sizeof values_[0], compare_values);
    }
}

RandomVariable::RandomVariable(const char* tool, const char* var, 
    Value* start, Value* stop, int nvalues, bool sorted
) : FreeVariable(tool, var) {
    start_ = start;
    stop_ = stop;
    nvalues_ = nvalues;
    counter_ = 0;
    done_ = false;

    // get seed from clock and set seed
    clock_t clockval = time(0);
#if CXX_NO_STATIC_CAST
    seed_ = (long)clockval;
#else
    seed_ = static_cast<long>(clockval);
#endif
    srand48(seed_);
    //
    // now pre-compute the random values. Avoid having to have multiple
    // independent streams.
    //
    values_ = new Value*[nvalues];
    for(int i = 0; i < nvalues; ++i) {
	values_[i] = create_random();
    }
    if (sorted) { 
	qsort(values_, nvalues_, sizeof values_[0], compare_values);
    }
}

RandomVariable::~RandomVariable() {
    delete start_; start_ = 0;
    delete stop_; stop_ = 0;
    for(int i = 0; i < nvalues_; ++i) {
	delete values_[i];
    }
    delete[] values_;
}

int RandomVariable::getnruns() const { 
    return nvalues_;
}

const Value* RandomVariable::getcurrent() const { return values_[counter_]; }

bool RandomVariable::done() const { return done_; }

Value* RandomVariable::create_random() const {
    Value* newval = stop_->clone();
    newval->subtract(start_);
    char buf[1024];
    sprintf(buf, "%.17g", drand48());
    Value* randomval = newval->clone(buf);
    newval->mul(randomval);
    delete randomval;
    newval->add(start_);
    return newval;
}

void RandomVariable::increment() {
    ++counter_;
    if (counter_ >= nvalues_) {
	counter_ = nvalues_-1;		// array index 0 ... nvalues_-1;
	done_ = true;
    } 
}

void RandomVariable::reset() { 
    done_ = false; 
    counter_ = 0;
}

FreeVariable::FreeVariableType RandomVariable::type() const { return random; }

/***************************************************************************/
