#if defined(_WIN32) || defined(USE_C_IO)

/*
 * Source: src/lib/rbeam_cygwin32.c
 */

#include <stdio.h>
#include <string.h>
#include <dim.h>

#ifdef _MSC_VER
# define F77_NAME(name) __stdcall name
#else
# if !NO_FORT_UNDERSCORE
#  define F77_NAME(name) name ## _
# else
#  define F77_NAME(name) name
# endif
#endif

#ifdef _MSC_VER
# define RBEAM F77_NAME(RBEAM)
#else
# define RBEAM F77_NAME(rbeam)
#endif

/*
C+++
C	SUBROUTINE	RBEAM	(FNAME,RAY,PHASE,AP,NCOL,NPOINT,IFLAG,IERR)
C
C	purpose		reads in a BEAM file from SHADOW and returns it
C			to the calling program. This is an intermediate
C			subroutine that will be easily modified to suit
C			the format of SHADOW.
C
C	Input	:	FNAME	(character*(*))	Input file name
C	
C	Output  :	RAY	Array which contains the basic 12 columns that
C				defined each ray
C			PHASE	Array which contains the OPD and the 2 phase 
C				angles of the A vector
C			AP	Array which contains the Ap vector for each ray
C			NCOL	Number of columns for each ray 
C			NPOINT	Number of rays
C			IFLAG	A flag which is stored on the input file
C			IERR = 0, normal return
C			     = 1, error opening file
C			     = 2, error reading file
C
C---
	SUBROUTINE	RBEAM 	(FNAME,RAY,PHASE,AP,NCOL,NPOINT,IFLAG,IERR)
     	CHARACTER*(*)	FNAME
     	REAL*8		RAY(12, N_DIM), PHASE(3,N_DIM), AP(3,N_DIM)
*/

void RBEAM (
    const char* fort_fname, 		/* filename to write to */
    double ray[N_DIM][12],
    double phase[N_DIM][3],
    double ap[N_DIM][3],
    int *ncol, int *npoint, int *iflag, int *ierr,
    unsigned long fname_len		/* thank you FORTRAN! */
) {
    extern void leave_();
    extern void mssg_();
    FILE *fp;
    int i, j;

    int io_dummy[5];
    int reclen;
    char fname[256];

    /* fix fortran filename */
    for(i = fname_len; fort_fname[i-1] == ' '; --i)
	;
    strncpy(fname, fort_fname, i);
    fname[i] = '\0';

    *ierr = 0;

    fp = fopen(fname, "rb");
    if (!fp) {
	*ierr = 1;
        mssg_ ("RBEAM:: Error opening ", fort_fname, ierr,
            strlen("RBEAM:: Error opening "), i
	);
	return;
    }
    if (fread(io_dummy, sizeof(int), 5, fp) != 5) {
        *ierr = 2;
	fclose(fp);
	return;
    }
    /*
     * io_dummy[0] and io_dummy[4] are the F77 record lengths, so ignore.
     */
    *ncol = io_dummy[1];
    *npoint = io_dummy[2];
    *iflag = io_dummy[3];

    if (*ncol != 12 && *ncol != 13 && *ncol != 18) {
	mssg_("RBEAM", "Invalid number of columns.", *ncol,
	    strlen("RBEAM"), strlen("Invalid number of columns.")
	);
    }
    reclen = *ncol * sizeof(double);
    for (i = 0; i < *npoint; ++i) {
	fread(&reclen, sizeof(int), 1, fp);
	for (j = 0; j < 12; ++j) {
	    fread(&ray[i][j], sizeof(double), 1, fp);
	}
	if (*ncol == 13) {
	    fread(&phase[i][1], sizeof(double), 1, fp);
	} else if (*ncol == 18) {
	    for (j = 0; j < 3; ++j) {
		fread(&phase[i][j], sizeof(double), 1, fp);
	    }
	    for (j = 0; j < 3; ++j) {
		fread(&ap[i][j], sizeof(double), 1, fp);
	    }
	}
	fread(&reclen, sizeof(int), 1, fp);
    }
    fclose(fp);
    return;
}

#else /* _WIN32 || USE_C_IO */

static int rbeam_c_dummy;

#endif
