//
// report.h: automated reporting facility
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//


#ifndef excon_batch_report_h
#define excon_batch_report_h

/**************************************************************************/

class Stq;
class Hashtable;
class ToolInstance;
class Value;

/**************************************************************************/

/****************************************************************************
 *
 *  REPORTERS
 *
 *  class Reporter: Base class for various types of reporters in the 
 *  EXCON report class.
 *
 *  All subclasses must implement the reportval() method, which is 
 *  called by the container class Report to report values.
 *
 ****************************************************************************/

class Reporter {
public:
    Reporter(const char* tname, const char* varname);
    virtual ~Reporter();
    const char* gettoolname() const;
    const char* getvarname() const;
    int getruntime() const;

    virtual Reporter* clone() const = 0;
    virtual const Value* reportval(int /*iteration*/) const = 0;
protected:
    char* tname_;		// TOOL INSTANCE NAME ... DO NOT STORE POINTER
    char* varname_;		// DO NOT STORE Var*...MAY CHANGE DURING RUN
    int runtime_;
};

inline const char* Reporter::gettoolname() const { return tname_; }
inline const char* Reporter::getvarname() const { return varname_; }
inline int Reporter::getruntime() const { return runtime_; }

/*************************************************************************/

//
// class IVReporter:
// This is the Reporter that returns values given by "$GETVAR" method.
// Reports its value simply by finding the variables current value
//
class IVReporter: public Reporter {
public:
    IVReporter(const char* tname, const char* varname);
    ~IVReporter();
    virtual Reporter* clone() const;
    virtual const Value* reportval(int iteration) const;
};

/**************************************************************************/

//
// class RUNReporter:
// This is the Reporter that returns values given by "$RUN" method.
// Reports its value by running a program (cf: $RUN arg) and reports
// the output from stdout (and stderr).
//
class RUNReporter : public Reporter {
public:
    RUNReporter(const char* tname, const char* varname);
    virtual ~RUNReporter();
    virtual Reporter* clone() const;
    virtual const Value* reportval(int iteration) const;
private:
    void genexec(int iteration);

private:
    char* program_;		// Name of program to run. First token in text.
    char* program_args_;	// Arguments to the program to run.
    char* program_text_;	// Expanded program text ... runnable now.
    Value* val_;		// RETURN VALUE
    Value* errorfopen_;		// ERROR VALUE IF fopen FAILS
    Value* errorfork_;		// ERROR VALUE IF fork FAILS
    Value* errorgetres_;	// ERROR VALUE IF COULD NOT RETRIEVE RESULT
};

/****************************************************************************
 *
 *  REPORTS
 *
 *  class Report: Report contains all the reporters and when asked to 
 *  produce the rows in the report (one per iteration), calls on each
 *  of the reporters to produce its value(s).
 *
 ****************************************************************************/

class Report {
public:
    Report(const char* name, const char* path, const char* sep = " ");
    ~Report();

    Report* clone() const;

    const char* getname() const;
    void setpath(const char* path);
    const char* getpath() const;
    void separator(const char* sep);
    void enqheader(const char* header);
    void enqreporter(const Reporter* reporter);
    void collectresults(int iteration);
    void printheader();
    void printrow();
    void reset();
    int getruntime() const;

    const Stq* get_reporters() const;

private:
    void errorreportfile(const char* file);

private:
    char* name_;		// REPORT NAME (e.g. "table1")
    char* path_;		// FILE WHERE REPORT IS TO BE STORED
    char* separator_;		// SEP string for report columns (eg., " ")
    Stq* headers_;		// Header items.
    Stq* reporters_;		// LIST OF Reporter*...1 FOR EACH COL OF REPORT
    Stq* row_;			// ROW OF RESULTS (Value*)
    int runtime_;		// ESTIMATE COST OF COLLECTING REPORT (seconds)
};


inline const char* Report::getname() const { return name_; }
inline const char* Report::getpath() const { return path_; }
inline int Report::getruntime() const { return runtime_; }

inline const Stq* Report::get_reporters() const { return reporters_; }

/**************************************************************************/

#endif/*excon_batch_report_h*/
