/*
 * nml-compare.c: compare parameter value(s) in SHADOW namelist files
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * ------------------------------------------------
 *              SHADOW
 *    Center for X-ray Lithography
 *   University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 */

/*
 * Compare two SHADOW namelist files and return the result. The
 * requirement is that both namelist files MUST be of the same type
 * and written by compatible versions of SHADOW. If the namelist item
 * sizes are not the same, then nml-compare returns -1, but you can use
 * the "force" -f option to check as many as it can.
 *
 * Return codes:
 *    -1: Incompatible versions? Unknown reasons?
 *     0: AOK
 *     n: Where n is the number of differences.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "common.h"
#include "namelist.h"

static int parse_args (int argc, char** argv, const char* prog_name, 
    char** nml_file1, char** nml_file2, NmlType* type, boolean* force,
    double* eps
);
static void usage(const char* prog_name);
static void dump_namelist (const Namelist* namelist);
static int do_compare (
    const Namelist* namelist1, const Namelist* namelist2, boolean force,
    double eps
);

static boolean g_verbose = 0;

int main (int argc, char** argv) {
    char* nml_file1 = nil;		/* the namelist files to compare */
    char* nml_file2 = nil;
    NmlType type;			/* type of namelist (OE/SOURCE) */
    Namelist* namelist1 = nil;		/* and the namelists read from */
    Namelist* namelist2 = nil;

    double eps = 1.0e-15;

    char* prog_name = nil;
    boolean force = FALSE;
    int status = 0;

    char *ptr = strrchr(argv[0], '/');

    if (ptr) 
	prog_name = ptr + 1;
    else 
	prog_name = argv[0];

    parse_args(
        argc, argv, prog_name, &nml_file1, &nml_file2, &type, &force, &eps
    );
 
    namelist1 = read_namelist(nml_file1, type);
    if (namelist1 == nil) {
	fprintf(stderr, "ERROR: Cannot read namelist file `%s'\n", nml_file1);
	exit(1);
    }
    namelist1 = clone_namelist(namelist1);

    namelist2 = read_namelist(nml_file2, type);
    if (namelist2 == nil) {
	fprintf(stderr, "ERROR: Cannot read namelist file `%s'\n", nml_file2);
	exit(1);
    }

    status = do_compare(namelist1, namelist2, force, eps);
    switch (status) {
        case -1:
	    fprintf(stdout, "Namelist size mismatch. Use -f to force\n");
	    break;
	case 0:
	    fprintf(stdout, "Compares ok. Use -v to see each entry\n");
	default:
	    fprintf(stdout, "** %d differences\n", status);
	    break;
    }
    exit(status);
}

static int do_compare (
    const Namelist* namelist1, const Namelist* namelist2, boolean force,
    double eps
) {
    int i = 0;
    unsigned size = 0;
    int differences = 0;

    if (namelist1 == nil || namelist2 == nil) {
	return -1;
    }

    size = namelist1->size;
    if (namelist1->size != namelist2->size) {
        if (!force)
	    return -1;
	else {
	    size = (namelist1->size < namelist2->size) ?
	        namelist1->size : namelist2->size;
	}
    }
    
    for (i = 0; i < size; i++) {
	boolean same = FALSE;
	const NamelistData* data1 = &namelist1->nml_data[i];
	const NamelistData* data2 = &namelist2->nml_data[i];

	if (g_verbose) {
	    fprintf (stdout, "comparing %s ... ", data1->name);
	}

	if (data1->data_type == data2->data_type) {
	    double dbldiff;
	    switch (data1->data_type) {
		case DT_int:
		    same = (*(int*)data1->value == *(int*)data2->value);
		    break;

		case DT_flt:
		    dbldiff = *(float*)data1->value - *(float*)data2->value;
		    if (dbldiff < 0.0)
		        dbldiff *= -1.0;
		    same = (dbldiff <= eps);
		    break;

		case DT_dbl:
		    dbldiff = *(double*)data1->value - *(double*)data2->value;
		    if (dbldiff < 0.0)
		        dbldiff *= -1.0;
		    same = (dbldiff <= eps);
		    break;

		case DT_str:
		    same = (strncmp(data1->value,data2->value,FNAME_LEN) == 0);
		    break;
		
		default:
		    break;
	    }
	}
	if (g_verbose) {
	    fprintf(stdout, "%s\n", (same) ? "ok" : "nope");
	}
	if (!same) {
	    ++differences;
	    fprintf (stdout, "********\n%s\n", data1->name);
	    fprintf (stdout, "< %s\n", sprint_value(data1));
	    fprintf (stdout, "> %s\n********\n", sprint_value(data2));
	}
    }
    /* dump_namelist(namelist1); */
    return differences;
}

int parse_args (int argc, char** argv, const char* prog_name, 
    char** nml_file1, char** nml_file2, NmlType* type, boolean* force,
    double* eps
) {
    int arg_error = 0;
    extern int optind;
    extern char *optarg;

    boolean type_supplied = FALSE;
    int c;

    *nml_file1 = nil;
    *nml_file2 = nil;

    while ((c = getopt (argc, argv, "hfve:t:")) != EOF) {
	switch (c) {
	    case 't':
		if (optarg[0] == 's' || optarg[0] == 'S') {
		    *type = NML_source;
		}
		else if (optarg[0] == 'o' || optarg[0] == 'O') {
		    *type = NML_oe;
		}
		else {
		    fprintf(stderr, "Illegal namelist type --  %s\n", 
			optarg
		    );
		    usage(prog_name);
		    exit(1);
		}
		type_supplied = TRUE;
		break;

	    case 'v':
	        g_verbose = 1;
		break;

	    case 'f':
	        *force = TRUE;
		break;

	    case 'e':
		if (sscanf(optarg, "%lg", eps) != 1) {
		    fprintf(stderr, "Bad EPS value %s for -e EPS\n", optarg);
		    ++arg_error;
		}
		fprintf(stderr, "EPS value %-.15g\n", *eps);
		break;

	    case 'h':
		usage (prog_name);
		exit (0);
		break;
	    
	    default:
		++arg_error;
		break;
	}
	if (arg_error)
	    break;
    }

    if (type_supplied == FALSE) {
	fprintf(stderr, "Must supply namelist type (SOURCE or OE).\n");
	++arg_error;
    }

    if (argc != optind + 2) {
	fprintf(stderr, "Must supply names of the two files to compare.\n");
	++arg_error;
    }

    if (arg_error) {
	usage(prog_name);
	exit(arg_error);
    }

    *nml_file1 = strcpy(malloc(strlen(argv[optind])+1), argv[optind]);
    *nml_file2 = strcpy(malloc(strlen(argv[optind+1])+1), argv[optind+1]);

    if (g_verbose) {
        fprintf(stdout, "File 1 : %s\n", *nml_file1);
        fprintf(stdout, "File 2 : %s\n", *nml_file2);
        fprintf(stdout, "Using DOUBLE_EPSILON: %-.15g\n", *eps);
    }

    return 0;
}


static void usage (const char* prog_name) {
    fprintf (stderr, "\n"
    "Usage:%s [-hf] [-e EPSILON] -t NamelistType Namelist_File1 Namelist_File2\n\n\
    -h:			print this info\n\
    -f:			Force comparison even if namelists don't match\n\
    -e:			Supply EPS for comparing double precision numbers\n\
    -t NamelistType:	\"source\"/\"oe\" (also accepts \"s\"/\"o\")\n\n\
    Namelist_File1,2:	The two files to compare.\n\n",
    prog_name);

    fprintf (stderr, 
    "Examples:\n\
    %s -t source start.00 start.00.new\n\n",
    prog_name);

    return;
}

static void dump_namelist (const Namelist* namelist) {
    int i = 0;
    unsigned size = 0;

    FILE* fp = fopen("junk.dump", "w");
    if(fp == nil || namelist == nil) {
	return;
    }
    
    size = namelist->size;
    for (i = 0; i < size; i++) {
	const NamelistData* data = &namelist->nml_data[i];
	fprintf (fp, "%s\t=\t%s\n", data->name, sprint_value(data));
    }
    fclose(fp);
}
