//
// exp2gfile.cc: extract the gfile corresponding to a/set-of TOOL/s
//
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1996 Mumit Khan
//
//

// ====================================================================== //

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <pwd.h>		// for user info
#include <sys/types.h>		// for uid_t, pid_t etc

#include <fstream.h>
#include <string>
#include <list>

#include <getopt.h>		// better pick up GNU getopt!
#include <libcxrl.h>

#include "exp.h"
#include "global.h"
#include "logger.h"
#include "job.h"
#include "misc.h"
#include "pmu.h"
#include "tool.h"
#include "utils.h"
#include "value.h"
#include "variable.h"

// ====================================================================== //

class Exp2GFILEMgr : public ExpMgr {
public:
    Exp2GFILEMgr() { }
    virtual ~Exp2GFILEMgr() { }
    virtual const Exp* exp() const { return exp_; }
    virtual Exp* exp() { return exp_; }
};

// ====================================================================== //
//
// globals.
//

//
// names of the logfile EXCON uses. Defaults to "exp2gfile.log";
//
static string logfile = "exp2gfile.log";

//
// the trace string on command line line.
//
static string trace_str;

//
// the search path given on command line.
//
static string search_path;

//
// debug flag
//
static bool g_verbose = false;

//
// default flags.
//
static unsigned default_traces = Logger::minimal | Logger::load;


// ====================================================================== //

//
// command line options and such
//
struct option long_options[] = {
    { "gfile",		required_argument,	0, 'g' },
    { "search-path",	required_argument,	0, 'I' },
    { "expfile",	required_argument,	0, 'e' },
    { "toolsfile",	required_argument,	0, 't' },
#if NOT_IMPLEMENTED
    { "export-all",	optional_argument,	0, 'a' },
    { "export",		required_argument,	0, 'T' },
#endif
    { "logfile",	required_argument,	0, 'l' },
    { "traces",		required_argument,	0, 'T' },
    { "debug",		optional_argument,	0, 'd' },
    { "verbose",	optional_argument,	0, 'v' },
    { "help",		no_argument,		0, 'h' },
    { "more-help",	no_argument,		0, 129 },
    { "print-gfile",	no_argument,		0, 130 },
    { 0,		0,			0,  0  }
};

static void usage(ostream& os, bool die = false,  int diecode = 0) {
    os << "\n"
    "exp2gfile: Create g-files for the Tools in the EXP file\n\n"
    "Usage: exp2gfile [OPTIONS]\n\n"
    "Most options have reasonable defaults or are prompted for\n"
    "The following OPTIONS are recognized (in either short or long form):\n\n" 
    "  -g FILE, --gfile=FILE        Read g-parameters from FILE\n"
    "  -I PATH, --search-path=PATH  Use PATH (eg., .:tools) to search\n"
    "  -t FILE, --tool=FILE         Read TOOLS from FILE\n"
    "  -e FILE, --exp=FILE          Read EXPERIMENT from FILE\n"
#if NOT_IMPLEMENTED
    "  -a,      --export-all        Export all the Tools in EXP (default)\n"
    "  -T TOOLS,--export=TOOLS      Export only the named Tools in EXP\n"
#endif
    "  -d,      --debug             Turn on debugging. Not pretty!\n"
    "  -v,      --verbose           Turn on verbose mode.\n"
    "  -h,      --help              Print this info\n"
    "           --more-help         Print more help info\n"
    "           --print-gfile       Show example gfile parameters\n"
    "\n\n"
    "Example:\n"
    "  % exp2gfile -t excon.tools -e excon.exp\n"
    "\n";

    if (die)
	exit(diecode);
}

static void more_usage(ostream& os, bool die = false,  int diecode = 0) {
    os << "\n"
    "exp2gfile: Create g-files for the Tools in the EXP file\n\n"
    "Usage: exp2gfile [OPTIONS]\n\n"
    "Most options have reasonable defaults or are prompted for\n"
    "The following OPTIONS are recognized (in either short or long form):\n"
    "(you can use unique substrings for the long form; eg., you could\n"
    "say --out instead of --output since that's unique)\n\n"
    "  -g FILE, --gfile=FILE\n"
    "   You can supply write-mm all the parameters via \"gfile\" just like\n"
    "   all the other Toolset programs. Command line parameters override\n"
    "   the values in FILE, so you can use these as default values.\n"
    "   Use --print-gfile to see an example g-file.\n"
    "\n"
    "  -I PATH, --search-path=PATH\n"
    "   PATH is a colon- or comma- separated list of directories that EXCON\n"
    "   searches for TOOLS and EXPERIMENT files, when these files are named\n"
    "   within < and > (eg., <excon.tools>). Multiple -I flags are simply\n"
    "   concatenated as they are seen.\n"
    "   An example of PATH would be /usr/local/excon/tools:toolsdir\n"
    "\n"
    "  -t FILE, --tool=FILE\n"
    "   TOOLS file contains EXCON tool definitions. Usually users use the\n"
    "   Toolset provided one in $TOOLSET_DATA directory. Specify the FILE\n"
    "   within < and > to force EXCON to search for it.\n"
    "   Typically named ``excon.tools''\n"
    "\n"
    "  -e FILE, --exp=FILE\n"
    "   EXPRIMENT file contains user experiment sequence and also parameters\n"
    "   for the models and tools. Specify the FILE within < and > to force\n"
    "   EXCON to search for it.\n"
    "   Typically named ``excon.exp''\n"
    "\n"
#if NOT_IMPLEMENTED
    "  -a,      --export-all\n"
    "   Export all the Tools used in experiment file as gfiles. This is the\n"
    "   default behaviour currently.\n"
    "\n"
    "  -T TOOLS,--export=TOOLS\n"
    "   Export only TOOLS (comma or colon separated list of Tool names) in\n"
    "   the experiment file. You can also use --export-all to export all the\n"
    "   Tools, which is the default behaviour. Ignored if --export-all is\n"
    "   true.\n"
    "\n"
#endif
    "  -d,      --debug\n"
    "   Print out gobs of information about EXCONs run. Not pretty. This is\n"
    "   equivalent to seting all the internal traces, ie, --traces=all\n"
    "\n"
    "  -v,      --verbose           Turn on verbose mode.\n"
    "  -h,      --help              Print this info\n"
    "           --more-help         Print more help info\n"
    "   Print a bit more elaborate command line help information\n"
    "\n"
    "           --print-gfile       Show example gfile parameters\n"
    "\n"
    "Examples:\n"
    "  % exp2gfile -t excon.tools -e excon.exp\n"
    "\n";

    if (die)
	exit(diecode);
}

static void print_gfile(ostream& os, bool die = false,  int diecode = 0) {
    os << "\n"
    "###\n"
    "#\n"
    "# Example Exp2GFILE gfile\n"
    "#\n"
    "###\n"
    "#\n"
    "# verbose mode (y/n).\n"
    "#\n"
    "verbose = n\n"
    "#\n"
    "# EXCON file search path for INCLUDE/TOOLS/EXP files. This is a\n"
    "# colon separated list of directories that EXCON searches when\n"
    "# sees a filename of the form <FILE>, where FILE is the filename\n"
    "# to open. EXCON always searches current directory and then\n"
    "# TOOLSET_DATA directory at the end.\n"
    "#\n"
    "search_path = ~/excon/tools:~khan/excon/tools\n"
    "#\n"
    "# EXCON Tools file. This can in turn INCLUDE others. If you enter\n"
    "# the file name as <FILE> (eg., <excon.tools>), EXCON will search\n"
    "# the directories in the search path for excon.tools file.\n"
    "#\n"
    "toolsfile = <excon.tools>\n"
    "#\n"
    "# EXCON Experiment file. This can in turn INCLUDE others. If you enter\n"
    "# the file name as <FILE> (eg., <excon.exp>), EXCON will search\n"
    "# the directories in the search path for excon.exp file.\n"
    "#\n"
    "expfile = excon.exp\n"
#if NOT_IMPLEMENTED
    "#\n"
    "# Export all the tools used in the experiment file (y/n). Default\n"
    "# is y.\n"
    "#\n"
    "export_all = y\n"
    "#\n"
    "# Export which tools in the experiment sequence? The list of TOOLS is\n"
    "# a comma or colon separated list (eg., xprocess,cutline1). You can\n"
    "# also export all the TOOLS using --export-all next, which is the\n"
    "# default behaviour. This option is ignored if export_all is true.\n"
    "#\n"
    "# tools = xprocess,cutline1\n"
#endif
    "\n";

    if (die)
	exit(diecode);
}

static int read_gfile(const string& gfile,
    string& toolsfile, string& expfile, bool& /* export_all */,
    string& /* tool_list */
) {
    //
    // now read the gfile is supplied before the rest of the command
    // is parsed, so we can override all of it via cmdline.
    //
    g_file(const_cast<char*>(gfile.c_str()));
    char buf[1024];
    if (g_peek("verbose")) {
	g_string("verbose", "Verbose mode (y/n)?", buf);
	if (buf[0] == 'y' || buf[0] == 'Y' || buf[0] == '1')
	    g_verbose = true;
    }
    if (g_peek("search_path")) {
	g_string("search_path", "Enter EXCON file search path ", buf);
	search_path = buf;
    }
    if (g_peek("toolsfile")) {
	g_string("toolsfile", "Enter EXCON TOOLS file", buf);
	toolsfile = buf;
    }
    if (g_peek("expfile")) {
	g_string("expfile", "Enter EXCON experiment file", buf);
	expfile = buf;
    }
#if NOT_IMPLEMENTED
    if (g_peek("export_all")) {
	g_string("export_all", "Export all tools?", buf);
	export_all = (buf[0] == 'y' || buf[0] == 'Y');
    }
    if (g_peek("tools")) {
	g_string("tools", "Which tools to export?", buf);
	tool_list = buf;
    }
#endif
    return 0;
}
    
//
// parse the command line. First see if the user supplied a g-file or
// not; if so, read the g-file first to build the default set of params
// and THEN read the command line to override the g-file parameters.
//
static int parse_args(int argc, char* argv[],
    string& toolsfile, string& expfile, bool& export_all,
    string& tool_list
) {
    int c, ind;
    string gfile;
    int errcnt = 0;

    //
    // first pass thru the command line args to read gfile and set verbose
    // flag. Also check for flags like --help, --print-examples etc and
    // handle in the first pass.
    //
    while ((c = getopt_long(
	argc, argv, "g:I:e:t:al:T:dvh", long_options, &ind)) != EOF
    ) {
    if (c == 0 && long_options[ind].flag == 0)
	c = long_options[ind].val;

	switch (c) {
	case 'g':
	    gfile = optarg;
	    break;

	case 'v':
	    if (optarg) {
		g_verbose = (
		    optarg[0] == 'y' || optarg[0] == 'Y' || optarg[0] == '1'
		);
	    } else {
		g_verbose = true;
	    }
	    break;

	case 'h':
	    usage(cout, true, 0);
	    break;
	
	case 129:
	    more_usage(cout, true, 0);
	    break;

	case 130:
	    print_gfile(cout, true, 0);
	    break;

	case '?':
	    ++errcnt;
	    break;
	    
	default:
	    break;
	}
    }
    if (errcnt) {
	usage(cerr, true, 1);
    }
    if (gfile.length()) {
	read_gfile(gfile, toolsfile, expfile, export_all, tool_list);
    }

    //
    // reset optind to make a second pass. This time we override the
    // g-file params (if any) with command line params (if any).
    //
    optind = 0;
    while ((c = getopt_long(
	argc, argv, "g:I:e:t:al:T:dvh", long_options, &ind)) != EOF
    ) {
    if (c == 0 && long_options[ind].flag == 0)
	c = long_options[ind].val;

	switch (c) {
	case 'I':
	    search_path = optarg;
	    break;

	case 'e':
	    expfile = optarg;
	    break;

	case 't':
	    toolsfile = optarg;
	    break;

#if NOT_IMPLEMENTED
	case 'a':
	    if (optarg) {
		export_all = (optarg[0] == 'y' || optarg[0] == 'Y');
	    } else {
		export_all = true;
	    }
	    break;

	case 'T':
	    tool_list = optarg;
	    break;
#endif

	case 'l':
	    logfile = optarg;
	    break;
	
	case 'T':
	    trace_str = optarg;
	    break;

	case 'd':
	    Globals::debug = true;
	    break;

	case '?':
	    ++errcnt;
	    break;
	}
    }

    if (argc != optind) {
	cerr << "ERROR: bogus command line argument(s)." << endl;
	++errcnt;
    } 
    
    if (errcnt) {
	usage(cerr, true, 1);
    }

    char buf[1024];
    if (expfile.size() == 0) {
	promptforval (
	    "Enter experiment file name (default = excon.exp) > ", buf
	);
	if (strlen(buf) == 0) 
	     strcpy (buf, "excon.exp");
	expfile = buf;
    }
    if (toolsfile.size() == 0) {
	promptforval (
	    "Enter tools file name   (default = <excon.tools>) > ", buf
	);
	if (strlen(buf) == 0) 
	    strcpy (buf, "<excon.tools>");
	toolsfile = buf;
    }
    return 0;
}

static int assign_trace_flag(const char* str) {
    int trace_flag = -1;
    if (strcasecmp(str, "default") == 0)
	trace_flag = default_traces;
    else if (strcasecmp(str, "none") == 0)
	trace_flag = Logger::none;
    else if (strcasecmp(str, "minimal") == 0)
	trace_flag = Logger::minimal;
    else if (strcasecmp(str, "load") == 0)
	trace_flag = Logger::load;
    else if (strcasecmp(str, "nested_load") == 0)
	trace_flag = Logger::nested_load;
    else if (strcasecmp(str, "parse") == 0)
	trace_flag = Logger::parse;
    else if (strcasecmp(str, "run") == 0)
	trace_flag = Logger::run;
    else if (strcasecmp(str, "report") == 0)
	trace_flag = Logger::report;
    else if (strcasecmp(str, "resolve") == 0)
	trace_flag = Logger::resolve;
    else if (strcasecmp(str, "collect") == 0)
	trace_flag = Logger::collect;
    else if (strcasecmp(str, "debug") == 0)
	trace_flag = Logger::debug;
    else if (strcasecmp(str, "job") == 0)
	trace_flag = Logger::job;
    else if (strcasecmp(str, "all") == 0)
	trace_flag = Logger::all;
    
    return trace_flag;
}

static int init_logger() {
    unsigned trace_flags = default_traces;

    if (Globals::debug) {
	trace_flags = Logger::all;
    } else if (trace_str.size()) {
	char* tmpbuf = new char[trace_str.size() + 1];
	strcpy(tmpbuf, trace_str.c_str());

	char* this_trace;
	for(char* p = tmpbuf; (this_trace = strtok(p, ",:+")); p = 0) {
	    int this_trace_flag = assign_trace_flag(this_trace);
	    if (this_trace_flag == -1) {
		cerr << "exp2gfile (ERROR): Illegal trace flag `" 
		    << this_trace << "'. Ignored." << endl;
	    } else {
		trace_flags |= this_trace_flag;
	    }
	}
	delete[] tmpbuf;
    }
    //
    // unless user explicitly says "none", turn on minimal logging.
    //
    if (trace_flags != Logger::none)
	trace_flags |= Logger::minimal;
    Globals::logger->set_loglevel(trace_flags);

    ofstream* ofp = new ofstream(logfile.c_str());
    if (!*ofp) {
	cerr << "exp2gfile: Error opening log file \"" << logfile 
	    << "\". Will not log to file. Check permissions." << endl;
    } else {
	Globals::logger->add(*ofp);
    }
    return 0;
}

static int init_file_search_path() {
    //
    // initialize the search path for include files in TOOLS and EXP
    // files. The syntax is the following:
    //    excon [options] -I dir1:[dir2[:dir3]]]
    // the search path is then set to dir1 ... dirn in order, and then
    // "." and $TOOLSET_DATA (which is the default if no searchpath is 
    // specified).
    //
    if (search_path.size()) {
	char* tmpbuf = new char[search_path.size() + 1];
	strcpy(tmpbuf, search_path.c_str());
	char* include_dir;
	for(char *p = tmpbuf; (include_dir = strtok(p, ",:+")); p = 0) {
	    char tmpbuf2[1024];
	    if (expandpath(include_dir, tmpbuf2)) {
		EXCON_LOG_ALWAYS
		    << endl
		    << "EXCON (Warning): Directory `" << include_dir
		    << "' in search path is invalid. Ignored."
		    << endl;
	    } else {
		Globals::file_search_path->enq(strcpy_with_alloc(tmpbuf2));
	    }
	}
	delete[] tmpbuf;
    }

    Globals::file_search_path->enq(strcpy_with_alloc("."));
    char const* datadir = getenv("TOOLSET_DATA");
    if (datadir == 0) {
	datadir = TOOLSET_DATA;
    }
    Globals::file_search_path->enq(strcpy_with_alloc(datadir));

    return 0;
}

static int init_program_search_path() {
    //
    // initialize the search path from PATH env variable for PROGRAM 
    // files in TOOLS file.
    //
    char const* pathvar = getenv("PATH");
    if (pathvar == 0) {
	EXCON_LOG_ALWAYS
	    << "EXCON Warning: "
	    << "No PATH variable defined in login environment!"
	    << endl
	    << "    May not be able to run programs."
	    << endl;
    } else {
	char* tmpbuf = new char[strlen(pathvar) + 1];
	strcpy(tmpbuf, pathvar);
	char* program_dir;
	for(char *p = tmpbuf; (program_dir = strtok(p, ":")); p = 0) {
	    Globals::program_search_path->enq(strcpy_with_alloc(program_dir));
	}
	delete[] tmpbuf;
    }
    return 0;
}

static int init_user() {
    struct passwd* pw_entry = getpwuid(getuid());
    assume(pw_entry != 0);
    Globals::username = pw_entry->pw_name;
    Globals::homedir = pw_entry->pw_dir;
    return 0;
}

//=======================================================================//

//
// initialize as many globals here as we can. Some require that cmd
// is parsed first (eg., JobMgr needs ``njobs'' param), so we need to
// initialize these in post_init_globals().
//
static int pre_init_globals(int /*argc*/, char* argv[]) {
    //
    // setup username and such.
    //
    init_user();

    //
    // set up the logger first so others can use it.
    //
    Globals::logger      = new Logger(cerr);
    Globals::programname = getprogramname (argv);
    Globals::cleanupmgr  = new CleanupMgr ();  // CLEANUP FILES WE LEAVE AROUND
    Globals::pmu         = new Pmu        ();  // GLOBAL PATTERN MATCH UNIT
    Globals::stats       = new Stats      ();  // PROGRAM STATS COLLECTOR
    Globals::toolsmgr    = new ToolsMgr   ();  // TOOLS MANAGER
    Globals::expmgr      = new Exp2GFILEMgr();  // EXPERIMENT MANAGER
    Globals::file_search_path = new Stq;
    Globals::program_search_path = new Stq;

    return 0;
}

static int post_init_globals(string& toolsfile, string& expfile) {
    init_logger();

    if (!getenv("TOOLSET_DATA")) {
	EXCON_LOG_ALWAYS
	    << "****** Exp2GFILE startup warning ****** " << endl
	    << "    TOOLSET_DATA variable is not defined in environment."
	    << endl
	    << "    Using " << TOOLSET_DATA << ". Some files may not be found."
	    << endl
	    << "    Please initialize Toolset environment with setup_toolset"
	    << endl
	    << "*********************************** "
	    << endl;
    }

    init_file_search_path();
    init_program_search_path();

    char buf[1024];
    switch(search_file(expfile.c_str(), Globals::file_search_path, buf)) {
	case 0:
	    expfile = buf;
	    break;
	
	case 1:
	    EXCON_LOG_ALWAYS 
		<< "Exp2GFILE Error: Cannot find EXPERIMENT file `" 
		<< expfile << "' in standard search path." 
		<< endl;
	    exit(1);
	    break;

	case 2:
	    EXCON_LOG_ALWAYS 
		<< "Exp2GFILE Error: Cannot open EXPERIMENT file `" 
		<< expfile << "'."
		<< endl;
	    exit(1);
	    break;
	
	default:
	    break;
    }

    switch(search_file(toolsfile.c_str(), Globals::file_search_path, buf)) {
	case 0:
	    toolsfile = buf;
	    break;
	
	case 1:
	    EXCON_LOG_ALWAYS 
		<< "Exp2GFILE Error: Cannot find TOOLS file `" << toolsfile
		<< "' in standard search path." 
		<< endl;
	    exit(1);
	    break;

	case 2:
	    EXCON_LOG_ALWAYS 
		<< "Exp2GFILE Error: Cannot open TOOLS file `" 
		<< toolsfile << "'."
		<< endl;
	    exit(1);
	    break;
	
	default:
	    break;
    }

    return 0;
}

static int build_tool_list(const string& tool_list,
    list<string>& toolnames
) {
    if (tool_list.size() == 0)
	return 1;
    char* tmpbuf = new char[tool_list.size() + 1];
    strcpy(tmpbuf, tool_list.c_str());

    char* this_tool;
    for(char* p = tmpbuf; (this_tool = strtok(p, ",:+")); p = 0) {
	toolnames.push_back(string(this_tool));
    }
    delete[] tmpbuf;
    return 0;
}


//=======================================================================//

static int exp2gfile_init(int argc, char* argv[],
    string& toolsfile, string& expfile, bool& export_all,
    list<string>& toolnames
) {
    pre_init_globals(argc, argv);
    string tool_list;
    parse_args(argc, argv, toolsfile, expfile, export_all, tool_list);
    post_init_globals(toolsfile, expfile);
    if (!export_all) {
	build_tool_list(tool_list, toolnames);
    } 
    return 0;
}

// ====================================================================== //

#if NOT_IMPLEMENTED
static int get_all_tools(const ToolsMgr& toolsmgr, list<string>& tools) {
    const Stq* all_tools = toolsmgr.gettools();
    for (int i = 0; i < all_tools->size(); ++i) {
	const Tool* tool = (const Tool*)all_tools->cycle();
	tools.push_back(tool->getname());
    }
    return all_tools->size();
}
#endif

static int export_gfile(const ToolInstance* ti) {
    string instname(ti->getinstname());
    string gfile = instname + ".g";
    ofstream ofp(gfile.c_str());
    if (!ofp) {
	EXCON_LOG_ALWAYS
	    << endl
	    << "exp2gfile: "
	    << "Cannot open output g-file `" << gfile << "'." << endl;
	return 1;
    }

    if (g_verbose) {
	EXCON_LOG_ALWAYS
	    << "Tool `" << instname << "' variables ==> `" << gfile << "'." 
	    << endl;
    }

    ofp << "######" << endl << "#" << endl
	<< "# TOOL : " << instname << endl
	<< "# GFILE automatically generated from EXCON TOOLS file" << endl
	<< "#" << endl << "######" << endl << endl; 

    const Tool* tool = ti->gettool();
    for (int i = 1; i <= tool->nvars(); ++i) {
	const Var* var = tool->getvar(i);  
	assume (var != 0);
	if (var->hasprop("$INPUTS")) {
	    const Value* val = ti->getvarval(var->getname());  
	    assume (val != 0);
	    ofp << var->getname() << " = " << val->getstrval() << endl;
	}
    }
    ofp << endl << "#" << endl << "#" << endl << "######" << endl << endl; 
    return 0;
}

//
// export_gfiles:
//
// read an EXCON toolsfile and exports the TOOL variables as gfile params.
//
static int export_gfiles(Exp2GFILEMgr& expmgr, 
    bool /* export_all */, const list<string>& /* toolnames */
) {
    //
    // now create the job sequence for the first iteration. That's all
    // we can do w/out having to actually run the job.
    //
    Exp* exp = expmgr.exp();
    exp->setup_links(exp->get_job_sequence(0));
    Stq* job = exp->get_next_job(1);
    assert(job != 0);

    exp->resolveunique(job);
    exp->resolveunique(job);
    unsigned jobsize = job->size();
    while(jobsize--) {
	ToolInstance* ti = (ToolInstance*)job->cycle();
	exp->resolvelinks(ti);
	export_gfile(ti);
    }

#if NOT_IMPLEMENTED
    list<string> export_list;
    if (export_all) {
	get_all_tools(toolsmgr, export_list);
    } else {
	export_list = toolnames;
    }

    list<string>::const_iterator it = export_list.begin();
    for(; it != export_list.end(); ++it) {
	const string& toolname = *it;
	export_gfile(toolsmgr, expmgr, toolname);
    }
#endif

    return 0;
}

// ====================================================================== //

int main(int argc, char** argv) {
    string toolsfile;
    string expfile;
    list<string> toolnames;
    bool export_all = true;

    exp2gfile_init(argc, argv, toolsfile, expfile, export_all, toolnames);

    if (Globals::toolsmgr->load (toolsfile.c_str())) {
	EXCON_LOG_ALWAYS
	    << "Exp2GFILE: Errors found int TOOLS file(s). Fix and re-run"
	    << endl;
	exit(1);
    }

    if (Globals::expmgr->load (expfile.c_str())) {
	EXCON_LOG_ALWAYS
	    << "Exp2GFILE: Errors found in EXPERIMENT file(s). Fix and re-run"
	    << endl;
	exit(1);
    }

    Exp2GFILEMgr& expmgr = (Exp2GFILEMgr&)*Globals::expmgr;
    return export_gfiles(expmgr, export_all, toolnames);
}

// ====================================================================== //
