//
// excon_init.cc: Initialize the EXCON subsystems.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1995 Mumit Khan
//

// ====================================================================== //

#include <string>
#include <string.h>
#include <stdio.h>
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_LIMITS_H
# include <limits.h>		// for PATH_MAX
#endif
#include <sys/param.h>
#include <stdlib.h>
#include <pwd.h>		// for user info
#include <sys/types.h>		// for uid_t, pid_t etc

#ifndef PATH_MAX
#ifndef POSIX
#define PATH_MAX        MAXPATHLEN
#endif  /* Not POSIX.  */
#endif  /* No PATH_MAX.  */
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif  /* No MAXPATHLEN.  */

#include <getopt.h>		// better pick up GNU getopt!
#include <libcxrl.h>

#include <fstream.h>

#include "excon_init.h"
#include "exp.h"
#include "global.h"
#include "job.h"
#include "logger.h"
#include "misc.h"
#include "pmu.h"
#include "tool.h"
#include "utils.h"

// ====================================================================== //

//
// number of parallel jobs to run.
//
static int njobs = 1;

//
// names of the logfile EXCON uses. Defaults to "excon.log";
//
static string logfile = "excon.log";

//
// the trace string on command line line.
//
static string trace_str;

//
// the search path given on command line.
//
static string search_path;

//
// debug flag
//
static bool g_verbose = false;

//
// default flags.
//
static unsigned default_traces = Logger::minimal | Logger::load;

#if NOT_IMPLEMENTED
//
// print the current EXCON defaults and then exit?
//
static bool print_defaults_and_exit = false;
#endif


// ====================================================================== //

//
// command line options and such
//
struct option long_options[] = {
    { "gfile",		required_argument,	0, 'g' },
    { "search-path",	required_argument,	0, 'I' },
    { "expfile",	required_argument,	0, 'e' },
    { "toolsfile",	required_argument,	0, 't' },
    { "output",		required_argument,	0, 'o' },
    { "logfile",	required_argument,	0, 'l' },
    { "jobs",		required_argument,	0, 'j' },
    { "background",	optional_argument,	0, 'b' },
    { "abort-on-error",	optional_argument,	0, 'a' },
    { "traces",		required_argument,	0, 'T' },
    { "debug",		optional_argument,	0, 'd' },
    { "verbose",	optional_argument,	0, 'v' },
    { "help",		no_argument,		0, 'h' },
    { "more-help",	no_argument,		0, 129 },
#if NOT_IMPLEMENTED
    { "print-defaults",	no_argument,		0, 130 },
#endif
    { "print-examples",	no_argument,		0, 131 },
    { "print-gfile",	no_argument,		0, 132 },
    { 0,		0,			0,  0  }
};

static void usage(ostream& os, bool die = false,  int diecode = 0) {
    os << "\n"
    "Usage: EXCON [OPTIONS]\n\n"
    "Most options have reasonable defaults or are prompted for\n"
    "The following OPTIONS are recognized (in either short or long form):\n\n" 
    "  -g FILE, --gfile=FILE        Read g-parameters from FILE\n"
    "  -I PATH, --search-path=PATH  Use PATH (eg., .:tools) to search\n"
    "  -t FILE, --tool=FILE         Read TOOLS from FILE\n"
    "  -e FILE, --exp=FILE          Read EXPERIMENT from FILE\n"
    "  -o FILE, --output=FILE       Write report (table) to FILE\n"
    "  -j NJOBS,--jobs=NJOBS        # of jobs to run in parallel (default: 1)\n"
    "  -b,      --background        Run in the background (default: no)\n"
    "  -a,      --abort-on-error    Abort on error.\n"
    "  -T TLIST,--traces=TLIST      List of events to trace\n"
    "                               (none,all,minimal,load,parse,run,result,report)\n"
    "  -d,      --debug             Turn on debugging (equiv to --traces=all)\n"
    "  -v,      --verbose           Turn on verbose mode.\n"
    "  -h,      --help              Print this info\n"
    "           --more-help         Print more help info\n"
#if NOT_IMPLEMENTED
    "           --print-defaults    Print current default EXCON option values\n"
#endif
    "           --print-examples    Show some more examples\n"
    "           --print-gfile       Show example gfile parameters\n"
    "\n\n"
    "Examples:\n"
    "  % excon -t shadow.tools -e shadow.exp -o table -I mytoolsdir\n"
    "  % excon -t shadow.tools -e shadow.exp -o table -T load,run\n"
    "\n";

    if (die)
	exit(diecode);
}

static void more_usage(ostream& os, bool die = false,  int diecode = 0) {
    os << "\n"
    "Usage: EXCON [OPTIONS]\n\n"
    "Most options have reasonable defaults or are prompted for\n"
    "The following OPTIONS are recognized (in either short or long form):\n"
    "(you can use unique substrings for the long form; eg., you could\n"
    "say --out instead of --output since that's unique)\n\n"
    "  -g FILE, --gfile=FILE\n"
    "   You can supply EXCON all the parameters via \"gfile\" just like\n"
    "   all the other Toolset programs. Command line parameters override\n"
    "   the values in FILE, so you can use these as default values.\n"
    "   use --print-gfile for example g-file.\n"
    "   Example g-file:\n"
    "\n"
    "       verbose = no\n"
    "       expfile = shadow.exp\n"
    "       toolsfile = <toolset.tools>\n"
    "\n"
    "  -I PATH, --search-path=PATH\n"
    "   PATH is a colon- or comma- separated list of directories that EXCON\n"
    "   searches for TOOLS and EXPERIMENT files, when these files are named\n"
    "   within < and > (eg., <shadow.tools>). Multiple -I flags are simply\n"
    "   concatenated as they are seen.\n"
    "   An example of PATH would be /usr/local/shadow.tools:toolsdir\n"
    "\n"
    "  -t FILE, --tool=FILE\n"
    "   TOOLS file contains EXCON tool definitions. Usually users use the\n"
    "   Toolset provided one in $SHADOW_DATA directory. Specify the FILE\n"
    "   within < and > to force EXCON to search for it.\n"
    "   Typically named ``shadow.tools''\n"
    "\n"
    "  -e FILE, --exp=FILE\n"
    "   EXPRIMENT file contains user experiment sequence and also parameters\n"
    "   for the models and tools. Specify the FILE within < and > to force\n"
    "   EXCON to search for it.\n"
    "   Typically named ``shadow.exp''\n"
    "\n"
    "  -o FILE, --output=FILE\n"
    "   EXCON writes the tabular report to this OUTPUT file.\n"
    "   Typically named ``table?'', where ? = 1,2,3,...\n"
    "\n"
    "  -l FILE, --log=FILE\n"
    "   EXCON logs all activities to standard error (stderr/cerr) as well\n"
    "   as to the log file so you can peruse the messages after EXCON run\n"
    "   is done. Default: ``excon.log''\n"
    "\n"
    "  -j NJOBS,--jobs=NJOBS\n"
    "   EXCON can execute multiple parallel job sequences (dangerous!), so\n"
    "   if you really know what you're doing, specify NJOBS to be > 1.  The\n"
    "   default is to run 1 job sequence at a time and I suggest that you\n"
    "   let EXCON use the default.\n"
    "\n"
    "  -b,      --background\n"
    "   Tell EXCON to run in the background (like when you use & at then end\n"
    "   of command line) as a daemon. Avoids problems when you log out on\n"
    "   while running EXCON. Defaults to running in the foreground.\n"
    "\n"
    "  -a,      --abort-on-error\n"
    "   If set, EXCON will abort if a tool exits abnormally.\n"
    "   Defaults to no, and EXCON keeps on chugging, but enabling it helps\n"
    "   debugging both the Tools and Experiment files\n"
    "\n"
    "  -T TLIST,--traces=TLIST\n"
    "   List of events to trace when EXCON is running. The valid list is\n"
    "   is a proper subset of ``none,all,default,minimal,load,nested_load,\n"
    "   parse,run,resolve,report,collect,debug''\n"
    "\n"
    "  -d,      --debug\n"
    "   Print out gobs of information about EXCONs run. Not pretty. This is\n"
    "   equivalent to seting all the traces, ie, --traces=all\n"
    "\n"
    "  -v,      --verbose           Turn on verbose mode.\n"
    "   Synonym for --traces=load,run,result\n"
    "\n"
    "  -h,      --help              Print this info\n"
    "           --more-help         Print more help info\n"
    "   Print a bit more elaborate command line help information\n"
    "\n"
#if NOT_IMPLEMENTED
    "           --print-defaults    Print the current defaults\n"
    "   Print the internal defaults that EXCON will use for this run.  Look\n"
    "   at the entire command line before printing the defaults of course\n"
    "\n"
#endif
    "           --print-examples    Show some more examples\n"
    "           --print-gfile       Show example gfile parameters\n"
    "\n"
    "Examples:\n"
    "  % excon -t shadow.tools -e shadow.exp -o table -T load,run\n"
    "  % excon -t shadow.tools -e shadow.exp -o table -I mytoolsdir\n"
    "The above is equivalent to:\n"
    "  % excon --tool=shadow.tools --exp=shadow.exp --output=table \\\n"
    "    --search-path=mytoolsdir\n"
    "\n";

    if (die)
	exit(diecode);
}

#if NOT_IMPLEMENTED
static void print_defaults(ostream& os, bool die = false,  int diecode = 0) {
    os << "\n\nEXCON global defaults: " << endl;
    os 
	<< "  Abort on error?       "
	<< ((Globals::abort_on_error) ? "yes" : "no")
	<< endl;
    os 
	<< "  Traces set:           "
	<< trace_str
	<< endl;
    os 
	<< "  Debug mode?           "
	<< ((Globals::debug) ? "yes" : "no")
	<< endl;

    if (die)
	exit(diecode);
}
#endif

static void print_examples(ostream& os, bool die = false,  int diecode = 0) {

    os << "\n"
    "EXCON examples:\n"
    "\n"
    "The following will prompt you for all the required parameters\n"
    "\n"
    "  % excon\n"
    "\n"
    "The following will load TOOLS definitions from \"shadow.tools\",\n"
    "EXPERIMENT from \"shadow.exp\", output the report table in files\n"
    "that start with \"table\", and run the job.\n"
    "\n"
    "  % excon --tool=shadow.tools --exp=line.exp --output=table\n"
    "\n"
    "The following will turn on traces for \"run\" and \"load\" events\n"
    "search and load \"shadow.tools\" from either current directory or\n"
    "from your search path, load \"line.exp\" and run the job. Note the\n"
    "' (single quotes) to prevent the shell from interpreting the < and >\n"
    "in the TOOLS filename, <shadow.tools>.\n"
    "\n"
    "  % excon --traces=run,load --tool='<shadow.tools>' --exp=line.exp\n"
    "\n";

    if (die)
	exit(diecode);
}

static void print_gfile(ostream& os, bool die = false,  int diecode = 0) {
    os << "\n"
    "###\n"
    "#\n"
    "# Example EXCON gfile\n"
    "#\n"
    "###\n"
    "#\n"
    "# verbose mode (y/n).\n"
    "#\n"
    "verbose = n\n"
    "#\n"
    "# EXCON file search path for INCLUDE/TOOLS/EXP files. This is a\n"
    "# colon separated list of directories that EXCON searches when\n"
    "# sees a filename of the form <FILE>, where FILE is the filename\n"
    "# to open. EXCON always searches current directory and then\n"
    "# SHADOW_DATA directory at the end.\n"
    "#\n"
    "search_path = ~/shadow.tools:~khan/shadow.tools\n"
    "#\n"
    "# EXCON Tools file. This can in turn INCLUDE others. If you enter\n"
    "# the file name as <FILE> (eg., <shadow.tools>), EXCON will search\n"
    "# the directories in the search path for shadow.tools file.\n"
    "#\n"
    "toolsfile = <shadow.tools>\n"
    "#\n"
    "# EXCON Experiment file. This can in turn INCLUDE others. If you enter\n"
    "# the file name as <FILE> (eg., <shadow.exp>), EXCON will search\n"
    "# the directories in the search path for shadow.exp file.\n"
    "#\n"
    "expfile = shadow.exp\n"
    "#\n"
    "# EXCON Output/Report file prefix. If you name the prefix \"table\",\n"
    "# then the reports will be in files \"table1\", \"table2\", etc.\n"
    "#\n"
    "output = table\n"
    "#\n"
    "# EXCON log file. All warnings and errors from EXCON will go to\n"
    "# standard error/output and to this file as well. Great for debugging\n"
    "# when something goes wrong.\n"
    "#\n"
    "logfile = excon.log\n"
    "#\n"
    "# EXCON can now use traces to log certain events. The set of events\n"
    "# is `none,all,minimal,load,parse,run,result,report'. The default is\n"
    "# minimal.\n"
    "#\n"
    "traces = minimal\n"
    "#\n"
    "# EXCON has the capability for running multiple jobs in parallel,\n"
    "# but we recommend that you do not change this value (1). It's a\n"
    "# dangerous option, only for gurus and thrill-seekers!\n"
    "#\n"
    "jobs = 1\n"
    "#\n"
    "# EXCON has the capability of running in the background as a UNIX\n"
    "# daemon. Defaults to no.\n"
    "#\n"
    "background = no\n"
    "\n";

    if (die)
	exit(diecode);
}

static int read_gfile(const string& gfile,
    string& toolsfile, string& expfile, string& output, bool& background
) {
    //
    // now read the gfile is supplied before the rest of the command
    // is parsed, so we can override all of it via cmdline.
    //
    g_file(const_cast<char*>(gfile.c_str()));
    char buf[1024];
    if (g_peek("verbose")) {
	g_string("verbose", "Verbose mode (y/n)?", buf);
	if (buf[0] == 'y' || buf[0] == 'Y' || buf[0] == '1')
	    g_verbose = true;
    }
    if (g_peek("search_path")) {
	g_string("search_path", "Enter EXCON file search path ", buf);
	search_path = buf;
    }
    if (g_peek("toolsfile")) {
	g_string("toolsfile", "Enter EXCON TOOLS file", buf);
	toolsfile = buf;
    }
    if (g_peek("expfile")) {
	g_string("expfile", "Enter EXCON experiment file", buf);
	expfile = buf;
    }
    if (g_peek("output")) {
	g_string("output", "Enter EXCON output file prefix", buf);
	output = buf;
    }
    if (g_peek("logfile")) {
	g_string("logfile", "Enter EXCON log file", buf);
	logfile = buf;
    }
    if (g_peek("traces")) {
	g_string("traces", "Enter EXCON events to trace", buf);
	trace_str = buf;
    }
    if (g_peek("jobs")) {
	g_int("jobs", "Enter # of jobs to run in parallel", &njobs);
    }
    if (g_peek("background")) {
	g_string("background", "Run in the background as a daemon (y/n)", buf);
	if (buf[0] == 'y' || buf[0] == 'Y' || buf[0] == '1')
	    background = true;
    }
    return 0;
}
    
//
// parse the command line. First see if the user supplied a g-file or
// not; if so, read the g-file first to build the default set of params
// and THEN read the command line to override the g-file parameters.
//
static int parse_args(int argc, char* argv[],
    string& toolsfile, string& expfile, string& output, bool& background
) {
    int c, ind;
    string gfile;
    int errcnt = 0;

    //
    // first pass thru the command line args to read gfile and set verbose
    // flag. Also check for flags like --help, --print-examples etc and
    // handle in the first pass.
    //
    while ((c = getopt_long(
	argc, argv, "g:I:e:t:o:l:j:baT:dvh", long_options, &ind)) != EOF
    ) {
    if (c == 0 && long_options[ind].flag == 0)
	c = long_options[ind].val;

	switch (c) {
	case 'g':
	    gfile = optarg;
	    break;

	case 'v':
	    if (optarg) {
		g_verbose = (
		    optarg[0] == 'y' || optarg[0] == 'Y' || optarg[0] == '1'
		);
	    } else {
		g_verbose = true;
	    }
	    break;

	case 'h':
	    usage(cout, true, 0);
	    break;
	
	case 129:
	    more_usage(cout, true, 0);
	    break;

	case 131:
	    print_examples(cout, true, 0);
	    break;

	case 132:
	    print_gfile(cout, true, 0);
	    break;

	case '?':
	    ++errcnt;
	    break;
	    
	default:
	    break;
	}
    }
    if (errcnt) {
	usage(cerr, true, 1);
    }
    if (gfile.length()) {
	read_gfile(gfile, toolsfile, expfile, output, background);
    }

    //
    // reset optind to make a second pass. This time we override the
    // g-file params (if any) with command line params (if any).
    //
    optind = 0;
    while ((c = getopt_long(
	argc, argv, "g:I:e:t:o:l:j:baT:dvh", long_options, &ind)) != EOF
    ) {
    if (c == 0 && long_options[ind].flag == 0)
	c = long_options[ind].val;

	switch (c) {
	case 'I':
	    search_path = optarg;
	    break;

	case 'e':
	    expfile = optarg;
	    break;

	case 't':
	    toolsfile = optarg;
	    break;

	case 'o':
	    output = optarg;
	    break;

	case 'l':
	    logfile = optarg;
	    break;
	
	case 'j':
	    if ((sscanf(optarg, "%d", &njobs)) != 1) {
		cerr << "ERROR: -j option requires integer argument." << endl;
		++errcnt;
	    }
	    if (njobs < 1) {
		cerr << "ERROR: -j must be >= 1." << endl;
		++errcnt;
	    }
	    break;

	case 'b':
	    if (optarg) {
		background = (
		    optarg[0] == 'y' || optarg[0] == 'Y' || optarg[0] == '1'
		);
	    } else {
		background = true;
	    }
	    break;
	
	case 'a':
	    if (optarg) {
		Globals::abort_on_error = (
		    optarg[0] == 'y' || optarg[0] == 'Y' || optarg[0] == '1'
		);
	    } else {
		Globals::abort_on_error = true;
	    }
	    break;
	
	case 'T':
	    trace_str = optarg;
	    break;

	case 'd':
	    Globals::debug = true;
	    break;

#if NOT_IMPLEMENTED
	case 130:
	    print_defaults_and_exit = true;
	    break;
#endif

	case '?':
	    ++errcnt;
	    break;
	}
    }

    if (argc != optind) {
	cerr << "ERROR: bogus command line argument(s)." << endl;
	++errcnt;
    } 
    
    if (errcnt) {
	usage(cerr, true, 1);
    }

#if NOT_IMPLEMENTED
    if (print_defaults_and_exit)
	print_defaults(cout, true, 0);
#endif

    char buf[1024];
    if (expfile.size() == 0) {
	promptforval (
	    "Enter experiment file name (default = shadow.exp) > ", buf
	);
	if (strlen(buf) == 0) 
	     strcpy (buf, "shadow.exp");
	expfile = buf;
    }
    if (toolsfile.size() == 0) {
	promptforval (
	    "Enter tools file name   (default = <shadow.tools>) > ", buf
	);
	if (strlen(buf) == 0) 
	    strcpy (buf, "<shadow.tools>");
	toolsfile = buf;
    }
    if (output.size() == 0) {
	promptforval (
	    "Enter output file name prefix  (default = table) > ", buf
	);
	if (strlen(buf) == 0) 
	    strcpy (buf, "table");
	output = buf;
    }

    return 0;
}

//
// assign_trace: given the name, return the Trace id define in Logger.
// Note that the actual trace values are unsigned, and there is no
// concept of illegal trace value we can return. Hence we pass the
// arg by reference and return an error code.
//
// Return: 0 if OK and trace_flag is set to the correct value.
//         1 if Not OK and trace_flag is left unchanged.
//
static int assign_trace_flag(const char* str, unsigned& trace_flag) {
    int retcode = 0;
    if (strcasecmp(str, "default") == 0)
	trace_flag = default_traces;
    else if (strcasecmp(str, "none") == 0)
	trace_flag = Logger::none;
    else if (strcasecmp(str, "minimal") == 0)
	trace_flag = Logger::minimal;
    else if (strcasecmp(str, "load") == 0)
	trace_flag = Logger::load;
    else if (strcasecmp(str, "nested_load") == 0)
	trace_flag = Logger::nested_load;
    else if (strcasecmp(str, "parse") == 0)
	trace_flag = Logger::parse;
    else if (strcasecmp(str, "run") == 0)
	trace_flag = Logger::run;
    else if (strcasecmp(str, "report") == 0)
	trace_flag = Logger::report;
    else if (strcasecmp(str, "resolve") == 0)
	trace_flag = Logger::resolve;
    else if (strcasecmp(str, "collect") == 0)
	trace_flag = Logger::collect;
    else if (strcasecmp(str, "debug") == 0)
	trace_flag = Logger::debug;
    else if (strcasecmp(str, "job") == 0)
	trace_flag = Logger::job;
    else if (strcasecmp(str, "all") == 0)
	trace_flag = Logger::all;
    else 
	retcode = 1;
    
    return retcode;
}

static int init_logger() {
    unsigned trace_flags = default_traces;

    if (Globals::debug) {
	trace_flags = Logger::all;
    } else if (trace_str.size()) {
	char* tmpbuf = new char[trace_str.size() + 1];
	strcpy(tmpbuf, trace_str.c_str());

	char* this_trace;
	for(char* p = tmpbuf; (this_trace = strtok(p, ",:+")); p = 0) {
	    unsigned this_trace_flag;
	    if (assign_trace_flag(this_trace, this_trace_flag)) {
		cerr << "ERROR: Illegal trace flag `" << this_trace
		    << "'. Ignored." << endl;
	    } else {
		trace_flags |= this_trace_flag;
	    }
	}
	delete[] tmpbuf;
    }
    //
    // unless user explicitly says "none", turn on minimal logging.
    //
    if (trace_flags != Logger::none)
	trace_flags |= Logger::minimal;
    Globals::logger->set_loglevel(trace_flags);

    ofstream* ofp = new ofstream(logfile.c_str());
    if (!*ofp) {
	cerr << "Error opening log file \"" << logfile 
	    << "\". Will not log to file. Check permissions." << endl;
    } else {
	Globals::logger->add(*ofp);
    }
    return 0;
}

static int init_file_search_path() {
    //
    // initialize the search path for include files in TOOLS and EXP
    // files. The syntax is the following:
    //    excon [options] -I dir1:[dir2[:dir3]]]
    // the search path is then set to dir1 ... dirn in order, and then
    // "." and $SHADOW_DATA (which is the default if no searchpath is 
    // specified).
    //
    if (search_path.size()) {
	char* tmpbuf = new char[search_path.size() + 1];
	strcpy(tmpbuf, search_path.c_str());
	char* include_dir;
	for(char *p = tmpbuf; (include_dir = strtok(p, ",:+")); p = 0) {
	    char tmpbuf2[1024];
	    if (expandpath(include_dir, tmpbuf2)) {
		EXCON_LOG_ALWAYS
		    << endl
		    << "EXCON (Warning): Directory `" << include_dir
		    << "' in search path is invalid. Ignored."
		    << endl;
	    } else {
		Globals::file_search_path->enq(strcpy_with_alloc(tmpbuf2));
	    }
	}
	delete[] tmpbuf;
    }

    Globals::file_search_path->enq(strcpy_with_alloc("."));
    char shadow_data_dir[PATH_MAX];
    char const* datadir = getenv("SHADOW_DATA");
    if (datadir) {
        strcpy(shadow_data_dir, datadir);
    } else {
	char const* rootdir = getenv("SHADOW_ROOT");
	if (rootdir) {
	    sprintf(shadow_data_dir, "%s/data", rootdir);
	} else {
	    strcpy(shadow_data_dir, SHADOW_DATA);
	}
    }
    Globals::file_search_path->enq(strcpy_with_alloc(shadow_data_dir));

    return 0;
}

static int init_program_search_path() {
    //
    // initialize the search path from PATH env variable for PROGRAM 
    // files in TOOLS file. Always *append* the $SHADOW_ROOT/bin 
    // directory to the current path just to be safe.
    //
    char const* pathvar = getenv("PATH");
    if (pathvar == 0) {
	EXCON_LOG_ALWAYS
	    << "EXCON Warning: "
	    << "No PATH variable defined in login environment!"
	    << endl
	    << "    May not be able to run programs."
	    << endl;
    } else {
	char* tmpbuf = new char[strlen(pathvar) + 1];
	strcpy(tmpbuf, pathvar);
	char* program_dir;
	for(char *p = tmpbuf; (program_dir = strtok(p, ":")); p = 0) {
	    Globals::program_search_path->enq(strcpy_with_alloc(program_dir));
	}
	delete[] tmpbuf;
    }
    const char* shroot = getenv("SHADOW_ROOT");
    if (shroot) {
	char bindir[PATH_MAX+1];
	sprintf(bindir, "%s/bin", shroot);
	Globals::program_search_path->enq(strcpy_with_alloc(bindir));
    }
    return 0;
}

static int init_user() {
    struct passwd* pw_entry = getpwuid(getuid());
    assume(pw_entry != 0);
    Globals::username = pw_entry->pw_name;
    Globals::homedir = pw_entry->pw_dir;
    char curdir[PATH_MAX+1];
    if (!getcwd(curdir, PATH_MAX + 1)) {
	EXCON_LOG_ALWAYS
	    << "EXCON: Cannot get working directory." << endl;
	perror("PWD");
	strcpy(curdir, ".");
    }
    Globals::curdir = curdir;
    return 0;
}

//=======================================================================//

//
// initialize as many globals here as we can. Some require that cmd
// is parsed first (eg., JobMgr needs ``njobs'' param), so we need to
// initialize these in post_init_globals().
//
static int pre_init_globals(int /*argc*/, char* argv[]) {
    //
    // setup username and such.
    //
    init_user();

    //
    // set up the logger first so others can use it.
    //
    Globals::logger      = new Logger(cerr);
    Globals::programname = getprogramname (argv);
    Globals::cleanupmgr  = new CleanupMgr ();  // CLEANUP FILES WE LEAVE AROUND
    Globals::pmu         = new Pmu        ();  // GLOBAL PATTERN MATCH UNIT
    Globals::stats       = new Stats      ();  // PROGRAM STATS COLLECTOR
    Globals::toolsmgr    = new ToolsMgr   ();  // TOOLS MANAGER
    Globals::expmgr      = new ExpMgr     ();  // EXPERIMENT MANAGER
    Globals::file_search_path = new Stq;
    Globals::program_search_path = new Stq;

    return 0;
}

static int post_init_globals(string& toolsfile, string& expfile) {
    init_logger();
    Globals::jobmgr = new JobMgr(njobs);  // JOB EXECUTION MANAGER

    if (!getenv("SHADOW_ROOT")) {
	EXCON_LOG_ALWAYS
	    << "****** EXCON startup warning ****** " << endl
	    << "    SHADOW_ROOT variable is not defined in environment."
	    << endl
	    << "    Using " << SHADOW_ROOT << ". Some files may not be found."
	    << endl
	    << "    Please initialize SHADOW environment with .shadowrc"
	    << endl
	    << "*********************************** "
	    << endl;
    }

    init_file_search_path();
    init_program_search_path();

    char buf[1024];
    switch(search_file(expfile.c_str(), Globals::file_search_path, buf)) {
	case 0:
	    expfile = buf;
	    break;
	
	case 1:
	    EXCON_LOG_ALWAYS 
		<< "EXCON Error: Cannot find EXPERIMENT file " << expfile 
		<< " in standard search path." << endl;
	    exit(1);
	    break;

	case 2:
	    EXCON_LOG_ALWAYS 
		<< "EXCON Error: Cannot open EXPERIMENT file " << expfile 
		<< endl;
	    exit(1);
	    break;
	
	default:
	    break;
    }

    switch(search_file(toolsfile.c_str(), Globals::file_search_path, buf)) {
	case 0:
	    toolsfile = buf;
	    break;
	
	case 1:
	    EXCON_LOG_ALWAYS 
		<< "EXCON Error: Cannot find TOOLS file " << toolsfile
		<< " in standard search path." << endl;
	    exit(1);
	    break;

	case 2:
	    EXCON_LOG_ALWAYS 
		<< "EXCON Error: Cannot open TOOLS file " << toolsfile << endl;
	    exit(1);
	    break;
	
	default:
	    break;
    }

    return 0;
}

//=======================================================================//

int excon_init(int argc, char* argv[],
    string& toolsfile, string& expfile, string& output, bool& background
) {
    pre_init_globals(argc, argv);
    parse_args(argc, argv, toolsfile, expfile, output, background);
    post_init_globals(toolsfile, expfile);
    return 0;
}

//=======================================================================//
