#if defined(__CYGWIN32__) || defined(USE_C_IO)

/*
 * writeoff_cygwin32.c: SHADOW Binary I/O
 *
 */

#include <stdio.h>
#include <string.h>
#include <dim.h>

#ifdef _MSC_VER
# define F77_NAME(name) __stdcall name
#else
# if !NO_FORT_UNDERSCORE
#  define F77_NAME(name) name ## _
# else
#  define F77_NAME(name) name
# endif
#endif


/*
 * G77 (GNU FORTRAN, upto and including v.0.5.18) adds a SECOND underscore
 * to names that already contain an underscore. Annoying as well, and
 * hence the G77_HACK_NAME(f) hack.
 */

#define G77_HACK_NAME(f) f##__

#if G77
# define WRITE_OFF G77_HACK_NAME(write_off)
#else
# ifdef _MSC_VER
#  define WRITE_OFF F77_NAME(WRITE_OFF)
# else
#  define WRITE_OFF F77_NAME(write_off)
# endif
#endif

/*
C+++
C	SUBROUTINE	WRITE_OFF (FNAME,RAY,PHASE,AP,NCOL,NPOINT,IFLAG,IERR)
C
C	Input	:	FNAME	(character*(*))	Output file name
C			RAY	Array which contains the basic 12 columns that
C				defined each ray
C			PHASE	Array which contains the OPD and the 2 phase 
C				angles of the A vector
C			AP	Array which contains the Ap vector for each ray
C			NCOL	Number of columns for each ray 
C				(valid NCOL = 12, 13, or 18)
C			NPOINT	Number of rays
C			IFLAG	A flag which is written to the output file.
C			IFORM = 0, Binary output
C			      = 1, Formatted output
C	
C	Output	:	IERR = 0, normal return
C			     = 1, error opening file
C			     = 2, error writing file
C
C---
*/
void WRITE_OFF (
    const char* fort_fname, 		/* filename to write to */
    double ray[][12], 			/* 1st dimension is npoint */
    double phase[][3], 			/*   ditto */
    double ap[][3], 			/*   ditto */
    int *ncol, int *npoint, int *iflag, int *iform, int *ierr,
    unsigned long fname_len		/* thank you FORTRAN! */
) {
    extern void leave_();
    char fname[256];
    int i, j;

    if (*ncol != 12 && *ncol != 13 && *ncol != 18) {
	leave_("WRITE_OFF", "Invalid argument of NCOL.", *ncol,
	    strlen("WRITE OFF"), strlen("Invalid argument of NCOL.")
	);
    }

    /* fix fortran filename */
    for(i = fname_len; fort_fname[i-1] == ' '; --i)
	;
    strncpy(fname, fort_fname, i);
    fname[i] = '\0';

    *ierr = 0;

    if (*iform == 0) {			/* binary output */
	int io_dummy[3];
	int reclen;
        FILE *fp = fopen(fname, "wb");
	if (!fp) {
	    *ierr = 1;
	    return;
	}
	io_dummy[0] = *ncol;
	io_dummy[1] = *npoint;
	io_dummy[2] = *iflag;
	reclen = sizeof(io_dummy);
	fwrite(&reclen, sizeof(int), 1, fp);
	fwrite(io_dummy, sizeof(io_dummy), 1, fp);
	fwrite(&reclen, sizeof(int), 1, fp);
	reclen = *ncol * sizeof(double);
	for (i = 0; i < *npoint; ++i) {
	    fwrite(&reclen, sizeof(int), 1, fp);
	    for (j = 0; j < 12; ++j) {
		fwrite(&ray[i][j], sizeof(double), 1, fp);
	    }
	    if (*ncol == 13) {
		fwrite(&phase[i][1], sizeof(double), 1, fp);
	    } else if (*ncol == 18) {
		for (j = 0; j < 3; ++j) {
		    fwrite(&phase[i][j], sizeof(double), 1, fp);
		}
		for (j = 0; j < 3; ++j) {
		    fwrite(&ap[i][j], sizeof(double), 1, fp);
		}
	    }
	    fwrite(&reclen, sizeof(int), 1, fp);
	}
	fclose(fp);
    } else if (*iform == 1) {		/* ascii/text output */
        FILE *fp = fopen(fname, "w");
	if (!fp) {
	    *ierr = 1;
	    return;
	}
	fprintf(fp, "%d %d %d\n", *ncol, *npoint, *iflag);
	for (i = 0; i < *npoint; ++i) {
	    for (j = 0; j < 12; ++j) {
		fprintf(fp, "%.15e ", ray[i][j]);
	    }
	    if (*ncol == 13) {
		fprintf(fp, "%.15e ", phase[i][1]);
	    } else if (*ncol == 18) {
		for (j = 0; j < 3; ++j) {
		    fprintf(fp, "%.15e ", phase[i][j]);
		}
		for (j = 0; j < 3; ++j) {
		    fprintf(fp, "%.15e ", ap[i][j]);
		}
	    }
	}
	fputc('\n', fp);
        fclose(fp);
    } else {
        leave_("WRITE_OFF", "Invalid argument of IO Format", iform,
            strlen("WRITE_OFF"), strlen("Invalid argument of IO Format")
	);
	return;
    }

    return;
}

#else /* __CYGWIN32__ || USE_C_IO */

static int write_off_c_dummy;

#endif
