//
// tool.cc: EXCON tool manager
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#include <iostream.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "global.h"
#include "link.h"
#include "misc.h"
#include "pmu.h"
#include "tool.h"
#include "utils.h"
#include "value.h"
#include "variable.h"


/****************************************************************************
 *
 *  TOOLS
 *
 ***************************************************************************/

Tool::Tool (const char* name) {
    name_ = strcpy_with_alloc (name);
    vartable_ = new Hashtable (19); 
    varnlist_ = new Stq (); 
    links_ = new Stq ();
    path_ = nil;
    exec_ = nil;
    runtime_ = 60;

    // add the PSEUDO variables that are automatically set.
    Var* var = new Var("$GFILE", "filename", new TextValue("unassigned"));
    var->addprop("$PSEUDO");
    addvar(var);
    var = new Var("$ITERATION", "integer", new IntegerValue(-1));
    var->addprop("$PSEUDO");
    addvar(var);
}

Tool::~Tool() {
    delete[] name_ ; name_ = nil;
    delete vartable_ ; vartable_ = nil;		// DELETE CONTAINER elems.
    delete varnlist_; varnlist_ = nil;

    unsigned size = links_->size();
    while(size--) {
	Link* link = (Link*)links_->pop();
	delete link; link = nil;
    }
    delete links_; links_ = nil;

    delete[] exec_ ; exec_ = nil;
    delete[] path_ ; path_ = nil;
}

void Tool::errorcpuenv() { 
    cerr 
	<< "Could not find CPU environmental variable.\n"
	<< "Set this variable and try again.\n";  
    exit (1);   
}

int Tool::nvars() const { return vartable_->size(); }

void Tool::addvar(Var* var) { 
    vartable_->insert (var->getname(), var); 
    varnlist_->enq (var->getname());
}

Var* Tool::getvar(const char* name) const { 
    return (Var*)vartable_->find (name); 
}

Var* Tool::getvar(int ith) const { return (Var*)vartable_->get(ith); }

void Tool::setexec(const char* str) {
    delete[] exec_ ; exec_ = nil;
    exec_ = strcpy_with_alloc (str);
}

void Tool::setpath(const char* str) { 
    // EXPAND ANY $??? VARIABLES
    char buf1[1024];  char buf2[1024];  
    strcpy (buf1, str);  
    char* master = buf1; char* temp = buf2;  char* swap = nil;

    // SUBSTITUTE FOR CPU TYPE
    if (strstr (buf1, "$CPU")) {
	char* env = getenv ("CPU");  
	if (env == nil) 
	    errorcpuenv ();
	strsub ("$CPU", env, master, temp); 
	swap = master;  master = temp;  temp = swap;
    }
    delete[] path_ ; path_ = nil;
    path_ = strcpy_with_alloc (master);
}

/****************************************************************************
 *
 *  TOOL INSTANCE
 *
 ****************************************************************************/

ToolInstance::ToolInstance (Tool* tool, int number, int iteration) { 
    tool_ = tool; number_ = number;
    instvars_ = new Hashtable (19); 
    char buf[1024]; 
    char buf2[32];  
    strcpy (buf, tool_->getname());
    if (number > 0) { 
	sprintf (buf2, "%d", number);  
	strcat (buf, buf2); 
    }
    instname_ = strcpy_with_alloc (buf);
    setiteration(iteration);
    gpath_ = nil;
    // WARNING: DO NOT GENERATE AN exec LINE until right before running.
    // since the gfile depends on the iteration number
    exec_ = nil;

    // this is usually reset by client in ToolInstance::set_resolved_links
    resolved_links_ = new Stq;

    // run tree optimization stuff.
    // the free variable is loaded by the experiment manager for each
    // instance that has any free variables associated with it. DO NOT
    // DELETE THE LIST MEMBERS.
    freevars_ = new Stq;
    needs_to_run_ = true;
}


ToolInstance::~ToolInstance() {
    tool_ = nil; number_ = -1;

    unsigned size = instvars_->size();
    for(unsigned i = 1; i <= size; i++) {
	Var* var = (Var*)instvars_->get(i);
	delete var ; var = nil;
    }
    delete instvars_ ; instvars_ = nil;

    delete[] instname_ ; instname_ = nil;
    delete[] gpath_ ; gpath_ = nil;
    delete[] exec_; exec_ = nil;

    size = resolved_links_->size();
    while(size--) {
	ResolvedLink* link = (ResolvedLink*)resolved_links_->pop();
	delete link; link = nil;
    }
    delete resolved_links_;

    // run tree optimization stuff.
    // DO NOT DELETE THE LIST MEMBERS. These are allocated and free'd by
    // class Exp; SEE Exp::setloopvar|setsetvar|setfvgroup.
    delete freevars_;
}

void ToolInstance::addvar(Var* var) { 
    instvars_->insert(var->getname(), var); 
}

// SIDE EFFECT: CHANGES PSEUDO VARIABLE VALUE.
void ToolInstance::setiteration(int iteration) {
    iteration_ = iteration;

    // add the gfile name as a pseudo variable.
    setvarval("$ITERATION", new IntegerValue (iteration));
}

void ToolInstance::setgpath() {
    char buf[1024];
    sprintf(buf, "%s+%d.g", instname_, iteration_);
    delete[] gpath_ ; gpath_ = nil;
    gpath_ = strcpy_with_alloc (buf); 

    // add the gfile name as a pseudo variable.
    setvarval("$GFILE", new TextValue (gpath_));
}




//
// Method:	ToolInstance::resolveunique
//
// Caller:	Exp::resolveunique
//
// Parameters:
//
// Description:
// 		for all FILENAME variables with a value of "*$UNIQUE*" 
//		replace the $UNIQUE part of the value with 
//		INSTANCE_NAME + ITERATION_ID. 
//
//		Scanning for the variables MUST be done on the values
//		in the TOOL, not the instances (because the're probably
//		overwritten by default G files).
//
//		If the variable in the Tool Instance DOES NOT have $SAVE
//		property set, then add it the CleanupMgr's list.
//
// Return code:	void
//


void ToolInstance::resolveunique() {
    int nvars = tool_->nvars();

    // pre-compute the UNIQUE substitution string.
    char unique[1024];
    sprintf(unique, "%s+%d", getinstname(), getiteration());

    for (int i=1; i<=nvars; i++) {
	const Var* tool_var = tool_->getvar (i);
	if (tool_var->getval()->isatext()) {
	    const char* text = tool_var->getval()->gettext();
	    if (strstr (text, "$UNIQUE") != nil) {
		char subst[1024];
		// PERFORM REPLACEMENT
		(void) strsub("$UNIQUE", unique, text, subst); 
		setvarval (tool_var->getname(), new TextValue (subst));

		// must check the $SAVE property for the instance
		// variable, and NOT the tool variable, since $SAVE is
		// always added to variables in expseq_;
		if (!getvar(tool_var->getname())->hasprop("$SAVE")) {
		    GLB_cleanupmgr->addunlink (subst, getiteration());
		}
	    }
	}
    }
}

void ToolInstance::genexec() {
    // EX: $PATH -g $GFILE

    char buf[1024];  buf[0] = '\0';
    char token[1024];
    const char* exec = tool_->getexec();
    int len = strlen (exec);  int pos = 0;  int tokenpos;
    while (1) {
	// SKIPBLANKS
	while (exec[pos]==' '||exec[pos]=='\t') 
	    pos++;  
	if (pos >= len) break;

	// GET NEXT TOKEN
	tokenpos = 0;  
	while (!((exec[pos]==' ')||(exec[pos]=='\t')||(exec[pos]=='\0'))) { 
	    token[tokenpos] = exec[pos];  tokenpos++;  pos++; 
	}
	token[tokenpos] = '\0';
	// IF ITS A KEY WORD SUBSTITUTE...OTHERWISE PASS-THRU
	if (strcmp(token,"$PATH")==0) {
	    // INSERT THE EXEC PATH HERE
	    strcat (buf, tool_->getpath());  strcat(buf," ");
	    continue;
	}
	if (strcmp(token,"$ITERATION")==0) {
	    // INSERT THE ITERATION COUNTER HERE
	    char buf2[32];
	    sprintf(buf2, "%d", getiteration());
	    strcat (buf, buf2);  strcat(buf," ");
	    continue;
	}
	if (strcmp(token,"$GFILE")==0) {
	    setgpath ();
	    strcat (buf, getgpath()); strcat (buf," ");
	    continue;
	}
	// IF WE ARE HERE IT IS NOT A KEY WORD...JUST APPEND IT
	strcat (buf,token);  strcat(buf," ");
    }
    delete[] exec_;
    exec_ = strcpy_with_alloc (buf);
}

Var* ToolInstance::getvar(const char* name) const {
   Var* v = (Var*)instvars_->find(name);
   return (v != nil) ? v : tool_->getvar(name);
}

const Value* ToolInstance::getvarval(const char* name) const {
   const Var* var = getvar (name);  
   return (var != nil) ? var->getval() : nil;
}

// ASSIGN val TO THE VARIABLE NAMED name
// WARNING: NO COPY OF val IS MADE...THE POINTER IS DIRECTLY ASSIGNED
void ToolInstance::setvarval (const char* name, Value* val) {
    // IF var IS NOT IN instvars ADD IT
    Var* var = (Var*)instvars_->find(name);
    if (var == nil) {
	// FIND VAR NAME IN DEFAULT TABLE SO WE CAN GET THE VAR TYPE
	const Var* defvar = tool_->getvar(name);  assume (defvar != nil);
	var = defvar->clone();
	var->setval(val);
	instvars_->insert(var->getname(), (void*)var);
    }
    else {
	// IF WE ARE HERE JUST ASSIGN THE NEW VALUE
	var->setval(val);
    }
}

void ToolInstance::set_resolved_links(Stq* newlinks) {
    unsigned size = resolved_links_->size();
    while(size--) {
	ResolvedLink* link = (ResolvedLink*)resolved_links_->pop();
	delete link; link = nil;
    }
    delete resolved_links_;
    resolved_links_ = newlinks;
}

// READ A A gfile (NAMELIST of var = val PAIRS) 
// RETURN 1 IF OK; -1 IF fopen FAILED RETURN -1
int ToolInstance::readgfile(const char* gfile) {
    FILE *f = fopen (gfile, "r");  
    if (f == nil) 
	return -1;
    
    int line = 0;   
    int errors = 0;
    char buf[1024];
    char text1[1024];  
    char text2[1024]; 
    Pmu pmu;
    while (1) {
	if (fgets (buf, sizeof(buf), f) == nil) 
	    break;  
	line++;
	if (pmu.isacomment(buf)) 
	    continue;

	// VARIABLE DEFINITIONS
	if (pmu.match (buf, "<ident>", "=", "<text>")) {
	    pmu.gettextfield (1, text1);
	    pmu.gettextfield (3, text2);
	    if (GLB_debug) 
		cerr <<  "   VARIABLE: " << text1 << "=" << text2 << "\n";
	    // ...MAKE SURE VARIABLE HAS BEEN DEFINED
	    const Var* var = getvar(text1);
	    if (var == nil) {
		cerr 
		    << "The variable `" << text1 << "' is not defined for tool"
		    << " " << instname_ 
		    << "\nMake sure `" << text1 << "' is defined in file.\n";
		errors++;  
		continue;
	    }
	    // make sure the value is of the right type.
	    Value* newval = var->getval()->clone(text2);
	    if(newval == nil) {
		cerr 
		    << "Unexpected value (" << text2 << ") assigned to variable"
		    << " `" << text1 << "'\n"
		    << "The variable `" << text1 << "' is of type "
		    << "`" << var->gettype() << "'.\n";
		errors++;  
		continue;
	    }
	    setvarval(var->getname(), newval);
	    continue;
	}

	if (pmu.match (buf, "<ident>", "=", "<nothing>")) {
	    pmu.gettextfield (1, text1);
	    if (GLB_debug) 
		cerr <<  "   VARIABLE: " << text1 << "=" << "<nothing>" << "\n";
	    // ...MAKE SURE VARIABLE HAS BEEN DEFINED
	    const Var* var = getvar(text1);
	    if (var == nil) {
		cerr 
		    << "The variable `" << text1 << "' is not defined for tool"
		    << " " << instname_ 
		    << "\nMake sure `" << text1 << "' is defined in file.\n";
		errors++;  
		continue;
	    }
	    // ...MAKE SURE VALUE MATCHES VARIABLE TYPE
	    const char* value = "NONE SPECIFIED";
	    Value* newval = var->getval()->clone(value);
	    if (newval == nil) {
		cerr 
		    << "Unexpected value (" << value << ") assigned to variable"
		    << " `" << text1 << "'\n"
		    << "The variable `" << text1 << "' is of type "
		    << "`" << var->gettype() << "'.\n";
		errors++;  
		continue;
	    }
	    setvarval(var->getname(), newval);
	    continue;
	}

	// UNRECOGNIZABLE PATTERN 
	cerr <<  "Unable to understand line " << line << "..." << buf << "\n";
	errors++;  
	if (errors > 3) {
	    // IF WE ARE HERE TOO MAY ERRORS
	    cerr << "Too many errors in gfile `" << gfile << "'\n";
	    exit (1);
	}
    }
    fclose (f);
    return 1;
}


// WRITE OUT A gfile (NAMELIST of var = val PAIRS) 
// RETURN 1 IF OK; -1 IF fopen FAILED RETURN -1
int ToolInstance::writegfile () const {
    char buf[1024];
    FILE *f = fopen (gpath_,"w");  
    if (f == nil) 
	return -1;
    // SCAN ALL VARIABLES KNOWN TO tool
    for (int i=1; i<=tool_->nvars(); i++) {
	const Var* var = tool_->getvar(i);  assume (var != nil);
	// DO NOT INCLUDE OUTPUT VARIABLES
	if (var->hasprop("$OUTPUTS")) 
	    continue;
	// NOTE: DO NOT QUERY var DIRECTLY FOR A VALUE BECAUSE IT COULD 
	// BE OVERRIDDEN BY THE INSTANCE VERSION...INSTEAD CALL getvarval
	const Value* val = getvarval (var->getname());  assume (val != nil);
	sprintf (buf,"%s = %s\n",var->getname(), val->getstrval());
	fputs (buf,f);
    }
    fclose (f);
    GLB_stats->editedinputfile();

    const Var* var = getvar("$GFILE"); assume(var != nil);
    if (!var->hasprop("$SAVE"))
	GLB_cleanupmgr->addunlink (gpath_, iteration_);
    return 1;
}

ToolInstance* ToolInstance::clone() const {
    ToolInstance* buddy = new ToolInstance (tool_, number_, iteration_);

    unsigned size = instvars_->size();
    for(unsigned i = 1; i <= size; i++) {
	const Var* var = (Var*)instvars_->get(i);
	buddy->addvar(var->clone());
    }

    size = resolved_links_->size();
    while(size--) {
	ResolvedLink* link = (ResolvedLink*)resolved_links_->cycle();
	buddy->resolved_links_->enq(link->clone());
    }

    // run tree optimization stuff.
    delete buddy->freevars_;
    buddy->freevars_ = freevars_->clone();
    buddy->needs_to_run_ = needs_to_run_;

    return buddy;
}
	
/***************************************************************************
 *
 *  TOOLS MANAGER
 *
 ***************************************************************************/

ToolsMgr::ToolsMgr() {
   tools_ = new Hashtable (17);
   toollist_ = new Stq;
   pmu_ = new Pmu;
   current_tool_ = nil;
   currentloadfile_ = nil;
}

ToolsMgr::~ToolsMgr() {
    unsigned size = tools_->size();
    for(unsigned i = 1; i <= size; i++) {
	Tool* tool = (Tool*)tools_->get(i);
	delete tool ; tool = nil;
    }
    delete tools_ ; tools_ = nil;
    delete toollist_; toollist_ = nil;
    delete pmu_ ; pmu_ = nil;
    current_tool_ = nil;
}

void ToolsMgr::errormissingtool(const char* file) {
   cerr 
       << "The tools file `" << file << "' must begin with a $TOOL definition."
       << "\nAdd a $TOOL line to the file and try again.\n";
   exit (1);
}

void ToolsMgr::errorfilenotfound(const char* file) {
   cerr
       << "Could not open the tools file `" << file << "'.\n"
       << "The file may not exists or may you may not have read permission.\n"
       << "Check the file and try again.\n";
   exit (1);
}

Tool* ToolsMgr::gettool(const char* toolname) const { 
    return (Tool*)tools_->find (toolname); 
}

const Stq* ToolsMgr::gettools() const { 
    return toollist_;
}

void ToolsMgr::loadinputs(FILE* f, int& line) {
    char text1[1024];  
    char text2[1024];  
    int i;  
    double d;

    // current_tool MUST BE SET SO WE CAN POPULATE ITS RECORD
    if (current_tool_ == nil) 
	errormissingtool (currentloadfile_);
    if (GLB_debug) 
	cerr << "...LOADING INPUTS\n";
    while (fgets (buf_, sizeof(buf_), f) != nil) {
	line++;
	if (pmu_->isacomment(buf_)) continue;
	if (pmu_->match (buf_,"<ident>",":","text",":","<text>")) {
	    pmu_->gettextfield (1,text1);
	    pmu_->gettextfield (5,text2);
	    if (GLB_debug) 
		cerr << text1 << ":text:" << text2 << "\n";
	    // SEMANTICS
	    Value* val = new TextValue (text2);
	    Var* var = new Var (text1,"text",val);
	    var->addprop("$INPUTS");
	    current_tool_->addvar (var);
	} 
	else if (pmu_->match (buf_,"<ident>",":","real",":","<real>")) {
	    pmu_->gettextfield (1,text1);
	    d = pmu_->getrealfield (5);
	    if (GLB_debug) 
		cerr << text1 << ":real:" << d << "\n";
	    // SEMANTICS
	    Value* val = new RealValue (d);
	    Var* var = new Var (text1,"real",val);
	    var->addprop("$INPUTS");
	    current_tool_->addvar (var);
	}
	else if (pmu_->match (buf_,"<ident>",":","boolean",":","<yn>")) {
	    pmu_->gettextfield (1,text1);
	    i = pmu_->getcodefield (5);
	    if (GLB_debug) 
		cerr << text1 << ":boolean:" << i << "\n";
	    // SEMANTICS
	    Value* val = new BooleanValue (i);
	    Var* var = new Var (text1,"boolean",val);
	    var->addprop("$INPUTS");
	    current_tool_->addvar (var);
	}
	else if (pmu_->match (buf_,"<ident>",":","integer",":","<integer>")) {
	    pmu_->gettextfield (1,text1);
	    i = pmu_->getintfield (5);
	    if (GLB_debug) 
		cerr << text1 << ":integer:" << i << "\n";;
	    // SEMANTICS
	    Value* val = new IntegerValue (i);
	    Var* var = new Var (text1,"integer",val);
	    var->addprop("$INPUTS");
	    current_tool_->addvar (var);
	}
	else if (pmu_->match (buf_,"<ident>",":","filename",":","<text>")) {
	    pmu_->gettextfield (1,text1);
	    pmu_->gettextfield (5,text2);
	    if (GLB_debug) 
		cerr << text1 << ":filename:" << text2 << "\n";
	    // SEMANTICS
	    Value* val = new TextValue (text2);
	    Var* var = new Var (text1,"filename",val);
	    var->addprop("$INPUTS");
	    current_tool_->addvar (var);
	}
	else {
	    // time to get control back to caller.
	    break;
	}
    }
}

// EX: inspectrum:sync  :$GETVAR outspectrum
void ToolsMgr::loadlinks(FILE* f, int& line) {
    char text1[1024];  char text2[1024];  char text3[1024];

    // current_tool MUST BE SET SO WE CAN POPULATE ITS RECORD
    if (current_tool_ == nil) 
	errormissingtool (currentloadfile_);
    if (GLB_debug) 
	cerr <<  "...LOADING LINKS\n";
    while (fgets(buf_, sizeof(buf_), f) != nil) {
	line++;
	if (pmu_->isacomment(buf_)) continue;
	if (pmu_->match (
	    buf_,"<ident>",":","<ident>",":","$GETVAR","<text>")
	) {
	    pmu_->gettextfield (1,text1);
	    pmu_->gettextfield (3,text2);
	    pmu_->gettextfield (6,text3);
	    if (GLB_debug) cerr 
		<<  "   " << text1 << ":" << text2 << ":$GETVAR "
		<< text3 << "\n";
	    
	    // make sure that the link variable is defined.
	    Var* var = current_tool_->getvar(text1);
	    if (var == nil) {
		cerr 
		    << "LoadLinks: bad link variable '" << text1 
		    << "' for tool `" << current_tool_->getname() 
		    << "'. Ignoring...\n";
	    }
	    else {
		var->addprop("$LINKS");

		// SEMANTICS. These links are deleted in Tool::~Tool.
		Link* link = new Link (buf_, text2, text1, "$GETVAR", text3);
		current_tool_->getlinks()->enq(link);
	    }
	}
	else if (pmu_->match (
	    buf_,"<ident>",":","<ident>",":","$RUN","<text>")
	) {
	    pmu_->gettextfield (1,text1);
	    pmu_->gettextfield (3,text2);
	    pmu_->gettextfield (6,text3);
	    if (GLB_debug) 
		cerr 
		    << "   " << text1 << ":" << text2 << ":$RUN "
		    << text3 << "\n";
	    
	    // make sure that the link variable is defined.
	    Var* var = current_tool_->getvar(text1);
	    if (var == nil) {
		cerr 
		    << "LoadLinks: bad link variable '" << text1 
		    << "' for tool `" << current_tool_->getname() 
		    << "'. Ignoring...\n";
	    }
	    else {
		var->addprop("$LINKS");

		// SEMANTICS
		Link* link = new Link (buf_,text2,text1,"$RUN",text3);
		current_tool_->getlinks()->enq (link);
	    }
	}
	else {
	    // IF WE ARE HERE WE ARE DONE
	    break;
	}
    }
}

void ToolsMgr::loadoutputs(FILE* f, int& line) {
    char text1[1024];  char text2[1024];  char text3[1024];

    // current_tool MUST BE SET SO WE CAN POPULATE ITS RECORD
    if (current_tool_ == nil) 
	errormissingtool (currentloadfile_);
    if (GLB_debug) 
	cerr << "...LOADING OUTPUTS\n";
    while (fgets (buf_, sizeof(buf_), f) != nil) {
	line++;
	if (pmu_->isacomment(buf_)) continue;
	if (pmu_->match (buf_,"<ident>",":","<ident>",":","$RUN","<text>")) {
	    // EX: TransmittedPower:real:$RUN tshow -i ttp $GPRED 
	    pmu_->gettextfield (1,text1);
	    pmu_->gettextfield (3,text2);
	    pmu_->gettextfield (6,text3);
	    if (GLB_debug) cerr
		<< "   " << text1 << ":" << text2 << ":$RUN "
		<< text3 << "\n";
	    // SEMANTICS
	    Value* val = new TextValue (text3);
	    Var* var = new Var (text1, text2, val);
	    var->addprop("$OUTPUTS");  var->addprop("$RUN");
	    current_tool_->addvar(var);
	}
	else {
	    // IF WE ARE HERE WE ARE DONE
	    break;
	}
    }
}

void ToolsMgr::load(const char* path, int nesting) {
    char text1[1024]; 
    int line = 0, errors = 0;

    currentloadfile_ = path;			// UNCONST
    if (GLB_debug) 
	cerr << "LOADING TOOLS FILE `" << path << "'\n";
    FILE* f = fopen (path, "r");  
    if (f == nil) 
	errorfilenotfound (path);
    while (1) {
	if (fgets (buf_,sizeof(buf_),f) == nil) 
	    break;  
	line++;

	// PUT ALL SUBROUTINE CALLS ARE FIRST AND LET THEM FALL THRU
	if (pmu_->match (buf_,"$INPUTS",":"))
	    loadinputs (f, line);
	if (pmu_->match (buf_,"$LINKS",":")) 
	    loadlinks (f, line);
	if (pmu_->match (buf_,"$OUTPUTS",":")) 
	    loadoutputs (f, line);

	// ALL OTHER PATTERNS GO HERE
	if (pmu_->isacomment(buf_)) continue;
	if (pmu_->match (buf_,"$TOOL",":","<ident>")) {
	    pmu_->gettextfield (3,text1);
	    if (GLB_debug) 
		cerr << "   $TOOL: " << text1 << "\n";
	    // SEMANTICS
	    // ...INSERT TOOL INTO TOOL TABLE
	    current_tool_ = new Tool(text1);
	    tools_->insert(current_tool_->getname(), current_tool_);
	    toollist_->enq(current_tool_);
	}
	else if (pmu_->match (buf_,"$PATH",":","<text>")) {
	    pmu_->gettextfield (3,text1);
	    if (GLB_debug) 
		cerr << "   $PATH: " << text1 << "\n";
	    // SEMANTICS
	    current_tool_->setpath (text1);
	}
	else if (pmu_->match (buf_,"$RUN",":","<text>")) {
	    pmu_->gettextfield (3,text1);
	    if (GLB_debug) 
		cerr <<  "   $RUN: " << text1 << "\n";
	    // SEMANTICS
	    current_tool_->setexec (text1);
	}
	else if (pmu_->match (buf_,"$RUNTIME",":","<integer>")) {
	    int runtime = pmu_->getintfield (3);
	    if (GLB_debug) 
	    cerr << "   $RUNTIME: " << runtime << "\n";
	    // SEMANTICS
	    current_tool_->setruntime (runtime);
	}
	//
	// scan for include files with both "<file>" and "file" formats.
	// steps:
	//    1. is it a standard include directive, ie., within < >
	//       yes - extract the filename portion.
	//       1.a. does the filename start with '/'?
	//          yes - ignore include search 
	//    2. scan for env vars and expand
	//    3. load
	//
	else if (
	    pmu_->match (buf_,"$INCLUDE","<name>") ||
	    pmu_->match (buf_,"include","<name>")
	) {
	    pmu_->gettextfield(2, text1);
	    if (GLB_debug) 
		cerr << "   $INCLUDE: " << text1 << "\n";
	    // search and load new file.
	    char newpath[1024];
	    switch(search_file(text1, GLB_file_search_path, newpath)) {
		case 0:
		    load(newpath, nesting+1);
		    break;
		
		case 1:
		    cerr << "Cannot find INCLUDE file " << text1 
			<< " in standard search path (" << path 
			<< ":" << line << ")." << endl;
		    ++errors;
		    break;

		case 2:
		    cerr << "Error opening INCLUDE file " << text1 
			<< " (" << path << ":" << line << ")." << endl;
		    ++errors;
		    break;
		
		default:
		    break;
	    }
	}
	else {
	    // UNRECOGNIZABLE PATTERN 
	    cerr 
		<<  "Ignoring unknown line " << line << " ... " 
		<< buf_ << "\n";
	    ++errors;
	}
	if (errors > 3) {
	    cerr << "Too many errors.  Check your tools file.\n";
	    fclose (f);  
	    exit (1);
	}
    }
    fclose (f);
    return;
}
