//
// report.cc: automated reporting facility
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <iostream.h>
#include <strstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_SYSENT_H
# include <sysent.h>			// USL cfront 3.0 specific?? (for open)
#endif
#include <time.h>
#include <unistd.h>
#include <sys/wait.h>


#include "exp.h"
#include "global.h"
#include "misc.h"
#include "pmu.h"
#include "report.h"
#include "tool.h"
#include "utils.h"
#include "value.h"
#include "variable.h"
#include "xrtypes.h"

/****************************************************************************
 *
 *  REPORTERS
 *
 ****************************************************************************/

Reporter::Reporter (const char* tname, const char* varname) {
    tname_ = strcpy_with_alloc(tname);
    varname_ = strcpy_with_alloc (varname);
    runtime_ = 0;
}

Reporter::~Reporter() {
    delete[] tname_; tname_ = nil;
    delete[] varname_; varname_ = nil;
}

/*************************************************************************/

IVReporter::IVReporter(
    const char* tname, const char* varname
) : Reporter(tname, varname) { }

IVReporter::~IVReporter() { }

Reporter* IVReporter::clone() const {
    return new IVReporter(gettoolname(), getvarname());
}

const Value* IVReporter::reportval(int iteration) const {
    ToolInstance* ti = GLB_expmgr->gettoolinstance(tname_, iteration);
    assume(ti != nil);
    return ti->getvarval(varname_);
}

/*************************************************************************/

RUNReporter::RUNReporter(
    const char* tname, const char* varname
) : Reporter(tname, varname), program_text_(nil) {
    // CREATE ERROR VALUES
    errorfopen_  = new TextValue ("\"ERROR Opening reporter file\"!");
    errorfork_   = new TextValue ("\"ERROR Running reporter\"!");
    errorgetres_ = new TextValue ("\"ERROR Getting value from reporter!\"");
    // GET THE ORIGINAL TOOL INSTANCE. INDEPENDENT OF THE ITERATION
    const ToolInstance* ti = GLB_expmgr->gettoolinstance(tname); 
    assume(ti != nil);
    // CREATE A VALUE OF THE RIGHT TYPE
    const Var* cvar = ti->getvar(varname);  assume (cvar != nil);
    val_ = cvar->genval();
    // THE VALUE OF varname IS THE program...EXPAND THE program TEXT
    // EX: tshow -i ttp $GPRED 
    const Value* cval = ti->getvarval (varname);  assume (cval != nil);

    char token[1024];  
    char buf[1024];
    strcpy (buf, cval->gettext());
    int pos = 0;  

    // get the first token, the program to run. rest of text is also saved.

    pos = GLB_pmu->match(buf, "<name>");
    assert(pos != 0);
    GLB_pmu->gettextfield(1, token);

    // IF ITS THE FIRST TOKEN ITS A PROGRAM NAME...FIND ITS FULL PATHNAME
    const Tool* tool = GLB_toolsmgr->gettool (token);  assume (tool != nil);
    program_ = strcpy_with_alloc(tool->getpath());

    // ESTIMATE OF GETTING RESULTS IS ESTIMATE PROGRAM RUN TIME
    runtime_ = tool->getruntime();

    // rest of the text is saved for expansion at runtime.
    program_args_ = strcpy_with_alloc(&buf[pos]);
}

RUNReporter::~RUNReporter() {
    delete errorfopen_;
    delete errorfork_;
    delete errorgetres_;
    delete val_;
    delete[] program_; program_ = nil;
    delete[] program_args_; program_args_ = nil;
    delete[] program_text_; program_text_ = nil;
}

Reporter* RUNReporter::clone() const {
    return new RUNReporter(gettoolname(), getvarname());
}

void RUNReporter::genexec(int iteration) {
    char tmpbuf[1024];

    delete[] program_text_; program_text_ = nil;

    strcpy(tmpbuf, program_args_);

    const ToolInstance* ti = GLB_expmgr->gettoolinstance(tname_, iteration);
    assume(ti != nil);

    // substitute for the key words '$KEYWORD'. Highly inefficient scanning,
    // but what the hell ... this is what computers are for, isn't it?
    // FIX the scanning and substitution algorithm. Take it from tcsh/csh.

    // proposed algorithm:
    //    char buf[tmpbuf];
    //    strcpy(buf, program_args_);
    //    while (token = get_keyword(program_args_)) {
    //        check_keyword(token);
    //        substitute_in_place(buf, token);
    //    }
    //


    // current scanning strategy. First scan and substitute for all simple
    // keywords like $CPU $PATH and weird ones like $GPRED $GPREDNS, which
    // don't have any qualifiers such as tool name.

    char tmpbuf2[1024];
    // SUBSTITUTE FOR $GPRED 
    if (strstr (tmpbuf, "$GPRED")) {
	char tmpbuf3[1024];

	tmpbuf3[0] = '\0';
	const Stq* job = GLB_expmgr->get_job_sequence(iteration); 
	assume (job != nil);
	int append = 1;
	for (int i=1;i<=job->size();i++) {
	    const ToolInstance *temp_ti = (ToolInstance*)job->cycle();
	    if (temp_ti == ti) 
		append = 0;
	    if (append) { 
		if (temp_ti->getgpath() != nil) { 
		    strcat (tmpbuf3, temp_ti->getgpath()); 
		    strcat (tmpbuf3, " "); 
		}
	    }
	}

	strsub("$GPRED", tmpbuf3, tmpbuf, tmpbuf2);
	strcpy(tmpbuf, tmpbuf2);
    }

    if (strstr (tmpbuf, "$GPREDNS")) {
	char tmpbuf3[1024];

	tmpbuf3[0] = '\0';
	const Stq* job = GLB_expmgr->get_job_sequence(iteration); 
	assume (job != nil);
	// SCAN THE EXPERIMENT SEQUENCE TILL YOU GET TO ti...APPENDING 
	// gfile PATHS
	for (int i=1;i<=job->size();i++) {
	    int append = 1;
	    const ToolInstance *temp_ti = (ToolInstance*)job->cycle();
	    if (temp_ti == ti) 
		append = 0;
	    const char* toolname = temp_ti->getname();
	    if (strcmp(toolname,"sync")==0) 
		append = 0;
	    if (append) {
		if (temp_ti->getgpath() != nil) { 
		    strcat (tmpbuf3, temp_ti->getgpath()); 
		    strcat (tmpbuf3, " "); 
		}
	    }
	}
	strsub("$GPREDNS", tmpbuf3, tmpbuf, tmpbuf2);
	strcpy(tmpbuf, tmpbuf2);
    }

    if (strstr (tmpbuf, "$ITERATION")) {
	char tmpbuf3[1024];

	tmpbuf3[0] = '\0';
	sprintf(tmpbuf3, "%d", iteration);
	strsub("$ITERATION", tmpbuf3, tmpbuf, tmpbuf2);
	strcpy(tmpbuf, tmpbuf2);
    }


    // now the qualified ones. All variables have to be fully qualified
    // (such as xlith.gap), else these are simply passed off as strings
    // to the output driver.  $GFILE is a special case, where by itself
    // it is assumed to be for the current tool, whatever that may be.

    // use the global PMU instance to scan.
    int pos = 0;  
    char tmpbuf3[1024];
    strcpy(tmpbuf3, tmpbuf);
    char const* rest = tmpbuf3;
    char toolname[256];
    char varname[256];
    char match[256];
    while(pos = GLB_pmu->match(rest, "<name>")) {
	rest = &rest[pos];
	GLB_pmu->gettextfield(1, match);
	if (GLB_pmu->match(match, "<ident>", ".", "<variable>")) {
	    GLB_pmu->gettextfield(1, toolname);
	    GLB_pmu->gettextfield(3, varname);

	    // check sanity. The only tool allowed is the current one.
	    const char* tname = ti->getname();
	    if(strcmp(tname, toolname) == 0) {
		const Value* value = ti->getvarval(varname);
		if(value != nil) {
		    strsub(match, value->getstrval(), tmpbuf, tmpbuf2);
		    strcpy(tmpbuf, tmpbuf2);
		} 
		else {
		    cerr 
			<< "RUNReport: bad variable name in tools file: ``"
			<< match << "''. Passed onto the output driver.\n";
		}
	    }
	    else {
		cerr 
		    << "RUNReport: bad tool name in tools file: ``"
		    << match << "''. Passed onto the output driver.\n";
	    }
	}
    }

    // catch the sole $GFILE if it's still remains.
    if (strstr (tmpbuf, "$GFILE")) {
	strsub("$GFILE", ti->getgpath(), tmpbuf, tmpbuf2);
	strcpy(tmpbuf, tmpbuf2);
    }


    // buf NOW CONTAINS THE EXPANDED PROGRAM
    strcpy(tmpbuf2, program_);
    strcat(tmpbuf2, " ");
    strcat(tmpbuf2, tmpbuf);
    program_text_ = strcpy_with_alloc (tmpbuf2);
}

const Value *RUNReporter::reportval(int iteration) const {
    char path[1024];
    char buf[1024];

    // pretend to a non-const member.
    ((RUNReporter*)this)->genexec(iteration);		// UNCONST

    if (GLB_coldebug)
	cerr << "   RUN Reporter for `" << getvarname() << "'\n";

    // GET UNUSED FILE NAME TO COPY PROGRAM INTO AND WRITE RETURN VALUE
    ::getunusedpath (path, "exp.", 1);  // 1 AT END MEANS CREATE THE FILE
    if (GLB_coldebug)
	cerr << "      Unused path `" << path << "' returned\n";

    int infd = -1, outfd = -1;
    // RUN THE JOB...WAITING FOR IT TO FINISH
    switch (fork()) {
	case -1 : 
	    return errorfork_; // TOO MANY PROCESSES OR OUT OF MEMORY
	    break;

	case  0 : 
	    // set up redirection for child process.
	    if ((outfd = open (path, O_WRONLY | O_CREAT)) == -1) {
		return errorfopen_;
	    }
	    if ((infd = open ("/dev/null", O_RDONLY)) == -1) {
		return errorfopen_;
	    }
	    if (GLB_coldebug)
		cerr 
		    << "      Command " << program_text_ 
		    << " written to `" << path << "'\n";
	    close (fileno(stdin)); dup2(infd, fileno(stdin));
	    close (fileno(stdout)); dup2(outfd, fileno(stdout));
	    close (fileno(stderr)); dup2(outfd, fileno(stderr));
	    execlp ("/bin/sh", "/bin/sh", "-c", program_text_, (char*)0); 
	    perror("CANNOT EXEC CHILD");
	    _exit (1);
	    break;

	default : 
	    wait(nil); 
	    break;
    }
    GLB_stats->ranprogram();
    // EXTRACT THE RESULT FROM path, PACKAGE AND RETURN THE RESULT
    FILE* fp = fopen (path, "r");
    if (fp == nil)
	return errorgetres_;
    if (fgets (buf, sizeof(buf), fp) == nil) 
	return errorgetres_;
    if (GLB_coldebug)
	cerr << "      Read return value `" << buf << "'\n";
    val_->setval (buf);
    fclose (fp);
    GLB_stats->searchedoutputfile();
    // GET RID OF THE FILE
    unlink (path);
    return val_;
}

/****************************************************************************
 *
 *  REPORTS
 *
 ***************************************************************************/

void Report::errorreportfile(const char* file) {
    cerr
	<< "Unable to open and append to the report file `" << file 
	<< "'.\nYou may not have write permission.\n"
	<< "Check the file and try running the experiment again.\n";
    exit (1);
}

Report::Report(const char* name, const char* path) {
    name_ = strcpy_with_alloc (name);
    path_ = strcpy_with_alloc (path);
    reporters_ = new Stq ();
    row_= new Stq ();
    runtime_ = 0;
}

Report::~Report() {
    delete[] name_; name_ = nil;
    delete[] path_; path_ = nil;
    delete reporters_; reporters_ = nil;
    delete row_; row_ = nil;
}

Report* Report::clone() const {
    Report* newreport = new Report(getname(), getpath());
    unsigned size = reporters_->size();
    while(size--) {
	const Reporter* reporter = (Reporter*)reporters_->cycle();
	Reporter* newreporter = reporter->clone();
	newreport->enqreporter(newreporter);
    }
    return newreport;
}

void Report::enqreporter(const Reporter* reporter) { 
    reporters_->enq(reporter); 
    runtime_ += reporter->getruntime(); 
}

void Report::reset() { row_->reset(); }

void Report::setpath(const char* path) {
    // APPEND NUMERIC SUFFIX FROM ORIGINAL NAME ONTO NEW ONE
    char* oldpath = path_;
    int pos = strlen(oldpath)-1; 
    while (isdigit(oldpath[pos])) 
	pos--;
    char buf[1024];  strcpy (buf, path);  strcat (buf, &oldpath[pos+1]);
    path_ = strcpy_with_alloc (buf);
    delete[] oldpath;
}

void Report::collectresults (int iteration) {
    if (GLB_coldebug) cerr
	<< "Collecting " << reporters_->size() << " results for report `"
	<< name_ << "'.\n";
    for (int i=1;i<=reporters_->size();i++) {
	Reporter *reporter = (Reporter*)reporters_->cycle();
	const Value *result = reporter->reportval(iteration);
	if (GLB_coldebug) cerr 
	    << "   Reporter `" << reporter->getvarname() << " returns `" 
	    << result->getstrval() << "'.\n";
	row_->enq (result); 
    }
}
void Report::printrow () {
    char buf[256];  buf[0] = '\0';
    FILE* fp = fopen (path_, "a");  
    if (fp == 0) { errorreportfile(path_); return; }
    for (int i=1;i<=row_->size();i++) {
	const Value *val = (Value*)row_->cycle();
	strcat (buf,val->getstrval ());  strcat (buf," ");
    }
    strcat (buf,"\n");
    if (GLB_coldebug) cerr 
	<<  "REPORT: Writing into file `" << path_ << "' the value `"
	<< buf << "'\n";
    fprintf(fp, buf);
    fclose(fp);
}
