//
// exp2gfile.cc: extract the gfile corresponding to a/set-of TOOL/s
//
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1996 Mumit Khan
//
//

// ====================================================================== //

#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include <fstream.h>
#include <string>
#include <list>

#include "exp.h"
#include "global.h"
#include "logger.h"
#include "job.h"
#include "misc.h"
#include "pmu.h"
#include "tool.h"
#include "utils.h"
#include "value.h"
#include "variable.h"

// ====================================================================== //

static void usage();
static int parse_args(int argc, char* argv[], bool& do_all,
    string& toolsfile, string& expfile, list<string>& toolnames
);
static int export_gfiles(const string& toolsfile, const string& expfile,
    bool do_all, const list<string>& toolnames
);

// ====================================================================== //

class ThisExpMgr : public ExpMgr {
public:
    ThisExpMgr() { }
    virtual ~ThisExpMgr() { }
    virtual const Exp* exp() const { return exp_; }
    virtual Exp* exp() { return exp_; }
};

// ====================================================================== //

//
// global variables. Need this for the EXCON main globals. FIX/CLEAN/TODO.
//

bool g_verbose = false;
bool GLB_debug    = false;  // TURN ON DEBUGGING INFO
bool GLB_resdebug = false;  // DEBUG RESOLVE STEPS
bool GLB_repdebug = false;  // DEBUG REPORT CONSTRUCTION
bool GLB_rundebug = false;  // DEBUG EXPERIMENT EXECUTION
bool GLB_coldebug = false;  // DEBUG REPORT COLLECTION
char* GLB_programname; // NAME USED TO INVOKE THIS PROGRAM
Pmu* GLB_pmu;
ExpMgr* GLB_expmgr;
ToolsMgr* GLB_toolsmgr;
CleanupMgr* GLB_cleanupmgr;
Stats* GLB_stats;
JobMgr* GLB_jobmgr;
Logger* GLB_logger;
Stq* GLB_file_search_path = 0;


// ====================================================================== //

int main(int argc, char** argv) {
    string toolsfile;
    string expfile;
    list<string> toolnames;
    bool do_all = false;

    parse_args(
        argc, argv, do_all, toolsfile, expfile, toolnames
    );

    return export_gfiles(toolsfile, expfile, do_all, toolnames);
}


// ====================================================================== //

#if 0
static int get_all_tools(const ToolsMgr& toolsmgr, list<string>& tools) {
    const Stq* all_tools = toolsmgr.gettools();
    for (int i = 0; i < all_tools->size(); ++i) {
	const Tool* tool = (const Tool*)all_tools->cycle();
	tools.push_back(tool->getname());
    }
    return all_tools->size();
}
#endif

static int export_gfile(const ToolInstance* ti) {
    string instname(ti->getinstname());
    string gfile = instname + ".g";
    ofstream ofp(gfile.c_str());
    if (!ofp) {
	cerr << "exp2gfile: cannot open file `" << gfile << "'." << endl;
	return 1;
    }

    if (g_verbose) {
	cerr << "Tool `" << instname << "' variables ==> `" << gfile
	    << "'." << endl;
    }

    ofp << "######" << endl << "#" << endl
	<< "# TOOL : " << instname << endl
	<< "# GFILE automatically generated from EXCON TOOLS file" << endl
	<< "#" << endl << "######" << endl << endl; 

    const Tool* tool = ti->gettool();
    for (int i = 1; i <= tool->nvars(); ++i) {
	const Var* var = tool->getvar(i);  
	assume (var != nil);
	if (var->hasprop("$INPUTS")) {
	    const Value* val = ti->getvarval(var->getname());  
	    assume (val != nil);
	    ofp << var->getname() << "\t=\t" << val->getstrval() << endl;
	}
    }
    ofp << endl << "#" << endl << "#" << endl << "######" << endl << endl; 
    return 0;
}

//
// export_gfiles:
//
// read an EXCON toolsfile and exports the TOOL variables as gfile params.
//
static int export_gfiles(const string& toolsfile, const string& expfile,
    bool do_all, const list<string>& toolnames
) {
    ToolsMgr toolsmgr;
    GLB_toolsmgr = &toolsmgr;
    toolsmgr.load(toolsfile.c_str());
    ThisExpMgr expmgr;
    GLB_expmgr = &expmgr;
    expmgr.load(expfile.c_str());

    GLB_cleanupmgr  = new CleanupMgr (); 
    GLB_pmu         = new Pmu        ();
    GLB_stats       = new Stats      ();
    GLB_logger      = new Logger     (nil, Logger::nolog, false);

    //
    // now create the job sequence for the first iteration. That's all
    // we can do w/out having to actually run the job.
    //
    Exp* exp = expmgr.exp();
    exp->setup_links(exp->get_job_sequence(0));
    Stq* job = exp->get_next_job(1);
    assert(job != 0);

    exp->resolveunique(job);
    exp->resolveunique(job);
    unsigned jobsize = job->size();
    while(jobsize--) {
	ToolInstance* ti = (ToolInstance*)job->cycle();
	exp->resolvelinks(ti);
	export_gfile(ti);
    }

#if 0
    list<string> export_list;
    if (do_all) {
	get_all_tools(toolsmgr, export_list);
    } else {
	export_list = toolnames;
    }

    list<string>::const_iterator it = export_list.begin();
    for(; it != export_list.end(); ++it) {
	const string& toolname = *it;
	export_gfile(toolsmgr, expmgr, toolname);
    }
#endif

    return 0;
}

// ====================================================================== //

static void usage () {
    cerr << endl
        // << "Usage: exp2gfile [-hv] -t toolsfile -e expfile [-a | TOOL_NAMES ]"
        << "Usage: exp2gfile [-hv] -t toolsfile -e expfile"
        << endl << endl
        << "    -v:     turn on verbose mode" << endl
        << "    -h:     print this info" << endl
        << endl
        << "    -t:     EXCON tools file to load" << endl
        << "    -e:     EXCON experiment file to load" << endl
	// << "    -a:     export ALL tools in Tools file"
	<< endl
        // << " TOOLS:     Name of the tools to export (if -a not specified)"
        << endl << endl
        << "Examples:\n"
        // << "    exp2gfile -v -t excon.tools -e excon.exp -a" << endl
        // << "    exp2gfile -t excon.tools -e excon.exp xprocess maskmaker" << endl
        << "    exp2gfile -v -t excon.tools -e excon.exp" << endl
	<< endl;
    return;
}

static int parse_args(int argc, char* argv[], bool& do_all,
    string& toolsfile, string& expfile, list<string>& toolnames
) {
    int errcnt = 0;
    extern int optind;
    extern char *optarg;

    int c;

    do_all = false;

    while ((c = getopt (argc, argv, "hvat:e:")) != EOF) {
        switch (c) {
            case 't':
		toolsfile = optarg;
                break;

            case 'e':
		expfile = optarg;
                break;

            case 'a':
                do_all = true;
                break;

            case 'v':
                g_verbose = true;
                break;

            case 'h':
		usage();
		exit(0);
                break;

            default:
                cerr << "exp2gfile:: Error in command line.\n";
                usage ();
                exit (1);
                break;
        }
    }

    if (argc > optind) {
	do_all = false;
	for(int i = optind; i < argc; ++i) {
	    toolnames.push_back(argv[i]);
	}
    }

#if 0
    if (!do_all && toolnames.size() == 0) {
        cerr << "exp2gfile: Must specify -a or toolnames to export"
            << endl;
        usage();
        exit(1);
    }
#endif

    if (toolsfile.size() == 0 || expfile.size() == 0) {
        cerr << "exp2gfile: Must specify tools AND exp files to export from"
            << endl;
        usage();
        exit(1);
    }
    return (errcnt) ? -1 : 0;
}

// ====================================================================== //
