//
// exp.h: Interface to EXCON experiment manager
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//

#ifndef excon_batch_experiment_h
#define excon_batch_experiment_h

#include "xrtypes.h"

/**************************************************************************/

class FreeVariable;
class FreeVariableGroup;
class Hashtable;
class Stq;
class ToolInstance;

/**************************************************************************/


/***************************************************************************
 *
 *  EXPERIMENTS
 *
 ***************************************************************************/

class Exp {
public:
    Exp();
    virtual ~Exp();
    virtual  bool expfileok() const;

    virtual  int buildexpseq(const char* exptext);
    virtual ToolInstance* gettoolinstance(
	const char* name, int iteration = 0
    ) const;
    virtual Stq* get_job_sequence(int iteration) const;

    virtual void setup_links(const Stq* sequence);
    virtual void resolvelinks(const Stq* sequence);
    virtual void resolveunique(const Stq* sequence);

    // experiment control loop
    virtual void setfvgroup(
       const char* tool, const char* var, const char* others, 
       const char* origline
    );
    virtual void setsetvar(
       const char* tool, const char* var, const char* valueset, 
       const char* origline
    );
    virtual void setloopvar(
       const char* tool, const char* var, const char* fromtext, 
       const char* totext, const char* incrtext, const char* origline
    );
    virtual void setrandomvar(
       const char* tool, const char* var, int nvalues,
       const char* fromtext, const char* totext, bool genseed, long seed,
       bool dosort, const char* origline
    );
    virtual FreeVariable* getfreevar(
	const char* tool, const char* var
    ) const;
    virtual FreeVariableGroup* getfvgroup(
	const char* tool, const char* var
    ) const;
    virtual void run();

    // reports
    virtual void reportresults(int future = 0) const;
    virtual void setoutput(const char* name);
    virtual void unlinkreports();
    virtual void settablecolumns(const char* tablename, const char* columns);
    virtual void dump() const;

// protected:
    virtual void resolvelinks(ToolInstance* ti);

    virtual int gettoolnum(const char* name, int &pos) const;
    virtual void addtool(const char* name);    

    virtual Stq* get_next_job(int iteration);
    virtual void delete_job_sequence(int iteration);

    virtual int getnruns() const;

    virtual void collectresults(int iteration);
    virtual void progresssetup(int nruns);
    virtual void progressreport(int runs);

    virtual void errormissingtool(const char* name);
    virtual void errormissingvar(const char* name, const char* toolname);
    virtual void errorforkfailed(const char* file);
    virtual void erroraddtool(const char* file);
    virtual void errorrunopen(const char* file);

private:
    virtual char* genscript(const Stq* job, int iteration);

protected:
    // EXPERIMENT
    Stq* expseq_;		// TOOLS IN EXEC SEQUENCE: LIST OF ToolInstance
    Stq* joblist_;		// JOB SEQUENCE: LIST OF (List of ToolInstance)
    // FREE VARIABLES
    Stq* fvgroups_;		// LIST OF FreeVarGroup*
    // PROGRESS
    int prstart_;		// TIME EXPERIMENT STARTED
    int prnext_;		// TIME OF NEXT RUN
    int prestimate_;		// ESTIMATED TIME TOTAL RUN TO COMPLETE
    int prnruns_;		// TOTAL NUMBER OF RUNS
    // REPORTS
    Stq* reports_;		// TABULAR REPORTS IN A LIST
    // STATE DATA
    char* expfile_;		// NAME OF EXPERIMENT FILE
    char* toolsfile_;		// NAME OF TOOLS FILE
};

/***************************************************************************
 *
 *  EXPERIMENT MANAGER
 *
 ***************************************************************************/


class ExpMgr {
public:
    ExpMgr();
    virtual ~ExpMgr();
    virtual void load(const char* path, int nesting = 0); 
    virtual void setoutput(const char* name);
    virtual void unlinkreports();
    virtual void run();
    virtual void reportresults();
    virtual ToolInstance* gettoolinstance(
	const char* tname, int iteration = 0
    ) const;
    virtual Stq* get_job_sequence(int iteration) const;
protected:
    virtual void errorfilenotfound(const char* path);
    virtual void setsystemparams(const char* param, const char* value);

protected:
    Exp* exp_;			// THE EXPERIMENT
};

#endif/*excon_batch_experiment_h*/
