//
// exp.cc: experiment manager.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <iostream.h>
#include <iomanip.h>
#include <math.h>				// for exp10()
#include <memory.h>
#include <stdlib.h>
#include <string.h>
#include <strstream.h>
#include <time.h>
#include <unistd.h>				// for unlink(2)

#include "exp.h"
#include "freevars.h"
#include "global.h"
#include "job.h"
#include "link.h"
#include "logger.h"
#include "misc.h"
#include "pmu.h"
#include "report.h"
#include "tool.h"
#include "utils.h"
#include "value.h"
#include "variable.h"


/************************************************************************/

Exp::Exp () {
    expseq_   = new Stq;
    joblist_ = new Stq;
    fvgroups_ = new Stq;
    reports_  = new Stq;
    prstart_ = (int)time(0);

    joblist_->enq(expseq_);		// iteration 0.

    expfile_ = nil;
    toolsfile_ = nil;
}

Exp::~Exp () {
    unsigned size = reports_->size();
    while(size--) {
	Report* report = (Report*)reports_->cycle();
	delete report; report = nil;
    }
    delete reports_; reports_ = nil;

    size = fvgroups_->size();
    while (size--) {
	FreeVariableGroup* fvgroup = (FreeVariableGroup*)fvgroups_->pop();
	delete fvgroup; fvgroup = nil;
    }
    delete fvgroups_; fvgroups_ = nil;

    size = joblist_->size();
    while(size--) {
	Stq* job = (Stq*)joblist_->pop();
	unsigned size2 = job->size();
	while (size2--) {
	    ToolInstance* ti = (ToolInstance*)job->pop();
	    delete ti; ti = nil;
	}
	delete job; job = nil;
    }
    delete joblist_; joblist_ = nil;

    // elements of expseq_ are already deleted when joblist_ is killed.
    delete expseq_; expseq_ = nil;

    delete[] expfile_; expfile_ = nil;
    delete[] toolsfile_; toolsfile_ = nil;
}

//
// various error reporting routines. All of these EXIT with a code of 1.
//

void Exp::erroraddtool (const char* name) {
    cerr 
	<< "Unable to find the tool named '" << name << "'.\n"
	<< "Add an entry for '" << name << "' into the tools file and "
	<< "try again.\n";
    exit (1);
}


void Exp::errorrunopen (const char* file) {
    cerr 
	<< "Could not create file '" << file << "' needed to hold the "
	<< "experiment script.\nCheck the file and try again.\n";
    exit (1);
}


void Exp::errorforkfailed (const char* file) { 
    cerr 
	<< "Could not create a UNIX process to run the experiment in file '"
	<< file << "'.\nEither there are too many processes running or we "
	<< "are out of memory.\nEliminate some processes and try again.\n";
    exit (1);
}


void Exp::errormissingtool (const char* name) {
    cerr 
	<< "The tool named '" << name << "' is referenced but has not been "
	<< "defined.\nThe reference was made in the experiment file in the "
	<< "table column definition statement.\nFirst check your spelling.\n"
	<< "Next make sure that tool '" << name << "' is defined in the "
	<< "tools file and try again.";
    exit (1);
}

void Exp::errormissingvar (const char* name, const char* toolname) {
    cerr 
	<< "The variable '" << toolname << "." << name << "' is referenced "
	<< "but has not been defined.\n"
	<< "The reference was made in the experiment file in the table "
	<< "column definition statement.\n"
	<< "First check your spelling.\n"
	<< "Next make sure that variable '" << name << "' is defined in the "
	<< "tools file and try again.\n";
    exit (1);
}


//
// Method:	Exp::getnruns:
//
// Caller:	Exp::run
// Parameters:	
//    job:	the job sequence, ie list of ToolInstance* to be exec'd.
//    iteration:the iteration id of the run (1 == first).
//
// Description:
// 		get the total number of runs needed to finish the experiment.
//
// Return code: name of the script. Caller delete storage.
//
// Side Effects:
//

int Exp::getnruns() const {
    int nruns = 1;
    unsigned ngroups = fvgroups_->size();
    while(ngroups--) {
	const FreeVariableGroup* gr = (FreeVariableGroup*)fvgroups_->cycle();
	nruns *= gr->getnruns();
    }
    return nruns;
}

//
// Method:	Exp::setoutput
//
// Caller:	ExpMgr::setoutput
//
// Parameters:	
//    name:	root name of the report file(s).
//
// Description:
// 		sets the ``report'' filename. This is usually the filename 
//		specified on the command line.
//
// Return code:	void
//
// Side Effects:
//

void Exp::setoutput (const char* name) {
    unsigned size = reports_->size();
    while(size--) {
	Report* report = (Report*)reports_->cycle();
	report->setpath (name);
    }
}

//
// Method:	Exp::reportresults
//
// Callers:	Exp::run, ExpMgr::reportresults
//
// Parameters:	
//    future:	If future == 1 is set, then tell the user where these will 
//		be, else say they already are.
//
// Description:
// 		tell the user where, if any, the report file(s) is (are). 
// Return code:	void
//
// Side Effects:
//

void Exp::reportresults (int future) const {
    int size = reports_->size();
    if (size == 1) {
	const Report* report = (Report*)reports_->top();
	if (future) 
	    cout 
		<< "Your results will be in the file '" 
		<< report->getpath() << "'.\n";
	else 
	    cout 
		<< "Your results are in the file '" << report->getpath()
		<< "'.\n";
	cout << flush;
    } 
    else if (size > 1) {
	if (future) 
	    cout << "Your results will be in the files ";
	else 
	    cout << "Your results are in the files ";
	for (int i = 1; i <= size; i++) {
	    const Report* report = (Report*)reports_->cycle();
	    if (i < size-1) 
		cout <<  "'" << report->getpath() << "', ";
	    else if (i == size-1) 
		cout << "'" << report->getpath() << "' and ";
	    else 
		cout <<  "'" << report->getpath() << "'\n";
	}
	cout << flush;
    }
}


//
// Method:	Exp::unlinkreports
//
// Caller:	Exp::run
//
// Parameters:
//
// Description:
// 		Make sure the report files don't exist by unlinking them. 
//		This is to make sure we end up with a clean set of report 
//		files for this run, instead of accidentally appending to 
//		old report files of the same names.
//
// Return code:	void
//
// Side Effects:
//

void Exp::unlinkreports () {
    unsigned size = reports_->size();
    while(size--) {
	const Report* report = (Report*)reports_->cycle();
	unlink (report->getpath());
    }
}

//
// Method:	Exp::getfreevar
//
// Caller:	Exp::setfvgroup
//
// Parameters:	
//    tiname:	name of the tool instance (eg., filter3)
//    varname:	name of the variable (eg., thickness)
//
// Description:
// 		given a tool instance name and a variable name defined 
//		for that tool, ie., the (tiname, varname) tuple, return 
//		the FreeVariable associated with that tuple. Return 
//		``nil'' if none found/defined. 
//
// Return code:	FreeVariable* or nil if not found.
//
// Side Effects:
//

FreeVariable* Exp::getfreevar(const char* tiname, const char* var) const{
    FreeVariable* fvret = nil;   
    unsigned ngroups = fvgroups_->size();
    while(ngroups--) {
	const FreeVariableGroup* gr = (FreeVariableGroup*)fvgroups_->cycle();
	const Stq* freevars = gr->getgroup();
	unsigned nfvars = freevars->size();
	while(nfvars--) {
	    FreeVariable* fv = (FreeVariable*)freevars->cycle();
	    if (
		strcmp(fv->gettool(), tiname)==0 && 
		strcmp(fv->getvar(), var)==0 
	    ) {
		assert(fvret == nil);		// ie., it's the only match!
		fvret = fv;
	    }
	}
    }
    return fvret;
}

//
// Method:	Exp::getfvgroup
//
// Caller:	Exp::setloopvar, Exp::setsetvar, Exp::setfvgroup
//
// Parameters:	
//    tiname:	name of the tool instance (eg., filter3)
//    varname:	name of the variable (eg., thickness)
//
// Description:
// 		given a tool instance name and a variable name defined 
//		for that tool, ie., the (tiname, varname) tuple, return 
//		the FreeVariableGroup that contains the FreeVariable 
//		associated with that tuple. Return ``nil'' if none 
//		found/defined. 
// Return code:	FreeVariableGroup* or nil if not found.
//
// Side Effects:
//

FreeVariableGroup* Exp::getfvgroup(
    const char* tiname, const char* var
) const{
    FreeVariableGroup* grret = nil;   
    unsigned ngroups = fvgroups_->size();
    while(ngroups--) {
	FreeVariableGroup* gr = (FreeVariableGroup*)fvgroups_->cycle();
	const Stq* freevars = gr->getgroup();
	unsigned nfvars = freevars->size();
	while(nfvars--) {
	    const FreeVariable* fv = (FreeVariable*)freevars->cycle();
	    if (
		strcmp(fv->gettool(), tiname)==0 && 
		strcmp(fv->getvar(), var)==0 
	    ) {
		assert(grret == nil);		// ie., it's the only match!
		grret = gr;
	    }
	}
    }
    return grret;
}


//
// Method:	Exp::run
//
// Caller:	main
//
// Parameters:	
//
// Description:
// 		now that all the Experiment has been loaded and ready to 
//		go, run the sequence in a tight loop. Returns when ALL the 
//		iterations are complete and ALL the reports are done. it 
//		also does the cleanup of intermediate files created by 
//		applications (minus those explicitly saved, of course).
//
// Return code:	void
//
// Side Effects:
//


#ifdef RUN_MULTIPLE
void Exp::run () {
    // SETUP REPORT AND MAKE INITIAL REPORT
    progresssetup(getnruns());
    cout << "\n";
    progressreport(0);
    reportresults (1);

    // do a preliminary resolution of the links.
    setup_links(get_job_sequence(0));		// get expseq_

    int nruns = 0;
    int iteration = 0;
    bool morejobs = true;
    while (true) {
	while (GLB_jobmgr->freecpu() && morejobs) {
	    iteration++;
	    Stq* job = get_next_job(iteration);
	    if (!job) {		// no more jobs left to do!
		morejobs = false;
		--iteration;
		break;
	    }
	    resolveunique(job);
	    resolvelinks(job);
	    char* scriptfile = genscript (job, iteration);
	    if (GLB_rundebug) {
		cerr 
		    << "Executing iteration `" << iteration << "', script `"
		    << scriptfile << "'\n"; 
	    }
	    GLB_jobmgr->execute_job(scriptfile, iteration);
	    delete[] scriptfile; scriptfile = nil;
	}
	assume(iteration != 0);		// at least one job must be started.

	if (GLB_jobmgr->njobs_running() == 0 && !morejobs) {
	    break;
	}

	// BLOCK waiting for job(s) to complete.
	GLB_jobmgr->wait_for_completed_job();

	int which_iteration = -1;
	int status = -1;
	while ((which_iteration = GLB_jobmgr->reap_job(status)) != -1) {
	    if (GLB_rundebug) {
		cerr << "reaping iteration " << which_iteration << "." << endl;
	    }
	    // COLLECT RESULTS
	    collectresults (which_iteration);

	    GLB_cleanupmgr->cleanup (which_iteration);

	    // REPORT ON PROGRESS
	    progressreport (++nruns);

	    // now kill the job sequence. BUGGY BUGGY
	    delete_job_sequence(which_iteration);
	}
    }
}
#else /*RUN_MULTIPLE*/
void Exp::run () {
    progresssetup(getnruns());
    progressreport(0);
    reportresults (1);

    // do a preliminary resolution of the links.
    setup_links(get_job_sequence(0));		// get expseq_

    int nruns = 0;
    int iteration = 0;
    bool morejobs = true;
    while(morejobs) {
	iteration++;
	Stq* job = get_next_job(iteration);
	if (!job) {		// no more jobs left to do!
	    morejobs = false;
	    --iteration;
	    break;
	}
	resolveunique(job);
	unsigned jobsize = job->size();
	while(jobsize--) {
	    ToolInstance* ti = (ToolInstance*)job->cycle();
	    resolvelinks(ti);
	    ti->genexec();
	    const char* exec_str = ti->getexec();
	    if (ti->getgpath() != nil) 
		ti->writegfile();
	    GLB_jobmgr->execute_job(exec_str, (int)ti);
	    GLB_jobmgr->wait_for_completed_job();
	    int status = -1;
	    int which_ti = GLB_jobmgr->reap_job(status);
	    assert((int)ti == which_ti);
	}

	int which_iteration = iteration;
	// COLLECT RESULTS
	collectresults (which_iteration);
	GLB_cleanupmgr->cleanup (which_iteration);

	// REPORT ON PROGRESS
	progressreport (++nruns);

	// now kill the job sequence. BUGGY BUGGY
	delete_job_sequence(which_iteration);
    }
}
#endif/*!RUN_MULTIPLE*/


//
// Method:	Exp::settablecolumns
//
// Caller:	ExpMgr::load
//
// Parameters:	
//    tablename:name of the table (eg., table1, table2, etc)
//    columns:	the text that defines the table columns
//
// Description:
//		Given a table name and the text the defines the columns,
//		add the appropriate variables from appropriate tool 
//		instances to the final report. If the specified table
//		does not exist, create a new one and add it to the list
//		of reports.
// 		eg: table1.columns = filter1.Thickness, filter1.dose
//
// Return code:	void
//
// Side Effects:
//		This method sets the $SAVE property for the variables.
//		Any variable added to any report is assumed to be SAVED,
//		ie., the user doesn't want EXCON to delete it even if it's
//		known to be an intermediate filename. 
//		
//

void Exp::settablecolumns (const char* tablename, const char* columns) {
    Report* report = nil;
    unsigned size = reports_->size();
    while(size--) {
	Report* tmpreport = (Report*)reports_->cycle();
	if (strcmp(tmpreport->getname(), tablename) == 0) {
	    assume(report == nil);		// better be only 1.
	    report = tmpreport;
	}
    }
    if (report == nil) {
	report = new Report(tablename, tablename);
	reports_->enq(report);
    }
    if (GLB_repdebug) 
	cerr << "Building reporting structure.\n";

    Pmu pmu;  
    int pos = 0;  
    char const* rest = columns;
    while (1) {
	pos = pmu.match (rest, "<ident>", ".", "<variable>", ",");
	if (pos == 0) 
	    pos = pmu.match (rest, "<ident>", ".", "<variable>");  
	if (pos == 0) 
	    break;  
	rest = &rest[pos];

	char toolname[256];  pmu.gettextfield (1, toolname);
	char varname[256];   pmu.gettextfield (3, varname);
	if (GLB_repdebug) 
	    cerr 
		<< "  Scanning table column " << toolname << "." << varname 
		<< ".\n";
	const ToolInstance* ti = gettoolinstance (toolname);
	if (ti == nil) 
	    errormissingtool (toolname);
	Var* var = ti->getvar (varname);
	if (var == nil) 
	    errormissingvar (varname, toolname);
    
	// make sure it doesn't kill itself
	var->addprop("$SAVE");

	if (!var->hasprop("$OUTPUTS")) {
	    // input variables use IV Reporter. This is also true for the
	    // pseudo variables, since these get entered in vartable as well.
	    if (GLB_repdebug) 
		cerr <<  "...ADD IVReporter.\n";
	    Reporter* reporter = new IVReporter(toolname, varname);
	    report->enqreporter(reporter);
	}
	else if (var->hasprop("$RUN")) {
	    if (GLB_repdebug) 
		cerr <<  "...ADD RUNReporter.\n";
	    // Variables extracted after program runs require RUN reporters
	    Reporter* reporter = new RUNReporter(toolname, varname);
	    report->enqreporter(reporter);
	}
    }
}


//
// Method:	Exp::genscript
//
// Caller:	Exp::run_multiple
// Parameters:	
//    job:	the job sequence, ie list of ToolInstance* to be exec'd.
//    iteration:the iteration id of the run (1 == first).
//
// Description:
//		Creates the Bourne shell script that has the shell commands
//		to execute the `ToolInstance's in the experiment for this
//		particular ``iteration''.
//
// Return code: name of the script. Caller delete storage.
//
// Side Effects:
//		Generates the exec line for the ToolInstance by calling
//		ToolInstance::genexec().
//		Also writes out the gfile via ToolInstance::writegfile().
//

char* Exp::genscript (const Stq* job, int iteration) {
    char buf[1024];
    sprintf (buf, "run.script-%d", iteration);
    char* runscript = strcpy_with_alloc(buf);
    FILE* fp = fopen(runscript, "w");
    if (fp == nil) {
	errorrunopen(runscript); 
	return nil; 
    }
    for (int i=1; i<=job->size(); i++) {
	ToolInstance* ti = (ToolInstance*)job->cycle();
	ti->genexec();
	const char* exec_str = ti->getexec();
	fprintf(fp, "%s\n", exec_str);
	if (ti->getgpath() != nil) 
	    ti->writegfile ();
    }
    // art of self-destruction
    fprintf(fp, "/bin/rm %s\n", runscript);
    fclose(fp);
    return runscript;
}


//
// Method:	Exp::collectresults
//
// Caller:	Exp::run
//
// Parameters:	
//    iteration:iteration counter of the run.(first == 1)
//
// Description:
//		Ask all the reports to first collect the results needed
//		for the reports and then report the findings in the report
//		files (specified by Exp::setoutput). Each report is reset
//		in prep for the next run.
// Return code:	void
//
// Side Effects:
//

void Exp::collectresults(int iteration) {
    unsigned size = reports_->size();
    while(size--) {
	Report* report = (Report*)reports_->cycle();
	report->collectresults(iteration);
	report->printrow();
	report->reset();  
    }
}

//
// Method:	Exp::progressetup
//
// Caller:	Exp::run
//
// Parameters:	
//    nruns:	number of runs needed to complete experiment
//
// Description:
//		set up the initial estimate of the run time.
//
// Return code:	void
//
// Side Effects:
//

void Exp::progresssetup(int nruns) {

    // make initial report
    prnext_ = prstart_;
    prnruns_ = nruns;

    // initial estimate of time to complete (seconds)
    // runtime = nruns*  (total sequence runtime + total collection run time)
    // ...get total sequence run time
    int tsrt = 0;
    for (int i=1;i<=expseq_->size();i++) {
	const ToolInstance* ti = (ToolInstance*)expseq_->cycle();
	tsrt += ti->gettool()->getruntime();
    }
    // ...get total collection run time
    int tcrt = 0;
    unsigned size = reports_->size();
    while(size--) {
	const Report* report = (Report*)reports_->cycle();
	tcrt += report->getruntime();
    }
    prestimate_ = nruns*  (tsrt + tcrt);
}


//
// Method:	Exp::progressreport
//
// Caller:	Exp::run
//
// Parameters:	
//    run:	# of runs that have completed so far
//
// Description:
//     		gives feedback to the user as to where the experiment is
//		and how long it might take to get it done. Typically 
//		reports in the format:
//		Experiment %% done. Estimating 1hr 2min to complete(12:45).
//
//		Feedback is only given if enough time has passed since the
//		last report time.
//
// Return code:	void
//
// Side Effects:
//

void Exp::progressreport(int run) {

    // report this run?  do not report if not yet time
    int now = (int)time(0);
    if (now < prnext_) 
	return;

    // calculate time of next run
    int runtime = prestimate_;
    // report frequency calculation: 
    // ...report about 10 times during the run, report a bit more early
    int divideby = 12;
    if (run > 0) divideby = 64 - (run*run);
    if (divideby < 12) divideby = 12;
    prnext_ = now + (runtime / divideby); 
    // ...special cases and boundry conditions
    if (run == 0)
	prnext_ = now;			// always report after 1st run
    else if (runtime > 600) 
	prnext_ = now + 120;		// if > 10 minutes report every 2 mins

    // perform calculations
    // ...percent done
    int percentdone = 0;
    if (run > 0) 
	percentdone = (int) (((float)run / (float)prnruns_)*  100.0);

    // ...elapsed time to complete
    int elapsed = now - prstart_;
    int willcomplete;
    if (percentdone == 0) 
	willcomplete = prestimate_;
    else 
	willcomplete = (int)(((float)elapsed*  (100.0/(float)percentdone)));
    int lefttocomplete = willcomplete - elapsed + 1;
    int seconds = lefttocomplete;
    int hours   = (int)((float)lefttocomplete / 3600);
    seconds -= hours*  3600;
    int minutes = (int)((float)seconds / 60);
    seconds -= minutes * 60;

    // handle first message as a special case
    if (run == 0) {
	char buf[1024]; buf[0] = '\0';
#ifdef USE_IOSTREAMS
	memset(buf, 0, sizeof(buf));
	ostrstream ostr(buf, sizeof(buf));

	// CONSTRUCT STRING: 
	//     Experiment 10% done.  Estimating 01hr 20min to complete (12:45).
	ostr 
	    << prnruns_ 
	    << " runs are required to complete this experiment.\n"
	    << "My initial estimate is: ";

	// ...ELAPSED TIME TO COMPLETE
	if (hours > 0) { 
	    ostr << hours << " hours ";
	}
	if (minutes > 0) { 
	    ostr << minutes << " minutes ";
	}
	ostr << seconds << " seconds to complete.";

	char buf2[1024]; buf2[0] = '\0';
	time_t cttc = now + willcomplete;
	struct tm* lt = localtime (&cttc);
	strftime (buf2, sizeof(buf2), "(%I:%M%p)", lt);

	ostr << buf2 << "\n";
	// ...PRINT
	cout << ostr.str() << flush; 
#else
	sprintf(buf, 
	    "%d runs are required to complete this experiment.\n"
	    "My initial estimate is: ", prnruns_
	);
	char buf2[1024]; buf2[0] = '\0';
	// ...ELAPSED TIME TO COMPLETE
	if (hours > 0) { 
	    sprintf(buf2, " %d hours ", hours);
	}
	if (minutes > 0) { 
	    sprintf(buf2, " %d minutes ", minutes);
	}
	strcat(buf, buf2); buf2[0] = '\0';
	sprintf(buf2, "%d seconds ", seconds);
	strcat(buf, buf2); buf2[0] = '\0';

	time_t cttc = now + willcomplete;
	struct tm* lt = localtime (&cttc);
	strftime (buf2, sizeof(buf2), "(%I:%M%p)", lt);

	strcat(buf, buf2);
	puts(buf); puts("\n");
#endif
	return;
    }

    // CONSTRUCT STRING: 
    //	Experiment 10% done.  Estimating 01hr 20min to complete (12:45).
    char buf[1024]; buf[0] = '\0';
#ifdef USE_IOSTREAMS
    memset(buf, 0, sizeof(buf));
    ostrstream ostr(buf, sizeof(buf));
    ostr 
	<< "Experiment " << setw(2) << percentdone << "% done(" << setw(3)
	<< run << " runs). Estimating ";
    // ...ELAPSED TIME TO COMPLETE
    if (hours > 0) { 
	ostr << setw(2) << hours << " hours ";
    }
    if (minutes > 0) { 
	ostr << setw(2) << minutes << " minutes ";
    }
    ostr << setw(2) << seconds << " seconds to complete ";

    time_t cttc = now + willcomplete;
    struct tm* lt = localtime (&cttc);
    char buf2[1024]; buf2[0] = '\0';
    strftime (buf2, sizeof(buf2), "(%I:%M%p)", lt);
    ostr << buf2 << "\n";
    // ...PRINT
    cout << ostr.str() << flush;
#else
    sprintf (buf, "Experiment %2d%% done(%2d runs).  Estimating ", 
	percentdone, run
    );
    // ...ELAPSED TIME TO COMPLETE
    char buf2[1024]; buf2[0] = '\0';
    if (hours > 0) { 
	sprintf (buf2, "%2dhr ", hours);  
	strcat (buf, buf2); buf2[0] = '\0';
    }
    if (minutes > 0) { 
	sprintf (buf2, "%2dmin ", minutes);  
	strcat (buf, buf2); buf2[0] = '\0';
    }

    sprintf (buf2, "%2dsec to complete ", seconds);  
    strcat (buf, buf2); buf2[0] = '\0';
    time_t cttc = now + willcomplete;
    struct tm *lt = localtime (&cttc);
    strftime (buf2, sizeof(buf), "(%I:%M%p)", lt);
    strcat (buf, buf2);
    printf ("%s\n", buf);
#endif
}

//
// Method:	Exp::setfvgroup
//
// Caller:	ExpMgr::load
//
// Parameters:	
//    tiname:	tool instance name (eg., filter3)
//    varname:	variable name for that tool instance (eg., thickness)
//    others:   rest of the (tiname, varname) tuples to add to group
//    origline:	the orig line from the EXP file (for error reporting)
//
// Description:
//		Creates a new (or adds to existing) FreeVariableGroup
//		for the experiment. The groups define vertical sets for
//		varying parameters, so the user can vary multiple params
//		in lock-step.
//
//		NOTE: a FreeVariable can be in ONE and ONLY group only.
//
//		EX:
//		VARY tool1.var1 SET a, b, c, d
//		VARY tool2.var2 FROM 1 TO 3 BY 1
//		VARY TOGETHER tool1.var1 AND tool2.var2
//
//		iteration#	tool1.var1	tool2.var2
//		1		a		1
//		2		b		2
//		3		c		3
//		4		d		3
//
// Return code:
//
// Side Effects:
//		Empty FreeVariableGroup's are deleted from the system.
//		
//

void Exp::setfvgroup(
    const char* tiname, const char* varname, const char* others,
    const char* origline
) {
    FreeVariableGroup* fvgr = getfvgroup(tiname, varname);
    if (fvgr == nil) {
	cerr 
	    << "ERROR: Free variable ``" << tiname << "." 
	    << varname << "'' not defined.\n"
	    << "Offending line: " << origline << "\n";
	exit(1);
    }
    char const* rest = others;
    while(true) {
	Pmu pmu;
	int pos = pmu.match (rest, "and", "<ident>", ".", "<ident>");
	if (pos == 0) {
	    break;
	}
	rest = &rest[pos];

	char ttiname[1024];
	char tvarname[1024];
	pmu.gettextfield (2, ttiname);
	pmu.gettextfield (4, tvarname);

	FreeVariableGroup* tfvgr = getfvgroup(ttiname, tvarname);
	const FreeVariable* tfv = getfreevar(ttiname, tvarname);
	if (tfvgr == nil || tfv == nil) {
	    cerr 
		<< "ERROR: Free variable ``" << ttiname << "." 
		<< tvarname << "'' not defined.\n"
		<< "Offending line: " << origline << "\n";
	    exit(1);
	}
	tfvgr->remove(tfv);
	fvgr->add(tfv);
    }

    // clean up empty Groups.
    unsigned ngroups = fvgroups_->size();
    while(ngroups--) {
	FreeVariableGroup* gr = (FreeVariableGroup*)fvgroups_->pop();
	if(gr->getgroup()->size() > 0)
	    fvgroups_->enq(gr);
    }
}


//
// Method:	Exp::setloopvar
//
// Caller:	ExpMgr::load
//
// Parameters:	
//    tiname:	tool instance name (eg., filter3)
//    varname:	variable name for that tool instance (eg., thickness)
//    fromtext:	text of starting value
//    totext:	text of stopping value
//    incrtext:	text of increment value
//    origline:	the orig line from the EXP file (for error reporting)
//
// Description:
//		Create a new LoopVariable by specifying a starting, ending
//		values and an increment. This is valid for values of certain
//		classes ONLY -- IntegerValue and RealValue for now -- for
//		which we can define the addition, substraction, and 
//		division operators meaningfully.
//
//		EX: VARY tool1.var1 FROM val1 TO val2 BY val3
//
// Return code:
//
// Side Effects:
//		If the current variable doesn't already belong to a group,
//		a new FreeVariableGroup is created in the system.
//


void Exp::setloopvar(
    const char* tiname, const char* varname, const char* fromtext, 
    const char* totext, const char* incrtext, const char* origline
) {
    ToolInstance* ti = gettoolinstance (tiname); 
    if (ti == nil) {
	cerr << "ERROR: No such tool instance ``" << tiname
	    << "'' exists in the experiment sequence.\n"
	    << "Offending line: " << origline << "\n";
	exit(1);
    }
    Var* var = ti->getvar(varname);
    if (var == nil) {
	cerr << "ERROR: No such variable ``" << varname
	    << "'' exists for TOOL ``" << tiname << "''.\n"
	    << "Offending line: " << origline << "\n";
	exit(1);
    }

    // create the right type values and such.
    Value* tmpval = var->genval();

    // these are deleted when the LoopVariable dies.
    Value* startval = tmpval->clone(fromtext);
    Value* stopval = tmpval->clone(totext);
    Value* incrval = tmpval->clone(incrtext);

    delete tmpval;

    if (startval == nil || stopval == nil || incrval == nil) {
	cerr << "ERROR: Wrong data type for variable ``" << varname
	    << "'' (TOOL ``" << tiname << "'').\n"
	    << "Offending line: " << origline << "\n";
	exit(1);
    }
    FreeVariable* freevar = new LoopVariable(
	tiname, varname, startval, stopval, incrval
    );
    FreeVariableGroup* gr = getfvgroup(tiname, varname);
    if (gr == nil) {
	gr = new FreeVariableGroup;
	fvgroups_->enq(gr);
    }
    gr->add(freevar);
}

//
// Method:	Exp::setsetvar
//
// Caller:	ExpMgr::load
//
// Parameters:	
//    tiname:	tool instance name (eg., filter3)
//    varname:	variable name for that tool instance (eg., thickness)
//    valueset:	text of all the values in the set
//    origline:	the orig line from the EXP file (for error reporting)
//
// Description:
//		Create a new SetVariable by specifying a set of values
//		the variable is going to be taking during the iterations.
//		This is valid for values of ALL classes ONLY, since we
//		don't need fancy operators like addition, substraction, and 
//		division.
//
//		EX: VARY tool1.var1 SET val1, val2, val3
//
// Return code:
//
// Side Effects:
//		If the current variable doesn't already belong to a group,
//		a new FreeVariableGroup is created in the system.
//

void Exp::setsetvar(
    const char* tiname, const char* varname, const char* valueset, 
    const char* origline
) {
    ToolInstance* ti = gettoolinstance (tiname); 
    if (ti == nil) {
	cerr << "ERROR: No such tool instance ``" << tiname
	    << "'' exists in the experiment sequence.\n"
	    << "Offending line: " << origline << "\n";
	return;
    }
    Var* var = ti->getvar(varname);
    if (var == nil) {
	cerr << "ERROR: No such variable ``" << varname
	    << "'' exists for TOOL ``" << tiname << "''.\n"
	    << "Offending line: " << origline << "\n";
	exit(1);
    }

    // create the right type values and such.
    Value* tmpval = var->genval();
    Stq* values = new Stq;

    char const* rest = valueset;
    while(true) {
	Pmu pmu;
	int pos = pmu.match (rest, "<name>", ",");
	if (pos == 0) {
	    pos = pmu.match (rest, "<name>");
	    if (pos == 0) {
		break;
	    }
	}
	rest = &rest[pos];

	char valtext[1024];
	pmu.gettextfield (1, valtext);
	Value* value = tmpval->clone(valtext);
	if (value == nil) {
	    cerr << "ERROR: Wrong range type for variable ``" << varname
		<< "'' (TOOL ``" << tiname << "'').\n"
		<< "Offending line: " << origline << "\n";
	    exit(1);
	}
	else {
	    values->enq(value);
	}
    }

    delete tmpval;
    if (values->size() == 0) {
	cerr 
	    << "WARNING: no values given for SET parameters.\n"
	    << "Offending line: " << origline << "\n";
	delete values;
    }
    else {
	FreeVariable* freevar = new SetVariable(tiname, varname, values);
	FreeVariableGroup* gr = getfvgroup(tiname, varname);
	if (gr == nil) {
	    gr = new FreeVariableGroup;
	    fvgroups_->enq(gr);
	}
	gr->add(freevar);
    }
}

//
// Method:	Exp::setrandomvar
//
// Caller:	ExpMgr::load
//
// Parameters:	
//    tiname:	tool instance name (eg., filter3)
//    varname:	variable name for that tool instance (eg., thickness)
//    nvalues:  how many random values to generate
//    fromtext:	text of starting value
//    totext:	text of stopping value
//    genseed:  generate seed or use supplied seed 
//    seed:	supplied seed if any
//    dosort:   sort the random sample?
//    origline:	the orig line from the EXP file (for error reporting)
//
// Description:
//		Create a new RandomValue by specifying a starting, ending
//		values and optionally a seed to use. The user must also
//              supply the number of random values to generate. This is valid 
//              for values of certain classes ONLY -- IntegerValue and 
//              RealValue for now -- for which we can define the addition, 
//              substraction, and dvision operators meaningfully. Using it
//              with anything other than RealValue is however highly 
//              discouraged.
//
//		EX: VARY tool1.var1 FROM val1 TO val2 BY val3
//		EX: VARY tool1.var1 FROM val1 TO val2 BY val3 SEED seed
//
// Return code:
//
// Side Effects:
//		If the current variable doesn't already belong to a group,
//		a new FreeVariableGroup is created in the system.
//


void Exp::setrandomvar(
    const char* tiname, const char* varname, int nvalues,
    const char* fromtext, const char* totext, bool genseed, long seed,
    bool dosort, const char* origline
) {
    ToolInstance* ti = gettoolinstance (tiname); 
    if (ti == nil) {
	cerr << "ERROR: No such tool instance ``" << tiname
	    << "'' exists in the experiment sequence.\n"
	    << "Offending line: " << origline << "\n";
	exit(1);
    }
    Var* var = ti->getvar(varname);
    if (var == nil) {
	cerr << "ERROR: No such variable ``" << varname
	    << "'' exists for TOOL ``" << tiname << "''.\n"
	    << "Offending line: " << origline << "\n";
	exit(1);
    }

    // create the right type values and such.
    Value* tmpval = var->genval();

    // these are deleted when the LoopVariable dies.
    Value* startval = tmpval->clone(fromtext);
    Value* stopval = tmpval->clone(totext);

    delete tmpval;

    if (startval == nil || stopval == nil) {
	cerr << "ERROR: Wrong data type for variable ``" << varname
	    << "'' (TOOL ``" << tiname << "'').\n"
	    << "Offending line: " << origline << "\n";
	exit(1);
    }
    FreeVariable* freevar = nil;
    if (genseed) {
	freevar = new RandomVariable(
	    tiname, varname, startval, stopval, nvalues, dosort
	);
    } else {
	freevar = new RandomVariable(
	    tiname, varname, seed, startval, stopval, nvalues, dosort
	);
    }
    FreeVariableGroup* gr = getfvgroup(tiname, varname);
    if (gr == nil) {
	gr = new FreeVariableGroup;
	fvgroups_->enq(gr);
    }
    gr->add(freevar);
}


//
// Method:	Exp::resolveunique
//
// Caller:	Exp::run
//
// Parameters:	
//    sequence:	the sequence of ToolInstance* to resolve $UNIQUE.
//
// Description:
//		Asks all the ToolInstance*'s in the sequence to resolve
//		it's variables with $UNIQUE filenames. These $UNIQUE
//		variables are typically output filenames of tools.
//		It simply calls ToolInstance::resolveunique() on each
//		instance.
//
// Return code:
//
// Side Effects:
//

void Exp::resolveunique(const Stq* sequence) {
    for (int i=1; i<=sequence->size(); i++) {
	ToolInstance* ti = (ToolInstance*)sequence->cycle();
	ti->resolveunique();
    }
}


//
// Method:	Exp::setup_links
//
// Caller:	Exp::run
//
// Parameters:	
//    sequence:	the sequence of ToolInstance* to resolve link variables.
//
// Description:
//		This method goes thru the experiment sequence and resolves
//		all the link variables, and creates ResolveLink*'s for
//		each link that is resolved. Only called once during the
//		experiment, and once the list of ResolvedLink*'s are set
//		up, each iteration need only fill the values.
//
// Return code:
//
// Side Effects:
//

void Exp::setup_links(const Stq* sequence) {
    Stq* ancestors = new Stq;
    Hashtable* linktable = new Hashtable(19);
    Stq* tmpsequence = sequence->clone();

    if (GLB_resdebug) 
	cerr << "BEGIN resolve links DEBUG\n";

    ToolInstance* thisti = nil;
    unsigned seqlen = tmpsequence->size();
    while(seqlen--) {
	linktable->clear();		// no links resolved yet.
	if (thisti) 
	    ancestors->push(thisti);
	thisti = (ToolInstance*)tmpsequence->cycle();

	// nobody to look up to, I guess.
	if (ancestors->size() == 0)
	    continue;

	// at this point THISTI is the current tool instance under 
	// consideration for all of THISTI'S link variables; if LASTTI is 
	// link tool then take the appropriate action

	if (GLB_resdebug) cerr
	    << "... RESOLVE LINKS FROM: " << thisti->getinstname()
	    << " ...\n";

	const Stq* links = thisti->gettool()->getlinks();
	Stq* resolved_links = new Stq;
	unsigned nancestors = ancestors->size();
	while(nancestors--) {
	    const ToolInstance* lastti = (ToolInstance*)ancestors->cycle();
	    if (GLB_resdebug) cerr
		<< "\t... RESOLVE LINKS FROM: " << thisti->getinstname()
		<< " TO " << lastti->getinstname() << "\n.";
	    unsigned nlinks = links->size();
	    while(nlinks--) {
		const Link* link = (const Link*)links->cycle();

		// if already resolved, skip and go on the next one.
		if (linktable->find(link->getlinkvar())) {
		    if (GLB_resdebug) 
			cerr << "LINK " << link->getlinkvar()
			    << "already resolved\n";
		    continue;
		}
		// if lastti is not the link tool, skip
		if (strcmp(link->getlinktool(), lastti->getname()) != 0) 
		    continue;
		
		// PERFORM APPROPRIATE ACTION
		if (strcmp(link->getactioncode(), "$GETVAR") == 0) {
		    // assign the value of lastvar to thisvar
		    const char* thisvarname = link->getlinkvar();
		    const Var* lastvar = lastti->getvar(link->getactionval());
		    if (lastvar == nil) {
			cerr 
			    << "Variable `" << thisvarname 
			    << "' is not defined for tool `"
			    << lastti->getname() << "'..Ignoring $LINK: `" 
			    << link->getlinktext() << "' in tools file\n";
			continue;
		    }

		    // we have a match!
		    Var* thisvar = thisti->getvar(thisvarname);
		    assert(thisvar != nil);

		    resolved_links->enq(
			new ResolvedLink(link, lastti->getinstname())
		    );

		    // mark this link to be already resolved.
		    linktable->insert(thisvarname, thisvarname);
		    if (GLB_resdebug) cerr
			<< "......FOUND LINK OF "
			<< lastti->getinstname() << "." 
			<< lastvar->getname() << " TO "
			<< thisti->getinstname() << "." << thisvarname
			<< "\n";
		}
	    }
	}
	thisti->set_resolved_links(resolved_links);
    }
    delete tmpsequence;
    delete ancestors;
    delete linktable;
}

//
// Method:	Exp::resolvelinks
//
// Caller:	Exp::resolvelinks
//
// Parameters:	
//    ti:	The ToolInstance whose link variables need to be resolved.
//
// Description:
//		This method simply looks at all the ResolvedLink*'s
//		(setup by setup_links) for the 	ToolInstance ``ti'' and 
//		fills the values.
//		Has to be called for every ToolInstance* every iteration.
//
//		NOTE: the $OUTPUTS case is a tricky one, since that 
//		implies running a program before this tool can be executed.
//		This puts a serial condition on the execution.
//		cf: SHADOW SEED program feeding SOURCE.
//
// Return code:	void
//
// Side Effects:
//

void Exp::resolvelinks (ToolInstance* ti) {
    if (GLB_resdebug) cerr
	<< "... RESOLVE LINKS FROM: " << ti->getinstname()
	<< " ...\n";
    
    const Stq* resolved_links = ti->get_resolved_links();
    unsigned nlinks = resolved_links->size();
    while(nlinks--) {
	const ResolvedLink* rlink = (ResolvedLink*)resolved_links->cycle();
	const Link* link = rlink->getlink();
	const char* instname = rlink->gettoolinstance();
	const char* varname = link->getlinkvar();

	ToolInstance* linkti = gettoolinstance(instname , ti->getiteration()); 
	assert(linkti != nil);
	const Var* linkvar = linkti->gettool()->getvar(link->getactionval()); 
	assert(linkvar != nil);

	Value* newval = nil;
	if (linkvar->hasprop("$OUTPUTS")) {
	    Reporter* reporter = new RUNReporter(instname, linkvar->getname());
	    const Value* value = reporter->reportval(ti->getiteration());
	    newval = value->clone();
	    delete reporter;
	}
	else {
	    newval = linkti->getvarval(linkvar->getname())->clone();
	}
	ti->setvarval(varname, newval);

	if (GLB_resdebug) cerr
	    << "......ASSIGNED VALUE OF "
	    << linkti->getinstname() << "." << linkvar->getname() 
	    << " TO "
	    << ti->getinstname() << "." << varname
	    << "\n";
    }
}

//
// Method:	Exp::resolvelinks
//
// Caller:	Exp::run
//
// Parameters:	
//    sequence:	the sequence of ToolInstance* to resolve link variables.
//
// Description:
//		This method simply calls Exp::resolvelinks for each of the
//		ToolInstance* in the sequence to resolve the links.
//		Called once every iteration.
//
//		NOTE: I clone the list (not a deep copy, simply the
//		pointers in the list) for safety. Now I can leave the
//		clone in any state I want, w/out messing up the original
//		List's order (cf: Iteration on class Stq).
//
// Return code:	void
//
// Side Effects:
//

void Exp::resolvelinks (const Stq* sequence) {
    if (sequence->size() > 0) {
	Stq* tmpsequence = sequence->clone();
	unsigned seqlen = tmpsequence->size();

	if (GLB_resdebug) 
	    cerr << "BEGIN resolve links DEBUG\n";
    
	// skip the first one, since it has no predecessors to link to.
	(void)tmpsequence->cycle(); seqlen--;
	while(seqlen--) {
	    ToolInstance* ti = (ToolInstance*)tmpsequence->cycle();
	    resolvelinks(ti);
	}
	delete tmpsequence;
    }
}


//
// Method:	Exp::dump
//
// Caller:	Exp::
//
// Parameters:	
//
// Description:
//		does a dump of the experiment sequence for debugging.
//
// Return code:	void
//
// Side Effects:
//

void Exp::dump() const{
    cerr << " Experiment Sequence: ";
    for (int i=1;i<=expseq_->size();i++) {
	const ToolInstance* ti = (ToolInstance*)expseq_->cycle();
	if (i > 1) 
	    cerr << ", ";  
	cerr << ti->getname() << ti->getnumber();
    }
    cerr << endl;
};

//
// Method:	Exp::gettoolnum
//
// Caller:	Exp::gettoolinstance, Exp::addtool
//
// Parameters:	
//    tiname:	name of the toolinstance (eg., filter3)
//    pos:	where is the tool instance number. (Returned)
//
// Description:
// 		given a tool instance name (which is typically a tool name
//		with the instance number tacked on to the end), return the
//		instance number.
//
//		eg., tool3 will return 3 and pos == 4.
//
// Return code:	
//     		the tool number
//
// Side Effects:
//

int Exp::gettoolnum(const char* tiname, int &pos) const {
    int val = 0;  
    pos = strlen(tiname) - 1;
    while ((tiname[pos] == ' ') && (pos > 0)) 
	pos--;   				// skip trailing blanks.
    for(int decade = 0; pos >= 0; decade++, pos--) {
	if (isdigit(tiname[pos])) 
	    val = val + (tiname[pos]-'0') * (int)pow(10.0L, (double)decade);
	else 
	    break;
    }
    pos++;
    return val;
}



//
// Method:	Exp::delete_job_sequence
//
// Caller:	Exp::run
//
// Parameters:	
//    iteration:the iteration for which to kill the job
//
// Description:
// 		delete the job sequence for the given iteration. Simply 
//		go thru the job sequence, pick out the correct job, and 
//		delete ALL the tool instances in the job. Next call to 
//		get_next_job(this_iteration) will return NIL.
//
//		Called after the iteration is execution, and the reporting
//		is done, to free the memory eaten up by the ToolInstance's
//		making up the job.
//
// Return code:
//
// Side Effects:
//

void Exp::delete_job_sequence(int iteration) {
    Stq* job = nil;
    unsigned size = joblist_->size();
    while(size--) {
	Stq* tmpjob = (Stq*)joblist_->pop();
	ToolInstance* ti = (ToolInstance*)tmpjob->pop(); tmpjob->push(ti);
	if (iteration == ti->getiteration())
	    job = tmpjob;
	else 
	    joblist_->enq(tmpjob);
    }
    assert(job != nil);

    size = job->size();
    while(size--) {
	ToolInstance* ti = (ToolInstance*)job->pop();
	delete ti; ti = nil;
    }
    delete job; job = nil;
}


//
// Method:	Exp::get_job_sequence
//
// Caller:	Exp::get_next_job, Exp::run, Exp::gettoolinstance
//
// Parameters:	
//    iteration:the iteration for which to get the job sequence
//
// Description:
//		Get the job sequence from Exp's joblist_ that matches
//		the iteration id. If the job sequence is completed and
//		has been deleted, return NIL.
//
// Return code:
//
// Side Effects:
//


Stq* Exp::get_job_sequence(int iteration) const {
    Stq* job = nil;
    for (int j = 1; j <= joblist_->size(); j++) {
	Stq* tmpjob = (Stq*)joblist_->cycle();
	const ToolInstance* ti = (ToolInstance*)tmpjob->pop(); tmpjob->push(ti);
	if (iteration == ti->getiteration()) {
	    assert(job == nil);		// ie., it's the only match!
	    job = tmpjob;
	}
    }
    return job;
}

//
// Method:	Exp::gettoolinstance
//
// Caller:	Exp::run
//
// Parameters:	
//    name:	Tool Instance name (eg., filter4)
//    iteration:the iteration for which to kill the job
//
// Description:
// 		NOTE: name is expected to have the instance number at end
//
// Return code:
// 		returns instance pointer or NIL if not found
//
// Side Effects:
//

ToolInstance* Exp::gettoolinstance(const char* name, int iteration) const {
    char* tiname = strcpy_with_alloc(name);
    int pos;  
    int number = gettoolnum (tiname, pos); 
    tiname[pos] = '\0';
    Stq* job = get_job_sequence(iteration); assume (job != nil);

    ToolInstance* ret = nil;
    for (int i=1;i<=job->size();i++) {
	ToolInstance* ti = (ToolInstance*)job->cycle();
	if (
	    strcmp(ti->gettool()->getname(), tiname) == 0 &&
	    ti->getnumber() == number
	) {
	    assert(ret == nil);		// ie., it's the only match!
	    ret = ti; 
	}
    }
    tiname[0] = '\0'; delete[] tiname;
    return ret;
}

//
// Method:	Exp::addtool
//
// Caller:	Exp::buildexpseq
//
// Parameters:	
//    name:	Tool Instance name (eg., filter4)
//
// Description:
// 		Given a new tool instance name, it creates a new instance 
//		of the tool and adds it to the Exp::expseq_ list, which
//		the is the virgin list kept by the system, and all other
//		tool sequences are cloned from this list.
//
//		NOTE: all initial tool instances, ie., the ones added here
//		to Exp::expseq_ have the instance/iteration id set to 0.
//
// Return code:
// 		returns instance pointer or NIL if not found
//
// Side Effects:
//

void Exp::addtool(const char* name) {
    char* tiname = strcpy_with_alloc(name);
    int pos;  
    int number = gettoolnum (tiname, pos); 
    tiname[pos] = '\0';
    Tool* tool = GLB_toolsmgr->gettool(tiname);
    if (tool == nil) { 
	erroraddtool (tiname); 
	// does not return. exits with error code.
    }
    ToolInstance* ti = new ToolInstance (tool, number, 0);
    expseq_->enq(ti);
    if (GLB_debug) 
	cerr << "Adding tool `" << tiname << "' to sequence\n";
    tiname[0] = '\0'; delete[] tiname;
}


//
// Method:	Exp::buildexpseq
//
// Caller:	ExpMgr::load
//
// Parameters:	
//    exptext:	the text of the experimental sequence 
//
// Description:
//		Build the experimental sequence from the text in the 
//		experiment file (eg., sync -> filter1 -> mirror2).
//
// 		NOTE: this method builds the SEQUENCE 0 Exp::expseq_,
//		which is the granddaddy of all tool sequences, since from
//		now on, all tool sequences will be cloned from Exp::expseq_
//
// Return code:
// 		returns -1 if there is a tool in the tool sequence not found 
//		in the TOOLS file
//
// Side Effects:
//

int Exp::buildexpseq (const char* exptext) {
    char tool[1024];
    char rest[1024];

    Pmu pmu;

    strcpy (rest, exptext);
    while (pmu.match (rest, "<ident>", "->", "<text>")) {
	pmu.gettextfield (1, tool);
	pmu.gettextfield (3, rest);
	addtool (tool);
    }
    // if we are here ``rest'' contains the last tool name
    addtool (rest);
    return 1;
}


//
// Method:	Exp::get_next_job
//
// Caller:	Exp::run
//
// Parameters:	
//    iteration:the iteration for which to create a job sequence
//
// Description:
//		Create a new job to run. There are new jobs as long as
//		there are FreeVariable's (from the VARY statements) in the 
//		system that are still not exhausted.
//
//		look at the list of free variables and find the next one 
//		to vary. Create another experiment tool instance sequence 
//		cloned from SEQUENCE 0, ie., Exp::expseq_, and change the 
//		corresponding free variable value and return the new 
//		sequence.
//
// Return code:
// 		returns pointer to the job list or NIL if no jobs left
//
// Side Effects:
//

Stq* Exp::get_next_job(int iteration) {

    if (iteration > 1) {
	int increment = 1;
	int ngroups = fvgroups_->size();
	// LOOP THRU ENTIRE LIST SO THAT ORDERING IS RETAINED
	for(int i = 1; i <= ngroups; i++) {
	    FreeVariableGroup* gr = (FreeVariableGroup*)fvgroups_->cycle();
	    if (i == increment) {
		gr->increment();
		if (gr->done()) {
		    gr->reset();
		    // INDICATE THAT NEXT fv NEEDS TO BE INCREMENTED
		    increment = i+1;
		}
	    }
	}
	if (increment > ngroups)	// no more variables left to vary. 
	    return nil;
    }


    // clone the 0th job sequence. ALL the job sequences are cloned from
    // the very first sequence built from the EXP file, ala expseq_
    // DO NOT clone from the previous job sequence, since that might've
    // already finished and thus deleted.

    Stq* nextseq = new Stq;
    Stq* lastseq = get_job_sequence(0); assume(lastseq != nil);
    for (int i = 1; i <= lastseq->size(); i++) {
	const ToolInstance* ti = (ToolInstance*)lastseq->cycle();
	ToolInstance* newti = ti->clone();
	newti->setiteration(iteration);
	nextseq->enq(newti);
    }

    // put the new sequence in the joblist_, so that gettoolinstance()
    // will work for this iteration.
    joblist_->enq(nextseq);

    // now set the FreeVariable values in various FreeVariableGroups.
    int ngroups = fvgroups_->size();
    while(ngroups--) {
	const FreeVariableGroup* gr = (FreeVariableGroup*)fvgroups_->cycle();
	const Stq* freevars = gr->getgroup();
	unsigned nfreevars = freevars->size();
	while(nfreevars--) {
	    const FreeVariable* fv = (FreeVariable*)freevars->cycle();
	    ToolInstance* ti = gettoolinstance (fv->gettool(), iteration);  
	    assume (ti != nil);

	    // now get the type of the value in the variable and set it.
	    // It is VERY IMPORTANT to call ToolInstance::setvarval,
	    // instead of simply setting the new value in the variable,
	    // since ToolInstance::setvarval sets the value in the correct
	    // ToolInstance. Also, note that the Value* created here is
	    // deleted when the Variable dies or gets a new value.

	    const Var* var = ti->getvar (fv->getvar());  assume (var != nil);
	    Value* newval = fv->getcurrent()->clone();
	    ti->setvarval(fv->getvar(), newval);
	}
    }
    return nextseq;
}

//
// Method:	Exp::expfileok
//
// Caller:	ExpMgr::load
//
// Parameters:	
//
// Description:
//		return true if the experiment file is ok semantically. 
//		The syntactic check is done during initial loading and 
//		this is called AFTER the entire file is loaded. Semantics 
//		cannot be checked before the loading since there is no
//		limitation on ordering in the experiment file
//
// Return code:
// 		returns true if the file is OK
//
// Side Effects:
//


bool Exp::expfileok() const {
    int fverrors = 0;
#ifdef NOTDEF
    for (int i=1; i<=freevars_->size(); i++) {
	const FreeVariable* fv = (FreeVariable*)freevars_->cycle();
	const char* tiname = fv->gettool();
	if (gettoolinstance(tiname) == nil) {
	    cerr 
		<< "Error in free variable defn: tool instance ``"
		<< tiname << "'' does not exist\n"
		<< "offending line ==> vary "
		    << fv->gettool() << "." << fv->getvar() << " from "
		    // << fv->getstart() << " to " << fv->getstop() << " by "
		    // << fv->getincr() 
		<< "\n";
	    ++fverrors;
	}
    }

    if (fverrors) {
	cerr << "\nthe current tool instances defined in experiment are:\n";
        for (int i = 1; i <= expseq_->size(); i++) {
  	    const ToolInstance* ti = (ToolInstance*)expseq_->cycle();
	    cerr << "\t" << i << ". " << ti->getinstname() << "\n";
	}
	cerr <<  "\n\n";
    }
#endif
    return (fverrors == 0) ? true : false;
}

/**************************************************************************
 * 
 *   EXPERIMENT MANAGER
 * 
 **************************************************************************/

//
// various error reporting routines. All of these EXIT with a code of 1.
//

void ExpMgr::errorfilenotfound (const char* path) {
    cerr 
	<<  "Could not open the experiment file ``" << path << "''.\n"
	<< "The file may not exist or may not have read permission.\n"
	<< "Check the file and try again.\n";
    exit (1);
}

//
// Method:	ExpMgr::load
//
// Caller:	main
//
// Parameters:	
//    name:	The experiment file name
//
// Description:
//		given an experiment file name, this method creates a new
//		Experiment and loads the Experiment from the file.
//		
//		NOTE: Exits on error
//
// Return code:
//
// Side Effects:
//


void ExpMgr::load (const char* path, int nesting) {
    char buf[1024];
    char text1[1024];  char text2[1024];  char text3[1024];
    char text4[1024];  char text5[1024];  
    int line = 0;   int errors = 0;

    Pmu pmu;

    if (GLB_debug) 
	cerr << "** LOADING EXPERIMENT FILE `" << path << "'\n";
    FILE* f = fopen (path, "r");  
    if (f == nil) 
	errorfilenotfound (path);

    // FOR NOW WE ARE DEALING WITH A SINGLE EXPERIMENT
    if (nesting == 0) {
	delete exp_;
	exp_ = new Exp ();
    }

    int block_comment_cnt = 0;
    while (1) {
	if (fgets (buf, sizeof(buf), f) == nil) break;  line++;
	if (pmu.isacomment(buf)) continue;

	//
	// here we implement block comments ala C/C++, but more restricted
	//
	if (pmu.match(buf, "<text>", "/*", "<text>")) {
	    cerr 
		<< "EXCON: block comment /* can only start on an empty line "
		<< "(line = " << line << ")" << endl;
	    ++errors;
	    continue;
	}

	if (pmu.match(buf, "/*", "<text>") || pmu.match(buf, "/*", "<nothing>")) 
	{
	    if (GLB_debug) {
		cerr << "   STARTING COMMENT: (line = " << line << ")." << endl;
	    }
	    ++block_comment_cnt;
	    continue;
	}

	if (
	    pmu.match(buf, "<text>", "*/") || 
	    pmu.match(buf, "*/") ||
	    pmu.match(buf, "<nothing>", "*/")
	) {
	    if (GLB_debug) {
		cerr << "   END COMMENT: (line = " << line << ")." << endl;
	    }
	    --block_comment_cnt;
	    continue;
	}

	if (block_comment_cnt) {
	    if (GLB_debug) {
		cerr << "   IGNORING comment (line: " << line << "): " 
		    << buf << endl;
	    }
	    continue;
	}


	// PROGRAM SEQUENCE: MATCH FIRST FIELD THEN TREAT REST AS TEXT
	// ...WE WILL PARSE IT OURSELVES SINCE Pmu IS LIMITIED TO 8 FIELDS 
	if (pmu.match (buf, "<ident>", "->", "<text>")) {
	    pmu.gettextfield (1, text1);
	    pmu.gettextfield (3, text2);
	    if (GLB_debug) 
		cerr << "   SEQUENCE: " << text1 << " -> " << text2 << "\n";
	    // SEMANTICS
	    // ...BUILD BASE EXPERIMENT FROM THE tool sequence STRING
	    exp_->buildexpseq (buf);
	    continue;
	}
	// THIS MUST BE A SINGLE TOOL SEQUENCE -- DEGENERATE CASE.
	if (pmu.match (buf, "<ident>", "<nothing>")) {
	    pmu.gettextfield (1, text1);
	    if (GLB_debug) 
		cerr <<  "   SINGLE TOOL SEQUENCE: " << text1 << "\n";
	    // SEMANTICS
	    // ...BUILD BASE EXPERIMENT FROM THE tool sequence STRING
	    exp_->buildexpseq (text1);
	    continue;
	}
	// EXPERIMENT 
	// eg., vary xlith.gap from 10.0 to 35.0 by 5.0
	if (pmu.match(buf,
	    "vary", "<ident>", ".", "<ident>", "from", "<name>", 
	    "to", "<name>", "by", "<name>")
	) {
	    pmu.gettextfield (2, text1);
	    pmu.gettextfield (4, text2);
	    pmu.gettextfield (6, text3);
	    pmu.gettextfield (8, text4);
	    pmu.gettextfield (10, text5);
	    // SEMANTICS
	    exp_->setloopvar(text1, text2, text3, text4, text5, buf);
	    continue;
	}

	// eg., vary xlith.th_ab set 0.20, 0.35, 0.45, 0.6, 0.7
	if (pmu.match(buf,
	    "vary", "<ident>", ".", "<ident>", "set", "<text>")
	){
	    pmu.gettextfield (2, text1);
	    pmu.gettextfield (4, text2);
	    pmu.gettextfield (6, text3);
	    // SEMANTICS
	    exp_->setsetvar(text1, text2, text3, buf);
	    continue;
	}

	// eg., vary xlith.th_ab random 10 from 0.20 to 0.35 seed 123456
	// this *must* come before the more general pattern w/out the seed.
	// See next one.
	if (pmu.match(buf,
	    "vary", "<ident>", ".", "<ident>", "random", "<integer>",
	    "from", "<real>", "to", "<real>", "seed", "<long>")
	){
	    pmu.gettextfield (2, text1);
	    pmu.gettextfield (4, text2);
	    int nvalues = pmu.getintfield(6);
	    pmu.gettextfield (8, text3);
	    pmu.gettextfield (10, text4);
	    long seed = pmu.getlongfield(12);
	    bool genseed = false;
	    bool dosort = false;
	    // SEMANTICS
	    exp_->setrandomvar(
		text1, text2, nvalues, text3, text4, genseed, seed, dosort,
		buf
	    );
	    continue;
	}

	// eg., vary xlith.th_ab random 10 from 0.20 to 0.35
	if (pmu.match(buf,
	    "vary", "<ident>", ".", "<ident>", "random", "<integer>",
	    "from", "<real>", "to", "<real>")
	){
	    pmu.gettextfield (2, text1);
	    pmu.gettextfield (4, text2);
	    int nvalues = pmu.getintfield (6);
	    pmu.gettextfield (8, text3);
	    pmu.gettextfield (10, text4);
	    bool genseed = true;
	    long seed = 0;
	    bool dosort = false;
	    // SEMANTICS
	    exp_->setrandomvar(
		text1, text2, nvalues, text3, text4, genseed, seed, dosort,
		buf
	    );
	    continue;
	}

	// eg., vary xlith.th_ab srandom 10 from 0.20 to 0.35 seed 123456
	// this *must* come before the more general pattern w/out the seed.
	// See next one.
	if (pmu.match(buf,
	    "vary", "<ident>", ".", "<ident>", "srandom", "<integer>",
	    "from", "<real>", "to", "<real>", "seed", "<long>")
	){
	    pmu.gettextfield (2, text1);
	    pmu.gettextfield (4, text2);
	    int nvalues = pmu.getintfield(6);
	    pmu.gettextfield (8, text3);
	    pmu.gettextfield (10, text4);
	    long seed = pmu.getlongfield(12);
	    bool genseed = false;
	    bool dosort = true;
	    // SEMANTICS
	    exp_->setrandomvar(
		text1, text2, nvalues, text3, text4, genseed, seed, dosort,
		buf
	    );
	    continue;
	}

	// eg., vary xlith.th_ab srandom 10 from 0.20 to 0.35
	if (pmu.match(buf,
	    "vary", "<ident>", ".", "<ident>", "srandom", "<integer>",
	    "from", "<real>", "to", "<real>")
	){
	    pmu.gettextfield (2, text1);
	    pmu.gettextfield (4, text2);
	    int nvalues = pmu.getintfield (6);
	    pmu.gettextfield (8, text3);
	    pmu.gettextfield (10, text4);
	    bool genseed = true;
	    long seed = 0;
	    bool dosort = true;
	    // SEMANTICS
	    exp_->setrandomvar(
		text1, text2, nvalues, text3, text4, genseed, seed, dosort,
		buf
	    );
	    continue;
	}

	// eg., vary xlith.ax and xlith.ay
	if (pmu.match(buf,
	    "vary", "together", "<ident>", ".", "<ident>", "<text>")
	){
	    pmu.gettextfield (3, text1);
	    pmu.gettextfield (5, text2);
	    pmu.gettextfield (6, text3);
	    // SEMANTICS
	    exp_->setfvgroup(text1, text2, text3, buf);
	    continue;
	}

	// REPORTS
	if (pmu.match (buf, "<ident>", ".", "columns", "=", "<text>")) {
	    pmu.gettextfield (1, text1);
	    pmu.gettextfield (5, text2);
	    if (GLB_debug) 
		cerr << "   TABLE: " << text1 << ".columns=" << text2 << "\n";
	    // SEMANTICS
	    exp_->settablecolumns (text1, text2);
	    continue;
	}

	// EXCON system-specific stuff, such as logging.
	if (pmu.match (buf, "excon", ".", "<ident>", "=", "<text>")) {
	    pmu.gettextfield (3, text1);
	    pmu.gettextfield (5, text2);
	    if (GLB_debug) 
		cerr << "   EXCON: excon." << text1 << "=" << text2 << "\n";
	    // SEMANTICS
	    setsystemparams (text1, text2);
	    continue;
	}
	//
	// nested includes.
	//
	// scan for include files with both "<file>" and "file" formats.
	// steps:
	//    1. is it a standard include directive, ie., within < >
	//       yes - extract the filename portion between < and >.
	//    2. scan for env vars and expand
	//       2.a. does the filename start with '/'?
	//          yes - ignore include search 
	//    3. if search directive is true in 1 AND no leading '/',
	//       search for the expanded filename in search dirs
	//    4. load if found
	//
	if (
	    pmu.match (buf,"$INCLUDE","<name>") ||
	    pmu.match (buf,"include","<name>")
	) {
	    pmu.gettextfield(2, text1);
	    if (GLB_debug) 
		cerr << "   $INCLUDE: " << text1 << "\n";
	    // search and load new file.
	    char newpath[1024];
	    switch(search_file(text1, GLB_file_search_path, newpath)) {
		case 0:
		    load(newpath, nesting+1);
		    break;
		
		case 1:
		    cerr << "Cannot find INCLUDE file " << text1 
			<< " in standard search path (" << path 
			<< ":" << line << ")." << endl;
		    ++errors;
		    break;

		case 2:
		    cerr << "Error opening INCLUDE file " << text1 
			<< " (" << path << ":" << line << ")." << endl;
		    ++errors;
		    break;
		
		default:
		    break;
	    }
	    continue;
	}

	// **NOTE** VARIABLE DEFINITIONS MUST COME LAST (PATTERN IS SUPERSET)
	// VARIABLE DEFINITIONS
	if (pmu.match (buf, "<ident>", ".", "<variable>", "=", "<text>")) {
	    pmu.gettextfield (1, text1);
	    pmu.gettextfield (3, text2);
	    pmu.gettextfield (5, text3);
	    if (GLB_debug) 
		cerr << "   VARIABLE: " << text1 << "." << text2 
		    << "=" << text3 << "\n";
	    // SEMANTICS
	    // ...FIND TOOL INSTANCE 
	    ToolInstance* ti = exp_->gettoolinstance (text1);
	    if (ti == nil) {
		cerr 
		    << "Could not find the tool ``" << text1 
		    << "'' on line " << line << "\nMake sure ``" 
		    << text1 << "'' is defined in tools file.\n";
		continue;
	    }
	    
	    // is it a special variable?
	    if (strcmp(text2, "$GFILE") == 0) {
		if (ti->readgfile(text3) < 0) {
		    cerr 
			<< "Error reading GFILE ``" << text3 
			<< "'' for tool " << text1 << "\n";
		    exit(1);
		}
		continue;
	    }

	    // ...MAKE SURE VARIABLE HAS BEEN DEFINED
	    Var* var = ti->getvar (text2);
	    if (var == nil) {
		cerr 
		    << "The variable ``" << text2 
		    << "'' is not defined for tool ``" << text1
		    << "\nMake sure ``" << text2 << "'' is defined in file.\n";
		continue;
	    }
#ifdef OVERRIDE_BUG
	    // ...MAKE SURE VALUE MATCHES VARIABLE TYPE
	    Var* newvar = var->clone (text3);
	    if (newvar == nil) {
		cerr 
		    << "Unexpected value (" << text3 
		    << ") assigned to variable ``" << text2
		    << "''\nThe variable ``" << text2 << "'' is of type "
		    << var->gettype() << "\n";
		continue;
	    }
	    ti->addvar (newvar);
#else
	    const Value* oldval = var->getval();
	    // ...MAKE SURE VALUE MATCHES VARIABLE TYPE
	    Value* newval = oldval->clone(text3);
	    if (newval == nil) {
		cerr 
		    << "Unexpected value (" << text3 
		    << ") assigned to variable ``" << text2
		    << "''\nThe variable ``" << text2 << "'' is of type "
		    << var->gettype() << "\n";
		continue;
	    }
	    ti->setvarval(var->getname(), newval);
#endif
	    continue;
	}

	// UNRECOGNIZABLE PATTERN 
	cerr << "Unable to understand line " << line <<  " ..." << buf << "\n";
	errors++;  
	if (errors <= 3) 
	    continue;
	// IF WE ARE HERE TOO MAY ERRORS
	cerr << "Too many errors. Check your experiment file.\n";
	fclose (f);  
	exit (1);
    }
    fclose (f);

    if (block_comment_cnt) {
	cerr << "EXCON: unbalanced block comments in experiment file" << endl;
    }

    if (!exp_->expfileok()) {
       cerr << "EXCON aborting due to errors in experiment file\n";
       exit(1);
    }

    if (GLB_debug) 
	exp_->dump();
}

ExpMgr::ExpMgr() : exp_(nil) { }
ExpMgr::~ExpMgr() { delete exp_; exp_ = nil; }
void ExpMgr::setoutput(const char* name) { 
    assume(exp_ != nil); exp_->setoutput(name); 
}
void ExpMgr::unlinkreports() { 
    assume(exp_ != nil);
    exp_->unlinkreports (); 
}
void ExpMgr::run() { 
    assume(exp_ != nil);
    exp_->run(); 
}

void ExpMgr::reportresults() { 
    assume(exp_ != nil);
    exp_->reportresults(); 
}

ToolInstance* ExpMgr::gettoolinstance(
    const char* tname, int iteration
) const{ 
    assume(exp_ != nil);
    return exp_->gettoolinstance(tname, iteration); 
}
Stq* ExpMgr::get_job_sequence(int iteration) const {
    assume(exp_ != nil);
    return exp_->get_job_sequence(iteration); 
}

void ExpMgr::setsystemparams(const char* param, const char* value) {

    if (strcmp(param, "logfile") == 0) {
	GLB_logger->setlogfile(value);
    }
    else if (strcmp(param, "log") == 0) {
	Pmu pmu;
	int pos = 0;  
	char const* rest = value;
	while (1) {
	    pos = pmu.match (rest, "<ident>", ",");  
	    if (pos == 0) pos = pmu.match (rest, "<ident>");  
	    if (pos == 0) break;  rest = &rest[pos];
	    char logcodename[256];  pmu.gettextfield (1, logcodename);
	    int logcode = Logger::nolog;
	    if (strcmp(logcodename, "nolog") == 0)
		logcode = Logger::nolog; 
	    else if (strcmp(logcodename, "debug") == 0)
		logcode = Logger::debug; 
	    else if (strcmp(logcodename, "exec") == 0)
		logcode = Logger::exec; 
	    else if (strcmp(logcodename, "resolve") == 0)
		logcode = Logger::resolve; 
	    else if (strcmp(logcodename, "run") == 0)
		logcode = Logger::run; 
	    else if (strcmp(logcodename, "report") == 0)
		logcode = Logger::report; 
	    else if (strcmp(logcodename, "reporter") == 0)
		logcode = Logger::reporter; 
	    else {
		cerr << "ExpMgr::load: bad log code `" << logcodename << "'\n";
	    }
	    GLB_logger->setlogcode(logcode);
	}
    } else {
	cerr << "ExpMgr::load: bad system parameter code `" << param << "'\n";
    }
    return;
}
