//
// excon.cc: main driver for EXCON engine.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

/****************************************************************************
 *
 *  excon.c: EXCON main driver program.
 *
 ****************************************************************************/

#include <iostream.h>

#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <fcntl.h>
#include <time.h>
#include <sys/wait.h>

#include "exp.h"
#include "global.h"
#include "job.h"
#include "link.h"
#include "logger.h"
#include "misc.h"
#include "pmu.h"
#include "tool.h"
#include "utils.h"
#include "value.h"
#include "variable.h"
#include "xrlstring.h"

/****************************************************************************
 *
 *  GLOBALS
 *
 *
 * any other program sharing code with excon batch back-end must also
 * declare these globals
 *
 ****************************************************************************/

bool GLB_debug    = false;  // TURN ON DEBUGGING INFO
bool GLB_resdebug = false;  // DEBUG RESOLVE STEPS
bool GLB_repdebug = false;  // DEBUG REPORT CONSTRUCTION
bool GLB_rundebug = false;  // DEBUG EXPERIMENT EXECUTION
bool GLB_coldebug = false;  // DEBUG REPORT COLLECTION
char* GLB_programname; // NAME USED TO INVOKE THIS PROGRAM
Pmu* GLB_pmu;
ExpMgr* GLB_expmgr;
ToolsMgr* GLB_toolsmgr;
CleanupMgr* GLB_cleanupmgr;
Stats* GLB_stats;
JobMgr* GLB_jobmgr;
Logger* GLB_logger;
Stq* GLB_file_search_path = 0;

/****************************************************************************
 *
 *  MAIN 
 *
 ****************************************************************************/

int main(int argc, char** argv) {

    cout << "EXCON Batch Version 1.0." << endl;
    
    if (!getenv("SHADOW_ROOT")) {
	cerr << "****** EXCON/SHADOW startup warning ****** " << endl
	    << "Please initialize SHADOW environment with .shadowrc"
	    << endl;
	exit(1);
    }

    // INIT GLOBAL OBJECTS
    GLB_programname = getprogramname (argv);
    GLB_cleanupmgr  = new CleanupMgr ();  // CLEANUP FILES WE LEAVE AROUND
    GLB_pmu         = new Pmu        ();  // GLOBAL PATTERN MATCH UNIT
    GLB_stats       = new Stats      ();  // PROGRAM STATS COLLECTOR
    GLB_toolsmgr    = new ToolsMgr   ();  // TOOLS MANAGER
    GLB_expmgr      = new ExpMgr     ();  // EXPERIMENT MANAGER
    GLB_logger      = new Logger     (nil, Logger::nolog, false);
    GLB_file_search_path = new Stq;

    char buf[1024];
    int njobs = 1;
    if (getflagval ('j', argc, argv, buf)) {
	int tmp;
	if (sscanf(buf, "%d", &tmp) == 1)
	    njobs = tmp;
    }
    GLB_jobmgr      = new JobMgr (njobs);  // JOB EXECUTION MANAGER

    if (getflag('d', argc, argv)) {	// DEBUG ON
	GLB_debug    = true;  // TURN ON DEBUGGING INFO
	GLB_resdebug = true;  // DEBUG RESOLVE STEPS
	GLB_repdebug = true;  // DEBUG REPORT CONSTRUCTION
	GLB_rundebug = true;  // DEBUG EXPERIMENT EXECUTION
	GLB_coldebug = true;  // DEBUG REPORT COLLECTION
    }

    if (!getflagval('l', argc, argv, buf)) {
	strcpy(buf, "excon.log");
    }
    GLB_logger->setlogfile(buf);
    GLB_logger->setlogcode(Logger::nolog);

    //
    // initialize the search path for include files in TOOLS and EXP
    // files. The syntax is the following:
    //    excon [options] -I dir1:[dir2[:dir3]]]
    // the search path is then set to dir1 ... dirn in order, and then
    // "." and $SHADOW_DATA_DIR (which is the default if no searchpath is 
    // specified).
    //
    if (getflagval ('I', argc, argv, buf)) {
	char const* include_dir = strtok(buf, ",:");
	if (include_dir)
	    GLB_file_search_path->enq(strcpy_with_alloc(include_dir));
	while((include_dir = strtok(0, ",:"))) {
	    GLB_file_search_path->enq(strcpy_with_alloc(include_dir));
	}
    }

    GLB_file_search_path->enq(strcpy_with_alloc("."));
    char* shadow_data_dir = getenv("SHADOW_DATA_DIR");
    if (!shadow_data_dir) {
	sprintf(buf, "%s/data", getenv("SHADOW_ROOT"));
    } else {
	strcpy(buf, shadow_data_dir);
    }
    GLB_file_search_path->enq(strcpy_with_alloc(buf));

    char exp[1024];  char tools[1024];  char output[1024];

    if (getflagval ('e', argc, argv, exp) == 0)
	promptforval (
	    "Enter experiment file name (default = excon.exp) > ", exp
	);
    if (strlen(exp) == 0) 
	 strcpy (exp, "excon.exp");

    switch(search_file(exp, GLB_file_search_path, buf)) {
	case 0:
	    strcpy(exp, buf);
	    cerr << "EXCON: Loading EXP file " << exp << " ..." << endl;
	    break;
	
	case 1:
	    cerr << "EXCON Error: Cannot find EXPERIMENT file " << exp 
		<< " in standard search path." << endl;
	    exit(1);
	    break;

	case 2:
	    cerr << "EXCON Error: Cannot open EXPERIMENT file " << exp << endl;
	    exit(1);
	    break;
	
	default:
	    break;
    }

    if (getflagval ('t', argc, argv, tools) == 0) {
	promptforval ("Enter tools file name   (default = <excon.tools>) > ", 
	    tools
	);
    }
    if (strlen(tools) == 0) 
	strcpy (tools,"<excon.tools>");
    
    switch(search_file(tools, GLB_file_search_path, buf)) {
	case 0:
	    strcpy(tools, buf);
	    cerr << "EXCON: Loading TOOLS file " << tools << " ..." << endl;
	    break;
	
	case 1:
	    cerr << "EXCON Error: Cannot find TOOLS file " << tools 
		<< " in standard search path." << endl;
	    exit(1);
	    break;

	case 2:
	    cerr << "EXCON Error: Cannot open TOOLS file " << tools << endl;
	    exit(1);
	    break;
	
	default:
	    break;
    }

    // ASK WHERE OUTPUT SHOULD GO
    char output2[1024];  char buf2[1024];
    if (getflagval ('o', argc, argv, output) == 0) 
	strcpy (output, "");
    while (1) {
	if (strlen(output) == 0) 
	    promptforval (
		"Enter output file name prefix  (default = table) > ", output
	    );
	if (strlen(output) == 0) 
	    strcpy (output, "table");

	// TEST TO SEE IF THE FILE EXISTS (WITH SUFFIX)
	strcpy (output2, output);  strcat (output2, "1");
	if (fileexists(output2)) {
	    strcpy (buf, "The file '"); strcat (buf, output2);  
	    strcat (buf,"' already exists. Overwrite?(y or n)(default = y) > ");
	    promptforval (buf, buf2);
	    if (buf2[0] == 'n') 
		strcpy (output, "");
	    else 
		break;
	}
	else 
	    break;
    }

    // LOAD TOOLS FILE
    GLB_toolsmgr->load (tools);

    // LOAD EXPERIMENT FILE
    GLB_expmgr->load (exp);

    // SET the output file (report) Must not set before loading the exp.
    GLB_expmgr->setoutput (output);

    GLB_expmgr->unlinkreports (); 

    // RUN THE EXPERIMENT...SET PRIORITY LOW
    if (getflag('n', argc, argv)) {	// Nice ON
	nice (16);
    }

    GLB_expmgr->run ();

    // REPORT ON RESULTS
    cout << "\nExperiment completed successfully.\n";

    // ...STATISTICS
    GLB_stats->report();
    // ...RESULTS
    GLB_expmgr->reportresults ();
}
