#
# autosave.tcl: routines for autosave mode in SHADOW GUI
#
# ------------------------------------------------
# Mumit Khan <khan@NanoTech.Wisc.EDU>
# Center for NanoTechnology
# University of Wisconsin-Madison
# 1415 Engineering Dr., Madison, WI 53706
# ------------------------------------------------
#
# Copyright (c) 2000 Mumit Khan
#

# Routines to support Autosave mode in the GUI. The basic idea is that,
# when enabled, auto-save feature will save the current workspace using
# either a user-defined timer trigger or explicit trigger (useful if the
# client wants to autosave before running code that may potentially kill
# the application for example). 
#
# The usual steps client takes are the following:
#   Initialization, typically during client's own initialization:
#     (a) source'ing the file initializes the default values, so no 
#     action is required.
#     (b) set the triggers (or just use the defaults). Triggers can be
#     set/unset or modified during the active state, so this step can
#     occur anywhere.
#
#   Activation/Deactivation:
#     (c) call autosave::activate to start. This is what the client
#     typically does when the user starts editing the system.
#         ... client code runs ...
#     (d) call autosave::deactivate to stop. This is what the client
#     typically does before exiting. Note that this *deletes* the current
#     autosave file.
#
# In the event of a crash, (d) is never called, and the user can
# retrieve hopefully most of the lost information from the autosave'd
# file.
#

# The client turns on/off autosave feature, and controls using the
# following interfaces:
#
#  autosave::activate .............. Activate the feature
#  autosave::deactivate ............ De-activate the feature. This also 
#                                    deletes any autosave files currently 
#                                    in play.
#  autosave::save .................. It'll autosave if it satisfies the 
#                                    criteria. Explicit trigger clients 
#                                    call this directly.
#
# Attribute query interfaces:
#  
#  autosave::active ................ Currently active?
#  autosave::triggers_on ........... Are any of the triggers on?
#  autosave::get_internal_state .... For debugging, no other use to clients.
#
# Attribute changing interfaces:
#
#  autosave::set_time_trigger ...... Time interval (0 disables)
#  autosave::get_time_trigger ...... current status
#  autosave::set_explicit_trigger .. Explicit trigger turn-on
#  autosave::get_explicit_trigger .. current status
#
# Misc interfaces:
#
#  autosave::scan .................. Return a list of autosave files
#  autosave::browse ................ Graphically browse autosave files [TODO]
#  autosave::clean ................. Remove all autosave files
#
#  autosave::set_message_callback .. Client supplied callback when saving
#
# Globals:
#  
#  active .......................... Activated?
#  time_trigger .................... Triggered by timer?
#  explicit_trigger ................ Triggered explicitly by user?
#
#  file_format_spec ................ Sprintf style spec for making filenames
#  file_regsub_spec ................ Regsub exp for getting id from filename
#  file_pattern .................... glob regexp for autosave filenames
#  filename ........................ current autosave filename
#
#  delta_time ...................... The time interval in minutes
#  timer_id ........................ The timer (after) id
#

#
# Namespace specific stuff.
#

namespace eval autosave {
  variable autosave

  # Don't export anything.
  namespace export -clear

  proc initialize {} {
    variable autosave

    set autosave(file_format_spec)	{autosave%d.ws}
    set autosave(file_regsub_spec)	{autosave([0-9]*).ws}
    set autosave(file_pattern)		{autosave[0-9]*.ws}
    set autosave(filename)		""

    set autosave(delta_time)		5
    set autosave(timer_id)		""

    set autosave(active)		0

    # Default policy: Turn auto-save on for everything.
    set autosave(time_trigger)		[expr {$autosave(delta_time) > 0}]
    set autosave(explicit_trigger)	1

    # message callback method.
    set autosave(message_callback)	""
  }

  initialize
}

proc autosave::get_internal_state {_state} {
  variable autosave
  upvar $_state state
  array set state [array get autosave]
}

proc autosave::active {} {
  variable autosave
  return $autosave(active)
}

proc autosave::triggers_on {} {
  variable autosave

  return [expr {$autosave(time_trigger) || $autosave(explicit_trigger)}]
}

proc autosave::set_time_trigger {delta_time} {
  variable autosave

  if {$autosave(active)} {
    catch {after cancel $autosave(timer_id)}

    if {$delta_time == 0} {
      set autosave(timer_id) ""
    } else {
      set time_ms [expr {$delta_time * 60 * 1000}]
      set autosave(timer_id) [after $time_ms autosave::_trigger_timer]
    }
  }
  set autosave(time_trigger) [expr {$delta_time > 0}]
  set autosave(delta_time) $delta_time
}

proc autosave::get_time_trigger {} {
  variable autosave

  return [list $autosave(time_trigger) $autosave(delta_time)]
}

proc autosave::set_explicit_trigger {activate} {
  variable autosave
  set autosave(explicit_trigger) $activate
}

proc autosave::get_explicit_trigger {activate} {
  variable autosave
  return autosave(explicit_trigger)
}

proc autosave::set_message_callback {callback} {
  variable autosave
  set autosave(message_callback) $callback
}

proc autosave::_set_message {text} {
  variable autosave
  if {$autosave(message_callback) != ""} {
    catch {eval $autosave(message_callback) $text}
  }
}

proc autosave::_trigger_timer {} {
  variable autosave

  # Autosave feature may have been deactivated since this one was
  # triggered.
  if {$autosave(active)} {
    autosave::save
    vputs "autosave triggered. re-arming if necessary"
    if {$autosave(delta_time) > 0} {
      set time_ms [expr {$autosave(delta_time) * 60 * 1000}]
      set autosave(timer_id) [after $time_ms autosave::_trigger_timer]
    }
  }
}

proc autosave::activate {} {
  variable autosave

  if {$autosave(active)} {
    return
  }

  vputs "Activating autosave mode!"

  set cur_files [lsort -dict [glob -nocomplain $autosave(file_pattern)]]
  if {[llength $cur_files]} {
    set last_file [lindex $cur_files end]
    regsub $autosave(file_regsub_spec) $last_file {\1} last_id
    set id [expr {$last_id + 1}]
  } else {
    set id 1
  }

  set autosave(filename) [file join [pwd] \
                          [format $autosave(file_format_spec) $id]]
  if {$autosave(time_trigger)} {
    set time_ms [expr {$autosave(delta_time) * 60 * 1000}]
    set autosave(timer_id) [after $time_ms autosave::_trigger_timer]
  }
  set autosave(active) 1

  _set_message "Activating autosave mode to $autosave(filename) ..."
}

proc autosave::deactivate {} {
  variable autosave

  if {! $autosave(active)} {
    return
  }

  vputs "De-activating autosave mode!"

  set autosave(active) 0
  catch {after cancel $autosave(timer_id)}
  vputs "Deleting autosave file $autosave(filename) ... "
  file delete $autosave(filename)

  _set_message "Deactivating autosave mode, deleted $autosave(filename) ..."
}

proc autosave::save {} {
  variable autosave

  if {! $autosave(active) || ![autosave::triggers_on]} {
    return
  }

  vputs "Saving autosave file $autosave(filename) ... "
  _set_message "Autosaving to $autosave(filename) ... "
  State:save $autosave(filename)
  _set_message "Autosaving to $autosave(filename) ... done."
}

proc autosave::scan {dir {full_paths 0}} {
  variable autosave

  if {[catch {set curdir [pwd]; cd $dir; pwd} msg]} {
    set errmsg "Error getting list of files in $dir ($msg)"
    if {[info command tk] != ""} {
      tk_dialog .error "Autosave scan error" $errmsg error 0 Dismiss
    } else {
      error "$errmsg"
    }
    return
  } else {
    cd $curdir
    set dir $msg
  }

  set autosave_files \
    [lsort -dict [glob -nocomplain \
		  [file join $dir $autosave(file_pattern)]]]
  if {! $full_paths} {
    # plus one for the directory separator
    set dir_len [expr {[string length $dir] + 1}]
    set tmp_list [list]
    foreach f $autosave_files {
      lappend tmp_list [string range $f $dir_len end]
    }
    set autosave_files $tmp_list
  }
  return $autosave_files
}

proc autosave::clean {{dir .}} {
  variable autosave

  set autosave_files [autosave::scan $dir]
  if {[llength $autosave_files]} {
    file delete [join $autosave_files]
  }
}

#
# Autosave browser
#
proc autosave::browse {dir} {
  variable autosave

  set autosave_files [autosave::scan $dir]
}
