/*
 * namelist.h: C interface to SHADOW FORTRAN NAMELIST data
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * ------------------------------------------------
 *             SHADOW
 *    Center for X-ray Lithography
 *   University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 */

#ifndef shadow_namelist_h
#define shadow_namelist_h

#ifdef __cplusplus 
extern "C" {
#endif

/*************************************************************************
 *
 * C library for reading/writing FORTRAN NAMELISTs in SHADOW.
 *
 *************************************************************************/

#ifndef nil
# define nil 0
#endif

#ifndef FALSE
# define FALSE 0
#endif

#ifndef TRUE
# define TRUE 1
#endif

/*
 * data types for various SHADOW common block variables. These are all
 * FORTRAN data types, which all have C equiv except for CHARACTER(*)
 */
typedef enum {
    DT_int, 				/* integer data type */
    DT_flt, 				/* real*4 data type */
    DT_dbl, 				/* real*8 data type */
    DT_str				/* FORTRAN character(*) data type */
} DataType;


/*
 * Various SHADOW namelists used in the start.* files.
 */
typedef enum {
    NML_source,				/* TSOUR namelist */
    NML_oe				/* TOTOE[1,2] namelists */
} NmlType;

/*
 * Individual elements of the NAMELIST data structure. The ``data'' member
 * is a pointer into the appropriate COMMON block, and is cast to char*
 * for convenience. The NAMELIST variable names are ALWAYS converted to
 * lower case.
 */
typedef struct {
    char* 	synopsis;		/* Description of parameter */
    int		allow_change;		/* should user be able to change it */
    char* 	name;			/* FORTRAN namelist name */
    DataType	data_type;		/* DT_int|DT_flt|DT_dbl|DT_str */
    char* 	value;			/* pointer into the common block */
} NamelistData;

/*
 * The C equiv of the namelist data structure.
 */
typedef struct {
    NmlType	nml_type;		/* the type (NML_source, NML_oe) */
    int		size;			/* size of the namelist */
    NamelistData* nml_data;		/* all the elements */
} Namelist;

/*************************************************************************/

/* 
 * routines to read and write fortran namelist and convert to gfiles. 
 */

extern Namelist* read_fnamelist(const char *nmlfile, NmlType type);
extern Namelist* read_gfile(const char *gfile, NmlType type);
extern int write_fnamelist(const char *nmlfile, NmlType type);
extern int write_gfile(const char *outfile, NmlType type);

/*
 * these 2 are wrappers for FORTRAN RWNAME, which chooses the right
 * format to read/write (either GFILE or FORTRAN namelist format).
 */
extern Namelist* read_namelist(const char* filename, NmlType type);
extern int write_namelist(const char* filename, NmlType type);

/*
 * now the reverse FORTRAN->C interface. Note that G77 requires 2
 * underscores when names already have one, hence the G77_NAME_HACK
 */

#if G77
extern void G77_HACK_NAME(read_gfile_f) (
    const char *gfile, int* type, int* status, unsigned long
);
extern void G77_HACK_NAME(write_gfile_f) (
    const char *outfile, int* type, int* status, unsigned long
);
#else
extern void F77_NAME(read_gfile_f) (
    const char *gfile, int* type, int* status, unsigned long
);
extern void F77_NAME(write_gfile_f) (
    const char *outfile, int* type, int* status, unsigned long
);
#endif

/* 
 * routines to get and manipulate the Namelists and its data members.
 */

extern Namelist* clone_namelist(const Namelist* namelist);
extern Namelist* get_namelist(NmlType type);
extern NamelistData* get_namelist_data(
    const char* name, const Namelist* namelist
);
extern int change_value(
    const char* name, const char* newval, Namelist* namelist
);
extern char* sprint_value(const NamelistData* data);

/*************************************************************************/

#ifdef __cplusplus 
}
#endif

#endif /*shadow_namelist_h*/
