#
# gui.tcl: routines to build the GUI
#
# ------------------------------------------------
# Mumit Khan <khan@xraylith.wisc.edu>
# Center for X-ray Lithography
# University of Wisconsin-Madison
# 3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Copyright (c) 1994-1996 Mumit Khan
#
#

#
# build all the GUI components here, but the callbacks do not belong
# in this file. Those are supplied by client routines.
#

#----------------------------------------------------------------------#
# Build various parts of the UI
#----------------------------------------------------------------------#

#----------------------------------------------------------------------#
# Make the top pulldown menu bar
#----------------------------------------------------------------------#

proc GUI:make_topmenu_pre_v8 {menu_name} {
    global gvars
    # build widget .topmenu
    frame $menu_name \
	-background {LightGray} \
	-borderwidth {2} \
	-relief {raised} \
	-class TopMenu

    lappend menu_names $menu_name.mFile
    # build widget $menu_name.mFile
    menubutton $menu_name.mFile \
	-background {LightGray} \
	-foreground {black} \
	-menu "$menu_name.mFile.m" \
	-text {File} \
	-underline {0}

    # build widget $menu_name.mFile.m
    menu $menu_name.mFile.m \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    # needed to add workspace files later on.
    set gvars(file_cmd_menu) $menu_name.mFile.m

    $menu_name.mFile.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {new_state_cmd} \
	-label {New Workspace} \
	-accelerator {Ctrl+N} \
	-underline {0}
    
    bind all <Control-n>	"new_state_cmd"

    $menu_name.mFile.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {load_state_cmd} \
	-label {Open Workspace ...} \
	-accelerator {Ctrl+O} \
	-underline {0}

    bind all <Control-o>	"load_state_cmd"

    $menu_name.mFile.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {save_state_cmd} \
	-label {Save Workspace} \
	-accelerator {Ctrl+S} \
	-underline {0}

    bind all <Control-s>	"save_state_cmd"

    $menu_name.mFile.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {save_state_as_cmd} \
	-label {Save As ...} \
	-underline {5}

    $menu_name.mFile.m add separator

    # ============================================== 

    # build cascade menu for importing/export shadow namelist files
    $menu_name.mFile.m add cascade \
	-menu "$menu_name.mFile.m.mImport" \
	-label {Import} \
	-underline {0}

    # build widget $menu_name.mFile.m.mImport
    menu $menu_name.mFile.m.mImport \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mFile.m.mImport add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Source ...} \
	-command {load_source_cmd} \
	-underline {-1}

    $menu_name.mFile.m.mImport add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {OE ...} \
	-command {load_oe_cmd} \
	-underline {-1}

    $menu_name.mFile.m.mImport add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {System ...} \
	-command {load_system_cmd} \
	-underline {-1}

    # build cascade menu for importing/export shadow namelist files
    $menu_name.mFile.m add cascade \
	-menu "$menu_name.mFile.m.mExport" \
	-label {Export} \
	-underline {0}

    # build widget $menu_name.mFile.m.mExport
    menu $menu_name.mFile.m.mExport \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mFile.m.mExport add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Source ...} \
	-command {save_source_cmd} \
	-underline {-1}

    $menu_name.mFile.m.mExport add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {OE ...} \
	-command {save_oe_cmd} \
	-underline {-1}

    $menu_name.mFile.m.mExport add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {System ...} \
	-command {save_system_cmd} \
	-underline {-1}

    $menu_name.mFile.m add separator

    $menu_name.mFile.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {exit_cmd} \
	-label {Exit} \
	-underline {1} \
	-accelerator {Ctrl+X}

    bind all <Control-x>	"exit_cmd"

    lappend menu_names $menu_name.mEdit
    # build widget $menu_name.mEdit
    menubutton $menu_name.mEdit \
	-background {LightGray} \
	-foreground {black} \
	-menu "$menu_name.mEdit.m" \
	-text {Edit} \
	-underline {0}

    # build widget $menu_name.mEdit.m
    menu $menu_name.mEdit.m \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mEdit.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Copy Selected} \
	-command {copy_selected_cmd} \
	-accelerator {Ctrl+C} \
	-underline {-1}
    
    bind all <Control-c>	"copy_selected_cmd"

    $menu_name.mEdit.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Delete Selected} \
	-command {delete_selected_cmd} \
	-underline {-1}

    $menu_name.mEdit.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Paste Selected} \
	-command {paste_selected_cmd} \
	-accelerator {Ctrl+V} \
	-underline {-1}

    bind all <Control-v>	"paste_selected_cmd"

    $menu_name.mEdit.m add separator

    # build cascade menu for Adding objects
    $menu_name.mEdit.m add cascade \
	-menu "$menu_name.mEdit.m.mAdd" \
	-label {Add} \
	-underline {-1}

    # build widget $menu_name.mEdit.m.mAdd
    menu $menu_name.mEdit.m.mAdd \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mEdit.m.mAdd add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {add_src_cmd} \
	-label {Source} \
	-underline {-1}

    $menu_name.mEdit.m.mAdd add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {add_oe_cmd 1} \
	-label {OE...} \
	-underline {-1}

    $menu_name.mEdit.m.mAdd add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {add_scr_cmd 1} \
	-label {Screen...} \
	-underline {-1}

    # build cascade menu for Deleting objects
    $menu_name.mEdit.m add cascade \
	-menu "$menu_name.mEdit.m.mParam" \
	-label {Parameters} \
	-underline {-1}

    # build widget $menu_name.mEdit.m.mParam
    menu $menu_name.mEdit.m.mParam \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mEdit.m.mParam add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Selected} \
	-command {edit_selected_cmd} \
	-underline {-1}

    $menu_name.mEdit.m.mParam add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Source} \
	-command {edit_src_cmd} \
	-underline {-1}

    $menu_name.mEdit.m.mParam add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {OE...} \
	-command {edit_oe_cmd} \
	-underline {-1}

    $menu_name.mEdit.m.mParam add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Screen...} \
	-command {edit_scr_cmd} \
	-underline {-1}

    # build cascade menu for Deleting objects
    $menu_name.mEdit.m add cascade \
	-menu "$menu_name.mEdit.m.mDelete" \
	-label {Delete} \
	-underline {-1}

    # build cascaded menu for $menu_name.mDelete
    menu $menu_name.mEdit.m.mDelete \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mEdit.m.mDelete add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {undo_deleted_cmd} \
	-label {Undo last} \
	-underline {-1}
    
    set gvars(undo_del_btn_config) "$menu_name.mEdit.m.mDelete entryconfigure 0"

    $menu_name.mEdit.m.mDelete add separator

    $menu_name.mEdit.m.mDelete add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {delete_selected_cmd} \
	-label {Selected} \
	-underline {-1}

    $menu_name.mEdit.m.mDelete add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {delete_src_cmd} \
	-label {Source} \
	-underline {-1}

    $menu_name.mEdit.m.mDelete add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {delete_oe_cmd} \
	-label {OE...} \
	-underline {-1}

    $menu_name.mEdit.m.mDelete add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {delete_all_oe_cmd} \
	-label {All OEs} \
	-underline {-1}

    $menu_name.mEdit.m.mDelete add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {delete_scr_cmd} \
	-label {Screen...} \
	-underline {-1}

    lappend menu_names $menu_name.mRun
    # build widget $menu_name.mRun
    menubutton $menu_name.mRun \
	-background {LightGray} \
	-foreground {black} \
	-menu "$menu_name.mRun.m" \
	-text {Run} \
	-underline {0}

    # build widget $menu_name.mRun.m
    menu $menu_name.mRun.m \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mRun.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Selected} \
	-command {run_selected_cmd} \
	-underline {-1}

    $menu_name.mRun.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Source} \
	-command {run_source_cmd} \
	-underline {-1}

    $menu_name.mRun.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {System} \
	-command {run_system_cmd} \
	-underline {-1}

    lappend menu_names $menu_name.mWorkspace
    # build widget $menu_name.mWorkspace
    menubutton $menu_name.mWorkspace \
	-background {LightGray} \
	-foreground {black} \
	-menu "$menu_name.mWorkspace.m" \
	-text {Workspace} \
	-underline {0}

    # build widget $menu_name.mWorkspace.m
    menu $menu_name.mWorkspace.m \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mWorkspace.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Annotate} \
	-command {annotate_workspace_cmd} \
	-underline {0}

    $menu_name.mWorkspace.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Show Annotation} \
	-command {show_annotation_cmd} \
	-underline {0}

    $menu_name.mWorkspace.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Show Info ...} \
	-command {show_workspace_info_cmd} \
	-underline {5}

    lappend menu_names $menu_name.mTools
    # build widget $menu_name.mTools
    menubutton $menu_name.mTools \
	-background {LightGray} \
	-foreground {black} \
	-menu "$menu_name.mTools.m" \
	-text {Tools} \
	-underline {0}

    # build widget $menu_name.mTools.m
    menu $menu_name.mTools.m \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mTools.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Plot} \
	-command {plot_cmd} \
	-underline {0}

    $menu_name.mTools.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Layout} \
	-command {layout_cmd} \
	-underline {0}

    $menu_name.mTools.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {New Shell Window} \
        -command {run_new_shell_cmd} \
	-underline {4}

    $menu_name.mTools.m add separator

    set tool_names [list \
	abrefc angle beam bragg col_2 comb_ray cutoff dose epath ergset \
	ffresnel filex fmax focnew gen_source genlib grade_mlayer \
	grating_set h2kol histo1 histo2 histo3 intens jntpscalc loss \
	machine_par make_id menu minmax mirinfo mlayer nphoton patch \
	plotfoc plotspace plotxy power pre_mlayer preplot preplot2 \
	prerefl presurface primvs pspread pwr_dens qeff readcdf recolor \
	reflag retrace seed sincrad slits sourcinfo srcdf srcomp srfull \
	srfunc surplot sync_spectra sysinfo sysplot torus translate \
	transmit undul undul_cdf undul_phot undul_set user_undul variance \
	wiggler_spectrum \
    ]

    set preproc_names [list \
	bragg ergset grade_mlayer histo3 jntpscalc patch pre_mlayer  \
	prerefl presurface qeff readcdf reflag \
    ]

    set postproc_names [list \
	h2kol histo1 plotxy plotspace preplot2 preplot angle comb_ray \
	ffresnel focnew histo2 intens loss minmax mirinfo plotfoc pspread \
	pwr_dens recolor retrace seed slits sourcinfo surplot sysinfo \
	sysplot tdconn translate \
    ]

    set optics_names [list \
	abrefc cutoff dose grating_set mlayer torus transmit \
    ]

    set general_names [list \
	col_2 filex fmax variance \
    ]

    set misc_names [list \
	make_id undul \
    ]

    # build cascade menu for Preprocs
    $menu_name.mTools.m add cascade \
	-menu "$menu_name.mTools.m.mPreprocs" \
	-label {Pre-processor Tools} \
	-underline {-1}

    # build cascaded menu for $menu_name.mTools.m.mPreprocs
    menu $menu_name.mTools.m.mPreprocs \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    foreach cmd $preproc_names {
	$menu_name.mTools.m.mPreprocs add command \
	    -activebackground {SteelBlue2} \
	    -background {LightGray} \
	    -label [list $cmd] \
	    -command "run_shell_cmd $cmd" \
	    -underline {-1}
    }

    # build cascade menu for Postprocs
    $menu_name.mTools.m add cascade \
	-menu "$menu_name.mTools.m.mPostprocs" \
	-label {Post-processor Tools} \
	-underline {-1}

    # build cascaded menu for $menu_name.mTools.m.mPostprocs
    menu $menu_name.mTools.m.mPostprocs \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    foreach cmd $postproc_names {
	$menu_name.mTools.m.mPostprocs add command \
	    -activebackground {SteelBlue2} \
	    -background {LightGray} \
	    -label [list $cmd] \
	    -command "run_shell_cmd $cmd" \
	    -underline {-1}
    }

    # build cascade menu for Optics
    $menu_name.mTools.m add cascade \
	-menu "$menu_name.mTools.m.mOptics" \
	-label {Optics Tools} \
	-underline {-1}

    # build cascaded menu for $menu_name.mTools.m.mOptics
    menu $menu_name.mTools.m.mOptics \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    foreach cmd $optics_names {
	$menu_name.mTools.m.mOptics add command \
	    -activebackground {SteelBlue2} \
	    -background {LightGray} \
	    -label [list $cmd] \
	    -command "run_shell_cmd $cmd" \
	    -underline {-1}
    }

    # build cascade menu for General
    $menu_name.mTools.m add cascade \
	-menu "$menu_name.mTools.m.mGeneral" \
	-label {General} \
	-underline {-1}

    # build cascaded menu for $menu_name.mTools.m.mGeneral
    menu $menu_name.mTools.m.mGeneral \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    foreach cmd $general_names {
	$menu_name.mTools.m.mGeneral add command \
	    -activebackground {SteelBlue2} \
	    -background {LightGray} \
	    -label [list $cmd] \
	    -command "run_shell_cmd $cmd" \
	    -underline {-1}
    }

    # build cascade menu for Misc
    $menu_name.mTools.m add cascade \
	-menu "$menu_name.mTools.m.mMisc" \
	-label {Misc} \
	-underline {-1}

    # build cascaded menu for $menu_name.mTools.m.mMisc
    menu $menu_name.mTools.m.mMisc \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    foreach cmd $misc_names {
	$menu_name.mTools.m.mMisc add command \
	    -activebackground {SteelBlue2} \
	    -background {LightGray} \
	    -label [list $cmd] \
	    -command "run_script_cmd $cmd" \
	    -underline {-1}
    }

    lappend menu_names $menu_name.mOptions
    # build widget $menu_name.mOptions
    menubutton $menu_name.mOptions \
	-background {LightGray} \
	-foreground {black} \
	-menu "$menu_name.mOptions.m" \
	-text {Options} \
	-underline {0}

    # build widget $menu_name.mOptions.m
    menu $menu_name.mOptions.m \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mOptions.m add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Verbose Mode (Debugging)} \
	-variable gvars(verbose) \
	-underline {-1}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Change directory} \
	-command {chdir_cmd} \
	-underline {-1}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Redraw Parameter Editor} \
	-command {redraw_current_page_cmd} \
	-underline {-1}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Redraw Beamline Editor} \
	-command {redraw_editbar_cmd} \
	-underline {-1}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Show Run Window} \
	-command {show_run_window_cmd} \
	-underline {-1}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Clear clipboard} \
	-command {clear_clipboard_cmd} \
	-underline {-1}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Clear Param Editor} \
	-command {edit_nothing} \
	-underline {-1}

    $menu_name.mOptions.m add separator

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Remember Current State} \
	-command {save_beamline_state_cmd} \
	-underline {-1}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Restore Remembered State} \
	-command {restore_beamline_state_cmd} \
	-underline {-1}

    $menu_name.mOptions.m add separator

    #
    # CHECK/FIXME: Add the autosave* preferences here as well. Currently 
    # only done for Tk 8.0+ menu proc. 
    #

    # build cascade menu for Preferences
    $menu_name.mOptions.m add cascade \
	-menu "$menu_name.mOptions.m.mPreferences" \
	-label {Preferences} \
	-underline {-1}

    # build cascaded menu for $menu_name.mOptions.m.mPreferences
    menu $menu_name.mOptions.m.mPreferences \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Add SRC at Startup} \
	-variable gprefs(add_src_at_startup) \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Edit SRC at Startup} \
	-variable gprefs(edit_src_at_startup) \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Edit after adding} \
	-variable gprefs(edit_added_obj) \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Change to workspace directory after loading} \
	-variable gprefs(cd_to_workspace_dir) \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Change to working directory from last time} \
	-variable gprefs(cd_to_last_working_dir) \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Restore Window size from last run} \
	-variable gprefs(restore_last_geom) \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Balloon Help} \
	-variable gprefs(balloon_help) \
	-command "balloon_help_cmd" \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Show Toolbar} \
	-variable gprefs(show_toolbar) \
	-command {show_toolbar_cmd} \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Message window at Bottom} \
	-variable gprefs(msg_window_at_bottom) \
	-command {show_msg_window_cmd} \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Show full path in Workspace history menu} \
	-variable gprefs(show_path_in_workspace_history) \
	-command {update_workspace_history} \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Show Command Window (Debugging)} \
	-variable gprefs(show_cmd_window) \
	-command {show_cmd_window_cmd} \
	-underline {-1}

    lappend menu_names $menu_name.mHelp
    # build widget $menu_name.mHelp
    menubutton $menu_name.mHelp \
	-background {LightGray} \
	-foreground {black} \
	-menu "$menu_name.mHelp.m" \
	-text {Help} \
	-underline {0}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Save preferences} \
	-command {save_prefs_cmd} \
	-underline {0}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Reload preferences} \
	-command {load_prefs_cmd} \
	-underline {0}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Reset to defaults} \
	-command {reset_prefs_cmd} \
	-underline {2}

    # build widget $menu_name.mHelp.m
    menu $menu_name.mHelp.m \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mHelp.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {about_cmd} \
	-label {About} \
	-underline {0}

    $menu_name.mHelp.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {about_author_cmd} \
	-label {Author} \
	-underline {1}

    $menu_name.mHelp.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {help_on_topmenu_cmd} \
	-label {On Menubar} \
	-underline {0}

    $menu_name.mHelp.m add separator

    # build cascade menu for Preferences
    $menu_name.mHelp.m add cascade \
	-menu "$menu_name.mHelp.m.mDemo" \
	-label {Demos} \
	-underline {-1}

    # build cascaded menu for $menu_name.mHelp.m.mDemo
    menu $menu_name.mHelp.m.mDemo \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mHelp.m.mDemo add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command {run_demo_cmd toroid.ws} \
	-label {Toroidal Mirror} \
	-underline {7}

    # pack widget $menu_name
    pack append $menu_name \
	$menu_name.mFile {left frame center} \
	$menu_name.mEdit {left frame center} \
	$menu_name.mRun {left frame center} \
	$menu_name.mWorkspace {left frame center} \
	$menu_name.mTools {left frame center} \
	$menu_name.mOptions {left frame center} \
	$menu_name.mHelp {right frame center}

    eval tk_menuBar $menu_name $menu_names
    focus $menu_name

    return $menu_name
}

proc GUI:make_topmenu {menu_name} {
    global gvars tcl_platform

    catch {tix resetoptions TK TK}

    # make menu bar itself
    menu $menu_name -tearoff 0

    # build widget $menu_name.file
    $menu_name add cascade -label File -menu $menu_name.file -underline 0
    menu $menu_name.file -tearoff 0

    # needed to add workspace files later on.
    set gvars(file_cmd_menu) $menu_name.file

    $menu_name.file add command \
	-command {new_state_cmd} \
	-label {New Workspace} \
	-accelerator {Ctrl+N} \
	-underline {0}
    
    bind all <Control-n>	"new_state_cmd"

    $menu_name.file add command \
	-command {load_state_cmd} \
	-label {Open Workspace ...} \
	-accelerator {Ctrl+O} \
	-underline {0}

    bind all <Control-o>	"load_state_cmd"

    $menu_name.file add command \
	-command {save_state_cmd} \
	-label {Save Workspace} \
	-accelerator {Ctrl+S} \
	-underline {0}

    bind all <Control-s>	"save_state_cmd"

    $menu_name.file add command \
	-command {save_state_as_cmd} \
	-label {Save As ...} \
	-underline {0}

    $menu_name.file add separator

    # ============================================== 

    menu $menu_name.file.import -tearoff 0

    # build cascade menu for importing/export shadow namelist files
    $menu_name.file add cascade \
	-menu "$menu_name.file.import" \
	-label {Import} \
	-underline {0}

    $menu_name.file.import add command \
	-label {Source ...} \
	-command {load_source_cmd} \
	-underline {-1}

    $menu_name.file.import add command \
	-label {OE ...} \
	-command {load_oe_cmd} \
	-underline {-1}

    $menu_name.file.import add command \
	-label {System ...} \
	-command {load_system_cmd} \
	-underline {-1}

    # build cascade menu for importing/export shadow namelist files
    $menu_name.file add cascade \
	-menu "$menu_name.file.export" \
	-label {Export} \
	-underline {0}

    # build widget $menu_name.file.export
    menu $menu_name.file.export \
	-tearoff 0

    $menu_name.file.export add command \
	-label {Source ...} \
	-command {save_source_cmd} \
	-underline {-1}

    $menu_name.file.export add command \
	-label {OE ...} \
	-command {save_oe_cmd} \
	-underline {-1}

    $menu_name.file.export add command \
	-label {System ...} \
	-command {save_system_cmd} \
	-underline {-1}

    $menu_name.file add separator

    $menu_name.file add command \
	-command {exit_cmd} \
	-label {Exit} \
	-underline {1} \
	-accelerator {Ctrl+X}

    bind all <Control-x>	"exit_cmd"

    # build widget $menu_name.edit

    $menu_name add cascade -label Edit -menu $menu_name.edit -underline 0

    # build widget $menu_name.edit
    menu $menu_name.edit -tearoff 0

    $menu_name.edit add command \
	-label {Copy Selected} \
	-command {copy_selected_cmd} \
	-underline {-1} \
	-accelerator {Ctrl+C}

    bind all <Control-c>	"copy_selected_cmd"

    $menu_name.edit add command \
	-label {Delete Selected} \
	-command {delete_selected_cmd} \
	-underline {-1}

    $menu_name.edit add command \
	-label {Paste Selected} \
	-command {paste_selected_cmd} \
	-underline {-1} \
	-accelerator {Ctrl+V}

    bind all <Control-v>	"paste_selected_cmd"

    $menu_name.edit add separator

    # build cascade menu for Adding objects
    $menu_name.edit add cascade \
	-menu "$menu_name.edit.add" \
	-label {Add} \
	-underline {-1}

    # build widget $menu_name.edit.add
    menu $menu_name.edit.add -tearoff 0

    $menu_name.edit.add add command \
	-command {add_src_cmd} \
	-label {Source} \
	-underline {-1}

    $menu_name.edit.add add command \
	-command {add_oe_cmd 1} \
	-label {OE...} \
	-underline {-1}

    $menu_name.edit.add add command \
	-command {add_scr_cmd 1} \
	-label {Screen...} \
	-underline {-1}

    # build cascade menu for Deleting objects
    $menu_name.edit add cascade \
	-menu "$menu_name.edit.param" \
	-label {Parameters} \
	-underline {-1}

    # build widget $menu_name.edit.param
    menu $menu_name.edit.param -tearoff 0

    $menu_name.edit.param add command \
	-label {Selected} \
	-command {edit_selected_cmd} \
	-underline {-1}

    $menu_name.edit.param add command \
	-label {Source} \
	-command {edit_src_cmd} \
	-underline {-1}

    $menu_name.edit.param add command \
	-label {OE...} \
	-command {edit_oe_cmd} \
	-underline {-1}

    $menu_name.edit.param add command \
	-label {Screen...} \
	-command {edit_scr_cmd} \
	-underline {-1}

    # build cascade menu for Deleting objects
    $menu_name.edit add cascade \
	-menu "$menu_name.edit.delete" \
	-label {Delete} \
	-underline {-1}

    # build cascaded menu for $menu_name.delete
    menu $menu_name.edit.delete -tearoff 0

    $menu_name.edit.delete add command \
	-command {undo_deleted_cmd} \
	-label {Undo last} \
	-underline {-1}
    
    set gvars(undo_del_btn_config) "$menu_name.edit.delete entryconfigure 0"

    $menu_name.edit.delete add separator

    $menu_name.edit.delete add command \
	-command {delete_selected_cmd} \
	-label {Selected} \
	-underline {-1}

    $menu_name.edit.delete add command \
	-command {delete_src_cmd} \
	-label {Source} \
	-underline {-1}

    $menu_name.edit.delete add command \
	-command {delete_oe_cmd} \
	-label {OE...} \
	-underline {-1}

    $menu_name.edit.delete add command \
	-command {delete_all_oe_cmd} \
	-label {All OEs} \
	-underline {-1}

    $menu_name.edit.delete add command \
	-command {delete_scr_cmd} \
	-label {Screen...} \
	-underline {-1}

    # build widget $menu_name.run.m
    $menu_name add cascade -label Run -menu $menu_name.run -underline 0
    menu $menu_name.run -tearoff 0

    $menu_name.run add command \
	-label {Selected} \
	-command {run_selected_cmd} \
	-underline {-1}

    $menu_name.run add command \
	-label {Source} \
	-command {run_source_cmd} \
	-underline {-1}

    $menu_name.run add command \
	-label {System} \
	-command {run_system_cmd} \
	-underline {-1}

    $menu_name add cascade \
        -label Workspace \
	-menu $menu_name.workspace \
	-underline 3

    menu $menu_name.workspace -tearoff 0

    $menu_name.workspace add command \
	-label {Annotate} \
	-command {annotate_workspace_cmd} \
	-underline {0}

    $menu_name.workspace add command \
	-label {Show Annotation} \
	-command {show_annotation_cmd} \
	-underline {0}

    $menu_name.workspace add command \
	-label {Show Info ...} \
	-command {show_workspace_info_cmd} \
	-underline {5}

    # build widget $menu_name.tools.m
    $menu_name add cascade \
	-menu "$menu_name.tools" \
	-label {Tools} \
	-underline {0}

    menu $menu_name.tools \
	-tearoff 0

    $menu_name.tools add command \
	-label {Plot} \
	-command {plot_cmd} \
	-underline {0}

    $menu_name.tools add command \
	-label {Layout} \
	-command {layout_cmd} \
	-underline {0}

    $menu_name.tools add command \
	-label {GO Shell} \
	-command "run_script_cmd GO" \
	-underline {0}
    global tcl_platform
    if {![string compare $tcl_platform(platform) "windows"]} {
	$menu_name.tools add command \
	    -label {Unix Shell} \
	    -command {win32_run_new_unix_shell_cmd} \
	    -underline {5}
	$menu_name.tools add command \
	    -label {DOS Shell} \
	    -command {win32_run_new_dos_shell_cmd} \
	    -underline {0}
    } else {
	$menu_name.tools add command \
	    -label {Shell Window} \
	    -command {unix_run_new_shell_cmd} \
	    -underline {0}
    }

    $menu_name.tools add separator

    set tool_names [list \
	abrefc angle beam bragg col_2 comb_ray cutoff dose epath ergset \
	ffresnel filex fmax focnew gen_source genlib grade_mlayer \
	grating_set h2kol histo1 histo2 histo3 intens jntpscalc loss \
	machine_par make_id menu minmax mirinfo mlayer nphoton patch \
	plotfoc plotspace plotxy power pre_mlayer preplot preplot2 \
	prerefl presurface primvs pspread pwr_dens qeff readcdf recolor \
	reflag retrace seed sincrad slits sourcinfo srcdf srcomp srfull \
	srfunc surplot sync_spectra sysinfo sysplot torus translate \
	transmit undul undul_cdf undul_phot undul_set user_undul variance \
	wiggler_spectrum \
    ]

    set preproc_names [list \
	bragg ergset grade_mlayer histo3 jntpscalc patch pre_mlayer  \
	prerefl presurface qeff readcdf reflag \
    ]

    set postproc_names [list \
	h2kol histo1 plotxy plotspace preplot2 preplot angle comb_ray \
	ffresnel focnew histo2 intens loss minmax mirinfo plotfoc pspread \
	pwr_dens recolor retrace seed slits sourcinfo surplot sysinfo \
	sysplot tdconn translate \
    ]

    set optics_names [list \
	abrefc cutoff dose grating_set mlayer torus transmit \
    ]

    set general_names [list \
	col_2 filex fmax variance \
    ]

    set misc_names [list \
	make_id undul \
    ]

    # build cascade menu for Preferences
    $menu_name.tools add cascade \
	-menu "$menu_name.tools.preprocs" \
	-label {Pre-processor Tools} \
	-underline {-1}

    # build cascaded menu for $menu_name.tools.preprocs
    menu $menu_name.tools.preprocs \
	-tearoff 0

    foreach cmd $preproc_names {
	$menu_name.tools.preprocs add command \
	    -label [list $cmd] \
	    -command "run_shell_cmd $cmd" \
	    -underline {-1}
    }

    # build cascade menu for Preferences
    $menu_name.tools add cascade \
	-menu "$menu_name.tools.postprocs" \
	-label {Post-processor Tools} \
	-underline {-1}

    # build cascaded menu for $menu_name.tools.postprocs
    menu $menu_name.tools.postprocs \
	-tearoff 0

    foreach cmd $postproc_names {
	$menu_name.tools.postprocs add command \
	    -label [list $cmd] \
	    -command "run_shell_cmd $cmd" \
	    -underline {-1}
    }

    # build cascade menu for Optics
    $menu_name.tools add cascade \
	-menu "$menu_name.tools.optics" \
	-label {Optics Tools} \
	-underline {-1}

    # build cascaded menu for $menu_name.tools.optics
    menu $menu_name.tools.optics \
	-tearoff 0

    foreach cmd $optics_names {
	$menu_name.tools.optics add command \
	    -label [list $cmd] \
	    -command "run_shell_cmd $cmd" \
	    -underline {-1}
    }

    # build cascade menu for General
    $menu_name.tools add cascade \
	-menu "$menu_name.tools.general" \
	-label {General} \
	-underline {-1}

    # build cascaded menu for $menu_name.tools.general
    menu $menu_name.tools.general \
	-tearoff 0

    foreach cmd $general_names {
	$menu_name.tools.general add command \
	    -label [list $cmd] \
	    -command "run_shell_cmd $cmd" \
	    -underline {-1}
    }

    # build cascade menu for Misc
    $menu_name.tools add cascade \
	-menu "$menu_name.tools.misc" \
	-label {Misc.} \
	-underline {-1}

    # build cascaded menu for $menu_name.tools.misc
    menu $menu_name.tools.misc \
	-tearoff 0

    foreach cmd $misc_names {
	$menu_name.tools.misc add command \
	    -label [list $cmd] \
	    -command "run_script_cmd $cmd" \
	    -underline {-1}
    }

    # build widget $menu_name.options
    $menu_name add cascade \
	-menu "$menu_name.options" \
	-label {Options} \
	-underline {0}

    # build widget $menu_name.options
    menu $menu_name.options \
	-tearoff 0

    $menu_name.options add checkbutton \
	-label {Verbose Mode (Debugging)} \
	-variable gvars(verbose) \
	-underline {-1}

    $menu_name.options add command \
	-label {Change directory} \
	-command {chdir_cmd} \
	-underline {-1}

    $menu_name.options add command \
	-label {Redraw Parameter Editor} \
	-command {redraw_current_page_cmd} \
	-underline {-1}

    $menu_name.options add command \
	-label {Redraw Beamline Editor} \
	-command {redraw_editbar_cmd} \
	-underline {-1}

    $menu_name.options add command \
	-label {Show Run Window} \
	-command {show_run_window_cmd} \
	-underline {-1}

    $menu_name.options add command \
	-label {Clear clipboard} \
	-command {clear_clipboard_cmd} \
	-underline {-1}

    $menu_name.options add command \
	-label {Clear Param Editor} \
	-command {edit_nothing} \
	-underline {-1}

    $menu_name.options add separator

    # Add options to add registry entries and add start menu items on Windows`.
    if {$tcl_platform(platform) == "windows"} {
      $menu_name.options add command \
	  -label {Repair Windows Registry items} \
	  -command {repair_win_registry_cmd} \
	  -underline {-1}

      $menu_name.options add command \
	  -label {Repair Windows Start Menu items} \
	  -command {repair_win_start_menu_cmd} \
	  -underline {-1}

      $menu_name.options add separator
    }

    $menu_name.options add command \
	-label {Remember Current State} \
	-command {save_beamline_state_cmd} \
	-underline {-1}

    $menu_name.options add command \
	-label {Restore Remembered State} \
	-command {restore_beamline_state_cmd} \
	-underline {-1}

    $menu_name.options add separator

    #
    # CHECK/FIXME: The autosave* options are preferences even if these 
    # are not in right place (ie., not under the preferences cascade). 
    # I'd rather have these prominently displayed instead of hiding
    # these under yet another level. What's the right thing to do?
    #

    $menu_name.options add checkbutton \
	-label {Autosave mode} \
	-variable gprefs(autosave_mode_active) \
	-command "autosave_mode_cmd $menu_name.options.autosave {0 1}" \
	-underline {-1}

    # build widget $menu_name.options.autosave
    $menu_name.options add cascade \
	-menu "$menu_name.options.autosave" \
	-label {Autosave Options} \
	-underline {0}

    # build widget $menu_name.options.autosave
    menu $menu_name.options.autosave \
	-tearoff 0

    $menu_name.options.autosave add checkbutton \
	-label {Time Trigger} \
	-variable gprefs(autosave_timer_trigger) \
	-command {autosave_mode_option_cmd timer} \
	-underline {-1}

    $menu_name.options.autosave add checkbutton \
	-label {Extra Safety Trigger} \
	-variable gprefs(autosave_explicit_trigger) \
	-command {autosave_mode_option_cmd explicit} \
	-underline {-1}

    $menu_name.options add separator

    # build cascade menu for Preferences
    $menu_name.options add cascade \
	-menu "$menu_name.options.preferences" \
	-label {Preferences} \
	-underline {-1}

    # build cascaded menu for $menu_name.options.preferences
    menu $menu_name.options.preferences \
	-tearoff 0

    $menu_name.options.preferences add checkbutton \
	-label {Add SRC at Startup} \
	-variable gprefs(add_src_at_startup) \
	-underline {-1}

    $menu_name.options.preferences add checkbutton \
	-label {Edit SRC at Startup} \
	-variable gprefs(edit_src_at_startup) \
	-underline {-1}

    $menu_name.options.preferences add checkbutton \
	-label {Edit after adding} \
	-variable gprefs(edit_added_obj) \
	-underline {-1}

    $menu_name.options.preferences add checkbutton \
	-label {Change to workspace directory after loading} \
	-variable gprefs(cd_to_workspace_dir) \
	-underline {-1}

    $menu_name.options.preferences add checkbutton \
	-label {Change to working directory from last time} \
	-variable gprefs(cd_to_last_working_dir) \
	-underline {-1}

    $menu_name.options.preferences add checkbutton \
	-label {Restore Window size from last time} \
	-variable gprefs(restore_last_geom) \
	-underline {-1}

    $menu_name.options.preferences add checkbutton \
	-label {Balloon Help} \
	-variable gprefs(balloon_help) \
	-command "balloon_help_cmd" \
	-underline {-1}

    $menu_name.options.preferences add checkbutton \
	-label {Show Toolbar} \
	-variable gprefs(show_toolbar) \
	-command {show_toolbar_cmd} \
	-underline {-1}

    $menu_name.options.preferences add checkbutton \
	-label {Message window at Bottom} \
	-variable gprefs(msg_window_at_bottom) \
	-command {show_msg_window_cmd} \
	-underline {-1}

    $menu_name.options.preferences add checkbutton \
	-label {Show full path in Workspace history menu} \
	-variable gprefs(show_path_in_workspace_history) \
	-command {update_workspace_history} \
	-underline {-1}

    $menu_name.options.preferences add checkbutton \
	-label {Show Command Window (Debugging)} \
	-variable gprefs(show_cmd_window) \
	-command {show_cmd_window_cmd} \
	-underline {-1}

    $menu_name.options add command \
	-label {Save preferences} \
	-command {save_prefs_cmd} \
	-underline {0}

    $menu_name.options add command \
	-label {Reload preferences} \
	-command {load_prefs_cmd} \
	-underline {0}

    $menu_name.options add command \
	-label {Reset to defaults} \
	-command {reset_prefs_cmd} \
	-underline {2}

    # build widget $menu_name.help
    $menu_name add cascade \
	-menu "$menu_name.help" \
	-label {Help} \
	-underline {0}

    # build widget $menu_name.help.m
    menu $menu_name.help \
	-tearoff 0

    $menu_name.help add command \
	-command {about_cmd} \
	-label {About} \
	-underline {0}

    $menu_name.help add command \
	-command {about_author_cmd} \
	-label {Author} \
	-underline {1}

    $menu_name.help add command \
	-command {help_on_topmenu_cmd} \
	-label {On Menubar} \
	-underline {0}

    $menu_name.help add separator

    # build cascade menu for Preferences
    $menu_name.help add cascade \
	-label {Demos} \
	-menu $menu_name.help.demo \
	-underline {-1}

    # build cascaded menu for $menu_name.help.demo
    menu $menu_name.help.demo \
	-tearoff 0

    $menu_name.help.demo add command \
	-command {run_demo_cmd toroid.ws} \
	-label {Toroidal Mirror} \
	-underline {7}

    catch {tix resetoptions TixGray 14Point}
    return $menu_name
}

#
# FIXME/TODO: The following doesn't work for cascaded menu items, so I had 
# to add "catch". Why?
#
proc GUI:set_undo_del_btn_state {state} {
    global gvars
    catch {eval $gvars(undo_del_btn_config) -state $state}
}

#----------------------------------------------------------------------#
# Make the Tcl command window for direct manipulation
#----------------------------------------------------------------------#

proc GUI:make_cmd_window {vname} {
    global gvars
    frame $vname -class XMenuCommand -bd 2 
    label $vname.label -text "Ready:: "
    entry $vname.entry -relief sunken
    bind $vname.entry <Return> {exec_tcl_command %W}
    bind $vname.entry <Any-question> "command_help %W $gvars(cmdhelpfile)"
    pack $vname.label -side left -anchor w -expand 0 
    pack $vname.entry -side left -anchor w -expand 1 -fill x
    return $vname
}

#----------------------------------------------------------------------#
# Make the menu page viewer/editor and return the frame
#----------------------------------------------------------------------#

proc GUI:make_page_viewer {vname} {
    global gvars gprefs
    tixScrolledWindow $vname \
        -width $gvars(page_w_wt) -height $gvars(page_w_ht) \
	-relief sunken -bd 2 \
	-expand 1 ;#-scrollbar both
    bind $vname <Enter> {
        set_msg "Parameter Editor: Enter `?' in any field for help"
    }
    bind $vname <Leave> {reset_msg}
    return $vname
}

#----------------------------------------------------------------------#
# Make the startup window (cutesy thing)
#----------------------------------------------------------------------#

set gvars(startup_w) ".startup"

proc GUI:set_startup_msg {{msg ""}} {
    global gvars
    set msg_w $gvars(startup_w).msg
    $msg_w configure -text $msg
    update idletasks
}

proc GUI:make_startup {} {
    global gvars

    set top [toplevel $gvars(startup_w) -class Startup]
    wm transient $top .
    #wm overrideredirect $top 1
    wm withdraw $top

    # load the SHADOW banner if needed.
    if {$gvars(shadow_image) == ""} {
	set image [image create photo]
	global tcl_version
	set imagefile [file join $gvars(shadow_tcl_library) \
	    bitmaps shadow.gif]
	if [catch {$image read $imagefile} msg] {
	    puts stderr "Warning: Cannot find SHADOW image ($msg)"
	} else {
	    set gvars(shadow_image) $image
	}
    }

    set blurb [label $top.blurb \
        -text "Please wait while Loading SHADOW ..."]
    set image [label ${top}.img -anchor center -image $gvars(shadow_image)]
    set msg [label $top.msg -text ""]

    grid $blurb -row 0 -column 0 -sticky news -ipadx 5 -ipady 5
    grid $image -row 1 -column 0 -sticky news -ipadx 20 -ipady 20
    grid $msg -row 2 -column 0 -sticky news -ipadx 5 -ipady 5
    grid columnconfigure $top 0 -weight 1

    update idletasks
    set x [expr [winfo screenwidth $top]/2 - [winfo reqwidth $top]/2 \
            - [winfo vrootx [winfo parent $top]]]
    set y [expr [winfo screenheight $top]/2 - [winfo reqheight $top]/2 \
            - [winfo vrooty [winfo parent $top]]]
    wm geom $top [winfo reqwidth $top]x[winfo reqheight $top]+$x+$y
    # win32 needs the following update to center correctly.
    update idletasks

    wm deiconify $top
    wm title $top "SHADOW GUI $gvars(shadow_gui_version)"
    update

    return $top
}

#----------------------------------------------------------------------#
# Make the menu page edit tool bar
#----------------------------------------------------------------------#

proc GUI:make_editbar {vname} {
    global gvars
    set initial_toolbox {
	{System header disabled {beamline}}
	{Tools header disabled {tools}}
    }

    frame $vname -class XMenuEditBar
    set editbar [ \
        editbar:make $vname $initial_toolbox editbar_callback \
	editbar_select_callback]
    pack $editbar -side top -fill both -expand yes
    #
    # do not use balloons yet, since there is no way of turning off
    # the balloon and keep only the status message. Use <Enter> and
    # <Leave> instead.
    #
    #	$gvars(balloon) bind $editbar \
    #	    -statusmsg "Editor:: Click on SHADOW element to edit parameters"
    #	    -balloonmsg "Select element to edit" 
    #
    bind $editbar <Enter> {
        set_msg "System: Single click selects, Double Click edit parameters"
    }
    bind $editbar <Leave> {reset_msg}
    return $vname
}

#
# make the popup menu for BEAMLINE stuff. Binding is done outside on the
# client side of course.
#
proc GUI:make_beamline_popup {vname} {
    global gvars
    #
    # make the popup menu. Binding is done by the caller.
    #

    catch {tix resetoptions TK TK}
    set menu [menu $vname -tearoff 0]

    $menu add command -label "Edit Selected" -command edit_selected_cmd
    $menu add command -label "Copy Selected" -command copy_selected_cmd
    $menu add command -label "Cut Selected" -command delete_selected_cmd
    $menu add command -label "Paste Selected" -command paste_selected_cmd
    $menu add separator
    $menu add command -label "Run Selected" -command run_selected_cmd
    $menu add command -label "Annotate Selected" -command rename_selected_cmd \
	-state disabled
    $menu add separator
    $menu add command -label "Load Workspace ..." -command load_state_cmd
    $menu add command -label "Save Workspace ..." -command save_state_cmd
    $menu add separator
    $menu add command -label Help \
        -command {set_msg "No HELP available yet."} \
	-state disabled

    catch {tix resetoptions TixGray 14Point}
    return $vname
}

#----------------------------------------------------------------------#
# Make the message command 
#----------------------------------------------------------------------#

proc GUI:make_msg_window {vname} {
    global gvars
    frame $vname -class XMenuMessage -bd 2
    frame $vname.top

    label $vname.top.label -text "Info:: "
    label $vname.top.msg -width 55 -relief sunken -bd 2 
    set gvars(statusbar) $vname.top.msg
    pack $vname.top.label -side left -anchor w -expand 0 
    pack $vname.top.msg -side left -anchor w -expand 0 -fill x

    label $vname.top.label2 -text "Selection:: "
    label $vname.top.msg2 -width 25 -relief sunken -bd 2 -text "None"
    set gvars(cur_selectbar) $vname.top.msg2
    pack $vname.top.label2 -side left -anchor w -expand 0 
    pack $vname.top.msg2 -side left -anchor w -expand 0 -fill x

    label $vname.top.label3 -text "Edit:: "
    label $vname.top.msg3 -width 25 -relief sunken -bd 3 -text "None"
    set gvars(cur_editbar) $vname.top.msg3
    pack $vname.top.label3 -side left -anchor w -expand 0 
    pack $vname.top.msg3 -side left -anchor w -expand 0 -fill x

    pack $vname.top -side top -expand 1 -fill x

    frame $vname.bottom 

    label $vname.bottom.label1 -text "Current directory:: "
    label $vname.bottom.msg1 -width 55 -relief sunken -bd 2 
    set gvars(cur_directory) $vname.bottom.msg1
    pack $vname.bottom.label1 -side left -anchor w -expand 0 
    pack $vname.bottom.msg1 -side left -anchor w -expand 0 -fill x

    button $vname.bottom.btn1 -text "ChDir" -command chdir_cmd
    pack $vname.bottom.btn1 -side left -anchor w 

    label $vname.bottom.label2 -text "Clipboard:: "
    label $vname.bottom.msg2 -width 20 -relief sunken -bd 2 
    set gvars(clipboard_w) $vname.bottom.msg2
    pack $vname.bottom.label2 -side left -anchor w -expand 0 
    pack $vname.bottom.msg2 -side left -anchor w -expand 0 -fill x

    pack $vname.bottom -side bottom -expand 1 -fill x

    return $vname
}

#----------------------------------------------------------------------#
# Make the toolbox
#----------------------------------------------------------------------#

# FIXME/CHECK: I believe the following is unused and should be removed.
# Thu Feb 25 16:07:19 CST 1999 -- MK

proc XXXXX_toolbox_callback {args} {
    global gvars
    vputs "toolbox_callback_args: $args"
    set tool_type [lindex $args 0]
    switch -exact $tool_type {
        beamline {
	    if {[llength $args] == 1} {
		show_about_page
	    } else {
	        add_tool [lindex $args 1]
	    }
	}
        inspector {
	    if {[llength $args] == 1} {
		show_about_page
	    } else {
	        add_inspector [lindex $args 1] [lrange $args 2 end]
	    }
	}
	ray_trace {
	    set trace_what [lindex $args 1]
	    switch -exact -- $trace_what {
	        source {
		    run_source_cmd
		}
		system {
		    run_system_cmd
		}
		default {
		    set_msg "Running $trace_what not implemented yet"
		}
	    }
	}
	default {
	    set_msg "ITEM $tool_type not implemented yet"
	}
    }
}

#
# GUI:make_toolbox: Make the SHADOW toolbox, which contains all the models,
# inspection tools and other utilities.
#
proc GUI:make_toolbox {vname} {
    global gvars
    set toolbox_tools {
	{System folder header enabled {beamline}}
	{System.Source file entry enabled {beamline source}}
	{System.OE file entry enabled {beamline oe}}
	{System.OE.Screen file entry enabled {beamline screen}}
	{Ray_Trace folder header disabled {ray_trace}}
	{Ray_Trace.Source file entry enabled {ray_trace source}}
	{Ray_Trace.System file entry enabled {ray_trace system}}
	{Inspector folder header disabled {inspector}}
	{Inspector.General folder header disabled {inspector general}}
	{Inspector.General.Plot file entry enabled {inspector plotxy}}
	{Inspector.General.MinMax file entry enabled {inspector minmax}}
	{Inspector.Source folder header disabled {inspector source}}
	{Inspector.Source.SRCInfo file entry enabled {inspector srcinfo}}
	{Inspector.OE folder header disabled {inspector oe}}
	{Inspector.OE.MirInfo file entry enabled {inspector mirinfo}}
	{Preprocessor folder header disabled {tool}}
	{Preprocessor.Prerefl file entry enabled {tool prerefl}}
    }

    frame $vname -class XMenuToolBox ;#-relief sunken -bd 2
    set toolbox [toolbox:make $vname $toolbox_tools toolbox_callback]
    pack $toolbox -side top -fill both -expand yes
    #
    # do not use balloons yet, since there is no way of turning off
    # the balloon and keep only the status message. Use <Enter> and
    # <Leave> instead.
    #
    #$gvars(balloon) bind $toolbox \
    #	-statusmsg "Toolbox: Click on element to add to beamline" \
    #	-balloonmsg "Select element to add"
    bind $toolbox <Enter> {
        set_msg "Toolbox: Click on element to add to System"
    }
    bind $toolbox <Leave> {
        reset_msg
    }
    return $vname
}

#----------------------------------------------------------------------#
# misc utils
#----------------------------------------------------------------------#

proc GUI:add_file_to_title {file} {
    global gvars
    if {$file == ""} {
	set filepart "\[No filename\]"
    } else {
	set filepart "\[$file\]"
    }
    wm title . "$gvars(win_title)   $filepart"
}

#----------------------------------------------------------------------#
# Make the top toolbar 
#----------------------------------------------------------------------#


proc GUI:make_toolbar {vname} {
    global gvars
    set toolbar {
	{"Exit" "" enabled exit_cmd {
	    "Exit application" "Click to Exit SHADOW"}}
	{Load openfile enabled {load_state_cmd} {
	    "Load Workspace" "Load the entire GUI state from file"}}
	{Save openfile enabled {save_state_cmd} {
	    "Save Workspace" "Save the entire GUI state to file"}}
	{"Edit" "" enabled {edit_selected_cmd} {
	    "Edit selected tool" "Edit currently selected tool in System"}}
	{"Cut" scissors enabled {delete_selected_cmd} {
	    "Cut selected tool" "Cut currently selected tool in System"}}
	{"Source" "" enabled {run_source_cmd} {
	    "Generate Source" "Generate the Source"}}
	{"Trace" "" enabled {run_system_cmd} {
	    "Trace system" "Trace the system"}}
	{"Run" "" enabled {run_selected_cmd} {
	    "Run selected" "Run the selected item"}}
	{"Plot" "" enabled {plot_cmd} {
	    "Plot" "Plot SHADOW data"}}
	{"Layout" "" enabled {layout_cmd} {
	    "Layout" "Layout SHADOW Optical System"}}
    }

    set toolbar [toolbar:make $vname $toolbar $gvars(balloon) 0]
    return $vname
}


#
# initialize the  UI
#
proc GUI:make {} {
    global gvars gprefs env

    global tcl_version
    tix addbitmap [file join $gvars(shadow_tcl_library) bitmaps]

    #
    # create the TixBalloon widget first.
    #
    set gvars(balloon) [tixBalloon .balloon]

    global tk_version

    frame .top -bg lightgrey -relief sunken
    if {$tk_version < 8.0} {
	set topmenu [GUI:make_topmenu_pre_v8 .top.topmenu]
    } else {
	set topmenu [GUI:make_topmenu .top.topmenu]
    }
    set gvars(toolbar_w_stub) .top.toolbar_w_stub
    frame $gvars(toolbar_w_stub)
    set gvars(toolbar) [GUI:make_toolbar .top.toolbar]
    set gvars(cmd_w_stub) .top.tcl_cmd_w_stub
    frame $gvars(cmd_w_stub)
    set gvars(msg_w_top_stub) .top.msg_w_top_stub
    frame $gvars(msg_w_top_stub)
    set gvars(cmd_w) [GUI:make_cmd_window .top.tcl_cmd_w]
    frame .top.body
    set gvars(page_w) [GUI:make_page_viewer .top.body.page]
    set gvars(msg_w_bot_stub) .top.msg_w_bot_stub
    frame $gvars(msg_w_bot_stub)
    set gvars(msg_w) [GUI:make_msg_window .top.msg]

    frame .top.body.toolbox -relief sunken -bd 2
    set gvars(tools_w) [GUI:make_toolbox .top.body.toolbox.tools]
    set downframe [frame .top.body.toolbox.downframe]
    set downimage [label $downframe.img -image [tix getimage down2]]
    pack $downimage -expand 0
    set gvars(editbar) [GUI:make_editbar .top.body.toolbox.editbar]


    # Window manager configurations
    wm positionfrom . user
    wm sizefrom . ""
    # uncomment the following line to make the interface resizable
    # (at your own risk) by the user via the window manager.
    #
    # wm minsize . $gvars(page_w_wt) $gvars(page_w_ht)
    # wm maxsize . [winfo screenwidth .] [winfo screenheight .]
    #
    wm title . $gvars(win_title)
    wm protocol . WM_DELETE_WINDOW exit_cmd
    wm protocol . WM_SAVE_YOURSELF exit_cmd

    $gvars(balloon) configure -statusbar $gvars(statusbar)

    # now pack the whole UI
    if {$tk_version < 8.0} {
	pack $topmenu -side top -expand 1 -fill x
    } else {
        . configure -menu $topmenu
    }
    pack $gvars(toolbar_w_stub) -side top -expand 1 -fill x
    pack $gvars(cmd_w_stub) -side top -expand 1 -fill x
    pack $gvars(msg_w_top_stub) -side top -expand 1 -fill x
    pack .top.body -side top -expand 1 -fill both -anchor center 
    pack $gvars(tools_w) -side top -expand 1 -fill both -padx 0 -pady 0 \
        -anchor center 
    pack $downframe -side top -expand 0 -fill y -padx 0 -pady 0 -anchor center
    pack $gvars(editbar) -side bottom -expand 1 -fill both -padx 0 -pady 0 \
        -anchor center 
    pack .top.body.toolbox -side left -expand 1 -fill both -anchor center \
	-padx 4 -pady 4 -ipadx 2 -ipady 2
    pack $gvars(page_w) -side left -expand 1 -fill both -anchor center \
	-padx 4 -pady 4 -ipadx 2 -ipady 2
    pack $gvars(msg_w_bot_stub) -side top -expand 1 -fill x

    # make the various popup menus for various areas in the GUI.

    set gvars(beamline_popup) [GUI:make_beamline_popup .top.bl_popup]

    #
    # make sure that the binding does NOT apply to other "foreign"
    # toplevels such as SHADOW Plotting utilities and such.
    #
    bind all <3> {+
	if {[winfo toplevel %W] == "."} {
	    tk_popup $gvars(beamline_popup) %X %Y
	}
    }

    #
    # HACK. This should be done better. Also, the following should be
    # set gvars(page_w) [$gvars(page_w) subwidget window].w
    # (Note the trailing .w). This for some reasons works around the
    # weird flicker problem when new pages would be up.
    #
    set gvars(page_w) [$gvars(page_w) subwidget window]
    #
    # END HACK
    #

    #
    # done building. Now pack the sucker.
    #

    pack .top -expand 1 -fill both

    # now pack the configurable windows
    show_toolbar_cmd
    show_cmd_window_cmd
    show_msg_window_cmd

    disable_undo_delete_cmd

    # focus default $topmenu
    focus $topmenu
}

