#
# run-shadow.tcl: Run various shadow programs
#
# ------------------------------------------------
# Mumit Khan <khan@xraylith.wisc.edu>
# Center for X-ray Lithography
# University of Wisconsin-Madison
# 3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Copyright (c) 1994-1996 Mumit Khan
#
#

set RunTool_privs(delay)		idle
set RunTool_privs(raise_delay)		250
set RunTool_privs(tmpfiles)		{}

set RunTool_privs(w)			.tool_runner
set RunTool_privs(w:text)		""
set RunTool_privs(w:cancel_btn)		""
set RunTool_privs(pipe)			""
set RunTool_privs(prog_running)		0
set RunTool_privs(use_blocking_mode)	1

set RunTool_privs(old_focus)		""
set RunTool_privs(old_grab)		""
set RunTool_privs(grab_status)		""


#----------------------------------------------------------------------#
# Running programs
#----------------------------------------------------------------------#

proc RunTool:_make_run_window {} {
  global RunTool_privs tcl_platform

  set w $RunTool_privs(w)

  set RunTool_privs(tmpfiles) {}

  toplevel $w
  wm title $w "SHADOW Run Tool"
  wm iconname $w "Run SHADOW"
  # The following command means that the dialog won't be posted if
  # [winfo parent $w] is iconified, but it's really needed;  otherwise
  # the dialog can become obscured by other windows in the application,
  # even though its grab keeps the rest of the application from being used.

  wm transient $w [winfo toplevel [winfo parent $w]]
  if {$tcl_platform(platform) == "macintosh"} {
    unsupported1 style $w dBoxProc
  }

  set f [frame $w.top -relief raised -bd 1]
  set f1 [frame $f.f1]
  set t  [text $f1.t -font {Helvetica 12}]

  set v  [scrollbar $f1.v -orient vertical]
  pack $v -side right -fill y
  pack $t -side top -expand yes -fill both

  set f2 [frame $f.f2]
  set h  [scrollbar $f2.h -orient horizontal]
  set x  [frame $f2.x -width [winfo reqwidth $v]]
  pack $x -side right -fill y
  pack $h -expand yes -fill both

  pack $f2 -side bottom -fill x
  pack $f1 -expand yes -fill both

  $t config -xscrollcommand "$h set" -yscrollcommand "$v set"
  $h config -command "$t xview"
  $v config -command "$t yview"

  set bot [frame $w.bot -relief raised -bd 1]
  set b [button $bot.b -text Stop -command RunTool:cancel_run_window -width 7]
  set b2 [button $bot.b2 -text Save -command RunTool:save_run_window -width 7]
  set b3 [button $bot.b3 -text Clear -command RunTool:clear_run_window -width 7]
  set b4 [button $bot.b4 -text "Insert File" \
          -command RunTool:insert_into_run_window -width 7]
  pack $b -side left -expand yes -padx 10 -pady 15
  pack $b2 -side left -expand yes -padx 10 -pady 15
  pack $b3 -side left -expand yes -padx 10 -pady 15
  pack $b4 -side left -expand yes -padx 10 -pady 15
  pack $bot -side bottom -fill both
  pack $f -expand yes -fill both

  #
  # do the text fonts and such things.
  #
  #
  $t tag configure bold \
    -font {Courier 12 bold}
  $t tag configure big \
    -font {Courier 14 bold}
  $t tag configure verybig \
    -font {Courier 18 bold}

  if {[winfo depth $w] > 1} {
    $t tag configure color1 -background #a0b7ce
    $t tag configure color2 -foreground red
    $t tag configure raised -relief raised -borderwidth 1
    $t tag configure sunken -relief sunken -borderwidth 1
  } else {
    $t tag configure color1 -background black -foreground white
    $t tag configure color2 -background black -foreground white
    $t tag configure raised -background white -relief raised \
      -borderwidth 1
    $t tag configure sunken -background white -relief sunken \
      -borderwidth 1
  }

  set RunTool_privs(w:text) $t
  set RunTool_privs(w:cancel_btn) $b
  set RunTool_privs(w) $w

  wm withdraw $w
  update idletasks
  set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	  - [winfo vrootx [winfo parent $w]]]
  set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	  - [winfo vrooty [winfo parent $w]]]
  wm geom $w +$x+$y
  wm deiconify $w
  update idletasks
  return $w
}

proc show_run_window {} {
  global RunTool_privs tcl_platform

  set w $RunTool_privs(w)

  set RunTool_privs(old_focus) ""
  set RunTool_privs(old_grab) ""
  set RunTool_privs(grab_status) ""

  if {![winfo exists $w]} {
    RunTool:_make_run_window
  }
  $RunTool_privs(w:cancel_btn) config -text Close
  wm deiconify $w
  update idletasks
}

proc RunTool:make_run_window {} {
  global RunTool_privs tcl_platform

  set w $RunTool_privs(w)

  if {![winfo exists $w]} {
    RunTool:_make_run_window
  } else {
    $RunTool_privs(w:cancel_btn) config -text Stop
  }
  wm deiconify $w

  # FIXME/BUG: Don't use grab, busy etc on Windows. Too flaky. Leads to 
  # crashes.
  if {![string compare $tcl_platform(platform) "unix"]} {
    set RunTool_privs(old_focus) [focus]
    set RunTool_privs(old_grab) [grab current $w]
    if {$RunTool_privs(old_grab) != ""} {
      set RunTool_privs(grab_status) [grab status $RunTool_privs(old_grab)]
    }
    grab $w
    focus $w
    blt::busy hold $RunTool_privs(w:text)
    blt::busy hold .
  }
}

proc RunTool:save_run_window {{file {}}} {
  global RunTool_privs tcl_platform

  if {$file != ""} {
    if {[catch {open $file w} f] != 0} {
      tk_messageBox -title "Save Error" -type ok -icon error -message \
	"Cannot open output file \"$file\" to save"
      set_msg "cancelled"
      return 1
    }
    puts $f [$RunTool_privs(w:text) get 1.0 end]
    close $f
    return 0
  }
  #
  # We only get there when called w/out an argument, eg., by user.
  #

  global tcl_platform
  set types {
    {{Text files}	{.txt}}
    {{Misc Data Files}	{.dat}}
    {{All Files}	{*}}
  }
  set file [tk_getSaveFile \
    -filetypes $types \
    -defaultextension .txt \
    -initialfile report.txt \
    -title "Save the window contents to text file"
  ]
  if {$file != ""} {
    RunTool:save_run_window $file
  }
  return 0
}

proc RunTool:insert_into_run_window {{file {}}} {
  global RunTool_privs tcl_platform

  if {$file != ""} {
    if {[catch {open $file r} f] != 0} {
      tk_messageBox -title "Error" -type ok -icon error -message \
	"Cannot open input file \"$file\" to read from"
      set_msg "cancelled"
      return 1
    }
    set data [read $f]
    $RunTool_privs(w:text) insert end "Inserting file $file\n\n" {big color2}
    $RunTool_privs(w:text) insert end "$data"
    $RunTool_privs(w:text) insert end "  ----- Done -------\n" {verybig color2}
    $RunTool_privs(w:text) see end
    close $f
    return 0
  }
  #
  # We only get there when called w/out an argument, eg., by user.
  #

  global tcl_platform
  set types {
    {{Text files}	{.txt}}
    {{Misc Data Files}	{.dat}}
    {{All Files}	{*}}
  }
  set file [tk_getOpenFile \
    -filetypes $types \
    -title "Enter filename to insert into this window"
  ]
  if {$file != ""} {
    RunTool:insert_into_run_window $file
  }
  return 0
}

proc RunTool:insert_text_into_run_window {text} {
  global RunTool_privs tcl_platform

  $RunTool_privs(w:text) insert end "Inserting file $file\n\n" {big color2}
  $RunTool_privs(w:text) insert end "$text"
  $RunTool_privs(w:text) insert end "  ----- Done -------\n" {verybig color2}
  $RunTool_privs(w:text) see end
  return 0
}

proc RunTool:clear_run_window {} {
  global RunTool_privs tcl_platform
  $RunTool_privs(w:text) delete 1.0 end
}

proc RunTool:cancel_run_window {} {
  global RunTool_privs tcl_platform
  if {[$RunTool_privs(w:cancel_btn) cget -text] == "Close"} {
    wm withdraw $RunTool_privs(w)
    #
    # also time to delete temp files.
    #
    foreach file $RunTool_privs(tmpfiles) {
      catch {file delete $file}
    }
  } else {
    $RunTool_privs(w:cancel_btn) config -text Close -width 7
    catch {
      close $RunTool_privs(pipe)
    }
    set RunTool_privs(prog_running) 0
    $RunTool_privs(w:text) insert end " \n"
    $RunTool_privs(w:text) insert end \
      "  ----- Cancelled by User -------\n\n" {verybig color2}
    set fmt "%a %b %d %H:%M:%S %Y"
    $RunTool_privs(w:text) insert end \
      "End Time: [clock format [clock seconds] -format $fmt]\n\n" {big color2}
    $RunTool_privs(w:text) see end

    # FIXME/BUG: Don't use grab, busy etc on Windows. Too flaky. Leads to 
    # crashes.
    if {![string compare $tcl_platform(platform) "unix"]} {
      blt::busy release .
      blt::busy release $RunTool_privs(w:text)
      grab release $RunTool_privs(w)
      catch {focus $RunTool_privs(old_focus)}
      if {$RunTool_privs(old_grab) != ""} {
	if {$RunTool_privs(grab_status) == "global"} {
	  grab -global $RunTool_privs(old_grab)
	} else {
	  grab $RunTool_privs(old_grab)
	}
      }
    }
  }
}

proc RunTool:get_run_output {{hook {}}} {
  global RunTool_privs tcl_platform

  #
  # FIXME/CHECK: There are two ways to do this: 
  #  blocking     : use blocking mode, line buffering, and use gets to read 
  #                 line by line.
  #  non-blocking : use blocking mode, no buffering and use read to read all
  #                 all available data and output without a newline.
  #  Which one is better?
  #
  #  Set the RunTool_privs(use_blocking_mode) to 0 to use the 2nd 
  #  method.
  #
  #  See RunTool:run_program for current set of channel options using
  #  fconfigure.
  #
  if {![eof $RunTool_privs(pipe)]} {
    if {$RunTool_privs(use_blocking_mode)} {
      gets $RunTool_privs(pipe) line
      $RunTool_privs(w:text) insert end "$line\n"
    } else {
      set line [read $RunTool_privs(pipe)]
      $RunTool_privs(w:text) insert end "$line"
    }
  } else {
    catch {
      close $RunTool_privs(pipe)
    }
    #
    # if there is a hook, run it now, but only if the user hasn't
    # killed the job using the Cancel button.
    # TODO/FIXME/CHECK:
    #
    if {[$RunTool_privs(w:cancel_btn) cget -text] == "Close"} {
      vputs "Cancelled = 1"
    } else {
      set cancelled 0
    }
    if {$hook == {} || $cancelled} {
      $RunTool_privs(w:text) insert end " \n"
      $RunTool_privs(w:text) insert end "  ----- Done -------\n" \
        {verybig color2}
      set fmt "%a %b %d %H:%M:%S %Y"
      $RunTool_privs(w:text) insert end \
	"End Time: [clock format [clock seconds] -format $fmt]\n\n" {big color2}
      $RunTool_privs(w:text) insert end \
        " you can choose to save the output now\n"
      $RunTool_privs(w:cancel_btn) config -text Close

      # FIXME/BUG: Don't use grab, busy etc on Windows. Too flaky. Leads to 
      # crashes.
      if {![string compare $tcl_platform(platform) "unix"]} {
	blt::busy release .
	blt::busy release $RunTool_privs(w:text)
	grab release $RunTool_privs(w)
      }
      # 
      # Clients can wait on this. Need to find a better way than using
      # vwait on a private variable, but will have to do for now. See
      # layout_cmd (commands.tcl) for an example usage.
      #
      set RunTool_privs(prog_running) 0
    } else {
      vputs "executing hook $hook"
      eval $hook
    }
  }
  if {[$RunTool_privs(w:text) dlineinfo insert] == {}} {
    $RunTool_privs(w:text) see end
  }
}

proc RunTool:run_error {program msg} {
  global RunTool_privs tcl_platform

  tk_messageBox -title "Run Error" -type ok -icon error -message \
    "ERROR running \"$program\". Not installed?\nMessage: $msg"
  set_msg "Error running program \"$program\"... ERROR"

  # FIXME/BUG: Don't use grab, busy etc on Windows. Too flaky. Leads to 
  # crashes.
  if {![string compare $tcl_platform(platform) "unix"]} {
    blt::busy release .
    blt::busy release $RunTool_privs(w:text)
    grab release $RunTool_privs(w)
  }
}

proc run_program {mode progname args} {
  global RunTool_privs tcl_platform

  if ![string compare $tcl_platform(platform) "unix"] {
    set program "| sh -c \"$progname $args 2>&1\""
  } else {
    #
    # use sh on cygwin as well.
    # set program "|$progname $args"
    #
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname $args 2>&1\""
  }

  RunTool:make_run_window

  if {[catch {eval set RunTool_privs(pipe) [open $program  "$mode"]} errmsg]} {
    RunTool:run_error $progname $errmsg
    return
  }
  # FIXME/CHECK: See note in RunTool:get_run_output
  if {$RunTool_privs(use_blocking_mode)} {
    # FIXME/CHECK: Setting buffersize to a large value doesn't seem to
    # work (still gets flushed after each newline). Why?
    fconfigure $RunTool_privs(pipe) -buffering full -blocking true
  } else {
    fconfigure $RunTool_privs(pipe) -buffering none -blocking false
  }
  # End FIXME/CHECK:
  set RunTool_privs(prog_running) 1
      
  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable RunTool:get_run_output]]

  set fmt "%a %b %d %H:%M:%S %Y"
  $RunTool_privs(w:text) insert end \
    "Start Time: [clock format [clock seconds] -format $fmt]\n\n" {big color2}
  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}
}

#
# Run programs in the background. This is used mainly on Windows (cf:
# run_plotxy).
#
proc run_detached_program {progname args} {
  global RunTool_privs tcl_platform

  if ![string compare $tcl_platform(platform) "unix"] {
    set program "sh -c \"$progname $args 2>&1\""
  } else {
    #
    # use sh on cygwin as well.
    # set program "$progname $args"
    #
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname $args 2>&1\""
  }

  RunTool:make_run_window

  if [catch {eval exec $program &} errmsg] {
    RunTool:run_error $progname $errmsg
    return
  }

  set fmt "%a %b %d %H:%M:%S %Y"
  $RunTool_privs(w:text) insert end \
    "Start Time: [clock format [clock seconds] -format $fmt]\n\n" {big color2}
  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}

  $RunTool_privs(w:text) insert end \
    "executing $program in the background\n\n" {big color2}

  $RunTool_privs(w:cancel_btn) config -text Close

  # FIXME/BUG: Don't use grab, busy etc on Windows. Too flaky. Leads to 
  # crashes.
  if {![string compare $tcl_platform(platform) "unix"]} {
    blt::busy release .
    blt::busy release $RunTool_privs(w:text)
    grab release $RunTool_privs(w)
    catch {focus $RunTool_privs(old_focus)}
    if {$RunTool_privs(old_grab) != ""} {
      if {$RunTool_privs(grab_status) == "global"} {
	grab -global $RunTool_privs(old_grab)
      } else {
	grab $RunTool_privs(old_grab)
      }
    }
  }
}

proc run_script {args} {
  global RunTool_privs tcl_platform

  RunTool:make_run_window

  if [catch {eval $args} errmsg] {
    RunTool:run_error $progname $errmsg
    return
  }

  $RunTool_privs(w:text) insert end " \n"
  $RunTool_privs(w:text) insert end "  ----- Done -------\n" {verybig color2}
  $RunTool_privs(w:text) insert end " you can choose to save the output now\n"

  $RunTool_privs(w:cancel_btn) config -text Close

  # FIXME/BUG: Don't use grab, busy etc on Windows. Too flaky. Leads to 
  # crashes.
  if {![string compare $tcl_platform(platform) "unix"]} {
    blt::busy release .
    blt::busy release $RunTool_privs(w:text)
    grab release $RunTool_privs(w)
    catch {focus $RunTool_privs(old_focus)}
    if {$RunTool_privs(old_grab) != ""} {
      if {$RunTool_privs(grab_status) == "global"} {
	grab -global $RunTool_privs(old_grab)
      } else {
	grab $RunTool_privs(old_grab)
      }
    }
  }

  # this is not needed anymore, but I'll leave it in for a while.
  set RunTool_privs(waitvar) ""
}

proc run_source {progname startfile} {
  global RunTool_privs tcl_platform

  global tcl_platform
  if ![string compare $tcl_platform(platform) "unix"] {
    set program "| sh -c \"$progname $startfile 2>&1\""
  } else {
    #
    # use sh on cygwin as well.
    # set program "|$progname $startfile"
    #
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname $startfile 2>&1\""
  }

  RunTool:make_run_window

  if [catch {eval set RunTool_privs(pipe) [open $program  "r"]} errmsg] {
    RunTool:run_error $progname $errmsg
    return
  }
  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable RunTool:get_run_output]]

  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}
}

#
# Run the whole system as a sequence of OEs. Not used right now. Note
# the use of tkwait.
#
proc run_system_chained {num_oe} {
  global RunTool_privs tcl_platform

  for {set oe 1} {$oe <= $num_oe} {incr oe} {
    run_oe_cmd $oe
    tkwait variable RunTool_privs(waitvar)
  }
}

#
# Run the whole system via a script on Unix or via run_system_chained
# on WIN32 where there is no viable shell. Not used right now.
#
proc run_system_old {progname num_oe startprefix} {
  global RunTool_privs tcl_platform

  global tcl_platform
  if ![string compare $tcl_platform(platform) "unix"] {
    set program "| sh -c \"$progname -n $num_oe -f $startprefix 2>&1\""
  } else {
    return [run_system_chained $num_oe]
  }

  RunTool:make_run_window

  if [catch {eval set RunTool_privs(pipe) [open $program  "r"]} errmsg] {
    RunTool:run_error $progname $errmsg
    return
  }
  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable RunTool:get_run_output]]

  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}
}

proc run_system {progname num_oe startprefix} {
  global RunTool_privs tcl_platform

  RunTool:make_run_window
  global tcl_platform
  if ![string compare $tcl_platform(platform) "unix"] {
    set program "|sh -c \"$progname -m batch 2>&1\""
  } else {
    #
    # use sh on cygwin as well.
    # set program "|$progname -m batch"
    #
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname -m batch 2>&1\""
  }

  if [catch {eval set RunTool_privs(pipe) [open $program "r+"]} errmsg] {
    RunTool:run_error $progname $errmsg
    return
  }
  set fid $RunTool_privs(pipe)
  fconfigure $fid -translation "auto lf"

  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable RunTool:get_run_output]]
  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}

  for {set oe 1} {$oe <= $num_oe} {incr oe} {
    puts $fid 0			;# start anew
    set gfile [format "${startprefix}.%.2d" $oe]
    puts $fid $gfile		;# input file?
    flush $fid
  }
  puts $fid 0
  puts $fid exit
  flush $fid
}

proc run_oe {progname oe gfile prev_image} {
  global RunTool_privs tcl_platform

  RunTool:make_run_window
  global tcl_platform
  if ![string compare $tcl_platform(platform) "unix"] {
    set program "|sh -c \"$progname -m batch 2>&1\""
  } else {
    #
    # use sh on cygwin as well.
    # set program "|$progname -m batch"
    #
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname -m batch 2>&1\""
  }
  if [catch {eval set RunTool_privs(pipe) [open $program "r+"]} errmsg] {
    RunTool:run_error $progname $errmsg
    return
  }
  set fid $RunTool_privs(pipe)
  fconfigure $fid -translation "auto lf"

  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable RunTool:get_run_output]]
  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}

  if {$oe == 1} {
    puts $fid 0			;# start anew
    puts $fid $gfile		;# input file?
  } else {
    set prev_oe [expr $oe - 1]
    puts $fid 1			;# start from particular OE
    puts $fid $prev_oe		;#   prev OE number
    puts $fid $prev_image  	;#     and prev OE image
    puts $fid 0			;# change mode?
    puts $fid $gfile		;# input file?
  }
  puts $fid 0			;# want to change mode?
  puts $fid exit		;# done.

  flush $fid
}

proc read_tool_gfile {file a} {
  upvar 1 $a array
  if {[catch {open $file r} fin] != 0} {
    puts stderr "cannot open gfile \"$file\". Going ahead anyway."
    return 1
  }
  while {[gets $fin line] >= 0} {
    # is it a comment or empty line?
    if {[regexp {^[ \t]*#} $line] == 0 &&
	[regexp {^[ \t]*$} $line] == 0} {
      set line [split $line]
      if {[lindex $line 1] != "="} {
	puts stderr \
	    "Illegal syntax in gfile \"$file\". wrong format file?"
	close $fin
	return 1
      }
      # use case-indepent variable names.
      set lname [string tolower [lindex $line 0]]
      set lvalue [lrange $line 2 end]
      #
      # now simply add a variable in gvars array with the same
      # name as the one in the gfile and the corresponding value.
      set array($lname) $lvalue
    }
  }
  return 0
}

#
# obj_info says what object we're inspecting here.
#    Examples: {source}
#            : {oe <oe#>}
#            : {screen <oe#> <scr#>}
#
proc run_plotxy {obj_info progname gfile} {
  global RunTool_privs tcl_platform

  set plotxy(DUMMY) ""			;# force to be array

  read_tool_gfile $gfile plotxy
  if {$plotxy(plx_device) == 2} {		;# postscript
    set ps_device 1
  } else {
    set ps_device 0
  }

  RunTool:make_run_window
  global tcl_platform
  if ![string compare $tcl_platform(platform) "unix"] {
    set program "|sh -c \"$progname 2>&1\""
  } else {
    # hack not needed anymore
    # set program "|$progname -no-run-primvs"
    #
    # use sh on cygwin as well.
    # set program "|$progname"
    #
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname 2>&1\""
  }
  set program_hook ""
  if $ps_device {
    set program_hook \
      "run_script $RunTool_privs(w:text) insert end \"Postscript output is in plotxy.ps\""
  }
  #
  # there is a problem with running X-based programs using the usual
  # method, so we might have to resort exec'ing in the background
  # with Tcl's exec primitive.
  # eval exec $program &
  #
  if [catch {eval set RunTool_privs(pipe) [open $program "r+"]} errmsg] {
      RunTool:run_error $progname $errmsg
      return
  }
  set fid $RunTool_privs(pipe)
  fconfigure $fid -translation "auto lf"

  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable "RunTool:get_run_output [list $program_hook]"]]

  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}

  #
  # shove the data into the program now.
  #
  set imagefile ""
  if {$plotxy(plx_spec_file) == 1} {		;# user specifid filename?
    set imagefile "$plotxy(plx_infile)"
  } else {
    switch $plotxy(plx_plot_what) {
      0 {					;# plotting screen here.
	if {[get_source] != ""} {
	  set imagefile [beamline vget source FILE_SOURCE]
	} else {
	  set imagefile "begin.dat"		;# hope it exists!
	}
      }
      1 {					;# plotting OE here
	set oe [lindex $obj_info 1]
	if {$oe == ""} {
	  error "run_plotxy (Internal): No OE number supplied"
	}
	switch $plotxy(plx_image) {
	  0 {					;# at cont. plane
	    set imagefile "[format star.%.2d $oe]"
	  }
	  1 {					;# at mirr plane
	    set imagefile "[format mirr.%.2d $oe]"
	  }
	}
      }
      2 {
	set oe [lindex $obj_info 1]
	set scr [lindex $obj_info 2]
	if {$oe == "" || $scr == ""} {
	  error "run_plotxy (Internal): No SCR/OE number supplied"
	}
	set imagefile "[format screen.%.2d%.2d $oe $scr]"
      }
    }

    if {$imagefile == ""} {
	error "Internal Error in run_plotxy: Check image/plot type"
    }
  }
  puts $fid "$imagefile"

  puts $fid "$plotxy(plx_rayopts)"
  puts $fid "$plotxy(plx_comments)"
  puts $fid "$plotxy(plx_row_horiz)"
  puts $fid "$plotxy(plx_row_vert)"

  if {$plotxy(plx_row_horiz) == 11 || $plotxy(plx_row_vert) == 11} { ;# eV
    puts $fid "$plotxy(plx_unit)"
  }

  puts $fid "$plotxy(plx_scale_opts)"

  if {$plotxy(plx_scale_opts) == 2} {		;# EXTERNAL limits
    puts $fid "$plotxy(plx_ext_hmin)"
    puts $fid "$plotxy(plx_ext_hmax)"
    puts $fid "$plotxy(plx_ext_vmin)"
    puts $fid "$plotxy(plx_ext_vmax)"
  }

  puts $fid "$plotxy(plx_plot_type)"
  puts $fid "$plotxy(plx_hairline)"
  puts $fid "$plotxy(plx_ovl_mirror)"

  #
  # histograms now.
  # 
  # if do_hist == 1, that means that PLOTXY expects 0
  #

  if {$plotxy(plx_do_hist) == 0} {
    puts $fid "-1"
  } elseif {$plotxy(plx_do_hist) == 1} {
    puts $fid "0"
    puts $fid "$plotxy(plx_hist_xbins)"
    puts $fid "$plotxy(plx_hist_ybins)"
    if {$plotxy(plx_hist_lim) == 1} {
      puts $fid "$plotxy(plx_hist_xctr)"
      puts $fid "$plotxy(plx_hist_xwid)"
      puts $fid "$plotxy(plx_hist_yctr)"
      puts $fid "$plotxy(plx_hist_ywid)"
    }
  }
  puts $fid "$plotxy(plx_device)"
  flush $fid
  global tcl_platform
  # This hack is not needed anymore, but I'll leave it around for a bit.
  if {0 && ![string compare $tcl_platform(platform) "windows"]} {
      tkwait variable RunTool_privs(waitvar)
      $RunTool_privs(w:text) insert end {
------------
Due to a bug in SHADOW under Win32 ('95 and NT), we have to now 
run PRIMVS to get the output (Postscript/X Window). If the device 
is Postscript, the output file is "plotxy.ps" in the current 
working directory. PRIMVS will run in the background.
------------
} {big color2}
    # run_program r primvs -x -i plotxy.prm &
    run_detached_program primvs -x -i plotxy.prm
    update idletasks
    $RunTool_privs(w:text) see end
  }
}

#
# NOT USED right now.
#
proc run_source_plotxy {progname gfile} {
  global RunTool_privs tcl_platform

  set plotxy(DUMMY) ""			;# force to be array

  read_tool_gfile $gfile plotxy
  if {$plotxy(pl1_device) == 2} {		;# postscript
    set ps_device 1
  } else {
    set ps_device 0
  }

  RunTool:make_run_window
  set program "|sh -c \"$progname 2>&1"
  if $ps_device {
    append program "; echo 'Postscript output is in plotxy.ps'"
  }
  append program "\""
  #
  # there is a problem with running X-based programs using the usual
  # method, so we might have to resort exec'ing in the background
  # with Tcl's exec primitive.
  #eval exec $program &
  #
  if [catch {eval set RunTool_privs(pipe) [open $program "r+"]} errmsg] {
    RunTool:run_error $progname $errmsg
    return
  }
  set fid $RunTool_privs(pipe)
  fconfigure $fid -translation "auto lf"

  #
  # shove the data into the program now.
  #
  puts $fid "$plotxy(pl1_infile)"
  puts $fid "$plotxy(pl1_rayopts)"
  puts $fid "$plotxy(pl1_comments)"
  puts $fid "$plotxy(pl1_row_horiz)"
  puts $fid "$plotxy(pl1_row_vert)"

  if {$plotxy(pl1_row_horiz) == 11 || $plotxy(pl1_row_vert) == 11} { ;# eV
    puts $fid "$plotxy(pl1_unit)"
  }


  puts $fid "$plotxy(pl1_scale_opts)"

  if {$plotxy(pl1_scale_opts) == 2} {		;# EXTERNAL limits
    puts $fid "$plotxy(pl1_ext_hmin)"
    puts $fid "$plotxy(pl1_ext_hmax)"
    puts $fid "$plotxy(pl1_ext_vmin)"
    puts $fid "$plotxy(pl1_ext_vmax)"
  }

  puts $fid "$plotxy(pl1_plot_type)"
  puts $fid "$plotxy(pl1_hairline)"
  puts $fid "$plotxy(pl1_ovl_mirror)"

  #
  # histograms now.
  # 
  # if do_hist == 1, that means that PLOTXY expects 0
  #

  if {$plotxy(pl1_do_hist) == 0} {
    puts $fid "-1"
  } elseif {$plotxy(pl1_do_hist) == 1} {
    puts $fid "0"
    puts $fid "$plotxy(pl1_hist_xbins)"
    puts $fid "$plotxy(pl1_hist_ybins)"
    if {$plotxy(pl1_hist_lim) == 1} {
      puts $fid "$plotxy(pl1_hist_xctr)"
      puts $fid "$plotxy(pl1_hist_xwid)"
      puts $fid "$plotxy(pl1_hist_yctr)"
      puts $fid "$plotxy(pl1_hist_ywid)"
    }
  }
  puts $fid "$plotxy(pl1_device)"
  flush $fid

  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable RunTool:get_run_output]]
  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}
}

proc run_mirinfo {obj_info progname gfile} {
  global RunTool_privs tcl_platform

  set mirinfo(DUMMY) ""			;# force to be array

  if {[string toupper [lindex $obj_info 0]] != "OE"} {
    tk_messageBox -title "Error" -type ok -icon error -message \
      "Mirinfo can inspect only OEs, not Source or Screens."
    set_msg "Run cancelled."
    return
  }

  read_tool_gfile $gfile mirinfo

  RunTool:make_run_window
  global tcl_platform
  if ![string compare $tcl_platform(platform) "unix"] {
    set program "|sh -c \"$progname 2>&1 && cat $mirinfo(mir_outfile)\""
    set program_hook ""
  } else {
    #
    # use sh on cygwin as well.
    #set program "|$progname"
    ## set program_hook "run_program r cat $mirinfo(mir_outfile)"
    #set program_hook "run_script insert_into_run_window $mirinfo(mir_outfile)"
    #
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname 2>&1 && cat $mirinfo(mir_outfile)\""
    set program_hook ""
  }
  if [catch {eval set RunTool_privs(pipe) [open $program "r+"]} errmsg] {
    RunTool:run_error $progname $errmsg
    return
  }
  set fid $RunTool_privs(pipe)
  fconfigure $fid -translation "auto lf"

  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable "RunTool:get_run_output [list $program_hook]"]]
  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}

  #
  # shove the data into the program now.
  #
  puts $fid "$mirinfo(mir_infile)"
  puts $fid "$mirinfo(mir_title)"
  puts $fid "$mirinfo(mir_comments)"
  puts $fid "$mirinfo(mir_outfile)"
  flush $fid
}

proc run_srcinfo {obj_info progname gfile} {
  global RunTool_privs tcl_platform

  set srcinfo(DUMMY) ""			;# force to be array

  if {[string toupper [lindex $obj_info 0]] != "SOURCE"} {
    tk_messageBox -title "Error" -type ok -icon error -message \
      "srcinfo can inspect only SOURCE, not OEs or Screens."
    set_msg "Run cancelled."
    return
  }

  read_tool_gfile $gfile srcinfo

  RunTool:make_run_window 
  global tcl_platform
  if ![string compare $tcl_platform(platform) "unix"] {
    set program "|sh -c \"$progname 2>&1 && cat $srcinfo(sri_outfile)\""
    set program_hook ""
  } else {
    #
    # use sh on cygwin as well.
    #set program "|$progname"
    ## set program_hook "run_program r cat $srcinfo(sri_outfile)"
    #set program_hook "run_script insert_into_run_window $srcinfo(sri_outfile)"
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname 2>&1 && cat $srcinfo(sri_outfile)\""
    set program_hook ""
  }
  if [catch {eval set RunTool_privs(pipe) [open $program "r+"]} errmsg] {
    RunTool:run_error $progname $errmsg
    return
  }
  set fid $RunTool_privs(pipe)
  fconfigure $fid -translation "auto lf"

  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
      readable "RunTool:get_run_output [list $program_hook]"]]
  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}

  #
  # shove the data into the program now.
  #
  puts $fid "$srcinfo(sri_infile)"
  puts $fid "$srcinfo(sri_title)"
  puts $fid "$srcinfo(sri_comments)"
  puts $fid "$srcinfo(sri_outfile)"
  flush $fid 
}

proc run_minmax {obj_info progname gfile} {
  global RunTool_privs tcl_platform

  set minmax(DUMMY) ""			;# force to be array

  read_tool_gfile $gfile minmax

  RunTool:make_run_window 

  if ![string compare $tcl_platform(platform) "unix"] {
    set program "|sh -c \"$progname 2>&1\""
  } else {
    #
    # use sh on cygwin as well.
    #set program "|$progname"
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname 2>&1\""
  }
  if [catch {eval set RunTool_privs(pipe) [open $program "r+"]} errmsg] {
      RunTool:run_error $progname $errmsg
      return
  }
  set fid $RunTool_privs(pipe)
  fconfigure $fid -translation "auto lf"

  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable RunTool:get_run_output]]
  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}

  #
  # shove the data into the program now.
  #
  puts $fid "$minmax(min_infile)"
  puts $fid "$minmax(min_rayopts)"

  if {$minmax(min_rayopts) == 1} {		;# include loss at OE <oe>
    set flagval [expr $minmax(min_oe_loss) * -11000]
    puts $fid $flagval
  }

  puts $fid "$minmax(min_comments)"
  flush $fid
}

# private proc
proc extract_prerefl_composition {composition} {
  set composition [string trim $composition]
  regsub -all "\[ \t\]\[ \t\]*" $composition " " composition
  set list [split $composition]
  set matlist ""
  foreach l $list {
    set mat ""
    set cnt ""
    set matches [scan $l "%\[^0-9\]%d" mat cnt]
    if {$matches == 0} {
      return $matlist
    } elseif {$matches == 1} {
      set cnt 1
    }
    lappend matlist [list [string toupper $mat] $cnt]
  }
  return $matlist
}

proc run_prerefl {progname gfile} {
  global RunTool_privs tcl_platform

  set prerefl(DUMMY) ""			;# force to be array
  
  # variables are the following:
  # PRL_COMPOSITION A
  # PRL_DENSITY     R 
  # PRL_ENERGY_MIN  R
  # PRL_ENERGY_MAX  R
  # PRL_ENERGY_STEP R
  # PRL_OUTFILE     F

  read_tool_gfile $gfile prerefl

  catch {unset prerefl(DUMMY)}

  set materials [extract_prerefl_composition $prerefl(prl_composition)]
  set num_materials [llength $materials]
  if {$num_materials == 0} {
    tk_messageBox -title "Run Prerefl" -type ok -icon error -message \
      "Cannot parse composition \"$prerefl(prl_composition)\""
    set_msg "cancelled"
    return 1
  }

  RunTool:make_run_window 

  set extra_text [format_array prerefl]
  regsub -all "prl_" $extra_text "    " extra_text
  $RunTool_privs(w:text) insert end "Parameters used:\n" {color2}
  $RunTool_privs(w:text) insert end "$extra_text\n\n" {color2}
  $RunTool_privs(w:text) see end

  if ![string compare $tcl_platform(platform) "unix"] {
    set program "|sh -c \"$progname 2>&1\""
  } else {
    #
    # use sh on cygwin as well.
    #set program "|$progname"
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname 2>&1\""
  }
  if [catch {eval set RunTool_privs(pipe) [open $program "r+"]} errmsg] {
      RunTool:run_error $progname $errmsg
      return
  }
  set fid $RunTool_privs(pipe)
  fconfigure $fid -translation "auto lf"

  # FIXME: hooks are broken!
  #set extra_text [format_array prerefl]
  #set program_hook \
  #  "run_script $RunTool_privs(w:text) insert end \"$extra_text\""

  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable "RunTool:get_run_output"]]
  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}

  #
  # shove the data into the program now.
  #

  # always force compound to keep it simple.
  puts $fid "1"			;# compound
  puts $fid "$prerefl(prl_density)"

  puts $fid "$num_materials"		;# num of species
  foreach mat $materials {
    puts $fid "[lindex $mat 0]"	;# At. symbol
    puts $fid "[lindex $mat 1]"	;# formula index.
  }
  puts $fid "$prerefl(prl_energy_min)"
  puts $fid "$prerefl(prl_energy_max)"
  puts $fid "$prerefl(prl_energy_step)"
  puts $fid "$prerefl(prl_outfile)"

  flush $fid
}

proc run_sector {progname gfile} {
  global RunTool_privs tcl_platform

  set sector(DUMMY) ""			;# force to be array
  
  # variables are the following:
  # SCT_OE          I
  # SCT_NSECTOR     I
  # SCT_PATH        D

  read_tool_gfile $gfile sector

  catch {unset sector(DUMMY)}

  set num_oe [get_num_oe]
  if {$sector(sct_oe) > $num_oe} {
    tk_messageBox -title "Invalid OE" -type ok \
      -icon error -message \
      "Cannot sector non-existent OE #$sector(sct_oe)"
    set_msg "cancelled"
    return 1
  }

  # Create the Mirr0 directory, and save SHADOW files.
  set mirr0_dir [file join $sector(sct_path) Mirr0]
  if {[catch {file mkdir $mirr0_dir}]} {
    tk_messageBox -title "Error creating directory" -type ok \
      -icon error -message \
      "Cannot create Mirr0 directory \"$mirr0_dir\" to save"
    set_msg "cancelled"
    return 1
  }

  set cwd [pwd]
  cd $mirr0_dir
  do_save_source start.00
  do_save_system systemfile.dat
  # Make the directory absolute to avoid problems with the nsector program.
  set mirr0_dir [fix_platform_filename [pwd]]
  cd $cwd

  # Create sector.inp file.
  set file "sector.inp"
  if {[catch {open "$file" w} f] != 0} {
    tk_messageBox -title "Save Error" -type ok -icon error -message \
      "Cannot open temporary data file file \"$file\" to save"
    set_msg "cancelled"
    return 1
  }
  puts $f "nsector = $sector(sct_nsector)"
  puts $f "path = $mirr0_dir"
  close $f

  RunTool:make_run_window 

  set extra_text [format_array sector]
  regsub -all "sct_" $extra_text "    " extra_text
  $RunTool_privs(w:text) insert end "Parameters used:\n" {color2}
  $RunTool_privs(w:text) insert end "$extra_text\n\n" {color2}
  $RunTool_privs(w:text) see end

  if ![string compare $tcl_platform(platform) "unix"] {
    set program "|sh -c \"$progname 2>&1\""
  } else {
    #
    # use sh on cygwin as well.
    #set program "|$progname"
    global gvars
    set shell [file join $gvars(shadow_bin) sh]
    set program "|$shell -c \"$progname 2>&1\""
  }
  if [catch {eval set RunTool_privs(pipe) [open $program "r+"]} errmsg] {
      RunTool:run_error $progname $errmsg
      return
  }
  set fid $RunTool_privs(pipe)
  fconfigure $fid -translation "auto lf"

  # FIXME: hooks are broken!
  #set extra_text [format_array sector]
  #set program_hook \
  #  "run_script $RunTool_privs(w:text) insert end \"$extra_text\""

  after $RunTool_privs(delay) [list catch [list fileevent $RunTool_privs(pipe) \
    readable "RunTool:get_run_output"]]
  $RunTool_privs(w:text) insert end "executing $program\n\n" {big color2}
}

