;+ 
;      ============================= XPLOT  ======================
;
; 
; XPLOT is a general package for creating X-Y plots that also allows for
; sophisticated data visualization, processing, manipulation and analysis.
; Some of the XPLOT features are: 
;
;  1) Direct loading of data from multicolumn ASCII files. Multiple sets of
; data can be accessed from several files and merged together. Interactive
; action on data columns with the creation of new data from scratch or by
; manipulating existing data 
;  2) Access to SPEC data files 
;  3) Direct printing from Unix workstations and PCs, and creation of
; PostScript and encapsulated PostScript files 
;  4) Linear and logarithmic X- and Y-axes. Apply and change symbols, line
; styles, colors, thickness, fonts etc. 
;  5) Label, annotate, add legends, titles, etc. to the plots. Zoom, shift,
; switch columns, and over-plot data. 
;  6) Save and restore plots and attributes in a XPLOT backup file. 
;  7) Data analysis: extrema, moments, integrals, derivatives, CDFs,
; smoothing, user-defined operations, interactive Fourier filtering,
; interpolations, convolutions and correlations. It also includes several
; fitting routines: linear regression, polynomial fit, Gaussian fit and
; nonlinear fit (the gradient-expansion algorithm or the Powell method) with an
; on-line user-defined function. 
;
;
; XPLOT also includes some packages for specific data analysis. At present
; the XAID package for XAFS (X-ray Absorption Fine Structure)
; analysis is available. New applications and interfaces for data
; processing applied to different synchrotron techniques are under
; development, and they will be integrated in future releases.
;
;
; On line help is available in most of the cases. A Web Xplot page and
; documentation can be foubd at:
; http://www.esrf.fr/computing/expg/subgroups/theory/idl/xplot/xplot.html
;
;
; Overplotting with Xplot
; =======================
;
; One of the most important features of Xplot is the capability to display
; multi-line graphs with data coming fro different origins. Data in Xplot
; is structured in the so-called sets. 
; 
; A set is an array or matrix containing N points (or rows) and M columns.
; 
; Xplot stores the data in multiple sets, and each set may hold a
; different number of points or columns. Xplot allows multi-line plots, in
; which the data for each line is a couple of columns from a given set.
; Xplot options Calculations and File/Load Data File... allows to create
; new columns in defined sets and new sets from files or from operations
; between existing sets. 
; 
; You can change the X and Y columns for the plot in the current set of
; data by just typing the column numbers in the bottom of the Xplot
; window. It is very important to enter <Enter> after modifying any text
; widget in Xplot. That permits Xplot to understand you have finished with
; the editing, and process the event.
;
; If you want to do a plot with multiple lines, first load the data set
; with the first line. Then use the "Save Plot" button to store this set
; as a "saved set". At this moment, the saved set is copied to the
; current set, and both will be displayed. Then, you can change columns
; of even load a new data set using File/Load menu and the new data set
; will be overplotted on the saved one. Repeat this process as many times
; as you want. Once a set is saved, the Xplot controls do do affect it.
; The window created by pressing the "Plot Mgr" (Plot manager) button
; permits to delete save sets, or to promote a saved set to the current
; set.
;
;
;
;
;
;
; Xplot has been written by:
;	Manuel Sanchez del Rio
;	European Synchrotron Radiation Facility
;	BP 200
;	38043 Grenoble-Cedex 9
;	Email: srio@esrf.fr
;
;-

;+
;      ================= USE OF XPLOT FROM IDL PROMPT  ==========
;
;
;
;	NAME:
;		XPLOT
;	PURPOSE:
;		to plot and interactive manipulate data
;	CALLING SEQUENCE:
;		XPLOT [,indata] [,keywords]
;	OPTIONAL INPUT PARAMETERS:
;	indata can be one of the following:
;		1) filename: a string containing the file name to be
;		plotted or an idl variable [fltarr(n_points,n_columns]
;		with data. The file must be ascii data arranged in 
;		columns. Lines starting with characters [except . + -]
;		are comments and not considered for the plot.
;		2) an idl variable containing the numerical data
;		set (a fltarr(n_columns,n_points)
;		3) x,y two arrays with the  abscissas and ordinate
;		arrays.
;		4) h an IDL handler pointing to a set or to a number
;		of sets.
;		
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls Xplot.
;		When this ID is specified, a death of the caller results 
;		in a death of Xplot.
;		WINDOW_SIZE = a 2-dim vector containing the width and
;		height of the graphical window (in pixels).
;		XCOL = for multicolumn entries, the column number to be
;		plotted in the X axis (default=1)
;		YCOL = for multicolumn entries, the column number to be
;		plotted in the Y axis (default=last column)
;		WTITLE = title for the main window
;		TITLE = title for the plot
;		XTITLE = abscissas title. In the case that xtitle is
;			set as XTITLE='-1', then xplot will place
;			the text of COLTITLES corresponding to the 
;			plotted column.
;		YTITLE = ordinates title. In the case that ytitle is
;                       set as YTITLE='-1', then xplot will place
;                       the text of COLTITLES corresponding to the
;                       plotted column.
;		COLTITLES = string array with column labels.
;		LEGEND = A legend call to the LEGEND procedure by 
;			 F K Knight (knight@ll.mit.edu)
;		STATE = a xplot state structure (only for internal 
;			use of Xplot (when creating clones))
;		PARENT = The wiggler id of the main Xplot window 
;			created. This is useful to pass this id 
;			as parent of other widgets that must die when
;			the present XPlot window dies.
;		[XY]LOG = when set to 1, initializes [XY] logaritmic axis.
;		SPEC = input, initialize xplot with a SPEC file. Input
;			may be a file name (string) or a spec handler (str).
;			Example, IDL> xplor,spec='file.dat'
;               XRANGE=[xmin,xmax] range of the x variable for the plot
;               YRANGE=[ymin,ymax]              y
;
;
;	OUTPUTS:
;		Open a widget utility and present a graphic.
;	COMMON BLOCKS:
;		None
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;	RESTRICTIONS:
;		Unknown.
;	PROCEDURE:
;		Uses the PLOTFILE routine for producing the plot
;	KNOWN BUGS:
;		When starting xplot with coltitles keyword, the "clone"
;		  option does not work.
;		[xy]title='-1' does not work properly with spec/mca data.
;	
;		When the COLTITLES keyword is used, or with SPEC files,
;		if the labels of the columns are very long and if the
;		font size is large, idl tries to resize the xplot window
;		to a larger area that the screen, and an error like this
;		is produced, hanging the session:
;		Xlib: unexpected async reply (sequence 0x72fc)!
;		% X windows protocol error: BadDrawable (invalid Pixmap \
;		  or Window parameter).
;		Temporary solution: Use a smaller font.
;
;	MODIFICATION HISTORY:
;       by  Manuel Sanchez del Rio. ESRF. December 1993
;		94-01-21 MSR improves the evaluate option
;		94-02-02 MSR add style options in LIMITS menu
;		94-02-18 MSR removes common blocks and allow
;			multiple copies of the widget
;		94-05-19 MSR fix a bug in the print buffer managment
;			found by R. Dejus.
;		94-06-27 MSR adds the /Calculations/correlation function 
;			option.
;		94-11-17 MSR adds Tools/Xplot and Tools/IDL_command options
;		95-02-11 MSR adds Calculations/Fitting option, check 
;			existence of input files.
;		95-03-08 MSR: updating of fourierfilter: we pass now also 
;			the abscissas values.
;		95-08-08 MSR: adds LEGEND keyword.
;
;		95-09-14 MSR Releases Xplot version 2.0
;		Nov 1995  version 2.1 major upgrades:
;			- Integration and interfacing of spec_acces
;			- COLTITLES keyword
;			- Options Polynomial and Gaussian fit integrated
;			in a single one.
;			- new interface of non-linear fit.
;		96-01-18 srio@esrf.fr changes xtext call by xedit from RJD.
;		96-01-19 srio@esrf.fr adds options [xy]title='-1'
;		96-01-22 srio@esrf.fr adds the possibility to edit column
;			titles for SPEC files, and number the columns.
;		96-04-12 srio@esrf.fr adds /xlog and /ylog kws. Change 
;			fieldlen in settitles.
;		96-05-29 srio@esrf.fr xplot/title: when using spec files, 
;			the symbol #x (i.e. #S) is interpreted as its 
;			corresponding value in the spec scan header.
;		96-07-09 srio@esrf.fr Adds SPEC keyword.
;		96-08-13 srio@esrf.fr changes "'" by '"' in xplot_plot
;		96-08-20 srio@esrf.fr makes the main window resizable,
;			and removes the ZoomSel button (now automatic selection
;			of zooming area by dragging the mouse with the left 
;			button down). Do not write intermediate files
;			for view/showData. Introduce xhelp.
;		96-12-04 srio@esrf.fr add "catch" in xplot_event to
;			avoid crassing the application when there is an error.
;		97-01-15 srio@esrf.fr complete the list of spec functions in
;			xplot_compile().
;		97-02-17 srio@esrf.fr debugging: log->lin change when saving, 
;			dynamic_resize of column widgets, correct cursor 
;			read-out when switching between Xplot windows. Add 
;			path to pickfile and uses xhelp for help.  Version 2.2
;		97-06-27 srio@esrf.fr added View/Mesh option. version 2.2.
;			Use copy_structure to restore backup file (assures 
;			compatibility between save files from different xplot
;			versions).
;		97-09-07 srio@esrf.fr added non-interactive use of xplot. 
;			Version 2.4
;		97-09-30 srio@esrf.fr substitutes read_ascii to rascii
;		97-10-27 srio@esrf.fr version 2.5, with lots of modifications:
;			New zoom rubberband code. Re-load file. Apply
;			button in many windows. Changed many calculations 
;			windows (smooth, convolution, etc), non-linaer fit 
;			with fixed parameters, new Plot Manager button 
;			obsoleting the Set Current one, fixed many bugs 
;			regarding zoom limits and resizing, etc, etc.
;		97-11-10 srio@esrf.fr adds Edit/Data... menu using xtable.
;		97-12-03 srio@esrf.fr adds xplot_cleardata() and fix
;			a bug with non-interactice use of plotmgr. V 2.5.1.
;			Adds xplot_setlimits().
;		98-02-18 srio@esrf.fr adds NOREFRESH keyword to 
;			 xplot_[setlimits,loadfile,changecol,exebuffer,
;			errorbars,addcol,plotmgr] routines.
;			Fix a bug (not displaying error bars in saved
;			sets.)
;		98-03-25 srio@esrf.fr small changes to allow starting
;			xplot when spec_access is not available.
;		98-04-07 srio@esrf.fr removed /no_copy kw in main 
;			widget_draw definition (problems in w95).
;
;-
;

;+
;
;      ================= NON-INTERACTIVE MANIPULATION OF XPLOT ==========
;
;	Since Xplot version 2.4, the Xplot internals are being modified
;	to allow also non-interactive manipulation of Xplot.
;	This is very useful when using Xplot from other widget applications
;	and one wants to prepare sophisticated plots with options not
;	availables as keywords in the main Xplot call. Typical examples
;	are to overplot several plots, to add commands in the EXEbuffer,
;	or to display error bars.
;	The access is done by a set of routines with general syntaxis:
;
;	xplot_<name>, parent [,other_inputs, keywords]
;
;	where <name> indicates the action to be performed (i.e. loadfile,
;	quit, etc.), parent is the widget id of the manin Xplot window to 
;	be manipulated and the other information  is passed using the
;	other inputs and keywords.
;
;	EXAPLE:
;		; you can execute these commands as an example
;		tmp = fltarr(5,100)
;		tmp(0,*) = findgen(100)
;		tmp(1,*) = tmp(0,*)^2     ; first curve
;		tmp(2,*) = tmp(0,*)^2+100 ; second curve
;		tmp(3,*) = tmp(0,*)^2-500 ; bottom of error bars
;		tmp(4,*) = tmp(0,*)^2+500 ; top of error bars
;		xplot,parent=p,/no_block  ; starts xplot, retrieved parent id
;		xplot_loadfile,p,tmp      ; load the data in xplot
;		xplot_changecol,p,ycol=3  ; set Y as column 3
;		xplot_controls_action,p,linestyle=1 ; set dotted line
;		xplot_savecurrent,p       ; save this plot
;		xplot_changecol,p,ycol=2  ; overplot column 2
;		xplot_controls_action,p,linestyle='0' ; overplotted line to solid
;		xplot_errorbars,p,4,5     ; plot error bars
;		xplot_exebuffer,p,setbuffer=$ ; add an annotation
;			"xyouts,.2,.7,'Example of non-interactive label',/norm"
;
;	The available routines are listed below.
;-
;
;=========================================================================
;
; this is just to define spec_* as functions to allow the use
; of xplot when specacess is not installed. MSR 95/11/28
; 
pro xplot_compile
forward_function spec_access, spec_scan ,spec_labels,spec_data,$
  spec_headers,spec_columns,spec_motors,spec_name,spec_points,spec_save
end
;
;=========================================================================
;
function xplot_version
return,(xplot_defaults('version')).version
;return,'2.4'
end
;
;=========================================================================
;
function xplot_text,text
;
; returns a text array with particular contents
;
case strupcase(text) of
  'ANNOTATE': text = ['This Xplot options calls the IDL utility ANNOTATE.',$
			'The ANNOTATE widget allows you to interactively ',$
			'annotate images and plots with text and drawings.',$
			' ',$
			'The ANNOTATE utility is *NOT* integrated in Xplot.',$
			'This means that the objects overplotted with ',$
			'ANNOTATE onto your Xplot graphic will not be passed',$
			'to XPlot, and will be lost when closing the ANNOTATE',$
			'window. The Xplot graphic modified by ANNOTATE',$
			'can only be printed from the ',$
			'ANNOTATE/File/Write PostScript/Everything option to',$
			'a PostScript file containing the bitmap of the',$
			'display, not the PostScript objects and lines, as',$
			'Xplot does. In any case, this application may be ',$
			'useful sometimes, and that is why we are here.']
  'HELPABOUT': text = [' ',' ', $
		  '                    Xplot                   ',$
		  ' ',$
		  '                     by                     ',$
		  ' ',$
		  '          Manuel Sanchez del Rio            ',$
		  ' ',' ', $
		  '   European Synchrotron Radiation Facility  ', $
		  '      BP 220, F-38043 Grenoble Cedex 9      ', $
		  ' ', $
		  '            email: srio@esrf.fr             ', $
		  '             Tel: +33-76882513              ', $
		  '             Fax: +33-76882160              ', $
		  ' ',' ']
  'LIMITS': text = [' ', $
	'Enter here the [xy]min, [xy]max and [xy]style. The values for',$
	'minima and maxima entered here will be applied to the graph. If',$
	'the minimum and the maximum vales are the same, the program',$
	'ignores the values and applies the default interval range. The',$
	'default interval range is calculated taking into account only',$
	'the current set, and not the saved sets. That is why sometimes the',$
	'data of the saved sets may be hidden, and one has to change manually',$
	'the limits to give a suitable interval that includes them. ',$
	' ',$
	'The style option must be chosen between the values: ',$
	'  Exact: adjust exactly the plot limits to the entered values',$
	'  Extended: Extends 5% the range in each direction',$
	'  No axis: the axis and its annotation are not drawn',$
	'  No box: avoids the box-style axis around the data',$
	'  No zero: inhibits setting the axis minimum to zero when the ',$
	'           data are all greater than zero',$
	' ',$
	' Technical note: this options controls the IDL system variable',$
	' ![x,y].style, and gives values of 1,2,4,8 and 16 for the',$
	' respective options. Although ![x,y].style is a bitwise variable,',$
	' the possible multiple option choice is not available from the ',$
	' current version of Xplot.']
  'TITLES': text = [' ', $
     'Place in these boxes the titles for the plot.',$
     ' ',$
     'You may use the positioning and font idl commands which start with ',$
     'the symbol "!". Few possibilities are: ',$
     '!D shift down to subscript level and decrease font by 62%',$
     '!L shift down to second-subscript level and decrease font by 62%',$
     '!E shift up to exponent level and increase font by 44%',$
     '!U shift up to upper-subscript level and increase font by 62%',$
     '!I shift down to index level and decrease font by 44%',$
     '!N shift back to normal level',$
     '!S save position',$
     '!R restore position',$
     '!C Carriage return',$
     '!! Display the ! symbol',$
     '!3 start Simplex Roman Font (default)',$
     '!4 start Simplex Greek Font (a=alpha, b=beta, l=mu, h=theta, m=nu',$
     '                              r=sigma, q=rho, v=chi, w=psi, x=omega)',$
     '!X return to the entry font',$
     ' ',$
     'When you start XPLOT with the keyword "COLTITLES" keyword, or when', $
     'you use SPEC files, then a text in the corresponding [xy]title equal ',$
     "to '-1' means that you will get as [xy]titles those defined in the",$
     "'Column Titles...' option.", $
     ' ',$
     'With SPEC files, for all the title entries, the symbol #x (i.e. #S for',$
     'scan number or #F for file name) is substituted by its corresponding',$
     'value in the scan header.']
  'MESH': text = [' ',' ', $
     'The mesh option permits to display a multiple data set in a 3D view. ',$
     ' ',$
     'A multiple data set is defined as a single data set with different',$
     'plots one after the other. All individual plots must have the same',$
     'number of points. The position where a plot ends and the next plot starts',$
     'can be defined by the line position (or the number of points of the',$
     'individual plots) of by an additional label column. The label column',$
     'has a different value for each indivudual plot.',$
     ' ',$
     'Three kind of MESH view are possible: lines, surface and contour.',$
     ' ',$
     'When the MESH option is set in XPLOT other XPLOT functions (like',$
     'Save Current, Plot symbols, etc.) do not work with it. The attributes',$
     "to the MESH option are given in the 'Additional keywords' box.",$
     ' ',$
     'Example: Consider two curves with points (x,y)=(1,1), (2,6) and (3,3) ',$
     'for the first one and (1,3), (2,7) and (3,0) for the second one. ',$
     'The first curve is labelled with the value 27 and the second is labelled ',$
     'with 60.  The data set may be (col1=x, col2=label and col3=y) :',$
     '1  1 27',$
     '2  6 27',$
     '3  3 27',$
     '1  3 60',$
     '2  7 60',$
     '3  0 60',$
     ' ',$
     'This format is also presented in the SPEC mesh scans.',$
     "If you load this set in Xplot, the 'split plot' can either be set by",$
     " 'each 3 points' or 'using a label column 3'. A negative value of the column",$
     '(i.e. col=-3) indicates that the plots are labelled with their label values,',$
     'otherwise Xplot labels them as 0,1,2,...',$
     ' ',$
     'Typical examples of accepted keywords are:',$
     'For the lines/surface kind: AZ(rotation angle around Z), ZAXIS (Z axis ',$
     'position),ZRANGE ([Zmin,Zmax] limits) and ZLOG (=1 for log Z scale).',$
     '(Note: These keywords are the only ones accepted if you run idl 4.x)',$
     'For contour mesh the NLEVELS kwyword can be used. See the IDL users',$
     'documentation for a complete list of accepted keywords. ' ]
  'COLUMNOPER': text = [' ',$
   "The 'Operation With Columns' option allows the user to ", $
   'calculate a new column data in the current set as',$
   'a function of the existing ones.', $
   '   ', $
   'The resulting column can either: ', $
   '    1) Be appended to the set, thus substituting the ', $
   '       current set by a new one with one more. ',$
   '    2) Substitute an existing column in the set.',$
   '    3) Export the result to a new Xplot window.' ]
  'SMOOTH': text = [' ',$
	'This option smooths the Y data of the current graph.',$
	' ',$
	'Two methods are available: ',$
	'  Boxcar average: the results are a current set smoothed with a ',$
	'                 boxcar average of the specified width.',$
	'  Least Squares: the results are a current set smoothed applying',$
	'                 a least squares smoothing polynomial filter',$
	'                 also called Savitzky-Golay smoothing filter.',$
	'   ', $
	'It uses the SMOOTH function of IDL for the Boxcar method and',$
	'the POLY_SMOOTH routine of Frank Varosi for the Savitzky-Golay',$
	'one.',' ']
  'OPERWITHSETS': text = [' ',$
      "The 'operations with sets' option allows the user to create", $
      'a new set of data called C (with abscissas Cx, and ', $
      'ordinates Cy) which is calculated as a function of two', $
      'existing sets: A (usually the current displayed data, ', $
      'with abscissas Ax and ordinates Ay) and an optional', $
      'set B read from a 2-column file.', $
      '   ',$
      'The user must provide the evaluation expression written', $
      'using IDL syntax, using Ax,Ay,Bx,By and Cx as variables.', $
      '   ',$
      'In the case that the two sets A and B have different', $
      'abscissas, the set B is substituted by an inter-', $
      'polated one with new abscissas as Ax.', $
      '   ',$
      'This option can also be used to create new data, for', $
      'example abscissas Cx:findgen(100) and ordinates Cy:Cx^2']
  'FITPOLGAUSS': text = [' ',$
        'The Polynomial/Gaussian fit option of XPlot permits to perform', $
	'different kind of linear, polynomial and gaussian fitting.',$
	'The different options refer to different IDL routines.', $
	'  ',$
	'Please read the documentation in the Help/Routines for',$
	'detailed references of the method used for each fit.', $
	'  ',$
	'A correspondence between Xplot polynomial fit options and ',$
	'the IDL routines is the following: ', $
	'  ',$
	'  1) Chi-Square Liner Regression                        - LINFIT', $
	'  2) Robust Estimation Linear Regression                - LADFIT', $
	'  3) Least Square Polynomial Fit                        - POLYFITW', $
	'  4) Singular Value Decomposition Polynomial Fit        - SVDFIT', $
	'  5) Gaussian + independent term fit                    - GAUSS_FIT', $
	'  6) Gaussian + 2nd degree polynomial                   - GAUSSFIT']
  'EXEBUFFEREDIT': text = [' ',$
		'This is an advanced option which permits ', $
		'to create and modify the Xplot internal buffer ',$
		'where the Legend and Label commands are stored.',$
		'This buffer is a set of IDL commands executed ',$
		'right after the plot. It is used to store labels ',$
		'annotations etc.',$
		'  ',$
		'It is also used for modifying the legend/labels ',$
		'(created with ethe Edit/Legend and Edit/Label options)',$
		'after being saved. Its use requires a small knowdledge ',$
		'of IDL syntax.', $
		'It is not recommended for the general user.', $
		'   ']
  'SAVEAS': text = [' ',$
	'This option permits to write a new ASCII file containing the ',$
	'numerical data of the current set. Just enter the file name',$
	'where to write the data. ',$
	' ',$
	'You can save only the two columns of current data in the current',$
	'set (default option), meaning the X and Y column currently',$
	'displayed, or all the columns present in the current set, thus',$
	'a multi column file.',$
	' ',$
	'If the file name entered already exists, then a confirmation window',$
	'will appear.',$
	' ',$
	'This option is also very useful to create temporal files, which can',$
	'be re-read in other Xplot window (because it is possible to create',$
	'many Xplot windows, holding simultaneously different data sets), ',$
	'or used for Operations with sets... (in the Calculations menu.)']
  'COLTITLES': text = [' ',$
	'This option permits to edit the column titles displayed in the ',$
	'droplist at the bottom of the Xplot window.']
  'SHIFT': text = [' ',$
	'This option permits to shift horizontally or vertically the current',$
	"displayed X/Y line of a given quantity measured in plot's ",$
	'coordinates.',$
	' ',$
	'This option is very useful when one has to present a multiple set',$
	'of lines (for example spectra) which change a little from one to',$
	'another, and we are not interested in the absolute coordinates.',$
	' ',$
	'Thus, it permits to shift the plot line without changing the data',$
	'or creating a new set of data. ',' ']
  'ERRORBARS': text = [' ',$
	'This option displays error bars in your plot. For that purpose the',$
	'upper values and bottom values of the error bars to be displayed ',$
	'have to be defined as columns in the current set. This can be done,',$
	'for example, by using the Calculations/Operations with columns...',$
	'option.']
  'DRAWDIM': text = [' ',$
	'This option permits to define the dimension (in pixels) of the',$
	'Xplot graph window. The Horizontal/Verical ratio is conserved',$
	'in the plotting option.',' ']
  'SECONDAXIS': text = [' ',$
	'This option permits to draw a second X axis (on the top of the',$
	'data window) and/or Y axis (on the right). Give the minimum and',$
	'maximum values for the axis. These values are completely',$
	'disconnected from the data of the graph, thus are not changed when',$
	'graph limits are modified. ',$
	'The same minimum and maximum values means that the second axis ',$
	'will not be plotted. ',$
	' ',$
	'Advanced feature: If your graph does not look very nice after the',$
	'insertion of the second axis, use the Edit/System Variables/',$
	'!p.position to reduce/center your graph.']
  'INTERPOLATION': text = [' ',$
	'This options creates a new Xplot window with interpolated data.',$
	' ',$
	'This option is normally used to create a new set of data with the',$
	'same look than the original, but with a larger number of points.',$
	' ',$
	'Two possible interpolations are possible: ',$
	'  Linear Interpolation (using INTERPOL from IDL)',$
	'  Spline interpolation (using SPLINE from IDL). ',$
	'  Parametric Spline (using SPLINE_P from IDL). Here the number',$
	'                    of points of the result is defined by the',$
	'                    program itself.',' ']
  'CONVOLUTION': text = [' ',$
	'This option permits to calculate convolution and correlation between',$
	'two data sets, A (the current set) and B.',$
	'B can be one of the following cases:',$
	'  B=A autoconvolution/autocorrelation cases.',$
	'  B   from an external file.',$
	'  B   a gaussian, lorentzian or a liner combination of both',$
	'     (the so-called pseudo-voigt function:',$
	'      v(x)=r*L(x)+(1-r)G(x), with r the ratio or linear coefficient',$
	'       between Lorentzian and Gaussian).',$
	' ',$
	' The Convolution and correelation are calculated from:',$
	' CONV(x) = integral( A(y) B(x-y) dy)',$
	' CORR(x) = integral( A(y) B(y-x) dy)',$
	' Of course, CONV(x) = CORR(x) if B(y) is symmetrical.',$
	' ',$
	'The prigram interpolate both curves (using INTERPOL) to the',$
	'biggest interval and smallest step between both abscissas and',$
	'convolute/correlate them by using CONV or INFL routines).',$
	'The result may be normalized. The program uses the FFT method',$
	'to perform the calculations.',$
	' ',$
	'The available controls are: ',$
	' Method: to use either convolution or correlation.',$
	' Type: Select to conv/corr A with itself (auto), with another',$
	' function from a file or with a pseudo-voigt function.',$
	' Normalization: Divide the result by integral(A) (to the first',$
	'                set), by integral(B) (to the second set), or by',$
	'                integral(A)*integral(B) (to both sets).',$
	' Voigt ration: when tyle="Gaussian and/or Lorentzian" this ',$
	'               permits to give the Gaussian and Lorentzian',$
	'               contriburions to the pseudo-voigt function (1=',$
	'               pure Lorentzian, 0=pure Gaussian).',$
	' Voigt Full Width at Half Maximum: The width for the voigt',$
	'               function (when selected).',$
	' Overplot original data: Overplots the original data on the',$
	'               top of the convoluted data.',$
	' Write output to a file: if Yes, give the name in the next entry.',$
	' Output file name: enter the nema for a 2-column file with the ',$
	'                   results.',$
	' Calculation by: The calculations may be done using either the Fast ',$
	'                 Fourier transform algorithm or by numerical ',$
	'                 integration. The FFT is much faster and should be',$
	'                 preferred.',$
	' ']
  else: text = ''
endcase
return,text
end ; xplot_text
;
;=========================================================================
;
function xplot_readspec, state, scan_index, mca, oldcol=col
on_error,2
   handle_value, state.spec.handle, hspec
   if not mca then begin
      data = spec_data(hspec, scan_index, /index, /float)
      ncol = n_elements(data(*,0))
      labels = spec_labels(hspec, scan_index, /index)
      if ncol ne n_elements(labels) then begin
         tmp = widget_message(DIALOG_PARENT=state.wids.main,$
		'Error reading data from scan #'+ $
                 strcompress(scan,/REM)+'.',/ERROR)
         return, 0
      endif
; srio adds:
      ;labels = strcompress((sindgen(ncol+1))(1:ncol) ) + ': '+labels
      tmplabels = labels(0)
	for i=1,n_elements(labels)-1 do tmplabels = tmplabels+'!'+labels(i)
      state.spec.labels = tmplabels

      xcol = 1
      ycol = ncol
      if keyword_set(col) then begin
         if state.xpdt.Xcol le ncol and state.xpdt.Ycol le ncol then begin
            xcol = state.xpdt.Xcol
            ycol = state.xpdt.Ycol
         endif
      endif
   endif else begin
      data = spec_data(hspec, scan_index, /index, /float, /mcdata, $
                                           channels=chann, calibrated=energy)
      data = transpose(temporary(data))
      ncol = n_elements(data(*,0))
      nch = n_elements(chann)
      if n_elements(data(0,*)) ne nch then begin
         tmp = widget_message(DIALOG_PARENT=state.wids.main,$
		'Error reading MCA data from scan #'+ $
                strcompress(scan,/REM)+'.',/ERROR)
         return, 0
      endif
      data = [reform(data,ncol,nch), reform(chann,1,nch), reform(energy,1,nch)]
      widget_control,state.wids.xcolmca,get_value = aux
      xcol = ncol+1+aux
      ycol = 1
      if keyword_set(col) then begin
         if state.xpdt.Ycol le ncol then begin
            ycol = state.xpdt.Ycol
         endif
      endif
   endelse

   handle_value, state.spec.handle, hspec, /set
   state.spec.scan = scan_index
   state.spec.mca = mca

   if not mca then begin
      tmplabels = strcompress((sindgen(n_elements(labels)+ $
	1))(1:n_elements(labels)) ) + ': '+labels
      widget_control,state.wids.xcolSPEC, SET_VALUE=tmplabels, $
	SET_DROPLIST_SELECT = xcol-1
      widget_control,state.wids.ycolSPEC,SET_VALUE=tmplabels, $
        SET_DROPLIST_SELECT = ycol-1
   endif else begin
      widget_control,state.wids.ycolmca,SET_VALUE=strcompress(ycol,/REM)
   endelse
   widget_control,state.wids.ncol,SET_VALUE=strcompress(ncol,/REM)

   state.xpdt.Xcol = xcol
   state.xpdt.Ycol = ycol
   state.xpdt.shift=[0.,0.]
   state.xpdt.ErrPlot=[0,0]

   return, data
end
;
;=========================================================================
;
pro get_fileinfo, state, spechandle, file
   on_error, 2
   file='unknown'
   list = spec_access(spechandle,/list,/compress,/str, $
                        filename=file,mcdata=state.mca, no_empty=(1-state.mca))
   ; coommented by srio 97/02/13. Useless?
   ;spawn, 'file '+file, aux, /sh
   widget_control, state.file, set_value=file
   if list eq '' then list = '<none>'
   widget_control, state.list, set_value=list
   return
end
;
;=========================================================================
;
pro xplot_specscan_init, state, error_status
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      tmp = widget_message(DIALOG_PARENT=state.wids.main,$
	"Can't find SPEC file: "+file, /ERROR)
      goto, out
   endif

   handle_value, state.handle, spechandle, /no_copy

   scan_index = spec_scan(spechandle, 'current',/return_index)
   state.scanidx = scan_index
   scan = spec_scan(spechandle, scan_index, /index)
   widget_control, state.field, set_value=scan
   widget_control, state.current, set_value='Current scan :'+string(scan)

   get_fileinfo, state, spechandle, file

Out:
   handle_value, state.handle, spechandle, /set, /no_copy
end
;
;=========================================================================
;
pro xplot_specscan_event, event
   widget_control, event.top, get_uvalue=state
   if event.id eq event.top then begin
      xplot_specscan_init, state, error
      widget_control, event.top, set_uvalue=state
      if error then widget_control, event.top, /destroy
      return
   endif

   handle_value, state.handle, spechandle, /no_copy
   widget_control, event.id, get_uvalue=evuval

   case evuval of
      'ACCEPT': begin
         widget_control, state.field, get_value=scan
         scan=strtrim(scan(0),2)
         index = -1
         if strpos(scan, '-') eq -1 then index = (spec_scan(spechandle, scan, $
                                  /return_index, /no_range,  $
                                  mcdata=state.mca, no_empty=(1-state.mca)))(0)
         if index(0) gt 0 then begin
            state.scanidx = index(0)
            dummyevent={NEW_SCAN_EVENT, ID:state.butid, TOP:0L, $
                        HANDLER:0L, FHANDLE:0L, $
			NEWSCAN:state.scanidx, MCA:state.mca}
            widget_control, state.butid, send_event=dummyevent
            widget_control, state.current, set_value='Current scan :' + $
                            string(spec_scan(spechandle, index, /index))
         endif else begin
            msg='Scan "'+scan+'" is not valid.'
            tmp = widget_message(DIALOG_PARENT=event.top,$
		msg, /error, dialog_parent=event.top)
         endelse
      end

      'CLOSE': 

      'DATATYPE': begin
         widget_control, event.id, get_value=aux
         state.mca = aux
         get_fileinfo, state, spechandle
      end
   endcase

   handle_value, state.handle, spechandle, /set, /no_copy
   widget_control, event.top, set_uvalue=state
   if evuval eq 'CLOSE' then widget_control, event.top, /destroy
end
;
;=========================================================================
;
function xplot_specscan, handle, butid, mca, group_leader=group,$
   no_block=no_block
   MainBase = widget_base(title='Scan Selection', mbar=menubar, $
                   column=1,uvalue=0)
      FileName = widget_label(MainBase)
      ScanCurrent = widget_label(MainBase)
      tmp0 = widget_base(MainBase, column=1, /frame)
         ScanList = widget_label(tmp0)
         tmp1 = widget_base(tmp0, row=1)
            ScanField = cw_field(tmp1, title='Scan id.', uvalue='ACCEPT', $
                                           xsize=5, /return_events)
            DataType = cw_bgroup(tmp1, ['Counters','MCA'],/exclusive, $
                               /row, set_value=mca, uvalue='DATATYPE', /frame)
         tmp1 = widget_base(tmp0, row=1)
            AcceptButt = widget_button(tmp1, value='Load', uvalue='ACCEPT')
            CloseButt = widget_button(tmp1, value='Close', uvalue='CLOSE')

   state={butid:butid, field:ScanField, file:FileName, current:ScanCurrent, $
          list:ScanList, handle:handle, scanidx:0, mca:mca}
   xplot_specscan_init, state, error
   if error eq 0 then begin
      widget_control, MainBase, /realize, set_uvalue=state
      if keyword_set(no_block) then $
      xmanager, 'xplot_specscan', MainBase, group_leader=group, /no_block else $ 
      xmanager, 'xplot_specscan', MainBase, group_leader=group
   endif else begin
      widget_control, MainBase, /destroy
   endelse
   return, MainBase
end
;
;=========================================================================
;
Function xplot_spectitle,title,spechandle,specscan
;
; Process a title for spec. It will substitute the characters
; #S, #D, etc by their values in the corresponding header.
;
if n_elements(specscan) LE 1 then specscan =spec_scan(spechandle, 'current')

; find number of # characters in the title

i=0
cnt=0
pos = -1
while (i ne -1) do begin
  i = strpos(title,'#',i)
  if (i ne -1) then begin 
    cnt = cnt+1
    i = i + 1
    if n_elements(pos) EQ 1 then begin
       if (pos EQ -1) then pos = i-1 else pos=[pos,i-1]
    endif else  pos=[pos,i]
  endif
endwhile 
if cnt EQ 0 then return,title

newtitle=title
for i=0,cnt-1 do begin
  s_from = strmid(title,pos(i),2)
  ; keyword ALL added srio@esrf.fr 97/08/29
  s_to = spec_headers(spechandle,specscan,s_from,/ALL)
  s_to = strmid(s_to,2,strlen(s_to)-2)
  newtitle=strsubstitute(newtitle,s_from, s_to)
endfor
return,newtitle
end
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_QUIT 
;	PURPOSE: Quits the Xplot window and destroy the internal handlers. 
;	CALLING SEQUENCE: XPLOT_QUIT,parent 
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_quit,wid

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  catch, /cancel
  if n_elements(state) NE 0 then widget_control, wid, SET_UVALUE = state, /NO_COPY
  return
endif
widget_control, wid, GET_UVALUE = state, /NO_COPY

widget_control,state.wids.draw,get_value=windownum
wset,windownum
xyouts,'!3'
widget_control,wid,/DESTROY
for i=0,n_elements(state.xpdt.sysvar)-1 do $
	tmp = execute(state.xpdt.sysvar_old(i))

;kill handles if window has not been cloned or is not a clone
if state.cloned EQ 0 then begin
  if keyword_set(state.dthbuffer) then $
    for i=0,n_elements(state.dthbuffer)-1 do $
      if handle_info(state.dthbuffer(i)) then $
      handle_free,state.dthbuffer(i)
  if keyword_set(state.dthcurrent) then $
    if handle_info(state.dthcurrent) then $
    handle_free,state.dthcurrent
  if handle_info(state.spec.handle) then $
    handle_free,state.spec.handle
endif
end ; xplot_quit

;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_CLEARDATA 
;	PURPOSE: clears graphs in xplot
;	CALLING SEQUENCE: XPLOT_CLEARDATA,parent 
;	INPUTS:
;	KEYWORD PARAMETERS:
;		CONFIRM = When set, display a confirmation window.
;	MODIFICATION HISTORY:
;		97-12-03 srio@esrf.fr initial version from xplot code
;
;-
PRO XPLOT_CLEARDATA,id,confirm=confirm

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  catch, /cancel
  goto,out
endif
;

widget_control, id , GET_UVALUE = state, /NO_COPY

if keyword_set(confirm) then begin
  tmp = widget_message(dialog_parent=id,$
    /CANCEL,/DEFAULT_CANCEL,$
    ['This options deletes all your Xplot graph and ',$
    'initialize command.','Please confirm:'])
  if tmp EQ 'Cancel' then goto,out
endif
;
;kill handles if window has not been cloned or is not a clone
;
if state.cloned EQ 0 then begin
  if keyword_set(state.dthbuffer) then $
    for i=0,n_elements(state.dthbuffer)-1 do $
    if handle_info(state.dthbuffer(i)) then $
    handle_free,state.dthbuffer(i)
  if keyword_set(state.dthcurrent) then $
    if handle_info(state.dthcurrent) then $
    handle_free,state.dthcurrent
  ;if handle_info(state.spec.handle) then $
  ;  handle_free,state.spec.handle
endif
;
; cleans the buffer
;
xpstr_update,state,/nobuffer,/nodthcurrent,/nodthbuffer,$
  /noexebuffer
;
; reset some valuse
;
state.xpdt.shift=[0.,0.]
state.xpdt.ErrPlot=[0,0]

tmpin = xplot_defaults('xpdt_titles')
tmpout = state.xpdt_titles
copy_structure,tmpin,tmpout
state.xpdt_titles = tmpout

tmpin = xplot_defaults('xpdt_limits')
tmpout = state.xpdt_limits
copy_structure,tmpin,tmpout

state.xpdt_limits = tmpout
tmpin = xplot_defaults('xpdt')
tmpout = state.xpdt
copy_structure,tmpin,tmpout
state.xpdt= tmpout

tmpin = xplot_defaults('strcontrols')
tmpout = state.strcontrols
copy_structure,tmpin,tmpout
state.strcontrols= tmpout

; 
; erase drawing window
;
erase

;
; reset labels
;
widget_control,state.wids.ncol,SET_VALUE= '0'
widget_control,state.wids.xcol,SET_VALUE= '0'
widget_control,state.wids.ycol,SET_VALUE= '0'
widget_control,state.wids.saved,SET_VALUE= '0'
;
; close control panel if open
;
if widget_info(state.wids.controls(0),/valid_id) then begin
  widget_control,state.wids.showcontrols, $
  set_value='Show Controls'
  widget_control,state.wids.controls(0),/DESTROY
endif
;
; set off spec
;
if state.spec.on then begin
  widget_control,state.spec.scanbase,/destroy, bad_id=bad
  state.spec.on = 0
  xplot_mappanels,state.wids,state.spec
endif
;
; reset message
;
widget_control,state.wids.message,set_value=$
'Xplot message: Xplot window and data cleaned up.'

out:
widget_control, id , SET_UVALUE = state, /NO_COPY
end ;xplot_cleardata

;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_SETLIMITS
;	PURPOSE: Sets the xrange and/or yrange to xplot.
;	CALLING SEQUENCE: XPLOT_SETLIMITS,parent
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named 
;		variable)
;
;	KEYWORD PARAMETERS:
;		XRANGE = [xmin,xmax] The extrema in X
;		YRANGE = [ymin,ymax] The extrema in Y
;		XSTYLE = The XSTYLE Graphic Keyword (see IDL doc)
;		YSTYLE = The YSTYLE Graphic Keyword (see IDL doc)
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	LIMITATIONS:
;		The XSTYLE and YSTYLE keywords sets this values to
;		the chosen values but do not update the "set limits..."
;		window.
;
;	MODIFICATION HISTORY:
;		97-12-03 srio@esrf.fr initial version from xplot code.
;
;-
PRO xplot_setlimits,id,xrange=xrange,yrange=yrange,xstyle=xstyle,$
	ystyle=ystyle,norefresh=norefresh

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  catch, /cancel
  goto,out
endif
interactive=1
if keyword_set(xrange) then interactive = 0
if keyword_set(yrange) then interactive = 0
if keyword_set(xstyle) then interactive = 0
if keyword_set(ystyle) then interactive = 0
;help,interactive
;
widget_control, id , GET_UVALUE = state, /NO_COPY
action=''
	if interactive  then begin
		tmp = state.xpdt_limits
		XscrMenu,tmp,/NOTYPE,GROUP=id,ACTION=action, $
		  WTITLE='Xplot limits',/INTERP,/APPLY,NCOL=2, TITLES=$
		  ['X min','X max','X style:','Y min','Y max','Y style'],$
		  help=xplot_text('LIMITS')
		if action eq 'DONT' then goto,out
		state.xpdt_limits = tmp
		state.xpdt.sysvar(2) = '!x.range = [' + $
			strcompress(state.xpdt_limits.Xmin,/rem)+','+ $
			strcompress(state.xpdt_limits.Xmax,/rem)+']'
		state.xpdt.sysvar(3) = '!y.range = [' + $
			strcompress(state.xpdt_limits.Ymin,/rem)+','+ $
			strcompress(state.xpdt_limits.Ymax,/rem)+']'
		tmp = fix( state.xpdt_limits.Xstyle(0) )
		case tmp of
		    0: state.xpdt.sysvar(4) = '!x.style = 1'
		    1: state.xpdt.sysvar(4) = '!x.style = 2'
		    2: state.xpdt.sysvar(4) = '!x.style = 4'
		    3: state.xpdt.sysvar(4) = '!x.style = 8'
		    4: state.xpdt.sysvar(4) = '!x.style = 16'
		endcase
		tmp = fix( state.xpdt_limits.Ystyle(0) )
		case tmp of
		    ;0: state.xpdt.sysvar(5) = '!y.style = 0'
		    0: state.xpdt.sysvar(5) = '!y.style = 1'
		    1: state.xpdt.sysvar(5) = '!y.style = 2'
		    2: state.xpdt.sysvar(5) = '!y.style = 4'
		    3: state.xpdt.sysvar(5) = '!y.style = 8'
		    4: state.xpdt.sysvar(5) = '!y.style = 16'
		endcase
	endif
	if keyword_set(xrange) then begin
		state.xpdt_limits.Xmin = xrange(0)
		state.xpdt_limits.Xmax = xrange(1)
		state.xpdt.sysvar(2) = '!x.range = [' + $
			strcompress(state.xpdt_limits.Xmin,/rem)+','+ $
			strcompress(state.xpdt_limits.Xmax,/rem)+']'
	endif
	if keyword_set(yrange) then begin
		state.xpdt_limits.Ymin = Yrange(0)
		state.xpdt_limits.Ymax = Yrange(1)
		state.xpdt.sysvar(3) = '!Y.range = [' + $
			strcompress(state.xpdt_limits.Ymin,/rem)+','+ $
			strcompress(state.xpdt_limits.Ymax,/rem)+']'
	endif
	if keyword_set(xstyle) then state.xpdt.sysvar(4)='!x.style = '+$
		strcompress(xstyle,/rem)
	if keyword_set(ystyle) then state.xpdt.sysvar(5)='!y.style = '+$
		strcompress(ystyle,/rem)
		state.xpdt.zoom_on=0
		if not(keyword_set(norefresh)) then Xplot_plot,state
                ; autocall xplot_event when Apply button is pressed
                if action eq 'DO_APPLY' then begin
                  widget_control, id, SET_UVALUE = state, /NO_COPY
                  xplot_event,event
                  return
                endif

out:
widget_control, id , SET_UVALUE = state, /NO_COPY
end ; xplot_setlimits
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_LOADFILE 
;	PURPOSE: load a file or variable in xplot
;	CALLING SEQUENCE: XPLOT_LOADFILE,parent [,input,spec=spec]
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;		INPUT  it can either be:
;		   1) a string with the ASCII file name
;		   2) an idl variable with a multi-column set (i.e.  fltarr(2,100) )
;	KEYWORD PARAMETERS:
;		SPEC set this keyword to a SPEC file name. In this case the
;		file is loaded and interpreted as SPEC file. IN this case, 
;		"INPUT" is not used.
;		PLOTNAME set this keyword to a string that identifies the 
;			plot (to be used by plotmgr).
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;		97-12-03 srio@esrf.fr adds plotname keyword.
;		98-02-17 srio@esrf.fr adds norefresh keyword. Xplot v 2.5.2
;
;-
PRO XPLOT_LOADFILE,id,input,spec=spec,plotname=plotname,norefresh=norefresh

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  if !err_string EQ $
   "Attempt to call undefined procedure/function: 'SPEC_ACCESS'." then begin
    spec_access_flag = -1
    message,/info,'Could not check for SPEC file. Input taken as standard' 
    spec_loaded=0 
  endif else spec_loaded=1
  if spec_loaded EQ 0 then goto,CONTINUE1
  catch, /cancel
  goto,out
endif
;
widget_control, id , GET_UVALUE = state, /NO_COPY

input_type = 'none'
state.plotname_current='<Variable>'
if n_params() GE 2 then begin
  if type(input) EQ 7 then begin
    file = input
    if checkfile(file) EQ 0 then begin
      itmp = widget_message(DIALOG_PARENT=state.wids.main,$
	'File not found: '+file,/error)
      goto,out
    endif else input_type = 'file'
    state.plotname_current=file
    state.fileinf.name=file
  endif  else begin
    input_type = 'variable'
    spec_access_flag = 0
  endelse
endif
if keyword_set(spec) then begin
  ntype = type(spec)
  if ntype EQ 7 then begin
    spec_access_flag=spec_access(hspec,spec) 
    state.plotname_current=spec
    state.fileinf.name=spec
  endif else begin
    hspec=spec
    spec_access_flag = spec_access(hspec)
    state.plotname_current='<SpecHandle>'
  endelse
  input_type = 'variable'
endif

		path = ''
if input_type EQ 'none' then begin
		if state.fileinf.dir EQ '' then cd,current=oldpath else $
		  oldpath=state.fileinf.dir 
		file = pickfile(GROUP=id,/NOCONF, /MUST_EXIST, $
                       TITLE='Select a data file',GET_PATH=path,PATH=oldpath)
		if file eq '' then GOTO,out
		input_type = 'file'
		state.plotname_current=file
endif
; commented by now
;		; Windows check *MSR 97/01/13*
;		if sdep() ne 'WINDOWS' then begin
;                  ; Check if ASCII
;                  spawn, 'file ' + file, aux, /sh
;		  ; add 'link' to allow soft links in SUN. MSR 27/11/95
;                  if ( (strpos(aux(0), 'text') eq -1) AND $ 
;                       (strpos(aux(0), 'link') eq -1) ) then begin
;                    tmp=widget_message('"'+file+'" is not an ASCII file.',/ERR)
;                    goto,OUT
;                  endif
;		endif
if input_type EQ 'file' then begin
                widget_control,/HOURGLASS
		state.fileinf.name=file
		state.fileinf.dir=path
                handle_value, state.spec.handle, hspec
		; If spec_access.pro is not somewhere in the directories
		; listed in !path, then suppose ASCII input.
		; This allows to run xplot without needing to install
		; spec_access.
		; MSR 97/02/10 This section is commented because if spec_acess 
		; is precompiled with xop.xdr, this option does not work. 
		;
		;if sdep() EQ 'WINDOWS' then tmpf='SPEC_A~1.DAT' else $
		;	tmpf='spec_access.pro'
		;if filepath1(tmpf) NE '' then $
		;  spec_access_flag = spec_access(hspec, file) else $
		;  spec_access_flag = -1
		spec_access_flag = spec_access(hspec, file) 
endif
CONTINUE1:
if input_type EQ 'file' or input_type EQ 'variable' then begin
                if spec_access_flag le 0 then begin  ; ASCII input
		   if input_type EQ 'file' then begin
		     tmp = rascii(file)
		     if n_elements(tmp) eq 1 then begin
		      tmp = widget_message(DIALOG_PARENT=state.wids.main,$
			'Error reading ASCII file.',/ERROR)
		      goto,OUT
		     endif
		   endif else begin
		     tmp = input
		     if n_elements(tmp) eq 1 then begin
		      tmp = widget_message(DIALOG_PARENT=state.wids.main,$
			'Wrong input to xplot.',/ERROR)
		      goto,OUT
		     endif
		   endelse
                   state.spec.on = 0
		   Ncol = n_elements(tmp(*,0))
		   state.xpdt.Xcol = 1
		   state.xpdt.Ycol = Ncol
		   if widget_info(state.wids.xcol,/NAME) EQ 'DROPLIST' $
		   then begin
		     coltitles = 'Col: '+$
			strcompress((sindgen(Ncol+1))(1:Ncol),/rem)
		     xpstr_update,state,coltitles=coltitles
		     widget_control,state.wids.xcol,set_value=coltitles,$
			set_droplist_select=0
		     widget_control,state.wids.ycol,set_value=coltitles,$
			set_droplist_select=ncol-1
		   endif else begin
		     widget_control,state.wids.xcol,SET_VALUE='1'
		     widget_control,state.wids.ycol,SET_VALUE=$
                                                     strcompress(Ncol,/REM)
		   endelse
                   widget_control,state.spec.scanbase,/destroy, bad_id=bad
		   state.xpdt.shift=[0.,0.]
		   state.xpdt.ErrPlot=[0,0]
		   widget_control,state.wids.ncol,SET_VALUE= $
			                              strcompress(Ncol,/REM)
                   xplot_mappanels, state.wids, state.spec
		   if input_type EQ 'file' then $
			widget_control,state.wids.message,set_value=$
			'Xplot message: ASCII data file ('+file+ $
			') read succesfully' else $
			widget_control,state.wids.message,set_value=$
			'Xplot message: ASCII data received.'
		endif else begin     ; spec input
                   mca = 0
                   ;fscan = (spec_scan(hspec, '-', /no_empty, /return_index))(0)
                   fscan = (spec_scan(hspec, 'LAST', /no_empty, /return_index))(0)
                   if fscan eq -1 then begin
                      ;fscan = (spec_scan(hspec, '-', /mcdata,/return_index))(0)
                      fscan = (spec_scan(hspec, 'LAST', /mcdata,/return_index))(0)
                      if fscan eq -1 then begin
		         tmp = $
			  widget_message(DIALOG_PARENT=state.wids.main,$
			  'No data found in SPEC file.',/ERROR)
		         goto,OUT
                      endif
                      mca = 1
                   endif
                   handle_value, state.spec.handle, hspec, /set
                   tmp = xplot_readspec(state, fscan, mca)
		   if n_elements(tmp) eq 1 then begin
		      goto,OUT
		   endif
                   state.spec.on = 1
		   widget_control,state.wids.message,set_value=$
			'Xplot message: SPEC data file ('+hspec.specfile+$
			') read succesfully'
                end 
endif
		if keyword_set(state.dthcurrent) then $
			handle_free,state.dthcurrent
		state.dthcurrent = handle_create(value=tmp)

                xplot_mappanels, state.wids, state.spec
		; reinitialize spec pannel
                widget_control,state.spec.scanbase,bad_id=tmp, send_event=$
                   {ID:state.spec.scanbase, TOP:0L, HANDLER:0L}
		; reset mesh flag
		;state.xpdt.mesh.flag(0)='0'
		if not(keyword_set(norefresh)) then xplot_plot,state
out:
if keyword_set(plotname) then state.plotname_current = plotname
widget_control, id , SET_UVALUE = state, /NO_COPY
end ;xplot_loadfile
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_SAVECURRENT 
;	PURPOSE: to save the current displayed plot in xplot (same
;		effect as pressing the "Save Current" button on the xplot
;		window)
;	CALLING SEQUENCE: XPLOT_SAVECURRENT,parent 
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_savecurrent,wid_top

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  catch, /cancel
  goto,out
endif

widget_control, wid_top , GET_UVALUE = state, /NO_COPY

if strcompress(state.xpdt.mesh.flag(0),/rem) EQ '1' then begin
  itmp = widget_message(dialog_parent=state.wids.main, /error,$
    'SAVE option not allowed when'+$
    ' View/Mesh option is selected. Not saved.')
  goto,out
endif
xplot_plot,state,/SAVE
widget_control,state.wids.saved,get_value=tmp
widget_control,state.wids.saved,set_value= $
strcompress(long(tmp)+1,/rem)
widget_control,state.wids.Message,set_value= $
'Xplot message: Current set saved. Use "Plot Mgr" button to'+$
' delete or move saved plots.'
widget_control,state.wids.plotmgr,/SENSITIVE
;commented to avoid bug when saving.
;state.xpdt.Kind = 2

out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_savecurrent
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_CHANGECOL 
;	PURPOSE: to change the X or Y displayed column in xplot
;		It works aso with SPEC files, including MCA. In the last
;		case the (X) column for Channels is ncol+1 and for Energy
;		is ncol+2, being ncol the number of channels.
;	CALLING SEQUENCE: XPLOT_CHANGECOL,parent ,XCOL=xcol, YCOL=ycol
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	KEYWORD PARAMETERS [USE AT LEAST ONE]:
;		XCOL The column index for X (starting from col 1, not 0)
;		YCOL The column index for Y (starting from col 1, not 0)
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_changecol,input,xcol=pxcol,ycol=pycol,norefresh=norefresh

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  catch, /cancel
  goto,out
endif

eventUVal=0
case type(input) of
8: begin
   event = input
   widget_control, event.top , GET_UVALUE = state, /NO_COPY
   Widget_Control, event.id, get_UValue=eventUVal
   wid_top=event.top
   end
3: begin
   widget_control, input, GET_UVALUE = state, /NO_COPY
   wid_top=input
   eventuval = ''
   if keyword_set(PXCOL) then begin
	eventUVal=[eventUVal,'XCOL']
	XCOL = PXCOL
   endif
   if keyword_set(PYCOL) then begin
	eventUVal=[eventUVal,'YCOL']
	YCOL = PYCOL
   endif
   my_value = 0
   if n_elements(eventuval) GE 2 then $
     eventuval=eventuval(1:n_elements(eventuval)-1)
   end
else: begin
   message,/info,'Usage: xplot_changecol, event (or parent_id)'
   return
   end
endcase

widget_control,state.wids.ncol,GET_VALUE=temp2
ncol = fix(temp2(0))

for i=0,n_elements(eventUVal)-1 do begin

  event_tmp = eventUVal(i)
  if type(input) EQ 3 then begin
     command = 'my_value = '+event_tmp
     itmp = execute(command)
  endif

  case event_tmp of
	'XCOL':	BEGIN
		if not(keyword_set(pxcol)) then begin
                  if state.spec.on then begin
                   if state.spec.mca then begin
                      widget_control,event.id,GET_VALUE=temp1
                      xcol = ncol + 1 + temp1
                   endif else begin
                      xcol = event.index + 1
                   endelse
                  endif else begin
                   if widget_info(state.wids.xcol,/NAME) EQ 'TEXT' then begin
			widget_control,event.id,GET_VALUE=temp1
		        xcol = fix(temp1(0))
		   endif else begin
		        xcol = fix(event.index)+1
		   endelse
                  endelse
		endif 
		if not state.spec.mca and xcol gt ncol then begin
		  tmp = widget_message(DIALOG_PARENT=state.wids.main,$
		  'Error selecting column.',/ERROR)
		  if not(keyword_set(PXCOL)) then $
                  widget_control,event.id,SET_VALUE= $
			strcompress(state.xpdt.Xcol,/rem)
		  goto,out
		endif else begin
                  state.xpdt.Xcol = xcol
		  if keyword_set(pxcol) then begin
                    if state.spec.on then begin
                     if state.spec.mca then begin
                        w_tmp = state.wids.xcolmca
                     endif else begin
                        w_tmp = state.wids.xcolspec
                     endelse
                    endif else w_tmp = state.wids.xcol
		    case widget_info(w_tmp,/NAME) of
		      'TEXT':widget_control,w_tmp,SET_VALUE=strcompress(xcol,/rem)
		      'DROPLIST': widget_control,w_tmp,set_droplist_select=xcol-1
		      'BASE': widget_control,w_tmp,set_value=xcol-1-ncol
		      else: goto,out
		    endcase
		  endif
		endelse
		END
	'YCOL':	BEGIN
		if not(keyword_set(pycol)) then begin
                  if state.spec.on and not state.spec.mca then begin
                   ycol = event.index + 1
                  endif else begin
                   if widget_info(state.wids.ycol,/NAME) EQ 'TEXT' OR $
		   (state.spec.mca EQ 1) then begin
			widget_control,event.id,GET_VALUE=temp1
		        ycol = fix(temp1(0))
		   endif else begin
		        ycol = fix(event.index)+1
		   endelse
                   ;widget_control,event.id,GET_VALUE=temp1
		   ;ycol = fix(temp1(0))
                  endelse
		endif
		if ycol gt ncol or ycol le 0 then begin
		  tmp = widget_message(DIALOG_PARENT=state.wids.main,$
		  'Error selecting column.',/ERROR)
		  if not(keyword_set(PYCOL)) then $
                  widget_control,event.id,SET_VALUE= $
			strcompress(state.xpdt.Ycol,/rem)
		  goto,out
		endif else begin
                  state.xpdt.Ycol = ycol
		  if keyword_set(pycol) then begin
                    if state.spec.on then begin
                     if state.spec.mca then begin
                        w_tmp = state.wids.ycolmca
                     endif else begin
                        w_tmp = state.wids.ycolspec
                     endelse
                    endif else w_tmp = state.wids.ycol
		    case widget_info(w_tmp,/NAME) of
		      'TEXT':widget_control,w_tmp,SET_VALUE=strcompress(ycol,/rem)
		      'DROPLIST': widget_control,w_tmp,set_droplist_select=ycol-1
		      else: goto,out
		    endcase
		  endif
		endelse
		END
  else:
  endcase
endfor
if not(keyword_set(norefresh)) then xplot_plot,state
out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_changecol
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_EXEBUFFER 
;	PURPOSE: to set a buffer or to add commands to the buffer
;		The EXEbuffer is a set of IDL commands that are executed
;		after the plot. It is used to make labels, annotations, etc.
;	CALLING SEQUENCE: XPLOT_EXEBUFFER,parent, ADDBUFFER=addbuffer, $
;		SETBUFFER=setbuffer 
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	KEYWORD PARAMETERS [MUST USE ONLY ONE (MUTUALLY EXCLUSIVE)]:
;		ADDBUFFER A string array with the commands to be added to the buffer
;
;		SETBUFFER A string array with the commands to be set to the buffer.
;			If the buffer contained commands, they are deleted before.
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_exebuffer,input,setbuffer=setbuffer,addbuffer=addbuffer,$
  norefresh=norefresh

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  catch, /cancel
  goto,out
endif

eventUVal=' '
case type(input) of
8: begin
   event = input
   widget_control, event.top , GET_UVALUE = state, /NO_COPY
   Widget_Control, event.id, get_UValue=eventUVal
   wid_top=event.top
   end
3: begin
   if keyword_set(setbuffer) then begin
     eventUVal = 'SETBUFFER'
     buffer = setbuffer
   endif
   if keyword_set(addbuffer) then begin
     eventUVal  = 'ADDBUFFER'
     buffer = addbuffer
   endif
   if buffer(0) EQ '' then begin
     message,/info,'Empty buffer'
     return
   endif
   if not(keyword_set(setbuffer)) AND not(keyword_set(addbuffer)) then begin
     message,/info,$
	'Usage: xplot_exebuffer,wid_top,setbuffer=setbuffer,addbuffer=addbuffer'
     message,/info,'       One keyword MUST be used'
     return
   endif
   widget_control, input, GET_UVALUE = state, /NO_COPY
   wid_top=input
   end
else: begin
   message,/info,'Usage: xplot_exebuffer, event (or parent_id)'
   return
   end
endcase

case eventUVal of
	'SETBUFFER': xpstr_update,state,EXEBUFFER=buffer
	'ADDBUFFER':	BEGIN
		tmp = state.exebuffer
		if tmp(0) NE '' then newbuffer=[tmp,buffer] else $
			newbuffer=buffer
                xpstr_update,state,EXEBUFFER=newbuffer
		END
	'EXEBUFFERCLEAN':	BEGIN
		tmp = widget_message(DIALOG_PARENT=wid_top, /CANCEL,$
		/DEFAULT_CANCEL, $
		  ['This option deletes all Legens, Labels',$
		  'and any other command stored in the buffer and',$
		  'present in the graph. Please comfirm'])
		if tmp EQ 'Cancel' then goto,out
		xpstr_update,state,/NOEXEBUFFER
		END
	'EXEBUFFEREDIT':	BEGIN
		tmp = state.exebuffer
		if type(tmp(0)) EQ 2 then begin
		  itmp = widget_message(DIALOG_PARENT=wid_top,/CANCEL,$
			/DEFAULT_CANCEL,['The buffer is empty. ',$
			'Do you want to create it?'])
		  if itmp EQ 'Cancel' then goto,out
		  tmp = $
		    '; please place below IDL commands to be executed after the plot'
		endif
		XDisplayFile1,text=tmp,group=wid_top,/MODAL,$
		  title='Editing EXE buffer',help=xplot_text('EXEBUFFEREDIT'),$
		  action=action
		if action EQ 'DONT' then goto,out
		if tmp(0) EQ '' then xpstr_update,state,/NOEXEBUFFER else $
			xpstr_update,state,EXEBUFFER=tmp
		xpstr_update,state,EXEBUFFER=tmp
		END
	'SETLABELS':	BEGIN
		buffer = 0
		xxyouts,BUFFER=buffer,GROUP=event.top
		if not(keyword_set(buffer)) then goto,out
		if keyword_set(state.exebuffer) then $
			exebuffer =  [state.exebuffer,buffer] else $
			exebuffer = buffer
		xpstr_update,state,EXEBUFFER=exebuffer
		END
	'SETLEGEND':	BEGIN
		command = 0
		xlegend,COMMAND=command,GROUP=event.top
		if command EQ '' then goto,out
		if keyword_set(state.exebuffer) then $
			buffer =  [state.exebuffer,command] else $
			buffer = command
		xpstr_update,state,EXEBUFFER=buffer
		END
	else:
ENDCASE


if not(keyword_set(norefresh)) then xplot_plot,state
out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_exebuffer
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_ERRORBARS 
;	PURPOSE: create error bars on a given plot
;	CALLING SEQUENCE: XPLOT_ERRORBARS,parent, col1, col2
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named 
;			variable)
;		COL1 the column containing the bottom limit of the error bars
;		COL2 the column containing the upper limit of the error bars
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_errorbars,input,col1,col2,norefresh=norefresh

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  catch, /cancel
  goto,out
endif

eventUVal=' '
case type(input) of
8: begin
   event = input
   widget_control, event.top , GET_UVALUE = state, /NO_COPY
   Widget_Control, event.id, get_UValue=eventUVal
   wid_top=event.top
   end
3: begin
   if n_params() NE 3 then begin
     message,/info,'Usage: xplot_errorbars,wid_top,col1,col2'
     return
   endif
   widget_control, input, GET_UVALUE = state, /NO_COPY
   wid_top=input
   end
else: begin
   message,/info,'Usage: xplot_errorbars, event (or parent_id)'
   return
   end
endcase


if eventuval EQ 'ERRPLOT' then BEGIN
		tmp = { Bottom:state.xpdt.ErrPlot(0), $
			Top:state.xpdt.ErrPlot(1) }
		XscrMenu,tmp,GROUP=event.top,/NOTYPE,ACTION=action, $
		  WTITLE='Xplot Error bar columns', Titles = $
		  ['Col with bottom data error:','Col with top data error:'],$
		  HELP=xplot_text('ERRORBARS')
		if action eq 'DONT' then goto,out
		  state.xpdt.ErrPlot = [tmp.bottom,tmp.top]
ENDIF ELSE state.xpdt.ErrPlot = [col1,col2]
if not(keyword_set(norefresh)) then xplot_plot,state
out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_errorbars
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_ADDCOL 
;	PURPOSE: load a numerical array and include it as new column in
;		Xplot.
;	CALLING SEQUENCE: XPLOT_ADDCOL,parent,input
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named 
;		variable)
;
;		INPUT  an array with the new column.
;
;	KEYWORD PARAMETERS:
;
;	MODIFICATION HISTORY:
;		97-10-21 srio@esrf.fr initial version slightly modifying
;			the old xplot_addcol.
;
;-
PRO xplot_addcol,id,yfit,norefresh=norefresh

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  catch, /cancel
  goto,out
endif
;
widget_control, id , GET_UVALUE = state, /NO_COPY
			plotfile,state.dthcurrent,tmpset,/NODATA
			ncol = n_elements(tmpset(*,0))
			npts = n_elements(tmpset(0,*))
			newset = fltarr(ncol+1,npts)
nyfit=n_elements(yfit)
if nyfit NE npts then begin
  itmp = widget_message(/QUESTION,dialog_parent=id,$
	['Error in xplot_addcol: The column to be added has a different',$
	'number of points of the existing columns.',$
	'Resize and continue?'])
  if itmp EQ 'No' then goto,out
  if nyfit GT npts then yfit = reform(yfit(0:npts-1)) else begin
     yfitnew = fltarr(npts)
     yfitnew(0:nyfit-1) = yfit
     yfit = reform(yfitnew)
  endelse
endif

			; if spec/mca insert instead appending...
			if state.spec.on AND state.spec.mca then begin
			  ycol = ncol-1
			  newset(0:ycol-2,*) = tmpset(0:ycol-2,*)
			  newset(ycol-1,*) = yfit
			  newset(ycol:ncol,*) = tmpset(ncol-2:ncol-1,*)
			  ncol = ncol-2
			  ncoladd='+2'
			endif else begin
			  newset(0:ncol-1,*) = tmpset
			  newset(ncol,*) = yfit
			  ncoladd=''
			  ycol = ncol+1
			endelse

			widget_control,state.wids.ncol,set_value = $
			  strcompress(ncol+1,/rem)+ncoladd

			if widget_info(state.wids.xcol,/NAME) EQ 'TEXT' $
			then begin
			  widget_control,state.wids.ycol,set_value = $
			  strcompress(ycol,/rem)
			endif else begin
			  tmp  = [state.coltitles, $
			   strcompress(n_elements(state.coltitles)+1, $
			   /rem)+': Fit']
			  xpstr_update,state,coltitles=tmp
			  widget_control,state.wids.xcol,set_value=tmp
			  widget_control,state.wids.ycol,set_value=tmp, $
			    set_droplist_select=ycol-1
			endelse


			state.xpdt.Ycol = ycol
			handle_free,state.dthcurrent
			h1 = handle_create(value=newset)
			state.dthcurrent = h1

			; spec setup
			if state.spec.on EQ 1 then begin
			  labels = state.spec.labels+'!Fit'
			  state.spec.labels = labels
			  tmp = strparse(labels,'!',labelslist)
			  labelslist=strcompress((sindgen(n_elements( $
			    labelslist)+1))(1:n_elements(labelslist)) ) + $
			    ': '+labelslist
			  if tmp EQ -1 then begin
			    tmp = widget_message(dialog_parent=state.wids.main,$
				'Error (column separator)',/ERROR)
			    return
			  endif
			  if state.spec.mca then begin
      			    widget_control,state.wids.ycolMCA,SET_VALUE=$
				strcompress(ycol,/rem)
			  endif else begin
      			    widget_control,state.wids.xcolSPEC,SET_VALUE=$
				labelslist
      			    widget_control,state.wids.ycolSPEC,SET_VALUE=$
				labelslist, SET_DROPLIST_SELECT = ncol
			  endelse
			endif


			widget_control,state.wids.message,set_value = $
			  'Xplot message: Result of the fit has been placed'+$
			  ' in column '+strcompress(ycol,/rem)
			if not(keyword_set(norefresh)) then xplot_plot,state

out:
widget_control, id , SET_UVALUE = state, /NO_COPY
end ; xplot_addcol
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_PLOTMGR
;	PURPOSE: manage the saved set (delete or move to current).
;	CALLING SEQUENCE: XPLOT_PLOTMGR,parent [,selected ]
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;		SELECTED = the saved set number (starting from 1) to be
;		edited (moved or deleted).
;	KEYWORD PARAMETERS 
;		[THE FOLLOWING ONES ARE EXCLUSIVE. AT LEAST ONE IS MANDATORY]:
;		DELETE_SEL will delete the selected saved set.
;		DELETE_ALL will delete all the saved sets.
;		MOVE will move the selected saved set to the current set.
;		[GENERAL KWYWORDS]:
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	MODIFICATION HISTORY:
;		97-10-24 srio@esrf.fr initial version from xplot code
;
;-
;=========================================================================
PRO xplot_plotmgr_win_event,event

widget_control,event.id,get_uvalue=uval
widget_control,event.top,get_uvalue=state,/no_copy

case uval of
 'CANCEL': begin
	widget_control,state.wids.action,get_uval=tmp
	tmp.action='CANCEL'
	widget_control,state.wids.action,set_uval=tmp
	widget_control,event.top,/destroy
	return
	end
 'ACCEPT':begin
	widget_control,state.wids.action,get_uval=actionstr
	widget_control,state.wids.buttons,get_value=tmp
	case tmp of
	  0:begin
		if state.selected EQ 0 then goto,accept_error
		actionstr.action='DELETE_SEL'
		end
	  1:actionstr.action='DELETE_CUR'
	  2:actionstr.action='DELETE_ALL'
	  3:begin
		if state.selected EQ 0 then goto,accept_error
		actionstr.action='MOVE'
		end
	  else:actionstr.action='CANCEL'
	endcase
	actionstr.sel=state.selected
	widget_control,state.wids.action,set_uval=actionstr
	widget_control,event.top,/destroy
	return
	accept_error:
		itmp=widget_message(/ERROR,'Plase select a set',$
			dialog_parent=event.top)
	end
 'SELECT':begin
	state.selected=event.index+1
	end
 'HELP':begin
	  infotext = [' ',$
	   'The Plot Manager permits to modify the saved data sets in Xplot.', $
	   ' ',$
	   'It will either delete set(s) or promote a saved set to the ',$
	   'current set.',$
	   ' ',$
	   'The sets are named either with the file name from which the ',$
	   'data was loaded, or with a name given by the user (with',$
	   'Edit/PlotNames Menu). In addition, there is a unique identifier ',$
	   'for each set (The id number).']
	XDisplayFile1,text=infotext,group=event.top,title=$
	'Help on Plot Manager'
	end
  else:
endcase

out:
widget_control,event.top,set_uvalue=state,/no_copy
end ; xplot_plotmgr_win_event
;=========================================================================
PRO xplot_plotmgr_win,buffer,current,$
	group=group,action=action,selected=selected
;
;
wbase = widget_base(title='Plot Manager',/column) ;,GROUP=group,/MODAL)
wAction = widget_base(UVALUE={action:'CANCEL',sel:0}) ; to store the action

wbase0 = widget_base(wbase,/row)
 wtmp = widget_button(wbase0,VALUE='Cancel',UVALUE='CANCEL')
 wtmp = widget_button(wbase0,VALUE='Accept',UVALUE='ACCEPT')
 wtmp = widget_button(wbase0,VALUE='Help',UVALUE='HELP')

wbase1 = widget_base(wbase,/col,/frame)
wtmp = widget_label(wbase1,VALUE='The following sets are saved:')
wlist = widget_list(wbase1,VALUE=buffer,UVALUE='SELECT',XSIZE=30,$
  YSIZE=n_elements(buffer)<6)
wtmp = widget_label(wbase1,VALUE=strcompress(current,/rem)+$
	' is the current set.',/ALIGN_LEFT)
;wtmp = widget_TEXT(wbase1,VALUE=current,XSIZE=30)

wbase2 = widget_base(wbase,/COL,/frame)
wtmp = widget_label(wbase2,VALUE='Actions: ')
wbuttons = cw_bgroup(wbase2,['Delete selected set',$
'Delete current set','Delete all saved sets',$
'Move selected set to current set'],$
UVALUE='',$
/EXCLUSIVE,/NO_RELEASE,SET_VALUE=0,COLUMN=2)

wids = {buttons:wbuttons, action:wAction}
state={ wids:wids, selected:0L}
widget_control,wbase,set_uvalue=state
widget_control,wbase,/REALIZE
xmanager,'xplot_plotmgr_win',wbase,GROUP_LEADER=group,/modal
;
; retrieve the result
straction=0
widget_control,wAction,get_uvalue=straction
widget_control,wAction,/destroy
action=straction.action
selected=straction.sel
end ; xplot_plotmgr_win
;=========================================================================
;
pro xplot_plotmgr,id,nn_sel,DELETE_SEL=delete_sel,DELETE_ALL=delete_all,$
	MOVE=move,NOREFRESH=norefresh
;
catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  catch, /cancel
  goto,out
endif

interactive=1
if keyword_set(DELETE_SEL) then begin
  if n_elements(nn_sel) EQ 0 then begin
     message,/info,'Usage: xplot_plotmgr,id,nn_sel,/DELETE_SEL'
     return
  endif
  action='DELETE_SEL'
  interactive=0
endif
if keyword_set(MOVE) then begin
  if n_elements(nn_sel) EQ 0 then begin
     message,/info,'Usage: xplot_plotmgr,id,nn_sel,/MOVE'
     return
  endif
  action='MOVE'
  interactive=0
endif
if keyword_set(DELETE_ALL) then begin
  action='DELETE_ALL'
  interactive=0
endif
;
widget_control, id , GET_UVALUE = state, /NO_COPY

;if n_elements(nn_sel) EQ 0 then begin
if interactive then begin
  action='' & nn_sel=0

  buffer = strcompress(state.plotname_buffer,/rem)
  buffer = buffer+'  (id: '+strcompress(state.dthbuffer,/rem)+')'
  current = state.plotname_current+$
	' (id: '+strcompress(state.dthcurrent,/rem)+')'

  xplot_plotmgr_win,buffer,current,Group=id,action=action,$
	selected=nn_sel
  ;if action EQ 'CANCEL' then goto,OUT
  ;if  nn_sel GT n_elements(state.buffer) then begin
  ;    tmp = widget_message(dialog_parent=event.top,$
  ;     'There are not so many saved sets.',/ERROR)
  ;     goto,out
  ;endif
endif
if action EQ 'DELETE_CUR' then begin
  action='MOVE'
  nn_sel = n_elements(state.buffer)
endif

case action of 
  'CANCEL': goto,OUT
  'DELETE_SEL': begin
		;
		; removing from buffer
		;
		hh = state.dthbuffer(nn_sel-1)
		nsaved = n_elements(state.buffer)
		if nsaved EQ 1 then begin
		  xpstr_update,state,/nobuffer,/nodthbuffer,/noplotname_buffer
		endif else begin
		  index = lindgen(nsaved)
		  newindex = where( index NE nn_sel-1)
		  buffer = state.buffer( newindex)
		  dthbuffer = state.dthbuffer( newindex )
		  plotname_buffer = state.plotname_buffer( newindex )
		  xpstr_update,state,buffer = buffer, $
			dthbuffer = dthbuffer,plotname_buffer = plotname_buffer
		endelse
		if n_elements(nsaved) EQ 1 then $  ; now zero
			widget_control,state.wids.plotmgr,sensitive=1
		widget_control,state.wids.saved, set_value= $
			strcompress(nsaved-1,/rem)
		handle_free,hh
	end
  'DELETE_ALL': begin
		for i=0,n_elements(state.buffer)-1 do $
		 handle_free,state.dthbuffer(i)
		xpstr_update,state,/nobuffer,/nodthbuffer,/noplotname_buffer
		widget_control,state.wids.plotmgr,sensitive=0
		widget_control,state.wids.saved, set_value='0'
	end
  'MOVE': begin
		;print,'<><> selected buffer: ',state.buffer(long(outtext)-1)
		;nlist = strparse(state.buffer(long(outtext)-1),',',list)
		handle_free,state.dthcurrent
		nlist = strparse(state.buffer(nn_sel-1),',',list)
		; 
		; execute buffer
		;
		;print,'<><> executing : ','hh = '+list(1)
		tmp = execute( 'hh = '+list(1))
		for i=2,7 do begin
		;print,'<><> executing : ',list(i)
			tmp = execute( list(i) )
		endfor
		;print,'<><> executing : ',list(8)+','+list(9)
		tmp = execute( list(8)+','+list(9) )
		;
		; saving values
		;
		state.xpdt.xcol = xcol
		state.xpdt.ycol = ycol
		state.xpdt.psym = psym
		state.xpdt.linestyle = linestyle
		state.xpdt.thick = thick
		state.xpdt.sysvar(7) = '!p.color = '+strcompress(color,/rem)
		state.xpdt.shift = shiftval

		;
		; removing from buffer
		;
		nsaved = n_elements(state.buffer)
		;help,nsaved
		if nsaved EQ 1 then begin
		  xpstr_update,state,/nobuffer,/nodthbuffer,/noplotname_buffer,$
			dthcurrent = hh 
		endif else begin
		  index = lindgen(nsaved)
		  newindex = where( index NE nn_sel-1)
		  buffer = state.buffer( newindex)
		  dthbuffer = state.dthbuffer( newindex )
		  plotname_buffer = state.plotname_buffer( newindex )
		  xpstr_update,state,buffer = buffer, $
			dthbuffer = dthbuffer, dthcurrent = hh, $
			plotname_buffer = plotname_buffer
		endelse
		;
		; close control panel and update its values
		;
                if widget_info(state.wids.controls(0),/valid_id) then begin
                     widget_control,state.wids.showcontrols, $
				set_value='Show Controls'
                     widget_control,state.wids.controls(0),/DESTROY
                endif
		state.strcontrols.symbol = psym
		state.strcontrols.linestyle = linestyle
		state.strcontrols.thickness = thick
		state.strcontrols.color = color
		;
		; updating widgets
		;
		if nsaved EQ 1 then $
			widget_control,state.wids.plotmgr,sensitive=0
		plotfile,hh,tmp,/nodata
		ncol = n_elements(tmp(*,0))
		
		if widget_info(state.wids.xcol,/NAME) EQ 'DROPLIST' then begin
		  coltitles = (sindgen(Ncol+1))(1:Ncol)
		  widget_control,state.wids.xcol, set_value = coltitles, $
			set_droplist_select = xcol-1
		  widget_control,state.wids.ycol, set_value = coltitles, $
			set_droplist_select = ycol-1
		endif else begin
		  widget_control,state.wids.xcol, set_value = $
			strcompress(xcol,/rem)
		  widget_control,state.wids.ycol, set_value = $
			strcompress(ycol,/rem)
		endelse
		widget_control,state.wids.ncol, set_value= $
			strcompress(ncol,/rem)
		widget_control,state.wids.saved, set_value= $
			strcompress(nsaved-1,/rem)
		widget_control,state.wids.message, set_value= $
			'Xplot message:'
		; switch out spec (if selected)
		if state.spec.on EQ 1 then begin
		  widget_control,state.spec.scanbase,/destroy, bad_id=bad
		  state.spec.on = 0
		  xplot_mappanels, state.wids, state.spec
		endif
	end
endcase
; re-plot
if not(keyword_set(norefresh)) then xplot_plot,state

out:
widget_control, id , SET_UVALUE = state, /NO_COPY
end ;xplot_plotmgr
;
;============== END OF ROUTINES FOR NON-INTERACTIVE CALLS. ===================
;
;
;=========================================================================
;
pro xplot_mappanels, wids, spec
;
; Maps the column selection panels.
;
   if spec.on then begin
      widget_control,wids.asciipanel,map=0
      widget_control,wids.specpanel,map=1
      if spec.mca then begin
         widget_control,wids.normalpanel,map=0
         widget_control,wids.mcapanel,map=1
      endif else begin
         widget_control,wids.mcapanel,map=0
         widget_control,wids.normalpanel,map=1
      endelse
   endif else begin
      widget_control,wids.asciipanel,map=1
      widget_control,wids.specpanel,map=0
   endelse
end
;
;=========================================================================
;
FUNCTION xplot_fitpolgauss,set,OUTTEXT=outtext,EXTRAP=extrap,OPLOT=oplot,$
  GROUP=GROUP
;
; drives the Calculations/Fitting/Polynomial/Gaussian option
; returns an array with the fitted point coordinates
;
; outtext: information text
; oplot: if set, overplots the fitting value. If set to a numerical 
;	value, tthen the number of overplotted points is set to its value.
; extrap: the resulting returned value is calculated from the 
;	abscissas points extrap(0,*)
;
on_error,2

back:
widget_control,sensitive=0
itmp = wmenu2([' 1) Chi-Square Linear Regression', $
	' 2) Robust Estimation Linear Regrassion', $
	' 3) Least Square Polynomial Fit', $
	' 4) Singular Value Decomposition Polynomial Fit', $
	' 5) F(x) = A0*EXP(-z^2/2) + A3 with  z=(x-A1)/A2', $
	' 6) F(x) = A0*EXP(-z^2/2) + A3 + A4*x + A5*x^2 with z=(x-A1)/A2', $
	' 7) Help',$
	' 8) Cancel'])
widget_control,sensitive=1


setx = reform(set(0,*))
sety = reform(set(1,*))

case  itmp  of
  0:BEGIN
	sdev = Replicate(1.0,n_elements(setx))
	Result = LinFit(setx,sety,CHISQR=chisqr,PROB=prob,SDEV=sdev,$
		SIG_AB=sig_ab)
	Equation = 'x*Result(1)+Result(0)'
	outtext = [ $
	'   Linear Regression (LINFIT) results ', $
	'   Fit result : yfit = A + Bx ', $
	'             A: '+strcompress(Result(0),/rem), $
	'             B: '+strcompress(Result(1),/rem), $
	'        ChiSqr: '+strcompress(ChiSqr,/rem), $
	'          Prob: '+strcompress(Prob,/rem), $
	'      Sigma(A): '+strcompress(sig_ab(0),/rem), $
	'      Sigma(B): '+strcompress(sig_ab(1),/rem)]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
        yfit = Result(1)*setx+Result(0)
    END
  1:BEGIN
	Result = LadFit(setx,sety)
	Equation = 'x*Result(1)+Result(0)'
	outtext = [ $
	'   Linear Regression (LADFIT) results ', $
	'   Fit result : yfit = A + Bx ', $
	'             A: '+strcompress(Result(0),/rem), $
	'             B: '+strcompress(Result(1),/rem)]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
        yfit = Result(1)*setx+Result(0)
    END
  2:BEGIN
	strtmp = { n:1 }
	XscrMenu,strtmp,/NoType,Titles='Polynomial Degree',Action=action
	W = Replicate(1.0,n_elements(setx))
	if action ne 'DO' then return,0
	deg = strtmp.n
	Result = PolyFitW(setx,sety,W,strtmp.n,yfit,yband,sigma,a)
	Equation = ' X^'+strcompress(deg,/rem) + $
		'* Result('+strcompress(deg,/rem)+ ')' 
	for i=deg-1,0,-1 do Equation = Equation+ $
		' + X^'+strcompress(i,/rem)+ $
		' * Result('+strcompress(i,/rem)+')'
        atext = strarr(strtmp.n+1)
	for i=0,strtmp.n do atext(i)=vect2string(a(*,i))
	outtext = [ $
        '   Polynomial fit (POLYFITW) Results',$
	'   Fit coefficients  : '+vect2string(result),$
	'   Standard Deviation: '+strcompress(sigma,/rem),$
	'   Correlation matrix: ',atext]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
  3:BEGIN
        strtmp = { n:1 }
        XscrMenu,strtmp,/NoType,Titles='Polynomial Degree',Action=action
        W = Replicate(1.0,n_elements(setx))
	Yfit = W & ChiSq = 0  &  Covar = 0  &  Variance = 0
        if action ne 'DO' then return,0
        Result = SVDFit(setx,sety,strtmp.n+1,WEIGHT=W, YFIT=yfit, $
	ChiSq=ChiSq,COVAR=covar,VARIANCE=variance)
	deg = strtmp.n
	Equation = ' X^'+strcompress(deg,/rem) + $
		'* Result('+strcompress(deg,/rem)+ ')' 
	for i=deg-1,0,-1 do Equation = Equation+ $
		' + X^'+strcompress(i,/rem)+ $
		' * Result('+strcompress(i,/rem)+')'
        atext = strarr(strtmp.n+1)
        for i=0,strtmp.n do atext(i)=vect2string(covar(*,i))
        outtext = [ $
        '   Polynomial fit (SVDFIT) Results',$
        '   Fit coefficients  : '+vect2string(result),$
        '             Variance: '+vect2string(Variance),$
        '               ChiSq : '+strcompress(ChiSq,/rem),$
        '   Correlation matrix: ',atext]

        for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
  4:BEGIN
	yfit  = 0
	yfit = Gauss_Fit(setx,sety,Result)
	Equation = 'Result(0)*EXP(-((x-Result(1))/Result(2))^2/2) + Result(3)'
	outtext = [ $
	'   Gaussina Fit (GAUSS_FIT) results ', $
	'   Fit result :  F(x) = A0*EXP(-z^2/2) + A3, z=(x-A1)/A2', $
	'            A0: '+strcompress(Result(0),/rem), $
	'            A1: '+strcompress(Result(1),/rem), $
	'            A2: '+strcompress(Result(2),/rem), $
	'            A3: '+strcompress(Result(3),/rem)]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
  5:BEGIN
	yfit  = 0
	yfit = GaussFit(setx,sety,Result)
	Equation = 'Result(0)*EXP(-((x-Result(1))/Result(2))^2/2) '+ $
		' + Result(3) + x*Result(4) + x^2 * Result(5)'
	outtext = [ $
	'   Gaussina Fit (GAUSS_FIT) results ', $
	'   Fit result :  F(x) = A0*EXP(-z^2/2) + A3 + A4*x + A5*x^2', $
	'                 z=(x-A1)/A2', $
	'            A0: '+strcompress(Result(0),/rem), $
	'            A1: '+strcompress(Result(1),/rem), $
	'            A2: '+strcompress(Result(2),/rem), $
	'            A3: '+strcompress(Result(3),/rem), $
	'            A4: '+strcompress(Result(4),/rem), $
	'            A5: '+strcompress(Result(5),/rem)]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
  '6': begin
	XDisplayFile1,TEXT=xplot_text('FITPOLGAUSS'),GROUP=group,$
	TITLE='Help on gaussian and polynomial fit'
	goto,back
	end
  '7': return,0
  else:
endcase

if keyword_set(oplot) then begin
  if oplot EQ 1 then x=setx else x = dindgen(oplot)/(oplot-1)* $
    (setx(n_elements(setx)-1)-setx(0)) + setx(0)
  yyfit = 0
  command = 'yyfit = '+Equation
  tmp = execute(command)
  oplot,x,yyfit,psym=-4
endif

if keyword_set(extrap) then begin
  x=double(extrap(0,*)) 
  yfit = 0
  command = 'yfit = '+Equation
  tmp = execute(command)
;help,yfit
endif

outtext = [outtext, ' ', $
  'Minimum y value for the fit: '+strcompress(min(yfit,max=yfitmax),/rem), $
  'Maximum y value for the fit: '+strcompress(yfitmax,/rem)]
return,yfit
end ; xplot_fitpolgauss
;
;=========================================================================
;
PRO xplot_columnoper,set,str=str,group=group, resultcololumn = resultcolumn
;
; drives the Calculations/Operations with columns option
;
;
;
; Inputs: set: data set to be oprated.
;	  str: structure with the window data.
;	  group: id of parent widget.
; Outputs: set: modified data set
;	   resultcolumn (keyword): modified column number. Returns -1
;	   if data is sent to a new Xplot window.
;
; Allow operations between columns. The resulting set overwrites
; the input set.
; MSR/95/08/29
;
;
on_error,2
;
; info window
;
col1 = 0  &  col2 = 0  &  col3 = 0  &  col4 = 0  &  col5 = 0
col6 = 0  &  col7 = 0  &  col8 = 0  &  col9 = 0  &  col10 = 0
col11 = 0  &  col12 = 0  &  col13 = 0  &  col14 = 0  &  col15 = 0
col16 = 0  &  col17 = 0  &  col18 = 0  &  col19 = 0  &  col20 = 0

ncol = n_elements(set(*,0))
if ncol gt 20 then begin
  txt = ['Operation with columns are only allowed', $
	'with set holding up to 20 columns.', $
	'The current set has '+strcompress(ncol,/rem)+' columns.']
  tmp = widget_message(dialog_parent=group,txt,/ERROR)
  return
end
;
; to evaluate an expression [Calculations Menu]
;
if not(keyword_set(str)) then str = xplot_defaults('XPcolumnoper')
;
flags=['1','1','w(1) EQ 1']
titles=['Expression for new column: ', 'Result to: ','Column to overwrite: ']

str_old = str
XscrMenu,str,/NOTYPE,/INTERP,FLAGS=flags,TITLES=titles, $
  ACTION=action,WTITLE='Operation with columns',GROUP=group,$
  HELP=xplot_text('COLUMNOPER')

if action eq 'DONT' then begin
  str = str_old
  return
endif
;
; start operations
;
; load data 
;
for i=1,ncol do begin
   command = 'col'+strcompress(i,/rem)+' = set('+$
	strcompress(i-1,/rem)+',*) '
   ;print,command
   tmp = execute(command)
endfor

  if not(keyword_set(set)) then begin
    ;xtext,TEXT=['Empty data set','Returning'],GROUP=group
    tmp = widget_message(dialog_parent=group,$
	['Empty data set','Returning'],/ERROR)
    str = str_old
    return
  endif

;
; evaluate the expression
;
command = 'result = '+str.command
tmp = execute(command)
if tmp ne 1 then begin
  tmp = widget_message(dialog_parent=group,$
	'Operation expression not understood',/ERROR)
  str = str_old
  return
endif   
;
; rearrange the result
;
;if strcompress(str.to(0),/rem) EQ '0' then begin
case  strcompress(str.to(0),/rem) of
  '0':BEGIN
      resultcolumn = ncol+1
      newset = fltarr(ncol+1,n_elements(col1))
      newset(0:ncol-1,*) = set
      command = 'newset('+strcompress(ncol,/rem)+',*) = result'
      ;print,command
      tmp = execute(command)
      END
  '1':BEGIN
      newset = set
      resultcolumn = fix(str.outcol)
      command = 'newset('+strcompress(str.outcol-1,/rem)+',*) = result'
      ;print,command
      tmp = execute(command)
      END
  '2':BEGIN
      newset = set
      xplot,col1,result,wtitle= $
	"Xplot 'Calculations/Operations with sets' results"
      END
  else:
endcase
;
;
set = newset
;
end
;
;=========================================================================
;
PRO XPLOT_CALCULATIONS,event
;
; Event handler whefor the Calculations menu (Except operations with
; columns)
; State is passed from the event handler because it has been extracted
; by using /NO_COPY
; state is not passed MSR 97/09/08
; added common block to remember the user inputs (smooth).
;
;
common xplot_calculations, smoothstr, interstr
catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  itmp=Dialog_Message(/Error,Dialog_Parent=event.top,$
    'XPLOT_CALCULATIONS: error catched '+!err_string)
  catch, /cancel
  if n_elements(state) NE 0 then goto,out
  return
endif
widget_control, event.top , GET_UVALUE = state, /NO_COPY
widget_control, event.id, get_value=eventval
if not(keyword_set(state.dthcurrent)) then begin
  ;it is allowed to call Operations with sets... without
  ;data!!
  if eventval EQ 'Operations with sets...' then goto,EVALUATE

  tmp = widget_message(DIALOG_PARENT=event.top,$
	' No data in current set')
  goto,OUT
endif

plotfile,state.dthcurrent,/nodata,tmpset_pre
tmpset1 = tmpset_pre([state.xpdt.xcol-1,state.xpdt.ycol-1],*)
xmin = state.xpdt_limits.xmin
xmax = state.xpdt_limits.xmax
if (xmin NE xmax) then begin
  cutset,tmpset1,tmpset,xrange=[xmin,xmax]
  ttext = $
  'Calculations have been performed in'+$
  ' the X interval ('+strcompress(xmin,/rem)+','+ $
  strcompress(xmax,/rem)+')'
endif else begin
  tmpset = tmpset1
  ttext = ''
endelse
widget_control,state.wids.message,set_value= $
'Xplot message: '+ttext
                
if n_elements(tmpset EQ 1) then plotfile,tmpset,tmpset
CASE eventval OF
 'Width/Integral/MinMax':	begin
	nn_val = n_elements(tmpset(0,*))
	fwhm_val = GETFWHM(tmpset,/PLOT)
	;int_val = INTEGSET(tmpset)
	int_val = INT_TABULATED(tmpset(0,*),tmpset(1,*))
	max_val = max(tmpset(1,*),max_i)
	min_val = min(tmpset(1,*),min_i)
	max_val_x = max(tmpset(0,*),max_x_i)
	min_val_x = min(tmpset(0,*),min_x_i)
	text= [ ttext,' ', ' ',$
	'number of points:    '+strcompress(nn_val,/RE), $
	'FWHM value is:    '+strcompress(fwhm_val,/RE), $
	'Integral is  :    '+strcompress(int_val,/RE), $
	' ',$
	'Maximum y (Peak) at point '+strcompress(1+max_i,/re)+' :  ('+$
	  strcompress(tmpset(0,max_i),/RE)+','+strcompress(max_val,/RE)+')', $
	'Minimum y at point '+strcompress(1+min_i,/RE)+' : ('+ $
	  strcompress(tmpset(0,min_i),/RE)+','+strcompress(min_val,/RE)+')', $
	'Maximum x is at point '+strcompress(1+max_x_i,/RE)+' : ('+ $
	  strcompress(tmpset(0,max_x_i),/RE)+','+strcompress(max_val_x,/RE)+ $
	')',	$
	 'Minimum x is at point '+strcompress(1+min_x_i,/RE)+' : ('+ $
	   strcompress(tmpset(0,min_x_i),/RE)+','+strcompress(min_val_x,/RE)+$
	')', $
	' ',' ', $
	'Do you want to write few of these values in the plot legend? ']

	tmp = widget_message(text,DIALOG_PARENT=event.top,$
		/INFO,/CANCEL,/DEFAULT_CANCEL)
	if tmp EQ 'Cancel' then goto,out
	legendtxt = 'legend,'+ $
	  '['+"'"+'FWHM value is: '+strcompress(fwhm_val,/RE)+"',"+ $
	  "'"+'Integral is  : '+strcompress(int_val,/RE)+"',"+ $
	  "'"+'Peak at  : ('+strcompress(tmpset(0,max_i),/RE)+','+$
	  strcompress(max_val,/RE)+')'+"']"
	if keyword_set(state.exebuffer) then $
		exebuffer  = [state.exebuffer,legendtxt] else $
		exebuffer  = legendtxt
	xpstr_update,state,EXEBUFFER=exebuffer
	xplot_plot,state
	end
 'Moments':        begin
	npoints = n_elements(tmpset(0,*))
	xmom = moment(tmpset(0,*),mdev=xmdev,sdev=xsdev)
	ymom = moment(tmpset(1,*),mdev=ymdev,sdev=ysdev)
	text= [ ttext,' ', ' ',$
	  'Number of data points: '+strcompress(npoints,/rem), $
	  '  ', $
	  'Values for the abscissas values are: ', $
	  '    Mean:       '+strcompress(xmom(0),/rem), $
	  '    Variance:   '+strcompress(xmom(1),/rem), $
	  '    Skewness:   '+strcompress(xmom(2),/rem), $
	  '    Kurtosis:   '+strcompress(xmom(3),/rem), $
	  '    Mean Absolute dev:   '+strcompress(xmdev,/rem), $
	  '    Standard Deviation:  '+strcompress(xsdev,/rem), $
	  '  ', $
	  'Values for the ordinates values are: ', $
	  '    Mean:       '+strcompress(ymom(0),/rem), $
	  '    Variance:   '+strcompress(ymom(1),/rem), $
	  '    Skewness:   '+strcompress(ymom(2),/rem), $
	  '    Kurtosis:   '+strcompress(ymom(3),/rem), $
	  '    Mean Absolute dev:   '+strcompress(ymdev,/rem), $
	  '    Standard Deviation:  '+strcompress(ysdev,/rem)]
	;ShowInfo1,TITLE='Xplot Calculations/Moments', $
	;  GROUP=event.top,INFOTEXT=text, WIDTH=$
	;  max(strlen(text)), HEIGH=n_elements(text)
	Xdisplayfile1,TEXT=text,Title='Xplot Calculations/Moments results.'
	end
 'Operations with sets...':		begin
	EVALUATE:
	tmp = state.calcstr.xpevaluate
	xplot_evaluate,tmpset,eval_str= tmp,group=event.top,$
	no_block=state.no_block
	state.calcstr.xpevaluate = tmp
	end
 'Fourier Filtering...':	FOURIERFILTER,GROUP=event.top, $
	DATA=tmpset(1,*),XDATA=tmpset(0,*),NO_BLOCK=state.no_block
 'Derivative':		begin
	der = tmpset
	der(1,*) = Deriv(tmpset(0,*),tmpset(1,*))
	xplot,der,wtitle='Xplot Calculations/Derivative',$
	  NO_BLOCK=state.no_block
	end
 'CDF':		begin
	tmpout = tmpset

	nelem = n_elements(tmpset(0,*))
	for i=1,nelem-1 do tmpout(1,i)= $
		int_tabulated(tmpset(0,0:i),tmpset(1,0:i))
	xplot,tmpout,wtitle='Xplot Calculations/CDF',$
	  NO_BLOCK=state.no_block
	end
 'Smooth...':		begin
	tmpout = tmpset
	if n_elements(smoothstr) EQ 0 then begin
	  smoothstr = { method:['1','Boxcar average','Least Squares'],$
	  width:5,edge:['0','No','Yes'],degree:2 }
	endif
	XscrMenu,smoothstr,/NoType,/Interp, $
	Titles=['Smoothing method','Width of smoothing window:', $
	 'Edge Truncation: ','Polynomial degree'],Action=action, $
	  help=xplot_text('SMOOTH'),wtitle='smooth',$
	  flags=['1','1','w(0) EQ 0','w(0) EQ 1']
	if action eq 'DONT' then goto,out
	tmpout = tmpset
	width = float(smoothstr.width)
	case strcompress(smoothstr.method(0),/rem) of
	  '0': begin ; BOXCAR
		;message,/info,'Using SMOOTH routine'
		wtitle='Xplot Calculations/Smooth(Boscar),width: '+$
		strcompress(width,/rem)
		edge = fix(smoothstr.edge(0))
		tmpout(1,*) = Smooth(tmpset(1,*),width,edge=edge)
	 	end
	  '1': begin ; LESAT-SQUARE
		;message,/info,'Using POLY_SMOOTH routine'
		degree = smoothstr.degree
		wtitle='Xplot Calculations/Smooth(LeastSquares),width: '+$
		strcompress(width,/rem)+', degree: '+$
		strcompress(degree,/rem)
		if width LE degree then begin
		  itmp = widget_message(/ERROR,dialog_parent=event.top,$
		  'Width must be LARGER than DEGREE. Aborting.')
		  goto,out
		endif
		tmpout(1,*) = Poly_Smooth(reform(tmpset(1,*)),width,$
			degree=degree)
	 	end
	endcase
	xplot,tmpout,wtitle=wtitle,NO_BLOCK=state.no_block
	end
 'Interpolation...': begin
	if n_elements(interstr) EQ 0 then begin
	  interstr = { kind:['0','Linear','Spline','Parametric spline'], $
	  npts:500 }
	endif
	XscrMenu,interstr,/NoType,/Interp,Titles= $
	  ['Interpolation: ','Number of points of new set:'], $
	  Wtitle='Interpolation',Action=action,FLAGS=['1','w(0) NE 2'],$
	  HELP=xplot_text('INTERPOLATION')
	if action eq 'DONT' then goto,out
	mymethod = strcompress(interstr.kind(0),/rem)
	if mymethod NE '2' then begin
	  n_interp=interstr.npts
	  int_step = $
	    (max(tmpset(0,*))-min(tmpset(0,*)))/float(n_interp-1)
	  newset = fltarr(2,n_interp)
      	  newset(0,*)=findgen(n_interp)*int_step+min(tmpset(0,*))
	endif
	case mymethod of
	'0': newset(1,*) = interpol(tmpset(1,*),tmpset(0,*),newset(0,*))
	'1': newset(1,*) = spline(tmpset(0,*),tmpset(1,*),newset(0,*))
	'2': begin
	  xx=0.0 & yy=0.0
	  spline_p,tmpset(0,*),tmpset(1,*),xx,yy
	  newset = fltarr(2,n_elements(xx))
	  newset(0,*) = reform(xx)
	  newset(1,*) = reform(yy)
	  end
	endcase
	wtitle='Xplot Calculations/Interpolation'
	if (interstr.kind(0) eq '0') then $
		wtitle = wtitle+'(Linear)' else $
		wtitle = wtitle+'(Spline)'
	xplot,newset,wtitle=wtitle
	end
 'Convolution & Correlation...':		begin
	Conv_data = state.calcstr.xpconvolution 
	;Conv_flag = ['1','w(0) EQ 1','1','1','w(3) EQ 0']
	Conv_flag = ['1','1','w(1) EQ 2','1',$
		'w(1) EQ 1','w(1) EQ 1','1','1','w(7) EQ 1','1']
	Conv_titles = $
	    ['Method','Type','External file:','Normalization:',$
	  'Voigt ratio (1=Lorentzian, 0=gaussian)',$
	  'Voigt Full Width at Half Maximum',$
	  'Overplot original data',$
	  'Write output to file:', 'Output file name:',$
	  'Calculation by:']
	XscrMenu,conv_data,GROUP=event.top,TITLES=conv_titles, $
	  FLAGS=conv_flag,ACTION=action,/INTER,/NOTYPE, $
	  WTITLE='Convolution data',HELP=xplot_text('CONVOLUTION'),$
	  ncol=2
	IF (action EQ 'DONT') THEN goto,out
	IF (STRCOMPRESS(Conv_data.Type(0),/REM) EQ '2') THEN  BEGIN
	  tmp = rascii(Conv_data.Filename)
	  if not(keyword_set(tmp)) then begin
	    itmp = widget_message(DIALOG_PARENT=event.top,$
	   'Error opening file'+$
	    Conv_data.Filename,/ERROR)
            goto,OUT
          endif
	  tmpset2 = tmp(0:1,*)
	ENDIF ELSE tmpset2 = tmpset
	if strcompress(Conv_data.type(0),/rem) EQ 1 then begin
	  FWHM=[Conv_data.voigt_fwhm,Conv_data.voigt_ratio]
	endif else FWHM=0
	;conv_res = CONV2(tmpset2,tmpset, FWHM=fwhm,$
	widget_control,/HOURGLASS
	conv_res = CONV2(tmpset,tmpset2, FWHM=fwhm,$
	  NORM=fix(Conv_data.Normalization(0)),$
	  CORRELATION=fix(Conv_data.method(0)),$
	  METHOD=fix(Conv_data.calculation(0)) )
	itmp = 0L

	IF (STRCOMPRESS(Conv_data.over(0),/REM) EQ '1')  THEN BEGIN
	  xplot,tmpset,wtitle= 'Xplot Calculations/Convolution&Correlation',$
	  NO_BLOCK=state.no_block,PARENT=itmp,$
	  xrange=[max(tmpset(0,*)),min(tmpset(0,*))]
	  state.calcstr.xpconvolution = Conv_data
	  xplot_savecurrent,itmp
	  xplot_loadfile,itmp,conv_res
	ENDIF ELSE BEGIN
	  xplot,conv_res,wtitle= 'Xplot Calculations/Convolution&Correlation',$
	  NO_BLOCK=state.no_block,PARENT=itmp
	  state.calcstr.xpconvolution = Conv_data
	ENDELSE

	IF (STRCOMPRESS(Conv_data.Write(0),/REM) EQ '1')  THEN BEGIN
	  tmp = widget_message(DIALOG_PARENT=itmp,$
	  ['File '+Conv_data.Fileout+' already exists.', $
	  'Overwrite it?'],/CANCEL,/DEFAULT_CANCEL)
	  if tmp EQ 'Cancel' then goto,out
	  OPENW,Unit,Conv_data.Fileout,/GET_LUN
	  PRINTF,Unit,conv_res
	  FREE_LUN,Unit
	  ;tmp = widget_message(DIALOG_PARENT=itmp,$
	  ;'File '+Conv_data.Fileout+' written to disk.',/INFO)
	  widget_control,state.wids.message,set_value='Xplot message: '+$
		'File '+Conv_data.Fileout+' written to disk.'
	ENDIF
	end
 'Polynomial&Gaussian...':		begin
	outtext = ''
	oplotflag = n_elements(tmpset(0,*)) 
	if oplotflag LT 100 then oplotflag = 100 else $
		oplotflag=1
	yfit = xplot_fitpolgauss(tmpset,OPLOT=oplotflag,$
		OUTTEXT=outtext,EXTRAP=tmpset1,GROUP=event.top)
	if not(keyword_set(yfit)) then goto,out
	XDisplayFile1, GROUP=event.top, TEXT=outtext ,$
	  TITLE='Xplot Polynomial&Gaussian Fit Results'
	;oplot,tmpset(0,*),yfit,psym=-4
	widget_control,state.wids.message,set_value = $
	  'Xplot message: Diamond points are the result'+$
	  ' of the fit.'
	tmp = widget_message( DIALOG_PARENT=event.top,$
	  /CANCEL,TITLE='Temporary Fit results',[ ttext,' ',$
	  'The overplotted diamond points are the result',$
	  'of the fitting routine. This data has not been',$
	  'saved yet, thus will disappear when updating', $
	  '(ReDraw) the Xplot window...', $
	  '  ',$
	  'You can append the result of the fit to ',$
	  'the current set in a new additional column.',$
	  '   ','Do you want to do that?'])
	if tmp EQ 'Cancel' then goto,out
	; add result as new column
	; place the "state" because it will be needed by xplot_addcol
	widget_control, event.top, SET_UVALUE = state, /NO_COPY
	xplot_addcol,event.top,yfit
	return
	end
 'Non linear...':begin
        if n_elements(nlfitstr) EQ 0 then begin
	  a = [max(tmpset(1,*)),(max(tmpset(0,*))+min(tmpset(0,*)))/2.0,$
	      getfwhm(tmpset)/2.35]
          nlfitstr = ["equation='a(0)*exp(-((x-a(1))/a(2))^2/2)'",$
		'a='+vect2string(a),'free_flags=[1,1,1]']
        endif

	xnlfit,tmpset, /NoDraw ,GROUP=event.top , $
	xplotmain = state.wids.main, NO_BLOCK=state.no_block, $
	init_pars=nlfitstr
	end
 'XAFS extraction (xaid_mu2chi)': $
	xaid_mu2chi,tmpset, GROUP=event.top, NO_BLOCK=state.no_block
	'XAFS Fourier Filter (xaid_ff)': $
	xaid_ff,tmpset, GROUP=event.top , NO_BLOCK=state.no_block
 else: print,'CALCULATIONS case not found: ',eventval
 ENDCASE
OUT:
widget_control, event.top, SET_UVALUE = state, /NO_COPY

end ;xplot_calculations
;
;=========================================================================
;
PRO xplot_controls_map,initval
on_error,2
;
; MAP the connected/Disconnected when Symbol=line (always connected)
;
if initval.Symbol EQ 0 then begin ; line selected
	widget_control,initval.wids.Connect,MAP=0 ; disappear
	widget_control,initval.wids.linestyle,MAP=1 ; appear
endif else begin
	widget_control,initval.wids.Connect,MAP=1 ; appear
	if initval.Connect EQ 0 then $ ; connected set
	  widget_control,initval.wids.linestyle,MAP=1 else $; appear
	  widget_control,initval.wids.linestyle,MAP=0 ; disappear
endelse
end ; xplot_controls_map
;
;=========================================================================
;
PRO xplot_controls_update,initval
;
; updates the control pannel 
;
if type(initval) NE 8 then begin
  message,/info,'Usage: xplot_controls_update,str_controls'
  return
endif
;values
if widget_info(initval.wids.main,/VALID_ID) then begin
  widget_control,initval.wids.Connect,set_droplist_select=initval.Connect
  widget_control,initval.wids.Symbol,set_droplist_select=initval.Symbol
  widget_control,initval.wids.Linestyle,set_droplist_select=initval.Linestyle
  widget_control,initval.wids.Axis,set_droplist_select=initval.Axis
  widget_control,initval.wids.Ticks,set_droplist_select=initval.Ticks
  widget_control,initval.wids.Font,set_droplist_select=initval.Font
  widget_control,initval.wids.Thickness,set_value=initval.Thickness
  widget_control,initval.wids.CharSize,set_value=initval.CharSize
  if widget_info(initval.wids.BackGround,/REALIZED) then $
    widget_control,initval.wids.BackGround,set_value=initval.BackGround
  if widget_info(initval.wids.Axiscolor,/REALIZED) then $
    widget_control,initval.wids.Axiscolor,set_value=initval.Axiscolor
  if widget_info(initval.wids.Color,/REALIZED) then $
    widget_control,initval.wids.Color,set_value=initval.Color
  ;
  ; MAP the connected/Disconnected when Symbol=line (always connected)
  ;
  xplot_controls_map,initval
endif
end ;xplot_controls_update
;
;=========================================================================
;
pro xplot_controls_event,event
;
; reduced event handler for Controls window (xplot_controls)
;
widget_control,event.top,get_uval=uval
        ooo = event
        ooo.top = uval
        ;if xregistered('xplot',/noshow) GT 0 then xplot_event,ooo
	xplot_controls_action,ooo
end ;xplot_controls_event
;
;=========================================================================
;
function xplot_controls,top,group=group , initval=initval, no_block=no_block
;
;
; creates and updates the controls panel.
;
;
on_error,2
if not(keyword_set(group)) then group=0
if not(keyword_set(top)) then top=0L

id = widget_base(/COLUMN,TITLE='Xplot controls')
;
; CONTROLS
;

wBquit = widget_button(id, value=  "Hide Controls",$
  UVALUE='CONTROLS')
wBdrop = widget_base(id,/COLUMN,/FRAME)

wBsymbol = widget_droplist(wBdrop,UVALUE='PSYMBOL', $
	VALUE = ['0: Line','1: +','2: *','3: Period',$
        '4: Diamond','5: Triangle','6: Square','7: X',$
	'8: User-defined     ',$
	'9: Undefined', '10:Histogram'], TITLE='   Symbol:')

; I had to add this tmp base to MAP correctly. I do not know why.  97/10/23
tmp = widget_base(wBdrop)
wBconnect = widget_droplist(tmp,UVALUE='PSYMSIGN', $
	VALUE = ['Connected','Disconnected        '], TITLE=' Sym Link:')

; I had to add this tmp base to MAP correctly. I do not know why.  97/10/23
tmp = widget_base(wBdrop)
wBlinestyle = widget_droplist(tmp, $
	VALUE=['0: Solid','1: Dotted','2: Dashed','3: Dash-Dot',$
	'4: -...','5: Long Dash        '],$
	TITLE='Linestyle:',UVALUE='LINESTYLE')

wBkind = widget_droplist(wBdrop, $
	VALUE=["Lin/Lin","Lin/Log","Log/Lin","Log/Log             "], $
	TITLE='     Axis:',UVALUE='KIND')
wBticks = widget_droplist(wBdrop, $
	VALUE=["In","Out","Grid                "], $
	TITLE='    Ticks:',UVALUE='TICKLEN',XSIZE=20)

wBfonts = widget_droplist(wBdrop, VALUE= $
               ['3:  Simplex Roman', $
                '4:  Simplex Greek', $
                '5:  Duplex Roman', $
                '6:  Complex Roman', $
                '7:  Complex Greek', $
                '8:  Complex Italic', $
                '9:  Math and Special', $
                '10: Special', $
                '11: Gothic English', $
                '12: Simplex Script', $
                '13: Complex Script', $
                '14: Gothic Italian', $
                '15: Gothic German', $
                '16: Cyrillic', $
                '17: Triplex Roman', $
                '18: Triplex Italic' ],TITLE='     Font:',UVALUE='FONT')


wBsli = widget_base(id,/COLUMN,/FRAME)
wBthick = cw_fslider(wBsli, TITLE='Line/Symbol thickness',/FRAME, $
 	UVALUE='THICK',MIN=0.,MAX=10.0, XSIZE=205, /EDIT)

wBchar = cw_fslider(wBsli, TITLE='Character Size',/FRAME, $
 	UVALUE='CHARSIZE',MIN=0.1,MAX=3.0, XSIZE=210, /EDIT)

wBc = widget_base(id,/COLUMN,/FRAME)
wBclr_reset = widget_button(wBc, value="      Reset initial color    ",$
  UVALUE='CLR_RESET')
wBxloadct = widget_button(wBc, value=  "      Change color table     ",$
  UVALUE='XLOADCT')
wBbackg= cw_clr_index(wBc, LABEL='Bkgd: ', $
 	UVALUE='CLR_BACK', XSIZE=90, YSIZE=25)
wBaxiscolor= cw_clr_index(wBc, LABEL='Axis: ', $
 	UVALUE='CLR_AXIS', XSIZE=90, YSIZE=25)
wBcolor= cw_clr_index(wBc, LABEL='Lines:', $
 	UVALUE='CLR_LINES', XSIZE=90, YSIZE=25)

;
; set initial values
;
if keyword_set(initval) then begin
  ;wids
  initval.wids.Main = id
  initval.wids.Connect = wBconnect
  initval.wids.Symbol = wBsymbol
  initval.wids.Linestyle = wBlinestyle
  initval.wids.Axis = wBkind
  initval.wids.Ticks = wBticks
  initval.wids.Font = wBfonts
  initval.wids.Thickness = wBthick
  initval.wids.CharSize = wBchar
  initval.wids.BackGround = wBbackg
  initval.wids.Axiscolor = wBaxiscolor
  initval.wids.Color = wBcolor
  ;values
  ;tmp = initval
;help,/str,tmp
  xplot_controls_update,initval
  ;widget_control,wBconnect,set_droplist_select=initval.Connect
  ;widget_control,wBsymbol,set_droplist_select=initval.Symbol
  ;widget_control,wBlinestyle,set_droplist_select=initval.Linestyle
  ;widget_control,wBkind,set_droplist_select=initval.Axis
  ;widget_control,wBticks,set_droplist_select=initval.Ticks
  ;widget_control,wBfonts,set_droplist_select=initval.Font
  ;widget_control,wBthick,set_value=initval.Thickness
  ;widget_control,wBchar,set_value=initval.CharSize
endif

widget_control,id,set_uval=top
widget_control,id,/realize,group=group,  $
  event_pro='xplot_controls_event'

if keyword_set(initval) then begin
  widget_control,wBbackg,set_value=initval.BackGround
  widget_control,wBaxiscolor,set_value=initval.Axiscolor
  widget_control,wBcolor,set_value=initval.Color
endif
if keyword_set(no_block) then $
   xmanager, 'xplot_controls', id, GROUP_LEADER=group,/no_block else $
   xmanager, 'xplot_controls', id, GROUP_LEADER=group
return,[id,wBbackg,wBaxiscolor,wBcolor]
end
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_CONTROLS_ACTION
;	PURPOSE: to change plot attributes in XPlot (i.e. Log plots,
;		symbols, thicknesses, colors, etc). In other words, 
;		permits the non-interactive manipulations of all the 
;		controls under the "Show Controls" button.
;		If the "Show Controls" window is opened, it also updates it.
;
;	CALLING SEQUENCE: XPLOT_CONTROLS_ACTION,parent [, keywords]
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	KEYWORD PARAMETERS [USE AT LEAST ONE]:
;		Note that some keyword values are assigned to zero. For 
;		setting them use KEYWORD='0', because KEYWORD=0 means not 
;		setting the keyword. For non-zero values, both KEYWORD='n'
;		and KEYWORD=n are, in general, accepted.
;
;		PSYMSIGN set plot to concected (PSYMSIGN='0') or disconnected
;			(PSYMSIGN='1') line.  This keyword does not make any 
;			effect if PSYMBOL='0'
;		PSYMBOL set symbol:
;			'0' line
;			1 +
;			2 *
;			3 Period
;			4 Diamond
;			5 Triangle
;			6 Square
;			7: X
;			8: User-defined
;			9: undefined
;			10: histogram
;		LINESTYLE define the line style:
;			'0' Solid
;			1 Dotted
;			2 Dashed
;			3 Dash-Dot
;			4 -...-...
;			5 Long dash
;		KIND log or lin plots in X/Y:
;			'0' lin/lin
;			1   lin/log
;			2   log/lin
;			3   log/log
;		TICKLEN internal/external ticks or grid
;			'0' internal
;			1   external
;			2   grid
;		THICK <float> the line thickness
;		CHARSIZE <float> the character size
;		CLR_BACK <integer> the background color
;		CLR_AXIS <integer> the axis color
;		CLR_LINES <integer> the lines color
;		CLR_RESET set this keyword to reset the original background,
;			axis and lines colors.
;		FONT selects a font 
;                       3:  Simplex Roman
;                       4:  Simplex Greek
;                       5:  Duplex Roman
;                       6:  Complex Roman
;                       7:  Complex Greek
;                       8:  Complex Italic
;                       9:  Math and Special
;                       10: Special
;                       11: Gothic English
;                       12: Simplex Script
;                       13: Complex Script
;                       14: Gothic Italian
;                       15: Gothic German
;                       16: Cyrillic
;                       17: Triplex Roman
;                       18: Triplex Italic
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_controls_action,input,PSYMSIGN=PSYMSIGN,PSYMBOL=PSYMBOL,$
  LINESTYLE=LINESTYLE,$
  KIND=KIND, TICKLEN=TICKLEN, THICK=THICK, CHARSIZE=CHARSIZE, $
  CLR_BACK=CLR_BACK, CLR_AXIS=CLR_AXIS, CLR_LINES=CLR_LINES, CLR_RESET=CLR_RESET,$
  FONT=FONT,NOREFRESH=NOREFRESH, $
  CONTROLS=CONTROLS ; this one for use with xplot.pto, not for interactive use
  
 
catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  catch, /cancel
  goto,out
endif

eventUVal=0
pannelupdate=0
case type(input) of
8: begin
   event = input
   widget_control, event.top , GET_UVALUE = state, /NO_COPY
   Widget_Control, event.id, get_UValue=eventUVal
   wid_top=event.top
   if keyword_set(CONTROLS) then eventUVal='CONTROLS'
   end
3: begin
   widget_control, input, GET_UVALUE = state, /NO_COPY
   wid_top=input
   eventuval = ''
   if keyword_set(PSYMSIGN) then eventUVal=[eventUVal,'PSYMSIGN']
   if keyword_set(PSYMBOL) then eventUVal=[eventUVal,'PSYMBOL']
   if keyword_set(LINESTYLE) then eventUVal=[eventUVal,'LINESTYLE']
   if keyword_set(KIND) then eventUVal=[eventUVal,'KIND']
   if keyword_set(TICKLEN) then eventUVal=[eventUVal,'TICKLEN']
   if keyword_set(THICK) then eventUVal=[eventUVal,'THICK']
   if keyword_set(CHARSIZE) then eventUVal=[eventUVal,'CHARSIZE']
   if keyword_set(CLR_BACK) then eventUVal=[eventUVal,'CLR_BACK']
   if keyword_set(CLR_AXIS) then eventUVal=[eventUVal,'CLR_AXIS']
   if keyword_set(CLR_LINES) then eventUVal=[eventUVal,'CLR_LINES']
   if keyword_set(CLR_RESET) then eventUVal=[eventUVal,'CLR_RESET']
   if keyword_set(FONT) then eventUVal=[eventUVal,'FONT']
   my_value = 0
   if n_elements(eventuval) GE 2 then $
     eventuval=eventuval(1:n_elements(eventuval)-1)
   if widget_info(state.strcontrols.wids.main,/VALID_ID) then $
	pannelupdate = 1 
   end
else: begin
   message,/info,'Usage: xplot_controls_action, event (or parent_id)'
   return
   end
endcase

for i=0,n_elements(eventUVal)-1 do begin

  event_tmp = eventUVal(i)
  if type(input) EQ 3 then begin
    command = 'my_value = '+event_tmp
     itmp = execute(command)
  endif

  CASE event_tmp of
	'PSYMSIGN':	BEGIN
		if not(keyword_set(PSYMSIGN)) then my_value=event.index
		state.xpdt.connect = my_value
		state.strcontrols.connect = my_value
		if pannelupdate then xplot_controls_map,state.strcontrols
		;xplot_plot,state
		END
	'PSYMBOL':	BEGIN
		if not(keyword_set(PSYMBOL)) then my_value=event.index
		state.xpdt.psym = my_value
		state.strcontrols.symbol = my_value
		if pannelupdate then xplot_controls_map,state.strcontrols
		;xplot_plot,state
		END
	'LINESTYLE':	BEGIN
		if not(keyword_set(LINESTYLE)) then my_value=event.index
                state.xpdt.linestyle = my_value
		state.strcontrols.LineStyle = my_value
		;xplot_plot,state
		END
	'KIND':	BEGIN
		if not(keyword_set(KIND)) then my_value=event.index
		temp = my_value
		state.strcontrols.Axis = my_value
                if temp eq 0 then state.xpdt.kind = 1 
                if temp eq 1 then state.xpdt.kind = 3 
                if temp eq 2 then state.xpdt.kind = 4 
                if temp eq 3 then state.xpdt.kind = 5 
                ;if temp eq 4 then state.xpdt.kind = 2 
		;xplot_plot,state
		END
	'TICKLEN':	BEGIN
		if not(keyword_set(TICKLEN)) then my_value=event.index
		state.strcontrols.Ticks = my_value
		temp = my_value
                if temp eq 0 then state.xpdt.sysvar(1) = '!p.ticklen = 0.02 '
                if temp eq 1 then state.xpdt.sysvar(1) = '!p.ticklen = -0.02 '
                if temp eq 2 then state.xpdt.sysvar(1) = '!p.ticklen = 1.0'
		;xplot_plot,state
		END
	'THICK':	BEGIN
		if not(keyword_set(THICK)) then $
			widget_control,event.id,GET_VALUE=my_value
		temp = my_value
		state.strcontrols.Thickness = temp
		state.xpdt.Thick = temp
		;xplot_plot,state
		END
	'CHARSIZE':	BEGIN
		if not(keyword_set(CHARSIZE)) then $
		  widget_control,event.id,GET_VALUE=my_value
		temp = my_value
		state.strcontrols.CharSize = temp
		state.xpdt.sysvar(0) = '!p.charsize = '+ $
			strcompress(temp,/rem)
		;xplot_plot,state
		END
	'CLR_BACK':	BEGIN
                if not(keyword_set(CLR_BACK)) then $
		  widget_control,event.id,GET_VALUE=my_value
		temp = my_value
		state.strcontrols.BackGround = temp
		state.xpdt.sysvar(6) = '!p.background = '+ $
                        strcompress(temp,/rem)
                ;xplot_plot,state
		END
	'CLR_AXIS':	BEGIN
		if not(keyword_set(CLR_AXIS)) then $
		  widget_control,event.id,GET_VALUE=my_value
		temp = my_value
		state.strcontrols.Axiscolor = temp
		;state.xpdt.sysvar(6) = '!p.background = '+ $
                ;        strcompress(temp,/rem)
                ;xplot_plot,state
		END
	'CLR_LINES':	BEGIN
		if not(keyword_set(CLR_LINES)) then $
		  widget_control,event.id,GET_VALUE=my_value
		temp = my_value
		state.strcontrols.Color = temp
                state.xpdt.sysvar(7) = '!p.color = '+ $
                        strcompress(temp,/rem)
		;xplot_plot,state
		END
	'CLR_RESET':	BEGIN
		state.xpdt.sysvar(6) = state.xpdt.sysvar_old(6)
		state.xpdt.sysvar(7) = state.xpdt.sysvar_old(7)
		tmp = execute(state.xpdt.sysvar(6))
		tmp = execute(state.xpdt.sysvar(7))
		if not(keyword_set(CLR_RESET)) then begin
		  widget_control,state.wids.controls(1),Set_val=!p.background
		  widget_control,state.wids.controls(2),Set_val=!p.color
		  widget_control,state.wids.controls(3),Set_val=!p.color
		endif
		state.strcontrols.BackGround = !p.background
		state.strcontrols.Axiscolor = !p.color
		state.strcontrols.Color = !p.color
		;xplot_plot,state
		END
	'FONT':	BEGIN
		if not(keyword_set(FONT)) then my_value=event.index
                state.strcontrols.Font = my_value
		xyouts,"'!"+strcompress(3+ $
			fix(state.strcontrols.Font),/REM)+"'"
		;xplot_plot,state
		END
	; the next two entries are only for interactive use
	'XLOADCT':	Xloadct,GROUP=event.top
	'CONTROLS':	BEGIN
                widget_control,event.id,get_value=tmp
                case tmp of
                  'Show Controls': begin
                     widget_control,event.id,set_value='Hide Controls'
		     tmpstr = state.strcontrols
		     state.wids.controls=xplot_controls(event.top, $
			group=event.top,initval=tmpstr, $
			no_block=state.no_block)
		     state.strcontrols = tmpstr
                     end
                  'Hide Controls': begin
                     widget_control,state.wids.showcontrols, $
			set_value='Show Controls'
                     if widget_info(state.wids.controls(0),/valid_id) then $
                       widget_control,state.wids.controls(0),/DESTROY
                     end
                  else:
                endcase
		goto,out
		END
	else:
  ENDCASE
endfor
if pannelupdate then xplot_controls_update,state.strcontrols
if not(keyword_set(norefresh)) then xplot_plot,state

out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_controls_action
;
;=========================================================================
;
PRO xplot_evaluate,set,eval_str=eval_str,group=group,no_block=no_block
;
; drives the Calculations/Operations with sets... option
;
on_error,2
;
; to evaluate an expression [Calculations Menu]
;
if not(keyword_set(eval_str)) then eval_str = $
          xplot_defaults('XPevaluate')
;
flags=['1','1','1','1','w(2) EQ 1','1','w(5) EQ 1','1']
titles=['Expression for x Cx=f(Ax): ', 'Expression for y Cy=f(Ay,By): ', $
  'Number of data sets:', $
  'A data set:', 'B data file:', $
  'Write output to file: ', 'Output file name:', $
  'Overplot original data:' ]

eval_str_old = eval_str
XscrMenu,eval_str,/NOTYPE,/INTERP,FLAGS=flags,TITLES=titles, $
  ACTION=action,WTITLE='Operations with sets',GROUP=group,$
  HELP=xplot_text('OPERWITHSETS')

if action eq 'DONT' then begin
  eval_str = eval_str_old
  return
endif
;
; start operations
;
; load data 
;
IF KEYWORD_SET(SET) THEN BEGIN
if eval_str.set1 ne 'current set' then begin
tmp = rascii(eval_str.set1)
  if not(keyword_set(tmp)) then begin
    tmp = widget_message(DIALOG_PARENT=group, $
    ['Error opening file','Returning'],/ERROR)
    eval_str = eval_str_old
    return
  endif
  set1 = tmp
endif else  begin
  if not(keyword_set(set)) then begin
    tmp = widget_message(DIALOG_PARENT=group, $
    ['Empty data set','Returning'],/ERROR)
    eval_str = eval_str_old
    return
  endif
  set1 = set
endelse
Ax = set1(0,*)
Ay = set1(1,*)
;
; when using two curves 
;
if strcompress(eval_str.nset(0),/REM)  eq '1' then begin
  ; load file
  tmp = rascii(strcompress(eval_str.set2,/REM))
  if not(keyword_set(tmp)) then begin
    ;xtext,TEXT=['Error opening file','Returning'],GROUP=group
    tmp = widget_message(DIALOG_PARENT=group, $
    ['Error opening file','Returning'],/ERROR)
    eval_str = eval_str_old
    return
  endif
  set2 = tmp
  Bx = tmp(0,*)
  By = tmp(1,*)
endif
ENDIF
;
; evaluate the expression
;
Cx = 0
tmp = execute(eval_str.expression_x)
if tmp ne 1 then begin
  ;xtext,TEXT='Expression for x not understood',GROUP=group $
  tmp = widget_message(DIALOG_PARENT=group, $
  'Expression for x not understood',/ERROR)
  eval_str = eval_str_old
  return
endif    ; else set1(0,*) = Cx
;
;
; interpolates the new array
;
if strcompress(eval_str.nset(0),/REM)  eq '1' then begin
  set2old = set2
  set2 = set1
  set2(1,*) = interpol(set2old(1,*),set2old(0,*),set2(0,*))
  ;
  ; y variable
  ;
  Bx = set2(0,*)
  By = set2(1,*)
endif
;
Cy = 0
tmp = execute(eval_str.expression_y)
if tmp ne 1 then begin
  tmp = widget_message(DIALOG_PARENT=group, $
  'Expression for y not understood',/ERROR)
  eval_str = eval_str_old
  return
endif
;
setnew = fltarr(2,n_elements(Cx)) ;set1
setnew(0,*) = Cx
setnew(1,*) = Cy
;
; show and exit
;
choice = strcompress(eval_str.save_as(0),/REM)
;
; overplot old sets of data
;
if ( strcompress(eval_str.over(0),/REM) eq '1') then begin
  if strcompress(eval_str.nset(0),/REM)  eq '1' then begin
    ; to insert two curves in xplot, I'm obligated to use handlers
    handle = handle_create()
    handle1 = handle_create(handle,value=setnew,/FIRST_CHILD)
    handle2 = handle_create(handle1,value=set1,/SIBLING)
    handle3 = handle_create(handle2,value=set2,/SIBLING)
    xmin = min( [setnew(0,*),set1(0,*),set2(0,*)],max = xmax)
    ymin = min( [setnew(1,*),set1(1,*),set2(1,*)],max = ymax)
  endif else begin
    ; to insert two curves in xplot, I'm obligated to use handlers
    handle = handle_create()
    handle1 = handle_create(handle,value=setnew,/FIRST_CHILD)
    handle2 = handle_create(handle1,value=set1,/SIBLING)
    xmin = min( [setnew(0,*),set1(0,*)],max = xmax)
    ymin = min( [setnew(1,*),set1(1,*)],max = ymax)
  endelse
  if keyword_set(no_block) then $
  xplot,handle,no_block,xrange=[xmin,xmax],yrange=[ymin,ymax],wtitle= $
	'Xplot '+xplot_version()+' Results of Operations with sets...' else  $
  xplot,handle,xrange=[xmin,xmax],yrange=[ymin,ymax],wtitle= $
	'Xplot '+xplot_version()+' Results of Operations with sets...' 
endif else begin
  if keyword_set(no_block) then $
  xplot,setnew,/no_block,wtitle=$
	'Xplot '+xplot_version()+' Results of Operations with sets...' else $
  xplot,setnew,wtitle=$
	'Xplot '+xplot_version()+' Results of Operations with sets...'
endelse


;
if choice eq '1' then begin
  if checkfile(eval_str.filename) EQ 1 then begin
     tmp = widget_message(DIALOG_PARENT=group, $
        ['File '+eval_str.filename+' already exists',$
	'Overwrite it?'],/CANCEL,/DEFAULT_CANCEL)
     if tmp EQ 'Cancel' then return 
  endif
  openw,Unit,strcompress(eval_str.filename,/REM),/GET_LUN
  printf,Unit,setnew
  free_lun,Unit
  tmp = widget_message(DIALOG_PARENT=group, $
	'File '+eval_str.filename+' written to disk', /INFORMATION)
endif
;
end ;xplot_evaluate
;
;=========================================================================
;
FUNCTION xplot_fitgauss,set,OUTTEXT=outtext
;
; drives the Calculations/Fitting/Gaussian option
;
on_error,2
text = ['The gaussian fit option of XPlot permits to fit', $
	'the current data y=f(x) to one of the following functions: ', $
	'  ',$
	'  1) F(x) = A0*EXP(-z^2/2) + A3', $ 
	'     with         z=(x-A1)/A2', $
	'     where:', $
	'     A0 = height of exp, A1 = center of exp, A2 = sigma (the width)',$
	'     A3 = constant term', $
	'     (it uses the function GAUSS_FIT)', $
	'  ',$
	'  2) F(x) = A0*EXP(-z^2/2) + A3 + A4*x + A5*x^2', $
	'     with         z=(x-A1)/A2', $
	'     where:', $
	'     A0 = height of exp, A1 = center of exp, A2 = sigma (the width)',$
	'     A3 = constant term, A4 = linear term, A5 = quadratic term.', $
	'     (it uses the function GAUSSFIT)', $
	'  ',$
	'Please read the documentation in the Help/Routines for',$
	'detailed references of the method used for each fit.']

tmp = widget_message(text,/INFO,/CANCEL)
if tmp EQ 'Cancel' then return,0

widget_control,sensitive=0
itmp = wmenu2([' F(x) = A0*EXP(-z^2/2) + A3', $
	'  F(x) = A0*EXP(-z^2/2) + A3 + A4*x + A5*x^2'])
widget_control,sensitive=1


setx = reform(set(0,*))
sety = reform(set(1,*))

case  itmp  of
  0:BEGIN
	yfit = Gauss_Fit(setx,sety,Result)
	outtext = [ $
	'   Gaussina Fit (GAUSS_FIT) results ', $
	'   Fit result :  F(x) = A0*EXP(-z^2/2) + A3, z=(x-A1)/A2', $
	'            A0: '+strcompress(Result(0),/rem), $
	'            A1: '+strcompress(Result(1),/rem), $
	'            A2: '+strcompress(Result(2),/rem), $
	'            A3: '+strcompress(Result(3),/rem)]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
  1:BEGIN
	yfit = GaussFit(setx,sety,Result)
	outtext = [ $
	'   Gaussina Fit (GAUSS_FIT) results ', $
	'   Fit result :  F(x) = A0*EXP(-z^2/2) + A3 + A4*x + A5*x^2', $
	'                 z=(x-A1)/A2', $
	'            A0: '+strcompress(Result(0),/rem), $
	'            A1: '+strcompress(Result(1),/rem), $
	'            A2: '+strcompress(Result(2),/rem), $
	'            A3: '+strcompress(Result(3),/rem), $
	'            A4: '+strcompress(Result(4),/rem), $
	'            A5: '+strcompress(Result(5),/rem)]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
  else:
endcase

outtext = [outtext, ' ', $
  'Minimum y value for the fit: '+strcompress(min(yfit,max=yfitmax),/rem), $
  'Maximum y value for the fit: '+strcompress(yfitmax,/rem)]
return,yfit
end

;
;=========================================================================
;
function xplot_restorestate,OLDSTATE=oldstate,FILENAME=filename,IERR=ierr,$
  WSIZE=wsize
;
; drives the FIle/Restore xplot backup file
;
on_error,2
catch,ierr
 
if ierr NE 0 then begin
  print,'XPLOT_RESTORESTATE: Error restoring Xplot backup.'
  return,0
endif

if not(keyword_set(filename)) then filename = 'xplot.bck'

;
; initialize variables
;
version = 0 & state = 0 & dtcurrent = 0 & wsize=[0,0]
red = 0  &  green = 0  &  blue = 0
for i=0,10 do tmp =  execute('dtbuffer'+strcompress(i,/rem)+' = 0')

;
; restore
;
restore,filename,/VERBOSE
print,'XPLOT_RESTORESTATE: Restored Xplot backup '+version
;
; re-arrange handlers
;
if keyword_set(dtcurrent) then begin
  h1 = handle_create(value=dtcurrent)
  ;handle_value, h1,dtcurrent,/set
  state.dthcurrent=h1
endif else state.dthcurrent = 0L

h1 = 0L
print,'XPLOT_RESTORESTATE: old dthbuffer: ',state.dthbuffer
if keyword_set(state.buffer) then begin
  for i=0L,n_elements(state.buffer)-1 do begin
    h1 = handle_create()
    command = 'handle_value, h1,dtbuffer'+strcompress(i,/rem)+',/set'
    tmp = execute(command)
    state.dthbuffer(i)=h1
    ; substitute the new handler name in the buffer text
     line = state.buffer(i)
     s1= strpos(line,'L')
     newline = 'plotfile, '+strcompress(h1,/rem)+strmid(line,s1,strlen(line)-s1)
     state.buffer(i) = newline
  endfor
endif
print,'XPLOT_RESTORESTATE: new dthbuffer: ',state.dthbuffer
;
; color tables
;
tvlct,red,green,blue
;
; use correct widget identifiers
;
if keyword_set(oldstate) then state.wids = oldstate.wids

return,state
end

;
;=========================================================================
;
pro xplot_savestate,state,FILENAME=filename
;
; drives the File/Save as Xplot backup ... option
;
on_error,2
version = xplot_version() ;'Xplot version 2.0, (c) MSR-ESRF'
tvlct,red,green,blue,/get
wsize = [!d.x_size,!d.y_size]
if not(keyword_set(filename)) then filemane = 'xplot.bck'

if keyword_set(state.dthcurrent) then $
	plotfile,state.dthcurrent,dtcurrent,/nodata else $
	dtcurrent = 0

strdtbuffer = ''
if keyword_set(state.buffer) then begin
  for i=0L,n_elements(state.buffer)-1 do begin
    command = 'plotfile,state.dthbuffer('+strcompress(i,/rem)+$
		'),/nodata,dtbuffer'+strcompress(i,/rem)
    print,'XPLOT_SAVESTATE: executing: '+command
    tmp = execute(command)
    strdtbuffer = strdtbuffer+'dtbuffer'+strcompress(i,/rem)+','
  endfor
endif

command = 'save,version,state,dtcurrent,'+$
    'red,green,blue,wsize,'+strdtbuffer+$
   'FILENAME='+"'"+strcompress(filename,/rem)+"'"+',/VERBOSE'
print,'XPLOT_SAVESTATE: executing: '+command
tmp = execute(command)
end

;
;=========================================================================
;
PRO xpstr_update,state, $
   BUFFER=buffer, NOBUFFER = nobuffer, $
   DTHCURRENT=dthcurrent, NODTHCURRENT=nodthcurrent, $
   DTHBUFFER=dthbuffer, NODTHBUFFER=nodthbuffer, $
   EXEBUFFER=exebuffer, NOEXEBUFFER=noexebuffer, $
   NOCOLTITLES=nocoltitles, COLTITLES=coltitles, $
   NOPLOTNAME_BUFFER=noplotname_buffer, PLOTNAME_BUFFER=plotname_buffer

;
; This procedure updates the 'date' or 'buffer' tags in the structure 'state'
;  for the XPlot program
;
;  Why a procedure to update a tag, and I do not use the simple
;  expression state.data = newdata or state.buffer = newbuffer ?
;
; Because when I create the state structure, the dimensions of
; the tags are those of the initialization values. Now, I need that
; data very dimensions depending on the data to be display.
; 
; This procedures copies and re-defines the structure
; The structure MUST be anonymous
;
; MSR 94-02-18
; MSR 95-02-18
;
on_error,2
;
state_old = state

if keyword_set(buffer) then newbuffer = buffer else newbuffer = state_old.buffer
if keyword_set(nobuffer) then newbuffer = 0

if keyword_set(dthbuffer) then newdthbuffer = dthbuffer else $
		newdthbuffer = state_old.dthbuffer
if keyword_set(nodthbuffer) then newdthbuffer = 0L

if keyword_set(dthcurrent) then newdthcurrent = dthcurrent else $
		newdthcurrent = state_old.dthcurrent
if keyword_set(nodthcurrent) then newdthcurrent = 0L
;
if keyword_set(exebuffer) then newexebuffer = exebuffer else $
		newexebuffer = state_old.exebuffer
if keyword_set(noexebuffer) then newexebuffer = 0
;
if keyword_set(coltitles) then newcoltitles = coltitles else $
		newcoltitles = state_old.coltitles
if keyword_set(nocoltitles) then newcoltitles = 0
;
if keyword_set(plotname_buffer) then newplotname_buffer = plotname_buffer $
	else newplotname_buffer = state_old.plotname_buffer
if keyword_set(noplotname_buffer) then newplotname_buffer = 0

state = { wids:state_old.wids, $
	  calcstr:state_old.calcstr, $
	  xpdt:state_old.xpdt, $
	  xpdt_limits:state_old.xpdt_limits, $
	  xpdt_titles:state_old.xpdt_titles, $
	  buffer:newbuffer, $
	  dthcurrent:newdthcurrent, $
	  dthbuffer:newdthbuffer, $
	  strcontrols:state_old.strcontrols, $
	  spec:state_old.spec, $
	  exebuffer:newexebuffer, $
	  version:state_old.version, $
	  cloned:state_old.cloned, $
	  coltitles:newcoltitles,$
	  fileinf:state_old.fileinf,$
	  psysvar:state_old.psysvar, $
	  no_block:state_old.no_block, $
	  plotname_current:state_old.plotname_current, $
	  plotname_buffer:newplotname_buffer }
;
end
;
;=========================================================================
;
; Makes the plot
;
PRO XPLOT_PLOT,state,SAVE = save, NOCOMMAND = nocommand, $
	PRINTBUFFER=printbuffer
;
;
; xplot 2.0:
;	/save: to update the buffer with a saved set
;	printbuffer: is the printbuffer keyword is initialized when 
;		calling xplot_plot, then the printer buffer is 
;		stored in such variable. Useful for printing.
;	/nocommand: obsolete
;
on_error,2
widget_control,state.wids.draw,get_value=windownum
wset,windownum
!p=state.psysvar.p
!x=state.psysvar.x
!y=state.psysvar.y
;
for i=0,n_elements(state.xpdt.sysvar)-1 do begin
  ;print,'<><><> ',state.xpdt.sysvar(i)
  tmp = execute(state.xpdt.sysvar(i))
endfor
;
; in the case that the image is zoomed, overwrite some variables
; already defined in state.xpdt.sysvar
if state.xpdt.zoom_on EQ 1 then begin
  !x.range = state.xpdt.zoomsel(0:1)
  !y.range = state.xpdt.zoomsel(2:3)
endif


if keyword_set(state.dthbuffer) then $
    ndatasets = n_elements(state.dthbuffer)  else $
    ndatasets = 1
    
;print,'>>>>>ndatasets : ',ndatasets

case state.xpdt.kind of
   3: logstyle=',/ylog'
   4: logstyle=',/xlog'
   5: logstyle=',/xlog,/ylog'
   else: logstyle=''
endcase

;Note: if command1 is changed, remember to change also 
;	xplot_event/SETCURRENT
if state.xpdt.connect EQ 0 then psymsign='-' else psymsign=''
;help,state.xpdt.connect
command1 = 'plotfile,'+ strcompress(state.dthcurrent) +'L'+ $
	',xcol='+strcompress(state.xpdt.Xcol,/REM)+ $
	',ycol='+strcompress(state.xpdt.Ycol,/REM)+ $
	',psym='+psymsign+strcompress(state.xpdt.psym,/REM) + $
	',linestyle='+strcompress(state.xpdt.Linestyle,/REM)+ $
	',thick='+strcompress(state.xpdt.thick,/REM)+ $
	',color='+strcompress(state.strcontrols.color,/rem) + $
	',shiftval='+vect2string(state.xpdt.Shift)

if strcompress(state.xpdt.mesh.flag(0),/rem) EQ '1' then begin
  command1 = command1+$
    ',meshkind='+strcompress(state.xpdt.mesh.kind(0),/rem)
  if state.xpdt.mesh.keywords NE '' then $
	command1 = command1+','+state.xpdt.mesh.keywords
  case strcompress(state.xpdt.mesh.split(0),/rem) of
    '0': command1 = command1+$
	',meshpoints='+strcompress(state.xpdt.mesh.points,/rem)
    '1': command1 = command1+$
	',meshcol='+strcompress(state.xpdt.mesh.col,/rem)
    else: message,/info,'Error: MESH case not found'
  endcase
endif


if state.xpdt.axis2(0) NE state.xpdt.axis2(1) then $
	command1 = command1 + ',x2axis='+vect2string([state.xpdt.axis2(0),$
	 state.xpdt.axis2(1)])
if state.xpdt.axis2(2) NE state.xpdt.axis2(3) then $
	command1 = command1 + ',y2axis='+vect2string([state.xpdt.axis2(2),$
	 state.xpdt.axis2(3)])
if state.xpdt.ErrPlot(0) NE state.xpdt.ErrPlot(1) then $
	command1 = command1 + ',errplot='+vect2string([state.xpdt.errplot(0),$
	 state.xpdt.errplot(1)])

xtitletext = state.xpdt_titles.xtitle
if xtitletext EQ '-1' then begin
  if state.spec.on NE 1 then $
    xtitletext = state.coltitles(state.xpdt.Xcol-1) else begin
    labels = state.spec.labels 
    tmp = strparse(labels,'!',labelslist)
    if tmp EQ -1 then $
      tmp = widget_message(dialog_parent=state.wids.main,$
	'Error (column separator xtitle)', /ERROR) else $
      xtitletext = labelslist(state.xpdt.Xcol-1)
  endelse
endif

ytitletext = state.xpdt_titles.ytitle
if ytitletext EQ '-1' then begin
  if state.spec.on NE 1 then $
    ytitletext = state.coltitles(state.xpdt.Ycol-1) else begin
    labels = state.spec.labels
    tmp = strparse(labels,'!',labelslist)
    if tmp EQ -1 then $
      tmp = widget_message(dialog_parent=state.wids.main,$
	'Error (column separator xtitle)', /ERROR) else $
  	ytitletext = labelslist(state.xpdt.Ycol-1)
  endelse
endif

;iok
titletext = state.xpdt_titles.title
if strcompress(titletext,/rem) NE '' AND state.spec.on EQ 1 then  begin
  handle_value, state.spec.handle, hspec
  titletext=xplot_spectitle(titletext,hspec)
endif

;help,xtitletext

command = command1 + $
	;',title='+"'"+state.xpdt_titles.title+"'" + $
	',title='+'"'+titletext+'"' + $
	;',xtitle='+"'"+state.xpdt_titles.xtitle+"'" + $
	',xtitle='+'"'+xtitletext+'"' + $
	;',ytitle='+"'"+state.xpdt_titles.ytitle+"'" + $
	',ytitle='+'"'+ytitletext+'"' + $
	',axiscolor='+strcompress(state.strcontrols.Axiscolor,/rem) + $
	logstyle

;
if keyword_set(save) then begin
  if state.spec.on  then begin
     handle_value, state.spec.handle, hspec
     currentscan=' [#S '+$
	strcompress(spec_scan(hspec,'CURRENT'),/rem)+'] ' 
  endif else currentscan=''
  ;copy handle and places the copy as current
  plotfile,state.dthcurrent,/nodata,tmp
  hh = handle_create(value=tmp)
  ;handle_value,hh,tmp,/set
  if not(keyword_set(state.buffer)) then begin
        ;<>print,'XPLOT: ....... writting buffer .............'
	xpstr_update,state,buffer = command1+',ki=2' , dthbuffer= $
	    [state.dthcurrent],plotname_buffer=state.plotname_current+$
	    currentscan
  endif else begin
        ;<>print,'XPLOT: ....... adding to  buffer .............'
	xpstr_update,state,buffer = [state.buffer,command1+',ki=2'], $
	    dthbuffer= [state.dthbuffer,state.dthcurrent]
  	xpstr_update,state,plotname_buffer=[state.plotname_buffer,$
	   state.plotname_current+currentscan]
  endelse
  state.dthcurrent = hh
  goto,OUT
endif
;
if keyword_set(printbuffer) then begin
  printbuffer = [command]
  if keyword_set(state.buffer) then printbuffer = [printbuffer,state.buffer]
  if keyword_set(state.exebuffer) then $
		printbuffer = [printbuffer,state.exebuffer]
  ;print,'XPLOT: ############ printbuffer: #############'
  ;print,printbuffer
  goto,OUT
endif
;
; executing
;
if not(keyword_set(nocommand)) then begin
  ;print,'XPLOT: ...... executing command ...........'
  erase
  ;print,command
  a=execute(command)
endif

; if BUFFER buffer exist, execute it
if keyword_set(state.buffer) then begin
  ;<>print,'XPLOT: ...... executing buffer ...........'
  ;<>print,state.buffer
    for i=0,n_elements(state.buffer)-1 do a=execute(state.buffer(i))
endif

; if EXEBUFFER buffer exist, execute it
if keyword_set(state.exebuffer) then begin
  ;<>print,'XPLOT: ...... executing exebuffer ...........'
  ;<>print,state.exebuffer
    for i=0,n_elements(state.exebuffer)-1 do a=execute(state.exebuffer(i))
endif

OUT:
state.psysvar.p=!p
state.psysvar.x=!x
state.psysvar.y=!y
end
;
;=========================================================================
;
PRO XPLOT_EVENT,event

;if tag_names(event,/structure_name) EQ 'WIDGET_TRACKING' then begin
;  help,/str,event
;  return
;endif

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched '+!err_string
  itmp=Dialog_Message(/Error,Dialog_Parent=event.top,$
    'XPLOT_EVENT: error catched '+!err_string)
  catch, /cancel
  if n_elements(state) NE 0 then goto,out
  return
endif

;
; the following lines put artificially eventUVal='RESIZE' when the
; user resizes the window (event created by the top base widget (tbw) ).
; This is not elegant: one should store the "state" in the first 
; child of tbl, but in xplot is done directly in tbl. 
; One day that I will have the time I'll fix it! 
; MSR 97/10/16
;  
if tag_names(event,/structure_name) EQ 'WIDGET_BASE' then begin
  eventUVal = 'RESIZE'
endif else begin
  Widget_Control, event.id, get_UValue=eventUVal
endelse
;
; First the actions that do not require the "state" explicitely
; =============================================================
;
CASE eventuval OF
  'LOADFILE':     BEGIN
    xplot_loadfile,event.top
    return
    END
  'CALCULATIONS':	BEGIN
    xplot_calculations,event
    return
    END
  'SAVE':	BEGIN
    xplot_savecurrent,event.top
    return
    END
  'CONTROLS':	begin
    xplot_controls_action,event,/CONTROLS
    return
    end
  'XCOL':	begin
    xplot_changecol,event
    return
    end
  'YCOL':	begin
    xplot_changecol,event
    return
    end
  'ERRPLOT':	begin
    xplot_errorbars,event
    return
    end
  'EXEBUFFERCLEAN': begin
    xplot_exebuffer,event
    return
    end
  'EXEBUFFEREDIT': begin
    xplot_exebuffer,event
    return
    end
  'SETLABELS': begin
    xplot_exebuffer,event
    return
    end
  'SETLEGEND': begin
    xplot_exebuffer,event
    return
    end
  'EXIT':	begin
    xplot_quit,event.top
    return
    end
  'PLOTMGR': begin
    xplot_plotmgr,event.top
    return
    end
  'CLEARDATA': begin
    xplot_cleardata,event.top,/confirm
    return
    end
  'SETLIMITS': begin
    xplot_setlimits,event.top
    return
    end
  else:
ENDCASE


;
; Second the actions that do require the "state" 
; ==============================================
;
widget_control, event.top, GET_UVALUE = state, /NO_COPY

; this section is now in xplot_plot and in 'CURSOR'
;widget_control,state.wids.draw,get_value=windownum 
;wset,windownum
;!p=state.psysvar.p
;!x=state.psysvar.x
;!y=state.psysvar.y

CASE eventuval OF
;
; resize
;
        'RESIZE': BEGIN
		if sdep() EQ 'WINDOWS' then widget_control,state.wids.draw,$
		  scr_xsize=event.x-5,scr_ysize=event.y-110 else $
		  widget_control,state.wids.draw,$
		  scr_xsize=event.x-5,scr_ysize=event.y-165 
		xplot_plot,state
		goto,out
	END
; 
; these are in the menu  bar
;
        'SSCAN': BEGIN
                widget_control,state.spec.scanbase,/show, bad_id=bad
                event_name = tag_names(event, /structure_name)
                if event_name eq 'NEW_SCAN_EVENT' then begin
                   widget_control,/HOURGLASS
                   tmp=xplot_readspec(state, event.newscan, event.mca, /oldcol)
                   state.dthcurrent = handle_create(value=tmp)
                   xplot_mappanels, state.wids, state.spec
                   xplot_plot,state
                endif else if bad eq state.spec.scanbase then begin
                   xspecscan, state.spec.handle, caller=event.id,             $
                              tlb_id=tmp, mca=state.spec.mca, /not_all,       $
                              group_leader=event.top, title='Scan Selection', $
			      no_block=state.no_block
                   state.spec.scanbase = tmp
                endif
                END
	'RESTORE':	BEGIN
		if sdep() eq 'UNIX' then filter='*.bck' else filter=0
		file = pickfile(FILTER=filter,GROUP=event.top,/NOCONF,$
		title='Select a file with an Xplot backup')
		if file eq '' then GOTO,out
		newstate = xplot_restorestate(OLDSTATE=state, $
			IERR=IERR,FILENAME=file,WSIZE=wsize)
		if ierr NE 0 then begin
		  tmp = widget_message(dialog_parent=event.top,$
			'Error restoring Xplot backup file',/ERROR)
		  goto, out
		endif
                ; close control panel if open
                if widget_info(state.wids.controls(0),/valid_id) then begin
		  widget_control,state.wids.showcontrols, $
			set_value='Show Controls'
		  widget_control,state.wids.controls(0),/DESTROY
                endif

		; substitute state by the new one
		;state = newstate
                copy_structure,newstate,state

		state.spec.on = 0
		; resizes window if necessary
		if ((wsize(0) NE !d.x_size) OR (wsize(1) NE !d.y_size)) $
			then begin
		   widget_control,state.wids.draw,xsize=wsize(0), ysize=wsize(1)
		endif
		xplot_plot,state
		; update info widgets
		widget_control,state.wids.message,set_value = $
		 'Xplot message: Xplot restored backup file: '+file

		if keyword_set(state.buffer) then $
			nsaved = n_elements(state.buffer) else $
			nsaved = 0
		widget_control,state.wids.saved,set_value = $
		     strcompress(nsaved,/rem)
		if  nsaved GE 1 then widget_control,state.wids.plotmgr, $
		     /sensitive
		plotfile,state.dthcurrent,tmp,/nodata
		ncol = n_elements(tmp(*,0))
		widget_control,state.wids.ncol,set_value = $
                     strcompress(ncol,/rem)
		if widget_info(state.wids.xcol,/NAME) EQ 'DROPLIST' then begin
		  if not(keyword_set(state.coltitles)) then begin
		    coltitles = 'COl: '+$
			strcompress((sindgen(Ncol+1))(1:Ncol),/rem)
		    xpstr_update,state,coltitles=coltitles
		  endif
		  if n_elements(state.coltitles) NE Ncol then begin
		    coltitles = (sindgen(Ncol+1))(1:Ncol)
		    xpstr_update,state,coltitles=coltitles
		  endif
		  widget_control,state.wids.xcol,set_value=state.coltitles, $
			set_droplist_sel=state.xpdt.xcol-1
		  widget_control,state.wids.ycol,set_value=state.coltitles, $
			set_droplist_sel=state.xpdt.ycol-1
		endif else begin
		  widget_control,state.wids.xcol,set_value = $
		     strcompress(state.xpdt.xcol,/rem)
		  widget_control,state.wids.ycol,set_value = $
		     strcompress(state.xpdt.ycol,/rem)
		endelse

		END
	'RELOADFILE': 	BEGIN
		if state.fileinf.name EQ '<none>' or $
		   state.fileinf.name EQ '' then begin
		   itmp = widget_message(/Error,dialog_parent=event.top,$
		    'Xplot data came from variable, not from file.')
		  goto,out
		endif
		widget_control,state.wids.message,set_value=$
		 'Xplot message: Reloading '+state.fileinf.name(0)+'...'
		specon = state.spec.on
		filename = state.fileinf.name
		widget_control,event.top,set_uvalue=state,/no_copy
		if specon then begin
		  xplot_loadfile,event.top,SPEC=filename
		endif else begin
		  xplot_loadfile,event.top,filename
		endelse
		return
		END
	'PRINT': 	BEGIN
		; initialize printing buffer
		printbuffer = '  '
		; load printing buffer
		xplot_plot,state,printbuffer=printbuffer
		; print it
		xprint,BUFFER=printbuffer,GROUP=event.top,/COLOR
		END
	'EDITDATA':	BEGIN
                if not(keyword_set(state.dthcurrent)) then begin
                  tmp = widget_message(dialog_parent=event.top,$
                        'No data in current set',/ERROR)
                  goto,out
                endif
                plotfile,state.dthcurrent,tmpset,/nodata
		ycol = state.xpdt.ycol-1
		Csel = [ycol,ycol]
		if (state.xpdt.zoomsel(0) ne state.xpdt.zoomsel(1)) and $
		   (state.xpdt.zoomsel(2) ne state.xpdt.zoomsel(3)) then begin
		 xcol = state.xpdt.xcol-1
		 xmin = min([state.xpdt.zoomsel(0),state.xpdt.zoomsel(1)])
		 xmax = max([state.xpdt.zoomsel(0),state.xpdt.zoomsel(1)])
		 xi0 = where(tmpset(xcol,*) LT xmin)  & xi0 = n_elements(xi0)
		 xie = where(tmpset(xcol,*) LT xmax)  & xie = n_elements(xie)
		 Rsel=[xi0,xie-1]
		endif else Rsel=0
		action=''
		xtable,tmpset,group=event.top,Wtitle='Editing Xplot data',$
		  Rsel=Rsel,Csel=Csel,Action=action
		if action EQ 'Cancel' then goto,out
		widget_control, event.top, SET_UVALUE = state, /NO_COPY
		xplot_loadfile,event.top,tmpset
		return
		END
	'SAVEAS':	BEGIN
		if not(keyword_set(state.dthcurrent)) then begin
		  tmp = widget_message(dialog_parent=event.top,$
			'Nothing to save',/ERROR)
		  goto,OUT
		endif
		plotfile,state.dthcurrent,/nodata,tmpset_pre
		tmpset = tmpset_pre([state.xpdt.xcol-1,state.xpdt.ycol-1],*)
		;
		ffile = { ffile:'xplot.dat', $
			save: ['0','Current data','All columns'] }
		XscrMenu,ffile,TITLES=['Output file name:','Data to save:'], $
		  /NOTYPE,WTITLE='Save as...',GROUP=event.top,ACTION=action, $
		  /INTER,FIELDLEN=30,HELP=xplot_text('SAVEAS')
		if action eq 'DONT' then goto,out
		if checkfile(ffile.ffile) EQ 1 then begin
		  tmp = widget_message(dialog_parent=event.top,$
		    /CANCEL,/DEFAULT_CANCEL,$
		   ['File '+ffile.ffile+' already exists.', 'Overwrite it?'])
		  if tmp EQ 'Cancel' then goto,out
		endif
		openw,Unit,ffile.ffile,/GET_LUN
		if ( strcompress(ffile.save(0),/REM) eq '0') then $
		  printf,Unit,tmpset else begin
		  npoints = n_elements(tmpset_pre(0,*))
		  ncol = n_elements(tmpset_pre(*,0))
		  format = '('+string(ncol)+'(G0.6," "))'
		  printf,Unit,format=format,tmpset_pre
		endelse
		free_lun,Unit
                widget_control,state.wids.message,set_value = $
                  'Xplot message: File '+ffile.ffile+' written to disk.'
		;tmp = widget_message(dialog_parent=event.top,/INFO, $
		; 'File '+ffile.ffile+' written to disk.')
		END
	'SAVEIDL': 	BEGIN
		gfile = pickfile(FILE='xplot.bck',$
		title='Select a file for writing Xplot backup')
                if (gfile eq '') then goto,out
		if keyword_set(state.buffer) then $
		  if (n_elements(state.buffer) GT 10) then $
		    tmp = widget_message(dialog_parent=event.top,$
			['Xplot MAY fail in restoring a',$
			' Xplot backup file when the number of saved sets', $
			' is greater than 10.'])
                xplot_savestate,state,FILENAME=gfile ;.gfile
                widget_control,state.wids.message,set_value = $
                  'Xplot message: Backup file '+gfile+' written to disk.'
		;tmp = widget_message(dialog_parent=event.top,/INFO, $
		; 'File '+gfile+' written to disk.')
		END
	'SETLIMITS':	BEGIN
		tmp = state.xpdt_limits
		XscrMenu,tmp,/NOTYPE,GROUP=event.top,ACTION=action, $
		  WTITLE='Xplot limits',/INTERP,/APPLY,NCOL=2, TITLES=$
		  ['X min','X max','X style:','Y min','Y max','Y style'],$
		  help=xplot_text('LIMITS')
		if action eq 'DONT' then goto,out
		state.xpdt_limits = tmp
		state.xpdt.sysvar(2) = '!x.range = [' + $
			strcompress(state.xpdt_limits.Xmin,/rem)+','+ $
			strcompress(state.xpdt_limits.Xmax,/rem)+']'
		state.xpdt.sysvar(3) = '!y.range = [' + $
			strcompress(state.xpdt_limits.Ymin,/rem)+','+ $
			strcompress(state.xpdt_limits.Ymax,/rem)+']'
		tmp = fix( state.xpdt_limits.Xstyle(0) )
		case tmp of
		    0: state.xpdt.sysvar(4) = '!x.style = 1'
		    1: state.xpdt.sysvar(4) = '!x.style = 2'
		    2: state.xpdt.sysvar(4) = '!x.style = 4'
		    3: state.xpdt.sysvar(4) = '!x.style = 8'
		    4: state.xpdt.sysvar(4) = '!x.style = 16'
		endcase
		tmp = fix( state.xpdt_limits.Ystyle(0) )
		case tmp of
		    ;0: state.xpdt.sysvar(5) = '!y.style = 0'
		    0: state.xpdt.sysvar(5) = '!y.style = 1'
		    1: state.xpdt.sysvar(5) = '!y.style = 2'
		    2: state.xpdt.sysvar(5) = '!y.style = 4'
		    3: state.xpdt.sysvar(5) = '!y.style = 8'
		    4: state.xpdt.sysvar(5) = '!y.style = 16'
		endcase
		state.xpdt.zoom_on=0
		Xplot_plot,state
                ; autocall xplot_event when Apply button is pressed
                if action eq 'DO_APPLY' then begin
                  widget_control, event.top, SET_UVALUE = state, /NO_COPY
                  xplot_event,event
                  return
                endif

		END
        'SETLIMITSDEF':    BEGIN
                state.xpdt_limits.Xmin = 0.0
                state.xpdt_limits.Xmax = 0.0
                state.xpdt_limits.Ymin = 0.0
                state.xpdt_limits.Ymax = 0.0

                state.xpdt.sysvar(2) = '!x.range = [0.0,0.0]'
                state.xpdt.sysvar(3) = '!y.range = [0.0,0.0]'
		state.xpdt.zoom_on=0
		Xplot_plot,state
		END
        'SETLIMITSZOOM':    BEGIN
                state.xpdt_limits.Xmin = state.xpdt.zoomsel(0)
                state.xpdt_limits.Xmax = state.xpdt.zoomsel(1)
                state.xpdt_limits.Ymin = state.xpdt.zoomsel(2)
                state.xpdt_limits.Ymax = state.xpdt.zoomsel(3)
                  state.xpdt.sysvar(2) = '!x.range = [' + $
                        strcompress(state.xpdt.zoomsel(0),/rem)+','+ $
                        strcompress(state.xpdt.zoomsel(1),/rem)+']'
                  state.xpdt.sysvar(3) = '!y.range = [' + $
                        strcompress(state.xpdt.zoomsel(2),/rem)+','+ $
                        strcompress(state.xpdt.zoomsel(3),/rem)+']'
		state.xpdt.zoom_on=0
		Xplot_plot,state
		END
	'SETTITLES':	BEGIN
		tmp = state.xpdt_titles
		XscrMenu,tmp,/NOTYPE,GROUP=event.top, $
		  WTITLE='Xplot titles',ACTION=action,FIELDLEN=80,$
		  HELP=xplot_text('TITLES'),/APPLY_BUTTON
		if action EQ 'DONT' then goto,out
		state.xpdt_titles = tmp
		Xplot_plot,state
		if action eq 'DO_APPLY' then begin
		  widget_control, event.top, SET_UVALUE = state, /NO_COPY
		  xplot_event,event
		  return
		endif
		END
	'PLTNAME_CUR':	BEGIN
		tmp = state.plotname_current
		tmpstr = {tmp:tmp}
		XscrMenu,tmpstr,/NOTYPE,GROUP=event.top, FIELDLEN=84, $
		  WTITLE='Edit name of the current set',ACTION=action,$
		  TITLES='Please edit the names for the current sets'
		if action EQ 'DONT' then goto,out
		state.plotname_current = tmpstr.tmp
		END
	'PLTNAME_SAV':	BEGIN
		tmp = state.plotname_buffer
		if not(keyword_set(tmp)) then begin
		  itmp = widget_message(/ERROR,dialog_parent=event.top,$
		    'No saved plots to edit.')
		  goto,out
		endif
		tmpstr = {tmp:tmp}
		XscrMenu,tmpstr,/NOTYPE,GROUP=event.top, FIELDLEN=84, $
		  WTITLE='Edit names of the saved set',ACTION=action,$
		  TITLES='Please edit the names for the saved sets'
		if action EQ 'DONT' then goto,out
		state.plotname_buffer = tmpstr.tmp
		END
	'SETCOLTITLES':	BEGIN
		if state.spec.on EQ 1 then begin 
                 labels = state.spec.labels 
                  junk = strparse(labels,'!',tmp)
                  if junk EQ -1 then begin
                    junk = widget_message(dialog_parent=event.top,$
			'Error (column separator)',/ERROR)
                    goto,out
                  endif
		endif else begin
		  if not(keyword_set(state.coltitles)) then begin
		    junk = widget_message(dialog_parent=event.top,$
			['Option only available',$
			'when starting xplot ',	"with the 'COLTILES'", $
			'keyword, or when ','using SPEC files'])
		    goto,OUT
		  endif
		  ; MSR debugging 23 Apr 96:
		  tmp = state.coltitles
		endelse
		XscrMenu,tmp,/NOTYPE,GROUP=event.top, FIELDLEN=84, $
		  WTITLE='Xplot column titles',ACTION=action,$
		  help=xplot_text('COLTITLES')
		if action EQ 'DONT' then goto,OUT
		if (state.spec.on) EQ 1 then begin
		  labels = tmp(0)
                  for i=1,n_elements(tmp)-1 do $
                        labels = labels + '!' + tmp(i)
                  state.spec.labels = labels
                  tmp=strcompress((sindgen(n_elements( $
                    tmp)+1))(1:n_elements(tmp)) ) + ': '+tmp
                  widx = state.wids.xcolSPEC
                  widy = state.wids.ycolSPEC
		endif else begin
		  state.coltitles = tmp
		  widx = state.wids.xcol
		  widy = state.wids.ycol
		endelse
		widget_control,widx,SET_VALUE=tmp, $
		  set_droplist_select=state.xpdt.xcol-1
		widget_control,widy,SET_VALUE=tmp, $
		  set_droplist_select=state.xpdt.ycol-1
		Xplot_plot,state
		END
        'ANNOTATE':    BEGIN
		text = xplot_text('ANNOTATE')
		tmp = widget_message(dialog_parent=event.top,$
		  text,/CANCEL,/DEFAULT_CANCEL)
		if tmp EQ 'Cancel' then goto,out
		annotate
                END

	'SHOWDATAXY':	BEGIN
                plotfile,state.dthcurrent,/nodata,tmpset_pre
                tmpset = tmpset_pre([state.xpdt.xcol-1,state.xpdt.ycol-1],*)
		Xdisplayfile1,text=tmpset,GROUP=event.top,title='Xplot XY data.'
		END
	'SHOWDATASET':	BEGIN
                plotfile,state.dthcurrent,/nodata,tmpset
		Xdisplayfile1,text=tmpset,GROUP=event.top, $
		  title='Xplot set data.'
		END
	'SHIFT':	BEGIN
		tmp = { Horizontal:state.xpdt.shift(0), $
			Vertical:state.xpdt.shift(1) }
		XscrMenu,tmp,GROUP=event.top,/NOTYPE,ACTION=action, $
		  WTITLE='Xplot shift',HELP=xplot_text('SHIFT'),$
		  /APPLY_BUTTON,FIELDLEN=20
		if action eq 'DONT' then goto,out
		state.xpdt.shift = [tmp.Horizontal,tmp.Vertical]
		xplot_plot,state
                if action eq 'DO_APPLY' then begin
                  widget_control, event.top, SET_UVALUE = state, /NO_COPY
                  xplot_event,event
                  return
                endif
		END
	'DRAWDIM':	BEGIN
		tmp = { Horizontal:!d.x_size, $
			Vertical:!d.y_size }
		XscrMenu,tmp,GROUP=event.top,/NOTYPE,ACTION=action, $
		  WTITLE='Xplot Drawing size (pixels)',$
		  HELP=xplot_text('DRAWDIM')
		if action eq 'DONT' then goto,out
		widget_control,state.wids.draw,xsize=tmp.Horizontal, $
			ysize=tmp.Vertical
		xplot_plot,state
		END
	'COLPANEL': BEGIN
		if state.spec.on then begin
		  tmp = widget_message(dialog_parent=event.top,/ERROR, $
			'This option is not allowed with SPEC data')
		  goto,out
		endif
		wkind = widget_info(state.wids.xcol,/NAME)
		plotfile,state.dthcurrent,tmp,/NODATA
		ncol = n_elements(tmp(*,0))
		if wkind EQ 'TEXT' then begin
		  if not(keyword_set(state.coltitles)) then begin
		    coltitles = (sindgen(Ncol+1))(1:Ncol)
		    xpstr_update,state,coltitles= coltitles
		  endif
		  if ncol NE n_elements(state.coltitles) then begin
			tmp = widget_message(dialog_parent=event.top,/ERROR, $
			 'Number of columns not equal to stored labels')
			goto,out
		  endif
		  widget_control,state.wids.xcol,/DESTROY
		  widget_control,state.wids.ycol,/DESTROY
		  new1 = widget_droplist(state.wids.xcolBase, $
			value=state.coltitles,UVALUE='XCOL')
		  new2 = widget_droplist(state.wids.ycolBase, $
			value=state.coltitles,UVALUE='YCOL')
		  widget_control,new1,set_droplist_sel=state.xpdt.xcol-1
		  widget_control,new2,set_droplist_sel=state.xpdt.ycol-1
		  state.wids.xcol = new1
		  state.wids.ycol = new2
		endif else begin
		  widget_control,state.wids.xcol,/DESTROY
		  widget_control,state.wids.ycol,/DESTROY
		  new1 = widget_text(state.wids.xcolBase, $
			value=strcompress(state.xpdt.xcol,/Rem), $
			XSIZE=4,YSIZE=1,/EDIT,/FRAME,UVALUE='XCOL')
		  new2 = widget_text(state.wids.ycolBase, $
			value=strcompress(state.xpdt.ycol,/Rem), $
			XSIZE=4,YSIZE=1,/EDIT,/FRAME,UVALUE='YCOL')
		  state.wids.xcol = new1
		  state.wids.ycol = new2
		endelse
		END
	'SECONDAXIS':	BEGIN
		tmp = state.xpdt.axis2
		tmpstr = { Xmin:tmp(0), Xmax:tmp(1), Ymin:tmp(2), Ymax:tmp(3)}
		XscrMenu,tmpstr,GROUP=event.top,/NOTYPE,ACTION=action, $
		  WTITLE='Xplot 2nd axis limits',HELP=xplot_text('SECONDAXIS')
		if action ne 'DO' then goto,out
		state.xpdt.axis2 = $
			[tmpstr.Xmin,tmpstr.Xmax,tmpstr.Ymin,tmpstr.Ymax]
		xplot_plot,state
		END
	'MESH':	BEGIN
		widget_control,state.wids.saved,get_value=tmp
		if strcompress(tmp,/rem) NE '0' then begin
		  itmp = widget_message(dialog_parent=event.top, $
			/error,'MESH option not available'+$
		      ' with Saved sets.')
		  goto,out
		endif
		tmp = state.xpdt.mesh
		XscrMenu,tmp,GROUP=event.top,/NOTYPE,ACTION=action, $
		  WTITLE='Mesh values',FIELDLEN=40,/INTERP,/APPLY_BUTTON,$
		  TITLES=$
		  ['Mesh plot:','Kind','Split plots','N for splitting',$
		   'Column with mesh label values',$
		   'Additional keywords'],HELP=xplot_text('MESH'), $
		  FLAGS=['1','w(0) EQ 1','w(0) EQ 1','w(0) EQ 1 and w(2) EQ 0',$
			'w(0) EQ 1 and w(2) EQ 1','w(0) EQ 1']
		if action eq 'DONT' then goto,out
		state.xpdt.mesh = tmp
		xplot_plot,state
		; autocall xplot_event when Apply button is pressed
		if action eq 'DO_APPLY' then begin
		  widget_control, event.top, SET_UVALUE = state, /NO_COPY
		  xplot_event,event
		  return
		endif
		END
	'SYSVAR':	BEGIN
		widget_control, event.id, GET_VALUE=eventval
		TEXT=['This is an advanced option which use ', $
		'requires a deep knowdledge of IDL.', $
		'Its use is not recommended for the general user.', $
		'   ',$
		'Some of the modifications done here will remain ', $
		'after quiting xplot']
		tmp = widget_message(text,/CANCEL,/DEFAULT_CANCEL,$
			DIALOG_PARENT=event.top)
		if tmp EQ 'Cancel' then goto,out
		case eventval of
                   '!x': BEGIN
			tmp = !x
			XscrMenu,tmp,GROUP=event.top,NCOL=4,ACTION=action
			if action eq 'DONT' then goto,out
			!x = tmp
			xplot_plot,state
			END
                   '!y': BEGIN
			tmp = !y
			XscrMenu,tmp,GROUP=event.top,NCOL=4,ACTION=action
			if action eq 'DONT' then goto,out
			!y = tmp
			xplot_plot,state
			END
                  '!p': BEGIN
			tmp = !p
			XscrMenu,tmp,GROUP=event.top,NCOL=4, $
		  	  ACTION=action, /NOTYPE
			if action eq 'DONT' then goto,out
			!p = tmp
			xplot_plot,state
			END
		endcase
		END
	'COLUMNOPER':	BEGIN
		if not(keyword_set(state.dthcurrent)) then begin
		  tmp = widget_message(dialog_parent=event.top,$
			'No data in current set',/ERROR)
		  goto,out
		endif
		plotfile,state.dthcurrent,tmpset,/nodata
		tmpstr = state.calcstr.xpcolumnoper
		tmpcol = 0
		ncol_old = n_elements(tmpset(*,0))
		xplot_columnoper,tmpset,str=tmpstr,res=tmpcol
		if not(keyword_set(tmpcol)) then goto,out
		if tmpcol EQ -1 then goto,out
		ncol = n_elements(tmpset(*,0))

		; if spec/mca is set, inset column instead of appending it.
		ncoladd=''
		if state.spec.on AND state.spec.mca then begin
		  ncol = ncol-2
		  tmpcol = tmpcol-2
		  ncoladd='+2' 
		  widget_control,state.wids.ycolMCA,set_value = $
			strcompress(tmpcol,/rem)
		endif
		widget_control,state.wids.ncol,set_value = $
			strcompress(ncol,/rem)+ncoladd

		if widget_info(state.wids.xcol,/NAME) EQ 'TEXT' then begin
		   widget_control,state.wids.ycol,set_value = $
		   strcompress(tmpcol,/rem) 
		endif else begin
		   tmptxt = ' '+$
		     strcompress(n_elements(state.coltitles)+1,/rem)+': '
		   tmp  = [state.coltitles,tmptxt+'Oper with columns']
		   if ncol_old NE ncol then begin
		     xpstr_update,state,coltitles=tmp
		     widget_control,state.wids.xcol,set_value=tmp
		     widget_control,state.wids.ycol,set_value=tmp
		   endif
		   widget_control,state.wids.ycol,set_droplist_select=tmpcol-1
		endelse
		state.xpdt.Ycol = tmpcol
		state.calcstr.xpcolumnoper = tmpstr


		; spec setup
		if state.spec.on EQ 1 then begin
		  labels = state.spec.labels ;+'!Column Oper'
		  ;state.spec.labels = labels
		  tmp = strparse(labels,'!',labelslist)
		  if tmp EQ -1 then begin
		    tmp = widget_message(dialog_parent=event.top,$
			'Error (column separator)', /ERROR)
		    goto,out
		  endif
		  if (ncol EQ ncol_old) then begin
		    labelslist(tmpcol-1) = 'Column Oper'
		  endif else begin
		    labelslist = [labelslist,'Column Oper']
		  endelse
		  labels = labelslist(0)
		  for i=1,n_elements(labelslist)-1 do $
			labels = labels + '!' + labelslist(i)
		  state.spec.labels = labels
                  labelslist=strcompress((sindgen(n_elements( $
                    labelslist)+1))(1:n_elements(labelslist)) ) + $
                    ': '+labelslist
      		  widget_control,state.wids.xcolSPEC,SET_VALUE=$
			labelslist
      		  widget_control,state.wids.ycolSPEC,SET_VALUE=$
			labelslist, SET_DROPLIST_SELECT = tmpcol-1
		endif

		handle_free,state.dthcurrent
		h1 = handle_create(value=tmpset)
		state.dthcurrent = h1
		xplot_plot,state
		widget_control,state.wids.message,set_value = $
		  'Xplot message: Current set has been modified '+$
		  "with the option 'Calculations/Operation with columns'"
		END
	'XPLOT':	XPlot,WTITLE= $
		'XPLOT '+xplot_version()+' -child of xplot-',$
			no_block=state.no_block
	'XPLOTCLONE':	BEGIN	
		state.cloned=1
		tmp = state
		XPlot,WTITLE='XPLOT '+xplot_version()+' -clone of xplot-',$
			state=tmp,WINDOW_SIZE=[!d.x_size,!d.y_size],$
			no_block=state.no_block
		END
	'IDLCOMM': myidlmacro,group=event.top
	'XLOADCT':	Xloadct,GROUP=event.top
	'FONT':	BEGIN
                  state.strcontrols.Font = event.index
		  xyouts,"'!"+strcompress(3+ $
			fix(state.strcontrols.Font),/REM)+"'"
		  xplot_plot,state
		END
	'WCALC':	scicalc,GROUP=event.top
	'HELPABOUT':		BEGIN
		tmp = widget_message(dialog_parent=event.top,$
		 xplot_text('HELPABOUT'),/info,title='About Xplot')
		END
	'HELPWWW':		BEGIN
		command = (xplot_defaults('www')).www
		text = ['Xplot Web page and manual will be charged',$
		 'with the command :',command,' ', $
		 'If you want to modify this call in your installation',$
		 'set the command:',$
		 "www.www='netscape etc' ",$
		 'in a file which must be pointed by the environment ',$
		 'variable XPLOT_DEFAULTS.','Then restart xplot.' ]
		tmp = widget_message(dialog_parent=event.top,$
			/CANCEL,text)
		if tmp EQ 'Cancel' then goto,out
		spawn,command
		END
	'HELP':		BEGIN
			widget_control, event.id, get_value=eventval
			xhelp,strlowcase(eventval),group=event.top
		END

;
; these are in the top button menu
;
	'REDRAW':	BEGIN
		Widget_control,state.wids.Message,set_value= 'Xplot message: '
		state.xpdt.zoom_on=0
		xplot_plot,state
		END
	'ZOOM':	BEGIN
		state.xpdt.zoom_on = 1
		;xrangeold = state.xpdt.sysvar(2)
		;yrangeold = state.xpdt.sysvar(3)
		;state.xpdt.sysvar(2) = '!x.range = '+ $
		;  vect2string(state.xpdt.zoomsel(0:1))
		;state.xpdt.sysvar(3) = '!y.range = '+ $
		;  vect2string(state.xpdt.zoomsel(2:3))
		xplot_plot,state
		;state.xpdt.sysvar(2) = xrangeold
		;state.xpdt.sysvar(3) = yrangeold
		;itmp = execute(xrangeold)
		;itmp = execute(yrangeold)
		Widget_control,state.wids.Message,set_value= $
		'Xplot message: '+$
		'Use ReDraw button to zoom back to the top level'
		END
;	'SETCURRENT':	BEGIN
;		;xtext, /CANCEL, FIELDTEXT='1', OUTTEXT = outtext, $
;		nn = 1L
;		xedit,nn,text=' ',title='Set Current...', ACTION=action, $
;		infotext = [' ', $
;		 'This option permits to modify an already saved set.', $
;		 ' ', $
;		 'It will move the already saved set to the current set.', $
;		 ' ', $
;		 'Please enter the saved set number to be moved: ']
;		if action EQ 'CANCEL' then goto,OUT
;		if  nn GT n_elements(state.buffer) then begin
;		  tmp = widget_message(dialog_parent=event.top,$
;			'There are not so many saved sets.',/ERROR)
;		  goto,out
;		endif
;		;print,'<><> selected buffer: ',state.buffer(long(outtext)-1)
;		;nlist = strparse(state.buffer(long(outtext)-1),',',list)
;		nlist = strparse(state.buffer(nn-1),',',list)
;		; 
;		; execute buffer
;		;
;		;print,'<><> executing : ','hh = '+list(1)
;		tmp = execute( 'hh = '+list(1))
;		for i=2,7 do begin
;		;print,'<><> executing : ',list(i)
;			tmp = execute( list(i) )
;		endfor
;		;print,'<><> executing : ',list(8)+','+list(9)
;		tmp = execute( list(8)+','+list(9) )
;		;
;		; saving values
;		;
;		state.xpdt.xcol = xcol
;		state.xpdt.ycol = ycol
;		state.xpdt.psym = psym
;		state.xpdt.linestyle = linestyle
;		state.xpdt.thick = thick
;		state.xpdt.sysvar(7) = '!p.color = '+strcompress(color,/rem)
;		state.xpdt.shift = shiftval
;
;		;
;		; removing from buffer
;		;
;		nsaved = n_elements(state.buffer)
;		;help,nsaved
;		if nsaved EQ 1 then $
;		  xpstr_update,state,/nobuffer,dthcurrent = hh else begin
;
;		  index = lindgen(nsaved)
;		  newindex = where( index NE nn-1)
;		  buffer = state.buffer( newindex)
;		  dthbuffer = state.dthbuffer( newindex )
;		  xpstr_update,state,buffer = buffer, $
;			dthbuffer = dthbuffer, dthcurrent = hh
;		endelse
;		;
;		; close control panel and update its values
;		;
;                if widget_info(state.wids.controls(0),/valid_id) then begin
;                     widget_control,state.wids.showcontrols, $
;				set_value='Show Controls'
;                     widget_control,state.wids.controls(0),/DESTROY
;                endif
;		state.strcontrols.symbol = psym
;		state.strcontrols.linestyle = linestyle
;		state.strcontrols.thickness = thick
;		state.strcontrols.color = color
;		;
;		; updating widgets
;		;
;		if nsaved EQ 1 then $
;			widget_control,state.wids.setcurrent,sensitive=0
;		plotfile,hh,tmp,/nodata
;		ncol = n_elements(tmp(*,0))
;		
;		if widget_info(state.wids.xcol,/NAME) EQ 'DROPLIST' then begin
;		  coltitles = (sindgen(Ncol+1))(1:Ncol)
;		  widget_control,state.wids.xcol, set_value = coltitles, $
;			set_droplist_select = xcol-1
;		  widget_control,state.wids.ycol, set_value = coltitles, $
;			set_droplist_select = ycol-1
;		endif else begin
;		  widget_control,state.wids.xcol, set_value = $
;			strcompress(xcol,/rem)
;		  widget_control,state.wids.ycol, set_value = $
;			strcompress(ycol,/rem)
;		endelse
;		widget_control,state.wids.ncol, set_value= $
;			strcompress(ncol,/rem)
;		widget_control,state.wids.saved, set_value= $
;			strcompress(nsaved-1,/rem)
;		widget_control,state.wids.message, set_value= $
;			'Xplot message:'
;		; switch out spec (if selected)
;		if state.spec.on EQ 1 then begin
;		  widget_control,state.spec.scanbase,/destroy, bad_id=bad
;		  state.spec.on = 0
;		  xplot_mappanels, state.wids, state.spec
;		endif
;		; re-plot
;		xplot_plot,state
;		END
	'CURSOR':	BEGIN
		widget_control,state.wids.draw,get_value=windownum
		wset,windownum
		!p=state.psysvar.p
		!x=state.psysvar.x
		!y=state.psysvar.y
		if state.no_block then  begin
                  itmp = execute(state.xpdt.sysvar(2))
                  itmp = execute(state.xpdt.sysvar(3))
		endif

		coords_d = [event.x,event.y]
		if (coords_d(0) GT !d.x_size or coords_d(0) LT 0 or $
		  coords_d(1) GT !d.y_size or coords_d(1) LT 0) then goto,out
		if keyword_set(state.dthcurrent) then begin
		  coords = convert_coord(event.x,event.y,/device,/to_data)
		endif else coords = coords_d
		widget_control,state.wids.cursor,SET_VALUE='X:'+$
			strcompress(coords(0),/REM)+' Y:'+$
			strcompress(coords(1),/REM)
                ;
                ; zoom
                ;
                if event.type EQ 0 then begin ;initialize zoom origin
		  xplot_plot,state
                  state.xpdt.zoomsel=[-1000,-1000,-1000,-1000]
                  state.xpdt.zoomsel(0)=coords_d(0)
                  state.xpdt.zoomsel(2)=coords_d(1)
		  ; this is needed for creating a pixmap an storing it
		  ; which contains the graph with no rubberband. This 
		  ; pixmap will be restored while dragging the cursor in
		  ; the zoom selection
		  wdelete,state.xpdt.zoompixmap
		  window,/free,xsize=!d.x_size,ysize=!d.y_size,/pixmap
		  device,copy=[0,0,!d.x_size,!d.y_size,0,0,windownum]
		  state.xpdt.zoompixmap = !d.window
		  goto,out
                endif
                if (state.xpdt.zoomsel(1) eq state.xpdt.zoomsel(3)) and $
                  (state.xpdt.zoomsel(1) EQ -1000)  then zoom_sel_flag = 1 $
                  else zoom_sel_flag = 0
                if zoom_sel_flag EQ 1 then begin  ; zoom rectangle
                  distx=abs(coords_d(0)-state.xpdt.zoomsel(0))
                  disty= abs(coords_d(1)-state.xpdt.zoomsel(2))
                  if distx NE 0.0 and disty NE 0.0 then begin
		    ; old zoom part
                    ;tvbox,min([coords_d(0),state.xpdt.zoomsel(0)]), $
                    ;min([coords_d(1),state.xpdt.zoomsel(2)]), distx,disty,-2
		    ; new zoom part
		    xd=coords_d(0)
		    yd=coords_d(1)
		    xs=state.xpdt.zoomsel(0)
		    ys=state.xpdt.zoomsel(2)
		    ;print,'start: ',xs,ys,' end: ',xd,yd
		    device,copy=$	
			[0,0,!d.x_size,!d.y_size,0,0,state.xpdt.zoompixmap]
		    plots,[xs,xs,xd,xd,xs],[ys,yd,yd,ys,ys],/device,linestyle=2
                  endif
                endif
                if ((event.type EQ 1) AND (zoom_sel_flag EQ 1)) then begin 
		;end zoom region
		  if keyword_set(state.dthcurrent) then begin
                    coords0 = convert_coord(state.xpdt.zoomsel(0),$
                      state.xpdt.zoomsel(2),/device,/to_data)
		  endif else coords0 = [state.xpdt.zoomsel(0),$
			state.xpdt.zoomsel(2)]
                  x0 = min([coords0(0),coords(0)])
                  y0 = min([coords0(1),coords(1)])
                  distx=abs(coords0(0)-coords(0))
                  disty= abs(coords0(1)-coords(1))
                  xrange = [x0,x0+distx] 
                  yrange = [y0,y0+disty]
                  state.xpdt.zoomsel=[x0,x0+distx,y0,y0+disty]
                  Widget_control,state.wids.Message,set_value= $
                  'Xplot message: Selection Limits: [x1,x2,y1,y2]='+ $
                  vect2string(state.xpdt.zoomsel)
                endif
		END
ENDCASE

OUT:
widget_control, event.top, SET_UVALUE = state, /NO_COPY

end ;xplot_event
;
;=======================================================================
;
PRO XPLOT, indata, indataopt, GROUP=group, WINDOW_SIZE=w_size, $
	Xcol = xcol, YCOL = ycol, QUIT_BUTTON=tmp, $
	WTITLE = wtitle, TITLE = title, XTITLE = xtitle, YTITLE = ytitle, $
	LEGEND = legend, STATE=pSTATE, $
	XRANGE=xrange, YRANGE=yrange, PARENT = wXplotWindow, $
	COLTITLES=coltitles, XLOG=xlog, YLOG=ylog, SPEC=spec, $
	NO_BLOCK=no_block
;
;
on_error,2

if keyword_set(group) then widget_control,/hourglass

;
; create data structures
;

;
; the data structure xpdt
;

;
; store the system variables that may be affected by xplot and reset when exit
;
xpdt = xplot_defaults('xpdt')
xpdt_limits = xplot_defaults('xpdt_limits')
xpdt_titles = xplot_defaults('xpdt_titles')

data = 0
;
;insert defaults
if n_elements(no_block) EQ 0  then no_block=0
if keyword_set(Xcol) 	then xpdt.Xcol = Xcol
if keyword_set(Ycol) 	then xpdt.Ycol = Ycol
if keyword_set(Wtitle) 	then Wtitle = Wtitle else Wtitle='Xplot '+$
		xplot_version()
if keyword_set(Title) 	then xpdt_titles.Title = Title
if keyword_set(Xtitle) 	then begin
  xpdt_titles.Xtitle = Xtitle
  if (Xtitle EQ '-1' and not(keyword_set(coltitles))) then coltitles=1
endif
if keyword_set(Ytitle) 	then begin
  xpdt_titles.Ytitle = Ytitle
  if (Ytitle EQ '-1' and not(keyword_set(coltitles))) then coltitles=1
endif


if keyword_set(xrange) 	then begin
  xpdt_limits.Xmin = xrange(0)
  xpdt_limits.Xmax = xrange(1)
  xpdt.sysvar(2) = '!x.range = [' + $
    strcompress(xpdt_limits.Xmin,/rem)+','+ $
    strcompress(xpdt_limits.Xmax,/rem)+']'
endif

if keyword_set(yrange) 	then begin
  xpdt_limits.Ymin = yrange(0)
  xpdt_limits.Ymax = yrange(1)
  xpdt.sysvar(3) = '!y.range = [' + $
    strcompress(xpdt_limits.Ymin,/rem)+','+ $
    strcompress(xpdt_limits.Ymax,/rem)+']'
endif

fileinf={name:'<none>',dir:''}
plotname='<variable>'
if keyword_set(indata) 	then begin
  if type(indata) eq 7 then begin
      plotname = indata
      fileinf.name=indata
		; MSR 97/02/10 This section is commented because if spec_acess 
		; is precompiled with xop.sav, this option does not work. 
		;
      ;if sdep() EQ 'WINDOWS' then tmpf='SPEC_A~1.DAT' else $
      ;  tmpf='spec_access.pro'
      ;if filepath1(tmpf) NE '' then $
      ;	spec_access_flag = spec_access(hspec, indata) else $
      ;  spec_access_flag = -1
      if not(keyword_set(spec)) then begin  ; ASCII input
	spec_access_flag = -1
	data = rascii(indata) 
                  if not(keyword_set(data)) then begin
		     tmp = widget_message(dialog_parent=group,$
			'Error opening file',/ERROR)
                     return
                  endif
	xpdt.infile=indata
      endif else begin     ; spec input
        spec_access_flag = spec_access(hspec, indata) 
        tmp = widget_message(dialog_parent=group,['SPEC input file.', $
		'Please use File/Load Data file option to store it'],/ERROR)
	indata = 0
      endelse
  endif else begin
    ; when calling xplot with two arrays (as plot) then collapse them in
    ; only one.
    if n_params() EQ 2 then begin
      print,'XPLOT: Using two input arrays.'
      data = fltarr(2,n_elements(indata))
      if  n_elements(indata) NE n_elements(indataopt) then begin
        print,'XPLOT: Error: the two arrays must be of the same dimension.'
        return
      endif
      data(0,*) = reform(indata)  &  data(1,*) = reform(indataopt)
    endif else data = indata
  endelse
  xpdt.Nfiles = 1
endif
;
;window
if n_elements(w_size) lt 2 then begin     ; default window size
  device, GET_SCREEN = w_size
  w_size = (w_size(0) * [5,4]) /9
endif

;
; widget definition
;
wXplotWindow = $
  Widget_base(title=Wtitle,MBAR=wMenuBar, $
	/TLB_SIZE_EVENTS)	; base window
wXplotBase = Widget_base(wXplotWindow,/Column)	; base widget

wXplotButtons   = Widget_base(wXplotBase,/Row)	; buttons
wXplotDraw = Widget_draw(wXplotBase, XSIZE=w_size(0), YSIZE=w_size(1), $
	/MOTION_EVENTS,UVALUE='CURSOR',RETAIN=2, $ ; /NO_COPY, $
	BUTTON_EVENTS=1)
;
; BUG IN IDL: If we use /frame here (as done up to xplot 2.5) the
; xplot window changes size when doing some actions, like pressing the
; REDRAW button. To avoid that, Do Not Use /frame
;
;wXplotInfo   = Widget_base(wXplotBase, ROW=1, /FRAME)	; buttons
wXplotInfo   = Widget_base(wXplotBase, ROW=1)	; buttons

if !version.release EQ '4.0' then $
  wXplotMessage   = Widget_label(wXplotBase,Value='Xplot message:', $
	/ALIGN_LEFT) else $
  wXplotMessage   = Widget_label(wXplotBase,Value='Xplot message:', $
	/ALIGN_LEFT,/DYNAMIC_RESIZE)

;
; build the menu bar
;

wFileMenu =  WIDGET_BUTTON(wMenuBar, VALUE='File', /MENU)
  wLoadFile=WIDGET_BUTTON(wFileMenu, VALUE='Load Data File...', $
	UVALUE='LOADFILE')
  wLoadFile=WIDGET_BUTTON(wFileMenu, VALUE='Re-Load Data File', $
	UVALUE='RELOADFILE')
  wSaveDataAs=WIDGET_BUTTON(wFileMenu, VALUE= $
	'Save current set as... (new ASCII file)', UVALUE='SAVEAS')
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Xplot backup File', /MENU)
  wSavePlotAs=WIDGET_BUTTON(wtmp, VALUE='Save current session...',  $
	UVALUE='SAVEIDL')
  wRestoreFile=WIDGET_BUTTON(wtmp, VALUE='Restore from file...', $
	UVALUE='RESTORE')
  wPrintPlot=WIDGET_BUTTON(wFileMenu, VALUE='Print Plot...', UVALUE='PRINT',$
	/SEPA)
  wExit=WIDGET_BUTTON(wFileMenu, VALUE='Quit', UVALUE='EXIT',/SEPA)

wEditMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Edit', /MENU)
  wClearAll=WIDGET_BUTTON(wEditMenu, VALUE='Clear All', UVALUE='CLEARDATA')
  wData=WIDGET_BUTTON(wEditMenu, VALUE='Data...', UVALUE='EDITDATA')
  wLimits=WIDGET_BUTTON(wEditMenu, VALUE='Limits', /MENU)
  wLimitsSet=WIDGET_BUTTON(wLimits, VALUE='Set...', UVALUE='SETLIMITS')
  wLimitsSel=WIDGET_BUTTON(wLimits, VALUE='Set Defaults', $
	UVALUE='SETLIMITSDEF')
  wLimitsSel=WIDGET_BUTTON(wLimits, VALUE='Set Limits as Selection Area', $
	UVALUE='SETLIMITSZOOM')

  wTitles=WIDGET_BUTTON(wEditMenu, VALUE='Titles...',UVALUE='SETTITLES')
  wColTitles=WIDGET_BUTTON(wEditMenu, VALUE='Column Titles...',$
	UVALUE='SETCOLTITLES')
  wLegend=WIDGET_BUTTON(wEditMenu, VALUE='Legend...',UVALUE='SETLEGEND')
  wLabels=WIDGET_BUTTON(wEditMenu, VALUE='Labels...',UVALUE='SETLABELS')

  wPlotNames=WIDGET_BUTTON(wEditMenu, VALUE='Plot Names',/MENU)
    wtmp=WIDGET_BUTTON(wPlotNames, VALUE='Current Set...',UVALUE='PLTNAME_CUR')
    wtmp=WIDGET_BUTTON(wPlotNames, VALUE='Saved Set...',UVALUE='PLTNAME_SAV')

  wShift=WIDGET_BUTTON(wEditMenu, VALUE='Shift data...', UVALUE='SHIFT')
  wErrPlot=WIDGET_BUTTON(wEditMenu, VALUE='Error bars...', UVALUE='ERRPLOT')
  wExeBuffer=WIDGET_BUTTON(wEditMenu, VALUE='Legend/Label buffer',/MENU)
  wExeBE=WIDGET_BUTTON(wExeBuffer, VALUE='Clean', UVALUE='EXEBUFFERCLEAN')
  wExeBC=WIDGET_BUTTON(wExeBuffer, VALUE='Edit...', UVALUE='EXEBUFFEREDIT')
  wSysVar=WIDGET_BUTTON(wEditMenu, VALUE='System Variables', /MENU)
    wSysVarx=WIDGET_BUTTON(wSysVar, VALUE='!x', UVALUE='SYSVAR')
    wSysVary=WIDGET_BUTTON(wSysVar, VALUE='!y', UVALUE='SYSVAR')
    wSysVarp=WIDGET_BUTTON(wSysVar, VALUE='!p', UVALUE='SYSVAR')

wViewMenu =  WIDGET_BUTTON(wMenuBar, VALUE='View', /MENU)
  wShowData=WIDGET_BUTTON(wViewMenu, VALUE='Show Data', /MENU)
    wShowDataXY=WIDGET_BUTTON(wShowData, VALUE= $
	'Plotted X Y columns (current set)', UVALUE='SHOWDATAXY')
    wShowDataSet=WIDGET_BUTTON(wShowData, VALUE= $
	'All columns of the current set', UVALUE='SHOWDATASET')
    wDrawDim=WIDGET_BUTTON(wViewMenu, VALUE='Draw window size...', $
	UVALUE='DRAWDIM')
    wColPanel=WIDGET_BUTTON(wViewMenu, VALUE='Change Column Panel', $
	UVALUE='COLPANEL')
    w2ndAxis=WIDGET_BUTTON(wViewMenu, VALUE='Second X/Y axis...', $
	UVALUE='SECONDAXIS')
    wMesh=WIDGET_BUTTON(wViewMenu, VALUE='Mesh plot...', $
	UVALUE='MESH')

wCalc=WIDGET_BUTTON(wMenuBar, VALUE='Calculations', /MENU)
  wWidthInt=WIDGET_BUTTON(wCalc,VALUE='Width/Integral/MinMax', $
	UVALUE='CALCULATIONS')
  wWidthInt=WIDGET_BUTTON(wCalc,VALUE='Moments', UVALUE='CALCULATIONS')
  wColumnOper=WIDGET_BUTTON(wCalc,VALUE='Operations with columns...', $
	UVALUE='COLUMNOPER')
  wEvaluation=WIDGET_BUTTON(wCalc,VALUE='Operations with sets...',$
	UVALUE='CALCULATIONS')
  wFouFil=WIDGET_BUTTON(wCalc,VALUE='Fourier Filtering...', $
	UVALUE='CALCULATIONS')
  wDerivative=WIDGET_BUTTON(wCalc,VALUE='Derivative',UVALUE='CALCULATIONS')
  wDerivative=WIDGET_BUTTON(wCalc,VALUE='CDF',UVALUE='CALCULATIONS')
  wDerivative=WIDGET_BUTTON(wCalc,VALUE='Smooth...',UVALUE='CALCULATIONS')
  wInterpolation=WIDGET_BUTTON(wCalc,VALUE='Interpolation...', $
	UVALUE='CALCULATIONS')
  wConvolution=WIDGET_BUTTON(wCalc,VALUE='Convolution & Correlation...',$
	UVALUE='CALCULATIONS')
  ;wCorrelation=WIDGET_BUTTON(wCalc,VALUE='Correlation...',$
	;UVALUE='CALCULATIONS')
  wFitting=WIDGET_BUTTON(wCalc,VALUE='Fitting', /MENU)
    wFitPoly=WIDGET_BUTTON(wFitting,VALUE='Polynomial&Gaussian...',UVALUE='CALCULATIONS')
    ;wFitGauss=WIDGET_BUTTON(wFitting,VALUE='Gaussian...',UVALUE='CALCULATIONS')
    wNLFit=WIDGET_BUTTON(wFitting,VALUE='Non linear...', $
	UVALUE='CALCULATIONS')
  ; MSR 97/02/10 This section is commented because if spec_acess 
  ; is precompiled with xop.sav, this option does not work. 
  ;
  ;if sdep() EQ 'WINDOWS' then tmpf='XAID_M~1.DAT' else $
  ;   tmpf='xaid_mu2chi.pro'
  ;if filepath1(tmpf) NE '' then begin
  wXAFS=WIDGET_BUTTON(wCalc,VALUE='XAFS', /MENU)
    wXcnpick=WIDGET_BUTTON(wXAFS,VALUE='XAFS extraction (xaid_mu2chi)', $
	UVALUE='CALCULATIONS')
    wXcnpick=WIDGET_BUTTON(wXAFS,VALUE='XAFS Fourier Filter (xaid_ff)', $
	UVALUE='CALCULATIONS')
  ;endif
wToolsMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Tools', /MENU)
  ;wXloadct=WIDGET_BUTTON(wToolsMenu, VALUE='Xloadct', UVALUE='XLOADCT')
  wAnnotate=WIDGET_BUTTON(wToolsMenu, VALUE='Annotate...',UVALUE='ANNOTATE')
  wCalculator=WIDGET_BUTTON(wToolsMenu, VALUE='Calculator', UVALUE='WCALC')
  wXplot=WIDGET_BUTTON(wToolsMenu, VALUE='Xplot (Empty)', UVALUE='XPLOT')
  wXplotClone=WIDGET_BUTTON(wToolsMenu, VALUE='Xplot (Clone)', $
	UVALUE='XPLOTCLONE')
  wIdlCommand=WIDGET_BUTTON(wToolsMenu, VALUE='IDL macro', UVALUE='IDLCOMM')

wHelpMenu = WIDGET_BUTTON(wMenuBar, VALUE='Help', /HELP)
  wHelptmp = WIDGET_BUTTON(wHelpMenu, VALUE='About Xplot', UVALUE='HELPABOUT')
  wHelptmp = WIDGET_BUTTON(wHelpMenu, VALUE='Xplot', UVALUE='HELP')
  wHelptmp = WIDGET_BUTTON(wHelpMenu, VALUE='Xplot WWW manual', $
	UVALUE='HELPWWW')
  wHelpRoutines = WIDGET_BUTTON(wHelpMenu, VALUE='Math Routines used by Xplot',/MENU)
  n=strparse((xplot_defaults('math_routines')).math_routines,':',mylist)
  for i=0,n do $
    wHelptmp = WIDGET_BUTTON(wHelpRoutines, VALUE=mylist(i), UVALUE='HELP')
;
; build the buttons panel
;

wB0 = widget_base(wXplotButtons,/Row)

wB1 = widget_base(wB0,/ROW,/FRAME)
wBquit = widget_button(wB1, VALUE="Quit",UVALUE='EXIT')
wBredraw =  widget_button(wB1, VALUE="ReDraw",UVALUE='REDRAW')
wBsave =  widget_button(wB1, VALUE="Save Plot",UVALUE='SAVE')
wBplotmgr =  widget_button(wB1, VALUE="Plot Mgr",UVALUE='PLOTMGR')
;wBsetcurrent =  widget_button(wB1, VALUE="Set Current",UVALUE='SETCURRENT')
;wBzoomsel = cw_bgroup(wB1, "Zoom Sel",UVALUE='ZOOMSEL')
wBzoom = cw_bgroup(wB1, "Zoom",UVALUE='ZOOM')
wBcontrols = widget_button(wB1, VALUE="Show Controls",UVALUE='CONTROLS')
wBcursor = widget_text(wB0,/FRAME,XSIZE=25)


;
; Info (bottom) line
;
if keyword_set(data) then begin
  if n_elements(data) EQ 1 then plotfile,data,tmp,/nodata else tmp=data
  Ncol = n_elements(tmp(*,0)) 
  if keyword_set(Ycol) then xpdt.Ycol = Ycol else xpdt.Ycol = Ncol
  if keyword_set(Xcol) then xpdt.Xcol = Xcol else xpdt.Xcol = 1
  ;current_set = '1'
endif else begin
  Ncol=0
  ;current_set = '0'
endelse

if keyword_set(coltitles) then begin
  if n_elements(coltitles) EQ 1 then coltitles=['<none>']
  if keyword_set(indata) then begin
    if n_elements(coltitles) EQ 1 then begin
	coltitles= (sindgen(Ncol+1))(1:Ncol)
      ;print,'Im now here'
      ;help, n_elements(coltitles)
    endif
    if n_elements(coltitles) NE Ncol then tmp = widget_message($
	dialog_parent=group,$
	'The number of column titles is different to the column number.')
  endif
endif else coltitles=0
;====================================================================
wBsetBase = widget_base(wXplotInfo,/Row,/align_center)
 wBsetsaved = widget_label(wBsetBase,value=strcompress(0,/rem))
 wBset_tmp =  widget_label(wBsetBase,value=' saved sets. Current set has ')
 wBsetncol = widget_label(wBsetBase,value=strcompress(Ncol,/REM))
 wBset_tmp = widget_label(wBsetBase,value=' columns,')
wBsetcol = widget_base(wXplotInfo)
 wBsetcolASCII = widget_base(wBsetcol,/Row, /align_center, MAP=1)
  wBset_tmp = widget_label(wBsetcolASCII,value=' X: ')
  if not(keyword_set(coltitles)) then begin
    wBsetBxcol = widget_base(wBsetcolASCII)
    wBsetxcol = widget_text(wBsetBxcol,value= $
      strtrim(string(xpdt.Xcol),2),XSIZE=4,YSIZE=1,/EDIT,/FRAME,UVALUE='XCOL')
    wBset_tmp = widget_label(wBsetcolASCII,value=' Y: ')
    wBsetBycol = widget_base(wBsetcolASCII)
    wBsetycol = widget_text(wBsetBycol,value= $
      strtrim(string(xpdt.Ycol),2),XSIZE=4,YSIZE=1,/EDIT,/FRAME,UVALUE='YCOL')
  endif else begin
    wBsetBxcol = widget_base(wBsetcolASCII)
    wBsetxcol = widget_droplist(wBsetBxcol,value= coltitles, UVALUE='XCOL')
    wBset_tmp = widget_label(wBsetcolASCII,value=' Y: ')
    wBsetBycol = widget_base(wBsetcolASCII)
    wBsetycol = widget_droplist(wBsetBycol,value= coltitles, UVALUE='YCOL')
  endelse

 wBsetcolSPEC = widget_base(wBsetcol,/Row,MAP=0)
  wBset_tmp0 = widget_base(wBsetcolSPEC,MAP=1)
   wBsetcolNormal = widget_base(wBset_tmp0,MAP=1, row=1)
    wBset_tmp = widget_label(wBsetcolNormal,value=' X:')
    if !version.release EQ '4.0' then $
      wBsetxcolSPEC = widget_droplist(wBsetcolNormal,value='<none>', $
	UVALUE='XCOL') else $
      wBsetxcolSPEC = widget_droplist(wBsetcolNormal,value='<none>', $
	UVALUE='XCOL',/DYNAMIC_RESIZE)
    wBset_tmp = widget_label(wBsetcolNormal,value=' Y:')
    if !version.release EQ '4.0' then $
      wBsetycolSPEC = widget_droplist(wBsetcolNormal,value='  <none>  ', $
	UVALUE='YCOL') else $
      wBsetycolSPEC = widget_droplist(wBsetcolNormal,value='  <none>  ', $
	UVALUE='YCOL',/DYNAMIC_RESIZE)
   wBsetcolMCA = widget_base(wBset_tmp0,MAP=0, row=1)
    wBsetxmca = cw_bgroup(wBsetcolMCA, ['Channel','Energy'], $
                      ROW=1, /EXCLUSIVE, UVALUE='XCOL', set_value=0, /frame)
    wBset_tmp = widget_label(wBsetcolMCA,value=' Y: ')
    wBsetymca = widget_text(wBsetcolMCA,value='0', $
                                   XSIZE=4,YSIZE=1,/EDIT,/FRAME,UVALUE='YCOL')
  wBset_tmp0 = widget_button(wBsetcolSPEC,value= 'Scan sel...', UVALUE='SSCAN')

;====================================================================
;
; activate widgets
;
widget_control, wXplotWindow, /NO_COPY, /REALIZE
widget_control, wBplotmgr, SENSITIVE=0

if keyword_set(data) then begin
  if type(data) NE 3 then begin
    h1 = handle_create(value=data)
    ;handle_value, h1, data, /set
    ;if handle_info(h1) then print,'Handle data -> handle h1=',h1
    dthcurrent=h1
  endif else dthcurrent=data
endif else dthcurrent = 0L


strcontrols = xplot_defaults('strcontrols')
if keyword_set(Xlog) then begin
   xpdt.kind = 4
   strcontrols.axis=2
endif
if keyword_set(Ylog) then begin
   xpdt.kind = 3
   strcontrols.axis=1
endif
if (keyword_set(Xlog) AND keyword_set(Ylog)) then begin
   xpdt.kind = 5
   strcontrols.axis=3
endif
;current_set = '1'

;
; create the state structure
;
;structure to store the widget ids to be change during the
;running time
wids = { XPWIDS, main:wXplotWindow, cursor:wBcursor, $
        asciipanel:wBsetcolASCII, specpanel:wBsetcolSPEC, $
        normalpanel:wBsetcolNormal, mcapanel:wBsetcolMCA, $
        saved:wBsetsaved, ncol:wBsetncol, $
	xcolBase:wBsetBxcol,ycolBase:wBsetBycol, $
	xcol:wBsetxcol,ycol:wBsetycol, $
        xcolspec:wBsetxcolSPEC, ycolspec:wBsetycolSPEC, $
        xcolmca:wBsetxmca, ycolmca:wBsetymca, $
        plotmgr:wBplotmgr, showcontrols:wBcontrols, $
        controls:[0L,0L,0L,0L], message:wXplotMessage, draw:wXplotDraw}

xpevaluate = xplot_defaults('XPevaluate')
xpcolumnoper = xplot_defaults('XPcolumnoper')
xpconvolution = xplot_defaults('XPconvolution')
;xpcorrelation = xplot_defaults('XPcorrelation')
calcstr = { xpevaluate:xpevaluate, xpcolumnoper:xpcolumnoper, $
	    xpconvolution:xpconvolution}

if keyword_set(Legend) 	then exebuffer = Legend else exebuffer = 0

spechandle = handle_create(value=0)
specstr = {on:0, handle:spechandle, mca:0, scan:0, scanbase:0L, labels:''}

; ************************* VERY IMPORTANT ***************************
; note: when updating state, update also xpstr_update!!!
; ************************* VERY IMPORTANT ***************************
;spechandle = handle_create(value=0)
version = float(xplot_version())
state = { wids:wids, calcstr:calcstr, $
  xpdt:xpdt, xpdt_limits:xpdt_limits, $
  xpdt_titles:xpdt_titles, buffer:0, $ 
  dthcurrent:dthcurrent, dthbuffer:0L, strcontrols:strcontrols, $
  spec:specstr, exebuffer:exebuffer, version:version, cloned:0, $
  coltitles:coltitles, fileinf:fileinf, psysvar:{p:!p,x:!x,y:!y}, $
  no_block:no_block,plotname_buffer:0,plotname_current:plotname}
; ********************************************************************

xplot_mappanels, state.wids, state.spec

if keyword_set(indata) then state1 = state
if keyword_set(pstate) then state1 = state

;if keyword_set(spec) then xplot_loadfilespec,state,spec
if keyword_set(spec) then begin
  widget_control, wXplotWindow, set_uvalue=state, /NO_COPY
  xplot_loadfile,wXplotWindow,spec=spec
  widget_control, wXplotWindow, get_uvalue=state, /NO_COPY
endif

;<><>widget_control, wXplotWindow, set_uvalue=state, /NO_COPY

if keyword_set(coltitles) then begin
  widget_control,wBsetxcol,set_droplist_select=xpdt.xcol-1
  widget_control,wBsetycol,set_droplist_select=xpdt.ycol-1
endif

if keyword_set(pstate) then BEGIN
  print,'<><> XPLOT: clonning..'
  wids = state1.wids
  widget_control,pstate.wids.saved,get_value=saved
  widget_control,pstate.wids.ncol,get_value=ncol
  widget_control,pstate.wids.xcol,get_value=xcol
  widget_control,pstate.wids.ycol,get_value=ycol
  widget_control,pstate.wids.message,get_value=message
  widget_control,pstate.wids.xcolmca,get_value=xcolmca
  widget_control,pstate.wids.ycolmca,get_value=ycolmca


  state1 = pstate
  state1.wids = wids
  widget_control,state1.wids.saved,set_value=saved
  widget_control,state1.wids.ncol,set_value=ncol
  widget_control,state1.wids.xcol,set_value=xcol
  widget_control,state1.wids.ycol,set_value=ycol
  widget_control,state1.wids.message,set_value=message
  widget_control,state1.wids.xcolmca,set_value=xcolmca
  widget_control,state1.wids.ycolmca,set_value=ycolmca

  if state1.spec.on then begin
    xplot_mappanels, state1.wids, state1.spec
    tmp = strparse(state1.spec.labels,'!',labelslist)
    widget_control,state1.wids.xcolspec,set_value=labelslist, $
	set_droplist_select=state1.xpdt.xcol-1
    widget_control,state1.wids.ycolspec,set_value=labelslist, $
	set_droplist_select=state1.xpdt.ycol-1
  endif
  
  widget_control,state1.wids.saved,get_value=saved
  if fix(saved) GE 1 then $
      widget_control,state1.wids.plotmgr,/sensitive
  xplot_plot,state1
  ;widget_control, wXplotWindow, set_uvalue=state1, /NO_COPY
ENDIF

IF keyword_set(indata) then BEGIN
  ;print,'XPLOT: call to xplot_plot from main program'
  xplot_plot,state1
  state.psysvar = state1.psysvar
ENDIF
;<><>
if keyword_set(pstate) then $
  widget_control, wXplotWindow, set_uvalue=state1, /NO_COPY else $
  widget_control, wXplotWindow, set_uvalue=state, /NO_COPY 

if no_block then $
  xmanager, 'xplot', wXplotWindow, GROUP_LEADER=group,/no_block else $
  xmanager, 'xplot', wXplotWindow, GROUP_LEADER=group

;handle_free, spechandle
end
