
;+
;	NAME:
;		XNLFIT
;	PURPOSE:
;		Widget interface to non-linear fits (curvefit &
;		powellfit).
;	CALLING SEQUENCE:
;		xnlfit,set (or xnlfit,x,y)
;	OPTIONAL INPUT PARAMETERS:
;		y: The ordinates array.
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls XPowellFit.
;		When this ID is specified, a death of the caller results 
;		in a death of PowellFit.
;		NODRAW = avoids creating a widget_draw for display.
;			It uses the current window as display.
;		XPLOTMAIN = Only for use from XPlot. This keyword
;			is the widget ID of the manin xplot (caller)
;			window.
;		NO_BLOCK = When set, the NO_BLOCK keyword is passed to Xmanager.
;		INIT_PARS = Set this keyword to an array of strings
;			defining the initial values for the equation, model 
;			parameters, fix/free flags and min and max for the fit.
;			 For example:
;			tmp = ["equation = 'a(0)*exp(-((x-a(1))/a(2))^2/2)'",$
;				"a=[1.0,0,.2]",$
;				"free_flags=[1,1,1]",$
;				"xmin=0.0",$
;				"xmax=0.0"]
;			xnlfit,data,INIT_PARS=tmp
;		
;	OUTPUTS:
;	COMMON BLOCKS:
;               COMMON powellfit,equation,x,y, sigma, xrange, chi2fit
;		(set PowellFit help for a description)
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;	RESTRICTIONS:
;		The limitations of curvefit and powell.
;	PROCEDURE:
;	MODIFICATION HISTORY:
;       	by  Manuel Sanchez del Rio. ESRF. 95-02-13
;		95-03-16 MSR adds statistic weight, covariance, external
;			input of sigma...
;		95-11-24 MSR creates XNLFIR based on XPOWELLFIT.
;		97-02-12 MSR changes the gaussian funct (it wasn't normalized)
;		97-09-08 MSR adds the NO_BLOCK keyword. Cosmetics in help.
;		97-10-22 MSR includes fix/free flags, load/save macro file, 
;			INIT_PARS kew and other changes. 
;-

PRO XNLFIT_HELP,GROUP=group
	text = ['This option permits to fit the current set of data to ',$
   'a user-defined function, by using a non-linear fit.', $
   '   ', $
   'For the non-linear fit it uses either the CURVEFIT or POWELL idl ',$
   'routines to find the minimum f the function: ',$
   'ChiSquare= Sum{ ([y-f(x;a)]/sigma)^2 }. ', $
   '   ', $
   'CURVEFIT is based in the Gradient-Expansion algorithm (Marquardt)',$
   'from the book Data Reduction and Error Analysis for the Physical',$
   'Sciences, P.R. Bevington & D.K. Robinson, 2nd ed. McGraw-Hill, 1992',$
   'pag 162.', $
   'POWELL is based on the routine powell described in section 10.5 of', $
   'Numerical Recipes in C: The Art of Scientific Computing (Second',$
   'Edition), published by Cambridge University Press.', $
   '   ', $
   'The user must define the fitting function in IDL syntax as an ',$
   'expression of the independent variable x and a parameter array a.',$
   'For example a straight line is : a(0)*x+a(1)', $
   '              gaussian is a(0)*exp( -((x-a(1))/a(2))^2/2) etc etc.', $
   '   ', $
;   'For fitting with complicated functions, the user may write an',$
;  'IDL customized function that has to be typed in the Equation',$
;  'entry. For example, write a file with:', $
;  '       FUNCTION MYFITTEST,x,a', $
;  '       return,a(0)*exp( (x-a(1))^2/a(2))', $
;  '       end', $
;  " and type in the equation entry: myfittest(x,a)",$
;  '   ',$
   'The initial parameter guess must be provided in array notation:',$
   '[a(0),a(1),a(2),etc], and it also needs the x range defining the',$
   'points to be used for the fit. By setting xmin=xmax, the program ',$
   'uses all the x interval for the fit.',$
   '   ', $
   'In the case that the  CURVEFIT is selected, it is possible to set',$
   'flags that indicate which parameters are fixed (flag=0) or free (flag=1)',$
   'during the fitting process.',$
   '   ',$
   'With curvefit, the calculations are done in double precission if the ',$
   'model parameters are in double precision (i.e. a=[0.0D0,1,3]).  Remember',$
   'to convert to double precision in subsequential runs, when wanted. Under',$
   'Windows, the calculations are always done in double precision.',$
   '   ',$
   'The sigma values can be the unit for all points, or can be read',$
   'from an external file, or can be set to statistical noise: ',$
   'sigma = sqrt(y).', $
   '   ',$
   'In addition, the user may select the option to calculate covariance ',$
   'on the model and fitted parameters.',$
   ' ',$
   'The fitting equation, current model parameters, fix/free flags and ',$
   'fitting limits can be stored in a file (XNLFit macro) which can be ',$
   'edit and reused. This macro defines the variables equations, a, ',$
   'free_flags, xmin and xmax and can use a set of variables a0,...,a10 ',$
   'for intermediate calculations. For example, the a file with the following',$
   'text will be accepted: ',$
   '     ; this is a comment ',$
   "     a0='a(0)*exp(-((x-a(1))/a(2))^2/2)'  ; gaussian",$
   "     a1='+ a(3)*(a(5)/2)^2 /((x-a(4))^2+(a(5)/2)^2)' ; lorentzian",$
   '     a2=[0.01,0,0.02]  ; gaussian model parameters',$
   '     a3=[0.01,0,0.02]  ; lorentzian model parameters',$
   '     equation = a0+a1',$
   '     a = [a2,a3]',$
   '     free_flags=fix(a*0)+1 ; everything 1',$
   '     xmin=0',$
   '     xmax=0']
   XDisplayfile1,TEXT=text,GROUP=group,TITLE='Help on XNLFit (Non Linear Fit)'
end ; XNLFIT_HELP
;
;============================================================================
;
PRO xnlfit_load,commands,wids
;
catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched'
  catch, /cancel
  return
endif
; initializa vars
equation = 0 & a=0 & free_flags=0 & xmin=0 & xmax=0
a0 = 0 & a1=0 & a2=0 & a3=0 & a4=0 & a5=0 
a6=0 & a7=0 & a8=0 & a9=0 & a10=0
; execute buffer
for i=0,n_elements(commands)-1 do begin
  itmp = execute(commands(i))
  if itmp NE 1 then begin
    itmp = widget_message(/ERROR,widget_parent=$
    event.top,'Error executing '+commands(i))
  endif
endfor
;update panels
widget_control,wids.equation,set_value=equation
widget_control,wids.guess,set_value=vect2string(a)
widget_control,wids.flags,set_value=vect2string(free_flags)
if xmin LT xmax then begin
  widget_control,wids.xmin,set_value=xmin
  widget_control,wids.xmax,set_value=xmax
endif

end
;
;============================================================================
;
PRO xnlfit_event, event
common powellfit, equation, x, y, sigma, xrange, chi2fit

catch, error_status
if error_status ne 0 then begin
  message,/info,'error catched'
  catch, /cancel
  if n_elements(state) NE 0 then widget_control, event.top, $
	SET_UVALUE = state, /NO_COPY
  return
endif

widget_control,event.top,GET_UVALUE=state,/NO_COPY
widget_control,event.id,GET_UVALUE=uval
x = state.set(0,*)
y = state.set(1,*)
wset,state.window

; 
; update widgets
;
if  event.id EQ state.wids.SigmaValue then begin
  state.SigmaValue = event.index
  if event.index EQ 2 then $
	widget_control,state.wids.SigmaFileBase,MAP=1 else $
	widget_control,state.wids.SigmaFileBase,MAP=0 
endif

if  event.id EQ state.wids.FitMethod then begin
  state.FitMethod = event.index
  if event.index EQ 0 then $
  widget_control,state.wids.FlagsBase,MAP=1 else $
  widget_control,state.wids.FlagsBase,MAP=0
endif
;
; action
;

;help,uval
;print,'<><><> uval: ',uval
IF keyword_set(uval) then BEGIN
  case uval of
    'Quit': begin
	widget_control,event.top,/DESTROY
	return
	end
    'Help': xnlfit_help,GROUP=event.top
    'FreeFlags': begin
	tmp0 = 0
	widget_control,state.wids.guess,get_value=tmp0
	nn = 0
	itmp = execute('nn = n_elements('+tmp0(0)+')')
	if itmp  NE 1 then begin
	   itmp = widget_message(/ERROR,dialog_parent=event.top,$
	   'Error freeing all parameters. Set Model Parameters before.')
	   goto,out
	endif
	widget_control,state.wids.flags,set_value=$
	 vect2string(replicate(1,nn))
	end
    'write': begin
	gfile = pickfile(FILE='xplot.nlf',$
	title='Select a file for writing XNLFit macro',GROUP=event.top)
	if (gfile eq '') then goto,out
	openw,unit,gfile,/get_lun
	printf,unit,';File created by Xplot/XNLFit. Can be edited and reloaded.'
	widget_control,state.wids.equation,get_value=tmp0
	printf,unit,"equation='"+tmp0(0)+"'"
	widget_control,state.wids.guess,get_value=tmp0
	printf,unit,"a="+tmp0(0)
	widget_control,state.wids.flags,get_value=tmp0
	printf,unit,"free_flags="+tmp0(0)
	widget_control,state.wids.xmin,get_value=tmp0
	printf,unit,"xmin="+strcompress(tmp0(0),/rem)
	widget_control,state.wids.xmax,get_value=tmp0
	printf,unit,"xmax="+strcompress(tmp0(0),/rem)
	free_lun,unit
	end
    'load': begin
	if sdep() eq 'UNIX' then filter='*.nlf' else filter=0
	gfile = pickfile(FILTER=filter,$
	title='Select a file for reading XNLFit macro',GROUP=event.top)
	if (gfile eq '') then goto,out
	commands = read_textfile(gfile)
	xnlfit_load,commands,state.wids
	end
    'Xplotaddcolumn': begin
        widget_control,state.wids.guess,get_val = guess
        guess = guess(0)
        a = 0
        y2 = 0
        command = 'a='+guess
        print,'command: ',command  &  error = not( execute(command) )
        if error then begin
          tmp  = widget_message(DIALOG_PARENT=event.top, $
	  'Error defining initial parameters.',/ERROR)
          goto,out
        endif
        widget_control,state.wids.equation,get_val = equation
        equation = equation(0)
        command = 'y2='+equation
        print,'command: ',command  &  error = not( execute(command) )
        if error then begin
          tmp  = widget_message(DIALOG_PARENT=event.top, $
	  'Error executing function.',/ERROR)
          goto,out
        endif

	;widget_control,state.wids.xplot,get_uvalue = xplotstate,/no_copy
	xplot_addcol,state.wids.xplot,y2
	;widget_control,state.wids.xplot,set_uvalue = xplotstate,/no_copy

	end
    'Fit': begin
	widget_control,state.wids.guess,get_value=guess
        widget_control,state.wids.xmin,GET_VALUE=xmin
        widget_control,state.wids.xmax,GET_VALUE=xmax
        widget_control,state.wids.guess,GET_VALUE=guess
        widget_control,state.wids.equation,GET_VALUE=equation
	guess = guess(0)  &  equation = equation(0)
	if strcompress(guess,/rem) EQ '' or $
		strcompress(equation,/rem) EQ '' then begin
		tmp = widget_message(DIALOG_PARENT=event.top, $
		/ERROR,'Error in inputs.')
		goto,out
	endif
	;
	; prepare Fit call
	;
	a = 0
	error = not( execute('a='+guess) )
	if error then begin
  	tmp  = widget_message(DIALOG_PARENT=event.top, $
		['Error defining initial parameters.', $
		'Non-linear fit FAILED.  '],/ERROR)
  		goto,out
	endif 
	xrange=[xmin,xmax]
	;
	; sigma values
	;
	if state.SigmaValue EQ 0 then sigma=0
	if state.SigmaValue EQ 1 then statsigma=1 else $
		statsigma=0
	if state.SigmaValue EQ 2 then begin
  	  widget_control,state.wids.SigmaFile,get_value=sigmafile
	  sigmafile = sigmafile(0)
  	  if checkfile(sigmafile) NE 1 then begin
    	    tmp = widget_message(DIALOG_PARENT=event.top, $
		/ERROR,'File '+sigmafile+' does not exist.')
    	    goto,out
	  endif
  	  print,'XNLFIT: Reading sigma from file: ',sigmafile
  	  ss = rascii(sigmafile)
  	  if not(keyword_set(ss)) then begin
    	    tmp =  widget_message(DIALOG_PARENT=event.top, $
		'File reading error',/ERROR)
    	    goto,out
  	  endif
  	  if n_elements(ss(0,*)) NE n_elements(x) then begin
	    tmp  = widget_message(DIALOG_PARENT=event.top, $
		'Sigma File dimensions not valid',/ERROR)
	    goto,out
  	  endif
  	  nncol = n_elements(ss(*,0))-1
  	  print,'XNLFIT: Reading sigma from last column of the file: ',$
		nncol+1
  	  sigma=reform(ss(nncol,*))  ;last column
	endif
	if not(keyword_set(sigma)) then sigma=reform(y*0.+1.)
	if keyword_set(statsigma) then sigma = reform(sqrt(y))

	; call fit
	widget_control,event.id,get_value=vval,/hourglass
	iter = 0
	widget_control,state.wids.IterLimit,get_value=itmax
	itmax = long(itmax(0))
	widget_control,state.wids.Tol,get_value=Tol
	Tol = float(Tol(0))
	;tol = 1e-3
	case state.FitMethod of 
	1: begin   ; POWELL
		fittype = 'Fitting method: Powell'
		message,/info,'XNLLFIT: calling powellfit.'
		powellfit,a,yfit=y2,error=error, iter=iter, $
		 itmax=itmax  ;,statsigma=statsigma,/show
		message,/info,'XNLFIT: back from powellfit.'
		if error then begin
		  tmp = widget_message(DIALOG_PARENT=event.top, $
			'Powellfit error found',/ERROR)
		  goto,out
		endif
		chi2tmp = chi2fit
		end
	0: begin ; CURVEFIT1
        	widget_control,state.wids.flags,GET_VALUE=flags
		fita = 0
		itmp = execute('fita = '+flags(0))
		if itmp NE 1 then begin	
		  itmp = widget_message(/Error,dialog_parent=event.top,$
		  'Error in Free/Fix parameters defintion.')
		  goto,out
		endif
		if n_elements(fita) NE n_elements(a) then begin
		  itmp = widget_message(/Error,dialog_parent=event.top,$
		  'Free/Fix parameters dimension '+strcompress($
		  n_elements(fita),/rem)+$
		  ') must be the same as Model Parameters dimension ('+$
		  strcompress(n_elements(fita),/rem)+').' )
		  goto,out
		endif
		; this is to speed the curvefit1 in case that all flags are 1
		if (where(fita NE 1))(0) EQ -1 then fita=0
		fittype = 'Fitting method: Curvefit'
		message,/info,'XNLFIT: calling curvefit1.'
		y2 = 0
		x = reform(x) & y = reform(y)
		 ;if statsigma EQ 1 then  begin
		   ;ss = sqrt(y)
		   ;w = 1./ss^2
		 ;endif else w = 1./sigma^2
		w = 1./sigma^2
		y2 = curvefit1(x,y,sigma,a,sigmaai,chi2=chi2i, $
			Function_name=equation,iter=iter,tol=tol,$
			itmax=itmax-1,fita=fita)
		message,/info,'XNLFIT: back from curvefit1.'
		chi2tmp = chi2i
		print,'Sigmaai: ',Sigmaai
		error = 0
		end
	else:
	endcase
	
	;
	; sort the outputs
	;
	if xrange(0) EQ xrange(1) then xrangetxt = 'all' else $
		xrangetxt=vect2string(xrange)
	if error then begin
	  tmp = widget_message(DIALOG_PARENT=event.top, $
		' Non-linear fit FAILED. ',/ERROR)
	  goto,out
	endif

	oplot,x,y2,psym=-5
	 txt = ['Fitting results overplots the original '+$
	  'graph with triangular symbols.',' ', $
	  fittype,' ',$
	  'Fitting tolerance: '+ strcompress(tol,/rem), ' ', $
	  'Fitting iterations: '+ strcompress(iter,/rem), ' ', $
	  'Fitting iterations limit: '+ strcompress(itmax,/rem), ' ', $
	  'fitting equation: '+ equation, ' ', $
	  'fitting parms: '+ vect2string(a), ' ', $
	  'fitting range: '+ xrangetxt, ' ', $
	  'Chi2: '+strcompress(chi2tmp,/rem),' ' ]
         ;ShowInfo1,TITLE='Fitting results', $
         ;  INFOTEXT=txt, WIDTH=max(strlen(txt)), HEIGH=n_elements(txt), $
         ;  GROUP=event.top
	 XDisplayFile1,TITLE='Fitting results',TEXT=txt,GROUP=event.top
	 tmp = widget_message(DIALOG_PARENT=event.top,/QUESTION, $
	   'Do you want to update parameters to fitting results?')
	 if tmp EQ 'Yes' then $
	    widget_control,state.wids.guess, set_value = vect2string(a)
	end
    'PlotModel': begin
	widget_control,state.wids.guess,get_val = guess
	guess = guess(0)
	a = 0
	y2 = 0
	command = 'a='+guess
	print,'command: ',command  &  error = not( execute(command) )
	if error then begin
  	  tmp  = widget_message(DIALOG_PARENT=event.top, $
		'Error defining initial parameters.',/ERROR)
	  goto,out
	endif
	widget_control,state.wids.equation,get_val = equation
	equation = equation(0)
	command = 'y2='+equation
	print,'command: ',command  &  error = not( execute(command) )
	if error then begin
	  tmp  = widget_message(DIALOG_PARENT=event.top, $
		'Error executing function.',/ERROR)
	  goto,out
	endif

	widget_control,event.id,get_value = tmp
	case tmp of
	'Covariance': begin
	  ;
	  ; sigma values
	  ;
	  if state.SigmaValue EQ 0 then sigma=0
	  if state.SigmaValue EQ 1 then statsigma=1 else $
		statsigma=0
	  if state.SigmaValue EQ 2 then begin
  	    widget_control,state.wids.SigmaFile,get_value=sigmafile
	    sigmafile = sigmafile(0)
;help,checkfile(sigmafile)
;help,sigmafile
  	    if checkfile(sigmafile) NE 1 then begin
    	      tmp = widget_message(DIALOG_PARENT=event.top, $
		/ERROR,'File '+sigmafile+' does not exist.')
    	      goto,out
	    endif
  	    print,'XNLFIT: Reading sigma from file: ',sigmafile
  	    ss = rascii(sigmafile)
  	    if not(keyword_set(ss)) then begin
    	      tmp =  widget_message('File reading error',/ERROR)
    	      goto,out
  	    endif
  	    if n_elements(ss(0,*)) NE n_elements(x) then begin
	      tmp  = widget_message(DIALOG_PARENT=event.top, $
		'Sigma File dimensions not valid',/ERROR)
	      goto,out
  	    endif
  	    nncol = n_elements(ss(*,0))-1
  	    print,'XNLFIT: Reading sigma from last column of the file: ',$
		nncol+1
  	    sigma=reform(ss(nncol,*))  ;last column
	  endif
	  if not(keyword_set(sigma)) then sigma=reform(y*0.+1.)
	  if keyword_set(statsigma) then sigma = reform(sqrt(y))
	  covar_fit,x,y,a,equation,covar=covar ,sigma=sigma, $
	  outsigma=outsigma ;,statsigma=statsigma
  	  nco = n_elements(covar(0,*))
  	  atext = strarr(nco)
	  for i=0,nco-1 do atext(i)=vect2string(covar(*,i))
	  txt = [ ' Equation: y = '+equation, $
	   'Model parameters: a = '+vect2string(a), $
	   'xrange = all', ' ',$
	   'Sigma of the model parameters: '+vect2string(outsigma), $
	   'covariance matrix:', atext]
	  ;ShowInfo1,TITLE='Covariance Calculations', $
	  ; INFOTEXT=txt, WIDTH=max(strlen(txt)), HEIGH=n_elements(txt), $
	  ; GROUP=event.top
	  XDisplayfile1,TITLE='Covariance Calculations', $
	   TEXT=txt,GROUP=event.top
	  end
	'(X)Plot model': xplot,x,y2,title=equation+', a='+vect2string(a)
	'Overplot model': oplot,x,y2,psym=-4
	endcase
	end
    'Equation': begin
	widget_control,event.id,get_value = evval
	widget_control,state.wids.Equation,get_value=Equation
	Equation = Equation(0)
	case evval of
	  'Clear': begin
		Equation = ''
		state.npar = 0
		end
	  'Add Gaussian': begin	
		npar = state.npar
		state.npar = npar + 3
		if npar eq 0 then sign = '' else sign = '  +'
		a0 = 'a('+strcompress(npar+0,/rem)+')'
		a1 = 'a('+strcompress(npar+1,/rem)+')'
		a2 = 'a('+strcompress(npar+2,/rem)+')'
		Equation = Equation + sign + a0+'*exp(' +$
		 ;'-((x-'+a1+')/'+a2+')^2/'+a2+')'
		 '-((x-'+a1+')/'+a2+')^2/2)'
		end
	  'Add Lorentzian': begin	
		npar = state.npar
		state.npar = npar + 3
		if npar eq 0 then sign = '' else sign = '  +'
		a0 = 'a('+strcompress(npar+0,/rem)+')'
		a1 = 'a('+strcompress(npar+1,/rem)+')'
		a2 = 'a('+strcompress(npar+2,/rem)+')'
		;Equation = Equation + sign + a0+'*'+a2+' /(' +$
		 ;'(x-'+a1+')^2+'+a2+')'
		Equation = Equation + sign + a0+'*('+a2+'/2)^2 /(' +$
		 '(x-'+a1+')^2+('+a2+'/2)^2)'
		end
	  'Add Polynomial': begin	
		strtmp = { n:1 }
		XscrMenu,strtmp,/NoType,Titles='Polynomial Degree',Action=action
		if action ne 'DO' then goto,out
        	deg = strtmp.n
		npar = state.npar
		state.npar = npar + deg + 1
		if npar eq 0 then sign = '' else sign = '  +'
		Equation = Equation + sign 
		for i=0,deg-1,+1 do Equation = Equation+ $
		  ' X^'+strcompress(i,/rem)+ $
		  '*a('+strcompress(npar+i,/rem)+')+'
		Equation = Equation +  $
		 'X^'+strcompress(deg,/rem) + $
		 '*a('+strcompress(npar+deg,/rem)+ ')'
		end
	  endcase
	  widget_control,state.wids.Equation, set_value = Equation
	  ;state.fpar.Equ = Equation
	  end
    'Guess': begin
	widget_control,event.id,get_value = evval
;help,evval
;help,evval(0)
	case evval(0) of
	  'Clear parameters': begin
		Guess = '[]'
		end
	  'Template': begin	
		strtmp = { n:state.npar }
		XscrMenu,strtmp,/NoType,Titles='Number of parameters:', $
			Action=action
		if action ne 'DO' then goto,out
		if strtmp.n eq 0 then Guess = '[]' else Guess = $
		  vect2string(lonarr(strtmp.n))
		end
	  else: guess = evval(0)
	  endcase
	  widget_control,state.wids.Guess,set_value= Guess
	  ;state.fpar.Guess = Guess
	end
    'PlotData': begin
	if (event.id eq state.wids.allinterval) then begin
	  widget_control,state.wids.xmin,SET_VALUE=x(0)
	  widget_control,state.wids.xmax,SET_VALUE=x(0)
	  xmin = 0.0  &  xmax=0.0
	endif
	widget_control,state.wids.xmin,GET_VALUE=xmin
	widget_control,state.wids.xmax,GET_VALUE=xmax

	if xmin NE xmax then begin
	    tmp1 = fltarr(2,n_elements(x))
	    tmp1(0,*) = reform(x)  &  tmp1(1,*) = reform(y)
	    if xmin LE xmax then xrange=[xmin,xmax] else xrange=[xmax,xmin]
	    cutset,tmp1,tmp2,xrange=xrange
	    if state.wids.xplot EQ 0 then plot,x,y else  begin
             widget_control,state.wids.xplot,get_uvalue = xplotstate
             xplot_plot,xplotstate
	    endelse
	    oplot,tmp2(0,*),tmp2(1,*),color=100
	endif else begin
	    xrange=[xmin,xmax]
	    if state.wids.xplot EQ 0 then plot,x,y,/nodata else begin
              widget_control,state.wids.xplot,get_uvalue = xplotstate
              xplot_plot,xplotstate
	    endelse
	    oplot,x,y,color=100
	endelse
	;state.fpar.xmin = float(xmin)
	;state.fpar.xmax = float(xmax)
	end
    'Xloadct': xloadct,group=event.top
    'SigmaFileBrowser': Begin
	tmp = pickfile(/READ,/NOCO)
	if  strcompress(tmp,/REM) EQ '' then goto,out
	widget_control,state.wids.SigmaFile,set_val=tmp
	end
    'EXTRACT': Begin
        widget_control,state.wids.xmin,GET_VALUE=xmin
        widget_control,state.wids.xmax,GET_VALUE=xmax
        if xmin NE xmax then begin
            tmp1 = fltarr(2,n_elements(x))
            tmp1(0,*) = reform(x)  &  tmp1(1,*) = reform(y)
            if xmin LE xmax then xrange=[xmin,xmax] else xrange=[xmax,xmin]
            cutset,tmp1,tmp2,xrange=xrange
	    xplot,tmp2,wtitle='Fitting data in '+vect2string(xrange)
	endif else xplot,x,y,wtitle='Fitting data in all interval'
	end
     else:
  endcase
ENDIF

out:
widget_control,event.top,SET_UVALUE=state,/NO_COPY

end
;
;============================================================================
;


PRO xnlfit, set, optinp, group = group, nodraw=nodraw, $
 xplotmain=xplotmain, no_block=no_block,init_pars=init_pars
;common powellfit, equation, x, y, sigma, xrange, chi2fit, fpar

on_error,2
if n_params() EQ 0 then begin
  print,'XNLFIT: Usage: xnlfit, set, keyword_pars'
  return
endif

if  n_params() EQ 2 then begin
  x = reform(set)
  y = reform(optinp)
  set = fltarr(2,n_elements(x))
  set(0,*) = x
  set(1,*) = y
endif ;else begin
;  x = set(0,*)
;  y = set(1,*)
;endelse
;help,x
;if not(keyword_set(xrange)) then xrange=[0.,0.]
;if n_elements(xrange) NE 2 then xrange=[0.,0.]
;xrange1=[min(x,max=tmp),tmp]
;print,'xrange: ',xrange
;print,'xrange1: ',xrange1

;
; XscrMenu part
;
equation = ''
guess = '[]' ;[1.,0.,1e-4]'
xmax = max(set(0,*),min=xmin)

if keyword_set(equation) then fpar.equ=equation


wBaseBuffer = Widget_base() ; to store the command to be returned
wWin = Widget_base(TITLE='XNLFit (Non-linear fitting)',MBAR=wMenu)
;
; define menu bar
;
wtmp0 = WIDGET_BUTTON(wMenu, VALUE='File', /MENU)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Load XNLFit Macro File...', $
	UVALUE='load')
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Write XNLFit Macro File...', $
	UVALUE='write')
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Quit', UVALUE='Quit',/SEPARATOR)
wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Tools', /MENU)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Change Color table', UVALUE='Xloadct')
wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Help', /HELP)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='xnlfit', UVALUE='Help')



wBase = Widget_Base(wWin,/Column)

wtmp = widget_base(wBase,/ROW)
;wQuit = widget_Button(wtmp,VALUE='Quit',UVALUE='Quit')
wRefresh = widget_Button(wtmp,VALUE='Refresh Screen',UVALUE='PlotData')
;wXloadct = widget_Button(wtmp,VALUE='Color table',UVALUE='Xloadct')
;wHelp = widget_Button(wtmp,VALUE='Help',UVALUE='Help')
;wtmp0 = widget_base(wtmp,/ROW)
wtmp0 = widget_Button(wtmp,VALUE='**Start Fit**',UVALUE='Fit')


if not(keyword_set(nodraw)) then $
wDr = widget_draw(wBase, XSIZE=390, YSIZE=300, RETAIN=2)

wEq = widget_base(wBase,/COLUMN,/FRAME)
wEquationLabel =  widget_label(wEq,VALUE='Model equation: y=f(x;a(0,1...))', $
	/ALIGN_LEFT)
wEquation =  widget_text(wEq,VALUE=equation,/ALIGN_LEFT,/EDIT,$
	UVALUE='Equation',XSIZE=75)
wtmp = widget_base(wEq,/ROW)
wtmp1 = widget_button(wtmp,VALUE='Clear',UVALUE='Equation')
wtmp2 = widget_button(wtmp,VALUE='Add Gaussian',UVALUE='Equation')
wtmp3 = widget_button(wtmp,VALUE='Add Lorentzian',UVALUE='Equation')
wtmp4 = widget_button(wtmp,VALUE='Add Polynomial',UVALUE='Equation')

wGu = widget_base(wBase,/COLUMN,/FRAME)
wGuessLabel =  widget_label(wGu,VALUE='Model parameters:',/ALIGN_LEFT)
wGuess =  widget_text(wGu,VALUE=guess,/ALIGN_LEFT,/EDIT,XSIZE=75, $
	UVALUE='Guess')
wtmp = widget_base(wGu,/ROW)
wtmp1 = widget_button(wtmp,VALUE='Clear parameters',Uvalue='Guess')
wtmp2 = widget_button(wtmp,VALUE='Template',Uvalue='Guess')
wtmp3 = widget_Button(wtmp,VALUE='(X)Plot model',UVALUE='PlotModel')
wtmp4 = widget_Button(wtmp,VALUE='Overplot model',UVALUE='PlotModel')
wtmp5 = widget_Button(wtmp,VALUE='Covariance',UVALUE='PlotModel')
if keyword_set(xplotmain) then wtmp6 = widget_Button(wGu, $
	VALUE='Put Model in a new column',UVALUE='Xplotaddcolumn')

wFlagsbase = widget_base(wBase,/COLUMN,/FRAME)
wFlagsLabel =  widget_label(wFlagsbase,VALUE=$
  'Free (1) or Fix (0) parameters for fit:',/ALIGN_LEFT)
wFlags =  widget_text(wFlagsbase,VALUE='[]',/ALIGN_LEFT,/EDIT,XSIZE=75, $
	UVALUE='')
wtmp = widget_base(wFlagsbase,/ROW)
wtmp1 = widget_button(wtmp,VALUE='Free all',Uvalue='FreeFlags')

wRa = widget_base(wBase,/COL,/FRAME)
wLa =  widget_label(wRa,VALUE='Fitting Abscissas Domain:',/ALIGN_LEFT)
wtmp = widget_base(wRa,/ROW)
wx1= cw_fslider(wtmp, TITLE='X minimum',UVALUE='PlotData',$
         MIN=xmin,MAX=xmax, XSIZE=190, /EDIT,VALUE=xmin,/FRAME)
wx2= cw_fslider(wtmp, TITLE='X maximum',UVALUE='PlotData',$
         MIN=xmin,MAX=xmax, XSIZE=190, /EDIT,VALUE=xmax,/FRAME)
wtmp0 = widget_base(wRa,/ROW)
wAllinterval =  widget_button(wtmp0,VALUE='Set all interval',UVALUE='PlotData')
wtmp1 =  widget_button(wtmp0,VALUE='Xplot with data in domain',$
	UVALUE='EXTRACT')

wtmp = widget_base(wBase,/ROW,/FRAME)
wtmp1 = widget_label(wtmp,VALUE='Sigma (error) value:',/ALIGN_LEFT)
wSigmaValue = widget_droplist(wtmp,VALUE=['1','Statistic','From file'])
wSigmaFileBase = widget_base(wtmp,/ROW)
wtmp = widget_button(wSigmaFileBase,VALUE='Browser', $
	UVALUE='SigmaFileBrowser')
wSigmaFile = widget_text(wSigmaFileBase,VALUE='xnlfit.dat', $
	/ALIGN_LEFT,/EDIT)

wtmp = widget_base(wBase,/COL,/FRAME)
wtmp0 = widget_base(wtmp,/ROW)
wtmp1 =  widget_label(wtmp0,VALUE='Fitting tolerance: ',/ALIGN_LEFT)
wTol = widget_text(wtmp0,VALUE='1e-4', /ALIGN_LEFT,/EDIT,XSIZE=6)
wtmp2 =  widget_label(wtmp0,VALUE='Iterations limit: ',/ALIGN_LEFT)
wIterLimit = widget_text(wtmp0,VALUE='20', /ALIGN_LEFT,/EDIT,XSIZE=6)
wtmp3 =  widget_label(wtmp0,VALUE='Method: ',/ALIGN_LEFT)
wFitMethod = widget_droplist(wtmp0,VALUE=['CurveFit','Powell'])

;wtmp0 = widget_base(wtmp,/ROW)
;wtmp1 = widget_Button(wtmp0,VALUE='**Start Fit**',UVALUE='Fit')

;wtmp1 = widget_label(wtmp,VALUE='Output to:',/ALIGN_LEFT)
;wtmp2 = widget_droplist(wtmp,VALUE=['Overplot current','Xplot window',$
;	'New file'])



if keyword_set(nodraw) then begin
  wnum = !d.window
  if wnum LT 0 then begin
    window,/free
    wnum = !d.window
  endif
endif  else wnum=-1

if not(keyword_set(xplotmain)) then xplotmain=0L

if not(keyword_set(xplotmain)) then xplotmain=0L
wids = { equation:wEquation, guess:wGuess, xmin:wx1, xmax:wx2, $
	Allinterval:wAllinterval,  SigmaValue:wSigmaValue, $
	SigmaFileBase:wSigmaFileBase, SigmaFile:wSigmaFile, $
	Tol:wTol, IterLimit:wIterLimit, Xplot:xplotmain,$
	FitMethod:wFitMethod, FlagsBase:wFlagsBase, Flags:wFlags}
state= { window:wnum, set: set, wids:wids, npar:0, SigmaValue:0L, $
	FitMethod:0L}

widget_control,wWin,/REALIZE 
widget_control,wSigmaFileBase,MAP=0
if not(keyword_set(nodraw)) then begin
  widget_control,wDr, get_val = wnum
  state.window=wnum
endif
if keyword_set(init_pars) then xnlfit_load,init_pars,wids
widget_control,wWin, SET_UVAL=state,/NO_COPY
if xplotmain EQ 0 then plotfile,set
plotfile,set,ki=2,color=100,xrange=xrange
if keyword_set(no_block) then $
  xmanager, 'XNLFit', wWin, GROUP_LEADER=group,/NO_BLOCK else $
  xmanager, 'XNLFit', wWin, GROUP_LEADER=group
end
