;+
; NAME:
;	XEDIT
; PURPOSE:
;	Routine for editing a scalar variable.
; CATEGORY:
;	GAMS Index N1: Data handling; input, output
; CALLING SEQUENCE:
;	XEDIT, var [, TEXT=text, TITLE=title, FORMAT=format, $
;		XOFFSET=xoffset, YOFFSET=yoffset]
; INPUTS:
;	var:	Name of variable to be edited
; KEYWORD PARAMETERS:
;	TEXT:	Text adjacent to the variable input field
;	TITLE:	Title of the window
;	FORMAT:	Format of the variable displayed in the input field
;	XOFFSET: Offset in the x-direction (pixels) of the pop-up widget
;	YOFFSET: Offset in the y-direction (pixels) of the pop-up widget
;	INFOTEXT:String array with an information  text to be display at
;		the top of the created window.
;	ACTION: (output) string variable with 'OK' or 'CANCEL' depending
;		on which button has been pressed.
; OUTPUTS:
;	var:	Modified variable
; RESTRICTIONS:
;	Variable to be edited must be of type scalar. The variable var is
;	a required parameter.
; PROCEDURE:
;	Straighforward. Uses handles to pass information.
; EXAMPLE:
;	XEDIT, a, TEXT='Variable a: New Value', TITLE='Edit Variable a'
; MODIFICATION HISTORY:
; 	Written by:	Roger J. Dejus, XFD/APS, April, 1995.
;		96/01/18 srio@esrf.fr adds the INFOTEXT and ACTION  keywords.
;			 Changes xtext by widget_message.
;-

PRO xedit_event, event

widget_control, event.top, get_uvalue=info, /no_copy
widget_control, event.id, get_uvalue=test
handle_value, info.actionptr, test, /set
CASE test OF
  'CANCEL': BEGIN
  widget_control, event.top, /destroy
  END

  'OK': BEGIN
  widget_control, info.id_var, get_value=var_st
  handle_value, info.ptr, var_st, /set, /no_copy
  widget_control, event.top, /destroy
  END
ENDCASE

END ; xedit_event

PRO xedit, var, TEXT=text, TITLE=title, FORMAT=format, $
		XOFFSET=xoffset, YOFFSET=yoffset, $
		INFOTEXT=infotext, ACTION=action

on_error, 2 ; return to caller if an error occurs
on_ioerror, conversion_error
if n_elements(var) eq 0 then $
  message, 'Incorrect number of parameters: supply name of variable'
sz_var = size(var)
if sz_var(0) ne 0 then $
  message, 'Incorrect size: variable must be scalar'
if sz_var(1) eq 6 then $
  message, 'Incorrect type: variable must not be of type complex'

if n_elements(text) eq 0 then text = 'variable'
if n_elements(title) eq 0 then title = 'xedit'
if n_elements(format) eq 0 then var_st = string(var) else $
  var_st = string(var, format=format)
if n_elements(xoffset) eq 0 then xoffset = 0
if n_elements(yoffset) eq 0 then yoffset = 0

base = widget_base(title=title, column=1, xoffset=xoffset, yoffset=yoffset)
;MSR adds infotext
if keyword_set(infotext) then begin
  for ii=0,n_elements(infotext)-1 do $
	junk = widget_label(base, value=infotext(ii), /align_left)
endif
fd_var = cw_field(base, title=text, value=var_st)

bs = widget_base(base, column=1, /frame)
bt = widget_button(bs, value='OK', uvalue='OK')
bt = widget_button(bs, value='Cancel', uvalue='CANCEL')

ptr  = handle_create(value=var_st)
actionptr  = handle_create(value='')
info = {id_var:fd_var, ptr:ptr, actionptr:actionptr}
widget_control, base, set_uvalue=info

widget_control, base, /realize
xmanager, 'xedit', base, /modal

handle_value, ptr, var_st, /no_copy
handle_free, ptr
handle_value, actionptr, action, /no_copy
handle_free, actionptr

if action EQ 'CANCEL' then return

CASE sz_var(1) OF
  0: BEGIN ; Undefined
  var1 = var_st(0)
  END

  1: BEGIN ; Byte
  var1 = byte(var_st(0))
  END

  2: BEGIN ; Integer
  var1 = fix(var_st(0))
  END

  3: BEGIN ; Longword integer
  var1 = long(var_st(0))
  END

  4: BEGIN ; Floating point
  var1 = float(var_st(0))
  END

  5: BEGIN ; Double-precision floating point
  var1 = double(var_st(0))
  END

  6: BEGIN ; Complex floating (excluded)
  END

  7: BEGIN ; String
  var1 = var_st(0)
  END
ENDCASE

; Value OK, store in original variable
var = var1
return

conversion_error:
;xtext, text=!err_string, size=[60,3]
tmp = widget_message(/ERROR,!err_string)

END ; xedit
