FUNCTION surf_replicate,a,ORIENTATION=orientation,REVERSE=reverse, $
COLLAPSE=collapse

;+
; NAME:
;	SURF_REPLICATE
;
; PURPOSE:
;	This function permits to replicate a given surface (or2-dim array)
;	in a given direction. It is useful when one has a piece of a total 
;	surface in one quadrant and wants to replicate the surface in the other 
;	quadrants by symetry operation (copy or mirror).
;
; CATEGORY:
;	general utilities.
;
; CALLING SEQUENCE:
;	surf_out = surf_replicate(surf_ini)
;
; INPUTS:
;	surf_ini: the surface array
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;	ORIENTATION: The position of the quadrant where the duplication
;		takes place. Valid values are 'N', 'S', 'E' and 'W'. The 
;		position of this areas related to the original surface is 
;		the following:
;
;                                [N]
;                        y axis
;                        ^================
;                        |====== my ======
;                  [E]   |================    [W]
;                        |==== surface ===
;                        |================
;                        |================
;        ................|_________________
;                        .(0,0)             -> x axis
;                        .       [S]
;                        .
;	REVERSE: When this keyword is set, a mirror reflection of the 
;		surface is produced.
;	COLLAPSE: When this keyword is set, the limiting values of the
;		initial surface and the added one are collapsed to a
;		single one.
;
; OUTPUTS:
;	surf_out: the new surface.
;
; PROCEDURE:
;	Straightforward.
;
; RESTRICTIONS
;
; EXAMPLE:
;	Lets create a pyramid-like surface:
;	1) define an inclinated surface:
;	a=fltarr(10,10) & for i=0,9 do for j=0,9 do a(i,j)=i*j
;	
;	2) Duplicate it with the mirror image North direction:
;	b=surf_replicate(a,orient='N',/reverse,/collapse)
;	
;	3) Duplicate it with the mirror image West direction:
;	c=surf_replicate(b,orient='W',/reverse,/collapse)
;
;	4) show result
;	surface,c
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio 
;	97-04-03 srio@esrf.fr Bug fixed in 'E'. Adds doc.
;-
if not(keyword_set(orientation)) then orientation = 'N'

if ((orientation NE 'N') and (orientation NE 'S') and  $
   (orientation NE 'E') and (orientation NE 'W')) then begin
   message,/info,'Orientation keyword not understood. Abort'
   return,0
endif

nx = n_elements(a(0,*))
ny = n_elements(a(*,0))

s = size(a)
nx = s(1)
ny = s(2)
typ = s(s(0)+1)

if not(keyword_set(collapse)) then begin
  if (orientation eq 'N' or orientation eq 'S') then $
    b = make_array(nx, ny*2, type=typ)
  if (orientation eq 'E' or orientation eq 'W') then $
    b = make_array(nx*2, ny, type=typ)
endif else begin
  if (orientation eq 'N' or orientation eq 'S') then $
    b = make_array(nx, ny*2-1, type=typ)
  if (orientation eq 'E' or orientation eq 'W') then $
    b = make_array(nx*2-1, ny, type=typ)
endelse

;help,a & help,b

case orientation of
  'N':  BEGIN
	b(0:nx-1,0:ny-1) = a
	if keyword_set(collapse) then begin
	   if keyword_set(reverse) then b(0:nx-1,ny-1:2*ny-2)=rotate(a,7) else $
		b(0:nx-1,ny-1:2*ny-2) = a 
	endif else begin
	   if keyword_set(reverse) then b(0:nx-1,ny:2*ny-1) = rotate(a,7) else $
		b(0:nx-1,ny:2*ny-1) = a 
	endelse
	END
  'S': BEGIN
	if keyword_set(collapse) then b(0:nx-1,ny-1:2*ny-2) = a else $
	  b(0:nx-1,ny:2*ny-1) =a
	if keyword_set(reverse) then b(0:nx-1,0:ny-1) = rotate(a,7) else $
				     b(0:nx-1,0:ny-1) = a 
	END
  'E': BEGIN
	if not(keyword_set(collapse)) then b(nx:2*nx-1,0:ny-1) = a else $
	  b(nx-1:2*nx-2,0:ny-1) = a
	if keyword_set(reverse) then b(0:nx-1,0:ny-1) = rotate(a,5) else $
				     b(0:nx-1,0:ny-1) = a 
	END
  'W': BEGIN
	b(0:nx-1,0:ny-1) = a
	if not(keyword_set(collapse)) then begin
	  if keyword_set(reverse) then b(nx:2*nx-1,0:ny-1) = rotate(a,5) else $
				     b(nx:2*nx-1,0:ny-1) = a 
	endif else begin
	  if keyword_set(reverse) then b(nx-1:2*nx-2,0:ny-1)=rotate(a,5) else $
				     b(nx-1:2*nx-2,0:ny-1) = a 
	endelse
	END
endcase
;print,'REPLICATE_SURF: nx,ny = ',nx,ny
return,b
end

