;+
;
;	NAME:
;		POWELLFIT
;	PURPOSE:
;		to make a fit of a data set (x,y) with an specified function
;		y=f(x;a) where a are the free parameters.
;	CALLING SEQUENCE:
;		POWELLFIT,a,keyword_parameters
;	INPUT PARAMETERS:
;		A = initial guess of the parameters
;	OPTIONAL INPUT PARAMETERS:
;	KEYWORD PARAMETERS:
;		SHOW = when set to 1, plot the data points and overplots
;		the initial guess.
;		YFIT = yfit returns the array with the best fit
;		evaluation points.
;		ERROR = logical with the error status
;		STATSIGMA = when set to 1, then takes 
;		sigma = sqrt(yfit) (statistical error)
;		ftolerance = the tolerance value for nr_powell (def 1e-3)
;		iter = output (number of iterations performed)
;		itmax = number of maximum iterations allowed (def=200)
;	OUTPUTS:
;		Overwrite the fitting values on the initial estimation a, 
;		and (optionally) puts the evaluated function in the yfit 
;		keyword.
;	COMMON BLOCKS:
;		COMMON powellfit,equation,x,y, sigma, xrange, chi2fit, fpar
;		where:
;		equation: a string text with the f(x;a) equation
;		x: the abscissas array of the data points
;		y: the ordinates array of the data points
;		sigma: an array with the experimental error for y points
;			(the program takes sigma=x*0.+1 per default)
;		xrange: the x range where to evaluate the ChiSquare
;			function to be minimized
;		chi2fit: (output) value of the ChiSquate for the
;			parameter values obtained in the fitting process.
;		
;		
;	SIDE EFFECTS:
;		Uses internally the powellfit6 function.
;
;	RESTRICTIONS:
;		Unknown.
;	PROCEDURE:
;		Uses the the nr_powell idl function to find the minimum
;		of the function ChiSquare= Sum{ ( (y-f(x;a))/sigma )^2 }. 
;		The data are passed trough common blocks because the 
;		"user defined function" which calculates ChiSquare must 
;		know them. It uses the function powellfit6 internally 
;		to evaluate ChiSquare.
;
;	EXAMPLE: 
;		; define common block
;		COMMON powellfit,equation,x,y, sigma, xrange, chi2fit
;		;create data set (gaussian + noise)
;		x=findgen(100) & y= 10.*exp( -(x-55.)^2 / 50.)
;		y = y+2.*randomu(seed,100)
;		;define the equation to be fitted:
;		equation='a(0)*exp(-(x-a(1))^2/a(2))'
;		a=[6,40,20]  ; the initial guess
;		powellfit,a,yfit=yfit
;		; plot the result:
;		plot,x,y & oplot,x,yfit
;               print,a ;print the fitted values=[10.6253,55.0800,68.7902]
;
;	MODIFICATION HISTORY:
;       by  Manuel Sanchez del Rio. ESRF. 95-02-13
;	95-03-15 MSR Adds statsigma keyword. Changes chi2fit->chi2fit/dof
;		Other small changes.
;	95-11-20 MSR adds ftolerance,iter and itmax  keyword and change 
;		default ftolerance 1e-4 -> 1e-3. Removes fpar in common block.
;
;-
FUNCTION powellfit6,a,error=error,statsigma=statsigma
common powellfit,equation,x,y, sigma, xrange, chi2fit

error = not(execute('y1 = '+equation) )
if error then begin
  print,'POWELLFIT: Error interpreting equation'
  return,0
endif else begin
  if keyword_set(statsigma) then begin
    sigma = sqrt(y1)
    print,'POWELLFIT6: considering statistical error.'
  endif
  ;data = fltarr(2,n_elements(x))
  ;data(0,*) = x
  data = [1,0]#x
  data(1,*) = ( (y-y1)/sigma )^2
  if xrange(0) NE xrange(1) then  cutset,data,data2,xrange=xrange $
	else data2=data
  nterms = n_elements(a) 			; npoints
  nfree = n_elements(data(1,*)) - nterms ; degrees of freedom
  if nfree le 0 then print,'POWELLFIT: Not enough data points.'
  return,total(data2(1,*)) / nfree
endelse
end

;
;
; 

PRO powellfit,a,show=show,yfit=yfit,error=error,statsigma=statsigma, $
ftolerance = ftolerance, iter=iter, itmax=itmax

common powellfit,equation,x,y, sigma, xrange, chi2fit

on_error,2
catch,ierr

if ierr NE 0 then begin
  print,'POWELLFIT: Error.'
  error = 1
  return
endif

if not(keyword_set(sigma)) then sigma = x*0.+1.

if not(keyword_set(xrange)) then xrange=[0.,0]
if not(keyword_set(itmax)) then itmax=200
if xrange(0) EQ xrange(1) then xrange=[min(x),max(x)]

if keyword_set(show) then begin
  plot,x,y
  error = not( execute ('y1 = '+equation) )
  if error then begin
    print,'POWELLFIT: Error evaluating equation'
    print,'POWELLFIT: Returning'
    return
  endif 
  oplot,x,y1
endif

npar = n_elements(a)
;xi = transpose([[1.,0.,0.],[0.,1.,0.],[0.,0.,1.]])
xi = fltarr(npar,npar)
for i=0,npar-1 do xi(i,i) = 1.0

if not(keyword_set(ftolerance)) then ftolerance = 1e-3
iter=1
powell, a, xi ,ftolerance, fmin, 'powellfit6', iter=iter, $
  itmax=itmax
chi2fit = powellfit6(a,error=error,statsigma=statsigma)

error = not( execute ('yfit = '+equation) )
if error then begin
  print,'POWELLFIT: Error evaluating equation'
  print,'POWELLFIT: Returning'
  return
endif 
if keyword_set(show) then begin
  oplot,x,yfit
endif


end
