;+
; NAME:
;	LIN_REG
;
; PURPOSE:
;	This procedure makes linear regression and performs
;	several non-linearity tests.
;
; CATEGORY:
;	Regression
;
; CALLING SEQUENCE:
;
;	LIN_REG,x,y [,fbeta0,fbeta1]
;
; INPUTS:
;	x and y: the arrays with x and y values
;
; OPTIONAL OUTPUTS:
;	fbeta0 : Regression intercept
;	fbeta1 : Regression slope
;	
; KEYWORD PARAMETERS:
;
; OUTPUTS:
;	the results are printed on the screen
;
; SIDE EFFECTS:
;
; PROCEDURE:
;	See: Introduction to Linear Regrassion Analysis,
;	by D. C. Montgomery, E. A. Peck. Wiley (1982)
;
;
; EXAMPLE:
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio
;	March, 1995	
;	95-12-12 MSR places it in idl.expg/util distribution.
;
;-
pro lin_reg,x,y,fbeta0,fbeta1
on_error,2

npar = 2
n=n_elements(x)
n1=n_elements(x)
if n NE n1 then begin
  print,'LIN_REG: Dimensions not compatible. Return.'
  return
endif
;
; regression
;
;
;print,'**************** LIN_REG *************************'
;print,'* Linear Regrassion program written by           *'
;print,'* M. Sanchez del RIo based in the book:          *'
;print,'* Introduction to Linear Regrassion Analysis     *'
;print,'* (by D. C. Montgomery, E. A. Peck. Wiley (1982) *'
;print,'**************************************************'
;print,' '
print,'**************** LIN_REG results ********************'


Sxx = total(x^2) - total(x)^2/n
Sxy = total(x*y) - total(x)*total(y)/n
meanx=total(x)/n
meany=total(y)/n
fbeta1 = Sxy/Sxx
fbeta0 = meany - fbeta1*meanx
;print,'Sxx = ',Sxx
;print,'Sxy = ',Sxy
print,'Regression slope: fbeta1 = ',fbeta1
print,'Regression intercept: fbeta0 = ',fbeta0
print,'The least square fit is: yfit = '+strcompress(fbeta0,/rem)+$
  ' + '+strcompress(fbeta1,/rem)+' x'
;print,''

;
; sigma
;

Syy = total(y^2) - total(y)^2/n
c = (x-meanx)/Sxx
fsigma2 = (Syy - fbeta1*Sxy)/(n - npar)
;print,'Syy = ',Syy
;print,'fsigma2 = ',fsigma2
;print,''

;
; significance of regression
;
print,'**> testing significance of regression... '
;print,'>> Testing significance of regression: failing'
;print,'>> to reject Ho:fbeta1=0 implies that there is'
;print,'>> no linear relationship between x and y.    '
;print,'>> An analisis of variance procedure is used. '
;print,'>> The test statistics is Fo=(MSR/MSE) where  '
;print,'>> MSR is the regression mean square and MSE  '
;print,'>> the residual mean squares. If Ho is true,  '
;print,'>> the Fo follows the F1,n-2 distribution.    '
;print,'>> '
;print,'>>------------------------------------------<<'
print,' Source of |  Sum of  | Degrees of | mean  |  Fo  '
print,' Variation |  squares | freedom    | square|      '
;print,' Regression   SSR          1          MSR    MSR/MSE'
;print,' Residual     SSE         n-2         MSE           '
;print,' Total        Syy         n-1                       '


SSR = fbeta1* Sxy
SSE = Syy - SSR
MSR = SSR/1
MSE = SSE/(n-npar)
Fo = MSR/MSE
;print,' In our case: '
print,' Regression   '+strcompress(SSR,/REM)+'   1  '+$
	strcompress(MSR,/REM)+'       '+strcompress(Fo,/rem)
print,' Residual     '+strcompress(SSE,/REM)+'      '+ $
	strcompress(n-2,/rem)+'    '+strcompress(MSE,/REM)
print,' Total      '+strcompress(Syy,/REM)+'      '+ $
	strcompress(n-1,/rem)
fval = F_TEST(0.05,1,n-npar)
print,' And Fval = F_TEST(0.05,1,n-npar) is',Fval
if Fo GT Fval then print,' *** We reject Ho -> MAY BE LINEAR ***'
if Fo LE Fval then print,' *** We failed to reject Ho -> NON LINEAR ***'

; 
; interval estimation
;
se_fbeta1=sqrt(MSE/Sxx)
se_fbeta0=sqrt(MSE*( (1./n)+ (meanx^2/Sxx) ))
print,'**> Interval estimation...'
print,' Standard error of the slope fbeta1:     ',se_fbeta1
print,' Standard error of the intercept fbeta0: ',se_fbeta0

PERC = 95.
alpha = (100.-PERC)/100.
beta1low = fbeta1 - student_t(alpha/2,n-npar) * sqrt(MSE/Sxx)
beta1high = fbeta1 + student_t(alpha/2,n-npar) * sqrt(MSE/Sxx)
print,' '+strcompress(beta1low,/rem)+' <= beta1 <= '+$
	strcompress(beta1high,/rem)
sigma2low = (n-npar)*MSE/chi_sqr(alpha/2,n-npar)
sigma2high = (n-npar)*MSE/chi_sqr(1-alpha/2,n-npar)
print,' '+strcompress(sigma2low,/rem)+' <= sigma^2 <= '+$
	strcompress(sigma2high,/rem)

; 
; coefficient of determination
;
R2=SSR/Syy
print,' Coefficient of determination  R^2: ',R2

; 
; residual analysis
;
err = y - (fbeta0 + fbeta1*x)
derr = err / sqrt(MSE)
;
; test of lack of fit (only if one x value is repeated (necessary condition))
;
irepmax = 0
for i=0,n-1 do begin
  irep=1
  for j=i+1,n-1 do begin
    if x(i) EQ x(j) then begin
       irep = irep+1
    endif
  endfor
  if irep GT irepmax then irepmax=irep
endfor
if irepmax GT 1 then begin
  print,'**> Testing the Lack of Fit...'
  ; lof test
  xnew=fltarr(2,n) ; col0: x, col1: multipl
  xnew(1,*) = 1
  ynew=fltarr(irepmax,n) 
  for i=0,n-1 do begin
    if xnew(1,i) NE -1 then begin
      xnew(0,i) = x(i)
      ynew(0,i) = y(i)
      for j=i+1,n-1 do begin
        if x(i) EQ x(j) then begin
	   xnew(0,i) = x(i)
           xnew(1,i) = xnew(1,i)+1
           xnew(1,j) = -1
           ynew(xnew(1,i)-1,i) = y(j)
        endif
      endfor
    endif
  endfor

  check = where(xnew(1,*) > 0)
  xnew=xnew(*,check)
  ynew=ynew(*,check)
  nn=n_elements(xnew(0,*))

  ;print,'x: ',x
  ;for i=0,nn-1 do print,'xnew: ',xnew(0,i),xnew(1,i)
  ;print,'y: ',y
  ;for i=0,nn-1 do print,'ynew: ',ynew(*,i)
  ynewfit = reform(fbeta0 + fbeta1*xnew(0,*))
  ynewmean = fltarr(nn)
  for i=0,nn-1 do ynewmean(i)=total(ynew(*,i))/xnew(1,i)
  sslof = total( xnew(1,*)*(ynewmean-ynewfit)^2)
  sspe = sse-sslof
  dof_pe = n-nn
  dof_lof = nn - npar
  mslof=sslof/dof_lof
  mspe = sspe/dof_pe
  Fo_lof=mslof/mspe
  print,' Source of |  Sum of  | Degrees of | mean  |  Fo  '
  print,' Variation |  squares | freedom    | square|      '
  print,' Regression   '+strcompress(SSR,/REM)+'   1  '+$
        strcompress(MSR,/REM)+'       '+strcompress(Fo,/rem)
  print,' Residual     '+strcompress(SSE,/REM)+'      '+ $
        strcompress(n-2,/rem)+'    '+strcompress(MSE,/REM)
  print,' (Lack of Fit)'+strcompress(SSLOF,/REM)+'      '+ $
        strcompress(dof_lof,/rem)+'    '+strcompress(MSLOF,/REM)+$
	'      '+strcompress(Fo_lof,/rem)
  print,' (Pure Error) '+strcompress(SSPE,/REM)+'      '+ $
        strcompress(dof_pe,/rem)+'    '+strcompress(MSpe,/REM)
  print,' Total      '+strcompress(Syy,/REM)+'      '+ $
        strcompress(n-1,/rem)
  fval = F_TEST(0.05,dof_lof,dof_pe)
  print,' And Fval = F_TEST(0.05,dof_lof,dof_pe) is',Fval
  if Fo_lof GT Fval then print,' *** The regression function is not linear ***'
  if Fo_lof LE Fval then print,' *** We failed to demostrate non linearity ***'
endif
print,'**************** LIN_REG results ended **************'
end

