;+
; NAME:
;	CW_ARRAY
;
; PURPOSE:
;	CW_ARRAY is a compound widget that permits to modify an array.
;	It presents a pulldown menu with two options: Default and Set.
;	If selected the default option, the returned value is the 
;	array with all elements set to zero. If selecting the "Set"
;	option, a modal window will appear that allows the user to modify 
;	the values. 
;
; CATEGORY:
;	Compound widgets.
;
; CALLING SEQUENCE:
;	Widget = CW_ARRAY(Parent)
;
; INPUTS:
;	Parent:	  The ID of the parent widget.
;
; KEYWORD PARAMETERS:
;	UVALUE:	The user value for the widget.
;	TITLE:	A label or title (Default='File: ')
;	VALUE:	The initial array name. It can be Flt, Int or String arrays. 
;	COLUMN:	Display widgets inside the compound widget in column.
;	FRAME:	Display a frame around the compound widget.
;	NAMES: A string array with the same dimension as "Value"
;		containing the titles to display while editing the values.
;	DEFAULT: Sets the pulldown to "Default". Note that this is done
;		independently of the given values in "Value". The values
;		are kept in memory although zeros are returned in the 
;		"Default" option is selected.
;
; OUTPUTS:
;	The ID of the created widget is returned.
;
; SIDE EFFECTS:
;
; PROCEDURE:
;	Uses cw_droplist.
;
; EXAMPLE:
;	;
;	; Use this program to test cw_array
;	;
;	PRO test_arr_event, event
;	
;	Widget_Control, event.id, get_UValue=eventUValue
;	if not(keyword_set(eventUValue)) then eventUValue=''
;	
;	stateid = Widget_Info(event.handler,/Child)
;	Widget_Control, stateid, get_UValue=state, /No_Copy
;	
;	Case eventUValue OF
;	    'DONE': Begin
;		Widget_Control, event.top, /destroy
;		Return
;		End
;	    'DEFAULT': Begin
;		Widget_Control,state.r1,Set_Value='Default'
;		end
;	    'GETVALUE': Begin
;		Widget_Control,state.r1,Get_Value=aa
;		help,aa
;		message,/info, '>> value is: '+vect2string(aa)
;		end
;	    else:
;	ENDCASE
;	Widget_Control, stateid, Set_UValue=state, /No_Copy
;	END ; test_arr_event
;	;
;	;
;	PRO test_arr
;	
;	tlb=widget_base(/col)
;	tmp=widget_base(tlb) ; to store state
;	
;	title = Widget_Button(tlb,Value='DONE',UValue='DONE')
;	title = Widget_Button(tlb,Value='GETVALUE',UValue='GETVALUE')
;	title = Widget_Button(tlb,Value='Set Default',UValue='DEFAULT')
;	
;	range1 = cw_array(tlb,Title='>>:',value=[1.,2,3],Default=1,$
;		NAMES=['My first value','My 2nd value','My 3rd value'])
;	
;	state = {r1:range1}
;	
;	widget_control,Widget_Info(tlb,/Child),set_uvalue=state
;	widget_control,tlb,/realize
;	xmanager, 'test_arr', tlb
;	
;	END ; test_arr
;	
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio, srio@esrf.fr
;	December, 1997	 (derived from the draft cw_xshrange.pro).
;	98-04-23 srio@esrf.fr fixes a bug in CW_Array_Set
;
;-
;
;===============================================================
;
Pro CW_Array_Set, id, value
;
; acceps 'Default' and , for instance, [0,3,3]
;
child=Widget_Info(id, /Child)
Widget_Control, child, Get_UValue=state, /No_Copy
tmp = state.value

IF Type(value) EQ 7 THEN BEGIN ; string
  IF value(0) EQ 'Default' THEN BEGIN
    Widget_Control,state.dropid,Set_Value='0' 
    GoTo,out
  ENDIF ELSE BEGIN 
    Widget_Control,state.dropid,Set_Value='1' 
  ENDELSE
ENDIF ELSE Widget_Control,state.dropid,Set_Value='1'

for i=0,n_elements(value)-1 do tmp(i) = value(i)
state.value = tmp

out:
Widget_Control, child, Set_UValue=state, /No_Copy
END
;
;===============================================================
;
Function CW_Array_Get, id
child=Widget_Info(id, /Child)
Widget_Control, child, Get_UValue=state, /No_Copy

val = state.value

tmp=''
Widget_Control,state.dropid,Get_Value=tmp

CASE tmp OF
 '0': BEGIN ; Default
	out = val * 0
     END
 '1': BEGIN ; Set...
	out = val
     END
  else:
ENDCASE

Widget_Control, child, Set_UValue=state, /No_Copy
Return, out
END
;
;===============================================================
;
Function CW_Array_Event, event

child = Widget_Info(event.handler, /Child)
Widget_Control, child, Get_UValue=state, /No_Copy

;
;update state
;
index=0
widget_control,state.dropid,get_value=index

IF index EQ '1' THEN BEGIN ; Set...
	val = state.value
	varnames='a'+strcompress(indgen(n_elements(val)),/Rem)
	strval = create_struct(varnames(0),val(0))
	if n_elements(val) GT 1 then begin
	  for i=1,n_elements(val)-1 do strval =$
		create_struct(strval, varnames(i),val(i))
	endif
	XScrMenu,strval,ACTION=action,/NoType,Titles=state.names,$
		WTitle='Set values...'
	IF action EQ 'DONT' THEN GoTo,out
	for i=0,n_tags(strval)-1 do val(i) = strval.(i)
	state.value = val
ENDIF
out:
;
; if all values are equal, set the droplist to 'Default'
;
isame = 0
tmp = state.value(0)
nn=n_elements(state.value)
IF nn GT 1 THEN $
	FOR i=1,nn-1 DO $
		IF tmp EQ state.value(i) THEN isame=1 ELSE isame=0
IF isame THEN Widget_Control,state.dropid,Set_value='0'
;
; exit
;
Widget_Control, child, Set_UValue=state, /No_Copy
;
; return event structure
;
WIDGET_CONTROL, event.id, get_uvalue=uvalue
if n_elements(uvalue) EQ 0 then uvalue=0
return, { ID:event.handler, TOP:event.top, HANDLER:0L, value:uvalue }

END
;
;
;===============================================================
;
Function CW_Array, parent, UValue=uval, Title=init_label, $
  Value=Init_Value,Column=Column,Frame=Frame,Names=names, $
  Default=default

On_Error,1
If N_Params() NE 1 THEN Message,'Must have one parameter'
If N_Elements(uval) EQ 0 THEN uval = 0
If N_Elements(Frame) EQ 0 THEN Frame = 0
If N_Elements(init_label) EQ 0 THEN init_label = 'Range: '
If N_Elements(Init_Value) EQ 0 THEN BEGIN
  Init_Value = [0.0, 0.0]
  idefault=1
ENDIF ELSE idefault=0
IF KeyWord_Set(default) then idefault=1

If N_Elements(names) EQ 0 THEN names = 'val'+$
	strcompress(indgen(n_elements(Init_Value)),/Remove)

row=1
If keyword_set(column) then row=0 else Column=0



cw_tlb = Widget_Base(parent, Frame=Frame, Uvalue=uval, $
   Column=Column,Row=Row, $
   Event_Func='CW_Array_Event',$
   Pro_Set_Value='CW_Array_Set', $
   Func_Get_Value='CW_Array_Get')

label = Widget_Label(cw_tlb, Value=init_label)

IF idefault EQ 1 THEN List = ['0','<Default>','Set...'] ELSE $
   List = ['1','<Default>','Set...']
drop = CW_DropList(cw_tlb,Value=List)


state = {labelid:label, dropid:drop, value:init_value, names:names}
Widget_Control, label, Set_UValue=state , /No_Copy

widget_control,cw_tlb,set_value=init_value

Return, cw_tlb
END
