/* d4ceff.c */
/* */
/* This program reads in numor files corresponding to a D4c scan for
/* calibrating detector cell efficiencies, calculates the efficiencies
/* of all 9*64 cells, and writes the result to a *.eff file having
/* sections for each detector's cells numbered 1 to 64.  The minimum
/* and maximum angles of D4c detector 1's lowest angle cell are
/* -66 (hard) or -63 (soft) deg, and +10.1 deg, respectively.
/*
/* For a given 2theta (lowest angle cell of detector 1), the angle of
/* detector n's cell m is given by:
/*     angle_scatt = 2theta + (n-1)*15 + (m-1)*0.125
/*
/* So, for a full 2theta scan of -78 < 2theta < 10, we can compute the
/* total angular range of detection: -78 < angle_scatt < +138, as well
/* as the common angle range for a given series of detectors:
/*
/* Detector series      common angle_scatt range (deg)
/*
/*     1,2,3                -40 to -22   (not +10 because of direct beam)
/*                                       (not -20 because of window/bs cutoff)
/*
/*     3,4,5,6              +19.5 to +40   (not +5 because of pic V a +18.5)
/*
/*     6,7,8,9              +50 to +85   (full range no problem here)
/*
/* so that detectors 3 and 6 become the two normalisation "pivots".
/*
/* */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "fischer/hfit/hnr.h"
#include "fischer/hfit/nrutil.h"
#include "fischer/c/henrylib.h"
#include "d4priv/c/d4.h"

main(int argc, char **argv)
{
	void quit(int code);
	int i,j,k,l,m,n,icell,idata,jn;  /* loop indices */
	int detid;
	int brokeninput = 0;  /* flag for non-sequential numor file input */
	int c;  /* for getc() */
	int argcsave;
	char **argvsave;
	int length,slen;
	char *cpars;
	int *ipars;  /* default is 4 bytes (INTEGER*4) */
	double *dpars,*params1;
	double avedate0,datedetson;
	double monitor,time;  /* from 1st numor */
	int inumor,numor,numnumors,numorbeg,numorfin;
	double *moncnts,*timcnts,*avedate;
	int *goodnumor,*numorfile,badnumor[1001],badn=0;
	int stepnum,numsteps;
	int *ignore,ignlen,ignflag;
	int normtype = 0;
	double *detdist;
	int *detused,*icellstep;
	double ***det;
	int **detdata;
	int **ithdes,**ithlue;
	int **anglesseen,**cellsseen;
	int **bad;
	double **eff;
	double **eff123;
	double **eff3456;
	double **eff6789;
	double eff3tot123,eff3tot3456,eff6tot3456,eff6tot6789;
	double ***cellcoups;
	int mindata,maxdata;
	int mindata123,maxdata123;
	int mindata3456,maxdata3456;
	int mindata6789,maxdata6789;
	/* double ***cellcoups123; */
	/* double ***cellcoups3456; */
	/* double ***cellcoups6789; */
	double dtemp;
	int status = 0;
	FILE *fp,*fph,*fpo,*fpe;
	int dets = 9;
	int type = 'S';
	/* char *helpfile = "/hosts/d4sgi/home/dif/d4priv/help/d4ceff.help"; */
	char *helpfile = SPACES80;
	char *ignfile = SPACES80;
	char *badfile = SPACES80;
	char *badused = SPACES80;
	char *outfile = SPACES80;
	char *syscmd = SPACES80;
	int outtype = SGI;
	char *cnumor = "      ";
	/* char *disk = "/hosts/romeo/d1/ascii"; */
        /* char *disk = "/hosts/idefix/illdata/data"; */
        /* char *disk = "/home/dif"; */
        /* CORRECT: char *disk = "/home/dif/d4c"; */
        char *disk = SPACES80;
        /* char *disk = "/hosts/idefix/illdata/data-1"; */
        /* char *instr = "d4c"; */
        char *instr = "";

/* -------------------------- end of declarations -------------------------- */

	/* build up helpfile and disk paths using environment variables */

	/* 14 juin 00 -- replace getenv() with getenvchk() */
	strcpy( helpfile,getenvchk("D4RESPPATH") );
	strcat( helpfile,"/help/d4ceff.help" );
	/* strcpy( disk,getenvchk("D4CDISK") ); */
	strcpy( disk,getenvchk("D4CDATADISK") );

	/* dimension and zero some once-and-only stuff */

	argvsave = ppchar(1000);
	argvsave = argv;
	argcsave = argc;
	detdata = ppint(NUMDETSD4C+1);  /* index starts at zero */
	detused = ivector(1,NUMDETSD4C);
	icellstep = ivector(1,NUMDETSD4C);
	detdist = dvector(1,NUMDETSD4C);
	bad = imatrix(1,NUMDETSD4C,1,NUMCELLS);
	anglesseen = imatrix(1,NUMDETSD4C,1,NUMCELLS);
	eff = dmatrix(1,NUMDETSD4C,1,NUMCELLS);
	eff123 = dmatrix(1,NUMDETSD4C,1,NUMCELLS);
	eff3456 = dmatrix(1,NUMDETSD4C,1,NUMCELLS);
	eff6789 = dmatrix(1,NUMDETSD4C,1,NUMCELLS);

	/* break up cellcoups into smaller arrays (2 juin 00) */
	/*    to avoid memory allocation problem */
	/* use angle ranges +/- 5 deg given in comments above */
	/* cellcoups = d3tensor(1,NUMDETSD4C,1,NUMCELLS, */
	/*	  -NUMDATAD4C,NUMDATAD4C); */

	mindata123 = -41*1000/BINANGLED4C;
	maxdata123 = -22*1000/BINANGLED4C;  /* eviter cutoff fenetre d'Al */
	/* cellcoups123 = d3tensor(1,3,1,NUMCELLS,mindata123,maxdata123); */
	/* mindata3456 = 19*1000/BINANGLED4C; */
	mindata3456 = 19500/BINANGLED4C; /* eviter le pic V a 18.5 deg */
	maxdata3456 = 41*1000/BINANGLED4C;
	/* cellcoups3456 = d3tensor(3,6,1,NUMCELLS,mindata3456,maxdata3456); */
	mindata6789 = 49*1000/BINANGLED4C;
	maxdata6789 = 86*1000/BINANGLED4C;
	/* cellcoups6789 = d3tensor(6,9,1,NUMCELLS,mindata6789,maxdata6789); */

	/* reduce size of cellcoups and cellsseen arrays (2 juin 00) */
	/* cellsseen = imatrix(1,NUMDETSD4C,-NUMDATAD4C,NUMDATAD4C); */
	/*
	/* absolute largest angle_scatt range: -- doesn't work, too big */
	/* mindata = -66*1000/BINANGLED4C;
	/* maxdata = 139*1000/BINANGLED4C;
	/*
	/* reduce it to only what the efficiency scans need +/- 5 deg: */
	mindata = -42*1000/BINANGLED4C;
	maxdata = 87*1000/BINANGLED4C;
	cellcoups = d3tensor(1,NUMDETSD4C,1,NUMCELLS,mindata,maxdata);
	cellsseen = imatrix(1,NUMDETSD4C,mindata,maxdata);


	for( i=1; i<=NUMDETSD4C; i++ ) {
		for( icell=1; icell<=NUMCELLS; icell++ ) {
			anglesseen[i][icell]=0; /* also done in calceffd4c.c */
			for( idata=mindata; idata<=maxdata; idata++ )
			  cellcoups[i][icell][idata]=0.0;
			/*
			for( idata=mindata123; idata<=maxdata123; idata++ ) 
			  cellcoups123[i][icell][idata]=0.0;
			for( idata=mindata3456; idata<=maxdata3456; idata++ ) 
			  cellcoups3456[i][icell][idata]=0.0;
			for( idata=mindata6789; idata<=maxdata6789; idata++ ) 
			  cellcoups6789[i][icell][idata]=0.0;
			*/
		}
	}
	for( i=1; i<=NUMDETSD4C; i++ ) {
		/* for( idata=-NUMDATAD4C; idata<=NUMDATAD4C; idata++ ) { */
		for( idata=mindata; idata<=maxdata; idata++ ) {
			cellsseen[i][idata]=0;
		}
	}

	/* if only entered "d4ceff", give syntax message and exit */
	if (argc==1) quit(9);  /* argc = # tokens, incl argv[0] = "d4ceff" */
	--argc; ++argv;  /* thus skip "d4ceff" */

	beginagain: ;   /* for continuing to enter numor ranges */

	/* look for hyphen "-" tokens */
	while (argc > 0 && **argv == '-') {
		switch ((*argv)[1]) {
		case 'h':
			if ((fph = fopen(helpfile, "r")) == NULL) {
				fprintf(stderr,
			  "@@@@d4ceff: Ne peut pas ouvrir \"%s\"\n",helpfile);
				if( brokeninput == 0 ) quit(1);
				else goto nextinput;
			}
			while ((c=getc(fph)) != -1) putc(c,stderr);
			fclose(fph);
			if( brokeninput == 0 ) quit(0);
			else goto nextinput;
			break;
		case 'o':
			strcpy(outfile,*++argv);
			/* fprintf(stderr,"outfile = %s\n",outfile); */
			--argc;
			break;
		case 'b':
			strcpy(badfile,*++argv);
			/* fprintf(stderr,"badfile = %s\n",badfile); */
			--argc;
			break;
		case 'i':
			strcpy(ignfile,*++argv);
			/* fprintf(stderr,"ignfile = %s\n",ignfile); */
			--argc;
			break;
		case 'd':
			brokeninput++;  /* incremented */
			break;
		default:
			fprintf(stderr, "@@@@d4ceff: Bad token: %s\n", *argv);
			if( brokeninput == 0 ) quit(1);
			else goto nextinput;
		}   /* end switch */
		--argc; ++argv;
	}  /* end while for finding '-' */

	/* get other tokens */
	if (argc < 1) {  /* argc always = # tokens remaining */
		fprintf(stderr, "@@@@d4ceff: Pas assez de tokens\n");
		printargs(argc,argv);
		if( brokeninput == 0 ) quit(1);
		else goto nextinput;
	}
	numorbeg = atoi(*argv);

	if (argc > 1) {
		--argc; ++argv;
		i = atoi(*argv);
		if (i<numorbeg) {
			fprintf(stderr,
			  "@@@@d4ceff: numorfin must >= numorbeg\n");
			if( brokeninput == 0 ) quit(1);
			else goto nextinput;
		}
		numorfin = i;
	}
	else numorfin=numorbeg;
	if (argc > 1) {
		--argc; ++argv;
		i = atoi(*argv);
		if (i<1 || i>9) {
			fprintf(stderr,
			  "@@@@d4ceff: must have 1 <= dets <= 9\n");
			if( brokeninput == 0 ) quit(1);
			else goto nextinput;
		}
		dets = i;
	}
	if (argc > 1) {
		--argc; ++argv;
		disk = strdup(*argv);  /* can't strcpy here, faute d'espace */
	}
	if (argc > 1) {
		--argc; ++argv;
		instr = strdup(*argv);  /* can't strcpy here, faute d'espace */
	}
	/*
	fprintf(stderr,
	  " %d %d %d %s %s\n",numorbeg,numorfin,dets,disk,instr);
	*/

	/* ---------------- end of getting command line args --------------- */

	switch (dets) {  /* for now, make the following detused assignments */
		case 1:
			detused[1] = TRUE;
			/* detused[2] = FALSE; */
			break;
		case 2:
			/* detused[1] = FALSE; */
			detused[2] = TRUE;
			break;
		case 3:
			detused[3] = TRUE;
			/* detused[1] = TRUE; */
			/* detused[2] = TRUE; */
			break;
		case 4:
			detused[4] = TRUE;
			break;
		case 5:
			detused[5] = TRUE;
			break;
		case 6:
			detused[6] = TRUE;
			break;
		case 7:
			detused[7] = TRUE;
			break;
		case 8:
			detused[8] = TRUE;
			break;
		case 9:
			detused[1] = TRUE;
			detused[2] = TRUE;
			detused[3] = TRUE;
			detused[4] = TRUE;
			detused[5] = TRUE;
			detused[6] = TRUE;
			detused[7] = TRUE;
			detused[8] = TRUE;
			detused[9] = TRUE;
			break;
		default:
			fprintf(stderr,
			  "@@@@d4ceff: this should never happen\n");
			quit(1);
			break;
	}

	numnumors = numorfin-numorbeg+1;  /* number of numor files */


	/* ---------- get some parameters from first numor file ----------- */

	if ((fp=opennumord4c(&disk,instr,numorbeg)) == NULL) {
		if( brokeninput <= 1 ) quit(9);
		else goto nextinput;
	}

	status=getpars(fp,numorbeg,1,'F',&cpars,&ipars,&params1,&length);
	if (status==EOF) {
		fprintf(stderr,
	  "@@@@d4ceff: EOF reached in getpars for numor file %d\n",numor);
		if( brokeninput <= 1 ) quit(EOF);  /* EOF = -1 by stdio.h */
		else goto nextinput;
	}
	if (status!=0) {
		if( brokeninput <= 1 ) quit(status);
		else goto nextinput;
	}

	/* get monitor and time -- one will be same for all numors */
        monitor = params1[1];  /* first parameter in numor, in fact */
        /* monitor = 100000.; */
        /* time = params1[2]/100.0;  /* in 100th's of sec, for now */
        time = params1[2];  /* in seconds, for now */
        normtype = (int) (params1[3]+0.5);
 
        stepnum = dround(params1[6]);
        if (stepnum!=1) {
                fprintf(stderr,
          "@@@@d4ceff: WARNING: stepnum = %d != 1 in 1st numor file %d\n",
          stepnum,numorbeg);
        }
 
        /* get cell spacing in angle */
        detdist[1] = params1[21];  /* in meters, for now */
        /* detdist[2] = params1[26]; */
        for( i=1; i<=NUMDETSD4C; i++ ) {
                if (!detused[i]) continue;
                /* set all 9 detectors to 1146 mm rayon */
                detdist[i] = params1[21];
                icellstep[i] = cstepd4c(i,detdist[i]);
                /* fprintf(stderr,"icellstep[%d] = %d\n",i,icellstep[i]); */
        }

	free_dvector(params1,1,length);

	/* get starting and stopping date/time */
        if( status = numdated4c(fp,numorbeg,&avedate0) != 0 ) {
                if( brokeninput <= 1 ) quit(status);
                else goto nextinput;
        }
        /* fprintf(stderr,"avedate0 = %f\n",avedate0); */
        datedetson = unixdate(DATEDETSOND4C);
        /* fprintf(stderr,"datedetson = %f\n",datedetson); */

	fclose(fp);  /* close 1st numor file */


	/* ------------ allocate memory dependent on numnumors ------------ */

	det = d3tensor(1,NUMDETSD4C,1,numnumors,1,NUMCELLS);
	ithdes = imatrix(1,NUMDETSD4C,1,numnumors);
	ithlue = imatrix(1,NUMDETSD4C,1,numnumors);
	moncnts = dvector(1,numnumors);
	timcnts = dvector(1,numnumors);
	avedate = dvector(1,numnumors);
	goodnumor = ivector(1,numnumors);
	numorfile = ivector(1,numnumors);


	/* get bad file */

	if (strcmp(badfile,SPACES80)==0) badfile="badd4c.bad";  /* default */
	if (status=getbadd4c(badfile,badused,bad) != 0) {
		if( brokeninput <= 1 ) quit(status);
		else goto nextinput;
	}

	/* for now, have NO bad cells for all 9 dets */
        /* for( i=1; i<=NUMDETSD4C; i++ ) {
        /*         for( j=1; j<=NUMCELLS; j++ ) {
        /*                 bad[i][j]=1;
        /*         }
        /* } */

	/* get ignore file */

	if (strcmp(ignfile,SPACES80)==0) ignfile="ign.ign";  /* default */
	if (status=getign(ignfile,&ignore,&ignlen) != 0) {
		if( brokeninput <= 1 ) quit(status);
		else goto nextinput;
	}


	/* ------------------- big reading loop on numor ------------------ */

	for( numor=numorbeg; numor<=numorfin; numor++ ) {

	inumor = numor-numorbeg+1;

	/* check for numors to be ignored */
	ignflag = 0;
	for( i=1; i<=ignlen; i++ ) if (ignore[i]==numor) ignflag = 1;
	if( ignflag ) {
		fprintf(stderr,
		  "@@@@d4ceff: Numor file %d ignored as requested\n",numor);
		goodnumor[inumor] = FALSE;
		badn++;
		badnumor[badn] = numor;
		continue;  /* get next numor file */
	}

	fprintf(stderr,"****d4ceff: Reading numor file %d ...\n",numor);

	/* Keep going if a numor can't be found, since there are
	   sometimes gaps in the numor numbering (camac errors, etc) */
	if ((fp=opennumord4c(&disk,instr,numor)) == NULL ) {
                goodnumor[inumor] = FALSE;
                badn++;
                badnumor[badn] = numor;
                fclose(fp);  /* close numor file */
                continue;  /* get next numor file */
        }
	/* okay, it's a successfully opened numor */

	/* get some parameters from each numor file */

	status=getpars(fp,numor,1,'F',&cpars,&ipars,&params1,&length);
	if (status==EOF) {
		fprintf(stderr,
	  "@@@@d4ceff: EOF reached in getpars for numor file %d\n",numor);
		if( brokeninput <= 1 ) quit(EOF);  /* EOF = -1 by stdio.h */
		else goto nextinput;
	}
	if (status!=0) {
		if( brokeninput <= 1 ) quit(status);
		else goto nextinput;
	}

	/* get desired and read detector angles */
	ithdes[1][inumor] = dround(params1[12]*1000.0);
	ithlue[1][inumor] = dround(params1[11]*1000.0);
	ithdes[2][inumor] = dround(params1[17]*1000.0);
	ithlue[2][inumor] = dround(params1[16]*1000.0);
	/* for( i=1; i<=NUMDETSD4C; i++ ) */
	for( i=1; i<=1; i++ )  /* only one 2theta for all 9 detectors */
		if( ithlue[i][inumor] != ithdes[i][inumor] && detused[i] )
			/*
			fprintf(stderr,
  "@@@@d4ceff: In numor file %d: (ithlue=%d)-(ithdes=%d) = %d for det %d\n",
			  numor,ithlue[i][inumor],ithdes[i][inumor],
			  (ithlue[i][inumor]-ithdes[i][inumor]),i);
			*/
			fprintf(stderr,
  "@@@@d4ceff: In numor file %d: (ithlue=%d)-(ithdes=%d) = %d/1000 degrees\n",
			  numor,ithlue[i][inumor],ithdes[i][inumor],
			  (ithlue[i][inumor]-ithdes[i][inumor]));

	stepnum = dround(params1[6]);  /* might be useful to check */

	/* get monitor counts and time -- check monitor counts are zero */
	moncnts[inumor] = params1[1];  /* first parameter in numor, in fact */
	/* moncnts[inumor] = 100000.; */
	/* timcnts[inumor] = params1[2]/100.0;  /* in 100th's of sec, for now */
	timcnts[inumor] = params1[2];  /* in seconds, for now */
	/* if( moncnts[inumor] < 0.1 ) { */
	if( moncnts[inumor] < 0.1 && normtype == 0 ) {
		fprintf(stderr,
  "@@@@d4ceff: In numor file %d, ZERO monitor counts found (numor skipped)\n",
		  numor);
		goodnumor[inumor] = FALSE;
		badn++;
		badnumor[badn] = numor;
		free_dvector(params1,1,length);
		fclose(fp);  /* close numor file */
		continue;  /* get next numor file */
	}
	/* double if statement to avoid accessing moncnts[0] : */
	if( inumor != 1 )
	  if( dround(moncnts[inumor]) != dround(moncnts[inumor-1])
	  && goodnumor[inumor-1] )
		  fprintf(stderr,
  "@@@@d4ceff: In numor file %d: change in moncnts(=%d) found (keep going)\n",
		  numor,dround(moncnts[inumor]));

	free_dvector(params1,1,length);

	/* get starting and stopping date/time */
	if( status = numdated4c(fp,numor,&avedate[inumor]) != 0 ) {
		if( brokeninput <= 1 ) quit(status);
		else goto nextinput;
	}
	avedate[inumor] -= datedetson;  /* origin is date dets regulated */
	/* fprintf(stderr,"avedate[%d] = %f\n",inumor,avedate[inumor]); */


	/* get data from numor file */

	for( i=1; i<=NUMDETSD4C; i++ ) {
		if (!detused[i]) continue;
		status=getpars(fp,numor,i,'S',&cpars,&detdata[i],&dpars,&length);
		if (status==EOF) {
			fprintf(stderr,
	  "@@@@d4ceff: EOF reached in getpars for numor file %d\n",numor);
			if( brokeninput <= 1 ) quit(EOF);  /* EOF = -1 */
			else goto nextinput;
		}
		if (status!=0) {
			if( brokeninput <= 1 ) quit(status);
			else goto nextinput;
		}
		if (length != NUMCELLS) {
			fprintf(stderr,
			  "@@@@d4ceff: (data)length=%d != NUMCELLS=%d\n",
			  length,NUMCELLS);
			if( brokeninput <= 1 ) quit(1);
			else goto nextinput;
		}
	}

	fclose(fp);  /* close numor file */

	fprintf(stderr,"$$$$d4ceff: Numor file %d decoded\n",numor);

	/* assign 3d double array det */
	/* NOTE: need to reverse order of data for numors < 3443 */
	/*   en faisant: det[i][inumor][j] --> det[i][inumor][NUMCELLS+1-j] */

	for( i=1; i<=NUMDETSD4C; i++ ) {
		if (!detused[i]) continue;
		for( j=1; j<=NUMCELLS; j++ )
		  det[i][inumor][j] = (double)detdata[i][j];
		free_ivector(detdata[i],1,length);
	}

	goodnumor[inumor] = TRUE;

	}
	/* ---------------- end big reading loop on numor ---------------- */


	/* -------------------------- bin data --------------------------- */
	
	for( i=1; i<=NUMDETSD4C; i++ ) {
		if (!detused[i]) continue;
		/* note change from ithdes[i] to ithdes[1] */
		if( status = bineffd4c(i,det[i],bad[i],numnumors,numorbeg,
		  goodnumor,ithdes[1],icellstep[i],moncnts,timcnts,
		  cellcoups[i],cellsseen[i],mindata,maxdata)
		    != 0 ) quit(status);
	}


	/* ----- calculate monitor counts per time and write result ------ */

	/* This doesn't work for some reason */
	/* if( status = outmon(outfile,numorbeg,numorfin,numnumors,avedate,
	/*  moncnts,timcnts,goodnumor) != 0 ) quit(status);
	/* */


	/* ---------- free up stuff dimensioned to numor range ----------- */

	free_d3tensor(det,1,NUMDETSD4C,1,numnumors,1,NUMCELLS);
	free_dvector(moncnts,1,numnumors);
	free_dvector(timcnts,1,numnumors);
	free_dvector(avedate,1,numnumors);
	free_ivector(goodnumor,1,numnumors);
	free_ivector(numorfile,1,numnumors);
	free_imatrix(ithdes,1,NUMDETSD4C,1,numnumors);
	free_imatrix(ithlue,1,NUMDETSD4C,1,numnumors);


	/* ----------- cycle on input if they used '-b' option ----------- */

	nextinput: ;

	if( brokeninput > 0 ) {   /* keep incrementing brokeninput */
		do {
			fprintf(stderr,"\n");
			fprintf(stderr,
">>>> Enter post-'d4ceff' command line for next numor range ('.' if done) :\n");
			fprintf(stderr,
"  { [-h -o f.out -b f.bad -i f.ign] nbeg [nfin dets disk instr] }");
			fprintf(stderr,"\n");
		} while( (argc=getargs(argv)) == 0 );
		if( **argv == '.' ) goto alldone;
		brokeninput++;
		goto beginagain;
	}

	alldone: ;  /* no more broken input */

	/* goto effsequalone;  /* for now (28 mai 00) */


	/* ------------ sum up data and calculate efficiencies ----------- */

	/* for( i=1; i<=NUMDETSD4C; i++ ) {
	/* 	if (!detused[i]) continue;
	/* 	if( status = calceff(i,bad[i],cellcoups[i],cellsseen[i],
	/* 	  anglesseen[i],eff[i]) != 0 ) quit(status);
	/* }
	/* */

	/* Note that the anglesseen array below gets overwritten for the */
	/* pivot detectors (dets 3 and 6). */

	if( status = calceffd4c(1,3,bad,cellcoups,cellsseen,anglesseen,
	  eff123,mindata123,maxdata123) != 0 ) quit(status);
	if( status = calceffd4c(3,6,bad,cellcoups,cellsseen,anglesseen,
	  eff3456,mindata3456,maxdata3456) != 0 ) quit(status);
	if( status = calceffd4c(6,9,bad,cellcoups,cellsseen,anglesseen,
	  eff6789,mindata6789,maxdata6789) != 0 ) quit(status);

	/* Calculate scaling using pivot detectors 3 and 6 */
	eff3tot123 = 0.0;
	eff3tot3456 = 0.0;
	eff6tot3456 = 0.0;
	eff6tot6789 = 0.0;
	for( icell=1; icell<=NUMCELLS; icell++ ) {
		if( bad[3][icell] == 1 ) {
			eff3tot123 += eff123[3][icell];
			eff3tot3456 += eff3456[3][icell];
		}
		if( bad[6][icell] == 1 ) {
			eff6tot3456 += eff3456[6][icell];
			eff6tot6789 += eff6789[6][icell];
		}
	}

	fprintf(stderr,"\n");
	fprintf(stderr,"****d4ceff: eff3tot123 = %g\n",eff3tot123);
	fprintf(stderr,"****d4ceff: eff3tot3456 = %g\n",eff3tot3456);
	fprintf(stderr,"****d4ceff: eff6tot3456 = %g\n",eff6tot3456);
	fprintf(stderr,"****d4ceff: eff6tot6789 = %g\n",eff6tot6789);

	/* Rescale efficiences of eff123 and eff6789 wrt eff3456 */
	/* and then assign final values to eff array -- note that */
	/* I take an average for dets 3 and 6 to improve statistics */

	for( icell=1; icell<=NUMCELLS; icell++ ) {
		for( detid=1; detid<=2; detid++ ) {
			if( bad[detid][icell] == 1 ) eff[detid][icell]
			  = eff123[detid][icell] * eff3tot3456 / eff3tot123;
			else eff[detid][icell] = -1;
		}
		if( bad[3][icell] == 1 )
		  eff[3][icell] = 0.5 * ( eff3456[3][icell]
		    + eff123[3][icell] * eff3tot3456 / eff3tot123 );
		else eff[3][icell] = -1;
		for( detid=4; detid<=5; detid++ ) {
			if( bad[detid][icell] == 1 ) eff[detid][icell]
			  = eff3456[detid][icell];
			else eff[detid][icell] = -1;
		}
		if( bad[6][icell] == 1 )
		  eff[6][icell] = 0.5 * ( eff3456[6][icell]
		    + eff6789[6][icell] * eff6tot3456 / eff6tot6789 );
		else eff[6][icell] = -1;
		for( detid=7; detid<=9; detid++ ) {
			if( bad[detid][icell] == 1 ) eff[detid][icell]
			  = eff6789[detid][icell] * eff6tot3456 / eff6tot6789;
			else eff[detid][icell] = -1;
		}
	}


	/* effsequalone: ; /* (28 May 00) for now, make all efficiencies = 1 */
	/*
	/* for( i=1; i<=NUMDETSD4C; i++ ) {
	/*	if (!detused[i]) continue;
	/*	for( j=1; j<=NUMCELLS; j++ ) {
	/*		eff[i][j] = 1.0;
	/* 	}
	/* }
	/* */


	/* ------------ open output file(s) and write result ------------- */

        /* if( status = outeff(dets,outfile,badfile,badused,argcsave,argvsave,
	  numorbeg,numorfin,eff,anglesseen) != 0 ) quit(status);
	*/

	/* Note that the anglesseen array below is not used in outeffd4c.c */

        if( status = outeffd4c(dets,outfile,badfile,badused,argcsave,argvsave,
	  numorbeg,numorfin,eff,anglesseen) != 0 ) quit(status);


	/* ------------------ free up rest of stuff ---------------------- */

	free(argvsave);
	free(detdata);  /* had simply allocated NUMDETSD4C+1 pointers */
	free_ivector(detused,1,NUMDETSD4C);
	free_ivector(icellstep,1,NUMDETSD4C);
	free_dvector(detdist,1,NUMDETSD4C);
	free_imatrix(bad,1,NUMDETSD4C,1,NUMCELLS);
	free_imatrix(anglesseen,1,NUMDETSD4C,1,NUMCELLS);
	free_dmatrix(eff,1,NUMDETSD4C,1,NUMCELLS);
	free_dmatrix(eff123,1,NUMDETSD4C,1,NUMCELLS);
	free_dmatrix(eff3456,1,NUMDETSD4C,1,NUMCELLS);
	free_dmatrix(eff6789,1,NUMDETSD4C,1,NUMCELLS);
	/* reduce range of cellcoups and cellsseen for now (2 juin 00) */
	/* free_d3tensor(cellcoups,1,NUMDETSD4C,1,NUMCELLS,
	  -NUMDATAD4C,NUMDATAD4C); */
	/* free_imatrix(cellsseen,1,NUMDETSD4C,-NUMDATAD4C,NUMDATAD4C); */
	free_d3tensor(cellcoups,1,NUMDETSD4C,1,NUMCELLS,mindata,maxdata);
	free_imatrix(cellsseen,1,NUMDETSD4C,mindata,maxdata);

	free_ivector(ignore,1,ignlen);

	quit(0);  /* All's well */
}

void quit(int code)
{
	if (code == 0) 
		fprintf(stderr,"$$$$Successful termination of d4ceff\n\n");
	else if (code == 9) {
		fprintf(stderr,
"--->  d4ceff [-h -d -o f.out -b f.bad -i f.ign] nbeg [nfin dets disk instr]");
		fprintf(stderr,"\n");
		fprintf(stderr,
"    ( [args] are optional; defaults are: -o numorfin.eff, -b badd4c.bad,\n");
		fprintf(stderr,
"       numorfin=numorbeg, dets=9, disk=/hosts/d4c/users/data, instr=.null. )\n");
	}
	else fprintf(stderr,"@@@@Error code %d in d4ceff\n",code);

	exit(code);
}

