#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <vector>
#include <string>
#include "fglobal.h"
#include "cmd.h"
#include "wshare.h"
#include "baseGraph.h"
#include "axis.h"
#include "spacebox.h"
#include "space.h"
#include "new_read.h"
#include "areaInt.h"
#include "extract.h"

const double PI(3.141592653589);


void
XrayMaps::changeXYsize(const int horr,const int vert)
  /* Allow the x,y size of the matrix ot be changed 
     Additionally change the size of the matrix */
{
  if (vert!=n_vp || horr!=n_hp)
    {
      if (values)
	freematrix(values);
      values = fmatrix(horr,vert);
      n_pts=horr * vert;
      n_hp=horr;
      n_vp=vert;
    }
  return;
} 

int 
XrayMaps::setmaxmin(float& minv,float& maxv,const int zero_flag) const
  /* Get the max and min values of the data set, 
     if zero_flag is set then ignore negative values 
  */
{
  if (!n_hp || !n_vp) return 1;
  float* V;
  float Mv=values[0][0];
  float Xv=values[0][0];
  for(int i=0;i<n_vp;i++)
    {
      V=values[i];
	for(int j=0;j<n_hp;V++,j++)
	  {
	    if (*V>Xv) 
	      Xv= *V;
	    else if (*V<Mv)
	      Mv= *V;
	  }
    }
  if (zero_flag && Mv<0.0) Mv=0.0;
  if (Xv<=Mv) return 1;
  minv=Mv;
  maxv=Xv;
  return 0;
}

void
XrayMaps::rebin(const int nxp,const int nyp,const double fcx,
		const double fcy,const double scx,const double scy)
{
  if (nxp<0 || nyp<0) return;
  float **vtemp=fmatrix(nyp,nxp);
  for(int i=0;i<nyp;i++)
    for(int j=0;j<nxp;j++)
      vtemp[i][j]=0.0;
  double X1,X2,Y1,Y2;
  double stepx=(scx-fcx)/nxp;
  double stepy=(scy-fcy)/nyp;
  Y1=fcy-stepx;  Y2=fcy; 
  int sx,ex,sy,ey;
  int isx,isy,iex,iey;
  double frac,frac2;
  for(int i=0;i<nyp;i++)
    {
      X1=fcx-stepx; X2=fcx;
      Y1+=stepy; Y2+=stepy;
      for(int j=0;j<nxp;j++)
	{
	  X1+=stepx;
	  X2+=stepx;
	  sx=(int) X1;
	  ex=1+(int) X2;
	  sy=(int) Y1;
	  ey=1+(int) Y2;
	  if (sx<n_hp && sy<n_vp && ex>=0 && ey>=0)
	    {
	      isx=(sx<0) ? 0 : sx;
	      isy=(sy<0) ? 0 : sy;
	      iex=(ex>n_hp) ? n_hp : ex;
	      iey=(ey>n_vp) ? n_vp : ey;
	      for(int ii=isy;ii<iey;ii++)
		{
		  if (ii==sy && ii==ey-1)
		    frac=stepy;
		  if (ii==sy)
		    frac=1.0-(Y1-sy);
		  else if (ii== ey-1)
		    frac=1.0+Y2-ey;
		  else
		    frac=1.0;
		  
		  for(int jj=isx;jj<iex;jj++)
		    {
		      if (jj==sx && jj==ex-1)
			frac2=stepx;
		      else if (jj==sx)
			frac2=1.0-(X1-sx);
		      else if (jj == ex-1)
			frac2=1.0+X2-ex;
		      else
			frac2=1.0;
		      vtemp[i][j]+=frac * frac2 * values[ii][jj];
		    }
		}
	    }
	}
    }

  freematrix(values);
  values=vtemp;
  n_hp=nxp;
  n_vp=nyp;
  n_pts=n_hp*n_vp;
  return;
}


void
XrayMaps::setminimum()
{
  if (!n_hp || !n_vp) return;
  float *A;
  float min=values[0][0];
  for(int i=0;i<n_vp;i++)
    {
      A=values[i];
      for(int j=0;j<n_hp;j++)
	{
	  if (min > *A)
	    min = *A;
	  A++;
	}
    }
  if (!min) return;
  for(int i=0;i<n_vp;i++)
    {
      A=values[i];
      for(int j=0;j<n_hp;j++)
	*A -= min;
    }
  return;
}
    

void
XrayMaps::enhance(const int numb)
{
  if (!n_hp || !n_vp || numb<1) return;
 
  float *sptA = new float[n_hp]; //array from 0 .. n_hp-1
  float *sptB = new float[n_hp];
  float *chg = sptA;

  int spj,epj;
  int spi,epi;
  for(int ii=0;ii<numb;ii++)
    {
      chg = sptA;
      for (int y=0; y<n_vp; y++)
	{
	  spi = (y ? -1 : 0);
	  epi = ((y == n_vp-1) ? 1 : 2);
	  for (int x=0;x<n_hp;x++)
	    {      
	      spj = (x ? -1 : 0);
	      epj = ((x == n_hp-1) ? 1 : 2);
	      
	      for(int i= spi;i<epi;i++)
		for(int j = spj;j<epj;j++)
		  chg[x]+= 0.125 * values[y+i][x+j]; //sums to 0.5
	      chg[x]-=0.125*values[y][x];
	      if (chg[x]>0.0)
		{
		  chg[x]=0.5*values[y][x]+
		    (values[y][x]-0.5*chg[x]);
		}
	      else
		chg[x]=values[y][x];
	    }
	  if (chg == sptA)
	    {
	      if (y)
		{
		  chg=values[y-1];
		  for(int i=0;i<n_hp;i++)
		    chg[i]=sptB[i];
		}
	      chg=sptB;
	    }
	  else 
	    {
	      chg=values[y-1];
	      for(int i=0;i<n_hp;i++)
		chg[i]=sptA[i];
	      chg=sptA; 
	    }
	}
      if (chg == sptA)
	{
	  chg=values[n_vp-1];
	  for(int i=0;i<n_hp;i++)
	    chg[i]=sptB[i];
	}
      else
	{
	  chg=values[n_vp-1];
	  for(int i=0;i<n_hp;i++)
	    chg[i]=sptA[i];
	}
    }

  delete [] sptA;
  delete [] sptB;
  
  return;	  
}


void 
XrayMaps::smooth(const int numb)
{
  if (!n_hp || !n_vp || numb<1) return;
 
  float *sptA = new float[n_hp]; //array from 0 .. n_hp-1
  float *sptB = new float[n_hp];
  float *chg = sptA;

  int spj,epj;
  int spi,epi;
  for(int ii=0;ii<numb;ii++)
    {
      chg = sptA;
      for (int y=0; y<n_vp; y++)
	{
	  spi = (y ? -1 : 0);
	  epi = ((y == n_vp-1) ? 1 : 2);
	  for (int x=0;x<n_hp;x++)
	    {      
	      spj = (x ? -1 : 0);
	      epj = ((x == n_hp-1) ? 1 : 2);
	      
	      chg[x]=0.4375 * values[y][x];
	      for(int i= spi;i<epi;i++)
		for(int j = spj;j<epj;j++)
		  chg[x]+= 0.0625 * values[y+i][x+j];
	    }
	  if (chg == sptA)
	    {
	      if (y)
		{
		  chg=values[y-1];
		  for(int i=0;i<n_hp;i++)
		    chg[i]=sptB[i];
		}
	      chg=sptB;
	    }
	  else 
	    {
	      chg=values[y-1];
	      for(int i=0;i<n_hp;i++)
		chg[i]=sptA[i];
	      chg=sptA; 
	    }
	}
      if (chg == sptA)
	{
	  chg=values[n_vp-1];
	  for(int i=0;i<n_hp;i++)
	    chg[i]=sptB[i];
	}
      else
	{
	  chg=values[n_vp-1];
	  for(int i=0;i<n_hp;i++)
	    chg[i]=sptA[i];
	}
    }

  delete [] sptA;
  delete [] sptB;
  
  return;	  
}
  

void
XrayMaps::power(const double mult)
{
  if (mult!=0.0)
    for(int i=0;i<n_vp;i++)
      for(int j=0;j<n_hp;j++)
	values[i][j]=pow(values[i][j],mult);
  else
    for(int i=0;i<n_vp;i++)
      for(int j=0;j<n_hp;j++)
	values[i][j]=1.0;
  return;
}

void
XrayMaps::logofV()
{
  for(int i=0;i<n_vp;i++)
    for(int j=0;j<n_hp;j++)
      {
	if (values[i][j]>0.0)
	  values[i][j]=log(values[i][j]);
	else
	  values[i][j]= -10.0;
      }
  return;
}

void
XrayMaps::expofV()
{
  for(int i=0;i<n_vp;i++)
    for(int j=0;j<n_hp;j++)
      {
	if (values[i][j]> -200 && values[i][j]< 200)
	  values[i][j]=exp(values[i][j]);
      }
  return;
}

void
XrayMaps::self(const double add,const double mult)
{
  if (mult==0.0) return;    
  for(int i=0;i<n_vp;i++)
    for(int j=0;j<n_hp;j++)
      {
	values[i][j]*=mult;
	values[i][j]+=add;
      }
  return;
}

void
XrayMaps::rotate()
  /* Swaps X and Y coordinates */
{
  if (!n_vp || !n_hp) return;
  float **v2=fmatrix(n_hp,n_vp);   //note change because of rotation!
  for(int i=0;i<n_vp;i++)
    for(int j=0;j<n_hp;j++)
      v2[n_hp-(j+1)][i]=values[n_vp-(i+1)][j];
  freematrix(values);
  values=v2;       //set values and n_vp etc...
  int temp=n_vp;    
  n_vp=n_hp;
  n_hp=temp;
  return;
}

void
XrayMaps::diag()
  /* Swaps X and Y coordinates */
{
  if (!n_vp || !n_hp) return;
  float **v2=fmatrix(n_hp,n_vp);   //note change because of rotation!
  for(int i=0;i<n_vp;i++)
    for(int j=0;j<n_hp;j++)
      v2[j][i]=values[i][j];
  freematrix(values);
  values=v2;       //set values and n_vp etc...
  int temp=n_vp;    
  n_vp=n_hp;
  n_hp=temp;
  return;
}

int
XrayMaps::write_esrf(ofstream& temp)
{
  std::string ohead;
  char ss[256];
  ohead+="{\n";
  ohead+="HeaderID       = EH:000001:000000:000000 ;\n";
  ohead+="Image          = 1 ;\n";
  ohead+="ByteOrder      = HighByteFirst ;\n";
  ohead+="DataType       = UnsignedShort ;\n";
  sprintf(ss,"Dim_1          = %d ;\n",n_hp);
  ohead+=ss;
  sprintf(ss,"Dim_2          = %d ;\n",n_vp);
  ohead+=ss;
  sprintf(ss,"Size           = %d ;\n",n_vp*n_hp*2);
  ohead+=ss;
  ohead+="count_time     = 30 ;\n";
  ohead+="point_no       = 1 ;\n";
  ohead+="scan_no        = 1 ;\n";
  ohead+="preset         = 1 ;\n";
  sprintf(ss,"col_end        = %d ;\n",n_vp-1);
  ohead+=ss;
  ohead+="col_beg        = 0 ;\n";
  sprintf(ss,"row_end        = %d ;\n",n_hp-1);
  ohead+=ss;
  ohead+="row_beg        = 0 ;\n";
  ohead+="counter_pos    =  30 2.45683e+06 8.30308e+06 ;\n";
  ohead+="counter_mne    =  sec mon pico1 ;\n";
  ohead+=" motor_pos      =  41.8952 -5 10.69 150 7 1.50625 0 -0.04 ";
  ohead+="-0.200018 -0.2 0 0 0 0.5 0 149 -0.5005 -5e-05 ;\n";
  ohead+="motor_mne      =  tabz say saz sarot sax smallz say2 slitz";
  ohead+=" slitchi slitom slitx slity tth th chi phi st0 st1 ;\n";
  ohead+="suffix         = .edf ;\n";
  ohead+="run            = 1 ;\n";
  ohead+="title          = CCD Image ;\n";
  temp<<ohead;
  for(int i=ohead.size();i<1022;i++)
    temp<<" ";                         //correct number of pad charateers
  temp<<"}\n";
  int Vv;
  int aVv,bVv;
  for(int j=0;j<n_vp;j++)
    for(int i=0;i<n_hp;i++)
      {
	Vv=static_cast<int>(values[j][i]);
	aVv=Vv % 256;
	bVv=Vv/256;
	temp.put(static_cast<char>(bVv));
	temp.put(static_cast<char>(aVv));
      }
  return n_hp*n_vp;
}


int
XrayMaps::write_data(ofstream &temp)
{
  Uns A;
  for(int i=0;i<n_vp;i++)  
    for(int j=0;j<n_hp;j++)
      {
        A.f = values[i][j];
	temp.put(A.c[3]);
	temp.put(A.c[2]);
	temp.put(A.c[1]);
	temp.put(A.c[0]);
      }
  return n_pts;
}


void
XrayMaps::makespace(const int M,const int N)
{
  if (M==n_hp && N==n_vp) return;
  if (M<1 || N<1) return;
  if(n_pts && values)
    freematrix(values);
  values = fmatrix(N,M);
  n_pts=N * M;
  n_hp=M;
  n_vp=N;
  return;
}

int 
XrayMaps::pts() const
{
  return n_pts;
}
int 
XrayMaps::ypt() const
{
  return n_vp;
}

int 
XrayMaps::xpt() const
{
  return n_hp;
}

int 
XrayMaps::verify_cord(char *in,int *xp) const
  /* 
     Get coordinates from string in ss and verify that they
     are in the domain of the Map.
  */

{
  if (n_pts<1) return 0;
  int i;
  int ofa,ofb;
  for(i=0;i<4;i++)
    xp[i]=0;
  for(i=0;i<4 && cmdnumber(in,xp[i]);i++);  
  if (i==4 && cmdnumber(in,ofa))
    {
      if(cmdnumber(in,ofb))
	{
	  xp[0]=xp[2];
          xp[1]=xp[3];
          xp[2]=ofa;
	  xp[3]=ofb;
	}
      else
	{
	  xp[0]=xp[1];
          xp[1]=xp[2];
          xp[2]=xp[3];
          xp[3]=ofa;
	} 
    }                                               

  int flag;
  char ss[256]={"Point value of x_start,x_end,y_start and y_end =>"};
  do 
    {
      flag=0;
      for(int j=0;j<2 && !flag;j++)
	{
	  if (xp[j]>n_hp || xp[j]<1)
	    flag=1;
	  else
	    xp[j]--;
	}
      for(int j=3;j<4 && !flag;j++)
	{
	  if (xp[j]>n_vp || xp[j]<1)
	    flag=1;
	  else
	    xp[j]--;
	}
      if (flag)
	{
	  l_c(ss,0);
	  for(i=0;i<4 && cmdnumber(ss,xp[i]);i++);
	  if (i!=4) return 0;
	}
    } while (flag);

  if (xp[0]>xp[1])
    {
      i=xp[0];
      xp[0]=xp[1];
      xp[0]=i;
    }
  if (xp[2]>xp[3])
    {
      i=xp[2];
      xp[2]=xp[3];
      xp[3]=i;
    } 
  return 1;
}

void
XrayMaps::rollx(const int rpt)
{
  if (rpt>=n_hp || rpt<1 || n_vp<1)  return;  //Nothing to do.
  float *Tpts=new float[n_vp];
  
  int savept=n_vp-rpt;
  for(int j=0;j<n_vp;j++)
    Tpts[j]=values[j][savept];

  int ptm=savept;
  int ptn=0;
  int ndone=0;
  while(ndone < n_hp)
    {
      do
	{
	  for(int j=0;j<n_vp;j++)
	    values[j][ptm]=values[j][ptn]; 
	  ndone++;
	  ptm=ptn;
	  ptn = (ptm+rpt) % n_hp;
	} while(ptn!= savept);
      for(int j=0;j<n_vp;j++)
	values[j][ptm]=Tpts[j];
      ndone++;
      if (ndone<n_hp)
	{
	  savept++;
	  for(int j=0;j<n_vp;j++)
	    Tpts[j]=values[j][savept];
	  ptm=savept;
	  ptn=(savept+1+rpt) % n_hp;
	}
    }
  return;
}

void
XrayMaps::rolly(const int rpt)
{
  if (rpt>=n_vp || rpt<1 || n_hp<1)  return;  //Nothing to do.
  float *Tpts=new float[n_hp];
  
  int savept=n_hp-rpt;
  for(int j=0;j<n_hp;j++)
    Tpts[j]=values[savept][j];

  int ptm=savept;
  int ptn=0;
  int ndone=0;
  while(ndone < n_vp)
    {
      do
	{
	  for(int j=0;j<n_hp;j++)
	    values[ptm][j]=values[ptn][j]; 
	  ndone++;
	  ptm=ptn;
	  ptn = (ptm+rpt) % n_hp;
	} while(ptn!= savept);
      for(int j=0;j<n_hp;j++)
	values[ptm][j]=Tpts[j];
      ndone++;
      if (ndone<n_vp)
	{
	  savept++;
	  for(int j=0;j<n_hp;j++)
	    Tpts[j]=values[savept][j];
	  ptm=savept;
	  ptn=(savept+1+rpt) % n_vp;
	}
    }
  return;
}

double
XrayMaps::average(const int zero) const
  /* 
     To get a average value from a set of points defined by hold_.is
     string 
  */

{
  extern wback hold_;
   
  int xp[4];
  if (!verify_cord(hold_.is,xp)) return 0.0;
  double ave=0.0;
  double dva=0.0;
  if (!zero)
    {
      dva=(1.0+xp[1]-xp[0]) * (1.0+xp[3]-xp[2]);
      for(int ky=xp[2];ky<=xp[3];ky++)
	for (int kx=xp[0];kx<=xp[1];kx++)
	  ave+=values[ky][kx];
    }
  else
    for(int ky=xp[2];ky<xp[3];ky++)
      for (int kx=xp[0];kx<xp[1];kx++)
	{
          if (values[ky][kx]>0.0)
	    {
	      ave+=values[ky][kx]; 
	      dva+=1.0;
	    }
	}
  if (dva>0.0)
    ave/=dva;
  else
    ave=0.0;
  return ave;
}


double
XrayMaps::average(const int Zero,const int a,const int b,
		  const int c,const int d) const

  /*
     Find the average of a number of points defined by the rectangle
     of a,b and c,d 
  */
{
  double ave=0.0;
  double dva=0.0;
  if (!Zero)
    {
      dva=(1.0+b-a) * (1.0+d-c);
      for(int ky=c;ky<d;ky++)
	for (int kx=a;kx<b;kx++)
	  ave+=values[ky][kx];
    }
  else	
    for(int ky=c;ky<d;ky++)
      for (int kx=a;kx<b;kx++)
	{
          if (values[ky][kx]>0.0)
	    {
	      ave+=values[ky][kx]; 
	      dva+=1.0;
	    }
	}
  if (dva>0.0)
    ave/=dva;
  else
    ave=0.0;
  return ave;
}



void
XrayMaps::handle(int &hp,int &vp,float **&V) const
  /* Return the values of the Xmaps */
{
  hp=n_hp;
  vp=n_vp;
  V=values;
  return;
}

XrayMaps& 
XrayMaps::operator=(const XrayMaps& A)
{
  if (this != &A)
    {
      if (n_hp!=A.n_hp || n_vp!=A.n_vp)
	{
	  if (values)
	    freematrix(values);
          n_pts=A.n_pts;
          n_hp=A.n_hp;
          n_vp=A.n_vp;
	  if (n_pts)
	    values=fmatrix(n_vp,n_hp);
	  else
	    values=0;
	}
      for(int i=0;i<n_vp;i++)
	for(int j=0;j<n_hp;j++)
	  values[i][j]=A.values[i][j];
    }
  return *this;
}	  


XrayMaps::XrayMaps() : n_pts(0),n_hp(0),n_vp(0),values(0)
{ }

XrayMaps::~XrayMaps()
{
  if (values)
    freematrix(values);
}

void
XrayMaps::polar(const double len,const double width,
                const float cx,const float cy)

  /* DOES NOT WORK */
  // Point Modify 
{
  /* assumes that centre is defined, and perfect polorization yuk!! */
  char ss[80];
  sprintf(ss,"Centre : %f , %f ",cx,cy);
  l_c(ss,1);
  double xx,c_eps; 
  double len2= len * len;
  for(int i=0;i<n_hp;i++)
    {
      xx = width * (( (i+1.0)/n_pts)-cx);
      c_eps= len / sqrt(len2 + xx*xx);    //cheap cos(eps) 
      for(int j=0;j<n_vp;j++)
        {
          values[j][i]/=c_eps; 
        }
    }
 return;
}

XrayMaps::XrayMaps(const XrayMaps& A) 
{
  if (this!= &A)
    {
      n_pts=A.n_pts;
      n_hp=A.n_hp;
      n_vp=A.n_vp;
      if (A.n_pts)
	{
	  values = fmatrix(n_vp,n_hp);
          for(int i=0;i<n_vp;i++)
	    for(int j=0;j<n_hp;j++)
	      values[i][j]=A.values[i][j];
	}
    }
  return;
}

int 
XrayMaps::read_ascii(const char *fname,const int xp,const int yp)
{
  char ss[255];
  if (xp<1 || yp<1) return 1;
  ifstream imfle;
  if (fname)
    imfle.open(fname,ios::in);
  if (!imfle)
    {   
      strcpy(ss,"What is the 2D ascii file =>");
      l_c(ss,0);
      s_line();
      int j;
      for(j=0;ss[j]>' ' && j<80;j++);
      ss[j]=0; 
      imfle.open(ss,ios::in); 
    }
  if (imfle)
    {
    if (yp!=n_vp || xp!=n_hp)
      {
	if (values)
	  freematrix(values);
	
	values = fmatrix(yp,xp);
	n_pts=yp * xp;
	n_hp=xp;
	n_vp=yp;
      } 
    int count; //length of string read in
    int cnt=0;     //number of x,y numbers 
    int xval=0;
    int yval=0;
    float n_read;
    int string_ct;
    int chk;
    while(imfle && cnt<n_pts)
      {
	imfle.getline(ss,255,'\n');
	chk=imfle.gcount();
	for(count=0;count<chk && ss[count] &&
	      ss[count]!='#' && ss[count]!='!'
	      ;count++);
	  ss[count]=0; //quick way to remove comments!
	string_ct=0;
	if (count>1) //we have some numbers to use (hopefully)
	  {
	    while(cnt<n_pts &&
		  (chk=findnumber(ss+string_ct,n_read,count-string_ct)))
	      {
		string_ct+=chk;
		values[yval][xval]=n_read;
		xval++;
		cnt++;
		if (xval==xp)
		  {
		    xval=0;
		    yval++;
		  }
	      }
	  }
      }
    imfle.close();
    if (cnt<n_pts)
      {
	for(int i=yval;i<yp;i++)
	  for(int j=xval;j<xp;j++)
	    values[i][j]=0.0;
      }
    return 0;
    }
  imfle.close();
  return 1;
}


int
XrayMaps::read_data(const int M,const int N,char* s,const int offset,
		    const int type)
{
 
  if (M<1 || N<1) return 1;
  Uns A;
  ifstream imfle;
  if (s)
    imfle.open(s,ios::in);
  if (!imfle)
    {
      strcpy(s,"What is the 2D data sets file =>");
      l_c(s,0);
      s_line();
      int j;
      for(j=0;s[j]>' ' && j<80;j++);
      s[j]=0; 
      imfle.open(s,ios::in); 
    }
   
 
  if (imfle)
    {
    if (N!=n_vp || M!=n_hp)
      {
	if (values)
	  freematrix(values);
	
	values = fmatrix(N,M);
	n_pts=N * M;
	n_hp=M;
	n_vp=N;
      } 
    imfle.seekg(N*M*4*(offset-1),ios::beg);
    
     if (type)           //Gadds type
       {                
	 imfle.seekg(15 * 512,ios::beg);
	 for(int i=0;i<N;i++)
	   for(int j=0;j<M;j++)
	     {
	       imfle.get(A.c[0]);
	       values[i][j]=(float) A.c[0];
	     }
       }
     else
       {
	 for(int i=0;i<M;i++)
	   for(int j=0;j<N;j++)
	     {
	       imfle.get(A.c[3]);
	       imfle.get(A.c[2]);         
	       imfle.get(A.c[1]);
	       imfle.get(A.c[0]);
	       values[j][i]=A.f;
	     }
       }
    imfle.close();
/*
    for(int j=0;j<5;j++)
      for(int i=0;i<300;i++)
	values[i][400+j]=256.0;
*/
    return 0;
    }
  return 1;     //could not load file
}

int
XrayMaps::read_esrf(char* s,const int normI,const int m,const int n,
                    const int offset,const int Big_Ind)
  /* To read Indian type data */
{
 
  int M=m;
  int N=n;
  int sea_type=0;
  if (M<1 || N<1)
    sea_type=1;
  Uns A;
  ifstream imfle;
  char infoFile[255];
  if (s)
    {
      strncpy(infoFile,s,254);
      imfle.open(infoFile,ios::in);
    }
  if (!imfle)
    {
      strcpy(infoFile,"What is the 2D data sets file =>");
      l_c(infoFile,0);
      s_line();
      int j;
      for(j=0;infoFile[j]>' ' && j<80;j++);
      infoFile[j]=0; 
      imfle.open(infoFile,ios::in); 
    }
   
  double norm=1.0; 
  if (imfle)
    {
      if (sea_type)
	{
	  std::string check;
	  std::string::size_type pos;
	  char ss[1024];
	  while(imfle && sea_type)
	    {
	      imfle.getline(ss,1024,'\n');
	      check=ss;
	      if (check.find("Dim_1")!=std::string::npos)
		{
		  if((pos=check.find("=")) != std::string::npos)
		    {
		      check.copy(ss,std::string::npos,pos+1); 
		      if (!cmdnumber(ss,M))
			M= -1;
		    }
		}
	      else if (check.find("Dim_2")!=std::string::npos)
		{
		  if((pos=check.find("=")) != std::string::npos)
		    {
		      check.copy(ss,std::string::npos,pos+1); 
		      if (!cmdnumber(ss,N))
			N= -1;
		    }
		}
	      else if (normI && check.find("counter_pos")!=std::string::npos)
	        {
	          if ((pos=check.find("=")) != std::string::npos)
		    {
		      check.copy(ss,std::string::npos,pos+1); 
		      if (!cmdnumber(ss,norm) || !cmdnumber(ss,norm))
                        norm=1.0;
	              if (norm<=0.0) 
                        {
                          l_c("Very low normalisation value",1);
	                  norm=1.0;
                        }
	             }
                 } 
	      if (check.find("}")!=std::string::npos)
		sea_type=0;
	    }
	  if (M<1 || N<1) return 1;
	}
      if (N!=n_vp || M!=n_hp)
	{
	  if (values)
	    freematrix(values);
	  
	  values = fmatrix(N,M);
	  n_pts=N * M;
	  n_hp=N;
	  n_vp=M;
	} 
      if (!imfle) 
	l_c("File has incomplete header information",1);
      imfle.clear();
      imfle.seekg(1024+((N*M*2)*(offset-1)),ios::beg);

      A.c[2]=0;
      A.c[3]=0;
      int cfirst,csnd;
      if (Big_Ind)
	{
	  cfirst=1; csnd=0;
	}
      else
	{	  
	  cfirst=0; csnd=1;
	}
	
      for(int i=0;i<M;i++)
	for(int j=0;j<N;j++)   //horr. pts.
	  {
	    imfle.get(A.c[cfirst]);
	    imfle.get(A.c[csnd]);         
	    if (imfle.good())
	      {
		values[i][j]=((unsigned char) A.c[1])*256 + 
                  ((unsigned char) A.c[0]);
		values[i][j]/=norm;
	      }
	    else
	      values[i][j]=0.0;
	  }
      imfle.close();
      return 0;
    }
  imfle.close();
  return 1;     //could not load file
}

int
XrayMaps::read_fit2d(const char* s,const int m,const int n)
  /* 
     To read Fit 2d data with an info file ::
      WARNING only takes some of the info 
  */
{
 
  int Mhorr=m;
  int Nvert=n;
  int infoI = (Mhorr<1 || Nvert<1) ? 1 : 0;
    //if No M / N then get the spacing from the file.
  Uns A;
  ifstream imfle;
  char infoFile[255];  
  if (s)
    {
      strncpy(infoFile,s,255);
      imfle.open(infoFile,ios::in);
    }
  if (!imfle)
    {
      strcpy(infoFile,"What is the Fit2D info file =>");
      l_c(infoFile,0);
      s_line();
      int j;
      for(j=0;infoFile[j]>' ' && j<80;j++);
      infoFile[j]=0; 
      imfle.open(infoFile,ios::in); 
    }
  int Nbyte;  //must be 4,2,1
  char bfname[255];
  int LittleEndF= -1;  //equivelent to undef.
  
  if (imfle)
    {
      std::string check;
      std::string::size_type pos;
      char ss[1024];
      while(imfle)
	{
	  imfle.getline(ss,1024,'\n');
	  check=ss;
	  if (check.find("image")!=std::string::npos)
	    {
	      if((pos=check.find("=")) != std::string::npos)
		{
		  check.erase(0,pos);
		  int start,len;
		  extractname(bfname,check.c_str(),
			      check.size(),start,len,255);
		}
	    }
	  else if (check.find("horizontal")!=std::string::npos &&
		   check.find("vertical")!=std::string::npos)
	    {
	      if(infoI && 
		 (pos=check.find("=")) != std::string::npos)
		{
		  check.copy(ss,std::string::npos,pos+1); 
		  if (!cmdnumber(ss,Mhorr,255) ||
		      !cmdnumber(ss,Nvert,255))
		    Nvert = -1;
		}
	    }
	  else if (check.find("bytes")!=std::string::npos &&
		   check.find("pixel")!=std::string::npos)
	    {
	      if((pos=check.find("=")) != std::string::npos)
		{
		  check.copy(ss,std::string::npos,pos+1);
		  if (!cmdnumber(ss,Nbyte,255))
		    {
		      l_c("Unable to read number of bytes per pixel",1);
		      Nbyte=0;
		    }
		}
	    }
	  else if (check.find("endian")!=std::string::npos)
	    {
	      if((pos=check.find("little")) != std::string::npos)
		LittleEndF=1;
	      else
  		LittleEndF=0;
	    }
	}

      if (LittleEndF == -1)
	{
	  l_c("Error with endien type",1);
	  return 1;
	}
      if (Mhorr<1 || Nvert<1) 
	{
	  l_c("Error with grid number",1);
	  return 1;
	}
      if (Nbyte<=0 || Nbyte>4)
	{
	  l_c("Error with Byte number",1);
	  return 1;
	}
      imfle.close();
      check=infoFile;
      pos=check.find_last_of("/");
      if (pos!=std::string::npos)
	check.erase(pos+1);
      else
	check.erase(0,std::string::npos);
      check+=bfname;
      strncpy(bfname,check.c_str(),255);
      imfle.open(bfname,ios::in);
      if (imfle)   //Have opened the 
	{
	  changeXYsize(Mhorr,Nvert);
	  //	  imfle.seekg(((N*M*2)*(offset-1)),ios::beg);	  
	  A.g=0;  //zero the input point
      
	  int aidx[4];
	  if (LittleEndF)
	    for(int i=0;i<Nbyte;i++)
		aidx[i]=i;
	  else
	    for(int i=0;i<Nbyte;i++)
		aidx[i]=(Nbyte-1)-i;

	  for(int i=0;i<Mhorr;i++)
	    for(int j=0;j<Nvert;j++)   //horr. pts.
	      {
		for(int kk=0;kk<Nbyte;kk++)
		  imfle.get(A.c[aidx[kk]]);
		if (imfle.good())
		  {
		    values[i][j]=0.0;
		    int mult=1;
		    for(int kk=0;kk<Nbyte;kk++)
		      {
			values[i][j]=((unsigned char) A.c[kk])*mult;
			mult*=256;
		      }
		  }
		else
		  values[i][j]=0.0;
	      }
	  imfle.close();
	  return 0;
	}
      l_c("No binary file",1);
      return 2;  //couldn't get the second file
    }
  l_c("No info file ",1);
  return 1;  //couldn't get info file 
}

int
XrayMaps::intXvalues(const double spt,const double ept,double* Y)
{
  if (!Y) return 0;
  int Sp=(int) spt;
  double frcA=1.0-(spt-Sp);
  int Ep=(int) ept;
  double frcB=ept-Ep;
  if (Ep>=n_hp) 
    {
      Ep=n_hp-1;
      frcB=1.0;
    }
  double* yv=Y;
  for(int i=0;i<n_vp-1;i++)
    {
      *yv = frcA * values[i][Sp];
      for(int j=Sp+1;j<Ep-1;j++)
	{
	  *yv += values[i][j];
	}
      *yv++ += frcB* values[i][Ep];
    }
  return n_vp;
}

int
XrayMaps::intYvalues(const double spt,const double ept,double* Y)
{
  int Sp=(int) spt;
  double frcA=1.0-(spt-Sp);
  int Ep=(int) ept;
  double frcB=ept-Ep;
  if (Ep>=n_hp) 
    {
      Ep=n_vp-1;
      frcB=1.0;
    }
  double* yv=Y;
  for(int i=0;i<n_hp-1;i++)
    *yv++ = frcA * values[Sp][i];
  for(int j=Sp+1;j<Ep-1;j++)
    {
      yv=Y;     //reset yv
      for(int i=0;i<n_hp-1;i++)
	{
	  *yv += values[j][i];
	  yv++;
	}
    }
  yv=Y;
  for(int i=0;i<n_hp-1;i++)
    *yv+=frcB * values[Ep][i];

  return n_hp;
}

int
XrayMaps::intArcValues(const double spt,const double ept,double* Y)
{
  AreaInt AA;
  AA.Initialise(values,n_hp,n_vp);
  AA.Set_centre(n_hp/2.0,n_vp/2.0);
  double R=1.2;
  double Rstep=1.0;
  for(int i=0;i<maxpts && !AA.EvalArcUnit(*Y,R,R+Rstep,spt,ept);i++)
    {
      R+=Rstep;
      Y++;
    }
  
  return 0;
}
 
int
XrayMaps::read_norm(char* s,const int offset)
{
  Uns A;
  ifstream imfle;
  if (s)
    imfle.open(s,ios::in);
  if (!imfle) return 1;
  imfle.seekg(4*(offset-1),ios::beg);
  imfle.get(A.c[3]);
  imfle.get(A.c[2]);         
  imfle.get(A.c[1]);
  imfle.get(A.c[0]);  
  imfle.close();
  if (A.f<=0.0) 
    {
      l_c("Strange norm value",1);
      return 1;
    }

  for(int i=0;i<n_vp;i++)
    for(int j=0;j<n_hp;j++)
      values[i][j]/=A.f;

  return 0;
}

void
XrayMaps::flip_y(const double dty)
  /* flip all the points around the line y=dty */
{
  if (!n_pts) return;
  if (dty>=n_vp-2 || dty<1.0) return;  
  int atx=(int) dty;
  double f1=dty-atx;
  double f2=1.0-f1;

  int ilow,ihigh;
  double alpha,beta,zeta,yeti;
  int cont;        //do we need to continue
  for(int i=0;i<n_hp;i++)
    {
      zeta=beta= f1* values[atx][i] + f2 * values[atx+1][i];
      alpha=beta=f2 * values[atx][i] + f1 * values[atx-1][i];
      for(int j=0;j<n_vp;j++)
	{
	  ilow=atx-j;
	  ihigh=atx+j;
	  cont=0;
	  if (ihigh<n_hp)
	    {
	      cont++;
	      values[ihigh][i]= f1*zeta+f2*yeti;
	    }
	  if (ilow>=0)
	    {
	      cont++;
	      values[ilow][i]= f2*beta+f1*alpha;
	    }
	  if (!cont) break;

	  alpha=beta;
	  zeta=yeti;
	  yeti=(ilow>=0) ? f1* values[ilow][i] : 0.0;
	  yeti+= (ilow>0) ? f2 * values[ilow-1][i] : 0.0;

	  beta=(ihigh<n_hp) ? f1 * values[ihigh][i] : 0.0;
	  beta+= (ihigh<n_hp-1) ? f2 * values[ihigh+1][i] : 0.0;
	  
	}
    }
  
  return;
}


void
XrayMaps::flip_x(const double dtx)
  /* Flip the data around axis give at y=dtx 
     zero's all points outside range */ 
{
  if (!n_pts) return;
  if (dtx>=n_hp-2 || dtx<1.0) return;  
  int atx=(int) dtx;
  double f1=dtx-atx;
  double f2=1.0-f1;

  int ilow,ihigh;
  double alpha,beta,zeta,yeti;
  float *pts;
  for(int i=0;i<n_vp;i++)
    {
      pts=values[i];
      zeta=beta= f1* pts[atx] + f2 * pts[atx+1];
      alpha=beta=f2 * pts[atx] + f1 * pts[atx-1];
      int cont;
      for(int j=0;j<n_hp;j++)
	{
	  cont=0;
	  ilow=atx-j;
	  ihigh=atx+j;
	  if (ihigh<n_hp)
	    {
	      cont++;
	      pts[ihigh]= f1*zeta+f2*yeti;
	    }
	  if (ilow>=0)
	    {
	      cont++;
	      pts[ilow]= f2*beta+f1*alpha;
	    }
	  if (!cont) break;
	  alpha=beta;
	  zeta=yeti;
	  yeti=(ilow>=0) ? f1* pts[ilow] : 0.0;
	  yeti+= (ilow>0) ? f2 * pts[ilow-1] : 0.0;

	  beta=(ihigh<n_hp) ? f1 * pts[ihigh] : 0.0;
	  beta+= (ihigh<n_hp-1) ? f2 * pts[ihigh+1] : 0.0;
	  
	}
    }
  
  return;
}

void
XrayMaps::move_x(const double dtx)
  /* move the data across by dtx points, 
     y=dtx 
     zero's all points outside range */ 
{
  if (!n_pts) return;
   
  double cnx= n_hp/2.0;
  if (dtx==(double) cnx) return;
 
  double diff= fabs(dtx-cnx);
  int idiff=(int) (dtx-cnx);
  int stp,ept,add;
  if (dtx>cnx) 
    {
      stp=0;
      ept=n_hp;
      add=1;
    }
  else
    {
      stp=n_hp-1;
      ept= -1;
      add= -1;
    }
  double f2=diff-fabs(idiff);
  double f1=1.0-f2;

  double sum;
  int pt1,pt2;
  float *vpts;
  for(int j=0;j<n_vp;j++)
    {
      vpts=values[j];
      for(int i=stp;i!=ept;i+=add)
	{
	  pt1=idiff+i;
	  pt2=pt1+add;
	  sum=0.0;
	  if (pt1<n_hp && pt1>=0) 
	    sum+=f1 * vpts[pt1];
	  if (pt2<n_hp && pt2>=0) 
	    sum+=f2 * vpts[pt2];
	  vpts[i]=sum;
	}
    }
  return;
}

void
XrayMaps::move_y(const double dty)
  /* move the data across by dtx points, 
     y=dtx 
     zero's all points outside range 
  This can be made a lot more efficient by indexing an array of zeros
  and a pointer substitution
*/ 
{
  if (!n_pts) return;
   
  double cny= n_vp/2.0;
  if (dty==(double) cny) return;
 
  double diff= fabs(dty-cny);
  int idiff=(int) (dty-cny);
  int stp,ept,add;
  if (dty>cny) 
    {
      stp=0;
      ept=n_vp;
      add=1;
    }
  else
    {
      stp=n_vp-1;
      ept= -1;
      add= -1;
    }
  double f2=diff - fabs(idiff);
  double f1=1.0-f2;

  double sum;
  int pt1,pt2;
  for(int i=stp;i!=ept;i+=add)
    {
      pt1=idiff+i;
      pt2=pt1+add;
      for(int j=0;j<n_hp;j++)
	{
	  sum=0.0;
	  if (pt1<n_vp && pt1>=0) 
	    sum+=f1 * values[pt1][j];
	  if (pt2<n_vp && pt2>=0) 
	    sum+=f2 * values[pt2][j];
	  values[i][j]=sum;
	}
    }
  return;
}

void
XrayMaps::re_centre(const double dtx,const double dty)
  /*
    Move to position dtx and dty  as the new zero point
  */
{
  move_x(dtx);
  move_y(dty);
  return;
}



