#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <map>
#include <vector>
#include <algorithm>
#include <string>
#include "fglobal.h"
#include "cmd.h"
#include "wshare.h"
#include "baseGraph.h"
#include "axis.h"
#include "spacebox.h"
#include "space.h"
#include "new_read.h"


double
Peak::sep(const Peak &A) const
  /*
    returns the seperation between
     A and B
  */
{
  double a=A.x-x;
  double b=A.y-y;
  return sqrt(a*a+b*b);
}


space_correct::space_correct() : 
  SpcWind(0),
  grp(-1),V(0),xp(0),yp(0),Xfrac(0.10),
  min_dist(0.0),npeak(0)
{}

void
space_correct::complete_draw()
{
  bin_file* BF=dynamic_cast<bin_file*>(this);
  if (!BF || grp<0 ) return;
  BF->draw_picture(grp,0);
  BF->give_axis(ax);    //get axis
  if (Xfrac>0.0)
    {
      lsx=(ax[1]-ax[0])*Xfrac;
      lsy=(ax[3]-ax[2])*Xfrac;
    }
  else
    {
      lsx=(ax[1]-ax[0])/50;
      lsy=(ax[3]-ax[2])/50;
    }
  draw();
  return;
}


void
space_correct::make_menu()
{
  if (!SpcWind)
    SpcWind=new spacebox(this);
  SpcWind->make_menu();   
  return;
}


int
space_correct::init_grp(const int gp)
  /* seems to correct only for the x,y possible positions.. */
{
  bin_file* BF = dynamic_cast<bin_file *>(this);
  if (!BF) 
    {
      l_c("Error space correct needs a picture file",1);
      return 1;
    }
  if (gp > BF->max_Maps || gp<0) return 1;
  grp=gp;
  BF->give_axis(ax);    //get axis
  if (Xfrac>0.0)
    {
      lsx=(ax[1]-ax[0])*Xfrac;
      lsy=(ax[3]-ax[2])*Xfrac;
    }
  else
    {
      lsx=(ax[1]-ax[0])/50;
      lsy=(ax[3]-ax[2])/50;
    }
    
  BF->DD[gp].handle(xp,yp,V);
  if (xp<10 || yp<10) 
    return 1;
  return 0;
}

int
space_correct::overlap()
  /* 
     This is to remove points that are far too close 
     Cdist is the minimum seperation between points.
  */
{
  if (Pk.size()<2) return 0;  //sorted and none to move.
  double Cdist;
  if (min_dist<=0.0) 
    {
      double xsep=static_cast<double>(xp);
      double ysep=static_cast<double>(yp);
      Cdist=(xsep>ysep) ? xsep : ysep;
      Cdist/=5.0*sqrt(npeak+1.0);
    }
  else
    Cdist=min_dist;

  // A bit aggressive first
  sort( Pk.begin(),Pk.end(),xPeak_less() );
  int c=0;
  int loop_count=0;
  int total_removed=0;
  do  
    {
      total_removed+=c;
      loop_count++;
      for(int i=0;i<npeak-1;i++)
	for(int j=i+1;j<npeak && Pk[j].x-Pk[i].x<Cdist;j++)
	  {
	    if (Pk[i].sep(Pk[j])<Cdist)
	      {
		c++;
		converge(Pk[j],Pk[i]);
		Pk.erase(Pk.begin()+i,Pk.begin()+i);
		npeak--;
		i--;    //we have removed one from the array!
		break;
	      }
	  }
    }  // May need re-sorting ?
  while (c>0 && loop_count<5);
  return total_removed+c;
}


void
space_correct::corr_values()
  /* values for the spacial correction */
{
  extern wback hold_;
  char c;
  char ss[255];
  get_next_char(c,hold_.is,80);  //get next value
  double dl; 
  switch(c) 
    {
    case 'D':
      if (!cmdnumber(hold_.is,dl) || dl<0.0)
	{  
	  strcpy(ss,"Closest Distance =>");
          l_c(ss,0);
          if (!cmdnumber(ss,dl) || dl<0.0) return;
        }
      update('D',dl);
      sprintf(ss,"Closest Distance : %10g",dl);	  
      l_c(ss,1);
      return;
    case 'X':
      if (!cmdnumber(hold_.is,dl) || dl<0.0)
	{  
	  strcpy(ss,"Cross Fractional to draw =>");
          l_c(ss,0);
          if (!cmdnumber(ss,dl) || dl<0.0) return;
        }
      update('X',dl);
      sprintf(ss,"Cross Fractional : %10g",dl);	  
      l_c(ss,1);
      return;

    default:
      l_c("Spacial Correction Parameters ::",1);
      sprintf(ss,"Closest Distance : %10g",getvalues('D'));
      l_c(ss,1);
      sprintf(ss,"Cross Fractional : %10g",getvalues('X'));	  
      l_c(ss,1);
    }
  return;
}


void
space_correct::update(const char stw,double& V)
{
  switch(stw)
    {
    case 'D':  //min distance
      if (V<=0.0)
	{ 
	  min_dist=0.0;
	  V=0.0;
	}
      else
	min_dist=V;
      return;
    case 'R':
      RefPeak=1-RefPeak;
    case 'X':
      if (V<=0.0) V=0.0;
      Xfrac=V/100.0;   //want to enter a percentage.
      return;
    default:
      break;
    }
  return;
}


double
space_correct::getvalues(const char stw) const
{
  switch(stw)
    {
    case 'D':
      return min_dist;
    case 'X':
      return Xfrac*100.0;
    case 'R':
      return static_cast<double>(RefPeak);
    default:
      return 0.0;
    }
  return 0.0;
}

int
space_correct::cursor_addition(const int gp)
  /* For the addition of peaks to the overall addition table. */
{
  extern wback hold_;

  l_c("Use cursor to add or remove peak",1);
  l_c("Left button adds a peak, right removes and ",1);
  l_c("Middle allows a box to be drawn to remove a number of peaks",1);
  if (RefPeak) 
    l_c("Refining peaks before acceptance",1);
  else
    l_c("No refining of peaks before acceptance",1);

  bin_file* BF = dynamic_cast<bin_file*>(this);
  if (!BF)
    return 1;
  writeCommand("Peak selection (E to exit) =>",0);
  Peak Tp;
  char ans;
  float x1,y1;
  do 
    {
      BF->sband(0,0,0.0,0.0,x1,y1,ans); //normal cursor
      BF->rearm_curs();
      if (ans=='E' || ans=='e')
	{ 
	  hold_.graph_flag=2;
	  s_line();
	  return 1;
	}
      if (ans=='A')
	{
	  Tp.x=x1;
	  Tp.y=y1;
	  if (RefPeak)
	    {
	      if (!add_refined(Tp,1e7))
		l_c("Peak added",1);
	    }
	  else
	    {
	      Pk.push_back(Tp);
	      draw_cross(Tp.x,Tp.y);
	    }
	}
    }
  while(x1>ax[1] || x1<ax[0] ||
	y1>ax[3] || y1<ax[2]);
  s_line();
  return 0;
}

int
space_correct::init_search(const int gp)
  /* Release control from the bin_file layer */
{
  extern wback hold_; 

  if (!SpcWind)
    SpcWind=new spacebox(this);
  SpcWind->make_menu();   
  bin_file* BF=dynamic_cast<bin_file*>(this);
  if (!BF) return 1;
  BF->draw_picture(gp,0);  //draw picture 
  if (init_grp(gp)) 
    {
      l_c("Error with group",1);
      return 0;
    }
  //  screen = XtScreen()
  //  pixmap = XCreatePixmap(display,RootWindowOfScreen(screen),
  //			 width,height,
  //			 DefaultDepthOfScreen(screen));
  //  char ss[256];
  l_c("Use cursor to highlight three different peaks",1);
  l_c("These should be at the minimum seperation",1);
  Pk.clear();
  writeCommand("Click on first peak (right button to cancel");
  for(int cntr=0;cntr<3;cntr++)
    {
      if (cntr==1)
	layer_command("Click on second peak");
      else if (cntr==2)
	layer_command("Click on third peak");
      char ans='X';
      float x1,y1;
      do 
	{
	  BF->sband(0,0,0.0,0.0,x1,y1,ans); //normal cursor
	  BF->rearm_curs();
	  if (ans=='X')
	    {
	      hold_.graph_flag=2;
	      return 1;
	    }
	}
      while(x1>ax[1] || x1<ax[0] ||
	    y1>ax[3] || y1<ax[2]);
      
      cpgsci(3);
      cpgslw(1);
      draw_cross(x1,y1);
      Pk.push_back(Peak(x1,y1));
    }
  hold_.graph_flag=2;
  l_c("Commencing search (this means you get coffee now!)",1);

  //first refine the three peaks that we have!
  //initial guess for vectors
  directions();
  
  double averS=refine(Pk[0]);
  averS+=refine(Pk[1]);
  averS+=refine(Pk[2]);
  averS/=3.0;
  BF->draw_picture(gp,0);  //draw picture   draw_cross(Pk[0].x,Pk[0].y);
  cpgslw(1);
  cpgsci(3);
  draw_cross(Pk[0].x,Pk[0].y);
  draw_cross(Pk[1].x,Pk[1].y);
  draw_cross(Pk[2].x,Pk[2].y);
  int start=directions();   //reset directions
  
  //Ok now try to get the remaining positions.
 
  double dist=(ax[1]-ax[0])*(ax[1]-ax[0])+
    (ax[3]-ax[2])*(ax[3]-ax[2]);
  dist=sqrt(dist);
  double vectV=sqrt(zoneX[1]*zoneX[1]+zoneY[1]*zoneY[1]);
  double vectH=sqrt(zoneX[0]*zoneX[0]+zoneY[0]*zoneY[0]);
  if (vectV > vectH)
    vectV=vectH;
  const int max_v(1+static_cast<int>(dist/vectV));
  const int max_h(1+static_cast<int>(dist/vectV));
  Peak Htest,Vgood,Hgood,Tp;
				  
  Vgood=Pk[start];
  Hgood=Pk[start];
  Tp=Pk[start];
  int vcount=0;  //absolute number in each directon
  int hcount=0;
  int relH=0;   //distance to best option
  int relV=0;   //distance to best option

  for(int hplus=-1;hplus<2;hplus+=2)
    {
      Hgood=Pk[start];                 //start in centre.
      relH = (hplus>0) ? 1 : 0;   //relH is subtracted
      hcount=0;
      while(hcount<max_h && hcount> -max_h)
	{
	  Tp=Hgood;
	  hcount+=hplus;
	  Tp.x+=(hcount-relH)*zoneX[0];
	  Tp.y+=(hcount-relH)*zoneY[0];
	  if (!add_refined(Tp,10*averS))
	    {
	      relH=hcount;  //hcount is added /sub as we go.
	      Hgood=Tp;
	    }
	  for(int vplus=-1;vplus<2;vplus+=2)
	    {
	      Vgood=Hgood; //start in centre.
	      Vgood.x+=(hcount-relH)*zoneX[0];
	      Vgood.y+=(hcount-relH)*zoneY[0];   //don't refine.
	      
	      relV=0;
	      vcount=0;
	      while(vcount<max_v && vcount> -max_v)
		{	
		  Tp=Vgood;
		  vcount+=vplus;
		  Tp.x+=(vcount-relV)*zoneX[1];
		  Tp.y+=(vcount-relV)*zoneY[1];
		  if (!add_refined(Tp,10*averS))
		    {
		      relV=vcount;  //hcount is added /sub as we go.
		      Vgood=Tp;
		    }
		}
	    }
	}
    }	
	  
  npeak=Pk.size();
  overlap();  //remove overlaps
  s_line();   //reset cursor and return;
  return 3;   //number of peak positions found.
}


int
space_correct::directions()
  /*
    Job is to find the directions that are best suited to 
    a square matrix.
  */
{
  double dista=Pk[0].sep(Pk[1]);
  double distb=Pk[0].sep(Pk[2]);
  double distc=Pk[1].sep(Pk[2]);
  if (dista>distb && dista>distc)
    {
      distX=distb;
      distY=distc;
      zoneX[0]=Pk[0].x-Pk[2].x;
      zoneX[1]=Pk[0].y-Pk[2].y;
      zoneY[0]=Pk[1].x-Pk[2].x;
      zoneY[1]=Pk[1].y-Pk[2].y;
      return 2;
    }
  else if (distb>distc)
    {
      distX=dista;
      distY=distc;
      zoneX[0]=Pk[0].x-Pk[1].x;
      zoneX[1]=Pk[0].y-Pk[1].y;
      zoneY[0]=Pk[2].x-Pk[1].x;
      zoneY[1]=Pk[2].y-Pk[1].y;
      return 1;
    }
  else
    {
      distX=distb;
      distY=dista;
      zoneX[0]=Pk[2].x-Pk[0].x;
      zoneX[1]=Pk[2].y-Pk[0].y;
      zoneY[0]=Pk[1].x-Pk[0].x;
      zoneY[1]=Pk[1].y-Pk[0].y;
      return 0;
    }
}

void
space_correct::converge(Peak& A,const Peak& B) const
{
  A.x=(B.x+A.x)/2.0;
  A.y=(B.y+A.y)/2.0;
  refine(A);
  return;
}


int
space_correct::add_refined(Peak& A,const double max_score)
  /* adds the point to the list or not 
     if score low rejects */
{
  if (A.x<1 || A.x>=xp ||  //check actual position.
      A.y<1 || A.y>=yp)
    return 1;
  double score=refine(A);
  if (score<=0.0) return 2;
  if (score>max_score) return 3; //largest possible score
  Pk.push_back(A);
  draw_cross(A.x,A.y);
  return 0;
}


double
space_correct::refine(Peak &A) const
  /* 
     Refine works by calculating the decay values to the edge of the 
     box (CURRENTLY NOT USED) then calculating the full system
     followed by calculating the spread (val*val*dist)/mean 
  */
{
  int a,b;
  a=int(A.x);
  b=int(A.y);
  int spd = 1+(int) ((distY>distX) ? distY : distX); 
  spd/=2;
  double ymon,xmon;
  double leftX,rightX;
  double lowY,topY;
  leftX=rightX=0.0;
  lowY=topY=0.0;
  ymon=xmon=0.0;
  if (a>=spd && a+spd<xp && b>=spd && b+spd<yp)
    {
      for(int i=-spd;i<spd;i++)  //get slope across sample
	{
	  leftX+=V[b+i][a-spd];
	  rightX+=V[b+i][a+spd];
	  lowY+=V[b-spd][a+i];
	  topY+=V[b+spd][a+i];
	}
    }
  else
    {
      rightX=leftX=topY=lowY=0.0;
    }
  rightX/=2.0*spd;
  topY/=2.0*spd;
  leftX/=2.0*spd;
  lowY/=2.0*spd;
  double splx=(rightX-leftX)/(2.0*spd);
  double sply=(topY-lowY)/(2.0*spd);
  leftX+=spd*splx;
  lowY+=spd*sply;
  double sumX=0.0;
  int ix,iy;
  double mxv,mnv;
  if (a>=0 && a<spd && b>=0 && b<xp)
    mxv=mnv=V[a][b];
  else
    {
      mxv= -1e200;
      mnv= 1e200;  //nice large numbers ! 
    }
  int mptx,mpty;
  for(int i=-spd;i<spd;i++)
    {
      ix=a+i;
      for(int j=-spd;j<spd;j++)
	{
	  iy=b+j;
	  if (ix<xp && ix>=0 && iy>=0 && iy<yp)
	    {
	      if (mxv<V[iy][ix])
		{
		  mxv=V[iy][ix];
		  mptx=ix;
		  mpty=iy;
		}
	      else if (mnv>V[iy][ix])
		mnv=V[ix][iy];
	    }
	}
    }
  mnv=(mxv+mnv)/2;
  double val;
  for(int i=-spd;i<spd;i++)
    {
      ix=a+i;
      if (ix>=0 && ix<xp)
	{
	  for(int j=-spd;j<spd;j++)
	    {
	      iy=b+j;
	      if (iy>=0 && iy<yp)
		{
		  val=V[iy][ix]-mnv;
		  if (val>0.0)
		    {
		      xmon+= i * val*val;  
		      ymon+= j * val*val;
		      sumX+= val*val;
		    }
		}
	    }
	}
    }
  if (sumX)
    {
      A.x=a+xmon/sumX;
      A.y=b+ymon/sumX;
      if (A.x>a+spd || A.x<a-spd ||
	  A.x>=xp || A.x<0  ||
	  A.y>b+spd || A.y<b-spd ||
	  A.y>=yp || A.y<0)
	{
	  A.x=mptx;
	  A.y=mpty;
	}
    }
  else
    {
      A.x=mptx;
      A.y=mpty;
    }

  for(int i=-spd;i<spd;i++)
    {
      ix=a+i;
      if (ix>=0 && ix<xp)
	{
	  for(int j=-spd;j<spd;j++)
	    {
	      iy=b+j;
	      if (iy>=0 && iy<yp)
		{
		  val=V[iy][ix]-mnv;
		  if (val>0.0)
		    xmon+= (A.x-ix)*(A.x-ix)*val*val*(A.y-iy)*(A.y-iy);
		}
	    }
	}
    }
  if (sumX)
    return A.x/sumX;
  return -1.0;
}

void
space_correct::draw_cross(const float x,const float y) const
{
  float xpts[4],ypts[4];
   //With the points convert to XPoint!
  xpts[0]=x;
  xpts[1]=x;
  xpts[2]=x-lsx;
  xpts[3]=x+lsx;
  ypts[0]=y-lsy;
  ypts[1]=y+lsy;
  ypts[2]=y;
  ypts[3]=y;      //these are the points required.
  cpgline(2,xpts,ypts);
  cpgline(2,xpts+2,ypts+2);
  return;
}

space_correct::~space_correct()
{
  delete SpcWind;
}


void
space_correct::draw(const int Tgrp) const
{
  if (Tgrp<0 || Tgrp==grp)
    {
      cpgsci(3);
      cpgslw(1);
      //  float xpts[4],ytps[4];
      for(int i=0;i<(int) Pk.size();i++)
	draw_cross(Pk[i].x,Pk[i].y);
    }
  return;
}

