#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <vector>
#include <map>
#include <algorithm>
#include "areaInt.h"

bool operator<(const std::pair<double,Point>&,
	       const std::pair<double,Point>&);

Point::Point(const Point &A)
{
  x=A.x;
  y=A.y;
}

Point&
Point::operator=(const Point &A)
{
  if (this != &A)
    {
      x=A.x;
      y=A.y;
    }
  return *this;
}

Point& Point::operator+=(const Point& A)
{
  x+=A.x;
  y+=A.y;
  return *this;
}

Point Point::operator+(const Point& A) const
{
  Point out = *this;
  out+=A;
  return out;
}

Point& Point::operator-=(const Point& A)
{
  x-=A.x;
  y-=A.y;
  return *this;
}

Point Point::operator-(const Point& A) const
{
  Point out = *this;
  out-=A;
  return out;
}

double
Point::dist(const Point &A) const 
{
  double tx=A.x-x;
  tx*=tx;
  double ty=A.y-y;
  ty*=ty;
  return sqrt(tx+ty);
}

double
Point::dist() const
{
  return sqrt(x*x+y*y);
}

void
Point::boundary(const double ca,const double sa,
		Point& LB,Point& TB) const
  /* 
     Get the boundary points that a line crosses an
     integer box LB = point that is -ve unit direction
      TB = +ve direction.
  */
{
  double xv[2],yv[2];
  yv[0] =static_cast<int>(y)-y;
  yv[1]=yv[0]+1.0;
  xv[0] = static_cast<int>(x)-x;
  xv[1] = xv[0]+1.0;
  
  double Rneg[4];
  for(int i=0;i<2;i++)
    {
      Rneg[2*i]=yv[i]/sa;
      Rneg[2*i+1]=xv[i]/ca;
    }
  double nval(-1000.0),pval(1000.0);     //1000 hopefully out of range :-)
  int npt,ppt;
  for(int i=0;i<4;i++)
    {
      if (Rneg[i]<0.0 && Rneg[i]>nval)   //least negative
	{
	  nval=Rneg[i];
	  npt=i;
	}
      if (Rneg[i]>0.0 && Rneg[i]<pval)    //least positive
	{
	  pval=Rneg[i];
	  ppt=i;
	}
    }
  LB.x=x+Rneg[npt]*ca;
  LB.y=y+Rneg[npt]*sa;
  TB.x=x+Rneg[ppt]*ca;
  TB.y=y+Rneg[ppt]*sa;
  return;
}

Point
Point::arcPoint(Point& C,double ssa,double csa,double R) const
  /* 
     Assumes that centre is in the middle  (0,0) 
     finds that exit points for the arc from the centre 
  */
{
  Point out = *this;   //to be removed
  if (R<=0.0)
    R=dist(C);
  if (csa<-1.0 || csa >1.0 ||  //points not given
      ssa<-1.0 || ssa>1.0)
    {
      csa=Flat_angle(C);
      ssa=sin(csa);
      csa=cos(csa);
    }
  int xpt,ypt;
  xpt=static_cast<int>(x);
  ypt=static_cast<int>(y);
  double Xaim=static_cast<double>((ssa>0) ? xpt : xpt+1); 
  double Yaim=static_cast<double>((csa>0) ? ypt+1 : ypt); 
  // Solve two intercept points  //start with Xaim
 
  for(int i=0;i<2;i++)     //loop to save me writing code!
    {
      double acs=(Xaim-C.x)/R;    // problem is the +/- sign!
      double ass=sqrt(1.0-acs*acs);
      double ypA=C.y+ass*R;
      double ypB=C.y-ass*R;
      if (ypA>=ypt && ypA<=ypt+1)
	{
	  out.x=Xaim;
	  out.y=ypA;
	  return out;
	}
      if (ypB>=ypt && ypA<=ypt+1) 
	{
	  out.x=Xaim;
	  out.y=ypB;
	}
      
      //Oh well that means it should be Yaim
      ass=(Yaim-C.y)/R;    // problem is the +/- sign!
      acs=sqrt(1.0-acs*acs);
      ypA=C.x+acs*R;        //NOTE not actually ypA but xpA!
      ypB=C.x-acs*R;
      if (ypA>=ypt && ypA<=ypt+1)
	{
	  out.x=Xaim;
	  out.y=ypA;
	  return out;
	}
      if (ypB>=ypt && ypA<=ypt+1) 
	{
	  out.x=Xaim;
	  out.y=ypB;
	  return out;
	}
      Xaim=static_cast<double>((ssa<=0) ? xpt : xpt+1); 
      Yaim=static_cast<double>((csa<=0) ? ypt+1 : ypt); 
    }
  out.x=0.0;
  out.y=0.0;
  return out;
}


double
Point::c_angle(const Point& A) const
  /* find the angle between This and A throught AOT . */ 
{
  return (x*A.x+A.y*y)/(A.dist() * dist());
}

double
Point::Flat_angle(const Point& C) const
  /* returns the y=c.y line angle */
{
  Point B = *this - C;
  return atan2(B.y,B.x);
}

double
Point::c_angle(Point A,Point B) const
  /* Given points, A,B and This find the cos(theta) of ATB */
{
  A-= *this;
  B-= *this;
  return B.c_angle(A);
}

double
Point::Tri(Point A,Point B) const
  /* Given triangle from this,A,B find the area */
{
  A-=*this;
  B-=*this;
  double distOB=B.dist();
  double distOA=A.dist();
  double cang=A.c_angle(B);
  return 0.5*distOB*distOA*sqrt(1.0-cang*cang);
}

bool operator<(const std::pair<double,Point>& A,
	       const std::pair<double,Point>& B) 
{
  return A.first<B.first;
}


AreaInt::AreaInt() : 
  V(0),cent(0,0),TArea(0.0),n_hp(0),n_vp(0)
{ }

AreaInt::~AreaInt()
{}

double 
AreaInt::ArcA(Point A,Point B,Point C,const double R)
/* 
   Input is point A and B on the surfact of the sphere,
   Point, C the place were the area is taken from. */
{
  A-=cent;
  B-=cent;
  C-=cent;
  double cang=A.c_angle(B);
  double ang=acos(cang);
  double arc=0.5*R*R*(ang-sin(ang)); //Area of arc only
  return C.Tri(A,B)+arc;
}

void
AreaInt::Set_centre(const double x,const double y)
{
  cent=Point(x,y);
}

void
AreaInt::Initialise(float **Vval,const int nhp,const int nvp)
{
  V=Vval;
  n_hp=nhp;
  n_vp=nvp;
}
      

int
AreaInt::find_line(std::vector<Point>& LL,const Point& Spt,const Point& Ept)
  /* given Spt and Ept that difine a line, then we have LL which
     is the points in the crossing boundary between them assuming
     that the grid is integer orientated. 
     Note: LL must already have the first point (Spt) in it. */ 
{
  int sx=static_cast<int>(Spt.x);
  int sy=static_cast<int>(Spt.y);
  int ex=static_cast<int>(Ept.x);
  int ey=static_cast<int>(Ept.y);
  
  if (sx==ex && sy==ey)   //Ok first off are we in the same square!!
    {
      LL.push_back(Ept);
      return 2;
    }
  
  /* Now we can solve between sx and sy and ex and ey
     all bundary points have to be added. */

  double rx=Ept.x-Spt.x;
  double ry=Ept.y-Spt.y;   //Now only 0>r>1 are acceptable :-)
  std::vector<std::pair<double,Point> > Dpts;   //distance and Point!
  Point Temp;
  double Dd;
  if ((ex-sx)>0)  
    {
      for(int i=sx+1;i<ex+1;i++)
	{
	  Dd= (i-Spt.x)/rx;
	  Temp.x=static_cast<double>(i);
	  Temp.y=Dd*ry+Spt.y;
	  Dpts.push_back(std::pair<double,Point>(Dd,Temp));
	}
    }
  else if ((ex-sx<0))
    {
      for(int i=sx;i>ex;i--)
	{
	  Dd= (i-Spt.x)/rx;
	  Temp.x=static_cast<double>(i);
	  Temp.y=Dd*ry+Spt.y;
	  Dpts.push_back(std::pair<double,Point>(Dd,Temp));
	}
    }

  if ((ey-sy)>0)  
    {
      for(int i=sy+1;i<ey+1;i++)
	{
	  Dd= (i-Spt.y)/ry;
	  Temp.x=Dd*rx+Spt.x;
	  Temp.y=static_cast<double>(i);
	  Dpts.push_back(std::pair<double,Point>(Dd,Temp));
	}
    }
  else if ((ey-sy<0))
    {
      for(int i=sy;i>ey;i--)
	{
	  Dd= (i-Spt.y)/ry;
	  Temp.x=Dd*rx+Spt.x;
	  Temp.y=static_cast<double>(i);
	  Dpts.push_back(std::pair<double,Point>(Dd,Temp));
	}
    }
  sort(Dpts.begin(),Dpts.end());  
  for(int i=0;i<(int) Dpts.size();i++)
    {
      if (Dpts[i].first>0.0 && Dpts[i].first<1.0)
	LL.push_back(Dpts[i].second);
    }
  LL.push_back(Ept);
  return LL.size();
}

int
AreaInt::EvalArcUnit(double& YV,const double R1,const double R2,
		   const double SA,const double EA)
{
  TArea=0.0;  //reset area
  double ssa=sin(SA);
  double csa=cos(SA);
  double sea=sin(SA);
  double cea=cos(SA);
  Point A(csa*R1,ssa*R1);   //Four main points
  Point B(csa*R2,ssa*R2);
  Point C(cea*R1,sea*R1);
  Point D(cea*R2,sea*R2);
  A+=cent; B+=cent; C+=cent; D+=cent;  //give then the correct offset.

  //Better way is to calculate the list of points in the square
  //going round the system
  std::vector<Point> AB;
  std::vector<Point> BC;
  std::vector<Point> CD;
  std::vector<Point> DA;
  
  AB.push_back(A);  //Ok start with 
  find_line(AB,A,B);
  find_line(CD,C,D);
  for(int i=0;i<AB.size();i++)
    cout<<" "<<AB[i].x<<" "<<AB[i].y<<endl;

  Point Iout,Iin;
  Point Anxt;
  A.boundary(ssa,csa,Iout,Iin);
  Anxt=A.arcPoint(cent,ssa,csa);  //pass centre ,R ,ssa, csa
  
  YV=TArea;
  return (TArea>0) ? 0 : 1;
}
















