#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>
#include <math.h>
#include <string>
#include <vector>
#include "corpy.h"
#include "datain.h"
#include "fglo.h"


DataIn::DataIn() :
  Angle(0),Yval(0),Err(0),Npts(0)
{}

DataIn::DataIn(const DataIn& A)  :
  Npts(A.Npts)
{
  if (Npts)
    {
      Angle = new double[Npts];
      Yval = new double[Npts];
      Err = new double[Npts];
    }
  else
    {
      Angle=0;
      Yval=0;
      Err=0;
    }
  for(int i=0;i<Npts;i++)
    {
      Angle[i] = A.Angle[i];
      Yval[i] = A.Yval[i];
      Err[i] = A.Err[i];
    }
  return;
}

DataIn
DataIn::operator-(const DataIn& A) const
{
  DataIn Temp(*this);
  Temp-=A;
  return Temp;
}

DataIn
DataIn::operator*(const DataIn& A) const
{
  DataIn Temp(*this);
  Temp*=A;
  return Temp;
}

DataIn& 
DataIn::operator-=(const DataIn& A)
{
  if (A.Npts != Npts)
    {
      cerr<<"Points not equal in minus operator"<<endl;
    }
  else
    {
      for(int i=0;i<A.Npts;i++)
	{
	  Yval[i] -= A.Yval[i];
	  Err[i] = sqrt(Err[i]*Err[i]+A.Err[i]*A.Err[i]);
	}
    }
  return *this;
}

DataIn& 
DataIn::operator*=(const DataIn& A)
{
  if (A.Npts != Npts)
    {
      cerr<<"Points not equal in multiply operator"<<endl;
    }
  else
    {
      for(int i=0;i<A.Npts;i++)
	{
	  Yval[i] *= A.Yval[i];
	  Err[i] = sqrt(A.Yval[i]*A.Yval[i]*Err[i]*Err[i]+
	    Yval[i]*Yval[i]*A.Err[i]*A.Err[i]);
	}
    }
  return *this;
}


DataIn& 
DataIn::mult(const corpy& A,const double* P) 
{
  for(int i=0;i<Npts;i++)
    {
      double cfact=A.polyval(P,Angle[i]);
      Yval[i] *= cfact;
      Err[i] *= cfact;
    }
  return *this;
}



DataIn&
DataIn::operator=(const DataIn& A)
{
  if (&A!=this)
    {
      resize(A.Npts);
      for(int i=0;i<Npts;i++)
	{
	  Angle[i]=A.Angle[i];
	  Yval[i]=A.Yval[i];
	  Err[i]=A.Err[i];
	}
    }
  return *this;
}

DataIn::~DataIn()
{
  delete [] Angle;
  delete [] Yval;
  delete [] Err;
}

void
DataIn::resize(const int N)   //private function
  /* Resize arrays */
{
  if (N!=Npts)
    {
      delete [] Angle;
      delete [] Yval;
      delete [] Err;
      if (N>0)
	{
	  Angle = new double[N];
	  Yval = new double[N];
	  Err = new double[N];
	}
      else
	{
	  Angle=0;
	  Yval=0;
	  Err=0;
	}
      Npts=N;
    }
  return;
}

int
DataIn::write(const char *Fname,const int append) const
{
  ofstream of;
  if (append)
    of.open(Fname,ios::out|ios::app|ios::nocreate);
  else
    of.open(Fname,ios::out);
  if (!of)
    {
      cerr<<"Error writing file "<<Fname<<endl;
      return -1;
    }
  of<<" 3 "<<Npts<<endl;
  of<<" # DataIn "<<endl;
  for(int i=0;i<Npts;i++)
    of<<Angle[i]<<" "<<Yval[i]<<" "<<Err[i]<<endl;
  of.close();
  return 0;
}


void
DataIn::progressFit(const int npow,const double Sx,const double Ex)
{
  const double chival[] = { 20.0,10.0,5.0 };
  double* X=new double[Npts];
  double* Xwork=new double[Npts];
  double* Ywork=new double[Npts];
  double* Ework=new double[Npts];
  double* Fit=new double[npow];

  double maxX=Angle[Npts-1];
  for(int i=0;i<Npts;i++)
    X[i]=Angle[i]/maxX;
  write_stream("Prg.dat",Npts,X,Yval,Err,0);
  svdpoly(X,Yval,Err,Npts,npow,Fit);
  for(int i=0;i<3;i++)
    {
      for(int ii=0;ii<npow;ii++)
	{
	  cerr<<"P["<<ii<<"] == "<<Fit[ii]<<endl;
	}
      cerr<<"-------------------------"<<endl;

      int cnt(0);
      for(int j=0;j<Npts;j++)
	{
	  if (Angle[j]>Sx && Angle[j]<Ex)
	    {
	      double Y=polyval(npow,Fit,X[i]);
	      if (fabs(Yval[j]-Y) < Err[j]*chival[i])
		{
		  Xwork[cnt]=X[j];
		  Ywork[cnt]=Yval[j];
		  Ework[cnt]=Err[j];
		  cnt++;
		}
	    }
	}
      write_stream("Prg.dat",cnt,Xwork,Ywork,Ework,1);
      cerr<<"Cnt == "<<cnt<<endl;
      svdpoly(Xwork,Ywork,Ework,cnt,npow,Fit);
    }
  for(int i=0;i<Npts;i++)
    Yval[i]=polyval(npow,Fit,X[i]);
  for(int i=0;i<npow;i++)
    {
      cerr<<"P["<<i<<"] == "<<Fit[i]<<endl;
    }
  delete [] X;
  delete [] Fit;
  delete [] Ywork;
  delete [] Ework;
}


double
DataIn::polyval(const int size,const double* P,const double x) const
{
  double sum=P[0];
  double xx=x;
  for(int i=1;i<size;i++)
    {
      sum+=P[i]*xx;
      xx*=x;
    }
  return sum;
}


int 
DataIn::read(const char* Fn)
  /* 
     job is to read in the experimental EXAFS file 
     File format is   
        title line
	X , Y    newline
   */
{
  ifstream ipf;
  ipf.open(Fn);
  if (!ipf) 
    {
      cerr<<"Unable to open file "<<Fn<<endl;
      return 1;
    }
  char ss[255];
  int npts;
  ipf.getline(ss,255,'\n'); //number of point
  if (!cmdnumber(ss,npts,255) || npts<1)
    {
      cerr<<"No point to read"<<endl;
      return -1;
    }
  int cnt=0;
  resize(npts);
  double xv,yv,zv;

  while(cnt<npts && ipf)
    {
      ipf.getline(ss,255,'\n');
      if (!cmdnumber(ss,xv,255) || !cmdnumber(ss,yv,255)
	  || !cmdnumber(ss,zv,255))
        {
	  cerr<<"Error with file :: Read "<<cnt<<" pts "<<endl;
	  return -2;
	}
      Angle[cnt]=xv;
      Yval[cnt]=yv;
      Err[cnt]=zv;
      cnt++;
      while(cmdnumber(ss,xv,255) && cmdnumber(ss,yv,255) && 
	    cmdnumber(ss,zv,255))
	{
	  Angle[cnt]=xv;
	  Yval[cnt]=yv;
	  Err[cnt]=zv;
	  cnt++;
	}
    }
  return 0;
}




