#
# Note that public data can be blessed directly 
# 
package rawData;
use strict;

use constant inAngle => 8.0; 
use constant subAngle => 7.0; 

sub new 
{
  my $type = shift;
  my $class = ref($type) || $type;
  my $self = {
    Offset => undef,
    Monitor => 0.0,
    Npts  => 0,
    E   => undef,
    Y   => undef,
    SS  => undef,
    type => 0      # 0 == energy, 1 == k (Default = energy)
    };
  $self->{E} = []; # fill with empty arrays....
  $self->{Y} = [];
  $self->{SS} = [];
  bless $self,$class;
  return $self;
}

sub makeSubUnit
 # Create a rawData unit from a simple array reference of the y data.
{
  my $self=shift;
  $self->{Monitor} = shift;
  $self->{Offset}=shift;
  my $effR =  shift;
  my $Ar = shift;
  $self->{E}= [];
  $self->{Y}= [];
  $self->{SS}= [];
  my $cnt=0;
  for(my $i=0;$i<=$#$Ar;$i++)
    {
      if ($effR->[$i]>0.0)
	{
	  $cnt++;
	  push(@{$self->{E}},(inAngle*$i/64+$self->{Offset}));
	  push(@{$self->{Y}},$Ar->[$i]/$effR->[$i]);
	  push(@{$self->{SS}},sqrt($Ar->[$i]/$effR->[$i]));
	}
    }
  $self->{Npts}=$cnt;
}

sub copy
# Equivienlent of a copy constructor.
{
  my $self = shift;
  my $sref = shift;
  if (ref($sref) eq "rawData")
    {
      $self->{Npts} = $sref->{Npts};
      $self->{Monitor}=$sref->{Monitor};
      $self->{Offset}=$sref->{Offset};
      $self->{E}= [];
      $self->{Y}= [];
      for(my $i=0;$i<$self->{Npts};$i++)
	{
	  push(@{$self->{E}},$sref->{E}[$i]);
	  push(@{$self->{Y}},$sref->{Y}[$i]);
	}
      $self->{type}=$sref->{type};
    }
  return $self;
}

sub Ktype
  # set/get Ktype of data
{
  my $self = shift;
  return (@_) ? $self->{type}=$_[0] : $self->{type};
}


sub load
  # This should load the data from a spec file.
  # inputs are Number, Ecol,Ycol, ss cole
{
  my $self=shift;
  my $number=shift;
  my $Ecol=shift;
  my $Xacol=shift;
  $Ecol--;
  $Xacol--;

  my $filename = shift;

  if (!open(FH,$filename))
    {
       print "Sorry the file ",$filename," could not be opened \n";
       return;
     }
  
  $self->{E} = [];
  $self->{Y} = [];

  my $line;
  my $Rd=0;
  while($line=<FH>)
    {
      if (!$Rd) 
	{
	  if ($line =~ /^\#S\s([0-9]*)/)
	    {
	      if ($1 == $number)
		{
		  print "Found section :: ",$line;
		  $Rd=1;
		  $number++;
		}
	    }
	}
      else
	{
	  if (($line =~ /^\#S\s([0-9]*)/) &&
	      ($1==$number))
	    {
	      $Rd=0;
	    }
	  elsif (!($line =~ /\#/))
	    {
              my @pts=split ' ',$line; 
	      if ($#pts>=$Ecol && $#pts>=$Xacol)
                { 
		  push(@{$self->{E}},$pts[$Ecol]);
		  push(@{$self->{Y}},$pts[$Xacol]);
		}
	    }
	}
    }
  $self->{Npts}=scalar(@{$self->{E}});
  $self->{type}=0;   # Energy type unless specified otherwise
  print "Number of point read == ",$self->{Npts},"\n";
  return;
}

sub loadstandard
  # This should load the data from a spec file.
  # inputs are Number, Ecol,Ycol, ss cole
{
  my $self=shift;
  my $filename = shift;  
  my $Ecol=shift;
  my $Xacol=shift;
  $Ecol--;
  $Xacol--;

  if (!open(FH,$filename))
    {
      print "Sorry the file ",$filename," could not be opened \n";
      return 0;
    }
  
  $self->{E} = [];
  $self->{Y} = [];

  my $line;
  my $Rd=0;
  while($line=<FH>)
    {
      if (!($line =~ /^\S*\#/))
	{
	  my @pts=split ' ',$line; 
	  if ($#pts>=$Ecol && $#pts>=$Xacol)
	    { 
	      push(@{$self->{E}},$pts[$Ecol]);
	      push(@{$self->{Y}},$pts[$Xacol]);
	    }
	}
    }
  $self->{Npts}=scalar(@{$self->{E}});
  $self->{type}=0;
  print "Number of point read == ",$self->{Npts},"\n";
  return $self->{Npts};
}
  
sub smooth
 # Calculates a smoothed spectra using simultanious point sum
{
  my $self=shift;
  my $niter=shift;  # Number of iterations 
  my ($sf,$sm,$se);

  if ($self->{Npts}<=3)
    {
      print "No points in spectra \n";
      return 1;
    }
  my $Yr = $self->{Y};
  print "Number of points for smoothing == ",$#$Yr+1,"\n";
  for(my $iv=0;$iv<$niter;$iv++)
    {
      $sf= $Yr->[0];
      $sm= $Yr->[0];
      for(my $i=0;$i<$#$Yr;$i++)
	{
	  $se=$Yr->[$i+1];
	  $Yr->[$i]=
	      0.25*$sf+0.5*$sm+0.25*$se;
	  $sf=$sm;
	  $sm=$se;
	}
      $Yr->[$#$Yr]=0.75*$se+0.25*$sf;
    }
  return 0;
}


sub Abs
# Makes the spectra all positive 
{
  my $self=shift;
  return 1 if ($self->{Npts}<=0);

  for(my $i=0;$i<$self->{Npts};$i++)
    {
      $self->{Y}[$i]=abs($self->{Y}[$i]);
    }
  return 0;
}

sub deriv
# Takes the derivative of the spectra.
{
  my $self=shift;
  return 1 if ($self->{Npts}<=0);

  my $Xr = $self->{E};
  my $Yr = $self->{Y};
  my @Dp = [ ];
  push(@Dp,0.0);

  for(my $i=1;$i<$self->{Npts}-1;$i++)
    {
      my $tmp = ($Yr->[$i]-$Yr->[$i-1])/($Xr->[$i]-$Xr->[$i-1]);
      $tmp += ($Yr->[$i]-$Yr->[$i+1])/($Xr->[$i]-$Xr->[$i+1]);
      push(@Dp,0.5*$tmp);
    }
  push(@Dp,0.0);
  $self->{Y}= \@Dp;
  return 0;
}

sub length
# Calculates the length of the spectra
{
  my $self=shift;
  my $leng=0.0;
  if ($self->{Npts}<=0)
    {
      return 0.0;
    }
  my $stpt= (@_) ? shift : 1;
  my $edpt= (@_) ? shift : $self->{Npts};
  if ($stpt<1 || $stpt>=$self->{Npts})
    {
      $stpt=1;
    }
  if ($edpt<=$stpt || $edpt>=$self->{Npts})
    {
      $edpt=$self->{Npts};
    }

  my $Er = $self->{E};
  my $Yr = $self->{Y};

  for(my $i=$stpt;$i<$edpt;$i++)
    {
      $leng+= abs( ($Yr->[$i] - $Yr->[$i-1])/($Er->[$i]-$Er->[$i-1]) );
    }
  return $leng;
}

sub scale
  # objective is to multiply 
{
  my $self=shift;
  my $sv=shift;

  my $i;

  for($i=0;$i<$self->{Npts};$i++)
    {
      $self->{Y}[$i]*=$sv;
    }
  return;
}


sub write
  # Write the E and Y data.
{
  my $self=shift;
  my $name = shift;
  my $append = (@_) ? shift : 0;
  my $comment = (@_) ? shift : 0;
  if ($append)
    { open(FH,">>$name") or return 1; }
  else
    { open(FH,">$name") or return 1; }

  print FH "# " if ($comment);
  print FH " 3 ",$self->{Npts}," 0.1 0.1\n";
  print FH "# ",$name,"\n";
  for(my $i=0;$i<$self->{Npts};$i++)
    {
      print FH " ",$self->{E}[$i]," ";
      print FH " ",$self->{Y}[$i]," ";
      print FH " ",sqrt( abs($self->{Y}[$i]) )," ";
      print FH "\n";
    }
  close FH;
  return 0;
}

sub writeNoHeader
  # Write the E and Y data.
{
  my $self=shift;
  my $name = shift;
  my $append = (@_) ? shift : 0;
  my $comment = (@_) ? shift : 0;
  my $monitor = $self->{Monitor}/6.0e6;
  if ($append)
    { open(FH,">>$name") or return 1; }
  else
    { open(FH,">$name") or return 1; }

  for(my $i=0;$i<$self->{Npts};$i++)
    {
      print FH " ",$self->{E}[$i]," ";
      print FH " ",$self->{Y}[$i]/$monitor," ";
      print FH " ",sqrt( abs($self->{Y}[$i]/$monitor) )," ";
      print FH "\n";
    }
  close FH;
  return 0;
}

sub normalise
 # Attempt to zero the baseline and make the jump 1
{
  my $self=shift;
  my $prebase=0.0;
  
  my $Yr=$self->{Y};
  for(my $i=0;$i<10;$i++)
    {
      $prebase+=$Yr->[$i];
    }
  $prebase/=10.0;
  my $mainline=0.0;
  for(my $i=$#$Yr;$i>$#$Yr-10;$i--)
    {
      $mainline+=$Yr->[$i];
    }
  $mainline/=10.0;
  $mainline=$mainline-$prebase;
  print "Normalise:: Main line == ",$mainline," pre == ",$prebase,"\n";
  for(my $i=0;$i<=$#$Yr;$i++)
    {
      $Yr->[$i]-=$prebase;
      $Yr->[$i]/=$mainline;
    }
  
  return $self;
}


sub offset
  # objective is to multiply 
{
  my $self=shift;
  my $sv=shift;

  my $i;

  for($i=0;$i<$self->{Npts};$i++)
    {
      $self->{Y}[$i]+=$sv;
    }
  return;
}

sub chi_diff
  # calc the total difference
{
  my $self=shift;
  my $sref = shift;
  my $chi=0.0;
  my $i;

  if (ref($sref) eq "rawData")
    {
      if ($self->{Npts} !=$sref->{Npts})
        {
	  print "Error with number of points\n";
	  return 0.0;
	}
      for($i=0;$i<$self->{Npts};$i++)
        {
	  my $td=$self->{Y}[$i]-$sref->{Y}[$i];
	  $chi+=$td * $td;
	}
    }
  return $chi;
}

sub self_chi
  # calc the total difference :: Assuming that the difference has
  # been put into the file.
{
  my $self=shift;
  my $chi=0.0;
  my $i;

  for($i=0;$i<$self->{Npts};$i++)
  {
    $chi+=$self->{Y}[$i]*$self->{Y}[$i];
  }
  return $chi;
}


sub subtract
  # calc explicit difference 
{
  my $self = shift;
  my $RawRef = shift;
  my $i;

  if (ref($RawRef) eq "rawData")
    {
      if ($self->{Npts} !=$RawRef->{Npts})
        {
	  print "Error with number of points\n";
	  return 1;
	}
      for($i=0;$i<$self->{Npts};$i++)
        {
	  $self->{Y}[$i]-=$RawRef->{Y}[$i];
	}
      $self->{Monitor}-=$RawRef->{Monitor};
    }
  return 0;
}

sub add
  # calc explicit difference takes two spectra calc A+=B
{
  my $self = shift;
  my $RawRef = shift;
  my $i;

  if (ref($RawRef) eq "rawData")
    {
      if ($self->{Npts} !=$RawRef->{Npts})
        {
	  print "Error with number of points\n";
	  return 1;
	}
      for($i=0;$i<$self->{Npts};$i++)
        {
	  $self->{Y}[$i]+=$RawRef->{Y}[$i];
	}
      $self->{Monitor}+=$RawRef->{Monitor};
    }
  return 0;
}

sub selfMult
  # calc explicit difference takes two spectra calc A+=B
{
  my $self = shift;
  my $pow = (@_) ? shift : 1.0;

  for(my $i=0;$i<$self->{Npts};$i++)
    {
      $self->{Y}[$i]*=$self->{E}[$i]**$pow;
    }
  return 0;
}

1;









