// **********************************************************************
//
// Copyright (c) 2002
// IONA Technologies, Inc.
// Waltham, MA, USA
//
// All Rights Reserved
//
// **********************************************************************

#ifndef OB_EXCEPT_H
#define OB_EXCEPT_H

#include <OB/RefCounted_fwd.h>

#include <OB/Basic.h>
#include <OB/Fixed.h>
#include <OB/AnyDeclaration.h>
#include <OB/TypeCodeConst.h>
#include <OB/CompletionStatus.h>

namespace CORBA
{

//
// Forward declaration for SystemException
//
class SystemException;

} // End of namespace CORBA

namespace OB
{

//
// The OB::ExceptionBase super class
//
class ExceptionBase
{
public:
    virtual ~ExceptionBase() { }
    virtual void _raise() const = 0;
};

//
// Stub implementation changed exception
//
class StubImplChanged : public ExceptionBase
{
public:

    virtual void _raise() const { throw *this; }
};

//
// Location forward exception
//
class LocationForward : public ExceptionBase
{
public:

    RefCountIOR_var ior; // Forwarded IOR
    bool perm; // Is this a LOCATION_FORWARD_PERM?

    LocationForward(RefCountIOR_ptr, bool);

    virtual void _raise() const { throw *this; }
};

//
// Location forward exception
//
class FailureException : public ExceptionBase
{
public:

    //
    // The system exception causing the failure
    //
    CORBA::SystemException* exception;

    FailureException() : exception(0) { }
    FailureException(const CORBA::SystemException&);
    FailureException(const FailureException&);
    virtual ~FailureException();

    FailureException& operator=(const FailureException&);

    virtual void _raise() const { throw *this; }
};

} // End of namespace OB


namespace CORBA
{

//
// The Exception classes
//
class Exception : public OB::ExceptionBase
{
protected:

    Exception() { }

public:

    //
    // Standard IDL to C++ Mapping
    //
    Exception(const Exception&) { }
    virtual ~Exception() { }
    Exception& operator=(const Exception&) { return *this; }

    virtual void _raise() const = 0;
    virtual const char* _name() const;
    virtual const char* _rep_id() const;

    //
    // Additional ORBacus specific functions
    //
    virtual char* _to_string() const = 0;

    //
    // ORBacus internal functions
    // Application programs must not use these functions directly
    //
    virtual Exception* _OB_clone() const = 0;
    virtual void _OB_insertInAny(Any&) = 0;
    virtual void _OB_marshal(OB::OutputStreamImpl*) const = 0;
};

} // End of namespace CORBA

//
// Exception output on ostream
//
OB_STD(ostream)& operator<<(OB_STD(ostream)&, const CORBA::Exception&);
OB_STD(ostream)& operator<<(OB_STD(ostream)&, const CORBA::Exception*);

namespace OB
{

//
// Exception output using the message viewer
//
//void PrintException(const CORBA::Exception&);

} // End of namespace OB

namespace CORBA
{

//
// When adding a new system exception, don't forget to add (or to add
// to) the following stuff:
//
// Except.h:
//   OB_MAKE_STANDARD_EXCEPTION
//
// Except.cpp:
//   OB_MAKE_STANDARD_EXCEPTION_I
//   SystemException::_OB_unmarshal
//
// Any.h:
//   OB_MAKE_STANDARD_EXCEPTION_INSERTION_EXTRACTION
//
// Any.cpp:
//   OB_MAKE_STANDARD_EXCEPTION_INSERTION_EXTRACTION_I
//
// Typecode.h:
//   OB_IMPORT extern const OB::TypeCodeSysExConst _tc_*;
//
// Typecode.cpp:
//   const OB::TypeCodeSysExConst _tc_*("*");
//   bool CORBA::TypeCode::_OB_isSystemException() const
//
// Also update the test suite:
//
// ob/test/main/TestIntf.idl:
//   Add void op_*_Ex()
//
// ob/test/main/TestDSI_impl.cpp:
//   Add new case for operation
//
// ob/test/main/TestStubException.cpp:
//   Add new call for exception
//
// ob/test/main/Test_impl.cpp:
//   Add new method
//
class SystemException : public Exception
{
    //
    // ORBacus specific. Not transmitted in remote method calls.
    //
    String_var reason_; 

    //
    // ORBacus specific. If true, exception has been received "over
    // the wire". If false, exception has been generated locally.
    //
    bool remote_;

    //
    // Standard CORBA
    //
    ULong minor_;
    CompletionStatus status_;

public:

    //
    // Standard IDL to C++ Mapping
    //
    SystemException(const char* = "");
    SystemException(ULong, CompletionStatus);
    SystemException(const char*, ULong, CompletionStatus);
    SystemException(const SystemException&);
    SystemException& operator=(const SystemException&);

    ULong minor() const { return minor_; }
    void minor(ULong minor) { minor_ = minor; }
 
    CompletionStatus completed() const { return status_; }
    void completed(CompletionStatus status) { status_ = status; }

    static SystemException* _downcast(Exception* p)
    { return dynamic_cast<SystemException*>(p); }
    static const SystemException* _downcast(const Exception* p)
    { return dynamic_cast<const SystemException*>(p); }

    //
    // Additional ORBacus specific functions
    //
    const char* reason() const { return reason_; }
    void reason(const char* reason) { reason_ = reason; }
    bool remote() const { return remote_; }
    void remote(bool remote) { remote_ = remote; }

    //
    // ORBacus internal functions
    // Application programs must not use these functions directly
    //
    char* _OB_defaultToString(const char*) const;
    virtual void _OB_marshal(OB::OutputStreamImpl*) const;
    static SystemException* _OB_unmarshal(OB::InputStreamImpl*);
};

class UserException : public Exception
{
public:

    //
    // Standard IDL to C++ Mapping
    //
    UserException() { }
    UserException(const UserException&) { }
    UserException& operator=(const UserException&)
    { return *this; }

    static UserException* _downcast(Exception* p)
    { return dynamic_cast<UserException*>(p); }
    static const UserException* _downcast(const Exception* p)
    { return dynamic_cast<const UserException*>(p); }

    //
    // ORBacus internal functions
    // Application programs must not use these functions directly
    //
    char* _OB_defaultToString(const char*) const;
    virtual void _OB_marshal(OB::OutputStreamImpl*) const;
};

#define OB_MAKE_STANDARD_EXCEPTION(T) \
\
class T : public SystemException \
{ \
public: \
\
    T(const char* r = ""); \
    T(ULong, CompletionStatus); \
    T(const char*, ULong, CompletionStatus); \
    T(const T&); \
\
    virtual void _raise() const { throw *this; } \
    virtual const char* _name() const; \
    virtual const char* _rep_id() const; \
    virtual char* _to_string() const; \
    static T* _downcast(Exception* p) \
    { return dynamic_cast<T*>(p); } \
    static const T* _downcast(const Exception* p) \
    { return dynamic_cast<const T*>(p); } \
    virtual Exception* _OB_clone() const; \
    virtual void _OB_insertInAny(Any&); \
};

OB_MAKE_STANDARD_EXCEPTION(UNKNOWN)
OB_MAKE_STANDARD_EXCEPTION(BAD_PARAM)
OB_MAKE_STANDARD_EXCEPTION(NO_MEMORY)
OB_MAKE_STANDARD_EXCEPTION(IMP_LIMIT)
OB_MAKE_STANDARD_EXCEPTION(COMM_FAILURE)
OB_MAKE_STANDARD_EXCEPTION(INV_OBJREF)
OB_MAKE_STANDARD_EXCEPTION(NO_PERMISSION)
OB_MAKE_STANDARD_EXCEPTION(INTERNAL)
OB_MAKE_STANDARD_EXCEPTION(MARSHAL)
OB_MAKE_STANDARD_EXCEPTION(INITIALIZE)
OB_MAKE_STANDARD_EXCEPTION(NO_IMPLEMENT)
OB_MAKE_STANDARD_EXCEPTION(BAD_TYPECODE)
OB_MAKE_STANDARD_EXCEPTION(BAD_OPERATION)
OB_MAKE_STANDARD_EXCEPTION(NO_RESOURCES)
OB_MAKE_STANDARD_EXCEPTION(NO_RESPONSE)
OB_MAKE_STANDARD_EXCEPTION(PERSIST_STORE)
OB_MAKE_STANDARD_EXCEPTION(BAD_INV_ORDER)
OB_MAKE_STANDARD_EXCEPTION(TRANSIENT)
OB_MAKE_STANDARD_EXCEPTION(FREE_MEM)
OB_MAKE_STANDARD_EXCEPTION(INV_IDENT)
OB_MAKE_STANDARD_EXCEPTION(INV_FLAG)
OB_MAKE_STANDARD_EXCEPTION(INTF_REPOS)
OB_MAKE_STANDARD_EXCEPTION(BAD_CONTEXT)
OB_MAKE_STANDARD_EXCEPTION(OBJ_ADAPTER)
OB_MAKE_STANDARD_EXCEPTION(DATA_CONVERSION)
OB_MAKE_STANDARD_EXCEPTION(OBJECT_NOT_EXIST)
OB_MAKE_STANDARD_EXCEPTION(TRANSACTION_REQUIRED)
OB_MAKE_STANDARD_EXCEPTION(TRANSACTION_ROLLEDBACK)
OB_MAKE_STANDARD_EXCEPTION(INVALID_TRANSACTION)
OB_MAKE_STANDARD_EXCEPTION(INV_POLICY)
OB_MAKE_STANDARD_EXCEPTION(CODESET_INCOMPATIBLE)
OB_MAKE_STANDARD_EXCEPTION(REBIND)
OB_MAKE_STANDARD_EXCEPTION(TIMEOUT)
OB_MAKE_STANDARD_EXCEPTION(TRANSACTION_UNAVAILABLE)
OB_MAKE_STANDARD_EXCEPTION(TRANSACTION_MODE)
OB_MAKE_STANDARD_EXCEPTION(BAD_QOS)

//
// The UnknownUserException exception
//
class UnknownUserException : public UserException
{
    Any* any_;

public:
    
    //
    // Standard IDL to C++ Mapping
    //
    UnknownUserException();
    UnknownUserException(const UnknownUserException&);
    UnknownUserException(const Any&);
    UnknownUserException& operator=(const UnknownUserException&);
    ~UnknownUserException();
    
    Any& exception();

    static UnknownUserException* _downcast(Exception* p)
    { return dynamic_cast<UnknownUserException*>(p); }
    static const UnknownUserException*
    _downcast(const Exception* p)
    { return dynamic_cast<const UnknownUserException*>(p); }
    virtual void _raise() const { throw *this; }
    virtual const char* _name() const { return "UnknownUserException"; }
    virtual const char* _rep_id() const
    { return "IDL:omg.org/CORBA/UnknownUserException:1.0"; }
    
    //
    // Additional ORBacus specific functions
    //
    virtual char* _to_string() const
    { return _OB_defaultToString("CORBA::UnknownUserException"); }

    //
    // ORBacus internal functions
    // Application programs must not use these functions directly
    //
    virtual Exception* _OB_clone() const
    { return new UnknownUserException(*this); }
    virtual void _OB_insertInAny(Any&);
    virtual void _OB_marshal(OB::OutputStreamImpl*) const;
    static void _OB_unmarshal(UnknownUserException&, OB::InputStreamImpl*);
};

} // End of namespace CORBA

//
// Insertion and extraction for UnknownUserException
//
void operator<<=(CORBA::Any&, CORBA::UnknownUserException*);
void operator<<=(CORBA::Any&, const CORBA::UnknownUserException&);
CORBA::Boolean operator>>=(const CORBA::Any&, CORBA::UnknownUserException*&);

inline void
operator<<=(CORBA::Any_var& any, CORBA::UnknownUserException* val)
{
    any.inout() <<= val;
}

inline void
operator<<=(CORBA::Any_var& any, const CORBA::UnknownUserException& val)
{
    any.inout() <<= val;
}

inline CORBA::Boolean
operator>>=(const CORBA::Any_var& any, CORBA::UnknownUserException*& val)
{
    return any.in() >>= val;
}

namespace OB
{

enum VMCID
{
    OMGVMCID = 0x4f4d0000,
    OOCVMCID = 0x4f4f0000
};

//
// Minor error codes for INITIALIZE
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorInitialize
{
    //
    // ORBacus specific minor codes
    //
    MinorORBDestroyed =  1 | OOCVMCID
};

//
// Minor error codes for NO_IMPLEMENT
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorNoImplement
{
    //
    // Standard minor codes
    //
    MinorMissingLocalValueImplementation = 1 | OMGVMCID,
    MinorIncompatibleValueImplementationVersion = 2 | OMGVMCID,
    MinorNotSupportedByLocalObject = 3 | OMGVMCID,
    MinorDIINotSupportedByLocalObject = 4 | OMGVMCID
};

//
// Minor error codes for OBJECT_NOT_EXIST
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorObjectNotExist
{
    //
    // Standard minor codes
    //
    MinorUnregisteredValue = 1 | OMGVMCID,
    MinorCannotDispatch = 2 | OMGVMCID
};

//
// Minor error codes for BAD_PARAM
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorBadParam
{
    //
    // Standard minor codes
    //
    MinorValueFactoryError = 1 | OMGVMCID,
    MinorRepositoryIdExists = 2 | OMGVMCID,
    MinorNameExists = 3 | OMGVMCID,
    MinorInvalidContainer = 4 | OMGVMCID,
    MinorNameClashInInheritedContext = 5 | OMGVMCID,
    MinorBadAbstractInterfaceType = 6 | OMGVMCID,
    MinorBadSchemeName = 7 | OMGVMCID,
    MinorBadAddress = 8 | OMGVMCID,
    MinorBadSchemeSpecificPart = 9 | OMGVMCID,
    MinorOther = 10 | OMGVMCID,
    MinorInvalidAbstractInterfaceInheritance = 11 | OMGVMCID,
    MinorInvalidValueInheritance = 12 | OMGVMCID,
    MinorIncompleteTypeCodeParameter = 13 | OMGVMCID,
    MinorInvalidObjectId = 14 | OMGVMCID,
    MinorInvalidName = 15 | OMGVMCID,
    MinorInvalidId = 16 | OMGVMCID,
    MinorInvalidMemberName = 17 | OMGVMCID,
    MinorDuplicateLabel = 18 | OMGVMCID,
    MinorIncompatibleLabelType = 19 | OMGVMCID,
    MinorInvalidDiscriminatorType = 20 | OMGVMCID,
    MinorNoExceptionInAny = 21 | OMGVMCID,
    MinorUnlistedUserException = 22 | OMGVMCID,
    MinorNoWcharCodeSet = 23 | OMGVMCID,
    MinorServiceContextIdOutOfRange = 24 | OMGVMCID,
    MinorEnumValueOutOfRange = 25 | OMGVMCID,
    MinorInvalidServiceContextId = 26 | OMGVMCID,
    MinorObjectIsNull = 27 | OMGVMCID,
    MinorInvalidComponentId = 28 | OMGVMCID,
    MinorInvalidProfileId = 29 | OMGVMCID,
    MinorDuplicatePolicyType = 30 | OMGVMCID,

    //
    // ORBacus specific minor codes
    //
    MinorDuplicateDeclarator =  1 | OOCVMCID,
    MinorInvalidValueModifier = 2 | OOCVMCID,
    MinorDuplicateValueInit = 3 | OOCVMCID,
    MinorAbstractValueInit = 4 | OOCVMCID,
    MinorDuplicateBaseType = 5 | OOCVMCID,
    MinorSingleThreadedOnly = 6 | OOCVMCID,
    MinorNameRedefinitionInImmediateScope = 7 | OOCVMCID,
    MinorInvalidValueBoxType = 8 | OOCVMCID,
    MinorInvalidLocalInterfaceInheritance = 9 | OOCVMCID,
    MinorConstantTypeMismatch = 10 | OOCVMCID
};

//
// Minor error codes for BAD_INV_ORDER
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorBadInvOrder
{
    //
    // Standard minor codes
    //
    MinorDependencyPreventsDestruction = 1 | OMGVMCID,
    MinorIndestructibleObject = 2 | OMGVMCID,
    MinorDestroyWouldBlock = 3 | OMGVMCID,
    MinorShutdownCalled = 4 | OMGVMCID,
    MinorDuplicateSend = 5 | OMGVMCID,
    MinorServantManagerAlreadySet = 6 | OMGVMCID,
    MinorInvalidUseOfDSIArguments = 7 | OMGVMCID,
    MinorInvalidUseOfDSIContext = 8 | OMGVMCID,
    MinorInvalidUseOfDSIResult = 9 | OMGVMCID,
    MinorRequestAlreadySent = 10 | OMGVMCID,
    MinorRequestNotSent = 11 | OMGVMCID,
    MinorResponseAlreadyReceived = 12 | OMGVMCID,
    MinorSynchronousRequest = 13 | OMGVMCID,
    MinorInvalidPICall = 14 | OMGVMCID,
    MinorServiceContextExists = 15 | OMGVMCID,
    MinorPolicyFactoryExists = 16 | OMGVMCID,
    MinorNoCreatePOA = 17 | OMGVMCID,

    //
    // ORBacus specific minor codes
    //
    MinorBadConcModel = 1 | OOCVMCID,
    MinorORBRunning = 2 | OOCVMCID
};

//
// Minor error codes for COMM_FAILURE
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorCommFailure
{
    //
    // Standard minor codes
    //
    /* None yet */

    //
    // ORBacus specific minor codes
    //
    MinorRecv = 1 | OOCVMCID,
    MinorSend = 2 | OOCVMCID,
    MinorRecvZero = 3 | OOCVMCID,
    MinorSendZero = 4 | OOCVMCID,
    MinorSocket = 5 | OOCVMCID,
    MinorSetsockopt = 6 | OOCVMCID,
    MinorGetsockopt = 7 | OOCVMCID,
    MinorBind = 8 | OOCVMCID,
    MinorListen = 9 | OOCVMCID,
    MinorConnect = 10 | OOCVMCID,
    MinorAccept = 11 | OOCVMCID,
    MinorSelect = 12 | OOCVMCID,
    MinorSetSoTimeout = 26 | OOCVMCID,
    MinorGetsockname = 27 | OOCVMCID,
    MinorGetpeername = 28 | OOCVMCID,

    MinorGethostname = 13 | OOCVMCID,
    MinorGethostbyname = 14 | OOCVMCID,

    MinorWSAStartup = 15 | OOCVMCID,
    MinorWSACleanup = 16 | OOCVMCID,

    MinorNoGIOP = 17 | OOCVMCID,
    MinorUnknownMessage = 18 | OOCVMCID,
    MinorWrongMessage = 19 | OOCVMCID,
    MinorMessageError = 21 | OOCVMCID,
    MinorFragment = 22 | OOCVMCID,
    MinorUnknownReqId = 24 | OOCVMCID,
    MinorVersion = 25 | OOCVMCID,

    MinorPipe = 23 | OOCVMCID,

    //
    // ORBacus deprecated minor codes
    //
    MinorDeprecatedCloseConnection = 20 | OOCVMCID
};

//
// Minor error codes for INTF_REPOS
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorIntfRepos
{
    //
    // Standard minor codes
    //
    /* None yet */

    //
    // ORBacus specific minor codes
    //
    MinorNoIntfRepos = 1 | OOCVMCID,
    MinorLookupAmbiguous = 2 | OOCVMCID,
    MinorIllegalRecursion = 3 | OOCVMCID,
    MinorNoEntry = 4 | OOCVMCID
};

//
// Minor error codes for MARSHAL
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorMarshal
{
    //
    // Standard minor codes
    //
    MinorNoValueFactory = 1 | OMGVMCID,
    MinorDSIResultBeforeContext = 2 | OMGVMCID,
    MinorDSIInvalidParameterList = 3 | OMGVMCID,
    MinorLocalObject = 4 | OMGVMCID,
    MinorWcharSentByClient = 5 | OMGVMCID,
    MinorWcharSentByServer = 6 | OMGVMCID,

    //
    // ORBacus specific minor codes
    //
    MinorReadOverflow = 1 | OOCVMCID,
    MinorReadBooleanOverflow = 2 | OOCVMCID,
    MinorReadCharOverflow = 3 | OOCVMCID,
    MinorReadWCharOverflow = 4 | OOCVMCID,
    MinorReadOctetOverflow = 5 | OOCVMCID,
    MinorReadShortOverflow = 6 | OOCVMCID,
    MinorReadUShortOverflow = 7 | OOCVMCID,
    MinorReadLongOverflow = 8 | OOCVMCID,
    MinorReadULongOverflow = 9 | OOCVMCID,
    MinorReadLongLongOverflow = 10 | OOCVMCID,
    MinorReadULongLongOverflow = 11 | OOCVMCID,
    MinorReadFloatOverflow = 12 | OOCVMCID,
    MinorReadDoubleOverflow = 13 | OOCVMCID,
    MinorReadLongDoubleOverflow = 14 | OOCVMCID,
    MinorReadStringOverflow = 15 | OOCVMCID,
    MinorReadStringZeroLength = 16 | OOCVMCID,
    MinorReadStringNullChar = 17 | OOCVMCID,
    MinorReadStringNoTerminator = 18 | OOCVMCID,
    MinorReadWStringOverflow = 19 | OOCVMCID,
    MinorReadWStringZeroLength = 20 | OOCVMCID,
    MinorReadWStringNullWChar = 21 | OOCVMCID,
    MinorReadWStringNoTerminator = 22 | OOCVMCID,
    MinorReadFixedOverflow = 23 | OOCVMCID,
    MinorReadFixedInvalid = 24 | OOCVMCID,
    MinorReadBooleanArrayOverflow = 25 | OOCVMCID,
    MinorReadCharArrayOverflow = 26 | OOCVMCID,
    MinorReadWCharArrayOverflow = 27 | OOCVMCID,
    MinorReadOctetArrayOverflow = 28 | OOCVMCID,
    MinorReadShortArrayOverflow = 29 | OOCVMCID,
    MinorReadUShortArrayOverflow = 30 | OOCVMCID,
    MinorReadLongArrayOverflow = 31 | OOCVMCID,
    MinorReadULongArrayOverflow = 32 | OOCVMCID,
    MinorReadLongLongArrayOverflow = 33 | OOCVMCID,
    MinorReadULongLongArrayOverflow = 34 | OOCVMCID,
    MinorReadFloatArrayOverflow = 35 | OOCVMCID,
    MinorReadDoubleArrayOverflow = 36 | OOCVMCID,
    MinorReadLongDoubleArrayOverflow = 37 | OOCVMCID,
    MinorReadInvTypeCodeIndirection = 38 | OOCVMCID,
    MinorLongDoubleNotSupported = 40 | OOCVMCID,

    //
    // ORBacus deprecated minor codes
    //
    MinorDeprecatedWriteObjectLocal = 39 | OOCVMCID // Use MinorLocalObject
};

//
// Minor error codes for IMP_LIMIT
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorImpLimit
{
    //
    // Standard minor codes
    //
    MinorNoUsableProfile = 1 | OMGVMCID,

    //
    // ORBacus specific minor codes
    //
    MinorMessageSizeLimit = 1 | OOCVMCID,
    MinorThreadLimit = 2 | OOCVMCID
};

//
// Minor error codes for NO_MEMORY
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorNoMemory
{
    //
    // ORBacus specific minor codes
    //
    MinorAllocationFailure = 1 | OOCVMCID
};

//
// Minor error codes for TRANSIENT
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorTransient
{
    //
    // Standard minor codes
    //
    MinorRequestDiscarded = 1 | OMGVMCID,
    MinorNoUsableProfileInIOR = 2 | OMGVMCID,
    MinorRequestCancelled = 3 | OMGVMCID,
    MinorPOADestroyed = 4 | OMGVMCID,

    //
    // ORBacus specific minor codes
    //
    MinorConnectFailed = 1 | OOCVMCID,
    MinorCloseConnection = 2 | OOCVMCID,
    MinorActiveConnectionManagement = 3 | OOCVMCID,
    MinorForcedShutdown = 4 | OOCVMCID,
    MinorLocationForwardHopCountExceeded = 5 | OOCVMCID
};

//
// Minor error codes for NO_RESOURCES
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorNoResources
{
    //
    // PortableInterceptor minor codes
    //
    MinorInvalidBinding = 1 | OMGVMCID
};

//
// Minor error codes for UNKNOWN
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorUnknown
{
    //
    // Standard minor codes
    //
    MinorUnknownUserException = 1 | OMGVMCID,
    MinorSystemExceptionNotSupported = 2 | OMGVMCID
};

//
// Minor error codes for INV_POLICY
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorInvPolicy
{
    //
    // Standard minor codes
    //
    MinorCannotReconcilePolicy = 1 | OMGVMCID,
    MinorInvalidPolicyType = 2 | OMGVMCID,
    MinorNoPolicyFactory = 3 | OMGVMCID
};

//
// Minor error codes for INV_OBJREF
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorInvObjref
{
    //
    // Standard minor codes
    //
    MinorNoWcharSupport = 1 | OMGVMCID,
    MinorWcharCodeSetRequired = 2 | OMGVMCID
};

//
// Minor error codes for BAD_TYPECODE
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorBadTypecode
{
    //
    // Standard minor codes
    //
    MinorIncompleteTypeCode = 1 | OMGVMCID,
    MinorInvalidMemberType = 2 | OMGVMCID,

    //
    // ORBacus specific minor codes
    //
    MinorInvalidUnionDiscriminator = 1 | OOCVMCID
};

//
// Minor error codes for OBJ_ADAPTER
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorObjAdapter
{
    //
    // Standard minor codes
    //
    MinorSystemExceptionInUnknownAdapter = 1 | OMGVMCID,
    MinorServantNotFound = 2 | OMGVMCID,
    MinorNoDefaultServant = 3 | OMGVMCID,
    MinorNoServantManager = 4 | OMGVMCID,
    MinorIncarnateViolatedPOAPolicy = 5 | OMGVMCID
};

//
// Minor error codes for DATA_CONVERSION
//
// Note: If something is changed here, it must also be changed in
// other ORBacus implementations, like ORBacus for Java. Furthermore,
// the corresponding _to_string() operation must be updated.
//
enum MinorDataConversion
{
    //
    // Standard minor codes
    //
    MinorNoCharacterMapping = 1 | OMGVMCID
};

} // End of namespace OB

#endif
