/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.BatchStatement;
import com.datastax.driver.core.BoundStatement;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.ColumnDefinitions;
import com.datastax.driver.core.DataType;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.LatencyTracker;
import com.datastax.driver.core.PerHostPercentileTracker;
import com.datastax.driver.core.ProtocolVersion;
import com.datastax.driver.core.RegularStatement;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.StatementWrapper;
import com.google.common.annotations.Beta;
import com.google.common.annotations.VisibleForTesting;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class QueryLogger
implements LatencyTracker {
    public static final long DEFAULT_SLOW_QUERY_THRESHOLD_MS = 5000L;
    public static final double DEFAULT_SLOW_QUERY_THRESHOLD_PERCENTILE = 99.0;
    public static final int DEFAULT_MAX_QUERY_STRING_LENGTH = 500;
    public static final int DEFAULT_MAX_PARAMETER_VALUE_LENGTH = 50;
    public static final int DEFAULT_MAX_LOGGED_PARAMETERS = 50;
    public static final Logger NORMAL_LOGGER = LoggerFactory.getLogger("com.datastax.driver.core.QueryLogger.NORMAL");
    public static final Logger SLOW_LOGGER = LoggerFactory.getLogger("com.datastax.driver.core.QueryLogger.SLOW");
    public static final Logger ERROR_LOGGER = LoggerFactory.getLogger("com.datastax.driver.core.QueryLogger.ERROR");
    private static final String NORMAL_TEMPLATE = "[%s] [%s] Query completed normally, took %s ms: %s";
    private static final String SLOW_TEMPLATE_MILLIS = "[%s] [%s] Query too slow, took %s ms: %s";
    private static final String SLOW_TEMPLATE_PERCENTILE = "[%s] [%s] Query too slow, took %s ms (%s percentile = %s ms): %s";
    private static final String ERROR_TEMPLATE = "[%s] [%s] Query error after %s ms: %s";
    @VisibleForTesting
    static final String TRUNCATED_OUTPUT = "... [truncated output]";
    @VisibleForTesting
    static final String FURTHER_PARAMS_OMITTED = " [further parameters omitted]";
    protected final Cluster cluster;
    private volatile ProtocolVersion protocolVersion;
    protected volatile int maxQueryStringLength;
    protected volatile int maxParameterValueLength;
    protected volatile int maxLoggedParameters;

    private QueryLogger(Cluster cluster, int maxQueryStringLength, int maxParameterValueLength, int maxLoggedParameters) {
        this.cluster = cluster;
        this.maxQueryStringLength = maxQueryStringLength;
        this.maxParameterValueLength = maxParameterValueLength;
        this.maxLoggedParameters = maxLoggedParameters;
    }

    public static Builder builder(Cluster cluster) {
        if (cluster == null) {
            throw new NullPointerException("QueryLogger.Builder: cluster parameter cannot be null");
        }
        return new Builder(cluster);
    }

    public int getMaxQueryStringLength() {
        return this.maxQueryStringLength;
    }

    public void setMaxQueryStringLength(int maxQueryStringLength) {
        if (maxQueryStringLength <= 0 && maxQueryStringLength != -1) {
            throw new IllegalArgumentException("Invalid maxQueryStringLength, should be > 0 or -1, got " + maxQueryStringLength);
        }
        this.maxQueryStringLength = maxQueryStringLength;
    }

    public int getMaxParameterValueLength() {
        return this.maxParameterValueLength;
    }

    public void setMaxParameterValueLength(int maxParameterValueLength) {
        if (maxParameterValueLength <= 0 && maxParameterValueLength != -1) {
            throw new IllegalArgumentException("Invalid maxParameterValueLength, should be > 0 or -1, got " + maxParameterValueLength);
        }
        this.maxParameterValueLength = maxParameterValueLength;
    }

    public int getMaxLoggedParameters() {
        return this.maxLoggedParameters;
    }

    public void setMaxLoggedParameters(int maxLoggedParameters) {
        if (maxLoggedParameters <= 0 && maxLoggedParameters != -1) {
            throw new IllegalArgumentException("Invalid maxLoggedParameters, should be > 0 or -1, got " + maxLoggedParameters);
        }
        this.maxLoggedParameters = maxLoggedParameters;
    }

    @Override
    public void update(Host host, Statement statement, Exception exception, long newLatencyNanos) {
        long latencyMs = TimeUnit.NANOSECONDS.toMillis(newLatencyNanos);
        if (exception == null) {
            this.maybeLogNormalOrSlowQuery(host, statement, latencyMs);
        } else {
            this.maybeLogErrorQuery(host, statement, exception, latencyMs);
        }
    }

    protected abstract void maybeLogNormalOrSlowQuery(Host var1, Statement var2, long var3);

    protected void maybeLogNormalQuery(Host host, Statement statement, long latencyMs) {
        if (NORMAL_LOGGER.isDebugEnabled()) {
            String message = String.format(NORMAL_TEMPLATE, this.cluster.getClusterName(), host, latencyMs, this.statementAsString(statement));
            this.logQuery(statement, null, NORMAL_LOGGER, message);
        }
    }

    protected void maybeLogErrorQuery(Host host, Statement statement, Exception exception, long latencyMs) {
        if (ERROR_LOGGER.isDebugEnabled()) {
            String message = String.format(ERROR_TEMPLATE, this.cluster.getClusterName(), host, latencyMs, this.statementAsString(statement));
            this.logQuery(statement, exception, ERROR_LOGGER, message);
        }
    }

    protected void logQuery(Statement statement, Exception exception, Logger logger, String message) {
        boolean showParameterValues = logger.isTraceEnabled();
        if (showParameterValues) {
            StringBuilder params = new StringBuilder();
            if (statement instanceof BoundStatement) {
                this.appendParameters((BoundStatement)statement, params, this.maxLoggedParameters);
            } else if (statement instanceof BatchStatement) {
                BatchStatement batchStatement = (BatchStatement)statement;
                int remaining = this.maxLoggedParameters;
                for (Statement inner : batchStatement.getStatements()) {
                    if (!(inner instanceof BoundStatement)) continue;
                    remaining = this.appendParameters((BoundStatement)inner, params, remaining);
                }
            }
            if (params.length() > 0) {
                params.append("]");
            }
            logger.trace(message + params, exception);
        } else {
            logger.debug(message, exception);
        }
    }

    protected String statementAsString(Statement statement) {
        StringBuilder sb = new StringBuilder();
        if (statement instanceof BatchStatement) {
            BatchStatement bs = (BatchStatement)statement;
            int statements = bs.getStatements().size();
            int boundValues = this.countBoundValues(bs);
            sb.append("[" + statements + " statements, " + boundValues + " bound values] ");
        } else if (statement instanceof BoundStatement) {
            int boundValues = ((BoundStatement)statement).wrapper.values.length;
            sb.append("[" + boundValues + " bound values] ");
        }
        this.append(statement, sb, this.maxQueryStringLength);
        return sb.toString();
    }

    protected int countBoundValues(BatchStatement bs) {
        int count = 0;
        for (Statement s : bs.getStatements()) {
            if (!(s instanceof BoundStatement)) continue;
            count += ((BoundStatement)s).wrapper.values.length;
        }
        return count;
    }

    protected int appendParameters(BoundStatement statement, StringBuilder buffer, int remaining) {
        if (remaining == 0) {
            return 0;
        }
        ColumnDefinitions metadata = statement.preparedStatement().getVariables();
        int numberOfParameters = metadata.size();
        if (numberOfParameters > 0) {
            int numberOfLoggedParameters;
            List<ColumnDefinitions.Definition> definitions = metadata.asList();
            if (remaining == -1) {
                numberOfLoggedParameters = numberOfParameters;
            } else {
                numberOfLoggedParameters = remaining > numberOfParameters ? numberOfParameters : remaining;
                remaining -= numberOfLoggedParameters;
            }
            for (int i = 0; i < numberOfLoggedParameters; ++i) {
                if (buffer.length() == 0) {
                    buffer.append(" [");
                } else {
                    buffer.append(", ");
                }
                buffer.append(String.format("%s:%s", metadata.getName(i), this.parameterValueAsString(definitions.get(i), statement.wrapper.values[i])));
            }
            if (numberOfLoggedParameters < numberOfParameters) {
                buffer.append(FURTHER_PARAMS_OMITTED);
            }
        }
        return remaining;
    }

    protected String parameterValueAsString(ColumnDefinitions.Definition definition, ByteBuffer raw) {
        String valueStr;
        if (raw == null || raw.remaining() == 0) {
            valueStr = "NULL";
        } else {
            DataType type = definition.getType();
            int maxParameterValueLength = this.maxParameterValueLength;
            if (type.equals(DataType.blob()) && maxParameterValueLength != -1) {
                boolean bufferTooLarge;
                int maxBufferLength = Math.max(2, (maxParameterValueLength - 2) / 2);
                boolean bl = bufferTooLarge = raw.remaining() > maxBufferLength;
                if (bufferTooLarge) {
                    raw = (ByteBuffer)raw.duplicate().limit(maxBufferLength);
                }
                Object value = type.deserialize(raw, this.protocolVersion());
                valueStr = type.format(value);
                if (bufferTooLarge) {
                    valueStr = valueStr + TRUNCATED_OUTPUT;
                }
            } else {
                Object value = type.deserialize(raw, this.protocolVersion());
                valueStr = type.format(value);
                if (maxParameterValueLength != -1 && valueStr.length() > maxParameterValueLength) {
                    valueStr = valueStr.substring(0, maxParameterValueLength) + TRUNCATED_OUTPUT;
                }
            }
        }
        return valueStr;
    }

    private ProtocolVersion protocolVersion() {
        if (this.protocolVersion == null) {
            this.protocolVersion = this.cluster.getConfiguration().getProtocolOptions().getProtocolVersionEnum();
            assert (this.protocolVersion != null) : "protocol version should be defined";
        }
        return this.protocolVersion;
    }

    protected int append(Statement statement, StringBuilder buffer, int remaining) {
        if (statement instanceof StatementWrapper) {
            statement = ((StatementWrapper)statement).getWrappedStatement();
        }
        if (statement instanceof RegularStatement) {
            remaining = this.append(((RegularStatement)statement).getQueryString().trim(), buffer, remaining);
        } else if (statement instanceof BoundStatement) {
            remaining = this.append(((BoundStatement)statement).preparedStatement().getQueryString().trim(), buffer, remaining);
        } else if (statement instanceof BatchStatement) {
            BatchStatement batchStatement = (BatchStatement)statement;
            remaining = this.append("BEGIN", buffer, remaining);
            switch (batchStatement.batchType) {
                case UNLOGGED: {
                    this.append(" UNLOGGED", buffer, remaining);
                    break;
                }
                case COUNTER: {
                    this.append(" COUNTER", buffer, remaining);
                }
            }
            remaining = this.append(" BATCH", buffer, remaining);
            for (Statement stmt : batchStatement.getStatements()) {
                remaining = this.append(" ", buffer, remaining);
                remaining = this.append(stmt, buffer, remaining);
            }
            remaining = this.append(" APPLY BATCH", buffer, remaining);
        } else {
            remaining = this.append(statement.toString(), buffer, remaining);
        }
        if (buffer.charAt(buffer.length() - 1) != ';') {
            remaining = this.append(";", buffer, remaining);
        }
        return remaining;
    }

    protected int append(CharSequence str, StringBuilder buffer, int remaining) {
        if (remaining != -2) {
            if (remaining == -1) {
                buffer.append(str);
            } else if (str.length() > remaining) {
                buffer.append(str, 0, remaining).append(TRUNCATED_OUTPUT);
                remaining = -2;
            } else {
                buffer.append(str);
                remaining -= str.length();
            }
        }
        return remaining;
    }

    public static class Builder {
        private final Cluster cluster;
        private int maxQueryStringLength = 500;
        private int maxParameterValueLength = 50;
        private int maxLoggedParameters = 50;
        private long slowQueryLatencyThresholdMillis = 5000L;
        private double slowQueryLatencyThresholdPercentile = 99.0;
        private PerHostPercentileTracker perHostPercentileLatencyTracker;
        private boolean constantThreshold = true;

        public Builder(Cluster cluster) {
            this.cluster = cluster;
        }

        public Builder withConstantThreshold(long slowQueryLatencyThresholdMillis) {
            this.slowQueryLatencyThresholdMillis = slowQueryLatencyThresholdMillis;
            this.constantThreshold = true;
            return this;
        }

        @Beta
        public Builder withDynamicThreshold(PerHostPercentileTracker perHostPercentileLatencyTracker, double slowQueryLatencyThresholdPercentile) {
            this.perHostPercentileLatencyTracker = perHostPercentileLatencyTracker;
            this.slowQueryLatencyThresholdPercentile = slowQueryLatencyThresholdPercentile;
            this.constantThreshold = false;
            return this;
        }

        public Builder withMaxQueryStringLength(int maxQueryStringLength) {
            this.maxQueryStringLength = maxQueryStringLength;
            return this;
        }

        public Builder withMaxParameterValueLength(int maxParameterValueLength) {
            this.maxParameterValueLength = maxParameterValueLength;
            return this;
        }

        public Builder withMaxLoggedParameters(int maxLoggedParameters) {
            this.maxLoggedParameters = maxLoggedParameters;
            return this;
        }

        public QueryLogger build() {
            if (this.constantThreshold) {
                return new ConstantThresholdQueryLogger(this.cluster, this.maxQueryStringLength, this.maxParameterValueLength, this.maxLoggedParameters, this.slowQueryLatencyThresholdMillis);
            }
            return new DynamicThresholdQueryLogger(this.cluster, this.maxQueryStringLength, this.maxParameterValueLength, this.maxLoggedParameters, this.slowQueryLatencyThresholdPercentile, this.perHostPercentileLatencyTracker);
        }
    }

    @Beta
    public static class DynamicThresholdQueryLogger
    extends QueryLogger {
        private volatile double slowQueryLatencyThresholdPercentile;
        private volatile PerHostPercentileTracker perHostPercentileLatencyTracker;

        private DynamicThresholdQueryLogger(Cluster cluster, int maxQueryStringLength, int maxParameterValueLength, int maxLoggedParameters, double slowQueryLatencyThresholdPercentile, PerHostPercentileTracker perHostPercentileLatencyTracker) {
            super(cluster, maxQueryStringLength, maxParameterValueLength, maxLoggedParameters);
            this.setSlowQueryLatencyThresholdPercentile(slowQueryLatencyThresholdPercentile);
            this.setPerHostPercentileLatencyTracker(perHostPercentileLatencyTracker);
        }

        public PerHostPercentileTracker getPerHostPercentileLatencyTracker() {
            return this.perHostPercentileLatencyTracker;
        }

        public void setPerHostPercentileLatencyTracker(PerHostPercentileTracker perHostPercentileLatencyTracker) {
            if (perHostPercentileLatencyTracker == null) {
                throw new IllegalArgumentException("perHostPercentileLatencyTracker cannot be null");
            }
            this.perHostPercentileLatencyTracker = perHostPercentileLatencyTracker;
        }

        public double getSlowQueryLatencyThresholdPercentile() {
            return this.slowQueryLatencyThresholdPercentile;
        }

        public void setSlowQueryLatencyThresholdPercentile(double slowQueryLatencyThresholdPercentile) {
            if (slowQueryLatencyThresholdPercentile < 0.0 || slowQueryLatencyThresholdPercentile >= 100.0) {
                throw new IllegalArgumentException("Invalid slowQueryLatencyThresholdPercentile, should be >= 0 and < 100, got " + slowQueryLatencyThresholdPercentile);
            }
            this.slowQueryLatencyThresholdPercentile = slowQueryLatencyThresholdPercentile;
        }

        @Override
        protected void maybeLogNormalOrSlowQuery(Host host, Statement statement, long latencyMs) {
            long threshold = this.perHostPercentileLatencyTracker.getLatencyAtPercentile(host, this.slowQueryLatencyThresholdPercentile);
            if (threshold >= 0L && latencyMs > threshold) {
                this.maybeLogSlowQuery(host, statement, latencyMs, threshold);
            } else {
                this.maybeLogNormalQuery(host, statement, latencyMs);
            }
        }

        protected void maybeLogSlowQuery(Host host, Statement statement, long latencyMs, long threshold) {
            if (SLOW_LOGGER.isDebugEnabled()) {
                String message = String.format(QueryLogger.SLOW_TEMPLATE_PERCENTILE, this.cluster.getClusterName(), host, latencyMs, this.slowQueryLatencyThresholdPercentile, threshold, this.statementAsString(statement));
                this.logQuery(statement, null, SLOW_LOGGER, message);
            }
        }
    }

    public static class ConstantThresholdQueryLogger
    extends QueryLogger {
        private volatile long slowQueryLatencyThresholdMillis;

        private ConstantThresholdQueryLogger(Cluster cluster, int maxQueryStringLength, int maxParameterValueLength, int maxLoggedParameters, long slowQueryLatencyThresholdMillis) {
            super(cluster, maxQueryStringLength, maxParameterValueLength, maxLoggedParameters);
            this.setSlowQueryLatencyThresholdMillis(slowQueryLatencyThresholdMillis);
        }

        public long getSlowQueryLatencyThresholdMillis() {
            return this.slowQueryLatencyThresholdMillis;
        }

        public void setSlowQueryLatencyThresholdMillis(long slowQueryLatencyThresholdMillis) {
            if (slowQueryLatencyThresholdMillis <= 0L) {
                throw new IllegalArgumentException("Invalid slowQueryLatencyThresholdMillis, should be > 0, got " + slowQueryLatencyThresholdMillis);
            }
            this.slowQueryLatencyThresholdMillis = slowQueryLatencyThresholdMillis;
        }

        @Override
        protected void maybeLogNormalOrSlowQuery(Host host, Statement statement, long latencyMs) {
            if (latencyMs > this.slowQueryLatencyThresholdMillis) {
                this.maybeLogSlowQuery(host, statement, latencyMs);
            } else {
                this.maybeLogNormalQuery(host, statement, latencyMs);
            }
        }

        protected void maybeLogSlowQuery(Host host, Statement statement, long latencyMs) {
            if (SLOW_LOGGER.isDebugEnabled()) {
                String message = String.format(QueryLogger.SLOW_TEMPLATE_MILLIS, this.cluster.getClusterName(), host, latencyMs, this.statementAsString(statement));
                this.logQuery(statement, null, SLOW_LOGGER, message);
            }
        }
    }
}

