#!/usr/bin/env python
"""tangodatabase.py - access to TANGO database for Tango-Python modules.

Blabla:
         author : Laurent Pointal (laurent.pointal@lure.u-psud.fr)
       creation : 8 april 2002
          modif : 11 april 2002

History:
8 april 2002 - Laurent Pointal - creation of the module.

TODO: 
* Check DbClass methods to have similar interface as DbDevice.
* Cleanup DbDevice/DbClass from potentially dangerous methods for clients.
* Test all DbClass/DbDevice methods.
"""

#===============================================================================
# Imports and exports.
__all__ = [ "Database","DbDevice","DbClass","DbServInfo","DbDevInfo",
            "DbDevImportInfo","DbDevExportInfo" ]
import types
from pyTango import tangocorba


#===============================================================================
# Globals:
# Connection to the database.
# Note: TangoName add missing host and port itself from TANGO_HOST.
gdbcon = tangocorba.DeviceConnection ("corbaloc:/database")
# To debug this module.
DEBUG = 0



#===============================================================================
class Database :
    #---------------------------------------------------------------------------
    def __init__ (self,database=None) :
        global gdbcon

        if database == None :
            self.db_con = gdbcon
        elif type(database)==types.StringType :
            self.db_con = tangocorba.DeviceConnection (database)
        else :
            self.db_con = database

    #---------------------------------------------------------------------------
    def dbstate (self) :
        """Query the database for its Tango Device.state CORBA attribute.

        Return a Tango.DevState value.
        """
        return self.db_con.state ()

    #---------------------------------------------------------------------------
    def dbstatus (self) :
        """Query the database for its Tango Device.status CORBA attribute.

        Return a string which represent the database status.
        """
        return self.db_con.status ()

    #---------------------------------------------------------------------------
    def get_info (self) :
        """Query the database for some general info about its tables.

        Result is returned as a string.
        """
        return self.db_con.command_inout ("DbInfo")
    dbgetinfo = get_info

    #---------------------------------------------------------------------------
    def add_device (self,devinf) :
        """Add a device to the database.

        The device name, server and class are given in a DbDevInfo object.

        Example:
            db.add_device (DbDevInfo ("my/own/device","MyDevice",
                                        "MyServer/test"))
        Or
            infos = DbDevInfo()
            infos.name = "my/own/device"
            infos._class = "MyDevice"
            infos.server = "MyServer/test"
            db.add_device (infos)
        """
        n,c,s = devinf.name,devinf._class,devinf.server
        self.db_con.command_inout ("DbAddDevice",[n,c,s])
    dbadddev = add_device

    #---------------------------------------------------------------------------
    def delete_device (self,name) :
        """Delete the device of the specified name from the database.

        Example:
            db.delete_device ("my/own/device")
        """
        self.db_con.command_inout ("DbDeleteDevice",name)
    dbdeldev = delete_device

    #---------------------------------------------------------------------------
    def import_device (self,name) :
        """Query the database for the export info of the specified device.

        The command returns the information in a DbDevImportInfo structure.

        Example:
            r = db.import_device ("my/own/device")
            print "device:",r.name
            print "exported:",r.exported
            print "ior:",r.ior
            print "version:",r.version
        """
        res = self.db_con.command_inout ("DbImportDevice",name)
        infos = DbDevImportInfo ()
        infos.name     = res.svalue[0]
        infos.ior      = res.svalue[1]
        infos.version  = res.svalue[2]
        infos.exported = res.lvalue[0]
        return infos
    dbimpdev = import_device

    #---------------------------------------------------------------------------
    def export_device (self,devexpinf) :
        """Update the export info for this device in the database.

        Device name, server, class, pid and version are specified in the
        DbDevExportInfo structure.

        Example:
            db.export_device (DbDevExportInfo("my/own/device","ior:...",
                                            "dumela","1.0",get_pid()))
        Or:
            infos = DbDevExportInfo ()
            infos.name = "my/own/device"
            infos.ior = "ior:..."
            infos.host = "dumela"
            infos.version = "1.0"
            infos.pid = get_pid()
            db.export_device (infos)
        """
        n,i,h,p,v = devexpinf.name,devexpinf.ior,devexpinf.host,devexpinf.pid,\
                                                            devexpinf.version
        self.db_con.command_inout ("DbExportDevice",[n,i,h,p,v])
    dbexpdev = export_device

    #---------------------------------------------------------------------------
    def unexport_device (self,name) :
        """Mark the specified device as unexported in the database.

        Example:
            db.unexport_device ("my/own/device")
        """
        self.db_con.command_inout ("DbUnExportDevice",name)
    dbunexpdev = unexport_device

    #---------------------------------------------------------------------------
    def list_host (self,wildcard="*") :
        """Query the database for a list of host names.

        Query the database for a list of host names that match the wildcard. 
        Wildcard character is * and matches any number of characters.
        The method returns a list of names.
        
        Example:
            hosts = db.list_host ()
        Return:
            hosts -> [ "id112", "corvus", "dumela" ]
        """
        return self.db_con.command_inout ("DbGetHostList",wildcard)
    dblisthost = list_host

    #---------------------------------------------------------------------------
    def list_host_server (self,hostname) :
        """Query the database for a list of servers running on a specified host.

        The method returns a list of names.

        Example:
            servs = db.list_host_server ("dumela")
        Return:
            servs -> [ "fluids/opc" ]
        """
        return self.db_con.command_inout ("DbGetHostServerList",hostname)
    dblisthostserv = list_host

    #---------------------------------------------------------------------------
    def get_server_info (self,servname) :
        """Query the database for the informations about the specified server.

        The method returns the information in a DbServInfo structure.

        Example:
            r = db.get_server_info ("steppermotor/id11")
            print "host:",r.host
            print "mode:",r.mode
            print "level:",r.level
        """
        res = self.db_con.command_inout ("DbGetServerInfo",servname)
        final = DbServInfo (res[0],res[1],res[2],res[3])
        return final
    dbgetservinfo = get_server_info

    #---------------------------------------------------------------------------
    def put_server_info (self,servinf) :
        """Put informations about the server in the database.

        The method takes the informations from a DbServInfo structure.

        Example:
            infos  = DbServInfo ()
            infos.name = "steppermotor/id11"
            infos.host = "dumela"
            infos.mode = "1"
            infos.level = "99"
            db.put_server_info (infos)
        """
        n,h,m,l = servinf.name,servinf.host,servinf.mode,servinf.level
        self.db_con.command_inout ("DbPutServerInfo",[n,h,m,l])
    dbputservinfo = put_server_info

    #---------------------------------------------------------------------------
    def add_server (self,devinflist) :
        """Add a group of devices to the database.

        the device names, server names and classes are specified in the list of
        DbDevInfo structures.
        """
        if isinstance(devinflist,DbDevInfo) : devinflist = [ devinflist ]

        server = devinflist[0].server
        for di in devinflist :
            if di.server != server :
                raise tangocorba.tango_error \
                    ("Bad parameter for Database.add_server.",
                    tangocorba.ERR,
                    "All DbDevInfo server names in the list must be the same")

        fullist = [ server ]
        for devinf in devinflist :
            n,c = devinf.name,devinf._class
            fullist += [n,c]

        self.db_con.command_inout ("DbAddServer",fullist)
    dbaddserv = add_server

    #---------------------------------------------------------------------------
    def delete_server (self,name) :
        """Delete the device server and associated devices from the database.
        """
        self.db_con.command_inout ("DbDeleteServer",name)
    dbdelserv = delete_server

    #---------------------------------------------------------------------------
    def delete_server_info (self,name) :
        """Delete the device server informations from the database.
        
        Note: the host, mode, level server informations are cleaned (ie.
        get_server_info return empty string for these informations).
        """
        self.db_con.command_inout ("DbDeleteServerInfo",name)
    dbdelservinfo = delete_server_info
    
    #---------------------------------------------------------------------------
    def export_server (self,devexpinflist) :
        """Export a group of devices to the database.

        The device names, IOR, class, server name, pid etc. are specified in the
        list of DbDevExportInfo structures.
        """
        if isinstance(devexpinflist,DbDevExportInfo) : 
            devexpinflist = [ devexpinflist ]
        for devexpinf in devexpinflist : self.export_device (devexpinf)
    dbexpserv = export_server

    #---------------------------------------------------------------------------
    def unexport_server (self,name) :
        """Mark all devices exported for this server as unexported."""
        self.db_con.command_inout ("DbUnExportServer",name)
    dbunexpserv = unexport_server

    #---------------------------------------------------------------------------
    def get_device_name (self,servname,classname) :
        """Query the database for a list of devices.

        Query the database for a list of devices served by the specified server
        (1st parameter) and of the specified class (2nd parameter). The method
        return a list of names.
        """
        return self.db_con.command_inout("DbGetDeviceList",[servname,classname])
    dblistdev = get_device_name

    #---------------------------------------------------------------------------
    def get_device_alias (self,name) :
        """Query the database for a list of aliases for the specified device.

        This method returns a list of aliases.
        """
        return self.db_con.command_inout ("DbGetDeviceAliasList",name)
    dblistdevalias = get_device_alias

    #---------------------------------------------------------------------------
    def get_device_exported (self,wildcard="*") :
        """Query the database for a list of exported devices.

        Query the database for a list of exported devices whose names satisfy 
        the supplied filter (* is wildcard for any character(s)). This method 
        returns a list of names.
        """
        return self.db_con.command_inout ("DbGetDeviceExportedList",wildcard)
    dblistdevexp = get_device_exported

    #---------------------------------------------------------------------------
    def get_device_domain (self,wildcard="*") :
        """Query the database for a list of device domain names.

        Query the database for a list of device domain names which match the
        wildcard provided. Wildcard character is * and match any number of
        characters. Domain names are case insensitive. this method returns a 
        list of domain names.
        """
        return self.db_con.command_inout ("DbGetDeviceDomainList",wildcard)
    dblistdevdom = get_device_domain

    #---------------------------------------------------------------------------
    def get_device_family (self,wildcard="*") :
        """Query the database for a list of device family names.

        Query the database for a list of device family names which match the
        wildcard provided. Wildcard character is * and matches any number of 
        characters.
        Family names are case insensitive. this method returns a list of family
        names.
        """
        return self.db_con.command_inout ("DbGetDeviceFamilyList",wildcard)
    dblistdevfam = get_device_family

    #---------------------------------------------------------------------------
    def get_device_member (self,wildcard="*") :
        """Query the database for a list of device member names.

        Query the database for a list of device member names which match the
        wildcard provided. Wildcard character is * and matches any number of
        characters. Member names are case insensitive. This method returns a list o
        member names.
        """
        return self.db_con.command_inout ("DbGetDeviceMemberList",wildcard)
    dbdevmemblist = get_device_member
    
    #---------------------------------------------------------------------------
    def list_property (self,wildcard="*") :
        """Query the database for a list of object properties names.

        Query the database for a list of object (ie. non-device) properties
        that match the wildcard. Wildcard character is * and matches any number 
        of characters. The method returns a list of names.
        """
        return self.db_con.command_inout ("DbGetPropertyList",wildcard)
    dblistprop = list_property

    #---------------------------------------------------------------------------
    def get_property (self,name,proplist,*otherprop) :
        """Query the database for a list of objects (ie. non-device) properties.

        Query the database for a list of objects (ie. non-device) properties for
        the specifiedobject. The property names are specified by the list of
        names. The method returns a dictionnary with properties names as keys
        indexing values (or a single value if you give a single property name
        in place of the list).

        Example:
            props = db.get_property ("mymotor", ["velocity", "acceleration"])
        Or:
            props = db.get_property ("mymotor", "velocity", "acceleration")
        Return:
            props -> {"velocity":["100000.0"],"acceleration":["500000.0"]}

        Example:
            props = db.get_property ("mymotor", "velocity")
        Return:
            props -> ["100000.0"]
        """
        return self.db_property_getter ("get_property","DbGetProperty",
                    name,proplist,otherprop)
    dbgetprop = get_property

    #----------------------------------------------------------------------------
    def put_property (self,name,_thepropdict={},**propbyname) :
        """Insert or update a list of properties.

        Insert or update a list of properties for the specified object. The
        property names and their values are specified by the dictionnary, with
        properties names as keys indexing values.

        Example:
            db.put_property ("mymotor", {"velocity":100000.0,
                                         "acceleration":500000.0})
        Or:
            db.put_property ("mymotor", velocity = 100000.0, 
                                        acceleration = 500000.0)

        Example:
            db.put_property ("mytraductor", 
                                    langage = ["french","english","german"],
                                    directions = ["fr2en","en2fr","en2ge"])
        """
        self.db_property_putter ("put_property","DbPutProperty",
                    name,_thepropdict,propbyname)
    dbputprop = put_property

    #---------------------------------------------------------------------------
    def delete_property (self,name,proplist,*otherprop) :
        """Delete a list of properties.

        Delete a list of properties for the specified object. The property names
        are specified by the list of names.

        Example:
            db.delete_property ("mymotor", ["velocity", "acceleration"])
        Or:
            db.delete_property ("mymotor", "velocity", "acceleration")
        """
        self.db_property_deleter ("delete_property","DbDeleteProperty",
                    name,proplist,otherprop)
    dbdelprop = delete_property

    #---------------------------------------------------------------------------
    def list_device_class (self,name) :
        """Query the database for a list of devices and their classes 
        
        Query the database for a list of devices and their classes to be served 
        by the speficied device server. Return a list of tuples: 
            (device name,class name)

        Example:
            dclist = db.list_device_class ("StepperMotor/id11")
        Return:
            dclist -> [ ("id11/motor/1","StepperMotor"), 
                        ("id11/motor/1","StepperMotor") ]
        """
        res = self.db_con.command_inout ("DbGetDeviceClassList",name)
        final = []
        for i in range (0,len(res),2) :
            final.append ((res[i],res[i+1]))
        return final
    dblistdevclass = list_device_class

    #---------------------------------------------------------------------------
    def list_device_property (self,name,wildcard="*") :
        """Query the database for a list of device properties names.

        Query the database for a list of device properties
        that match the wildcard. Wildcard character is * and matches any number 
        of characters. The method returns a list of names.
        """
        return self.db_con.command_inout ("DbGetDevicePropertyList",\
                                            [name,wildcard])
    dblistdevprop = list_device_property

    #---------------------------------------------------------------------------
    def get_device_property (self,name,proplist,*otherprop) :
        """Query the database for a list of device properties.

        Query the database for a list of device properties for the specified 
        object.
        The property names are specified in a list. The method return a 
        dictionnary indexing properties values by names, or a single value if a 
        single property name was given as proplist (ie. not a list).


        Example:
            props = db.get_device_property ("id11/motor/1", 
                                            ["velocity", "acceleration"])
        Or:
            props = db.get_device_property ("id11/motor/1", 
                                            "velocity", "acceleration")
        Return:
            props -> {"velocity":["100000.0"],"acceleration":["500000.0"]}

        Example:
            props = db.get_device_property ("mymotor", "velocity")
        Return:
            props -> ["100000.0"]
        """
        return self.db_property_getter ("get_device_property",\
                    "DbGetDeviceProperty",\
                    name,proplist,otherprop)
    dbgetdevprop = get_device_property

    #---------------------------------------------------------------------------
    def put_device_property (self,name,_thepropdict={},**propbyname) :
        """Insert or update a list of properties for the specified device.

        The property names and their values are specified by the dictionnary,
        where values are indexed by name.

        Example:
            db.put_device_property ("id11/motor/1", {"velocity":100000.0,
                                                     "acceleration":500000.0})
        Or:
            db.put_device_property ("id11/motor/1", velocity = 100000.0, 
                                                    acceleration = 500000.0)
        """
        self.db_property_putter ("put_device_property","DbPutDeviceProperty",\
                    name,_thepropdict,propbyname)
    dbputdevprop = put_device_property

    #---------------------------------------------------------------------------
    def delete_device_property (self,name,proplist,*otherprop) :
        """Delete a list of properties for the specified device.

        The property names are specified by the list of names.

        Example:
            db.delete_device_property ("id11/motor/1", 
                                        ["velocity", "acceleration"])
        Or:
            db.delete_device_property ("id11/motor/1", 
                                        "velocity", "acceleration")
        """
        self.db_property_deleter ("delete_device_property",\
                    "DbDeleteDeviceProperty",\
                    name,proplist,otherprop)
    dbdeldevprop = delete_device_property

    #---------------------------------------------------------------------------
    def list_device_attribute (self,name,wildcard="*") :
        """Query the database for a list of device attribute names.

        Query the database for a list of device attributes names that match the
        wildcard. Wildcard character is * and matches any number of characters.
        The method returns a list of names.

        Example:
            attlist = db.list_device_attribute ("id11/motor/1","*")
        Return:
            attlist -> [ "position", "acceleration", "velocity" ]
        """
        return self.db_con.command_inout ("DbGetDeviceAttributeList",\
                                                [name,wildcard])
    dblistdevatt = list_device_attribute

    #---------------------------------------------------------------------------
    def delete_device_attribute (self,name,attlist,*otheratt) :
        """Delete a device attribute and all its properties from the database.
        """
        self.db_attribute_deleter ("delete_device_attribute",\
                    "DbDeleteDeviceAttribute",\
                    name,attlist,otheratt)

    #---------------------------------------------------------------------------
    def get_device_attribute_property (self,name,attlist,*otheratt) :
        """Query the database for a list of device attribute properties.

        Query the database for a list of device attribute properties for the
        specified device. The attribute names are specified by the list of
        names. The method return all the properties for the specified
        attributes, in two level of dictionnaries (first level use attribute
        name as key and second level use property name as key).

        Example:
            props = db.get_device_attribute_property ("id11/motor/1",
                                        ["velocity","acceleration"])
        Or:
            props = db.get_device_attribute_property ("id11/motor/1",
                                        "velocity","acceleration")
        Return:
            props -> {"velocity":{"min":"400","max":"4000","unit":"steps/sec"},
                      "acceleration":{"min":"100","max":"10000",
                                      "unit":"steps/sec2"}}
            props["velocity"]["min"] -> "400"

        Example:
            props = db.get_device_attribute_property ("id11/motor/1","velocity")
        Return:
            props -> {"min":"400","max":"4000","unit":"steps/sec"}
            props["min"] -> "400"
        """
        return self.db_attribute_property_getter(\
                    "get_device_attribute_property",\
                    "DbGetDeviceAttributeProperty",name,attlist,otheratt)
    dbgetdevattprop = get_device_attribute_property
    
    #---------------------------------------------------------------------------
    def put_device_attribute_property (self,name,_theattdict={},**attbyname) :
        """Insert or update a list of attribute properties for the device.

        The attributes property names and their values are specified in two
        level of dictionnaries (first level use attribute name as key and
        second level use property name as key).

        Example:
            db.put_device_attribute_property ("id11/motor/1",
                    {"velocity":{"min":"400","max":"4000","unit":"steps/sec"},
                     "acceleration":{"min":"100","max":"10000",
                                     "unit":"steps/sec2"}})
        Or:
            db.put_device_attribute_property ("id11/motor/1",
                    velocity = {"min":"400","max":"4000","unit":"steps/sec"},
                    acceleration ={"min":"100","max":"10000",
                                   "unit":"steps/sec2"})
        """
        self.db_attribute_property_putter ("put_device_attribute_property",
                    "DbPutDeviceAttributeProperty",name,_theattdict,attbyname)
    dbputdevattprop = put_device_attribute_property

    #---------------------------------------------------------------------------
    def delete_device_attribute_property (self,name,attrib,proplist="*",\
                                                                *otherprop) :
        """Delete a list of attribute properties for the specified device.

        You can use wildcard "*" to specify all attributes or all properties
        (by default properties list is set to "*").
        You can use one attribute name, or a list of attribute names.

        Examples:
            db.delete_device_attribute_property ("id11/motor/1","velocity")
            db.delete_device_attribute_property ("id11/motor/1","*")
            db.delete_device_attribute_property ("id11/motor/1","*","unit")
            db.delete_device_attribute_property ("id11/motor/1",
                                            ["velocity","acceleration"],"unit")
            db.delete_device_attribute_property ("id11/motor/1","velocity",
                                            "min","max")
            db.delete_device_attribute_property ("id11/motor/1","velocity",
                                            ["min","max"])
        """
        if type (attrib) in (types.ListType,types.TupleType) :
            for att in attrib :
                apply (self.delete_device_attribute_property, \
                            (name,att,proplist)+otherprop)
            return

        if attrib == "*" :
            for att in self.list_device_attribute (name) :
                apply (self.delete_device_attribute_property, \
                            (name,att,proplist)+otherprop)
            return

        if proplist == "*" :
            proplist = self.get_device_attribute_property (name,attrib).keys()
        self.db_attribute_property_deleter ("delete_device_attribute_property",\
                    "DbDeleteDeviceAttributeProperty",name,attrib,proplist,\
                    otherprop)
    dbdeldevattprop = delete_device_attribute_property

    #---------------------------------------------------------------------------
    def list_class (self,name) :
        """Query the database for a list of classes served by a device server.

        Query the database for a list of classes to be served by the specified
        device server. Return a list of class names.

        Example:
            clist = db.list_class ("StepperMotor/id11")
        Return:
            clist -> [ "DServer", "StepperMotor" ]
        """
        return self.db_con.command_inout ("DbGetClassList",name)
    dblistclass = list_class

    #---------------------------------------------------------------------------
    def list_class_property (self,name) :
        """Query the database for a list of class properties names.

        Query the database for a list of class properties for the specified 
        class.
        The method returns a list of names.
        """
        return self.db_con.command_inout ("DbGetClassPropertyList",name)
    dblistclassprop = list_class_property

    #---------------------------------------------------------------------------
    def get_class_property (self,name,proplist,*otherprop) :
        """Query the database for a list of class properties.

        Query the database for a list of device properties for the specified 
        object.
        The property names are specified in a list. The method return a 
        dictionnary indexing properties values by names, or a single value if a 
        single property name was given as proplist (ie. not a list).

        Example:
            props = db.get_class_property ("StepperMotor", 
                                        ["velocity", "acceleration"])
        Or:
            props = db.get_class_property ("StepperMotor", 
                                        "velocity", "acceleration")
        Return:
            props -> {"velocity":["100000.0"],"acceleration":["500000.0"]}

        Example:
            props = db.get_class_property ("mymotor", "velocity")
        Return:
            props -> ["100000.0"]
        """
        return self.db_property_getter ("get_class_property",\
                    "DbGetClassProperty",name,proplist,otherprop)
    dbgetclassprop = get_class_property

    #---------------------------------------------------------------------------
    def put_class_property (self,name,_thepropdict={},**propbyname) :
        """Insert or update a list of properties for the specified class.

        The property names and their values are specified by the dictionnary,
        where values are indexed by name.

        Example:
            db.put_class_property ("StepperMotor", {"velocity":100000.0,
                                                    "acceleration":500000.0})
        Or:
            db.put_class_property ("StepperMotor", velocity = 100000.0, 
                                                   acceleration = 500000.0)
        """
        self.db_property_putter ("put_class_property","DbPutClassProperty",\
                    name,_thepropdict,propbyname)
    dbputclassprop = put_class_property

    #----------------------------------------------------------------------------
    def delete_class_property (self,name,proplist,*otherprop) :
        """Delete a list of properties for the specified class.

        The property names are specified by the list of names.

        Example:
            db.delete_class_property ("StepperMotor", 
                                      ["velocity", "acceleration"])
        Or:
            db.delete_class_property ("StepperMotor",
                                      "velocity", "acceleration")
        """
        self.db_property_deleter ("delete_class_property",\
                    "DbDeleteClassProperty",\
                    name,proplist,otherprop)
    dbdelclassprop = delete_class_property

    #---------------------------------------------------------------------------
    def list_class_attribute (self,name,wildcard="*") :
        """Query the database for a list of class attribute names.

        Query the database for a list of class attributes names that match the
        wildcard. Wildcard character is * and matches any number of characters.
        The method returns a list of names.

        Example:
            attlist = db.list_class_attribute ("StepperMotor","*")
        Return:
            attlist -> [ "position", "acceleration", "velocity" ]
        """
        return self.db_con.command_inout ("DbGetClassAttributeList",\
                                        [name,wildcard])
    dblistclassatt = list_class_attribute

#     #-----------------------------------------------------------------------
#     def delete_class_attribute (self,name,attlist,*otheratt) :
#         """Delete a class attribute and all its properties from the database.
#         """
#         self.db_attribute_deleter ("delete_class_attribute",\
#                                    "DbDeleteClassAttribute",\
#                                    name,attlist,otheratt)

    #---------------------------------------------------------------------------
    def get_class_attribute_property (self,name,attlist,*otheratt) :
        """Query the database for a list of class attribute properties.

        Query the database for a list of class attribute properties for the
        specified class. The attribute names are specified by the list of
        names. The method return all the properties for the specified
        attributes, in two level of dictionnaries (first level use attribute
        name as key and second level use property name as key).

        Example:
            props = db.get_class_attribute_property ("StepperMotor",
                                        ["velocity","acceleration"])
        Or:
            props = db.get_class_attribute_property ("StepperMotor",
                                        "velocity","acceleration")
        Return:
            props -> {"velocity":{"min":"400","max":"4000","unit":"steps/sec"},
                      "acceleration":{"min":"100","max":"10000",
                                      "unit":"steps/sec2"}}
            props["velocity"]["min"] -> "400"

        Example:
            props = db.get_class_attribute_property ("StepperMotor","velocity")
        Return:
            props -> {"min":"400","max":"4000","unit":"steps/sec"}
            props["min"] -> "400"
        """
        return self.db_attribute_property_getter (\
                    "get_class_attribute_property",\
                    "DbGetClassAttributeProperty",\
                    name,attlist,otheratt)
    dbgetclassattprop = get_class_attribute_property

    #---------------------------------------------------------------------------
    def put_class_attribute_property (self,name,_theattdict={},**attbyname) :
        """Insert or update a list of attribute properties for the class.

        The attributes property names and their values are specified in two 
        level of dictionnaries (first level use attribute name as key and second 
        level use property name as key).

        Example:
            db.put_class_attribute_property ("StepperMotor",
                    {"velocity":{"min":"400","max":"4000","unit":"steps/sec"},
                     "acceleration":{"min":"100","max":"10000",
                     "unit":"steps/sec2"}})
        Or:
            db.put_class_attribute_property ("StepperMotor",
                    velocity = {"min":"400","max":"4000","unit":"steps/sec"},
                    acceleration ={"min":"100","max":"10000",
                    "unit":"steps/sec2"})
        """
        self.db_attribute_property_putter ("put_class_attribute_property",
                    "DbPutClassAttributeProperty",name,_theattdict,attbyname)
    dbputclassattprop = put_class_attribute_property

#     #-----------------------------------------------------------------------
#     def delete_class_attribute_property (self,name,attrib,proplist="*",*otherprop) :
#         """Delete a list of attribute properties for the specified class.
#
#         You can use wildcard "*" to specify all attributes or all properties
#         (by default properties list is set to "*").
#         You can use one attribute name, or a list of attribute names.
#
#         Examples:
#             db.delete_class_attribute_property ("StepperMotor","velocity")
#             db.delete_class_attribute_property ("StepperMotor","*")
#             db.delete_class_attribute_property ("StepperMotor","*","unit")
#             db.delete_class_attribute_property ("StepperMotor",["velocity","acceleration"],"unit")
#             db.delete_class_attribute_property ("StepperMotor","velocity","min","max")
#             db.delete_class_attribute_property ("StepperMotor","velocity",["min","max"])
#
#
#         """
#         self.db_attribute_property_deleter ("delete_class_attribute_property",
#                     "DbDeleteClassAttributeProperty",name,attrib,proplist,otherprop)
#     dbdelclassattprop = delete_class_attribute_property
#
    #---------------------------------------------------------------------------
    def db_property_getter (self,method,command,name,proplist,otherprop) :
        """INTERNAL: do the job for all get_xxx_property methods."""
        if type(proplist) in (types.ListType,types.TupleType) :
            if len(otherprop) :
                raise tangocorba.tango_error (\
                        "Bad parameter for Database.%s."%method,\
                        tangocorba.ERR,\
                        "You cannot give a list as property names parameter "\
                        "and then other parameters.")
            fullproplist = [ name ] + list(proplist)
        elif type(proplist) == types.StringType :
            fullproplist = [ name, proplist] + list(otherprop)
        else :
            raise tangocorba.tango_error ("Bad parameter for Database.%s."%\
                        method,tangocorba.ERR,\
                        "Bad parameter type for %s (%s)."%\
                        (proplist,type(proplist)))
        ret = self.db_con.command_inout (command,fullproplist)

        if type(proplist)==types.StringType and len(otherprop)==0:
            return tangocorba.proplist2dict (ret)[proplist]
        else :
            return tangocorba.proplist2dict (ret)

    #---------------------------------------------------------------------------
    def db_property_putter (self,method,command,name,propdict,propbyname) :
        """INTERNAL: do the job for all put_xxx_property methods."""
        if len(propbyname) and len(propdict) :
            raise tangocorba.tango_error (\
                        "Bad parameter for Database.%s."%method,\
                        tangocorba.ERR,\
                        "You cannot give an anonymous dictionnary as second "\
                        "parameter and then other parameters.")
        if len(propdict) : propbyname = propdict
        if len(propbyname)==0 : return
        fullproplist = [ name,str(len(propbyname)) ]
        for propname in propbyname.keys () :
            fullproplist.append (propname)
            propvals = propbyname[propname]
            if type (propvals) in (types.ListType,types.TupleType) :
                fullproplist.append (str(len(propvals)))
                fullproplist += list(propvals)
            else :
                fullproplist.append ("1")
                fullproplist.append (str(propvals))

        self.db_con.command_inout (command,fullproplist)

    #---------------------------------------------------------------------------
    def db_property_deleter (self,method,command,name,proplist,otherprop) :
        """INTERNAL: do the job for all del_xxx_property methods."""
        if type(proplist) in (types.ListType,types.TupleType) :
            if len(otherprop) :
                raise tangocorba.tango_error (\
                        "Bad parameter for Database.%s."%method,\
                        tangocorba.ERR,\
                        "You cannot give a list as property names parameter "\
                        "and then other parameters.")
            fullproplist = [ name ] + list(proplist)
        elif type(proplist) == types.StringType :
            fullproplist = [ name, proplist] + list(otherprop)
        else :
            raise tangocorba.tango_error (\
                        "Bad parameter for Database.%s."%method,\
                        tangocorba.ERR,\
                        "Bad parameter type for %s (%s)."%\
                        (proplist,type(proplist)))

        self.db_con.command_inout (command,fullproplist)

    #---------------------------------------------------------------------------
    def db_attribute_deleter (self,method,command,name,attlist,otheratt) :
        """INTERNAL: do the job for all del_xxx_attribute methods."""
        if type(attlist) in (types.ListType,types.TupleType) :
            if len(otheratt) :
                raise tangocorba.tango_error (\
                        "Bad parameter for Database.%s."%method,\
                        tangocorba.ERR,\
                        "You cannot give a list as attribute names parameter "\
                        "and then other parameters.")
            fullattlist = [ name ] + list(attlist)
        elif type (attlist) == types.StringType :
            fullattlist = [ name, attlist] + list(otheratt)
        else :
            raise tangocorba.tango_error (\
                        "Bad parameter for Database.%s."%method,\
                        tangocorba.ERR,\
                        "Bad parameter type for %s (%s)."%\
                        (attlist,type(attlist)))

        self.db_con.command_inout (command,fullattlist)

    #---------------------------------------------------------------------------
    def db_attribute_property_getter(self,method,command,name,attlist,otheratt):
        """INTERNAL: do the job for all get_xxx_attribute_property methods."""
        if type(attlist) in (types.ListType,types.TupleType) :
            if len(otheratt) :
                raise tangocorba.tango_error ("Bad parameter for Database.%s."%\
                        method,tangocorba.ERR,\
                        "You cannot give a list as attribute names parameter "\
                        "and then other parameters.")
            fullattlist = [ name ] + list(attlist)
        elif type (attlist) == types.StringType :
            fullattlist = [ name, attlist] + list(otheratt)
        else :
            raise tangocorba.tango_error (\
                        "Bad parameter for Database.%s."%method,\
                        tangocorba.ERR,\
                        "Bad parameter type for %s (%s)."%\
                        (attlist,type(attlist)))

        ret = self.db_con.command_inout (command,fullattlist)

        if type(attlist)==types.StringType and len(otheratt)==0 :
            return tangocorba.attproplist2dict(ret)[attlist]
        else :
            return tangocorba.attproplist2dict(ret)

    #---------------------------------------------------------------------------
    def db_attribute_property_putter (self,method,command,name,attdict,\
                                                                attbyname) :
        """INTERNAL: do the job for all put_xxx_attribute_property methods."""
        if len(attbyname) and len(attdict) :
            raise tangocorba.tango_error (\
                        "Bad parameter for Database.%s."%method,\
                        tangocorba.ERR,\
                        "You cannot give an anonymous dictionnary as second "\
                        "parameter and then other parameters.")
        if len(attdict) : attbyname = attdict
        if len(attbyname)==0 : return
        fullattlist = [ name,str(len(attbyname)) ]
        for attname in attbyname.keys () :
            propdict = attbyname[attname]
            fullattlist.append (attname)
            fullattlist.append (len(propdict))
            for propname in propdict.keys () :
                fullattlist.append (str(propname))
                propval = propdict[propname]
                if type(propval) in (types.ListType,types.TupleType) :
                    if len(propval)==1 :
                        propval = propval[0]
                    elif len(propval)==0 :
                        propval = ""
                assert type(propval) in (types.IntType,types.LongType,\
                                         types.FloatType,types.StringType)
                fullattlist.append (str(propval))

        self.db_con.command_inout (command,fullattlist)

    #---------------------------------------------------------------------------
    def db_attribute_property_deleter (self,method,command,name,attrib,\
                                                        proplist,otherprop) :
        """INTERNAL: do the job for all del_xxx_attribute_property methods."""
        if type(proplist) in (types.ListType,types.TupleType) :
            if len(otherprop) :
                raise tangocorba.tango_error (\
                        "Bad parameter for Database.%s."%method,\
                        tangocorba.ERR,\
                        "You cannot give a list as property names parameter "\
                        "and then other parameters.")
            fullproplist = [ name,attrib ] + list(proplist)
        elif type(proplist) == types.StringType :
            fullproplist = [ name,attrib,proplist] + list(otherprop)
        else :
            raise tangocorba.tango_error (\
                        "Bad parameter for Database.%s."%method,\
                        tangocorba.ERR,\
                        "Bad parameter type for %s (%s)."%\
                        (proplist,type(proplist)))

        self.db_con.command_inout (command,fullproplist)




#===============================================================================
# TO DO: ALLOW **ALL** SAME FUNCTIONNALITIES AS Database *_device_* METHODS.
class DbDevice :
    """Shortcut to get a specific device informations from a database.

    All requests miss the device name parameter as it is given at construction 
    and automatically inserted at begining of names before sending requests to 
    the database.
    """
    #---------------------------------------------------------------------------
    def __init__ (self,name,database=None) :
        """Construct the DbDevice.

        Parameters:
            -name       : the device name.
            -database   : the Database object to use for connexion (or a 
                          connexion to a database). It is optional, if missing
                          we use the default "corbaloc:iiop:TANGO_HOST/database" 
                          database.
        """
        global gdbcon

        self.mdd_name = name
        if database == None and name == None :
            self.mdd_database = None
        elif database == None :
            self.mdd_database = Database (gdbcon)
        elif isinstance (database,Database) :
            self.mdd_database = database
        elif isinstance (database,tangocorba.DeviceConnection) :
            self.mdd_database = Database (database)
        else :
            raise tangocorba.tango_error ("DbDevice bad parameter",\
                        tangocorba.ERR, "The database parameter is not None "\
                        "or a Database object or a DeviceConnection object.")

    #---------------------------------------------------------------------------
    def add_device (self,devinf) :
        self.dd_checkready ()
        devinf.name = self.mdd_name
        self.mdd_database.add_device (devinf)

    #---------------------------------------------------------------------------
    def delete_device (self) :
        self.dd_checkready ()
        self.mdd_database.delete_device (self.mdd_name)

    #---------------------------------------------------------------------------
    def import_device (self) :
        self.dd_checkready ()
        return self.mdd_database.import_device (self.mdd_name)

    #---------------------------------------------------------------------------
    def export_device (self,devexpinf) :
        self.dd_checkready ()
        devexpinf.name = self.mdd_name
        self.mdd_database.export_device (devexpinf)

    #---------------------------------------------------------------------------
#     def unexport_device (self,devexpinf) :
#         self.dd_checkready ()
#         devexpinf.name = self.mdd_name
#         self.mdd_database.export_device (devexpinf)

    #---------------------------------------------------------------------------
#     def get_device_alias (self,)

    #---------------------------------------------------------------------------
#    def list_device_property

    #---------------------------------------------------------------------------
    def get_property (self,proplist,*otherprop) :
        self.dd_checkready ()
        return apply (self.mdd_database,"get_device_property", \
                    (self.mdd_name,proplist)+otherprop)

    #---------------------------------------------------------------------------
    def put_property (self,_thepropdict,**propkeyvals) :
        self.dd_checkready ()
        apply (self.mdd_database,"put_device_property",\
                    (self.mdd_name,_thepropdict),propkeyvals)

    #---------------------------------------------------------------------------
    def delete_property (self,proplist,*otherprop) :
        self.dd_checkready ()
        apply (self.mdd_database,"delete_device_property", \
                    (self.mdd_name,proplist)+otherprop)

    #---------------------------------------------------------------------------
    def list_attribute (self,wildcard="*") :
        self.dd_checkready ()
        return self.mdd_database.list_device_attribute (self.mdd_name,wildcard)

    #---------------------------------------------------------------------------
#     def delete_device_attribute

    #---------------------------------------------------------------------------
    def get_attribute_property (self,attlist,*otheratt) :
        self.dd_checkready ()
        return apply(self.mdd_database,"get_device_attribute_property", \
                    (self.mdd_name,attlist)+otheratt)

    #---------------------------------------------------------------------------
    def put_attribute_property (self,attdict,**attbyname) :
        self.dd_checkready ()
        apply (self.mdd_database,"put_device_attribute_property",\
                    (self.mdd_name,attdict),attbyname)

    #---------------------------------------------------------------------------
    def delete_attribute_property (self,attrib,proplist,*otherprop) :
        self.dd_checkready ()
        apply(self.mdd_database,"delete_device_attribute_property", \
                    (self.mdd_name,attrib,proplist)+otherprop)

    #---------------------------------------------------------------------------
    def dd_checkready (self) :
        if self.mdd_name == None :
            raise tangocorba.tango_error ("DbDevice unusable",\
                        tangocorba.ERR, "No Device name was given... maybe"\
                        "because you use a device without using the "\
                        "database.")
            
    
#===============================================================================
class DbClass :
# TO DO: ALLOW **ALL** SAME FUNCTIONNALITIES AS Database *_class_* METHODS.
    #---------------------------------------------------------------------------
    def __init__ (self,name,database=None) :
        """Construct the DbClass.

        Parameters:
            -name       : the device name.
            -database   : the Database object to use for connexion (or a 
                          connexion to a database). It is optional, if missing 
                          we use the default "corbaloc:iiop:TANGO_HOST/database" 
                          database.
        """
        global gdbcon

        self.mdc_name = name
        if database == None :
            self.mdc_database = Database (gdbcon)
        elif isinstance (database,Database) :
            self.mdc_database = database
        elif isinstance (database,tangocorba.DeviceConnection) :
            self.mdc_database = Database (database)
        else :
            raise tangocorba.tango_error ("DbClass bad parameter",\
                        tangocorba.ERR, "The database parameter is not None "\
                        "or a Database object or a DeviceConnection object.")

    #---------------------------------------------------------------------------
    def get_property (self,proplist,*otherprop) :
        return apply (self.mdc_database,"get_class_property",\
                    (self.mdc_name,proplist)+otherprop)

    #---------------------------------------------------------------------------
    def put_property (self,_thepropdict,**propkeyvals) :
        apply (self.mdc_database,"put_class_property",\
                    (self.mdc_name,_thepropdict),propkeyvals)

    #---------------------------------------------------------------------------
    def delete_property (self,proplist,*otherprop) :
        apply (self.mdc_database,"delete_class_property",\
                    (self.mdc_name,proplist)+otherprop)

    #---------------------------------------------------------------------------
    def get_attribute_property (self,attlist,*otheratt) :
        return apply(self.mdc_database,"get_class_attribute_property",\
                    (self.mdc_name,attlist)+otheratt)

    #---------------------------------------------------------------------------
    def put_attribute_property (self,attdict,**attbyname) :
        apply (self.mdc_database,"put_class_attribute_property",\
                    (self.mdc_name,attdict),attbyname)

    #---------------------------------------------------------------------------
    def delete_attribute_property (self,attrib,proplist,*otherprop) :
        apply(self.mdc_database,"delete_class_attribute_property",\
                    (self.mdc_name,attrib,proplist)+otherprop)

#===============================================================================
class DbServInfo :
    """Placeholder to store server informations.

    Used as Database.get_server_info output, and Database.put_server_info input.
    """
    def __init__ (self,name = "", host = "",mode = "",level = "") :
        """Construct a DbServInfo, with fields as empty string by default."""
        self.name = name
        self.host = host
        self.mode = mode    # 1 for automatic start by starter, 
                            # 0 for manual start.
        self.level = level  # run order for the starter (like unix runlevel 
                            #of daemons).
    def __str__ (self) :
        format = "DbServInfo:\n\tname: %(name)r\n\thost: %(host)r"\
                 "\n\tmode: %(mode)r\n\tlevel: %(level)r"
        return format%self.__dict__

#===============================================================================
class DbDevInfo :
    """Placeholder to store device informations.

    Used as Database.add_device and add_server paramater.
    """
    def __init__ (self,name = "",_class = "",server = "") :
        """Construct a DbDevInfo, with fields as empty string by default."""
        self.name = name
        self._class = _class
        self.server = server
    def __str__ (self) :
        format = "DbDevInfo:\n\tname: %(name)r\n\tclass: %(_class)r"\
                 "\n\tserver: %(server)r"
        return format%self.__dict__

#===============================================================================
class DbDevImportInfo :
    """Placeholder to store imported device informations.

    Used as Database.import_device return value.
    """
    def __init__ (self) :
        """Construct a DbDevImportInfo, with fields as None by default."""
        self.name = None
        self.exported = None
        self.ior = None
        self.version = None
    def __str__ (self) :
        format = "DbDevInfo:\n\tname: %(name)r\n\texported: %(exported)r"\
                 "\n\tior: %(ior)r\n\tversion: %(version)r"
        return format%self.__dict__

#===============================================================================
class DbDevExportInfo :
    """Placeholder to store exported device informations.

    Used as Database.export_device paramater.
    """
    def __init__ (self,name="",ior="",host="",version="",pid="") :
        """Construct a DbDevExportInfo, with fields as empty string by default.
        """
        self.name = name
        self.ior = ior
        self.host = host
        self.version = version
        self.pid = pid
    def __str__ (self) :
        format = "DbDevInfo:\n\tname: %(name)r\n\tior: %(ior)r\n\thost: "\
                "%(host)r\n\tversion: %(version)r\n\tpid: %(pid)r"
        return format%self.__dict__




# End of pyTango/tangodatabase.py


