#!/usr/bin/env python
"""tangocorba.py - general CORBA interface for Tango-Python modules.

Blabla:
         author : Laurent Pointal (laurent.pointal@lure.u-psud.fr)
       creation : 8 april 2002
          modif : 1 october 2002

History:
8 april 2002 - Laurent Pointal - creation of the module.
14 june 2002 - LP - Moved name analysis and storage into TangoName class.
1 october 2002 - LP - Commented out __getattr__ for DeviceConnection.
"""

#===============================================================================
# Imports and exports.
__all__ = [ "Tango","PyTangoError","DeviceConnection","tango_error",
            "exception_to_tango_error", "WARN","ERR","PANIC" ]
import os,sys,threading,string,traceback,inspect,types
import omniORB
# Following replace "from omniORB import CORBA", avoid pychecker warning.
CORBA = omniORB.CORBA
from pyTango.corbacom import Tango
#import Tango
WARN  = Tango.WARN
ERR   = Tango.ERR
PANIC = Tango.PANIC


#===============================================================================
# To debug CORBA calls uncomment the following line.
# sys.argv += [ "-ORBtraceLevel", "25" ]
# sys.argv += [ "-ORBtraceInvocations", "1" ]

# To resolve inter-gorb communication problems try to uncomment some of 
# the following lines.
# sys.argv += [ "-ORBstrictIIOP", "1" ]
# sys.argv += [ "-ORBlcdMode" ]    # Activation of the lower common protocol.
# sys.argv += [ "-ORBverifyObjectExistsAndType", "0" ]


#===============================================================================
# Constants.

#===============================================================================
# Globals.
# Types codes from C++ source (enum CmdArgType)
DEV_VOID                    =  0
DEV_BOOLEAN                 =  1
DEV_SHORT                   =  2
DEV_LONG                    =  3
DEV_FLOAT                   =  4
DEV_DOUBLE                  =  5
DEV_USHORT                  =  6
DEV_ULONG                   =  7
DEV_STRING                  =  8
DEVVAR_CHARARRAY            =  9
DEVVAR_SHORTARRAY           = 10
DEVVAR_LONGARRAY            = 11
DEVVAR_FLOATARRAY           = 12
DEVVAR_DOUBLEARRAY          = 13
DEVVAR_USHORTARRAY          = 14
DEVVAR_ULONGARRAY           = 15
DEVVAR_STRINGARRAY          = 16
DEVVAR_LONGSTRINGARRAY      = 17
DEVVAR_DOUBLESTRINGARRAY    = 18
DEV_STATE                   = 19
CONST_DEV_STRING            = 20

# Type codes constants list.
gtypescodes = [
    # As tuples:(Tango in/out numeric value, name, 
    #                                       corresponding CORBA type code, 
    #                                       corresponding type )
    (  0,   "None",                 
            None,                                           
            None),
    (  1,   "DevBoolean",
            CORBA.TypeCode (CORBA.id(Tango.DevBoolean)),    
            Tango.DevBoolean ),
    (  2,   "DevShort",             
            CORBA.TypeCode (CORBA.id(Tango.DevShort)),
            Tango.DevShort ),
    (  3,   "DevLong",              
            CORBA.TypeCode (CORBA.id(Tango.DevLong)),       
            Tango.DevLong ),
    (  4,   "DevFloat",             
            CORBA.TypeCode (CORBA.id(Tango.DevFloat)),      
            Tango.DevFloat ),
    (  5,   "DevDouble",            
            CORBA.TypeCode (CORBA.id(Tango.DevDouble)),     
            Tango.DevDouble ),
    (  6,   "DevUShort",            
            CORBA.TypeCode (CORBA.id(Tango.DevUShort)),     
            Tango.DevUShort ),
    (  7,   "DevULong",             
            CORBA.TypeCode (CORBA.id(Tango.DevULong)),      
            Tango.DevULong ),
    (  8,   "DevString",            
            CORBA.TypeCode (CORBA.id(Tango.DevString)),     
            Tango.DevString ),
    (  9,   "DevVarCharArray",      
            CORBA.TypeCode (CORBA.id(Tango.DevVarCharArray)), 
            Tango.DevVarCharArray ),
    ( 10,   "DevVarShortArray",     
            CORBA.TypeCode (CORBA.id(Tango.DevVarShortArray)), 
            Tango.DevVarShortArray ),
    ( 11,   "DevVarLongArray",
            CORBA.TypeCode (CORBA.id(Tango.DevVarLongArray)),
            Tango.DevVarLongArray ),
    ( 12,   "DevVarFloatArray",
            CORBA.TypeCode (CORBA.id(Tango.DevVarFloatArray)), 
            Tango.DevVarFloatArray ),
    ( 13,   "DevVarDoubleArray",    
            CORBA.TypeCode (CORBA.id(Tango.DevVarDoubleArray)),
            Tango.DevVarDoubleArray ),
    ( 14,   "DevVarUShortArray",    
            CORBA.TypeCode (CORBA.id(Tango.DevVarUShortArray)), 
            Tango.DevVarUShortArray ),
    ( 15,   "DevVarULongArray",     
            CORBA.TypeCode (CORBA.id(Tango.DevVarULongArray)), 
            Tango.DevVarULongArray ),
    ( 16,   "DevVarStringArray",    
            CORBA.TypeCode (CORBA.id(Tango.DevVarStringArray)), 
            Tango.DevVarStringArray ),
    ( 17,   "DevVarLongStringArray",
            CORBA.TypeCode (CORBA.id(Tango.DevVarLongStringArray)), 
            Tango.DevVarLongStringArray ),
    ( 18,   "DevVarDoubleStringArray",
            CORBA.TypeCode (CORBA.id(Tango.DevVarDoubleStringArray)),
            Tango.DevVarDoubleStringArray ),
    ( 19,   "DevState",             
            CORBA.TypeCode (CORBA.id(Tango.DevState)), 
            Tango.DevState ),
    ( 20,   "const DevString",      
            CORBA.TypeCode (CORBA.id(Tango.DevString)), 
            Tango.DevString )
    ]

# CORBA type codes and name indexed by Tango numeric value.
gtcbynum = {}
gtcnamebynum = {}
gtctypebynum = {}
for i in gtypescodes :
    gtcbynum[i[0]] = i[2]
    gtcnamebynum[i[0]] = i[1]
    gtctypebynum[i[0]] = i[3]
    
# For mapping
gmaptypes = { DEVVAR_LONGSTRINGARRAY : "lvalue", 
              DEVVAR_DOUBLESTRINGARRAY : "dvalue" }

# Debug this module.
DEBUG = 0
# Trace DevFailed errors.
TRACE = 0

#===============================================================================
gorb = CORBA.ORB_init(sys.argv,CORBA.ORB_ID)

#===============================================================================
def type_to_typecode (tangotypenum) :
    """Convert Tango type number into a CORBA typecode.
    """
    global gtcbynum
    tc = gtcbynum.get (tangotypenum,"unknown")
    if tc == "unknown" :
        raise RuntimeError, "Unknown Tango type number: %d"%(tangotypenum,)
    return tc

#===============================================================================
class PyTangoError :
    """Exception class for all errors coming from pyTango. 

    """
    #Note: I failed to subclass Tango.DevFailed... it look like OmniORBpy make
    # it a 'terminal' class... which cant be subclassed (impossible to call the
    # inherited __init__ method.
    def __init__ (self,err=None) :
        self.errors = []    # To store DevError objects, as in DevFailed.
        if err : self.errors.append (Tango.DevError ("PyTangoError",\
                            Tango.ERR,err,"pyTango package"))
        self.tracebackstring = ""

    def __str__ (self) :
#         if DEBUG :
#             return self.dumpstr ()
#         el
        if len(self.errors) :
            return "<PyTangoError: %(reason)s: (%(severity)s) %(desc)s "\
                        "{%(origin)s}>"%self.errors[0].__dict__
        else :
            return "<PyTangoError: ???>"

    def dumpstr (self) :
        slist = []
        if len (self.errors) :
            slist.append ("### PyTangoError:")
            for err in self.errors :
                slist.append ("  # %(reason)s: (%(severity)s) %(desc)s "\
                            "{%(origin)s}"%err.__dict__)
            slist.append ("###")
        else :
            slist.append ("### PyTangoError (no error specified).")
        slist.append (self.tracebackstring)
        slist.append ("### end of PyTangoError.")
        return string.join (slist,"\n")

    def __repr__ (self) :
        s = "<PyTangoError"
        if len(self.errors) :
            s += self.errors[0].reason
        s += ">"
        return s

    def build_from_exception (self,einfos) :
        """Construct the object members from an exception.

        Parameters:
            -einfos  : the tuple describing the exception (from sys.exc_info()).
        """
        eclass,eobj,etb = einfos

        if isinstance (eobj,Tango.DevFailed) or isinstance (eobj,PyTangoError) :
            # Copy error informations from the DevFailed/PyTangoError.
            for err in eobj.errors : self.errors.append (err)
        else :
            # Build one error from the exception.
            tb = traceback.extract_tb (etb,1)
            curdesc = traceback.format_exception_only (eclass, eobj)
            if tb!=None and len(tb)>=1 :
                tb = tb[0]
                curorigin = "%s:line %s (in %s) %s"%(tb[0],tb[1],tb[2],tb[3])
            else :
                curorigin = "origin?"
            errsrc = Tango.DevError (eobj.__class__.__name__,Tango.ERR,\
                                     curdesc,curorigin)
            self.errors.append (errsrc)

        # common part: store exception traceback information.
        self.tracebackstring = string.join (traceback.format_exception(eclass, \
                                                                    eobj, etb))

    def makedevfailed (self) :
        return Tango.DevFailed (self.errors)
        
#===============================================================================
def tango_error (reason,severity,desc,origin=None,calldeep=0) :
    """Signal a Tango error.

    Parameters:
        -reason     : short description of the error reason.
        -severity   : error level, in Tango.WARN, Tango.ERR, Tango.PANIC.
        -desc       : long description of the error.
    (the function automatically add file and line informations, calldeep level 
    deep from caller)

    The function build a DevError with the given informations.
    If there is currently no exception, then a new PyTangoError exception is 
    created with this DevError.
    If there is currently a PyTangoError exception, then the DevError is 
    append to those in the PyTangoError errors list.
    If there is currently another exception kind, then a second DevError object 
    is created with informations from this exception, and the two DevErrors are 
    stored in a PyTangoError object.

    In all cases the function exit with returning the PyTangoError object (the
    caller must raise this object).
    """
    try :
        # Prepare DevError for the requested error.
        if origin == None :
            calldeep += 1
            frame = inspect.stack()[calldeep]  # sys._getframe (calldeep)
                # frame is a tuple with: the frame object, the filename, the 
                # line number of the current line, the function name, a list of 
                # lines of context from the source code, and the index of the 
                # current line within that list.
            #for m in dir(frame) : print m,getattr (frame,m)
            origin = "%s:line %s (in %s)"%(frame[1],frame[2],frame[3])
        err = Tango.DevError (reason, severity,desc,origin)

        # Get a DevFailed to raise.
        curexc = sys.exc_info ()
        if isinstance (curexc[1],PyTangoError) :
            exc = curexc[1]
        else :
            exc = PyTangoError ()
            if curexc != (None,None,None) :
                exc.build_from_exception (curexc)
        # Append the error and return it.
        exc.errors.append (err)
        return exc
    except :
        if DEBUG : traceback.print_exc()
        return RuntimeError("pyTango internal error in tango_error function.")

#===============================================================================
def exception_to_tango_error () :
    """Transform an exception into a tango error which is returned.

    The Tango error is a PyTangoError object, which can be modified (ie. 
    eventually add another Tango.DevError in its errors list), and must be 
    raised by caller.
    """
    try :
        excinfo = sys.exc_info ()
        if isinstance (excinfo[1],PyTangoError) : return excinfo[1]
        newe = PyTangoError ()
        newe.build_from_exception (excinfo)
        return newe
    except :
        if DEBUG : traceback.print_exc()
        return RuntimeError("pyTango internal error in "\
                            "exception_to_tango_error function.")

#===============================================================================
class DeviceConnection :
    """Manage connexion to a Tango Device, map CORBA Tango API.

    Hide device server connexion and reconnexion.
    Hide complex CORBA types.
    
    Members start by mcon.
    Methods start by con.
    """
    #---------------------------------------------------------------------------
    def __init__ (self,devname,orb=None) :
        """Construct the connection with the device server reference.

        Members:
        --------
        mcon_devname       : TangoName build from string or given at
                             construction.
        mcon_orb           : ORB to use.
        mcon_trace         : indicator to trace proxy activity.
        mcon_mutex         : mutex to protect proxy data.
        mcon_corba_device  : Tango.Device CORBA object reference (or None if not 
                             resolved).
        mcon_tango_version : version identified from the interface.
        mcon_trycount      : number of communication tentatives before raising 
                             error.
        mcon_commands      : dictionnary of commands infos indexed by command
                             name.
        """
        global gorb

        if isinstance (devname,TangoName) :
            self.mcon_devname = devname
        else :
            self.mcon_devname = TangoName (devname)

        if orb == None :
            self.mcon_orb = gorb
        else :
            self.mcon_orb = orb
            
        self.mcon_trace = 0
        self.mcon_mutex = threading.RLock ()
        self.mcon_corba_device = None   # Connected CORBA Tango Device object.
        self.mcon_tango_version = 0     # API cast found.
        self.mcon_trycount = 1          # Number of try when communicating.

        # We store commands argument references, indexed by command name, in 
        # this dictionnary, and we use it to setup right CORBA typecodes when 
        # communicating.
        self.mcon_commands = {}

        #if DEBUG :
        #    print "Created a new DeviceConnection."
        #    self.con_dump ()


    #---------------------------------------------------------------------------
    def con_dump (self,all=0) :
        """Print informations about the connection."""
        print "DeviceConnection:"
        s = self.mcon_devname.infos().split ("\n")
        s[0] = "-" + s[0]   # to have a - before the TangoName title.
        for numligne in range(len(s)) : s[numligne] = "\t"+s[numligne]
        print string.join (s,"\n")
        print "\t-Connected: ",yesno(self.mcon_corba_device!=None)
        print "\t-Tango version:",self.mcon_tango_version
        print "\t-Trace:",yesno(self.mcon_trace)
        cmds = self.mcon_commands.keys()
        cmds.sort ()
        print "\t-Retrieved commands informations:"
        for c in cmds :
            if all :
                cmd = self.mcon_commands[c]
                print "\t\tCommand name: %s"%(c,)
                if cmd == None : continue
                print "\t\t  tag          : %d"%(cmd.cmd_tag,)
                print "\t\t  in_type      : %d = %s"%\
                                    (cmd.in_type,gtcnamebynum[cmd.in_type])
                print "\t\t  in_type_desc : %s"%(cmd.in_type_desc,)
                print "\t\t  out_type     : %d = %s"%\
                                    (cmd.out_type,gtcnamebynum[cmd.out_type])
                print "\t\t  out_type_desc: %s"%(cmd.out_type_desc,)
            else :
                print "\t\t+%s"%(c,)


    #---------------------------------------------------------------------------
    def con_reset (self) :
        """INTERNAL: reset variables which are dynamically built.
        """
        try :
            self.mcon_mutex.acquire ()
            self.mcon_corba_device = None
            self.mcon_tango_version = None
            self.mcon_commands = {}
        finally :
            self.mcon_mutex.release ()

    #---------------------------------------------------------------------------
    def con_connect_device (self) :
        """INTERNAL: (re)connect the Device object if necessary and return it.

        Return:
            the CORBA Tango Device object.
        """
        try :
            self.mcon_mutex.acquire ()
            # Setup the connection with the right protocol.
            if self.mcon_corba_device == None :
                try :
                    if self.mcon_devname.protkey == "tango" :
                        self.con_connect_device_tango ()
                    elif self.mcon_devname.protkey == "corbaloc" :
                        self.con_connect_device_corbaloc ()
                    else :
                        raise tango_error ("Connexion failed",Tango.ERR,\
                                "Dont know how to handle protocol %s"%\
                                (self.mcon_devname.protkey,))
                except :
                    raise tango_error ("Connexion failed",Tango.ERR,\
                                "Cant connect device %s"%\
                                (self.mcon_devname.protkey,))

            # Retrieve supported commands (as requesting informations on an
            # unknown command lock the system).
            commands = self.mcon_corba_device.command_list_query ()
            for cmd in commands :
                self.mcon_commands[cmd.cmd_name] = cmd

            # Return the CORBA object reference for the caller.
            return self.mcon_corba_device
        finally :
            self.mcon_mutex.release ()

    #---------------------------------------------------------------------------
    def con_connect_device_tango (self) :
        """INTERNAL: connection for tango identification.

        This method must be in DeviceProxy... which know about Tango Database
        (here we only know about Device objects and simple API).
        """
        raise NotImplementedError, "To do in DeviceProxy subclass."

    #---------------------------------------------------------------------------
    def con_connect_device_corbaloc (self) :
        """INTERNAL: connection for corbaloc identification."""
        if self.mcon_corba_device != None : return self.mcon_corba_device

        try :
            objstr = "corbaloc:iiop:"+self.mcon_devname.devicehost+"/"+\
                                            self.mcon_devname.devicename
            objref = self.mcon_orb.string_to_object (objstr)
        except :
            raise tango_error ("string_to_object failed",Tango.ERR,
                        "Cant resolve string '%s'."%objstr)

        self.con_set_device (objref)
        return self.mcon_corba_device

    #---------------------------------------------------------------------------
    def con_set_device (self,objref):
        """INTERNAL: from a CORBA objref, cast it and store it.

        Parameters:
            -objref : CORBA.Object to cast to Tango.Device.
        """
        # In case of error, ensure these are properly reset.
        self.mcon_corba_device = None
        self.mcon_tango_version = 0

        try :
#             # try to see if support Device_2 API at CORBA level.
#             devref = objref._narrow (Tango.Device_2)
#             if devref != None :
#                 self.mcon_corba_device = devref
#                 self.mcon_tango_version = 2
            # Back to Device API.
            devref = objref._narrow (Tango.Device)
            if devref != None :
                self.mcon_corba_device = devref
                self.mcon_tango_version = 1
        except :
            raise tango_error ("CORBA object dont narrow to Tango.Device.",
                                Tango.ERR,str(objref))

    #----------------------------------------------------------------------------
    def con_cmd_codes (self,command) :
        """INTERNAL: get in and out args CORBA typecode.

        Parameter:
            -command: name of the command.
        Return:
            a tuple: (in typecode,out typecode)
        """
        try :
            self.mcon_mutex.acquire ()
            
            # If device has not even been connected, then connect it to retrieve
            # list of supported commands.
            if self.mcon_corba_device == None :
                    self.con_connect_device ()

            if self.mcon_commands.has_key (command) :
                if self.mcon_commands[command] == None :
                    self.mcon_commands[command] = self.command_query (command)
                cmddesc = self.mcon_commands[command]
                return cmddesc.in_type,cmddesc.out_type
            else :
                raise tango_error ("Unknown command",Tango.ERR,\
                            "The command %s is unknown by this server."%command)
        finally :
            self.mcon_mutex.release ()

    #---------------------------------------------------------------------------
    def con_secured_corba_call (self,methname,*args) :
        """INTERNAL: execute a call on the CORBA object, managing connections.

        Parameter:
            -methname   : name (string) of the method to call.
            -args       : other arguments for the call.
        Return:
            the value returned by the call.
            
        This method takes care of connecting/reconnecting to the device.
        """
        nbtry = self.mcon_trycount

        try :
            while nbtry :
                try :
                    corbadev = self.con_connect_device ()
                    if self.mcon_trace : print "Calling",methname,"on",\
                                self.mcon_devname.sourceurl
                    res = apply (getattr (corbadev,methname),args)
                    return res
                except Tango.DevFailed :
                    if DEBUG : print "Tango.DevFailed with",methname,\
                                "on",self.mcon_devname.sourceurl
                    self.con_reset ()
                    if nbtry :
                        nbtry -= 1
                    else :
                        raise
                except CORBA.COMM_FAILURE :
                    if DEBUG : print "CORBA.COM_FAILURE with",methname,\
                                "on",self.mcon_devname.sourceurl
                    self.con_reset ()
                    if nbtry :
                        nbtry -= 1
                    else :
                        raise
                except CORBA.OBJECT_NOT_EXIST :
                    if DEBUG : print "CORBA.OBJECT_NOT_EXIST with",methname,\
                                "on",self.mcon_devname.sourceurl
                    self.con_reset ()
                    if nbtry :
                        nbtry -= 1
                    else :
                        raise
                except CORBA.SystemException :
                    if DEBUG : print "CORBA.SystemException with",methname,\
                                "on",self.mcon_devname.sourceurl
                    self.con_reset ()
                    if nbtry :
                        nbtry -= 1
                    else :
                        raise
                except omniORB.LOCATION_FORWARD :
                    if DEBUG : print "omniORB.LOCATION_FORWARD with",methname,\
                                "on",self.mcon_devname.sourceurl
                    self.con_reset ()
                    if nbtry :
                        nbtry -= 1
                    else :
                        raise
                except :
                    ei = sys.exc_info ()
                    if DEBUG : print "Exception",ei[0],":",ei[1],"with",\
                                methname,"on",self.mcon_devname.sourceurl
                    raise
                # End of the "while nbtry" bloc
        except PyTangoError :
            if DEBUG : print sys.exc_info()[1].dumpstr()
            raise
        except :
            exc = exception_to_tango_error ()
            if DEBUG : print exc.dumpstr()
            raise exc


    #---------------------------------------------------------------------------
    def __getattr__ (self,name) :
        """see any unknown member as a command name to apply to server.
        """
        # Just to verify that the server support the command, try to get its
        # args type codes.
        try :
            self.con_cmd_codes (name)
        except :
            raise AttributeError,name
        # Now return handler for that command.
        return ClientRequestHandler (self,name)

    #---------------------------------------------------------------------------
    def __str__ (self) :
        """Return representation string of the Connection."""
        return "<Connection to %s>"%self.mcon_devname.sourceurl


    #===========================================================================
    #
    #                       MAPPING OF Tango.Device METHODS
    #
    #===========================================================================

    #---------------------------------------------------------------------------
    def name (self) :
        return self.con_secured_corba_call ("_get_name")

    #---------------------------------------------------------------------------
    def description (self) :
        return self.con_secured_corba_call ("_get_description")

    #---------------------------------------------------------------------------
    def state (self) :
        return self.con_secured_corba_call ("_get_state")

    #---------------------------------------------------------------------------
    def status (self) :
        return self.con_secured_corba_call ("_get_status")

    #---------------------------------------------------------------------------
    def adm_name (self) :
        return self.con_secured_corba_call ("_get_adm_name")

    #---------------------------------------------------------------------------
    def command_inout (self,command,argin=None) :
        """execute a command on a device.
        

        Parameters:
            -command: string representing the command.
            -argin: argument associated to the command, if providen.
        Return:
            The output value if the command has such one.

        The command is executed synchronously with one input parameter and one 
        output parameter.
        """
        codein,codeout = self.con_cmd_codes (command)

        # Cast the Python data into a Tango type encapsuled in a CORBA.Any.
        param = python_to_tango_any_type (codein,argin)
        
        # Do the CORBA call (manage connection, reconnection, error processing...).
        argout = self.con_secured_corba_call ("command_inout", command, param)
        
        return tango_any_to_python_type (codeout,argout)
        
    #---------------------------------------------------------------------------
    def get_attribute_config (self,names) :
        """read the configuration for a variable list of attributes from a device.

        Parameters:
            -names: list of names.
        Return:
            list of attribute configurations read
        """
        return self.con_secured_corba_call ("get_attribute_config",names)

    #---------------------------------------------------------------------------
    def set_attribute_config (self,new_conf) :
        """set the configuration for a variable list of attributes from the device.

        Parameters:
            -new_conf: list of attribute configuration to be set
        """
        self.con_secured_corba_call ("set_attribute_config",new_conf)

    #---------------------------------------------------------------------------
    def read_attributes (self,names) :
        """read a variable list of attributes from a device.

        Parameters:
            -names: list of names.
        Return:
            list of attribute values read
        """
        return self.con_secured_corba_call ("read_attributes",names)

    #---------------------------------------------------------------------------
    def write_attributes (self,values) :
        """write a variable list of attributes to a device.

        Parameters:
            -values: list of attribute values to write
        """
        self.con_secured_corba_call ("write_attributes",values)

    #---------------------------------------------------------------------------
    def ping (self) :
        """ping a device to see if it is alive."""
        self.con_secured_corba_call ("ping")

    #---------------------------------------------------------------------------
    def black_box (self,n) :
        """read list of last N commands executed by clients.

        Parameter:
            -n: number of commands to return.
        Return:
            list of command and clients
        """
        return self.con_secured_corba_call ("black_box", n)

    #---------------------------------------------------------------------------
    def info (self) :
        """return general information about object e.g. class, type, ...

        Return:
             device info
        """
        return self.con_secured_corba_call ("info")

    #---------------------------------------------------------------------------
    def command_list_query (self) :
        """query device to see what commands it supports.

        Return:
            commands and their types in a list of DevCmdInfo structures.
        """
        return self.con_secured_corba_call ("command_list_query")

    #---------------------------------------------------------------------------
    def command_query (self,command) :
        """query device to see command argument

        Parameter:
            -command: command name.
        Return:
            command and its types in a DevCmdInfo structure.
        """
        assert type(command) == types.StringType,"Command of bad type '%s'."%\
                                                                        command
        assert self.mcon_commands.has_key (command),"Command '%s' unsuported."%\
                                                                        command
        return self.con_secured_corba_call ("command_query", command)

    #===========================================================================
    #
    #                       MAPPING OF Tango.Device_2 METHODS
    #
    #===========================================================================
    def command_inout_2(self,command,argin,source) :
        return self.con_secured_corba_call ("command_inout_2",command,argin,\
                                                                        source)

    def read_attributes_2(self,names,source) :
        return self.con_secured_corba_call ("read_attributes_2",names,source)

    def get_attribute_config_2(self,names) :
        return self.con_secured_corba_call ("get_attribute_config_2",names)

    def command_list_query_2(self) :
        return self.con_secured_corba_call ("command_list_query_2")

    def command_query_2(self,command) :
        return self.con_secured_corba_call ("command_query_2",command)

    def command_inout_history_2(self,command,n) :
        return self.con_secured_corba_call ("command_inout_history_2",command,n)

    def read_attribute_history_2(self,name,n) :
        return self.con_secured_corba_call ("read_attribute_history_2",name,n)


#===============================================================================
class ClientRequestHandler :
    """Temporary handler for client commands used as methods.

    A simple class to write "dev.dothat(xxx)"
    in place of "dev.command_inout ('dothat',xxx)"
    """
    def __init__ (self,connection,command) :
        """Build the handler - store parameters."""
        self.__connection = connection
        self.__command = command
    def __call__ (self,*args) :
        return apply (self.__connection.command_inout,(self.__command,)+args)


#===============================================================================
def python_to_tango_any_type (codein,argin) :
    """Cast a Python data into a Tango type encapsuled in a CORBA.Any.

    Parameters:
        -codein : Tango code number value.
        -argin  : Python value to cast.
    """
    # Hide CORBA types parameters...
    if codein in [DEV_BOOLEAN,DEV_SHORT,DEV_LONG,DEV_USHORT,DEV_ULONG] :
        argin = int(argin)
    elif codein in [DEV_FLOAT,DEV_DOUBLE] :
        argin = float(argin)
    elif codein == DEV_STRING :
        argin = str(argin)
    elif codein == DEVVAR_CHARARRAY :
        if type (argin) in (types.ListType,types.TupleType) :
            argin = string.join ([chr(x) for x in argin])
        # Else, it can be a string as it is send to CORBA layer as a string
        # of bytes.
    elif codein in [DEVVAR_SHORTARRAY,DEVVAR_LONGARRAY,
                    DEVVAR_USHORTARRAY,DEVVAR_ULONGARRAY,] :
        argin = [ int(x) for x in argin ]
    elif codein in [DEVVAR_FLOATARRAY,DEVVAR_DOUBLEARRAY] :
        argin = [ float(x) for x in argin ]
    elif codein == DEVVAR_STRINGARRAY :
        argin = [ str(x) for x in argin ]
    elif codein in gmaptypes.keys () :
        if type (argin)==types.DictType :
            # Allow the use of dictionnary for mapping d[svalue]->lvalue|dvalue.
            d = argin
            argin = gtctypebynum[codein] ([],[]) # Construct data of right type.
            for k in d.keys () :
                getattr (argin, gmaptypes[codein]).append (d[k])
                argin.svalue.append (k)
        elif isinstance (argin,gtctypebynum[codein]) :
            pass    # Good type.
        elif hasattr (argin,gmaptypes[codein]) and hasattr (argin,"svalue") :
            pass    # Compatible type.
        elif type(argin) in (types.TupleType,types.ListType) and len(argin)==2 :
            # We have two parameters: names & values.
            names,values = argin
            argin = gtctypebynum[codein] ([],[]) # Construct data of right type.
            if type (names) in (types.TupleType,types.ListType) :
                for name in names :
                    argin.svalue.append (name)
            else :
                argin.svalue.append (names)
            if type (values) in (types.TupleType,types.ListType) :
                for value in values :
                    getattr (argin, gmaptypes[codein]).append (value)
            else :
                getattr (argin, gmaptypes[codein]).append (value)
        else :
            raise tango_error ("Bad command parameter type.", Tango.ERR, 
                "argin is not compatible with DevVar[Long|Double]StringArray.")
        # Cast in right type.
        argin.svalue = [ str(x) for x in argin.svalue ]
        if codein == DEVVAR_LONGSTRINGARRAY :
            argin.lvalue = [ int(x) for x in argin.lvalue ]
        else :
            argin.dvalue = [ float(x) for x in argin.dvalue ]

    # Prepare parameter as argin encapsulated into an Any.
    if codein == DEV_VOID :
        param = CORBA.Any(CORBA.TypeCode (CORBA.id(Tango.DevBoolean)),1)
    else :
        param = CORBA.Any(type_to_typecode(codein),argin)
        
    return param


#===============================================================================
def tango_any_to_python_type (codeout,argout) :
    """Cast a tango value encapsulated in a CORBA.Any value into a Python data.

    Parameters:
        -codeout : Tango code number value.
        -argout  : Tango CORBA.Any value to cast.
    """
    if codeout != DEV_VOID :
#         if argout.typecode() != type_to_typecode(codeout) :
#             raise RuntimeError, "command_inout returned value has not "\
#                                "expected type code."
        return argout.value()
    else :
        return None # to avoid pychecker warning.




#===============================================================================
# Imports and exports.
__all__ = [ "yesno","attproplist2dict","proplist2dict" ]


#===============================================================================
def yesno (v) :
    """Simple map to yes or no."""
    if v : return "yes"
    else : return "no"

#===============================================================================
def proplist2dict (propvallist) :
    """Transform a Tango list of properties into a dictionnary.

    The dictionnary use property names as keys, and store values for the 
    properties in lists.
    propvallist is in the form:
        [ "name",
          "number of properties",
          "name of property 0",
          "number of values of property 0",
          "value 1 of property 0",
          "value 2 of property 0",
          ...
          "value n0 of property 0",
          "name of property 1",
          "number of values of property 1",
          "value 1 of property 1",
          "value 2 of property 1",
          ...
          "value n1 of property 1",
          ...
          "name of property m",
          "number of values of property m",
          "value 1 of property m",
          "value 2 of property m",
          ...
          "value nm of property m"
        ]
    The output is in the form:
        { "name of property 0" : [ "value 1 of property 0",...,
                                                    "value n of property n0" ],
          "name of property 1" : [ "value 1 of property 1",...,
                                                    "value n of property n1" ],
          ...
          "name of property m" : [ "value 1 of property m",...,
                                                    "value n of property nm"] }
    """
    d = {}
    curprop = None
    propcount = int(propvallist[1])
    propindex = 0
    valcount = 0
    valindex = 0
    index = 2   # Index in propvallist, miss name and number of properties.
    while propindex < propcount :
        curprop = propvallist[index]
        valcount = int(propvallist[index+1])
        d[curprop] = [None]*valcount
        index += 2
        for valindex in range (valcount) :
            d[curprop][valindex] = propvallist[index]
            index += 1
        propindex += 1

    return d

#===============================================================================
def attproplist2dict (propvallist) :
    """Transform a Tango list of properties into a dictionnary.

    The dictionnary use attribute names as first level dictionary key indexing
    second level dictionnaries, and property names as second level dictionary 
    keys indexing properties value.
    propvallist is in the form:
        [ "name",
          "number of attributes",
          "name of attribute 0",
          "number of properties of attribute 0",
          "name of property 0 of attribute 0",
          "value of property 0 of attribute 0",
          "name of property 1 of attribute 0",
          "value of property 1 of attribute 0",
          ...
          "name of property t0 of attribute 0",
          "value of property t0 of attribute 0",
          "name of attribute 1",
          "number of properties of attribute 1",
          "name of property 0 of attribute 1",
          "value of property 0 of attribute 1",
          "name of property 1 of attribute 1",
          "value of property 1 of attribute 1",
          ...
          "name of property t1 of attribute 1",
          "value of property t1 of attribute 1",
          ...
          "name of attribute m",
          "number of properties of attribute m",
          "name of property 0 of attribute m",
          "value of property 0 of attribute m",
          "name of property 1 of attribute m",
          "value of property 1 of attribute m",
          ...
          "name of property tm of attribute m",
          "value of property tm of attribute m"
        ]
    The output is in the form:
        { "name of attribute 0" : { "name of property 0 of attribute 0" : 
                                        "value of property 0 of attribute 0",
                                    "name of property 1 of attribute 0" : 
                                        "value of property 1 of attribute 0",
                                    ...
                                    "name of property t0 of attribute 0" : 
                                        "value of property t0 of attribute 0" },
            ...
          "name of attribute m" : { "name of property 0 of attribute m" : 
                                        "value of property 0 of attribute m",
                                    ...
                                    "name of property tm of attribute m" : 
                                        "value of property tm of attribute m"}
          }
    """
    __pychecker__ = 'unusednames=attindex,propindex'

    d = {}
    curatt = None
    attcount = int(propvallist[1])
    propcount = 0
    index = 2   # Index in propvallist, miss name and number of attributes.
    for attindex in range(attcount) :
        curatt = propvallist[index]
        propcount = int(propvallist[index+1])
        d[curatt] = {}
        index += 2
        for propindex in range(propcount) :
            d[curatt][propvallist[index]] = propvallist[index+1]
            index += 2

    return d




#===============================================================================
class TangoName :
    """A class for Tango names (device/attribute/property).

    Manage the individual access to informations.
    
    Note: see "Tango object naming (device, attribute and property)" 
        by E.Taurel, and "Interoperable Naming Service" OMG TC Document
        available at http://www.omg.org/.
    
    Tango naming schema:
        [tango://][host:port/]device_name[/attribute][->property][#dbase=xx]
    Corbaloc naming schema:
        corbaloc:[//host:port]/object_key

    Note: for corbaloc // is a synonym for iiop: and can be replaced by iiop:.
    """
    #---------------------------------------------------------------------------
    def __init__ (self,url,attname=None,propname=None) :
        """Construct a Tango Name.
        
        For corbaloc, you can use attname and propname to specify an
        attribute and/or property.
        """
        if isinstance (url,TangoName) :
            self.__dict__.update (url.__dict__)
        else :
            self.sourceurl = url        # Keep a trace of the original string.
            self.protkey = None         # Protocol used to resolve.
            self.hostport = None        # Host and port found.
            self.databasehost = None    # Host of database to connect to.
            self.devicehost = None      # Direct host of Device object.
            self.attribute = None       # Name of an attribute.
            self.property = None        # Name of a property.
            # And now, process the URL to extract the "substantifique moelle".
            if type(url)==types.StringType : self.analyse_url (url)

        # Attribute and property names given at construction take priority
        # over those found in the url.
        if attname != None : 
            if attname : 
                self.attribute = attname
            else :
                self.attribute = None
        if propname != None : 
            if propname :
                self.property = propname
            else :
                self.property = None

    #---------------------------------------------------------------------------
    def __str__ (self) :
        return "<TangoName("+self.sourceurl+")>"

    #---------------------------------------------------------------------------
    def infos (self) :
        s = []
        s.append ("TangoName:")
        s.append ("\turl      : %s"%(self.sourceurl,))
        s.append ("\tprotocol : %s"%(self.protkey,))
        s.append ("\thost&port: %s"%(self.hostport,))
        s.append ("\tdb host  : %s"%(self.databasehost,))
        s.append ("\tdev host : %s"%(self.devicehost,))
        s.append ("\tattribute: %s"%(self.attribute,))
        s.append ("\tproperty : %s"%(self.property,))
        s.append ("\tdev name : %s"%(self.devicename,))
        return string.join (s,"\n")
        
    #---------------------------------------------------------------------------
    def analyse_url (self,url) :
        """Identifies the protocol used and analyse remaining in other methods.
        
        The remaining analyse depend on the protocol found.
        """
        #--- Find protocol.
        protsep = url.find(":")
        if protsep == -1 :
            self.protkey = "tango"
        else  :
            self.protkey = url[:protsep].lower()
            if self.protkey.find ("/") != -1 :  
                # we have a device name without protocol indication.
                self.protkey = "tango"
            else :
                url = url[protsep+1:]

        #--- Complement to protocol.
        if self.protkey == "tango" :
            self.analyse_url_tango (url)
        elif self.protkey == "corbaloc" :
            self.analyse_url_corbaloc (url)
        else :
            raise tango_error ("Unknown protocol",Tango.ERR, \
                        "Cant build TangoName with "\
                        "protocol %s."%(self.protkey,))
            
    #---------------------------------------------------------------------------
    def analyse_url_tango (self,url) :
        #[protocol://][host:port/]device_name[/attribute][->property][#dbase=xx]

        # We allow the //host:port syntax.
        if url[:2]=="//" : url = url[2:]

        #--- Find host name.
        hostsep = url.find("/")
        if hostsep != -1 and url[:hostsep].find(":") != -1 :
            self.hostport = url[:hostsep]
            url = url[hostsep+1:]
        else :
            self.hostport = os.getenv ("TANGO_HOST")
            if self.hostport == None :
                raise tango_error ("Missing parameter",Tango.ERR, \
                            "Cant build TangoName without "\
                            "hostname:port specified and without TANGO_HOST.")

        items = url.split("/")
        if not len(items) in (3,4) :
            raise tango_error ("Bad tango name",Tango.ERR, \
                        "Cant understant this name (%s)."%(self.sourceurl,))

        #--- Extract #dbase option.
        dbsep = items[-1].find ("#")
        if dbsep != -1 :
            s = items[-1][dbsep:].lower()
            if s == "#dbase=no" :
                self.devicehost = self.hostport
            elif s == "#dbase=yes" :
                self.databasehost = self.hostport
            else :
                raise tango_error ("Bad tango name",Tango.ERR, \
                            "Cant understand # option in %s."%(self.sourceurl,))
            items[-1] = items[-1][:dbsep]
        else :
            self.databasehost = self.hostport

        #--- Extract property name.
        propsep = items[-1].find ("->")
        if propsep != -1 :
            self.property = items[-1][propsep+2:]
            items[-1] = items[-1][:propsep]

        #--- Extract attribute name.
        if len(items)==4 :
            self.attribute = items[3]
            del items[3]

        #--- Find device name and reference.
        self.domain = items[0]
        self.family = items[1]
        self.member = items[2]
        self.devicename = string.join (items,"/")


    #---------------------------------------------------------------------------
    def analyse_url_corbaloc (self,url) :
        # corbaloc:[//host:port]/object_key

        #--- Extract host information.
        if url[:2] == "//" or url[:5] == "iiop:" :
            if url[:2] == "//" :
                url = url[2:]
            else :
                url = url[5:]
            items = url.split("/")
            self.hostport = items[0]
            self.devicehost = self.hostport
            url = string.join (items[1:],"/")
        else :
            self.hostport = os.getenv ("TANGO_HOST")
            if self.hostport == None :
                raise tango_error ("Missing parameter",Tango.ERR, \
                            "Cant build TangoName without "\
                            "hostname:port specified and without TANGO_HOST.")
            self.devicehost = self.hostport
            if url[0] == "/" : url = url[1:]

        #--- Extract object key.
        # Later, if servant key is normalized, then process it deeper and
        # try to extract attribute and property.
        self.devicename = url


# End of pyTango/tangocorba.py

