function result = tango_reg_test
%TANGO_REG_TEST Executes a TANGO binding regression test using the TangoTest device.
%  Change the instance name of the TANGO test device (dev_name - near line 11).
%  Obviously, a TangoTest device should be running somewhere.

%==================================================================================
% DEVICE NAME (OUR OWN TANGOTEST INSTANCE NAME GOES HERE) 
%==================================================================================
dev_name = 'tango/tangotest/3';
%==================================================================================
print_title('* TANGO BINDING FOR MATLAB *');
disp(sprintf('- regression test on %s', dev_name));
%**********************************************************************************
%==================================================================================
%                                 MISC. FUNCTIONS
%==================================================================================
%**********************************************************************************
print_title('* TESTING MISC. FUNCTIONS *');
%----------------------------------------------------------------------------------
disp('- testing tango_ping');
elapsed_usec = tango_ping(dev_name);
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('%s is up and ready [RRT: %f usec]\r', dev_name, elapsed_usec));
%----------------------------------------------------------------------------------
disp('- testing tango_admin_name');
adm_name = tango_admin_name(dev_name);
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('  `--> %s\r', adm_name));
%----------------------------------------------------------------------------------
disp('- testing tango_description');
desc = tango_description(dev_name);
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('  `--> %s\r', desc));
%----------------------------------------------------------------------------------
disp('- testing tango_status');
status = tango_status(dev_name);
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('  `--> %s\r', status));
%----------------------------------------------------------------------------------
disp('- testing tango_state');
state = tango_state(dev_name);
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('  `--> %s [%.0f]\r', state.name, state.value));
%----------------------------------------------------------------------------------
disp('- testing tango_info');
info = tango_info(dev_name);
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(info);
%----------------------------------------------------------------------------------
disp('- testing tango_idl_version');
idl_ver = tango_idl_version(dev_name);
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('  `--> passed [%s supports TANGO IDL version %d]\r', dev_name, idl_ver));
%----------------------------------------------------------------------------------
disp('- testing tango_get_source');
src = tango_get_source(dev_name);
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('  `--> %s [%.0f]\r', src.name, src.value));
%----------------------------------------------------------------------------------
disp('- testing tango_set_source (as numeric value)');
tango_set_source(dev_name, mod(src.value + 1, 2));
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
%----------------------------------------------------------------------------------
disp('- testing tango_set_source (as string value)');
tango_set_source(dev_name, 'DevICe');
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
%----------------------------------------------------------------------------------
disp('- testing tango_set_source (as string value - error expected)');
tango_set_source(dev_name, 'DUMMY-SOURCE');
if (tango_error == 0) 
  disp('  `--> ERROR (tango_error should be -1)');
  return;
end
tango_print_error_stack;
%----------------------------------------------------------------------------------
disp('- testing tango_get_timeout');
tmo = tango_get_timeout(dev_name);
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('  `--> %.0f ms\r', tmo));
%----------------------------------------------------------------------------------
disp('- testing tango_set_timeout');
tango_set_timeout(dev_name, tmo + 1);
if (tango_error == -1) 
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
%----------------------------------------------------------------------------------
disp('-testing tango_black_box');
bb = tango_black_box(dev_name, 5);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
[m, n] = size(bb);
for i = 1:n
  disp(sprintf('\t%s\r', char(bb(i))));
end

%**********************************************************************************
%==================================================================================
%                                     COMMANDS
%==================================================================================
%**********************************************************************************
print_title('* TESTING COMMANDS AND COMMAND ORIENTED FUNCTIONS *');
%==================================================================================
% DEVICE CMD QUERY
%==================================================================================
disp('-testing tango_command_query');
cmd_info = tango_command_query(dev_name, 'DevVarDoubleStringArray');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(cmd_info);
%==================================================================================
% DEVICE CMD LIST QUERY
%==================================================================================
disp('-testing tango_command_list_query');
cmd_list = tango_command_list_query(dev_name);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
[m, n] = size(cmd_list);
for i = 1:n
  disp(cmd_list(i));
end
%==================================================================================
% TEST CMD: DevVoid
%==================================================================================
disp('-testing DevVoid command');
tango_command_inout(dev_name, 'DevVoid');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevVarLongStringArray
%==================================================================================
disp('-testing DevVarLongStringArray command');
argin.lvalue = int32(0:1023);
argin.svalue = cell(1,256);
argin.svalue(1,1:256) = {'dev-var-long-string-array-test'};
argout = tango_command_inout(dev_name, 'DevVarLongStringArray', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
[argin_m, argin_n] = size(argin.lvalue);
[argout_m, argout_n] = size(argin.lvalue);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin.lvalue) ~= size(argout.lvalue)');
  return;
end
if (argin.lvalue ~= argout.lvalue)
  disp('  `--> ERROR::argin.lvalue ~= argout.lvalue');
  return;
end
[argin_m, argin_n] = size(argin.svalue);
[argout_m, argout_n] = size(argin.svalue);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin.svalue) ~= size(argout.svalue)');
  return;
end
diff = 0;
for i = 1:argin_n
  argin_str = argin.svalue{1,i};
  argout_str = argout.svalue{1,i};
  if (argin_str ~= argout_str)
    diff = diff + 1;
  end
end
if (diff ~= 0)
  disp('  `--> ERROR::argin.svalue ~= argout.svalue');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevVarDoubleStringArray
%==================================================================================
disp('-testing DevVarDoubleStringArray command');
argin.dvalue = 1024:2047;
argin.svalue = cell(1,256);
argin.svalue(1,1:256) = {'dev-var-double-string-array-test'};
argout = tango_command_inout(dev_name, 'DevVarDoubleStringArray', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
[argin_m, argin_n] = size(argin.dvalue);
[argout_m, argout_n] = size(argin.dvalue);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin.dvalue) ~= size(argout.dvalue)');
  return;
end
if (argin.dvalue ~= argout.dvalue)
  disp('  `--> ERROR::argin.dvalue ~= argout.dvalue');
  return;
end
[argin_m, argin_n] = size(argin.svalue);
[argout_m, argout_n] = size(argin.svalue);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin.svalue) ~= size(argout.svalue)');
  return;
end
diff = 0;
for i = 1:argin_n
  argin_str = argin.svalue{1,i};
  argout_str = argout.svalue{1,i};
  if (argin_str ~= argout_str)
    diff = diff + 1;
  end
end
if (diff ~= 0)
  disp('  `--> ERROR::argin.svalue ~= argout.svalue');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevVarStringArray
%==================================================================================
disp('-testing DevVarStringArray command');
argin = cell(1,256);
argin(1,1:256) = {'dev-var-string-array-test'};
argout = tango_command_inout(dev_name, 'DevVarStringArray', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
diff = 0;
for i = 1:argin_n
  if (argin{i} ~= argout{i})
    diff = diff + 1;
  end
end
if (diff ~= 0)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevVarDoubleArray
%==================================================================================
disp('-testing DevVarDoubleArray command');
argin = 0:0.1:128;
argout = tango_command_inout(dev_name, 'DevVarDoubleArray', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevVarFloatArray
%==================================================================================
disp('-testing DevVarFloatArray command');
argin = single(-128:0.1:0);
argout = tango_command_inout(dev_name, 'DevVarFloatArray', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevVarULongArray
%==================================================================================
disp('-testing DevVarULongArray command');
argin = uint32(0:256);
argout = tango_command_inout(dev_name, 'DevVarULongArray', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevVarLongArray
%==================================================================================
disp('-testing DevVarLongArray command');
argin = int32(-1024:0);
argout = tango_command_inout(dev_name, 'DevVarLongArray', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevVarUShortArray
%==================================================================================
disp('-testing DevVarUShortArray command');
argin = uint16(0:1024);
argout = tango_command_inout(dev_name, 'DevVarUShortArray', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevVarShortArray
%==================================================================================
disp('-testing DevVarShortArray command');
argin = int16(-1024:0);
argout = tango_command_inout(dev_name, 'DevVarShortArray', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevVarCharArray
%==================================================================================
disp('-testing DevVarCharArray command');
argin = char(0:256);
argout = tango_command_inout(dev_name, 'DevVarCharArray', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevString
%==================================================================================
disp('-testing DevString command');
argin = 'dev-string';
argout = tango_command_inout(dev_name, 'DevString', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevBoolean
%==================================================================================
disp('-testing DevBoolean command');
argin = uint16(1);
argout = tango_command_inout(dev_name, 'DevBoolean', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevUShort
%==================================================================================
disp('-testing DevUShort command');
argin = uint16(2^16 - 1);
argout = tango_command_inout(dev_name, 'DevUShort', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevShort
%==================================================================================
disp('-testing DevShort command');
argin = int16(-2^16);
argout = tango_command_inout(dev_name, 'DevShort', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevULong
%==================================================================================
disp('-testing DevULong command');
argin = uint32(2^32 -1);
argout = tango_command_inout(dev_name, 'DevULong', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevLong
%==================================================================================
disp('-testing DevLong command');
argin = int32(-2^32);
argout = tango_command_inout(dev_name, 'DevLong', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevFloat
%==================================================================================
disp('-testing DevFloat command');
argin = single(-pi);
argout = tango_command_inout(dev_name, 'DevFloat', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%==================================================================================
% TEST CMD: DevDouble
%==================================================================================
disp('-testing DevDouble command');
argin = pi;
argout = tango_command_inout(dev_name, 'DevDouble', argin);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[argin_m, argin_n] = size(argin);
[argout_m, argout_n] = size(argout);
if (argin_m ~= argout_m | argin_n ~= argout_n)
  disp('  `--> ERROR::size(argin) ~= size(argout)');
end
if (argin ~= argout)
  disp('  `--> ERROR::argin ~= argout');
  return;
end
disp('  `--> passed');
%**********************************************************************************
%==================================================================================
%                                 ATTRIBUTES
%==================================================================================
%**********************************************************************************
print_title('* TESTING ATTRIBUTE ORIENTED FUNCTIONS *');
%**********************************************************************************
% DEVICE ATTRIBUTES - GET ATTRIBUTE LIST
%**********************************************************************************
disp('-testing tango_get_attribute_list');
attr_list = tango_get_attribute_list(dev_name);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
[m, n] = size(attr_list);
for i = 1:n
  disp(attr_list(i));
end
%**********************************************************************************
% DEVICE ATTRIBUTES - READ ATTRIBUTE CONFIG
%**********************************************************************************
disp('-testing tango_get_attribute_config (get config. for one attribute)');
attr_config = tango_get_attribute_config(dev_name, 'short_scalar');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(attr_config);
%**********************************************************************************
% DEVICE ATTRIBUTES - READ ATTRIBUTES CONFIG
%**********************************************************************************
disp('-testing tango_get_attributes_config (get config. for several attributes)');
attr_list = {'short_scalar', 'long_scalar', 'double_scalar', ...
             'short_spectrum', 'long_spectrum', 'double_spectrum', ...
             'short_image', 'long_image', 'double_image'};
attr_config = tango_get_attributes_config(dev_name, attr_list);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
[m, n] = size(attr_config);
for i = 1:n
  disp(attr_config(i));
end
%**********************************************************************************
% DEVICE ATTRIBUTES - WRITE ATTRIBUTE CONFIG
%**********************************************************************************
disp('-testing tango_set_attribute_config');
attr_config = tango_get_attribute_config(dev_name, 'short_scalar');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> original <short_scalar> config');
disp(attr_config);
%change some values
attr_config.description = 'a simple short scalar attribute';
attr_config.label = 'sc';
attr_config.unit = 'A.U.';
attr_config.standard_unit = attr_config(1).unit;
attr_config.display_unit = attr_config(1).unit;
attr_config.format = 'none';
attr_config.min_value = '-16384';
attr_config.max_value = '16384';
tango_set_attributes_config(dev_name, attr_config);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return; 
end
disp('  `--> passed');
disp('  `--> modified <short_scalar> config');
attr_config = tango_get_attribute_config(dev_name, 'short_scalar');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(attr_config);
%**********************************************************************************
% DEVICE ATTRIBUTES - READ/WRITE INDIVIDUALLY EACH ATTRIBUTE
%**********************************************************************************
print_title('* READING/WRITTING ATTRIBUTES (INDIVIDUALLY) *');
%==================================================================================
% READ/WRITE ATTRIBUTE: short_scalar (read/write)
%==================================================================================
disp('-writing short_scalar');
val_in = int16(-4096);
tango_write_attribute(dev_name, 'short_scalar', val_in);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('  `--> passed (wrote %d)', double(val_in)));
disp('-reading short_scalar');
val_out = tango_read_attribute(dev_name, 'short_scalar');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(val_out);
if (val_in(1) ~= val_out.value(2)) 
  disp('  `--> ERROR');
  return;
end
%==================================================================================
% READ/WRITE ATTRIBUTE: long_scalar (read/write)
%==================================================================================
disp('-writing long_scalar');
val_in = int32(123456);
tango_write_attribute(dev_name, 'long_scalar', val_in);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('  `--> passed (wrote %d)', double(val_in)));
disp('-reading long_scalar');
val_out = tango_read_attribute(dev_name, 'long_scalar');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(val_out);
if (val_in(1) ~= val_out.value(2)) 
  disp('  `--> ERROR');
  return;
end
%==================================================================================
% READ/WRITE ATTRIBUTE: double_scalar (read/write)
%==================================================================================
disp('-writing double_scalar');
val_in = pi;
tango_write_attribute(dev_name, 'double_scalar', pi);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(sprintf('  `--> passed (wrote %d)', val_in));
disp('-reading double_scalar');
val_out = tango_read_attribute(dev_name, 'double_scalar');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(val_out);
if (val_in(1) ~= val_out.value(2)) 
  disp('  `--> ERROR');
  return;
end
%==================================================================================
% READ ATTRIBUTE: short_spectrum
%==================================================================================
disp('-reading short_spectrum');
val = tango_read_attribute(dev_name, 'short_spectrum');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(val);
%==================================================================================
% READ ATTRIBUTE: long_spectrum
%==================================================================================
disp('-reading long_spectrum');
val = tango_read_attribute(dev_name, 'long_spectrum');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(val);
%==================================================================================
% READ ATTRIBUTE: double_spectrum
%==================================================================================
disp('-reading double_spectrum');
val = tango_read_attribute(dev_name, 'double_spectrum');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(val);
%==================================================================================
% READ ATTRIBUTE: short_image
%==================================================================================
disp('-reading short_image');
val = tango_read_attribute(dev_name, 'short_image');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(val);
%==================================================================================
% READ ATTRIBUTE: long_image
%==================================================================================
disp('-reading long_image');
val = tango_read_attribute(dev_name, 'long_image');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(val);
%==================================================================================
% READ ATTRIBUTE: double_image
%==================================================================================
disp('-reading double_image');
val = tango_read_attribute(dev_name, 'double_image');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(val);
%**********************************************************************************
% DEVICE ATTRIBUTES - READ/WRITE ALL ATTRIBUTES IN ONE CALL
%**********************************************************************************
print_title('* READING/WRITTING ATTRIBUTES (GLOBALLY) *');
%==================================================================================
% WRITE ALL ATTRIBUTES
%==================================================================================
disp('-testing tango_write_attributes (writting several attributes in one call)');
attr_name_val_list(1).name  = 'short_scalar';
attr_name_val_list(1).value = int16(4096);
attr_name_val_list(2).name  = 'long_scalar';
attr_name_val_list(2).value = int32(654321);
attr_name_val_list(3).name  = 'double_scalar';
attr_name_val_list(3).value = -pi;
tango_write_attributes(dev_name, attr_name_val_list);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
else
  disp('  `--> passed');
end
%==================================================================================
% READ ALL ATTRIBUTES
%==================================================================================
disp('-testing tango_read_attributes (reading all attributes in one call)');
attr_list = {'short_scalar', 'long_scalar', 'double_scalar', ...
             'short_spectrum', 'long_spectrum', 'double_spectrum', ...
             'short_image', 'long_image', 'double_image'};
val = tango_read_attributes(dev_name, attr_list);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
[m, n] = size(val);
for i = 1:n
  disp(val(i));
end
%**********************************************************************************
%==================================================================================
%                                   POLLING
%==================================================================================
%**********************************************************************************
print_title('* TESTING POLLING ORIENTED FUNCTIONS *');
%==================================================================================
% CMD POLLING
%==================================================================================
disp('-testing tango_poll_command');
tango_poll_command(dev_name, 'DevVoid', 500);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
%==================================================================================
% CHECK CMD POLLING
%==================================================================================
disp('-testing tango_is_command_polled');
polled = tango_is_command_polled(dev_name, 'DevVoid');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
if (polled ~= 1)
  disp('  `--> ERROR::cmd DevVoid should be polled');
  return;
end
disp('  `--> passed');
%==================================================================================
% CMD POLLING PERIOD
%==================================================================================
disp('-testing tango_get_command_poll_period');
pp = tango_get_command_poll_period(dev_name, 'DevVoid');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
if (pp ~= 500)
  disp(sprintf('  `--> ERROR::polling period should be 500 ms - got %.2fms', pp));
  return;
end
disp('  `--> passed');
%==================================================================================
% ATTR POLLING
%==================================================================================
disp('-testing tango_poll_attribute');
tango_poll_attribute(dev_name, 'short_scalar', 250);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
%==================================================================================
% CHECK ATTR POLLING
%==================================================================================
disp('-testing tango_is_attribute_polled');
polled = tango_is_attribute_polled(dev_name, 'short_scalar');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
if (polled ~= 1)
  disp('  `--> ERROR::attr short_scalar should be polled');
  return;
end
disp('  `--> passed');
%==================================================================================
% ATTR POLLING PERIOD
%==================================================================================
disp('-testing tango_get_attribute_poll_period');
pp = tango_get_attribute_poll_period(dev_name, 'short_scalar');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
if (pp ~= 250)
  disp(sprintf('  `--> ERROR::polling period should be 250 ms - got %.2fms', pp));
  return;
end
disp('  `--> passed');
%==================================================================================
% WAIT A SECOND FOR POLLING BUFFERS TO BE FILLED
%==================================================================================
disp('-waiting for polling buffers to be filled');
pause(1);
%==================================================================================
% POLLING STATUS
%==================================================================================
disp('-testing tango_polling_status');
ps = tango_polling_status(dev_name);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
[m, n] = size(ps);
for i = 1:n
  disp(ps(i));
end
%==================================================================================
% DEVICE CMD HISTORY
%==================================================================================
disp('-testing tango_command_history');
hist = tango_command_history(dev_name, 'DevVoid', 5);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  %return;
else
  disp('  `--> passed');
  [m, n] = size(hist);
  for i = 1:n
    disp(hist(i));
  end
end
%==================================================================================
% DEVICE ATTRIBUTE HISTORY
%==================================================================================
disp('-testing tango_attribute_history');
hist = tango_attribute_history(dev_name, 'short_scalar', 5);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  %return;
else
  disp('  `--> passed');
  [m, n] = size(hist);
  for i = 1:n
    disp(sprintf('\t-Attr.History(%d) ----------------\r', i));
    disp(hist(i));
    disp(sprintf('\t-Attr.History(%d).value ----------\r', i));
    disp(hist(i).value);  
  end
end
%==================================================================================
% STOP POLLING CMD 
%==================================================================================
disp('-testing tango_stop_poll_command');
tango_stop_poll_command(dev_name, 'DevVoid');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
polled = tango_is_command_polled(dev_name, 'DevVoid');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
if (polled ~= 0)
  disp('  `--> ERROR:: cmd DevVoid should NOT be polled');
  return;
end
disp('  `--> passed');
%==================================================================================
% STOP POLLING ATTR 
%==================================================================================
disp('-testing tango_stop_poll_attribute');
tango_stop_poll_attribute(dev_name, 'short_scalar');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
polled = tango_is_attribute_polled(dev_name, 'short_scalar');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
if (polled ~= 0)
  disp('  `--> ERROR::attr short_scalar should NOT be polled');
  return;
end
disp('  `--> passed');
%**********************************************************************************
%==================================================================================
%                                   PROPERTIES
%==================================================================================
%**********************************************************************************
print_title('* TESTING PROPERTY ORIENTED FUNCTIONS *');
%==================================================================================
% GET SINGLE PROPERTY
%==================================================================================
disp('-testing tango_get_property');
prop = tango_get_property(dev_name, 'mthreaded_impl');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
disp(prop);
%==================================================================================
% GET SEVERAL PROPERTIES
%==================================================================================
disp('-testing tango_get_properties');
prop_list = {'mthreaded_impl', 'sleep_period', 'polled_attr'};
prop = tango_get_properties(dev_name, prop_list);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
[m, n] = size(prop);
for i = 1:n
  disp(prop(i));
end
%==================================================================================
% PUT SINGLE PROPERTY
%==================================================================================
disp('-testing tango_put_property');
tango_put_property(dev_name, 'dummy_0', {'dummy_0_value_1', 'dummy_0_value_2'});
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
prop = tango_get_property(dev_name, 'mthreaded_impl');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp(prop);
%==================================================================================
% PUT SEVERAL PROPERTIES
%==================================================================================
disp('-testing tango_put_properties');
clear prop
prop.name = 'mthreaded_impl';
prop.value = {'1'};
prop(2).name = 'sleep_period';
prop(2).value = {'100'};
prop(2).name = 'dummy_1';
prop(2).value = {'dummy_1_value_1', 'dummy_1_value_2', 'dummy_1_value_3'};
prop(3).name = 'dummy_2';
prop(3).value = {'dummy_2_value_1', 'dummy_2_value_2'};
prop(4).name = 'dummy_3';
prop(4).value = {'dummy_3_value_1'};
tango_put_properties(dev_name, prop);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
clear prop
prop = {'mthreaded_impl', 'sleep_period'};
prop = tango_get_properties(dev_name, prop);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[m, n] = size(prop);
for i = 1:n
  disp(prop(i));
end
%==================================================================================
% DELETE SINGLE PROPERTY
%==================================================================================
disp('-testing tango_del_property');
tango_del_property(dev_name, 'dummy_0');
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
clear prop
prop = {'dummy_0', 'dummy_1', 'dummy_2', 'dummy_3'};
prop = tango_get_properties(dev_name, prop);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[m, n] = size(prop);
for i = 1:n
  disp(prop(i));
end
%==================================================================================
% DELETE SEVERAL PROPERTIES
%==================================================================================
disp('-testing tango_del_properties');
tango_del_properties(dev_name, {'dummy_1', 'dummy_2', 'dummy_3'});
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
disp('  `--> passed');
clear prop
prop = {'dummy_1', 'dummy_2', 'dummy_3'};
prop = tango_get_properties(dev_name, prop);
if (tango_error == -1)
  disp('  `--> ERROR');
  tango_print_error_stack;
  return;
end
[m, n] = size(prop);
for i = 1:n
  disp(prop(i));
end
%==================================================================================
% END OF REGRESSION TEST
%==================================================================================
return;

%**********************************************************************************
% UTILITIES
%**********************************************************************************
function print_title (title)
[m, n] = size(title);
l(1:n) = char('*');
disp(l);
disp(title);
disp(l);  
return;
