// ============================================================================
//
// = CONTEXT
//   TANGO binbing for Matlab
//
// = FILENAME
//   MexFile.cpp
//
// = AUTHOR
//   Nicolas Leclercq - SOLEIL
//
// ============================================================================

//=============================================================================
// DEPENDENCIEs
//=============================================================================
#include "StandardHeader.h"
#include "TangoBinding.h"

//=============================================================================
// STATIC MEMBERS
//=============================================================================
MexArgs MexFile::args;
int MexFile::initialized = 0;
const char *MexFile::kMEX_VERSION = "1.1.0";

//=============================================================================
// MexFile::init
//=============================================================================
int MexFile::init (void) 
{
  // return if already initialized
  if (MexFile::initialized) {
    return kNoError;
  }

  // initialize utilities
  if (MexUtils::init() == kError) {
    //-- can't use error stack since it may not be initialized
    return kError;
  }

  // initialize the TANGO binding class
  if (TangoBinding::init() == -1) {
    ::mexPrintf("mexFunction::TangoBinding initialization failed");
    return kError;
  }

  // set cleanup function (called when mex-file is discarded from memory)
  ::mexAtExit(MexFile::cleanup); 

  // initialization done 
  MexFile::initialized = 1;

  MEX_MSG("MEX-File initialized");

  return kNoError;
}

//=============================================================================
// MexFile::cleanup
//=============================================================================
void MexFile::cleanup (void) 
{
  if (MexFile::initialized) {
    MEX_MSG("mex-file is about to be cleared::executing cleanup\r");
    TangoBinding::cleanup();
    MexUtils::cleanup();
    MexFile::initialized = 0;
  }
}

//=============================================================================
// MexFile::version
//=============================================================================
void MexFile::version (void) 
{
  //- Create a 1-by-n char array
  mxArray* mex_version = ::mxCreateString(kMEX_VERSION);
  if (mex_version == 0) {
    MEX_UTILS->set_error("out of memory",
                         "mxCreateString failed",
                         "MexFile::version");
    _SET_DEFAULT_PRHS; 
  }
  //- Set output arg
  if (MEX_ARGS->set_output_array(k1ST_ARG, mex_version) == kError) {
    _SET_DEFAULT_PRHS; 
  }
}

//=============================================================================
// MexFile::exec
//=============================================================================
void MexFile::exec (int nlhs, mxArray **plhs, int nrhs, const mxArray **prhs) 
{
  // check initialization
  if (MexFile::initialized == 0 && MexFile::init() == kError) {
    ::mexErrMsgTxt("initialization failed");
    //-- no <return> needed here (done by mexErrMsgTxt)
  }

  //- check num of input arg - at least 1 arg expected (cmd-id).
  if (nrhs == 0) {
    ::mexErrMsgTxt("no input argument specified");
  }

  //- check first arg - must be a 16 bits integer (func_id).
  if (mxIsInt16(prhs[0]) == false) {
    ::mexErrMsgTxt("first mex function argument must be a 16 bits integer");
  }

  //- get cmd_id
  short cmd_id = *(static_cast<short*>(::mxGetData(prhs[0])));

  //- store mex function in/out arguments for later retrieval
  MEX_ARGS->set(nlhs, plhs, nrhs, prhs);

  //- reset global error code and stack
  if (cmd_id != MexFile::ERROR_CODE && cmd_id != MexFile::ERROR_STACK) {
    MEX_UTILS->reset_error();
  }

  //- exec cmd
  switch (cmd_id) 
  {
    //-- MEX_VERSION
    case MexFile::MEX_VERSION:
      MexFile::version();
      break;
    //-- TANGO_VERSION
    case MexFile::TANGO_VERSION:
      TANGO_BINDING->tango_version();
      break;
    //-- EXPORT_ERROR_CODE
    case MexFile::ERROR_CODE:
      MEX_UTILS->error_code();
      break;
    //-- EXPORT_ERROR_STACK
    case MexFile::ERROR_STACK:
      MEX_UTILS->error_stack();
      break;
    //-- OPEN_DEVICE
    case MexFile::OPEN_DEVICE:
      TANGO_BINDING->open_device();
      break;
    //-- CLOSE_DEVICE
    case MexFile::CLOSE_DEVICE:
      TANGO_BINDING->close_device();
      break;
    //-- COMMAND_IN_OUT
    case MexFile::COMMAND_IN_OUT:
      TANGO_BINDING->command_in_out();
      break;
    //-- READ_ATTRIBUTE
    case MexFile::READ_ATTRIBUTE:
      TANGO_BINDING->read_attribute();
      break;
    //-- READ_ATTRIBUTES
    case MexFile::READ_ATTRIBUTES:
      TANGO_BINDING->read_attributes();
      break;
    //-- WRITE_ATTRIBUTE
    case MexFile::WRITE_ATTRIBUTE:
      TANGO_BINDING->write_attribute();
      break;
    //-- WRITE_ATTRIBUTES
    case MexFile::WRITE_ATTRIBUTES:
      TANGO_BINDING->write_attributes();
      break;
    //-- STATUS
    case MexFile::STATUS:
      TANGO_BINDING->status();
      break;
    //-- PING
    case MexFile::PING:
      TANGO_BINDING->ping();
      break;
    //-- INFO
    case MexFile::INFO:
      TANGO_BINDING->info();
      break;
    //-- STATE
    case MexFile::STATE:
      TANGO_BINDING->state();
      break;
    //-- DESCRIPTION
    case MexFile::DESCRIPTION:
      TANGO_BINDING->description();
      break;
    //-- ADMIN_NAME
    case MexFile::ADMIN_NAME:
      TANGO_BINDING->admin_name();
      break;
    //-- GET_ATTR_CONFIG
    case MexFile::GET_ATTR_CONFIG:
      TANGO_BINDING->get_attr_config();
      break;
    //-- SET_ATTR_CONFIG
    case MexFile::SET_ATTR_CONFIG:
      TANGO_BINDING->set_attr_config();
      break;
    //-- BLACK_BOX
    case MexFile::BLACK_BOX:
      TANGO_BINDING->black_box();
      break;
    //-- CMD_LIST_QUERY
    case MexFile::CMD_LIST_QUERY:
      TANGO_BINDING->command_list_query();
      break;
    //-- CMD_QUERY
    case MexFile::CMD_QUERY:
      TANGO_BINDING->command_query();
      break; 
    //-- SET_TIMEOUT
    case MexFile::SET_TIMEOUT:
      TANGO_BINDING->set_timeout();
      break; 
    //-- GET_TIMEOUT
    case MexFile::GET_TIMEOUT:
      TANGO_BINDING->get_timeout();
      break;
    //-- SET_SOURCE
    case MexFile::SET_SOURCE:
      TANGO_BINDING->set_source();
      break; 
    //-- GET_SOURCE
    case MexFile::GET_SOURCE:
      TANGO_BINDING->get_source();
      break; 
    //-- COMMAND_HISTORY
    case MexFile::COMMAND_HISTORY:
      TANGO_BINDING->command_history();
      break; 
    //-- ATTRIBUTE_LIST
    case ATTRIBUTE_LIST:
      TANGO_BINDING->get_attribute_list();
      break; 
    //-- ATTRIBUTE_HISTORY
    case MexFile::ATTRIBUTE_HISTORY:
      TANGO_BINDING->attribute_history();
      break; 
    //-- POLLING_STATUS
    case MexFile::POLLING_STATUS:
      TANGO_BINDING->polling_status();
      break;
    //-- IS_CMD_POLLED
    case MexFile::IS_CMD_POLLED:
      TANGO_BINDING->is_command_polled();
      break;
    //-- IS_ATTR_POLLED
    case MexFile::IS_ATTR_POLLED:
      TANGO_BINDING->is_attribute_polled();
      break; 
    //-- POLL_CMD
    case MexFile::POLL_CMD:
      TANGO_BINDING->poll_command();
      break;
    //-- POLL_ATTR
    case MexFile::POLL_ATTR:
      TANGO_BINDING->poll_attribute();
      break;
    //-- STOP_POLL_CMD
    case MexFile::STOP_POLL_CMD:
      TANGO_BINDING->stop_poll_command();
      break;
    //-- STOP_POLL_ATTR
    case MexFile::STOP_POLL_ATTR:
      TANGO_BINDING->stop_poll_attribute();
      break;
    //-- GET_CMD_POLL_PERIOD
    case MexFile::GET_CMD_POLL_PERIOD:
      TANGO_BINDING->get_command_poll_period();
      break;
    //-- GET_ATTR_POLL_PERIOD
    case MexFile::GET_ATTR_POLL_PERIOD:
      TANGO_BINDING->get_attribute_poll_period();
      break;
    //-- IDL_VERSION
    case IDL_VERSION:
      TANGO_BINDING->get_idl_version();
      break;
    //-- GET_PROPERTIES
    case GET_PROPERTIES:
      TANGO_BINDING->get_properties();
      break;
    //-- PUT_PROPERTIES
    case PUT_PROPERTIES:
      TANGO_BINDING->put_properties();
      break;
    //-- DEL_PROPERTIES
    case DEL_PROPERTIES:
      TANGO_BINDING->del_properties();
      break;
    //-- GET_PROPERTY
    case GET_PROPERTY:
      TANGO_BINDING->get_property();
      break;
    //-- PUT_PROPERTY
    case PUT_PROPERTY:
      TANGO_BINDING->put_property();
      break;
    //-- DEL_PROPERTY
    case DEL_PROPERTY:
      TANGO_BINDING->del_property();
      break;
    //-- DEFAULT
    default:
      MEX_UTILS->set_error("invalid command tag specified", 
                           "unknown command tag",
                           "mexFunction");
      _SET_DEFAULT_PRHS; 
      break;
  }
}

