// ============================================================================
//
// = CONTEXT
//   Tango Generic Client for Matlab
//
// = FILENAME
//   MexUtils.cpp
//
// = AUTHOR
//   Nicolas Leclercq - SOLEIL
//
// ============================================================================

//=============================================================================
// DEPENDENCIEs
//=============================================================================
#include <cstdarg>
#include "StandardHeader.h"

//=============================================================================
// INLINED CODE
//=============================================================================
#if !defined (_MEX_INLINE_)
  #include "MexUtils.i"
#endif 

//=============================================================================
// #DEFINEs                                                 
//=============================================================================
#if defined (_MEX_DEBUG_)
# define INLINE_BUF_SIZE 1024
#else
# define INLINE_BUF_SIZE  512
#endif

//=============================================================================
// STATIC MEMBERS
//=============================================================================
MexUtils* MexUtils::instance_ = 0;

//=============================================================================
// LOCAL CONSTS
//=============================================================================
static const char *lkTANGO_GERR_CODE  = "tango_error";
static const char *lkTANGO_GERR_STACK = "tango_error_stack";
static const char *lkTANGO_ERR_CODE   = "error";
static const char *lkTANGO_ERR_DESC   = "desc";
static const char *lkTANGO_ERR_ORIG   = "origin";
static const char *lkTANGO_ERR_REAS   = "reason";

//=============================================================================
// MexUtils::init
//=============================================================================
int MexUtils::init (void)
{
  if (MexUtils::instance_ != 0)
    return 0;

  MexUtils::instance_ = new MexUtils;

  return (MexUtils::instance_) ? 0 : -1;
}

//=============================================================================
// MexUtils::cleanup
//=============================================================================
void MexUtils::cleanup (void)
{
  if (MexUtils::instance_) {
    delete MexUtils::instance_;
    MexUtils::instance_ = 0;
  }
}

//=============================================================================
// MexUtils::MexUtils                                                
//=============================================================================
MexUtils::MexUtils ()
{
  this->global_err_code_ = ::mxCreateDoubleMatrix(1, 1, ::mxREAL);
  if (this->global_err_code_) {
    ::mxSetName(this->global_err_code_, lkTANGO_GERR_CODE);
    ::mexMakeArrayPersistent(this->global_err_code_);
    this->reset_error();
  }
}

//=============================================================================
// MexUtils::~MexUtils                                                
//=============================================================================
MexUtils::~MexUtils (void)
{
  if (this->global_err_code_) {
    ::mxDestroyArray(this->global_err_code_);
    this->global_err_code_ = 0;
  }
}

//=============================================================================
// MexUtils::notify                                                 
//=============================================================================
void MexUtils::notify (const char *format, ...)
{
  static char notify_str[INLINE_BUF_SIZE];

	char *buf = notify_str;

  size_t fmtlen = ::strlen(format) + 1;

  if (fmtlen > INLINE_BUF_SIZE)  
    buf = new char[fmtlen];
    
  va_list argptr;
  va_start(argptr, format);
    int len = ::vsprintf(buf, format, argptr);
  va_end(argptr);

	if (len > 0) {
    ::mexPrintf(buf);
	} 
	
  if (buf != notify_str) {  
    delete[] buf;
  }
}

//=============================================================================
// MexUtils::set_global_error_code                                       
//=============================================================================
int MexUtils::set_global_error_code (int _err_code)
{
  if (this->global_err_code_ == 0) {
    return kError;
  }
  ::mxGetPr(this->global_err_code_)[0] = _err_code;
  return kNoError;
}

//=============================================================================
// MexUtils::set_error                                       
//=============================================================================
void MexUtils::set_error (const Tango::DevFailed &e)
{
  this->reset_error();
  this->set_global_error_code();
  long nb_err = e.errors.length();
  this->error_.errors.length(nb_err);
  for (int i = 0; i < nb_err; i++) {
    this->error_.errors[i].severity = e.errors[i].severity;
    this->error_.errors[i].reason = CORBA::string_dup(e.errors[i].reason);
    this->error_.errors[i].desc = CORBA::string_dup(e.errors[i].desc);
    this->error_.errors[i].origin = CORBA::string_dup(e.errors[i].origin);
  }
}

//=============================================================================
// MexUtils::set_error                                       
//=============================================================================
void MexUtils::set_error (const char *_r, 
                          const char *_d, 
                          const char *_o,
                          Tango::ErrSeverity _s) 
{
  this->reset_error();
  this->set_global_error_code();
  this->error_.errors.length(1);
  this->error_.errors[0].severity = _s;
  this->error_.errors[0].reason = CORBA::string_dup(_r);
  this->error_.errors[0].desc = CORBA::string_dup(_d);
  this->error_.errors[0].origin = CORBA::string_dup(_o);
}

//=============================================================================
// MexUtils::push_error                                       
//=============================================================================
void MexUtils::push_error (const char *_r, 
                           const char *_d, 
                           const char *_o,
                           Tango::ErrSeverity _s) 
{
  long nerr = this->error_.errors.length();
  this->error_.errors.length(nerr + 1);
  this->error_.errors[nerr].severity = _s;
  this->error_.errors[nerr].reason = CORBA::string_dup(_r);
  this->error_.errors[nerr].desc = CORBA::string_dup(_d);
  this->error_.errors[nerr].origin = CORBA::string_dup(_o);
}

//=============================================================================
// MexUtils::push_error                                       
//=============================================================================
void MexUtils::push_error (const Tango::DevFailed &e)
{
  long add_nb_err = e.errors.length();
  long cur_nb_err = this->error_.errors.length();
  long total_err = cur_nb_err + add_nb_err;
  this->error_.errors.length(total_err);
  int i, j;
  for (i = cur_nb_err, j = 0; i < total_err; i++, j++) {
    this->error_.errors[i].severity = e.errors[j].severity;
    this->error_.errors[i].reason = CORBA::string_dup(e.errors[j].reason);
    this->error_.errors[i].desc = CORBA::string_dup(e.errors[j].desc);
    this->error_.errors[i].origin = CORBA::string_dup(e.errors[j].origin);
  }
}

//=============================================================================
// MexUtils::error_code                                       
//=============================================================================
int MexUtils::error_code (void)
{
  if (this->global_err_code_ == 0) {
    return kError;
  }
  if (MEX_ARGS->set_output_array(0, this->global_err_code_) == kError) {
    return kError;
  }
  return kNoError;
}

//=============================================================================
// MexUtils::error_stack                                    
//=============================================================================
int MexUtils::error_stack (void)
{
  // exports (on demand) the error stack as an array of struct
  // each struct mapps a Tango::DevError (see tango.h)

  // get number of errors in the stack
  long nerr = this->error_.errors.length();
  // create a 1-by-n array of structs (each containing 4 fields)
  int dims[2];
  dims[0] = 1;
  dims[1] = (int)nerr;
  const char *field_names[] = {"reason", "desc", "origin", "severity", "severity_str"};
  mxArray *struct_array = ::mxCreateStructArray(2, dims, 5, field_names);
  if (struct_array == 0) {
    MEX_ERROR("mxCreateStructArray failed - could not export TANGO error stack");
    return kError;
  }
  // set array name
  ::mxSetName(struct_array, lkTANGO_GERR_STACK);
  // populate the array
  mxArray *severity_value;
  for (int i = 0; i < nerr; i++) {
    // set each string field of the ith struct
    // note: the field number indices are zero based (reason->0, desc->1, ...)
    ::mxSetFieldByNumber(struct_array, i, 0, ::mxCreateString(this->error_.errors[i].reason.in()));
    ::mxSetFieldByNumber(struct_array, i, 1, ::mxCreateString(this->error_.errors[i].desc.in()));
    ::mxSetFieldByNumber(struct_array, i, 2, ::mxCreateString(this->error_.errors[i].origin.in()));
    // set the severity field 
    severity_value = ::mxCreateDoubleMatrix(1,1,mxREAL);
    ::mxGetPr(severity_value)[0] = this->error_.errors[i].severity;
    ::mxSetFieldByNumber(struct_array, i, 3, severity_value);
    // set the severity_str field 
    switch (this->error_.errors[i].severity) {
      case Tango::WARN:
        ::mxSetFieldByNumber(struct_array, i, 4, ::mxCreateString("Warning"));
        break;
      case Tango::ERR:
        ::mxSetFieldByNumber(struct_array, i, 4, ::mxCreateString("Error"));
        break;
      case Tango::PANIC:
        ::mxSetFieldByNumber(struct_array, i, 4, ::mxCreateString("Panic"));
        break;
      default:
        ::mxSetFieldByNumber(struct_array, i, 4, ::mxCreateString("Unknown"));
        break;
    }
  }
  // set output arg
  if (MEX_ARGS->set_output_array(0, struct_array) == kError) {
    return kError;
  }
  return kNoError;
}
