//+======================================================================
// $Source: /cvsroot/tango-cs/tango/tools/pogo/gene/PogoGeneDoc.java,v $
//
// Project:   Tango
//
// Description:	java source code for the Pogo class definition .
//
// $Author: pascal_verdier $
//
// $Version$
//
// $Log: PogoGeneDoc.java,v $
// Revision 3.11  2005/06/14 08:44:25  pascal_verdier
// SuperClass is replaced by AbstractClass.
//
// Revision 3.10  2005/03/02 10:27:11  pascal_verdier
// Managing Super Classes and inherited classes.
//
// Revision 3.9  2005/02/23 11:59:39  pascal_verdier
// Force a new header in Description file.
//
// Revision 3.8  2004/11/22 15:29:31  pascal_verdier
// Javadoc tags correction.
//
// Revision 3.7  2004/11/12 08:05:17  pascal_verdier
// minor bugs fixed.
//
// Revision 3.6  2004/10/25 14:03:56  pascal_verdier
// Minor changes.
//
// Revision 3.5  2004/10/21 06:59:29  pascal_verdier
// Default value management added for properties.
// And minor changes
//
// Revision 3.4  2004/10/01 09:16:51  pascal_verdier
// Check Description.html header.
//
// Revision 3.3  2004/09/07 11:59:28  pascal_verdier
// Remove CVS log messages and info from templates.
//
// Revision 3.2  2004/08/26 07:19:30  pascal_verdier
// Attributes are now generated as class.
//
// Revision 3.1  2003/09/10 08:11:30  pascal_verdier
// Minor bugs fixed.
// HTML doc generated with more details.
//
// Revision 3.0  2003/04/29 10:39:50  pascal_verdier
// TANGO 3.0 compatibility
// little bugs fixed.
//
// Revision 1.33  2003/01/16 14:32:36  verdier
// Tango classe files detected for open JFileChooser.
//
// Revision 1.32  2002/10/03 13:53:47  verdier
// Pogo has been used without known bug.
// Put class description as class property.
//
// Revision 1.31  2002/04/25 12:05:08  verdier
// IDL 2 implemented for c++ servers
//
// Revision 1.30  2002/02/06 15:20:20  verdier
// Java code generation updated.
//
// Revision 1.26  2001/12/18 10:12:22  verdier
// Attribute user default property code added.
//
// Revision 1.25  2001/11/09 09:46:13  verdier
// Many bugs fixed.
//
// Revision 1.24  2001/04/04 12:21:27  verdier
// Property management added for cpp.
//
// Revision 1.23  2000/10/24 06:20:13  verdier
// The compatibility with TANGO2 has been tested on DatabaseDs.
//
// Revision 1.22  2000/10/02 05:52:20  verdier
// Attribute code generated is now compatible with Tango 2.
//
// Revision 1.21  2000/09/22 08:54:31  verdier
// DevState & DevStatus are virtual.
// First tests with Tango2
// Taco import utility added.
//
// Revision 1.20  2000/07/07 13:29:17  verdier
// Utilities added after first users.
//
// Revision 1.18  2000/06/20 07:01:38  verdier
// Right button double click management added for :
// editing src code, creating item, editing class....
// Little bugs fixed on generation/re-read src code.
//
// Revision 1.17  2000/05/12 07:35:41  verdier
// Attributes management added for java generation.
//
// Revision 1.16  2000/04/26 06:06:52  verdier
// The save/restore file (.pogo) does not exist anymore.
// DevStates and DevStates allowed management is now available for java.
//
// Revision 1.15  2000/04/18 08:13:48  verdier
// Management of DevStates to allow command added.
//
// Revision 1.14  2000/04/12 09:24:10  verdier
// Methods to manage attributes are now generated
// Only if at leat one attribute exists.
//
// Revision 1.13  2000/04/11 09:34:13  verdier
// Attributes Management Added.
//
// Revision 1.12  2000/03/29 13:14:00  verdier
// Doc generation added.
//
//
// copyleft 1999 by European Synchrotron Radiation Facility, Grenoble, France
//							 All Rights Reversed
//-======================================================================
package pogo.gene;

import java.io.*;
import java.util.*;
import fr.esrf.Tango.*;
import fr.esrf.TangoDs.*;


/**
 *	This class generates html Doc file.
 *
 * @author	$Author: pascal_verdier $
 * @version	$Revision: 3.11 $
 */
public class PogoGeneDoc extends PogoGene implements PogoDefs, TangoConst
{
/**
 *	Pogo Class to be used for generation.
 */
	private PogoClass	pogo;
	private String		outDoc;
	private	String		header;
	private String		footer = null;
	private Cmd			init_cmd;

	private	final String	end_html = "\n\n</BODY>\n</HTML>\n";

//=======================================================================
/**
 *	Constructor for Pogo Class Doc files generation.
 *
 *	@param	pogo	Pogo Class definition used for generation.
 */
//=======================================================================
	public PogoGeneDoc(PogoClass pogo)
	{
		super(pogo);
		this.pogo = pogo;
		outDoc = new String(pogo.projectFiles.getDocHtml());
		header = new String("\"" + pogo.class_name +
								" Device Server generated by POGO\"");
		//	Build the non overriding Init command
		init_cmd = new Cmd("Init", "",
							"Tango::DEV_VOID",
							"Tango::DEV_VOID",
							"none.",
							"none.",
							PogoDefs.init_desc);
	}
//=======================================================================
/**
 *	Build header for table.
 *	@param	title	table title
 *	@param	nb_col	column table number.
 *	@return the table header.
 */
//=======================================================================
	static private String buildTableHeader(String title, int nb_col)
	{
		StringBuffer	sb = 
			new StringBuffer("<Table Border=2 Cellpadding=3 CELLSPACING=0 WIDTH=\"100%\">\n");
		sb.append("<TR BGCOLOR=\"#CCCCFF\" CLASS=\"TableHeadingColor\">\n");

		sb.append("<Td COLSPAN=" + nb_col + "> <Font Size=+2>");
		sb.append("<Center>");
		sb.append("<b>" +  title + "</b></td></Font></Center>\n");

		return sb.toString();
	}
//=======================================================================
/**
 *	Build header for table.
 *	@param	title	table title
 *	@param	col_titles	column table titles.
 *	@return the table header.
 */
//=======================================================================
	static public String buildTableHeader(String title, String[] col_titles)
	{
		StringBuffer	sb = 
			new StringBuffer(buildTableHeader(title, col_titles.length));

		sb.append("<TR BGCOLOR=\"#CCCCFF\" CLASS=\"TableHeadingColor\">\n");
		for (int i=0 ; i<col_titles.length ; i++)
		{
			sb.append("<Td>");
			sb.append("<Center>");
			sb.append("<b>" +  col_titles[i] + "</b></td></Center>\n");
		}

		//sb.append("</Tr>\n");
		return sb.toString();
	}

//=======================================================================
/**
 *	Create the Output director for Doc, if doesn't already exist
 */
//=======================================================================
	private void makeDocDir()throws	SecurityException
	{
		//	Do it for all sub-directories
		for (int end=1 ; (end=outDoc.indexOf("/", end))>0 ; end++)
		{
			String	path = outDoc.substring(0, end);
			File	f = new File(path);
			if (f.exists()==false)
			{
				f.mkdir();
				System.out.println(path + "  created");
			}
		}
		//	Do it for full name
		File	f = new File(outDoc);
		if (f.exists()==false)
		{
			f.mkdir();
			System.out.println(outDoc + "  created");
		}
	}
//=======================================================================
/**
 *	Generate Html Doc files for tango device Server Programmer's guide.
 *	This class use tools as javadoc and doc++ 
 *
 *	@return	Output doc. generation process.
 *	@throws FileNotFoundException
 *	@throws SecurityException
 *	@throws IOException
 *	@throws	PogoException	If a synthax error occured on input file
 */
//=======================================================================
	private String generateProgrammersGuide() throws
										FileNotFoundException,
										SecurityException,
										IOException,
										InterruptedException,
										PogoException
	{
		//	First time get OS used
		//----------------------------------------------
		boolean		isUnix = PogoUtil.osIsUnix();

		//	Build doxigen command
		String[]	cmd;
		makeDocDir();
		switch(pogo.language)
		{
		case javaLang:
			cmd = buildCmdDocFromJava();
			break;
		case cppLang:
			cmd = buildCmdDocFromCpp();
			if (cmd==null)
			{
				removePgGuideLink();
				return "Doxygen NOT installed !";
			}
			break;
		default:
			return "Languange not supported !";
		}
		//System.out.println("\n" + cmd + "\n");
		Process proc = Runtime.getRuntime().exec(cmd);
		if (isUnix==false)
			return"";

//	The following part is done only under Unix (or Linux)
//	because there are zombi problems under Windows !!!
//=======================================================================
		

		// get command's output stream and
		// put a buffered reader input stream on it.
		//-------------------------------------------
		InputStream istr = proc.getInputStream();
		BufferedReader br =
                new BufferedReader(new InputStreamReader(istr));
		StringBuffer	sb = new StringBuffer("Generating Doc:\n\n");
    
		// read output lines from command
		//-------------------------------------------
		String str;
		while ((str = br.readLine()) != null)
		{
			System.out.println(str);
			sb.append(str+"\n");
		}

		// wait for end of command
		//---------------------------------------
		proc.waitFor();

		// check its exit value
		//------------------------
		int retVal;
		if ((retVal=proc.exitValue()) != 0)
		{
			//	An error occured try to read it
			InputStream errstr = proc.getErrorStream();
			br = new BufferedReader(new InputStreamReader(errstr));
			while ((str = br.readLine()) != null)
			{
				System.out.println(str);
				sb.append(str+"\n");
			}
			throw new PogoException(
				"Programmer's guide generation returns : " + retVal + " !\n\n"+
				sb.toString());
		}
		System.out.println(sb);

		return sb.toString();
	}
//=======================================================================
/**
 *	Build the Doxyge template file to generate class doc
 *
 *	@return	Output doc generation process.
 */
//=======================================================================
	private String[] buildCmdDocFromCpp() throws 
										FileNotFoundException,
										SecurityException,
										IOException,
										PogoException
	{
		String	template  = new String(pogo.templates_dir + "/html/Doxyfile");
		String	doxyfile  = new String(pogo.projectFiles.getPath() + "/Doxyfile");

		//	First build command if Doxygene installed
 		String[]	cmd = new String[2];

   		String	docPath = System.getProperty("CPP_DOC_PATH");;
		int	n = 0 ;
		if (docPath==null)
			cmd[n++] = new String("doxygen");
		else
		if (docPath.equals("NOT_INSTALLED"))
			return null;
		else
			cmd[n++] = new String(docPath + "/doxygen");
		cmd[n++] = doxyfile;

		//	Then time buid Doxyfile from template
		FileInputStream		fidin  = new FileInputStream(template);
		int nb = fidin.available();
		byte[]	inStr  = new byte[nb];
		fidin.read(inStr);
		fidin.close();
		PogoString	code = new PogoString(inStr);
		code.replace("POGO_TITLE",     "\"" + pogo.title + "\"");
		code.replace("POGO_SRC_FILES", pogo.projectFiles.getPath());
		code.replace("POGO_OUT_DOC",   outDoc);

		FileOutputStream	fidout = new FileOutputStream(doxyfile);
		fidout.write(code.str.getBytes());
		fidout.close();
	
		return cmd;
	}
//=======================================================================
/**
 *	Build the javadoc command to gfenerate class doc
 *
 *	@return	Output doc. generation process.
 */
//=======================================================================
	private String[] buildCmdDocFromJava() 
	{
		String[]	cmd = new String[pogo.commands.size() + 9];
		int	n = 0 ;
		cmd[n++] = "javadoc";
		cmd[n++] = "-private";
		cmd[n++] = "-author";
		cmd[n++] = "-windowtitle";
		cmd[n++] = header;
		cmd[n++] = "-header";
		cmd[n++] = header;
		cmd[n++] = "-d";
		cmd[n++] = outDoc + "/html";
		cmd[n++] = new String(pogo.projectFiles.getPath() + "/" +
							pogo.class_name + javaExtention);
		cmd[n++] = new String(pogo.projectFiles.getPath() + "/" +
							pogo.class_name + "Class" + javaExtention);
		for (int i=2 ; i < pogo.commands.size() ; i++)
		{
			Cmd		c = pogo.commands.cmdAt(i);
			cmd[n++] = new String(pogo.projectFiles.getPath() + "/" +
									c.cmd_class  + javaExtention);
		}
		return cmd;
	}
//=======================================================================
/**
 *	Create the index.html file from template file and PogoClass Object.
 *
 *	@throws IOException
 *	@throws	PogoException	If a synthax error occured on input file
 */
//=======================================================================
	protected void generateIndexFile()
						throws	IOException, PogoException
	{
		PogoString	pgs = readTemplate("index.html", "");

		//	Open Output file and search place to write
		//----------------------------------------------
		String	f_out = new String(outDoc + "/index.html");
		writeDocFile(f_out, pgs.str, true);

		//	Add inheritance if any
		if (pogo.inherited_from!=null &&
			pogo.inheritedClassName().equals(tangoDeviceImpl)==false)
		{
			PogoString	code = new PogoString(PogoUtil.readFile(outDoc + "/index.html"));
			String	inheritance_message = "<b>This class inherits from " + 
						pogo.inheritedClassName() + " class.</b><Br>\n";

			String	intro = "<h2>Introduction:</h2>\n";
			int pos = code.str.indexOf(intro);
			pos += intro.length();
			code.insert(pos, inheritance_message);
			PogoUtil.writeFile(outDoc + "/index.html", code.str);
		}
	}
//=======================================================================
/**
 *	Create the Description.html file from template file and PogoClass Object.
 *	This file will be filled by the programmer.
 *	For this reason, if the file already exists, it will not be overwritten.
 *
 *	@throws IOException
 *	@throws	PogoException	If a synthax error occured on input file
 */
//=======================================================================
	protected void generateDescriptionFile()
						throws	IOException, PogoException
	{
		//	DO NOT GENERATE IF ALREADY EXISTS.
		//------------------------------------------------
		String	f_out = new String(outDoc + "/Description.html");
		if (new File(f_out).exists() == true)
		{
			//	check header (old or new)
			boolean	modified = false;
			PogoString	code = new PogoString(PogoUtil.readFile(f_out));
			//	It is the old one --> put the new header
			int	end = code.str.indexOf("<HR WIDTH=\"100%\"></H5>");
			if (end>0)
			{
				String	templname  = pogo.templates_dir + "/html/header.html";
				String	old_header = code.str.substring(0, end);
				String	new_header = PogoUtil.readFile(templname);

				//	Remove end of header and change it
				int p = new_header.indexOf("</Body>");
				if (p>0)
					new_header = new_header.substring(0, p);
				code.replace(old_header, new_header);
				modified = true;
			}
			//	Check if old default message
			String	old_mess = "This Page Must Be Filled by <Br>\n" +
								"The Programmer";
			int	start = code.str.indexOf(old_mess);
			if (start>0)
			{
				code.replace(old_mess, pogo.class_desc);
				modified = true;
			}

			//	Write new code if modified
			if (modified)
				PogoUtil.writeFile(f_out, code.str);
		}
		else
		{
			//	If does not already exist generate a new one
			PogoString	pgs = readTemplate("page.html", "Device Description");
			pgs.append("<Center>\n");
			pgs.append(pogo.class_desc);
			pgs.append("\n<Br>\n<Br>\n");
			pgs.append("</Center>\n");

			//	Open Output file and write
			//----------------------------------------------
			writeDocFile(f_out, pgs.str, true);
		}
		//	Check if doc or pdf files
		checkDocFile(outDoc);
	}
//=======================================================================
/**
 *	Create the tangoDevStates.html file from template file and PogoClass Object.
 *	This file will describe the device server states in a table.
 *
 *	@throws IOException
 *	@throws	PogoException	If a synthax error occured on input file
 */
//=======================================================================
	protected void generateStateDescriptionFile()
						throws	IOException, PogoException
	{
		PogoString	pgs = readTemplate("page.html", "Device States Description");
		//	Open Output file and search place to write
		//----------------------------------------------
		pgs.append("<Center>\n");
		pgs.append("<Br> <Br> <Br> \n");
		String[]	titles = { "Names" , "Descriptions" };
		pgs.append(buildTableHeader("States", titles));

		for (int i=0 ; i<pogo.states.size() ; i++)
		{
			DevState state = (DevState)(pogo.states.elementAt(i));
			pgs.append("<Tr><Td>"+ removeTangoNamespace(state.name) +"</Td>\n");
			pgs.append("<Td>" + state.description + "</Td></Tr>\n\n");
		}

		pgs.append("\n");
		pgs.append("\n");
		pgs.append("\n");
		pgs.append("</Table>\n");
		pgs.append("</Center>\n");

		String	f_out = new String(outDoc + "/TangoDevStates.html");
		writeDocFile(f_out, pgs.str, true);
	}
//=======================================================================
/**
 *	Create the DevProperites.html file from template file and PogoClass Object.
 *	This file will describe the device server property in a table.
 *
 *	@throws IOException
 *	@throws	PogoException	If a synthax error occured on input file
 */
//=======================================================================
	protected void generatePropertyDescriptionFile()
						throws	IOException, PogoException
	{
		boolean	class_prop  = false;
		boolean	device_prop = false;
		PogoString	pgs =
				readTemplate("page.html", "Properties Description");

		pgs.append("<Br> <Br> <Br> \n");
		pgs.append("<Center>\n");
		String[]	titles = { "Property name", "Property type", "Description" };

		//	Class Properties if any
		//----------------------------------
		if (pogo.class_properties.size() > 0)
		{
			class_prop = true;
			pgs.append(buildTableHeader("Class Properties", titles));
			pgs.append("\n");

			for (int i=0 ; i<pogo.class_properties.size() ; i++)
			{
				Property	property = pogo.class_properties.propertyAt(i);
				pgs.append("<Tr><Td>"+
								property.name +"</Td>\n");
				pgs.append("<Td>"+
								property.type.cpp_code_str +"</Td>\n");
				pgs.append("<Td>" +
								property.description + "</Td></Tr>\n\n");
			}
			pgs.append("</Table>\n");
			pgs.append("\n");
		}
		//	Device Properties if any
		//----------------------------------
		if (pogo.dev_properties.size() > 0)
		{
			device_prop = true;
			pgs.append("<Br> <Br> <Br> \n");
			pgs.append(buildTableHeader("Device Properties", titles));

			for (int i=0 ; i<pogo.dev_properties.size() ; i++)
			{
				Property	property = pogo.dev_properties.propertyAt(i);

				pgs.append("<Tr><Td>"+
								property.name +"</Td>\n");
				pgs.append("<Td>"+
								property.type.cpp_code_str +"</Td>\n");
				pgs.append("<Td>" +
								property.description + "</Td></Tr>\n\n");
			}
			pgs.append("</Table>\n");
			pgs.append("\n");
		}
		if (device_prop==false || class_prop==false)
		{
			pgs.append("<Br><Br><Br>\n");
			pgs.append("<Center><b>\n");
			if (class_prop==false)
				pgs.append("There is no Class properties.<Br><Br>\n");
			if (device_prop==false)
				pgs.append("There is no Device Properties.<Br><Br>\n");
			pgs.append("</Center></b>\n");
		}

		//	Open Output file and  write
		//----------------------------------------------
		String	f_out = new String(outDoc + "/Properties.html");
		writeDocFile(f_out, pgs.str, true);
	}
//=======================================================================
/**
 *	Create the DevCommands.html file from template file and PogoClass Object.
 *	This file will describe the device server commands in a table.
 *
 *	@throws IOException
 *	@throws	PogoException	If a synthax error occured on input file
 */
//=======================================================================
	protected void generateAttributesFile()
						throws	IOException, PogoException
	{
		PogoString	pgs =
				readTemplate("page.html", "Device Attributes Description");
		pgs.append("<Br>\n");


		//	Check if scalar, Spectrum and Image attributes are defined
		//----------------------------------------------------------------
		boolean	assAttr = false;
		boolean scalar  = false;
		boolean spectrum= false;
		boolean image   = false;
		for (int i=0 ; i<pogo.attributes.size() ; i++)
		{
			Attrib attr = pogo.attributes.attributeAt(i);
			if (attr.attrType==ATTR_SCALAR)
			{
				scalar = true;
				if (attr.rwType==ATTR_READ_WITH_WRITE)
					assAttr  = true;
			}
			else
			if (attr.attrType==ATTR_SPECTRUM)
				spectrum = true;
			else
			if (attr.attrType==ATTR_IMAGE)
				image = true;
		}

		//	Generate for scalar attributs if exist
		//------------------------------------------

		if (scalar)
		{
			String[]	titles = new String[4 + ((assAttr)? 1 : 0)];
			int	n =0;
			titles[n++] = "Attribute name";
			titles[n++] = "Data Type";
			titles[n++] = "R/W Type";
			if (assAttr)
				titles[n++] = "Ass. Attr.";
			titles[n++] = "Expert";
			pgs.append(buildTableHeader("Scalar Attributes", titles));

			for (int i=0 ; i<pogo.attributes.size() ; i++)
			{
				Attrib attr = pogo.attributes.attributeAt(i);
				if (attr.attrType==ATTR_SCALAR)
				{
					//	Write att name
					pgs.append("<Tr><Td><b><Center>"+ attr.name + "</b>");

					//	Write att description if exists
					String	desc = attr.getDescriptionLine();
					if (desc!=null)
						if (desc.length()>0)
							pgs.append("<Br>( "+ desc + " )");
					pgs.append("</Center></Td>\n");
					String	str_type = simpleType(attr.dataType);
					String	str_rw   = AttrRWtypeArray[attr.rwType];
					String	level;
					if (attr.disp_level==DispLevel.OPERATOR)
						level = "No";
					else
						level = "Yes";

					pgs.append("<Td><Center>" + str_type + "</Center></Td>");
					pgs.append("<Td><Center>" + str_rw + "</Center></Td>");
					if (attr.rwType==ATTR_READ_WITH_WRITE)
						pgs.append("<Td><Center>" + attr.assAttr + "</Center></Td>");
					else
					if (assAttr)
						pgs.append("<Td><Center> - </Center></Td>");

					pgs.append("<Td><Center>" + level + "</Center></Td>");
					pgs.append("</Tr>\n\n");
				}
			}
			pgs.append("</Table>\n");
			pgs.append("</Center>\n");
			pgs.append("<Br><Br><Br><Br><Br>\n");
		}
		//	Generate for spectrum attributs if exist
		//------------------------------------------
		if (spectrum)
		{
			String[]	titles = new String[4];
			titles[0] = "Attribute name";
			titles[1] = "Data Type";
			titles[2] = "X Data Length";
			titles[3] = "Expert";
			pgs.append(buildTableHeader("Spectrum Attributes", titles));

			for (int i=0 ; i<pogo.attributes.size() ; i++)
			{
				Attrib attr = pogo.attributes.attributeAt(i);
				if (attr.attrType==ATTR_SPECTRUM)
				{
					pgs.append("<Tr><Td><b><Center>"+ attr.name + "</b>");

					//	Write att description if exists
					String	desc = attr.getDescriptionLine();
					if (desc!=null)
						if (desc.length()>0)
							pgs.append("<Br>( "+ desc + " )");
					pgs.append("</Center></Td>\n");

					String	str_type = simpleType(attr.dataType);
					String	level;
					if (attr.disp_level==DispLevel.OPERATOR)
						level = "No";
					else
						level = "Yes";
					pgs.append("<Td><Center>" + str_type + "</Center></Td>");
					pgs.append("<Td><Center>" + attr.xSize + "</Center></Td>");
					pgs.append("<Td><Center>" + level + "</Center></Td>");
					pgs.append("</Tr>\n\n");
				}
			}
			pgs.append("</Table>\n");
			pgs.append("</Center>\n");
			pgs.append("<Br><Br><Br><Br><Br>\n");
		}
		//	Generate for image attributs if exist
		//------------------------------------------
		if (image)
		{
			String[]	titles = new String[5];
			titles[0] = "Attribute name";
			titles[1] = "Data Type";
			titles[2] = "X Data Length";
			titles[3] = "Y Data Length";
			titles[4] = "Expert";
			pgs.append(buildTableHeader("Image Attributes", titles));

			for (int i=0 ; i<pogo.attributes.size() ; i++)
			{
				Attrib attr = pogo.attributes.attributeAt(i);
				if (attr.attrType==ATTR_IMAGE)
				{
					pgs.append("<Tr><Td><b><Center>"+ attr.name + "</b>");

					//	Write att description if exists
					String	desc = attr.getDescriptionLine();
					if (desc!=null)
						if (desc.length()>0)
							pgs.append("<Br>( "+ desc + " )");
					pgs.append("</Center></Td>\n");

					String	str_type = simpleType(attr.dataType);
					String	level;
					if (attr.disp_level==DispLevel.OPERATOR)
						level = "No";
					else
						level = "Yes";
					pgs.append("<Td><Center>" + str_type + "</Center></Td>");
					pgs.append("<Td><Center>" + attr.xSize + "</Center></Td>");
					pgs.append("<Td><Center>" + attr.ySize + "</Center></Td>");
					pgs.append("<Td><Center>" + level + "</Center></Td>");
					pgs.append("</Tr>\n\n");
				}
			}
			pgs.append("</Table>\n");
			pgs.append("</Center>\n");
		}
		//	Open Output file and search place to write
		//----------------------------------------------
		String	f_out = new String(outDoc + "/Attributes.html");
		writeDocFile(f_out, pgs.str, true);
	}
//=======================================================================
/**
 *	Create the DevCommands.html file from template file and PogoClass Object.
 *	This file will be a list of device server commands for the left frame part.
 *
 *	@throws IOException
 *	@throws	PogoException	If a synthax error occured on input file
 */
//=======================================================================
	protected void generateCmdListFile()
						throws	IOException, PogoException
	{
		String	str = "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0//EN\">\n";

		str += "<HTML><HEAD>\n";
		str += "<TITLE>Commands\n</TITLE></HEAD>\n";
		str += "<BODY BGCOLOR=\"white\">\n";
		str += "<FONT size=\"+1\" ID=\"FrameHeadingFont\">\n";
		str += "<B>Commands:</B></FONT>\n";
		str += "<Br>\n<Br>\n<Br>\n";

		//	Add the non overwritable command init
		str += "<A Href=\"DevCommands.html#Init";
		str += "\" TARGET=\"DevCommands\"> ";
		str += 	"Init</a><Br>\n";
				
		//	Add all commands
		for (int i=0 ; i<pogo.commands.size() ; i++)
		{
			Cmd		cmd = pogo.commands.cmdAt(i);
			str += "<A Href=\"DevCommands.html#" + cmd.name;
			str += "\" TARGET=\"DevCommands\"> ";
			str += 	cmd.name + "</a><Br>\n";
		}

		//	Add the abstract link
		/*
		str += "<Br>\n<Br>\n<A Href=\"DevCommandsTable.html\""+
						" TARGET=\"DevCommands\"> Abstract </a><Br>\n";
		*/
		str += end_html;

		//	Open Output file and  write
		//----------------------------------------------
		String	f_out = new String(outDoc + "/DevCommandsList.html");
		writeDocFile(f_out, str, false);

		//	Copy the Commands Frame file
		//-------------------------------------------
		copyFile(pogo.templates_dir + "/html/frame.html",
					outDoc + "/DevCommandsFrame.html");
	}
//=======================================================================
//=======================================================================
	private String buildCmdDesc(Cmd cmd, int cmd_idx)
	{
		PogoString	pgs =
			new PogoString("<A NAME=\"" + cmd.name  + "\"><!-- --></A>\n");

			//	Label for Href html command
			pgs.append("<A NAME=\"" + cmd.name  + "\"><!-- --></A>\n");
			//	Cmd name
			pgs.append("<h2>" + cmd_idx + " - " + cmd.name);

			if (cmd.level==fr.esrf.Tango.DispLevel.EXPERT)
				pgs.append(" (for expert only)");

 			pgs.append("</h2>\n<ul>\n");

			//	Cmd Description
			pgs.append("<Li><Strong>Description: </Strong> ");
			if (cmd.description!=null)
				pgs.append(cmd.description);

			//	Argin
			pgs.append("<Br>&nbsp\n");
			pgs.append("<Li><Strong>Argin:<Br>" + simpleType(cmd.argin) +
								"</Strong>\n");
			if (cmd.argin.description!=null)
			{
				PogoString	desc = new PogoString(cmd.argin.description);
				while (desc.indexOf("\\n")>=0)
					desc.replace("\\n", "<Br>\n");
				pgs.append(" : " + desc);
			}
			//	Argout
			pgs.append("<Br>&nbsp\n");
			pgs.append("<Li><Strong>Argout:<Br>" + simpleType(cmd.argout) +
								"</Strong>\n");
			if (cmd.argout.description!=null)
			{
				PogoString	desc = new PogoString(cmd.argout.description);
				while (desc.indexOf("\\n")>=0)
					desc.replace("\\n", "<Br>\n");
				pgs.append(" : " + desc);
			}

			//	Command allowed for states
			if (cmd.notAllowedFor!=null)
			{
				pgs.append("<Br>&nbsp\n");
				pgs.append("<Li><Strong>Command allowed for: " +
											"</Strong><Ul>\n");

				for (int idx=0 ; idx<pogo.states.size() ; idx++)
				{
					DevState	st = pogo.states.stateAt(idx);
					boolean 	found=false;
					for (int alIdx=0 ;alIdx<cmd.notAllowedFor.size() ; alIdx++)
					{
						DevState	st1 = cmd.notAllowedFor.stateAt(alIdx);
						if (st1.name.equals(st.name)==true)
							found = true;
					}
					if (found==false)
						pgs.append("<Li>" + st.name);
				}

				pgs.append("</Ul>\n");
			}
			else
				pgs.append("<Li><Strong>Command allowed for all device states." +
											"</Strong><Br>\n");
			
			pgs.append("<Br>&nbsp\n");
			if (cmd.polled_period>0)
				pgs.append("<Li><Strong>Polled every " + cmd.polled_period +
										 " ms</Strong><Br>\n");
			pgs.append("</ul><Br>\n<Br>\n");
		return pgs.str;
	}
//=======================================================================
/**
 *	Create the DevCommands.html file from template file and PogoClass Object.
 *	This file will describe the device server commands in a printable list.
 *
 *	@throws IOException
 *	@throws	PogoException	If a synthax error occured on input file
 */
//=======================================================================
	protected void generateCmdListDescriptionFile()
						throws	IOException, PogoException
	{
		PogoString	pgs =
				readTemplate("page.html", "Device Commands Description");

		
		pgs.append("<Br>\n<Br>\n<Br>\n");
		
		//	Add the non overwritable command init
		int	cmd_idx = 1;
		pgs.append(buildCmdDesc(init_cmd, cmd_idx++));

		//	Add all commands
		for (int i=0 ; i<pogo.commands.size() ; i++)
		{
			Cmd		cmd = pogo.commands.cmdAt(i);
			pgs.append(buildCmdDesc(cmd, cmd_idx++));
		}

		//	Open Output file and search place to write
		//----------------------------------------------
		String	f_out = new String(outDoc + "/DevCommands.html");
		writeDocFile(f_out, pgs.str, true);
	}
//=======================================================================
/**
 *	Create the DevCommands.html file from template file and PogoClass Object.
 *	This file will describe the device server commands in a table.
 *
 *	@throws IOException
 *	@throws	PogoException	If a synthax error occured on input file
 */
//=======================================================================
	protected void generateCmdTableFile()
						throws	IOException, PogoException
	{
		PogoString	pgs =
			readTemplate("page.html", "Device Commands Description");
		String[]	titles = new String[3];
		titles[0] = "Command name";
		titles[1] = "Argument In";
		titles[2] = "Argument Out";


		pgs.append("<Center>\n");
		pgs.append("<Br><Br>\n<A Href=\"DevCommandsFrame.html\">"+
						" More Details on commands.... </a><Br>\n");
		//	Build table for DispLevel.OPERATOR Commands
		pgs.append("<Br> <Br>\n");

		pgs.append(buildTableHeader("Device Commands for Operator Level", titles));

		//	Add the non overwritable command init
		pgs.append(new String("<Tr><Td>"+ init_cmd.name +"</Td>\n"));
		pgs.append(new String("<Td>"+ simpleType(init_cmd.argin) +"</Td>\n"));
		pgs.append(new String("<Td>"+ simpleType(init_cmd.argout) +"</Td>\n"));

		//	Add all  commands
		int	nb_expert = 0;
		for (int i=0 ; i<pogo.commands.size() ; i++)
		{
			Cmd		cmd = pogo.commands.cmdAt(i);
			if (cmd.level==DispLevel.OPERATOR)
			{
				pgs.append(new String("<Tr><Td>"+ cmd.name +"</Td>\n"));
				pgs.append(new String("<Td>"+ simpleType(cmd.argin) +"</Td>\n"));
				pgs.append(new String("<Td>"+ simpleType(cmd.argout) +"</Td>\n"));
			}
			else
				nb_expert++;
		}
		pgs.append("\n\n\n</Table></Center>\n");

		//	Build table for DispLevel.EXPERT Commands if any
		if (nb_expert>0)
		{
			pgs.append("<Br> <Br> <Br> \n");
			pgs.append("<Center>\n");
			pgs.append(buildTableHeader("Device Commands for Expert Level Only", titles));

			for (int i=0 ; i<pogo.commands.size() ; i++)
			{
				Cmd		cmd = pogo.commands.cmdAt(i);

				if (cmd.level==DispLevel.EXPERT)
				{
					pgs.append("<Tr><Td>"+ cmd.name +"</Td>\n");
					pgs.append("<Td>"+ simpleType(cmd.argin) +"</Td>\n");
					pgs.append("<Td>"+ simpleType(cmd.argout) +"</Td>\n");
				}
			}
			pgs.append("\n\n\n</Table></Center>\n");
		}
		//	Open Output file and write
		//----------------------------------------------
		String	f_out = new String(outDoc + "/DevCommandsTable.html");
		writeDocFile(f_out, pgs.str, true);
	}
//=======================================================================
/**
 *	Add footer if request and write file.
 */
//=======================================================================
	private void writeDocFile(String filename, String str, boolean write_footer)
						throws	IOException, PogoException
	{
		if (write_footer)
		{
			if (footer==null)
			{
				String	footname = pogo.templates_dir + "/html/footer.html";
				FileInputStream		fidin = new FileInputStream(footname);
				int nb = fidin.available();
				byte[]	inStr  = new byte[nb];
				fidin.read(inStr);
				footer = new String(inStr);
				fidin.close();
			}
			str += footer;
		}
		FileOutputStream	fidout = new FileOutputStream(filename);
		fidout.write(str.getBytes());
		fidout.close();
	}
//=======================================================================
//=======================================================================
	protected void removePgGuideLink() throws
								FileNotFoundException,
								IOException
	{
		String[]	pgGuideLink = {
			"<h2>Programmer's guide:</h2>",
			"<ul>",
			"<li> <a href=\"html/index.html\">		Software description.</a>",
			"</ul>"
			};

		String	filename = pogo.projectFiles.getDocHtml() + "/index.html";
		PogoString	pgs = new PogoString(PogoUtil.readFile(filename));

		for (int i=0 ; i<pgGuideLink.length ; i++)
			pgs.replace(pgGuideLink[i], "");

		FileOutputStream	fidout = new FileOutputStream(filename);
		fidout.write(pgs.str.getBytes());
		fidout.close();
	}
//=======================================================================
/**
 *	Read a template file and update the page title.
 *	@param	filename	template file name
 *	@param	page_title	text to be used as title.
 */
//=======================================================================
	private PogoString readTemplate(String filename, String page_title)
						throws	IOException, PogoException
	{
		//	Open and read file
		//----------------------------------------
		FileInputStream		fidin 
			= new FileInputStream(pogo.templates_dir + "/html/" + filename);
		int nb = fidin.available();
		byte[]	inStr  = new byte[nb];
		fidin.read(inStr);
		PogoString pgs = new PogoString(inStr);
		fidin.close();

		//	Update page title
		pgs.replace(projectTitleRes, pogo.title);
		pgs.replace(pageTitleRes, page_title);
		pgs.replace("ClassName", pogo.class_name);
		pgs.replace("1.0", pogo.revision);
		pgs.replace("Verdier", pogo.author);
		pgs.replace(classDescRes, pogo.class_desc);
		return pgs;
	}
//=======================================================================
/**
 *	Remove 'Tango::' from type and return as string.
 */
//=======================================================================
	private String simpleType(TangoTypes type)
	{
		return removeTangoNamespace(type.cpp_code_str);
	}
//=======================================================================
/**
 *	Remove 'Tango::' from string.
 */
//=======================================================================
	private String removeTangoNamespace(String s)
	{
		if (s.indexOf(PogoDefs.cppNameSpace)!=0)
			return s;
		else
			return s.substring(PogoDefs.cppNameSpace.length());
	}

//===============================================================
/**
 *	Search if .doc or .pdf in directory --> add link on it
 */
//===============================================================
	static public void checkDocFile(String targetdir)
	{
		File	f = new File(targetdir);
		if (f.exists()==false)
			return;

		//	Get file list in directory
		String[]	src_files = f.list();
		for (int i=0 ; i<src_files.length ; i++)
		{
			String	src_file = targetdir + "/" + src_files[i];
			if (new File(src_file).isDirectory()==false)	//	Not recursive
			{
				//	Check if doc or pdf file
				int	idx = src_file.lastIndexOf(".");
				if (idx<0)
					break;

				if (src_file.substring(idx).equals(".doc")  ||
					src_file.substring(idx).equals(".pdf") )
				{
					//	Get filename without targetdir
					idx = src_file.lastIndexOf("/");
					if (idx<0)
						idx = 0;
					else
						idx++;
					String	filename = src_file.substring(idx);
					System.out.println(filename);

					//	Add a link on description file
					addLinkOnDoc(targetdir, filename);
				}
			}
		}
	}
//===============================================================
/**
 *	Add link on .doc or .pdf file in Description.html file
 */
//===============================================================
	static public void addLinkOnDoc(String targetdir, String filename)
	{
		try
		{
			//	Add a link on description file
			PogoString	code = new PogoString(
				PogoUtil.readFile(targetdir + "/Description.html"));

			//	Check if link already exists
			if (code.str.indexOf(filename)>0)
				return;

			//	else search place to insert after Revision number
			int	idx = code.str.indexOf("Revision");
			if (idx<0)
				return;
			idx = code.str.toLowerCase().indexOf("</center>", idx);
			if (idx<0)
				return;

			String	link =  "<Br><Br><Br>\n" +
					"Download <A href=" + filename + "> " + filename +" </a>\n";
			code.insert(idx, link);
			PogoUtil.writeFile(targetdir + "/Description.html", code.str);
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
	}
//=======================================================================
/**
 *	Generate Html Doc files for tango device Server
 *	from Pogo Class definition
 *
 *	@return	Output doc. generation process.
 *	@throws FileNotFoundException
 *	@throws SecurityException
 *	@throws IOException
 *	@throws	PogoException	If a synthax error occured on input file
 */
//=======================================================================
	public String generate() throws	FileNotFoundException,
									SecurityException,
									IOException,
									InterruptedException,
									PogoException
	{
		System.out.println("Generating HTML Doc files...");

		//	Generate the User's Guide files.
		//-----------------------------------------
		makeDocDir();
		generateIndexFile();
		generateDescriptionFile();
		generateStateDescriptionFile();
		generatePropertyDescriptionFile();
		generateCmdListDescriptionFile();
		generateCmdListFile();
		generateCmdTableFile();
		generateAttributesFile();

		//	Generate the Programme'r Guide
		//-----------------------------------------
		//generateProgrammersGuide();

		return "";
	}		
}
//-----------------------------------------------------------------------------
/* end of $Source: /cvsroot/tango-cs/tango/tools/pogo/gene/PogoGeneDoc.java,v $ */
