package jive;

/****************************************************************************
 *
 * To use these classes, CLASSPATH include a path to this file location
 * (for instance, if this file is located in the same directory than
 *  the source file using these classes then CLASSPATH must contain "./"
 *
 * $Revision: 1.3 $
 *
 */

import fr.esrf.Tango.*;
import fr.esrf.TangoApi.DeviceAttribute;
import fr.esrf.TangoApi.DeviceData;
import fr.esrf.TangoDs.TangoConst;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

public final class ExtractHelper implements TangoConst {

  /**
   * Extract from a string (typically coming from a string input field) the
   * specified type values and put them in Tango API DeviceData type.
   *
   * The rules for decoding types from string (i.e. how the user must
   * enter the values in the string input field) are the same as the ones
   * of "xdevmenu" Unix application.
   *
   * Anyway they are explained here for each complicated type.
   *
   */
  public static void string_to_devicedata(
          String argin_string,
          DeviceData send,
          int command_in_type) throws NumberFormatException {

    if(command_in_type==Tango_DEV_VOID) return;

    ArgParser arg = new ArgParser(argin_string);

    switch (command_in_type) {
      case Tango_DEV_BOOLEAN:
        send.insert(arg.parse_boolean());
        break;
      case Tango_DEV_USHORT:
        send.insert_us(arg.parse_ushort());
        break;
      case Tango_DEV_SHORT:
        send.insert(arg.parse_short());
        break;
      case Tango_DEV_ULONG:
        send.insert_ul(arg.parse_ulong());
        break;
      case Tango_DEV_LONG:
        send.insert(arg.parse_long());
        break;
      case Tango_DEV_FLOAT:
        send.insert(arg.parse_float());
        break;
      case Tango_DEV_DOUBLE:
        send.insert(arg.parse_double());
        break;
      case Tango_DEV_STRING:
        send.insert(arg.parse_string());
        break;
      case Tango_DEVVAR_CHARARRAY:
        send.insert(arg.parse_char_array());
        break;
      case Tango_DEVVAR_USHORTARRAY:
        send.insert_us(arg.parse_ushort_array());
        break;
      case Tango_DEVVAR_SHORTARRAY:
        send.insert(arg.parse_short_array());
        break;
      case Tango_DEVVAR_ULONGARRAY:
        send.insert_ul(arg.parse_ulong_array());
        break;
      case Tango_DEVVAR_LONGARRAY:
        send.insert(arg.parse_long_array());
        break;
      case Tango_DEVVAR_FLOATARRAY:
        send.insert(arg.parse_float_array());
        break;
      case Tango_DEVVAR_DOUBLEARRAY:
        send.insert(arg.parse_double_array());
        break;
      case Tango_DEVVAR_STRINGARRAY:
        send.insert(arg.parse_string_array());
        break;
      case Tango_DEVVAR_LONGSTRINGARRAY:
        send.insert(new DevVarLongStringArray(arg.parse_long_array(),arg.parse_string_array()));
        break;
      case Tango_DEVVAR_DOUBLESTRINGARRAY:
        send.insert(new DevVarDoubleStringArray(arg.parse_double_array(),arg.parse_string_array()));
        break;
      case Tango_DEV_STATE:
        send.insert(DevState.from_int(arg.parse_ushort()));
        break;

      default:
        throw new NumberFormatException("Command type not supported code=" + command_in_type);

    }

  }


  /**
   * Extract from a string (typically coming from a string input field) the
   * specified type values and put them in Tango API DeviceData type.
   */
  public static void string_to_deviceattribute(
          String argin_string,
          DeviceAttribute send,
          AttrDataFormat attr_format,
          int attr_type) throws NumberFormatException {

    ArgParser arg = new ArgParser(argin_string);

    switch (attr_type) {

      case Tango_DEV_UCHAR:
        switch (attr_format.value()) {
          case AttrDataFormat._SCALAR:
            send.insert(arg.parse_uchar());
            break;
          case AttrDataFormat._SPECTRUM:
            send.insert(arg.parse_uchar_array());
            break;
          case AttrDataFormat._IMAGE:
            send.insert(arg.parse_uchar_image(),arg.get_image_width(),arg.get_image_height());
            break;
        }
        break;

      case Tango_DEV_BOOLEAN:
        switch (attr_format.value()) {
          case AttrDataFormat._SCALAR:
            send.insert(arg.parse_boolean());
            break;
          case AttrDataFormat._SPECTRUM:
            send.insert(arg.parse_boolean_array());
            break;
          case AttrDataFormat._IMAGE:
            send.insert(arg.parse_boolean_image(),arg.get_image_width(),arg.get_image_height());
            break;
        }
        break;

      case Tango_DEV_SHORT:
        switch (attr_format.value()) {
          case AttrDataFormat._SCALAR:
            send.insert(arg.parse_short());
            break;
          case AttrDataFormat._SPECTRUM:
            send.insert(arg.parse_short_array());
            break;
          case AttrDataFormat._IMAGE:
            send.insert(arg.parse_short_image(),arg.get_image_width(),arg.get_image_height());
            break;
        }
        break;

      case Tango_DEV_USHORT:
        switch (attr_format.value()) {
          case AttrDataFormat._SCALAR:
            send.insert_us(arg.parse_ushort());
            break;
          case AttrDataFormat._SPECTRUM:
            send.insert_us(arg.parse_ushort_array());
            break;
          case AttrDataFormat._IMAGE:
            send.insert_us(arg.parse_ushort_image(),arg.get_image_width(),arg.get_image_height());
            break;
        }
        break;

      case Tango_DEV_LONG:
        switch (attr_format.value()) {
          case AttrDataFormat._SCALAR:
            send.insert(arg.parse_long());
            break;
          case AttrDataFormat._SPECTRUM:
            send.insert(arg.parse_long_array());
            break;
          case AttrDataFormat._IMAGE:
            send.insert(arg.parse_long_image(),arg.get_image_width(),arg.get_image_height());
            break;
        }
        break;

      case Tango_DEV_FLOAT:
        switch (attr_format.value()) {
          case AttrDataFormat._SCALAR:
            send.insert(arg.parse_float());
            break;
          case AttrDataFormat._SPECTRUM:
            send.insert(arg.parse_float_array());
            break;
          case AttrDataFormat._IMAGE:
            send.insert(arg.parse_float_image(),arg.get_image_width(),arg.get_image_height());
            break;
        }
        break;

      case Tango_DEV_DOUBLE:
        switch (attr_format.value()) {
          case AttrDataFormat._SCALAR:
            send.insert(arg.parse_double());
            break;
          case AttrDataFormat._SPECTRUM:
            send.insert(arg.parse_double_array());
            break;
          case AttrDataFormat._IMAGE:
            send.insert(arg.parse_double_image(),arg.get_image_width(),arg.get_image_height());
            break;
        }
        break;

      case Tango_DEV_STRING:
        switch (attr_format.value()) {
          case AttrDataFormat._SCALAR:
            send.insert(arg.parse_string());
            break;
          case AttrDataFormat._SPECTRUM:
            send.insert(arg.parse_string_array());
            break;
          case AttrDataFormat._IMAGE:
            send.insert(arg.parse_string_image(),arg.get_image_width(),arg.get_image_height());
            break;
        }
        break;

      default:
        throw new NumberFormatException("Attribute type not supported code=" + attr_type);

    }

  }

  /**
   * Conmvert DeviceData to string for printing.
   */
  public static String devicedata_to_string(
          DeviceData received,
          int command_out_type) {
    String ret_string = new String("");

    switch (command_out_type) {
      case Tango_DEV_VOID:
        break;
      case Tango_DEV_BOOLEAN:
        Boolean bidon = new Boolean(received.extractBoolean());
        ret_string = bidon.toString();
        break;
      case Tango_DEV_USHORT:
        // Note: The extract() return a signed 16bits short therefore we need
        //       to do the cast from signed to unsigned by hand
        ret_string = Integer.toString(received.extractUShort());
        break;
      case Tango_DEV_SHORT:
        // Note: Java int type is 32bits wide and is therefore mapped
        //       to Tango_DevLong (Java long type is 64bits wide)
        ret_string = Short.toString(received.extractShort());
        break;
      case Tango_DEV_ULONG:
        // Note: The extract() return a signed 32bits int therefore we need
        //       to do the cast from signed to unsigned by hand
        ret_string = Long.toString(received.extractULong());
        break;
      case Tango_DEV_LONG:
        ret_string = Integer.toString(received.extractLong());
        break;
      case Tango_DEV_FLOAT:
        ret_string = Float.toString(received.extractFloat());
        break;
      case Tango_DEV_DOUBLE:
        ret_string = Double.toString(received.extractDouble());
        break;
      case Tango_DEV_STRING:
        ret_string = received.extractString();
        break;
      case Tango_DEVVAR_CHARARRAY:
        {
          byte[] dummy = received.extractByteArray();
          ret_string += "array length: " + dummy.length + "\n";
          if (dummy.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.length; i++) {
              ret_string += "[" + i + "]\t " + Integer.toString(dummy[i]);
              if (dummy[i] >= 32)
                ret_string += " '" + (new Character((char) dummy[i]).toString()) + "'";
              else
                ret_string += " '.'";
              ret_string += "\n";
            }
          }
        }
        break;
      case Tango_DEVVAR_USHORTARRAY:
        // Note: The extract() return a signed 16bits short therefore we need
        //       to do the cast from signed to unsigned by hand
        {
          int[] dummy = received.extractUShortArray();
          ret_string += "array length: " + dummy.length + "\n";
          if (dummy.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.length; i++)
              ret_string += "[" + i + "]\t " + Integer.toString(dummy[i]) + "\n";
          }
        }
        break;
      case Tango_DEVVAR_SHORTARRAY:
        {
          short[] dummy = received.extractShortArray();
          ret_string += "array length: " + dummy.length + "\n";
          if (dummy.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.length; i++)
              ret_string += "[" + i + "]\t " + Short.toString(dummy[i]) + "\n";
          }
        }
        break;
      case Tango_DEVVAR_ULONGARRAY:
        {
          long[] dummy = received.extractULongArray();
          ret_string += "array length: " + dummy.length + "\n";
          if (dummy.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.length; i++)
              ret_string += "[" + i + "]\t " + Long.toString(dummy[i]) + "\n";
          }
        }
        break;
      case Tango_DEVVAR_LONGARRAY:
        {
          int[] dummy = received.extractLongArray();
          ret_string += "array length: " + dummy.length + "\n";
          if (dummy.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.length; i++)
              ret_string += "[" + i + "]\t " + Integer.toString(dummy[i]) + "\n";
          }
        }
        break;
      case Tango_DEVVAR_FLOATARRAY:
        {
          float[] dummy = received.extractFloatArray();
          ret_string += "array length: " + dummy.length + "\n";
          if (dummy.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.length; i++)
              ret_string += "[" + i + "]\t " + Float.toString(dummy[i]) + "\n";
          }
        }
        break;
      case Tango_DEVVAR_DOUBLEARRAY:
        {
          double[] dummy = received.extractDoubleArray();
          ret_string += "array length: " + dummy.length + "\n";
          if (dummy.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.length; i++)
              ret_string += "[" + i + "]\t" + Double.toString(dummy[i]) + "\n";
          }
        }
        break;
      case Tango_DEVVAR_STRINGARRAY:
        {
          String[] dummy = received.extractStringArray();
          ret_string += "array length: " + dummy.length + "\n";
          if (dummy.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.length; i++)
              ret_string += "[" + i + "]\t " + dummy[i] + "\n";
          }
        }
        break;
      case Tango_DEVVAR_LONGSTRINGARRAY:
        {
          DevVarLongStringArray dummy = received.extractLongStringArray();

          ret_string += "long array length: " + dummy.lvalue.length + "\n";
          ret_string += "lvalue:\n";
          if (dummy.lvalue.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.lvalue.length; i++)
              ret_string += "[" + i + "]\t " + Integer.toString(dummy.lvalue[i]) + "\n";
          }

          ret_string += "string array length: " + dummy.svalue.length + "\n";
          ret_string += "svalue:\n";
          if (dummy.svalue.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.svalue.length; i++)
              ret_string += "[" + i + "]\t " + dummy.svalue[i] + "\n";
          }
        }
        break;
      case Tango_DEVVAR_DOUBLESTRINGARRAY:
        {
          DevVarDoubleStringArray dummy = received.extractDoubleStringArray();

          ret_string += "double array length: " + dummy.dvalue.length + "\n";
          ret_string += "lvalue:\n";
          if (dummy.dvalue.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.dvalue.length; i++)
              ret_string += "[" + i + "]\t " + Double.toString(dummy.dvalue[i]) + "\n";
          }

          ret_string += "string array length: " + dummy.svalue.length + "\n";
          ret_string += "svalue:\n";
          if (dummy.svalue.length > ExecDev.answerLimit) {
            ret_string += "Array too long to be displayed. (You may change the AnswerLimit)\n";
          } else {
            for (int i = 0; i < dummy.svalue.length; i++)
              ret_string += "[" + i + "]\t " + dummy.svalue[i] + "\n";
          }
        }
        break;

      case Tango_DEV_STATE:
        ret_string = Tango_DevStateName[received.extractDevState().value()];
        break;
      default:
        break;
    }

    return ret_string;
  }

  /**
   *  Append the value of the given attribute to ret_string.
   */
  private static void appendAttributeValue(
          Object[] values,
          AttrWriteType writable,
          AttrDataFormat attr_format,
          int nb_read,
          StringBuffer ret_string) {

    // Treat specific print out for read_write attributes
    // The spectrum and image attributes are only pure read attributes
    if (attr_format.value() == AttrDataFormat._SCALAR) {

      // Add read point value (always available)
      if (values.length > 0) {
        if (writable == AttrWriteType.WRITE)
          ret_string.append("last written value: " + values[0].toString() + "\n");
        else
          ret_string.append("read point: " + values[0].toString() + "\n");
      }
      // Add set point value (available only for read_write attributes)
      if (values.length > 1)
        ret_string.append("set  point: " + values[1].toString() + "\n");

    } else {

      ret_string.append("array length: " + nb_read + "\n");
      if (values.length > ExecDev.answerLimit) {
        ret_string.append("Array too long to be displayed. (You may change the AnswerLimit)\n");
      } else {
        int i,j;
        for (i = 0; i < nb_read; i++)
          ret_string.append("[" + i + "]\t" + values[i].toString() + "\n");
        if (values.length > nb_read) {
          ret_string.append("\n");
          for (j = 0; i < values.length; j++, i++)
            ret_string.append("Set [" + j + "]\t" + values[i].toString() + "\n");
        }
      }

    }

  }

  /**
   * Conmvert DeviceAttribute to string for printing.
   */
  public static String deviceattribute_to_string(
          DeviceAttribute received,
          AttrWriteType writable,
          AttrDataFormat attr_format,
          int attr_type) {

    StringBuffer ret_string = new StringBuffer();

    //
    // Add some common information to all attributes
    //

    try {

      // Add the date of the measure in two formats
      TimeVal t = received.getTimeVal();
      ret_string.append("measure date: " + t.tv_sec + "sec   " + t.tv_usec + "usec\n");

      Date date = new Date((long) (t.tv_sec * 1000.0 + t.tv_usec / 1000.0));
      SimpleDateFormat dateformat =
              new SimpleDateFormat("dd/MM/yyyy HH:mm:ss");
      dateformat.setTimeZone(TimeZone.getDefault());
      ret_string.append("measure date: " + dateformat.format(date) + "\n");

      // Add the quality information
      AttrQuality q = received.getQuality();

      ret_string.append("quality: ");
      switch (q.value()) {
        case AttrQuality._ATTR_VALID:
          ret_string.append("VALID\n");
          break;
        case AttrQuality._ATTR_INVALID:
          ret_string.append("INVALID\n");
          return ret_string.toString();
        case AttrQuality._ATTR_ALARM:
          ret_string.append("ALARM\n");
          break;
        default:
          ret_string.append("UNKNOWN\n");
          break;
      }

      // Add dimension of the attribute but only if having a meaning
      switch (attr_format.value()) {
        case AttrDataFormat._SCALAR:
          break;
        case AttrDataFormat._SPECTRUM:
          ret_string.append("dim x: " + received.getDimX() + "\n");
          break;
        case AttrDataFormat._IMAGE:
          ret_string.append("dim x: " + received.getDimX() + "\n");
          ret_string.append("dim y: " + received.getDimY() + "\n");
          break;
        default:
          break;
      }

      // Add values
      switch (attr_type) {

        case Tango_DEV_STATE:
          ret_string.append(Tango_DevStateName[received.extractState().value()]+"\n");
          break;

        case Tango_DEV_UCHAR:
          {
            // Get the attribute values
            short[] dummy = received.extractUCharArray();
            Short[] dummy2 = new Short[dummy.length];
            for(int i=0;i<dummy.length;i++) dummy2[i]=new Short(dummy[i]);
            appendAttributeValue(dummy2,writable,attr_format,received.getNbRead(),ret_string);
          }
          break;

        case Tango_DEV_SHORT:
          {
            // Get the attribute values
            short[] dummy = received.extractShortArray();
            Short[] dummy2 = new Short[dummy.length];
            for(int i=0;i<dummy.length;i++) dummy2[i]=new Short(dummy[i]);
            appendAttributeValue(dummy2,writable,attr_format,received.getNbRead(),ret_string);
          }
          break;

        case Tango_DEV_BOOLEAN:
          {
            // Get the attribute values
            boolean[] dummy = received.extractBooleanArray();
            Boolean[] dummy2 = new Boolean[dummy.length];
            for(int i=0;i<dummy.length;i++) dummy2[i]=new Boolean(dummy[i]);
            appendAttributeValue(dummy2,writable,attr_format,received.getNbRead(),ret_string);
          }
          break;

        case Tango_DEV_USHORT:
          {
            // Get the attribute values
            int[] dummy = received.extractUShortArray();
            Integer[] dummy2 = new Integer[dummy.length];
            for(int i=0;i<dummy.length;i++) dummy2[i]=new Integer(dummy[i]);
            appendAttributeValue(dummy2,writable,attr_format,received.getNbRead(),ret_string);
          }
          break;

        case Tango_DEV_LONG:
          {
            int[] dummy = received.extractLongArray();
            Integer[] dummy2 = new Integer[dummy.length];
            for(int i=0;i<dummy.length;i++) dummy2[i]=new Integer(dummy[i]);
            appendAttributeValue(dummy2,writable,attr_format,received.getNbRead(),ret_string);
          }
          break;

        case Tango_DEV_DOUBLE:
          {
            double[] dummy = received.extractDoubleArray();
            Double[] dummy2 = new Double[dummy.length];
            for(int i=0;i<dummy.length;i++) dummy2[i]=new Double(dummy[i]);
            appendAttributeValue(dummy2,writable,attr_format,received.getNbRead(),ret_string);
          }
          break;

        case Tango_DEV_FLOAT:
          {
            float[] dummy = received.extractFloatArray();
            Float[] dummy2 = new Float[dummy.length];
            for(int i=0;i<dummy.length;i++) dummy2[i]=new Float(dummy[i]);
            appendAttributeValue(dummy2,writable,attr_format,received.getNbRead(),ret_string);
          }
          break;

        case Tango_DEV_STRING:
          {
            String[] dummy = received.extractStringArray();
            appendAttributeValue(dummy,writable,attr_format,received.getNbRead(),ret_string);
          }
          break;

        default:
          ret_string.append("Unsupported attribute type code="+attr_type+"\n");
          break;
      }

    } catch (DevFailed e) {

      return (new DevHelperDevFailed(e)).toString();

    }

    return ret_string.toString();
  }

  /**
   * Returns a description of the input argument syntax.
   */
  public static String cmd_argin_example(int command_in_type) {
    String ret_string = new String("Ex: ");


    switch (command_in_type) {
      case Tango_DEV_VOID:
        ret_string = new String("");
        break;
      case Tango_DEV_STATE:
        ret_string += "0 (16bits value)";
        break;
      case Tango_DEV_BOOLEAN:
        ret_string += "true,false or 0,1";
        break;
      case Tango_DEV_UCHAR:
        ret_string += "10 or 0xa (unsigned 8bits)";
        break;
      case Tango_DEV_USHORT:
        ret_string += "10 or 0xa (unsigned 16bits)";
        break;
      case Tango_DEV_SHORT:
        ret_string += "10 or 0xa (signed 16bits)";
        break;
      case Tango_DEV_ULONG:
        ret_string += "10 or 0xa (unsigned 32bits)";
        break;
      case Tango_DEV_LONG:
        ret_string += "10 or 0xa (signed 32bits)";
        break;
      case Tango_DEV_FLOAT:
        ret_string += "2.3 (32bits float)";
        break;
      case Tango_DEV_DOUBLE:
        ret_string += "2.3 (64bits float)";
        break;
      case Tango_DEV_STRING:
        ret_string = "quotes needed for string with space or special char";
        break;
      case Tango_DEVVAR_CHARARRAY:
        ret_string += "2,0xa,4 (signed 8bits)";
        break;
      case Tango_DEVVAR_USHORTARRAY:
        ret_string += "2,0xa,4 (unsigned 16bits)";
        break;
      case Tango_DEVVAR_SHORTARRAY:
        ret_string += "2,0xa,4 (signed 16bits)";
        break;
      case Tango_DEVVAR_ULONGARRAY:
        ret_string += "2,0xa,4 (unsigned 32bits)";
        break;
      case Tango_DEVVAR_LONGARRAY:
        ret_string += "2,0xa,4 (signed 32bits)";
        break;
      case Tango_DEVVAR_FLOATARRAY:
        ret_string += "2.3,4 (32bits floats)";
        break;
      case Tango_DEVVAR_DOUBLEARRAY:
        ret_string += "2.3,4 (64bits floats)";
        break;
      case Tango_DEVVAR_STRINGARRAY:
        ret_string += "Dance,\"the TANGO\" (quotes needed for string with space or special char)";
        break;
      case Tango_DEVVAR_LONGSTRINGARRAY:
        ret_string += "[1,2][A,\"B C\",D] (quotes needed for string with space or special char)";
        break;
      case Tango_DEVVAR_DOUBLESTRINGARRAY:
        ret_string = "[1.0,2.0][A,\"B C\",D] (quotes needed for string with space or special char)";
        break;
      default:
        ret_string = new String("");
        break;
    }


    return ret_string;
  }

  /**
   * Returns a description of the input argument syntax.
   */
  public static String att_argin_example(AttrInfo attInfo) {

    String ret_string = "";

    switch (attInfo.data_type_num) {

      case Tango_DEV_STATE:
        ret_string += "0 (16bits value)";
        break;
      case Tango_DEV_UCHAR:
        ret_string += "10 or 0xa (unsigned 8bits)";
        break;
      case Tango_DEV_BOOLEAN:
        ret_string += "true,false or 0,1";
        break;
      case Tango_DEV_USHORT:
        ret_string += "10 or 0xa (unsigned 16bits)";
        break;
      case Tango_DEV_SHORT:
        ret_string += "10 or 0xa (signed 16bits)";
        break;
      case Tango_DEV_LONG:
        ret_string += "10 or 0xa (signed 32bits)";
        break;
      case Tango_DEV_FLOAT:
        ret_string += "2.3 (32bits float)";
        break;
      case Tango_DEV_DOUBLE:
        ret_string += "2.3 (64bits float)";
        break;
      case Tango_DEV_STRING:
        ret_string = "quotes needed for string with space or special char";
        break;
      default:
        ret_string = new String("");
        break;

    }

    if (attInfo.data_format.equals("SPECTRUM")) {
      return "Array of " + ret_string + " Ex: a,b,c";
    } else if (attInfo.data_format.equals("IMAGE")) {
      return "Image of " + ret_string + " Ex: [a,b,c][d,e,f][g,h,i]";
    } else {
      return "Ex: " + ret_string;
    }

  }

}

