/*
 * JArrowButton.java
 */

package fr.esrf.tangoatk.widget.util;

import java.awt.event.*;
import java.awt.*;
import javax.swing.*;

/**
 * An arrow button that also have auto repeat capabilities.
 */
public class JArrowButton extends JComponent implements MouseListener, ActionListener {

  /** UP arrow */
  public static int UP = 1;
  /** Down arrow */
  public static int DOWN = 2;

  private int state;
  private int orientation;
  private Color lightColor;
  private Color darkColor;
  private Timer autoRepeat;
  private int timerDelay;

  // General constructor
  public JArrowButton() {

    setLayout(null);
    setBackground(new Color(200, 200, 200));
    setPreferredSize(new Dimension(20,20));
    setOpaque(false);
    state = 0;
    orientation = UP;
    timerDelay = 300;
    autoRepeat = new Timer(1000, this);
    addMouseListener(this);

  }

  /**
   * Sets the orientation
   * @param o Orientation value
   * @see #UP
   * @see #DOWN
   */
  public void setOrientation(int o) {
    orientation = o;
  }

  /**
   * Gets the orientation
   * @see #setOrientation
   */
  public int getOrientation() {
    return orientation;
  }

  /**
   * Sets the state.
   * @param s 1->Push 0->Released
   */
  public void setState(int s) {
    state = s;
  }

  public int getState() {
    return state;
  }

  public void setBackground(Color c) {

    if(getBackground()!=null && c.getRGB()==getBackground().getRGB())
        return;
    lightColor = computeShadowColor(c, 90);
    darkColor = computeShadowColor(c, -90);
    super.setBackground(c);

  }

  // Action listener interface (Generated by the timer)
  public void actionPerformed(ActionEvent e) {
    if (e.getSource() == autoRepeat) {

      if (autoRepeat.getDelay() != timerDelay)
        autoRepeat.setDelay(timerDelay);

      // Send mouseClicked to all listeners
      MouseListener[] listenerList = (MouseListener[]) (getListeners(MouseListener.class));
      MouseEvent m = new MouseEvent(this, MouseEvent.MOUSE_CLICKED,
                                    e.getWhen(), e.getModifiers(), 3, 3, 1, false);
      for (int i = 0; i < listenerList.length; i++) listenerList[i].mouseClicked(m);

    }
  }

  // MouseListener interface
  public void mouseClicked(MouseEvent e) {
  }

  public void mouseEntered(MouseEvent e) {
  }

  public void mouseExited(MouseEvent e) {
    state = 0;
    autoRepeat.stop();
    autoRepeat.setDelay(1000);
    repaint();
    //System.out.println("Exit");
  }

  public void mousePressed(MouseEvent e) {
    state = 1;
    autoRepeat.start();
    repaint();
    //System.out.println("Press");
  }

  public void mouseReleased(MouseEvent e) {
    state = 0;
    autoRepeat.stop();
    autoRepeat.setDelay(1000);
    repaint();
    //System.out.println("Release");
  }

  // Paint the component
  protected void paintComponent(Graphics g) {

    int w = getWidth() - 1;
    int h = getHeight() - 1;
    int m = w / 2;

// Draw background
    int pointx[] = new int[3];
    int pointy[] = new int[3];

    if (orientation == UP) {

      pointx[0] = 0;
      pointy[0] = h;
      pointx[1] = m;
      pointy[1] = 0;
      pointx[2] = w;
      pointy[2] = h;

      g.setColor(getBackground());
      g.fillPolygon(pointx, pointy, 3);

      // Draw border
      if (state == 0) {
        g.setColor(lightColor);
        g.drawLine(0, h, m, 0);
        g.setColor(darkColor);
        g.drawLine(m, 0, w, h);
        g.drawLine(w, h, 0, h);
      } else {
        g.setColor(darkColor);
        g.drawLine(0, h, m, 0);
        g.setColor(lightColor);
        g.drawLine(m, 0, w, h);
        g.drawLine(w, h, 0, h);
      }

    } else {

      pointx[0] = 0;
      pointy[0] = 0;
      pointx[1] = m;
      pointy[1] = h;
      pointx[2] = w;
      pointy[2] = 0;

      g.setColor(getBackground());
      g.fillPolygon(pointx, pointy, 3);

      // Draw border
      if (state == 0) {
        g.setColor(lightColor);
        g.drawLine(0, 0, m, h);
        g.setColor(darkColor);
        g.drawLine(m, h, w, 0);
        g.drawLine(w, 0, 0, 0);
      } else {
        g.setColor(darkColor);
        g.drawLine(0, 0, m, h);
        g.setColor(lightColor);
        g.drawLine(m, h, w, 0);
        g.drawLine(w, 0, 0, 0);
      }

    }

  }

  private void RGBtoYUV(Color c, double[] yuv) {
    double R = (double) c.getRed();
    double G = (double) c.getGreen();
    double B = (double) c.getBlue();

    yuv[0] = 0.299 * R + 0.587 * G + 0.114 * B;
    yuv[1] = -0.169 * R - 0.331 * G + 0.500 * B + 128.0;
    yuv[2] = 0.500 * R - 0.419 * G - 0.081 * B + 128.0;
  }

  private Color createColorFromYUV(double[] yuv) {

    int r = (int) (yuv[0] + 1.4075 * (yuv[2] - 128.0));
    int g = (int) (yuv[0] - 0.7169 * (yuv[2] - 128.0) - 0.3455 * (yuv[1] - 128.0));
    int b = (int) (yuv[0] + 1.779 * (yuv[1] - 128.0));

    if (r > 255) r = 255;
    if (r < 0) r = 0;
    if (g > 255) g = 255;
    if (g < 0) g = 0;
    if (b > 255) b = 255;
    if (b < 0) b = 0;
    return new Color(r, g, b);

  }

  // Compute lighter or darker color
  private Color computeShadowColor(Color c, double ratio) {

    double[] yuv = new double[3];
    RGBtoYUV(c,yuv);
    yuv[0] = yuv[0]+ratio;
    return createColorFromYUV(yuv);

  }

  /* main */
  public static void main(String args[]) {

    JFrame f = new JFrame();
    JArrowButton b = new JArrowButton();
    f.getContentPane().setLayout(null);
    f.getContentPane().add(b);
    b.setBounds(10, 10, 10, 10);
    f.setSize(200, 100);
    f.setVisible(true);

  }

}
