//+============================================================================
//
// file :               MultiAttribute.java
//
// description :        Java source code for the MultiAttribute class.
//			This class is used to manage attribute.
//			There is one object of this class
//			per device. This object is mainly an aggregate of 
//			Attribute objects
//
// project :            TANGO
//
// author(s) :          E.Taurel
//
// $Revision: 3.8 $
//
// $Log: MultiAttribute.java,v $
// Revision 3.8  2005/06/13 09:08:42  pascal_verdier
// Attribute historic buffer can be filled by trigger.
//
// Revision 3.7  2005/03/07 10:10:48  pascal_verdier
// Bug fixed on int READ_WRITE attribute fixed (DEV_LONG is int not a long)
//
// Revision 3.6  2004/05/14 13:47:58  pascal_verdier
// Compatibility with Tango-2.2.0 cpp
// (polling commands and attibites).
//
// Revision 3.5  2004/03/12 14:07:57  pascal_verdier
// Use JacORB-2.1
//
// Revision 2.0  2003/01/09 16:02:58  taurel
// - Update release number before using SourceForge
//
// Revision 1.1.1.1  2003/01/09 15:54:39  taurel
// Imported sources into CVS before using SourceForge
//
// Revision 1.6  2001/10/10 08:11:25  taurel
// See Tango WEB pages for list of changes
//
// Revision 1.5  2001/07/04 15:06:37  taurel
// Many changes due to new release
//
// Revision 1.2  2001/05/04 12:03:21  taurel
// Fix bug in the Util.get_device_by_name() method
//
// Revision 1.1.1.1  2001/04/04 08:23:54  taurel
// Imported sources
//
// Revision 1.3  2000/04/13 08:23:01  taurel
// Added attribute support
//
//
// copyleft :           European Synchrotron Radiation Facility
//                      BP 220, Grenoble 38043
//                      FRANCE
//
//-============================================================================

package fr.esrf.TangoDs;

import java.util.*;
import org.omg.CORBA.*;
import fr.esrf.Tango.*;
import fr.esrf.TangoApi.*;

/**
 * There is one instance of this class for each device. This class is mainly
 * an aggregate of Attribute or WAttribute objects. It eases management of 
 * multiple attributes
 *
 * @author	$Author: pascal_verdier $
 * @version	$Revision: 3.8 $
 */
 
public class MultiAttribute implements TangoConst
{
/**
 * The Attribute objects vector.
 *
 * This vector is often referred as the main attributes vector
 */
	protected	Vector			attr_list = new Vector();
/**
 * The list of writable attribute.
 *
 * It is a vector of index in the main attribute vector
 */
	protected	Vector			writable_attr_list = new Vector();
/**
 * The list of attribute with an alarm level defined.
 *
 * It is a vector of index in the main attribute vector
 */
	protected	Vector			alarm_attr_list = new Vector();
	
	private static final AttrProperty[] def_opt_prop = new AttrProperty[] {
	new AttrProperty("label",Tango_LabelNotSpec),
	new AttrProperty("description",Tango_DescNotSpec),
	new AttrProperty("unit",Tango_UnitNotSpec),	
	new AttrProperty("standard_unit",Tango_StdUnitNotSpec),
	new AttrProperty("display_unit",Tango_DispUnitNotSpec),
	new AttrProperty("format",Tango_FormatNotSpec),
	new AttrProperty("min_value",Tango_AlrmValueNotSpec),
	new AttrProperty("max_value",Tango_AlrmValueNotSpec),
	new AttrProperty("min_alarm",Tango_AlrmValueNotSpec),
	new AttrProperty("max_alarm",Tango_AlrmValueNotSpec),
	new AttrProperty("writable_attr_name",Tango_AssocWritNotSpec)};

/**
 * Create a new MultiAttribute object.
 *
 * This constructor will in-turn call the constructor of the Attribute or
 * WAttribute class of all the device class attributes.
 *
 * @param dev_name The device name
 * @param dev_class Reference to the device DeviceClass object
 * @exception DevFailed If the command sent to the database failed.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
 
	public MultiAttribute(String dev_name,DeviceClass dev_class) throws DevFailed
	{
		int i;
		Util.out4.println("Entering MultiAttribute class constructor for device " + dev_name);
	
//
// Retrieve attr name list
//

		Vector tmp_attr_list = dev_class.get_class_attr().get_attr_list();
		int nb_attr = tmp_attr_list.size();
	
//
// Get device attribute properties
//

		if (nb_attr != 0)
		{
			Util tg = Util.instance();
			String[] db_dev_attr_prop = null;
			
			if (tg._UseDb == true)
			{	
				String[] send_list = new String[nb_attr + 1];
	
				send_list[0] = dev_name;
				for (i = 0;i < nb_attr;i++)
					send_list[i + 1] = ((Attr)(tmp_attr_list.elementAt(i))).get_name();
			
				DeviceData send = new DeviceData();
				send.insert(send_list);	
				DeviceData received_prop = tg.get_database().command_inout("DbGetDeviceAttributeProperty",send);
		
				String[] db_attr_prop_list = received_prop.extractStringArray();
			
//
// Create a vector of strings with the values returned from db
//

				int nb_db_data = db_attr_prop_list.length - 2;
				db_dev_attr_prop = new String[nb_db_data];
				for (i = 0;i < nb_db_data;i++)
					db_dev_attr_prop[i] = db_attr_prop_list[i + 2];
			}
				
//
// Build property list for each attribute
//

			int ind = 0;
	
			for (i = 0;i < nb_attr;i++)
			{
	
//
// Get attribute class properties
//

				Attr attr = dev_class.get_class_attr().get_attr(((Attr)(tmp_attr_list.elementAt(i))).get_name());
				Vector class_prop = attr.get_class_properties();
				Vector def_user_prop = attr.get_user_default_properties();

//
// If the attribute has some properties defined at device level, build a vector
// of these properties
//
		
				Vector dev_prop = new Vector();

				if (tg._UseDb == true)
				{
					ind++;
					int nb_prop = Integer.parseInt(db_dev_attr_prop[ind]);
					ind++;

					for (int j = 0;j < nb_prop;j++)
					{
						dev_prop.addElement(new AttrProperty(db_dev_attr_prop[ind],
									     db_dev_attr_prop[ind + 1]));
						ind = ind + 2;
					}
				}

//
// Concatenate these two attribute properties levels
//
	
				Vector prop_list = new Vector();
				concat(dev_prop,class_prop,prop_list);
				add_user_default(prop_list,def_user_prop);
				add_default(prop_list);
		
//
// Check and get attribute data type
//

				int data_type = attr.get_type();
		
//
// Create an Attribute instance
//

				if ((attr.get_writable() == AttrWriteType.WRITE) || 
				    (attr.get_writable() == AttrWriteType.READ_WRITE))
					attr_list.addElement(new WAttribute(prop_list,attr,dev_name));
				else
					attr_list.addElement(new Attribute(prop_list,attr,dev_name));
		
//
// If it is writable, add it to the writable attribute list
//

				if ((((Attribute)(attr_list.elementAt(i))).get_writable() == AttrWriteType.WRITE) ||
				    (((Attribute)(attr_list.elementAt(i))).get_writable() == AttrWriteType.READ_WRITE))
				{
					writable_attr_list.addElement(new Integer(i));
				}
		
//
// If one of the alarm properties is defined, add it to the alarmed attribute
// list
//

				if (((Attribute)(attr_list.elementAt(i))).is_alarmed() == true)
				{
					if (((Attribute)(attr_list.elementAt(i))).get_writable() != AttrWriteType.WRITE)
						alarm_attr_list.addElement(new Integer(i));
				}
			
				Util.out4.println((Attribute)(attr_list.elementAt(i)));
			}
		}
	
//
// For each attribute, check if the writable_attr_name is set and in this
// case, check if the associated attribute exists and is writable
//

		for (i = 0;i < nb_attr;i++)
		{
			check_associated(i,dev_name);
		}
	
		Util.out4.println("Leaving MultiAttribute class constructor");
	}	

//+-------------------------------------------------------------------------
//
// method : 		concat
// 
// description : 	Concatenate porperties defined at the class level and
//			at the device level. Prperties defined at the device
//			level have the highest priority
//
// in :			dev_prop : The device properties
//			class_prop : The class properties
//
// out :		result : The resulting vector
//
//--------------------------------------------------------------------------


	private void concat(Vector dev_prop,
			    Vector class_prop,
			    Vector result)
	{
	
//
// Copy all device properties
//

		int i = 0;
		for (i = 0;i < dev_prop.size();i++)
			result.addElement(dev_prop.elementAt(i));
		
//
// Add class properties if they have not been redefined at the device level
//

		Vector tmp_result = (Vector)(result.clone());
		int nb_class_check = class_prop.size();
		for (i = 0;i < nb_class_check;i++)
		{
			int j = 0;
			for (j = 0;j < tmp_result.size();j++)
			{
				if (((AttrProperty)(tmp_result.elementAt(j))).get_name().equals(((AttrProperty)(class_prop.elementAt(i))).get_name()) == true)
				{
					break;
				}
			}
			
			if (j == tmp_result.size())
				result.addElement(class_prop.elementAt(i));
		}
	}

//+-------------------------------------------------------------------------
//
// method : 		add_default
// 
// description : 	Add default value for optional property if they
//			are not defined
//
// in :			prop_list : The already defined property vector
//
//--------------------------------------------------------------------------

	private void add_default(Vector prop_list)
	{
		int nb_opt_prop = def_opt_prop.length;

//
// For all the optional attribute properties, search in the already built
// vector of attributes if they are defined. If yes, continue. Otherwise,
// add a new property with the default value
//
	
		for (int i = 0;i < nb_opt_prop;i++)
		{
			String opt_prop_name = def_opt_prop[i].get_name();

			int j = 0;
			for (j = 0;j < prop_list.size();j++)
			{
				if (((AttrProperty)(prop_list.elementAt(j))).get_name().equals(opt_prop_name) == true)
					break;
			}
			if (j == prop_list.size())
			{
				prop_list.addElement(def_opt_prop[i]);
			}
		}
	}

//+-------------------------------------------------------------------------
//
// method : 		add_user_default
// 
// description : 	Add user default value for property if they
//			are not defined
//
// in :			prop_list : The already defined property vector
//			user_default : The user defined default property values
//
//--------------------------------------------------------------------------

	void add_user_default(Vector prop_list,
			      Vector user_default)
	{

//
// Add default user properties if they have not been defined in the database
//

		int nb_user = user_default.size();
		for (int i = 0;i < nb_user;i++)
		{
			String user_prop_name = ((AttrProperty)(user_default.elementAt(i))).get_name();

			int j = 0;
			for (j = 0;j < prop_list.size();j++)
			{
				if (((AttrProperty)(prop_list.elementAt(j))).get_name().equals(user_prop_name) == true)
					break;
			}
		
			if (j == prop_list.size())
				prop_list.addElement(user_default.elementAt(i));
		}
	}

//+-------------------------------------------------------------------------
//
// method : 		MultiAttribute::check_associated
// 
// description :	Check if the writable_attr_name property is set and
//			in this case, check if the associated attribute exists
//			and is writable. This is necessary only for attribute
//			of the READ_WITH_WRITE or READ_WRITE types 
//
// argument : in : 	- index : The index of the attribute to checked in the
//				  attr vector
//			- dev_name : The device name
//
//--------------------------------------------------------------------------

	void check_associated(int index,String dev_name) throws DevFailed
	{

		Attribute att = ((Attribute)(attr_list.elementAt(index)));
		if ((att.get_writable() == AttrWriteType.READ_WITH_WRITE) ||
		    (att.get_writable() == AttrWriteType.READ_WRITE))
		{
		
			if (att.get_data_format().value() != AttrDataFormat._SCALAR)
			{
				StringBuffer o = new StringBuffer("Device --> ");
				
				o.append(dev_name);
				o.append("\nProperty writable_attr_name for attribute ");
				o.append(att.get_name());
				o.append(" is defined but this attribute data format is not SCALAR");
				Except.throw_exception("API_AttrOptProp",
						       o.toString(),
						       "MultiAttribute.check_associated");
			}
			
			int j = 0;
			int tmp_ind = 0;
			String assoc_name = att.get_assoc_name();
			for (j = 0;j < writable_attr_list.size();j++)
			{
				tmp_ind = ((Integer)(writable_attr_list.elementAt(j))).intValue();
				if (((Attribute)(attr_list.elementAt(tmp_ind))).get_name().equals(assoc_name) == true)
					break;
			}
			if (j == writable_attr_list.size())
			{
				StringBuffer o = new StringBuffer("Device --> ");

				o.append(dev_name);		
				o.append("\nProperty writable_attr_name for attribute ");
				o.append(att.get_name()); 
				o.append(" is set to ");
				o.append(assoc_name);
				o.append(", but this attribute does not exists or is not writable");
				Except.throw_exception("API_AttrOptProp",
						       o.toString(),
						       "MultiAttribute.check_associated");
			}

//
// Also check if the associated write attribute is a scalar one
//

			Attribute tmp_att = ((Attribute)(attr_list.elementAt(tmp_ind)));			
			if (tmp_att.get_data_format().value() != AttrDataFormat._SCALAR)
			{
				StringBuffer o = new StringBuffer("Device --> ");

				o.append(dev_name);		
				o.append("\nProperty writable_attr_name for attribute ");
				o.append(att.get_name()); 
				o.append(" is set to ");
				o.append(assoc_name);
				o.append(", but this attribute is not of the SCALAR data format");
				Except.throw_exception("API_AttrOptProp",
						       o.toString(),
						       "MultiAttribute.check_assiocated");
			}

//
// Check that the two associated attributes have the same data type
//
			
			if (tmp_att.get_data_type() != att.get_data_type())
			{
				StringBuffer o = new StringBuffer("Device --> ");

				o.append(dev_name);		
				o.append("\nProperty writable_attr_name for attribute ");
				o.append(att.get_name()); 
				o.append(" is set to ");
				o.append(assoc_name);
				o.append(", but these two attributes do not support the same data type");
				Except.throw_exception("API_AttrOptProp",o.toString(),
						       "MultiAttribute.check_associated");
			}
			
			att.set_assoc_ind(tmp_ind);
		}

	}
	
//+-------------------------------------------------------------------------
//
// method : 		add_attribute
// 
// description : 	Construct a new attribute object and add it to the
//			device attribute list
//
// argument : in : 	- dev_name : The device name
//			- dev_class_ptr : Pointer to the DeviceClass object
//			- index : Index in class attribute list of the new
//				  device attribute
//
//--------------------------------------------------------------------------

	void add_attribute(String dev_name,DeviceClass dev_class,int index) throws DevFailed
	{
		Util.out4.println("Entering MultiAttribute::add_attribute");
		int i;
	
//
// Retrieve attr name list
//

		Vector tmp_attr_list = dev_class.get_class_attr().get_attr_list();
		int nb_attr = 1;
	
//
// Get device attribute properties
//

		String[] db_dev_attr_prop = null;
		if (Util._UseDb == true)
		{
			Util tg = Util.instance();
			String[] send_list = new String[2];
	
			send_list[0] = dev_name;
			send_list[1] = ((Attr)(tmp_attr_list.elementAt(index))).get_name();
			
			DeviceData send = new DeviceData();
			send.insert(send_list);	
			DeviceData received_prop = tg.get_database().command_inout("DbGetDeviceAttributeProperty",send);
		
			String[] db_attr_prop_list = received_prop.extractStringArray();		
	
//
// Create a vector of strings with the values returned from db
//

			int nb_db_data = db_attr_prop_list.length - 2;
			db_dev_attr_prop = new String[nb_db_data];
			for (i = 0;i < nb_db_data;i++)
				db_dev_attr_prop[i] = db_attr_prop_list[i + 2];
		}

		int ind = 0;
		
//
// Get attribute class properties
//

		Attr attr = dev_class.get_class_attr().get_attr(((Attr)(tmp_attr_list.elementAt(index))).get_name());
		Vector class_prop = attr.get_class_properties();
		Vector def_user_prop = attr.get_user_default_properties();

//
// If the attribute has some properties defined at device level, build a vector
// of these properties
//

		Vector dev_prop = new Vector();
		if (Util._UseDb == true)
		{		
			ind++;
			int nb_prop = Integer.parseInt(db_dev_attr_prop[ind]);
			ind++;

			for (int j = 0;j < nb_prop;j++)
			{
				dev_prop.addElement(new AttrProperty(db_dev_attr_prop[ind],
									     db_dev_attr_prop[ind + 1]));
				ind = ind + 2;
			}
		}

//
// Concatenate these two attribute properties levels
//
	
		Vector prop_list = new Vector();
		concat(dev_prop,class_prop,prop_list);
		add_user_default(prop_list,def_user_prop);
		add_default(prop_list);

//
// Check and get attribute data type
//

		int data_type = attr.get_type();
		
//
// Create an Attribute instance
//

		if ((attr.get_writable() == AttrWriteType.WRITE) || 
		    (attr.get_writable() == AttrWriteType.READ_WRITE))
			attr_list.addElement(new WAttribute(prop_list,attr,dev_name));
		else
			attr_list.addElement(new Attribute(prop_list,attr,dev_name));
		index = attr_list.size() - 1;
				
//
// If it is writable, add it to the writable attribute list
//

		if ((((Attribute)(attr_list.elementAt(index))).get_writable() == AttrWriteType.WRITE) ||
		    (((Attribute)(attr_list.elementAt(index))).get_writable() == AttrWriteType.READ_WRITE))
		{
			writable_attr_list.addElement(new Integer(index));
		}
		
//
// If one of the alarm properties is defined, add it to the alarmed attribute
// list
//

		if (((Attribute)(attr_list.elementAt(index))).is_alarmed() == true)
		{
			if (((Attribute)(attr_list.elementAt(index))).get_writable() != AttrWriteType.WRITE)
				alarm_attr_list.addElement(new Integer(index));
		}

//
// For each attribute, check if the writable_attr_name is set and in this
// case, check if the associated attribute exists and is writable
//

		check_associated(index,dev_name);
	
		Util.out4.println("Leaving MultiAttribute::add_attribute");
	}
//=================================================================
/**
 * Remove an attribute on the device attribute list.
 *
 * @param attname Attribute's name to be removed to the list
 * @throw DevFailed.
 */
//=================================================================
	public void remove_attribute(String attname) throws DevFailed
	{
		// Check if this attribute is already defined for this device.
		Attribute attr = get_attr_by_name(attname);
		
		//	And remove it
		attr_list.remove(attr);
	}

	
/**
 * Get Attribute object from its name.
 *
 * This method returns a reference to the Attribute object with a name passed
 * as parameter. The equality on attribute name is case independant.
 *
 * @param attr_name The attribute name
 * @return A reference to the Attribute object
 * @exception DevFailed If the attribute is not defined.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
 
	public Attribute get_attr_by_name(String attr_name) throws DevFailed
	{
		int i = 0;
		int nb = attr_list.size();
		
		for (i = 0;i < nb;i++)
		{
			if (((Attribute)(attr_list.elementAt(i))).get_name().equalsIgnoreCase(attr_name) == true)
				break;
		}
		      
		if (i == nb)
		{
			Util.out3.println("MultiAttribute.get_attr throwing exception");
			Except.throw_exception("API_AttrNotFound",
					    attr_name + " attribute not found" ,
					     "MultiAttribute.get_attr");
		}
	
		return ((Attribute)(attr_list.elementAt(i)));
	}

/**
 * Get Writable Attribute object from its name.
 *
 * This method returns a reference to the WAttribute object with a name passed
 * as parameter. The equality on attribute name is case independant.
 *
 * @param attr_name The attribute name
 * @return A reference to the writable attribute object
 * @exception DevFailed If the attribute is not defined.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */

	public WAttribute get_w_attr_by_name(String attr_name) throws DevFailed
	{
		Attribute att = get_attr_by_name(attr_name);
		
		return ((WAttribute)(att));
	} 

/**
 * Get Attribute index into the main attribute vector from its name.
 *
 * This method returns the index in the Attribute vector (stored in the
 * MultiAttribute object) of an attribute with a given name. The name equality
 * is case independant
 *
 * @param attr_name The attribute name
 * @return The index in the main attributes vector
 * @exception DevFailed If the attribute is not found in the vector.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 */
 
	public int get_attr_ind_by_name(String attr_name) throws DevFailed
	{
		int i;

		int nb_attr = attr_list.size();
		for (i = 0;i < nb_attr;i++)
		{
			if (((Attribute)(attr_list.elementAt(i))).get_name().equalsIgnoreCase(attr_name) == true)
				break;
		}
	
		if (i == nb_attr)
		{
			Util.out3.println("MultiAttribute.get_attr_ind_by_name throwing exception");
			
			StringBuffer o = new StringBuffer(attr_name);
			o.append(" attribute not found");
			
			Except.throw_exception("API_AttrNotFound",
					       o.toString(),
					       "MultiAttribute.get_attr_ind_by_name");
		}
	
		return i;
	}

/**
 * Check alarm on all attribute(s) with an alarm defined.
 *
 * This method returns a boolean set to true if one of the attribute with an 
 * alarm level defined is in alarm condition.
 *
 * @return A boolean set to true if one attribute is in alarm
 * @exception DevFailed If the alarm level are not defined for one of the
 * attribute in the list of alarmable one
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 *
 */
 
	public boolean check_alarm() throws DevFailed
	{
		boolean ret = false;
		int nb_alarm = alarm_attr_list.size();

		for (int i = 0;i < nb_alarm;i++)
		{
			boolean tmp_ret = check_alarm(((Integer)(alarm_attr_list.elementAt(i))).intValue());
			if (tmp_ret == true)
			{
				ret = true;
				break;
			}
		}
	
		return ret;
	}

/**
 * Check alarm for one attribute with a given name.
 *
 * This method returns a boolean set to true if the attribute with the given
 * name is in alarm condition
 *
 * @param  attr_name  The attribute name
 * @return A boolean set to true if the attribute is in alarm
 * @exception DevFailed If the attribute does not have any alarm level defined.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 *
 */
 	
	public boolean check_alarm(String attr_name) throws DevFailed
	{
		return get_attr_by_name(attr_name).check_alarm();
	}

/**
 * Check alarm for one attribute from its index in the main attributes vector.
 *
 * This method returns a boolean set to true if the attribute with the given
 * index in the attrobite object vector is in alarm condition
 *
 * @param  ind  The attribute index
 * @return A boolean set to true if the attribute is in alarm
 * @exception DevFailed If the attribute does not have any alarm level defined.
 * Click <a href="../../tango_basic/idl_html/Tango.html#DevFailed">here</a> to read
 * <b>DevFailed</b> exception specification
 *
 */
 	
	public boolean check_alarm(int ind) throws DevFailed
	{
		return get_attr_by_ind(ind).check_alarm();
	}

//+-------------------------------------------------------------------------
//
// method : 		add_write_value
// 
// description : 	For scalar attribute with an associated write 
//			attribute, the read_attributes CORBA operation also
//			returns the write value. This method gets the associated
//			write attribute value and adds it to the read
//			attribute
//
// in :			att : Reference to the attribute which must be read
//
//--------------------------------------------------------------------------

	void add_write_value(Attribute att)
	{
		WAttribute assoc_att = get_w_attr_by_ind(att.get_assoc_ind());
	
		switch (att.get_data_type())
		{
		case Tango_DEV_SHORT :
			short sh_write_val = assoc_att.get_sh_write_value();		
			att.add_write_value(sh_write_val);
			break;
		
		case Tango_DEV_LONG :
			int lg_write_val = assoc_att.get_lg_write_value();		
			att.add_write_value(lg_write_val);
			break;
		
		case Tango_DEV_DOUBLE :
			double db_write_val = assoc_att.get_db_write_value();		
			att.add_write_value(db_write_val);
			break;
		
		case Tango_DEV_STRING :
			String str_write_val = assoc_att.get_str_write_value();		
			att.add_write_value(str_write_val);
			break;
		}
	}
	
//+-------------------------------------------------------------------------
//
// method : 		read_alarm
// 
// description : 	Add a message in the device status string if one of
//			the device attribute is in the alarm state
//
// in :			status : The device status
//
//--------------------------------------------------------------------------

/**
 * Add alarm message to device status.
 *
 * This method add alarm mesage to the string passed as parameter. A message
 * is added for each attribute which is in alarm condition
 *
 * @param  status  The string (should be the device status)
 */
	public void read_alarm(StringBuffer status)
	{
		int i = 0;
	
		for (i = 0;i < alarm_attr_list.size();i++)
		{
			Attribute att = get_attr_by_ind(((Integer)(alarm_attr_list.elementAt(i))).intValue());
			if (att.get_quality().value() == AttrQuality._ATTR_ALARM)
			{
		
//
// Add a message for low level alarm
//

				if (att.is_min_alarm() == true)
				{
					status.append("\nAlarm : Value too low for attribute ");
					status.append(att.get_name());				
				}

//
// Add a message for high level alarm
//
			
				if (att.is_max_alarm() == true)
				{
					status.append("\nAlarm : Value too high for attribute ");
					status.append(att.get_name());
				}
			}			
		}
	}

					
//+-------------------------------------------------------------------------
//
// Methods to retrieve/set some data members from outside the class and all
// its inherited classes
//
//--------------------------------------------------------------------------

/**
 * Get list of attribute with an alarm level defined.
 *
 * @return  A vector of Integer object. Each object is the index in the main
 * attribute vector of attribute with alarm level defined
 */
 
	public Vector get_alarm_list()
	{
		return alarm_attr_list;
	}

/**
 * Get attribute number.
 *
 * @return  The attribute number
 */
 	
	public int get_attr_nb()
	{
		return attr_list.size();
	}
	
/**
 * Get Attribute object from its index.
 *
 * This method returns a reference to the Attribute object from the index in the
 * main attribute vector
 *
 * @param ind The attribute index
 * @return A reference to the Attribute object
 */
 	
	public Attribute get_attr_by_ind(int ind)
	{
		return ((Attribute)(attr_list.elementAt(ind)));
	}

/**
 * Get Writable Attribute object from its index.
 *
 * This method returns a reference to the Writable Attribute object from the
 * index in the main attribute vector
 *
 * @param ind The attribute index
 * @return A reference to the WAttribute object
 */
 	
	public WAttribute get_w_attr_by_ind(int ind)
	{
		return ((WAttribute)(attr_list.elementAt(ind)));
	}
}
