//+===========================================================================
// $Source: /cvsroot/tango-cs/tango/api/java/fr/esrf/TangoApi/TacoDevice.java,v $
//
// Project:   Tango API
//
// Description:  Java source for conversion between Tango/TACO library
//
// $Author: pascal_verdier $
//
// $Revision: 3.6 $
//
// $Log: TacoDevice.java,v $
// Revision 3.6  2004/12/07 09:30:30  pascal_verdier
// Exception classes inherited from DevFailed added.
//
// Revision 3.5  2004/03/12 13:15:23  pascal_verdier
// Using JacORB-2.1
//
// Revision 3.1  2004/03/08 11:35:40  pascal_verdier
// AttributeProxy and aliases management added.
// First revision for event management classes.
//
// Revision 3.0  2003/04/29 08:03:29  pascal_verdier
// Asynchronous calls added.
// Logging related methods.
// little bugs fixed.
//
// Revision 2.0  2003/01/09 14:00:37  verdier
// jacORB is now the ORB used.
//
// Revision 1.8  2002/06/26 09:02:17  verdier
// tested with atkpanel on a TACO device
//
// Revision 1.7  2002/04/09 12:21:51  verdier
// IDL 2 implemented.
//
// Revision 1.6  2002/01/09 12:18:15  verdier
// TACO signals can be read as TANGO attribute.
//
// Revision 1.5  2001/12/10 14:19:42  verdier
// TACO JNI Interface added.
// URL syntax used for connection.
// Connection on device without database added.
//
// Revision 1.3  2001/11/08 10:47:27  verdier
// Package added.
//
// Revision 1.2  2001/11/07 14:45:02  verdier
// Standard argument types implemented.
//
// Revision 1.1  2001/10/24 07:19:24  verdier
// Initial revision
//
//
// Copyright 2001 by European Synchrotron Radiation Facility, Grenoble, France
//               All Rights Reversed
//-===========================================================================
//         (c) - Software Engineering Group - ESRF
//============================================================================


package fr.esrf.TangoApi;


import fr.esrf.Tango.*;
import fr.esrf.TangoDs.*;
import fr.esrf.TangoApi.*;

import java.util.*;


/**
 *	<b>Class Description:</b><Br>
 *	This class loads a library (written in cpp) to access TACO device.<Br>
 *	This library must be found in the LD_LIBRARY_PATH environment variable.<Br>
 *	The communication between this class and the library use the JNI convention.<Br>
 *	The connection to the TACO device is done when the class is instancied 
 *	(when the constructor is called) and a DevVarCmdArray is created for
 *	furthur command.<Br>
 *	If the input/output data are not a sequence, the data are passed to the
 *	library as a fr.esrf.TangoApi.DeviceData object.<Br>
 *	If the input/output data are java.lang.String array, the data are passed 
 *	to the library as an array of java.lang.Object.<Br>
 *	Else they are passed to the library as an array of native type.
 *
 *
 * @author  verdier
 * @version	$Revision: 3.6 $
 */


public class TacoDevice implements TangoConst
{
	//======================================================
	//	Done when class is instancied.
	//======================================================
	static
	{
		//	Load the Taco C++ library
		System.loadLibrary("jtaco");
	}



	//======================================================
	//	Native methods (call the c++ lib)
	//======================================================
	/**
	 *	Native method to create TacoDevice object and
	 *	create a connection oin the Taco device.
	 */
	native int		importDevice(String devname)		throws DevFailed;
	/**
	 *	Native method to get the Taco device command list.
	 */
	native Object[]	commandListQuery(int devnum)		throws DevFailed;
	/**
	 *	Native method to get the Taco device information.
	 */
	native Object[]	devInform(int devnum)	throws DevFailed;
	/**
	 *	Native method to set the Taco RCP protocol in UDP or TCP mode.
	 */
	native void	rpcProtocol(int devnum, int mode)	throws DevFailed;

	/**
	 *	Native method to get the RCP protocol timeout.
	 */
	native int	getRpcTimeout(int devnum)	throws DevFailed;

	/**
	 *	Native method to set the RCP protocol timeout.
	 */
	native int	setRpcTimeout(int devnum, int ms)	throws DevFailed;








	/**
	 *	Native method to insert input data in the argin Taco command
	 * if argin is not a sequence.
	 */
	native void		tacoInsertObject(int devnum, Object argin, int in_type)
														throws DevFailed;
	/**
	 *	Native method to insert input data in the argin Taco command
	 * if argin is DevCharArray
	 */
	native void		tacoInsertByteArray(int devnum, byte[] argin, int length)
														throws DevFailed;
	/**
	 *	Native method to insert input data in the argin Taco command
	 * if argin is DevShortArray
	 */
	native void		tacoInsertShortArray(int devnum, short[] argin, int length)
														throws DevFailed;
	/**
	 *	Native method to insert input data in the argin Taco command
	 * if argin is DevLongArray (int)
	 */
	native void		tacoInsertIntArray(int devnum, int[] argin, int length)
														throws DevFailed;
	/**
	 *	Native method to insert input data in the argin Taco command
	 * if argin is DevFloatArray
	 */
	native void		tacoInsertFloatArray(int devnum, float[] argin, int length)
														throws DevFailed;
	/**
	 *	Native method to insert input data in the argin Taco command
	 * if argin is DevLongArray
	 */
	native void		tacoInsertDoubleArray(int devnum, double[] argin, int length)
														throws DevFailed;
	/**
	 *	Native method to insert input data in the argin Taco command
	 * if argin is DevStringArray passed as Object array.
	 */
	native void		tacoInsertObjArray(int devnum, Object[] argin)throws DevFailed;






	/**
	 *	Native method to send a command to the Taco device
	 *	and return argout as DeviceData if it is not a sequence..
	 */
	native Object	commandInout(int devnum, String cmdname)	throws DevFailed;
	/**
	 *	Native method to send a command to the Taco device
	 *	and return argout if it is a DevStringArray.
	 */
	native Object[]	commandInoutObjArray(int devnum, String cmdname)	throws DevFailed;
	


	/**
	 *	Native method to send a command to the Taco device
	 *	and return argout if it is a DevCharArray.
	 */
	native byte[]	commandInoutByteArray(int devnum, String cmdname)	throws DevFailed;

	/**
	 *	Native method to send a command to the Taco device
	 *	and return argout if it is a DevShortArray.
	 */
	native short[]	commandInoutShortArray(int devnum, String cmdname)	throws DevFailed;

	/**
	 *	Native method to send a command to the Taco device
	 *	and return argout if it is a DevLongArray (int).
	 */
	native int[]	commandInoutIntArray(int devnum, String cmdname)	throws DevFailed;

	/**
	 *	Native method to send a command to the Taco device
	 *	and return argout if it is a DevFloatArray.
	 */
	native float[]	commandInoutFloatArray(int devnum, String cmdname)	throws DevFailed;

	/**
	 *	Native method to send a command to the Taco device
	 *	and return argout if it is a DevDoubleArray.
	 */
	native double[]	commandInoutDoubleArray(int devnum, String cmdname)	throws DevFailed;




	//======================================================
	//	Fields
	//======================================================
	/**
	 *	the device name
	 */
	private	String	devname;
	/**
	 *	The device ID
	 */
	private	int	dev;
	/**
	 *	The result of command_list_query for this device.
	 */
	public	DevVarCmdArray	cmd_info = null;
	/**
	 *	Result of DevGetSigConfig command
	 */
	private String[]	attr_strconfig;
	/**
	 *	Attribute data type.
	 */
	private int		attr_data_type = -1;
	private int		attr_orig_type = -1;

	//======================================================
	/**
	 *	Constructor for a Taco Device Object.
	 *
	 *	@param	name	the device name
	 */
	//======================================================
	public TacoDevice(String name) throws DevFailed
	{
		this.devname = name ;
		dev = importDevice(name);
		read_commandListQuery();	
		//System.out.println("new TacoDevice("+name+")");
	}
	//======================================================
	/**
	 *	Execute the dev_cmd_query on Taco device.
	 *	This command is equivalent of command_list_query in Tango.
	 */ 
	//======================================================
	private void read_commandListQuery() throws DevFailed
	{
		//	Call command list query in c++
		//---------------------------------------
		Object[]	obj = commandListQuery(dev);

		//	Covert Object array to DevCmdInfo array
		//----------------------------------------------
		DevCmdInfo[]	info = new DevCmdInfo[obj.length];
		for (int i=0 ; i<obj.length ; i++)
			info[i] = (DevCmdInfo) obj[i];

		cmd_info = new DevVarCmdArray(devname, info);
	}
	//======================================================
	/**
	 *	returns the result of dev_cmd_query Taco command.
	 */
	//======================================================
	public CommandInfo[] commandListQuery() throws DevFailed
	{
		if (cmd_info==null)
			read_commandListQuery();
		return cmd_info.getInfoArray();
	}
	//======================================================
	/**
	 *	returns the result of dev_cmd_query Taco command for specified command
	 *	@param	cmdname	the command specified name .
	 */
	//======================================================
	public CommandInfo commandQuery(String cmdname) throws DevFailed
	{
		if (cmd_info==null)
			read_commandListQuery();
		CommandInfo[]	info = cmd_info.getInfoArray();
		CommandInfo		result = null;
		for (int i=0 ; i<info.length ; i++)
			if (info[i].cmd_name.equals(cmdname))
				result = info[i];
		if (result==null)
			Except.throw_wrong_syntax_exception("BAD_PARAMETER",
							"Command name not found for " + devname,
							"TacoDevice.commandQuery()");
		//System.out.println(result);
		return result;
	}

	//======================================================
	/**
	 *	Return TACO device information as String array.
	 *	<li> Device name.
	 *	<li> Class name
	 *	<li> Device type
	 *	<li> Device server name
	 *	<li> Host name
	 */
	//======================================================
	public String[] dev_inform() 	throws DevFailed
	{
		Object[]	obj = devInform(dev);
		String[]	info = new String[obj.length];
		for (int i=0 ; i<obj.length ; i++)
			info[i] = (String)obj[i];
		return info;
	}

	//======================================================
	/**
	 *	Execute the dev_rpc_protocol TACO command to change RPC protocol mode.
	 *	@param	mode RPC protocol mode to be seted 
	 *		(TangoApi.ApiDefs.<b>D_TCP</b> or ApiDefs.TacoDevice.<b>D_TCP</b>).
	 */
	//======================================================
	public void dev_rpc_protocol(int mode)	throws DevFailed
	{
		if (mode != ApiDefs.D_UDP  &&  mode != ApiDefs.D_TCP)
			Except.throw_wrong_syntax_exception("BAD_PARAMETER",
							"Bad parameter for dev_rpc_protocol command",
							"TacoDevice.dev_rpc_protocol()");

		//	Call native method in c++
		//---------------------------------------
		rpcProtocol(dev, mode);
	}
	//======================================================
	/**
	 *	Execute the dev_rpc_timeout TACO command to get RPC timeout.
	 */
	//======================================================
	public int get_rpc_timeout()	throws DevFailed
	{
		return getRpcTimeout(dev);
	}

	//======================================================
	/**
	 *	Execute the dev_rpc_timeout TACO command to change RPC timeout.
	 *	
	 *	@param	ms	new timeout in milliseconds.
	 *	@return the new timeout value;
	 */
	//======================================================
	public int set_rpc_timeout(int ms)	throws DevFailed
	{
		return setRpcTimeout(dev, ms);
	}

	//======================================================
	//======================================================










	//======================================================
	/**
	 *	Execute a dev_putget on the Taco device.
	 *
	 *	@param	cmdname	The command name.
	 *	@param	in_data	Input parameter (argin in taco).
	 *	@return	the output command argument (argout in Taco).
	 */
	//======================================================
	public DeviceData command_inout(String cmdname, DeviceData in_data)
						throws DevFailed
	{
		DeviceData	argout = new DeviceData();
		int	in_type  = cmd_info.arginType(cmdname);
		int	out_type = cmd_info.argoutType(cmdname);
				
		//	check if args are supported type
		//--------------------------------------------
		if (in_type<0)
			Except.throw_wrong_syntax_exception("TangoApi_ARG_NOT_SUPPORTED",
							"Input argument type not supported",
							"TacoDevice.command_inout()");
		if (out_type<0)
			Except.throw_wrong_syntax_exception("TangoApi_ARG_NOT_SUPPORTED",
							"Output argument type not supported",
							"TacoDevice.command_inout()");
		
		switch(in_type)
		{
		//	In case of a non sequence argument,
		//	use DeviceData Object to transfer data
		//---------------------------------------------
		case Tango_DEV_VOID:
		case Tango_DEV_BOOLEAN:
		case Tango_DEV_SHORT:
		case Tango_DEV_LONG:
		case Tango_DEV_FLOAT:
		case Tango_DEV_DOUBLE:
		case Tango_DEV_USHORT:
		case Tango_DEV_ULONG:
			//	if input is not a sequence then
			//	insert it inside a DeviceData object.
			//-----------------------------------------
			{
				Object	argin = null;
				if (in_type!=Tango_DEV_VOID)
				{
					argin = new Object();
					argin = (Object)in_data;
				}
				//	Call commandInout in c++
				//---------------------------------------
				tacoInsertObject(dev, argin, T2Ttypes.tacoType(in_type));
			}
			break;
		case Tango_DEV_STRING:
			{
				Object	argin = null;
				argin = new Object();
				argin = (Object)in_data.extractString();
				//	Call commandInout in c++
				//---------------------------------------
				tacoInsertObject(dev, argin, T2Ttypes.tacoType(in_type));
			}
			break;
		//	if argin is a sequence insert it in argin before device call
		//----------------------------------------------------------------
		case Tango_DEVVAR_CHARARRAY:
			{
				byte[]		array = in_data.extractByteArray();
				tacoInsertByteArray(dev, array, array.length);
			}
			break;
		//	if argin is a sequence insert it in argin before device call
		//----------------------------------------------------------------
		case Tango_DEVVAR_SHORTARRAY:
			{
				short[]		array = in_data.extractShortArray();
				tacoInsertShortArray(dev, array, array.length);
			}
			break;
		//	if argin is a sequence insert it in argin before device call
		//----------------------------------------------------------------
		case Tango_DEVVAR_LONGARRAY:
			{
				int[]		array = in_data.extractLongArray();
				tacoInsertIntArray(dev, array, array.length);
			}
			break;
		//	if argin is a sequence insert it in argin before device call
		//----------------------------------------------------------------
		case Tango_DEVVAR_FLOATARRAY:
			{
				float[]		array = in_data.extractFloatArray();
				tacoInsertFloatArray(dev, array, array.length);
			}
			break;
		//	if argin is a sequence insert it in argin before device call
		//----------------------------------------------------------------
		case Tango_DEVVAR_DOUBLEARRAY:
			{
				double[]	array = in_data.extractDoubleArray();
				tacoInsertDoubleArray(dev, array, array.length);
			}
			break;
		//	if argin is a String sequence insert it in an object array
		//---------------------------------------------------------------
		case Tango_DEVVAR_STRINGARRAY:
			{

				String[]	array = in_data.extractStringArray();
				Object[]	obj = new Object[array.length];
				for(int i=0 ; i<array.length ; i++)
					obj[i] = array[i];
				tacoInsertObjArray(dev, obj);			
			}
			break;
		}

		switch (out_type)
		{
		//	In case of a non sequence argument,
		//	use DeviceData Object to transfer data
		//---------------------------------------------
		case Tango_DEV_VOID:
		case Tango_DEV_BOOLEAN:
		case Tango_DEV_SHORT:
		case Tango_DEV_LONG:
		case Tango_DEV_FLOAT:
		case Tango_DEV_DOUBLE:
		case Tango_DEV_USHORT:
		case Tango_DEV_ULONG:
		case Tango_DEV_STRING:
			{
				Object	obj = commandInout(dev, cmdname);
				//	if no argout -> return null
				//---------------------------------
				if (obj==null)	return null;
				//	Create output object
				//---------------------------------
				argout = (DeviceData)obj;
			}
			break;

		//	read device and insert out data in DeviceData object
		case Tango_DEVVAR_CHARARRAY:
				argout.insert(commandInoutByteArray(dev, cmdname));				
				break;

		//	read device and insert out data in DeviceData object
		case Tango_DEVVAR_SHORTARRAY:
				argout.insert(commandInoutShortArray(dev, cmdname));				
				break;

		//	read device and insert out data in DeviceData object
		case Tango_DEVVAR_LONGARRAY:
				argout.insert(commandInoutIntArray(dev, cmdname));				
				break;

		//	read device and insert out data in DeviceData object
		case Tango_DEVVAR_FLOATARRAY:
				argout.insert(commandInoutFloatArray(dev, cmdname));				
				break;

		//	read device and insert out data in DeviceData object
		case Tango_DEVVAR_DOUBLEARRAY:
				argout.insert(commandInoutDoubleArray(dev, cmdname));				
				break;


		//	Extract Strings from objects and insert inside DeviceData
		case Tango_DEVVAR_STRINGARRAY:
				Object[]	obj = commandInoutObjArray(dev, cmdname);
				String[]	str = new String[obj.length];
				for (int i=0 ; i<obj.length ; i++)
					str[i] = (String)obj[i];
				argout.insert(str);				
				break;
		}

		return argout;
	}
	//======================================================
	//
	//	ATTRIBUTE MANAGEMENT
	//
	//======================================================
	private	boolean	isWildcard(String s, String wildcard)
	{
		StringTokenizer st = new StringTokenizer(wildcard, "*");
		switch(st.countTokens())
		{
		case 0:		//	"*" char -> OK for all
			return true;

		case 1:		//	just one string
		{
			String	pattern = st.nextToken();
			if (s.indexOf(pattern)<0)
				return false;
			else
				return true;
		}
		case 2:		//	2 strings separated by a "*"
		{
			String[]	pattern = new String[2];
			pattern[0] = st.nextToken();
			pattern[1] = st.nextToken();
			int	start;
			if ((start=s.indexOf(pattern[0]))<0)
				return false;	//	first string not found
			else
			if (s.indexOf(pattern[1], start)<0)
				return false;	//	second string not found
			else
				return true;	//	both found in order
		}
		default:
			return false;
		}
		
	}
	//======================================================
	//======================================================
	private String onlyAttName(String fullname) throws DevFailed
	{
		int	start = 0;
		for (int i=0 ; i<3 ; i++, start++)
		{
			start = fullname.indexOf('/', start);
			if (start<0)
				Except.throw_wrong_syntax_exception("BAD_PARAMETER",
						new String("Attribute name not found in " + fullname),
						"TacoDevice.onlyAttName()");
		}
		return fullname.substring(start);
	}
	//======================================================
	//======================================================
	private String fullName(String attname)
	{
		return new String(devname + "/" + attname);
	}
	//======================================================
	//======================================================
	String[] get_attribute_list() throws DevFailed
	{
		String[]	attr_list = null;
		DeviceData	argout = command_inout("DevGetSigConfig", null);
		String[]	config = argout.extractStringArray();
		int			modulo = Integer.parseInt(config[0]);
		
		int	nb_attr = config.length/modulo;	//	All attributes

		//	Allocate an array of string and fill it
		attr_list = new String[nb_attr];
		for (int i=1, j=0 ; i<config.length ; i+=modulo)
		{
			attr_list[j++] = onlyAttName(config[i]);
			//System.out.println(j-1 + ": " + attr_list[j-1]);
		}
		return attr_list;
	}
	//======================================================
	//======================================================
	AttributeConfig[] get_attribute_config(String[] attrnames) throws DevFailed
	{
		int			nb_x = 1;
		int			nb_y = 0;

		//	Read Signal config if not already done
		if (attr_strconfig==null)
		{
			try
			{
				DeviceData	argout = command_inout("DevGetSigConfig", null);
				attr_strconfig = argout.extractStringArray();
				setAttrDataType();
			}
			catch(DevFailed e)
			{
				//	Check if command implemented (if not -> no signal available)
				if (e.errors[0].reason.equals("TACO_CMD_UNAVAILABLE"))
					return new AttributeConfig[0];
				else
					throw e;
			}
		}
		
		//	allocate output array (for all attr or just some of them ?)
		int					modulo = Integer.parseInt(attr_strconfig[0]);
		AttributeConfig[]	config;
		if (attrnames[0].equals("All attributes"))
		{
			config = new AttributeConfig[attr_strconfig.length/modulo];
			//	reconstruct attrnames with all attr found
			attrnames = new String[attr_strconfig.length/modulo];
			for (int i=0, j=1 ; j<attr_strconfig.length ; j+=modulo)
				attrnames[i++] = onlyAttName(attr_strconfig[j]);
		}
		else
			config = new AttributeConfig[attrnames.length];
	
	
		for (int i=0 ; i<attrnames.length ; i++)
		{
			//	search attribute name
			String	fullname = fullName(attrnames[i]);
			boolean found = false;
			for (int j=1 ; j<attr_strconfig.length ; j+=modulo)
			{
				if (attr_strconfig[j].equals(fullname))
				{
					//	default initilization of config fields
					String		description = "";
					String		label = "";
					String		unit = "";
					String		standard_unit = "";
					String		display_unit = "";
					String		format = "";
					String		min_value = "";
					String		max_value = "";
					String		min_alarm = "";
					String		max_alarm = "";
					String		writable_attr_name = "";
					String[]	extensions = null;
					found = true;

					//	Fill values
					String	not_specified = "No Description";
					if (attr_strconfig[j+4].equals(not_specified)==false)
						description = attr_strconfig[j+4];
					not_specified = "Not specified";
					if (attr_strconfig[j+1].equals(not_specified)==false)
						label = attr_strconfig[j+1];
					if (attr_strconfig[j+2].equals(not_specified)==false)
					{
						unit          = attr_strconfig[j+2];
						display_unit  = attr_strconfig[j+2];	
					}
					if (attr_strconfig[j+3].equals(not_specified)==false)
						format = attr_strconfig[j+3];
					if (attr_strconfig[j+5].equals(not_specified)==false)
						max_value = attr_strconfig[j+5];
					if (attr_strconfig[j+6].equals(not_specified)==false)
						min_value = attr_strconfig[j+6];
					if (attr_strconfig[j+7].equals(not_specified)==false)
						max_alarm = attr_strconfig[j+7];
					if (attr_strconfig[j+8].equals(not_specified)==false)
						min_alarm = attr_strconfig[j+8];
					config[i] = new AttributeConfig(attrnames[i],
											AttrWriteType.READ,
											AttrDataFormat.SCALAR,
											attr_data_type,
											nb_x,
											nb_y,
											description,
											label,
											unit,
											standard_unit,
											display_unit,
											format,
											min_value,
											max_value,
											min_alarm,
											max_alarm,
											writable_attr_name,
											extensions);
				}
			}
			//	check if found
			if (found==false)
				Except.throw_wrong_syntax_exception("BAD_PARAMETER",
							"Attribute name not found for this device",
							"TacoDevice.get_attribute_config()");
	
		}
		return config;
	}


	//======================================================
	/**
	 *	Set the attribute data type from DevReadSigValues
	 *	TACO command argout.
	 */
	//======================================================
	private void setAttrDataType() throws DevFailed
	{
		//	Check attr_data_type
		switch(cmd_info.argoutType("DevReadSigValues"))
		{
		case Tango_DEVVAR_LONGARRAY:
			attr_orig_type = Tango_DEV_LONG;
			attr_data_type = Tango_DEV_LONG;
			break;
		case Tango_DEVVAR_FLOATARRAY:
			attr_orig_type = Tango_DEV_FLOAT;
			attr_data_type = Tango_DEV_DOUBLE;
			break;
		case Tango_DEVVAR_DOUBLEARRAY:
			attr_orig_type = Tango_DEV_DOUBLE;
			attr_data_type = Tango_DEV_DOUBLE;
			break;
		default:
			Except.throw_wrong_syntax_exception("BAD_PARAMETER",
							"Output parameter not supported.",
							"TacoDevice.setAttrDataType()");
		}
	}
	//======================================================
	/**
	 *	Convert TACO format (C language convention) to 
	 *	TANGO display format.
	 *	@param	taco_format the TACO format (C language convention).
	 */
	//======================================================
	private String tangoFormat(String taco_format) throws DevFailed
	{
		StringBuffer	format = new StringBuffer("");
		String			scientific = "";
		switch(attr_data_type)
		{
		case Tango_DEVVAR_LONGARRAY:	//	Nothing special.
			break;
		default:
			int	start = taco_format.indexOf("%");
			if (start<0)
				break;
			//	Check if scientific notation
			if (taco_format.toUpperCase().indexOf('E') >=0 ||
				taco_format.toUpperCase().indexOf('G') >=0 )
				scientific = "scientific";
			format.append(scientific);

			//	Check if precision specified
			start = taco_format.indexOf(".");
			if (start>0)
			{
				start++;
				//	Get number of decimals to set precision
				int	 end = start;
				while (taco_format.charAt(end)>='0' &&taco_format.charAt(end)<='9')
					end++;
				String	s  = taco_format.substring(start, end);
				int	nb_dec = Integer.parseInt(s);
				if (nb_dec>0)
				{
					if (scientific.length()>0)
						format.append(";");
					format.append(new String("setprecision(" + nb_dec + ")"));
				}
			}
		}
		return format.toString();
	}
	//======================================================
	//======================================================
	DeviceAttribute[] read_attribute(String[] attrnames) throws DevFailed
	{
		//	Read Signal config if not already done
		if (attr_strconfig==null)
		{
			DeviceData	argout = command_inout("DevGetSigConfig", null);
			attr_strconfig = argout.extractStringArray();
			setAttrDataType();
		}

		//	Get the TACO signal and fill an array of out type
		DeviceData	argout = command_inout("DevReadSigValues", null);
		int[]		l_val = null;
		float[]		f_val = null;
		double[]	d_val = null;
		switch(attr_orig_type)
		{
		case Tango_DEV_LONG:
			l_val = argout.extractLongArray();
			break;
		case Tango_DEV_FLOAT:
			f_val = argout.extractFloatArray();
			break;
		case Tango_DEV_DOUBLE:
			d_val = argout.extractDoubleArray();
			break;
		}

		//	Allocate an array to be returned and fill it
		DeviceAttribute[]	dev_attr = new DeviceAttribute[attrnames.length];
		int					modulo = Integer.parseInt(attr_strconfig[0]);
		for (int i=0 ; i<attrnames.length ; i++)
		{
			String	fullname = fullName(attrnames[i]);
			//	search attribute name
			int	idx = -1;
			for (int j=1 ; j<attr_strconfig.length ; j+=modulo)
				if (attr_strconfig[j].equals(fullname))
					idx = j/modulo;

			//	check if found
			if (idx==-1)
				Except.throw_wrong_syntax_exception("BAD_PARAMETER",
							"Attribute name not found for this device",
							"TacoDevice.read_attribute()");

			//	Fill the attribute value
			switch(attr_orig_type)
			{
			case Tango_DEV_LONG:
				dev_attr[i] =  new DeviceAttribute(attrnames[i], l_val[idx]);
				break;
			case Tango_DEV_FLOAT:
				//	Because Float not supported by taco but mainly used in TACO
				if (attr_data_type==Tango_DEV_DOUBLE)
				{
					double	d = (double) f_val[idx];
					dev_attr[i] =  new DeviceAttribute(attrnames[i], d);
				}
				else
					dev_attr[i] =  new DeviceAttribute(attrnames[i], f_val[idx]);
				break;
			case Tango_DEV_DOUBLE:
				dev_attr[i] =  new DeviceAttribute(attrnames[i], d_val[idx]);
				break;
			}
		}
		return dev_attr;
	}
	//======================================================
	//======================================================
}
